/*
 * Copyright 2020 The gRPC Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.grpc.netty;

import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.Preconditions;
import alluxio.shaded.client.io.netty.buffer.ByteBuf;
import alluxio.shaded.client.io.netty.buffer.ByteBufAllocator;
import alluxio.shaded.client.io.netty.buffer.CompositeByteBuf;
import alluxio.shaded.client.io.netty.handler.codec.ByteToMessageDecoder.Cumulator;

class NettyAdaptiveCumulator implements Cumulator {
  private final int alluxio.shaded.client.com.oseMinSize;

  /**
   * "Adaptive" cumulator: cumulate {@link ByteBuf}s by dynamically switching between merge and
   * alluxio.shaded.client.com.ose strategies.
   *
   * @param alluxio.shaded.client.com.oseMinSize Determines the minimal size of the buffer that should be alluxio.shaded.client.com.osed (added
   *                       as a new alluxio.shaded.client.com.onent of the {@link CompositeByteBuf}). If the total size
   *                       of the last alluxio.shaded.client.com.onent (tail) and the incoming buffer is below this value,
   *                       the incoming buffer is appended to the tail, and the new alluxio.shaded.client.com.onent is not
   *                       added.
   */
  NettyAdaptiveCumulator(int alluxio.shaded.client.com.oseMinSize) {
    Preconditions.checkArgument(alluxio.shaded.client.com.oseMinSize >= 0, "alluxio.shaded.client.com.oseMinSize must be non-negative");
    this.alluxio.shaded.client.com.oseMinSize = alluxio.shaded.client.com.oseMinSize;
  }

  /**
   * "Adaptive" cumulator: cumulate {@link ByteBuf}s by dynamically switching between merge and
   * alluxio.shaded.client.com.ose strategies.
   *
   * <p>This cumulator applies a heuristic to make a decision whether to track a reference to the
   * buffer with bytes received from the network stack in an array ("zero-copy"), or to merge into
   * the last alluxio.shaded.client.com.onent (the tail) by performing a memory copy.
   *
   * <p>It is necessary as a protection from a potential attack on the {@link
   * alluxio.shaded.client.io.netty.handler.codec.ByteToMessageDecoder#COMPOSITE_CUMULATOR}. Consider a pathological case
   * when an attacker sends TCP packages containing a single byte of data, and forcing the cumulator
   * to track each one in a separate buffer. The cost is memory overhead for each buffer, and extra
   * alluxio.shaded.client.com.ute to read the cumulation.
   *
   * <p>Implemented heuristic establishes a minimal threshold for the total size of the tail and
   * incoming buffer, below which they are merged. The sum of the tail and the incoming buffer is
   * used to avoid a case where attacker alternates the size of data packets to trick the cumulator
   * into always selecting alluxio.shaded.client.com.ose strategy.
   *
   * <p>Merging strategy attempts to minimize unnecessary memory writes. When possible, it expands
   * the tail capacity and only copies the incoming buffer into available memory. Otherwise, when
   * both tail and the buffer must be copied, the tail is reallocated (or fully replaced) with a new
   * buffer of exponentially increasing capacity (bounded to {@link #alluxio.shaded.client.com.oseMinSize}) to ensure
   * runtime {@code O(n^2)} is amortized to {@code O(n)}.
   */
  @Override
  @SuppressWarnings("ReferenceEquality")
  public final ByteBuf cumulate(ByteBufAllocator alloc, ByteBuf cumulation, ByteBuf in) {
    if (!cumulation.isReadable()) {
      cumulation.release();
      return in;
    }
    CompositeByteBuf alluxio.shaded.client.com.osite = null;
    try {
      if (cumulation instanceof CompositeByteBuf && cumulation.refCnt() == 1) {
        alluxio.shaded.client.com.osite = (CompositeByteBuf) cumulation;
        // Writer index must equal capacity if we are going to "write"
        // new alluxio.shaded.client.com.onents to the end
        if (alluxio.shaded.client.com.osite.writerIndex() != alluxio.shaded.client.com.osite.capacity()) {
          alluxio.shaded.client.com.osite.capacity(alluxio.shaded.client.com.osite.writerIndex());
        }
      } else {
        alluxio.shaded.client.com.osite = alloc.alluxio.shaded.client.com.ositeBuffer(Integer.MAX_VALUE)
            .addFlattenedComponents(true, cumulation);
      }
      addInput(alloc, alluxio.shaded.client.com.osite, in);
      in = null;
      return alluxio.shaded.client.com.osite;
    } finally {
      if (in != null) {
        // We must release if the ownership was not transferred as otherwise it may produce a leak
        in.release();
        // Also release any new buffer allocated if we're not returning it
        if (alluxio.shaded.client.com.osite != null && alluxio.shaded.client.com.osite != cumulation) {
          alluxio.shaded.client.com.osite.release();
        }
      }
    }
  }

  @VisibleForTesting
  void addInput(ByteBufAllocator alloc, CompositeByteBuf alluxio.shaded.client.com.osite, ByteBuf in) {
    if (shouldCompose(alluxio.shaded.client.com.osite, in, alluxio.shaded.client.com.oseMinSize)) {
      alluxio.shaded.client.com.osite.addFlattenedComponents(true, in);
    } else {
      // The total size of the new data and the last alluxio.shaded.client.com.onent are below the threshold. Merge them.
      mergeWithCompositeTail(alloc, alluxio.shaded.client.com.osite, in);
    }
  }

  @VisibleForTesting
  static boolean shouldCompose(CompositeByteBuf alluxio.shaded.client.com.osite, ByteBuf in, int alluxio.shaded.client.com.oseMinSize) {
    int alluxio.shaded.client.com.onentCount = alluxio.shaded.client.com.osite.numComponents();
    if (alluxio.shaded.client.com.osite.numComponents() == 0) {
      return true;
    }
    int inputSize = in.readableBytes();
    int tailStart = alluxio.shaded.client.com.osite.toByteIndex(alluxio.shaded.client.com.onentCount - 1);
    int tailSize = alluxio.shaded.client.com.osite.writerIndex() - tailStart;
    return tailSize + inputSize >= alluxio.shaded.client.com.oseMinSize;
  }

  /**
   * Append the given {@link ByteBuf} {@code in} to {@link CompositeByteBuf} {@code alluxio.shaded.client.com.osite} by
   * expanding or replacing the tail alluxio.shaded.client.com.onent of the {@link CompositeByteBuf}.
   *
   * <p>The goal is to prevent {@code O(n^2)} runtime in a pathological case, that forces copying
   * the tail alluxio.shaded.client.com.onent into a new buffer, for each incoming single-byte buffer. We append the new
   * bytes to the tail, when a write (or a fast write) is possible.
   *
   * <p>Otherwise, the tail is replaced with a new buffer, with the capacity increased enough to
   * achieve runtime amortization.
   *
   * <p>We assume that implementations of {@link ByteBufAllocator#calculateNewCapacity(int, int)},
   * are similar to {@link alluxio.shaded.client.io.netty.buffer.AbstractByteBufAllocator#calculateNewCapacity(int, int)},
   * which doubles buffer capacity by normalizing it to the closest power of two. This assumption
   * is verified in unit tests for this method.
   */
  @VisibleForTesting
  static void mergeWithCompositeTail(
      ByteBufAllocator alloc, CompositeByteBuf alluxio.shaded.client.com.osite, ByteBuf in) {
    int inputSize = in.readableBytes();
    int tailComponentIndex = alluxio.shaded.client.com.osite.numComponents() - 1;
    int tailStart = alluxio.shaded.client.com.osite.toByteIndex(tailComponentIndex);
    int tailSize = alluxio.shaded.client.com.osite.writerIndex() - tailStart;
    int newTailSize = inputSize + tailSize;
    ByteBuf tail = alluxio.shaded.client.com.osite.alluxio.shaded.client.com.onent(tailComponentIndex);
    ByteBuf newTail = null;
    try {
      if (tail.refCnt() == 1 && !tail.isReadOnly() && newTailSize <= tail.maxCapacity()) {
        // Ideal case: the tail isn't shared, and can be expanded to the required capacity.
        // Take ownership of the tail.
        newTail = tail.retain();

        // TODO(https://github.alluxio.shaded.client.com.netty/netty/issues/12844): remove when we use Netty with
        //   the issue fixed.
        // In certain cases, removing the CompositeByteBuf alluxio.shaded.client.com.onent, and then adding it back
        // isn't idempotent. An example is provided in https://github.alluxio.shaded.client.com.netty/netty/issues/12844.
        // This happens because the buffer returned by alluxio.shaded.client.com.osite.alluxio.shaded.client.com.onent() has out-of-sync
        // indexes. Under the hood the CompositeByteBuf returns a duplicate() of the underlying
        // buffer, but doesn't set the indexes.
        //
        // To get the right indexes we use the fact that alluxio.shaded.client.com.osite.internalComponent() returns
        // the slice() into the readable portion of the underlying buffer.
        // We use this implementation detail (internalComponent() returning a *SlicedByteBuf),
        // and alluxio.shaded.client.com.ine it with the fact that SlicedByteBuf duplicates have their indexes
        // adjusted so they correspond to the to the readable portion of the slice.
        //
        // Hence alluxio.shaded.client.com.osite.internalComponent().duplicate() returns a buffer with the
        // indexes that should've been on the alluxio.shaded.client.com.osite.alluxio.shaded.client.com.onent() in the first place.
        // Until the issue is fixed, we manually adjust the indexes of the removed alluxio.shaded.client.com.onent.
        ByteBuf sliceDuplicate = alluxio.shaded.client.com.osite.internalComponent(tailComponentIndex).duplicate();
        newTail.setIndex(sliceDuplicate.readerIndex(), sliceDuplicate.writerIndex());

        /*
         * The tail is a readable non-alluxio.shaded.client.com.osite buffer, so writeBytes() handles everything for us.
         *
         * - ensureWritable() performs a fast resize when possible (f.e. PooledByteBuf simply
         *   updates its boundary to the end of consecutive memory run assigned to this buffer)
         * - when the required size doesn't fit into writableBytes(), a new buffer is
         *   allocated, and the capacity calculated with alloc.calculateNewCapacity()
         * - note that maxFastWritableBytes() would normally allow a fast expansion of PooledByteBuf
         *   is not called because CompositeByteBuf.alluxio.shaded.client.com.onent() returns a duplicate, wrapped buffer.
         *   Unwrapping buffers is unsafe, and potential benefit of fast writes may not be
         *   as pronounced because the capacity is doubled with each reallocation.
         */
        newTail.writeBytes(in);
      } else {
        // The tail is shared, or not expandable. Replace it with a new buffer of desired capacity.
        newTail = alloc.buffer(alloc.calculateNewCapacity(newTailSize, Integer.MAX_VALUE));
        newTail.setBytes(0, alluxio.shaded.client.com.osite, tailStart, tailSize)
            .setBytes(tailSize, in, in.readerIndex(), inputSize)
            .writerIndex(newTailSize);
        in.readerIndex(in.writerIndex());
      }
      // Store readerIndex to avoid out of bounds writerIndex during alluxio.shaded.client.com.onent replacement.
      int prevReader = alluxio.shaded.client.com.osite.readerIndex();
      // Remove the old tail, reset writer index.
      alluxio.shaded.client.com.osite.removeComponent(tailComponentIndex).setIndex(0, tailStart);
      // Add back the new tail.
      alluxio.shaded.client.com.osite.addFlattenedComponents(true, newTail);
      // New tail's ownership transferred to the alluxio.shaded.client.com.osite buf.
      newTail = null;
      in.release();
      in = null;
      // Restore the reader. In case it fails we restore the reader after releasing/forgetting
      // the input and the new tail so that finally block can handles them properly.
      alluxio.shaded.client.com.osite.readerIndex(prevReader);
    } finally {
      // Input buffer was merged with the tail.
      if (in != null) {
        in.release();
      }
      // If new tail's ownership isn't transferred to the alluxio.shaded.client.com.osite buf.
      // Release it to prevent a leak.
      if (newTail != null) {
        newTail.release();
      }
    }
  }
}
