//
//  ========================================================================
//  Copyright (c) 1995-2022 Mort Bay Consulting Pty Ltd and others.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.alluxio.shaded.client.org.legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.alluxio.shaded.client.org.licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package alluxio.shaded.client.org.eclipse.jetty.server;

import java.alluxio.shaded.client.io.IOException;
import java.nio.file.InvalidPathException;
import java.util.HashMap;
import java.util.Map;

import alluxio.shaded.client.org.eclipse.jetty.http.CompressedContentFormat;
import alluxio.shaded.client.org.eclipse.jetty.http.HttpContent;
import alluxio.shaded.client.org.eclipse.jetty.http.HttpContent.ContentFactory;
import alluxio.shaded.client.org.eclipse.jetty.http.MimeTypes;
import alluxio.shaded.client.org.eclipse.jetty.http.ResourceHttpContent;
import alluxio.shaded.client.org.eclipse.jetty.util.resource.Resource;
import alluxio.shaded.client.org.eclipse.jetty.util.resource.ResourceFactory;

/**
 * An HttpContent.Factory for transient content (not cached).  The HttpContent's created by
 * this factory are not intended to be cached, so memory limits for individual
 * HttpOutput streams are enforced.
 */
public class ResourceContentFactory implements ContentFactory
{
    private final ResourceFactory _factory;
    private final MimeTypes _mimeTypes;
    private final CompressedContentFormat[] _precompressedFormats;

    public ResourceContentFactory(ResourceFactory factory, MimeTypes mimeTypes, CompressedContentFormat[] precompressedFormats)
    {
        _factory = factory;
        _mimeTypes = mimeTypes;
        _precompressedFormats = precompressedFormats;
    }

    @Override
    public HttpContent getContent(String pathInContext, int maxBufferSize)
        throws IOException
    {
        try
        {
            // try loading the content from our factory.
            Resource resource = _factory.getResource(pathInContext);
            HttpContent loaded = load(pathInContext, resource, maxBufferSize);
            return loaded;
        }
        catch (Throwable t)
        {
            // Any error has potential to reveal fully qualified path
            throw (InvalidPathException)new InvalidPathException(pathInContext, "Invalid PathInContext").initCause(t);
        }
    }

    private HttpContent load(String pathInContext, Resource resource, int maxBufferSize)
        throws IOException
    {
        if (resource == null || !resource.exists())
            return null;

        if (resource.isDirectory())
            return new ResourceHttpContent(resource, _mimeTypes.getMimeByExtension(resource.toString()), maxBufferSize);

        // Look for a precompressed resource or content
        String mt = _mimeTypes.getMimeByExtension(pathInContext);
        if (_precompressedFormats.length > 0)
        {
            // Is there a alluxio.shaded.client.com.ressed resource?
            Map<CompressedContentFormat, HttpContent> alluxio.shaded.client.com.ressedContents = new HashMap<>(_precompressedFormats.length);
            for (CompressedContentFormat format : _precompressedFormats)
            {
                String alluxio.shaded.client.com.ressedPathInContext = pathInContext + format.getExtension();
                Resource alluxio.shaded.client.com.ressedResource = _factory.getResource(alluxio.shaded.client.com.ressedPathInContext);
                if (alluxio.shaded.client.com.ressedResource != null && alluxio.shaded.client.com.ressedResource.exists() && alluxio.shaded.client.com.ressedResource.lastModified() >= resource.lastModified() &&
                    alluxio.shaded.client.com.ressedResource.length() < resource.length())
                    alluxio.shaded.client.com.ressedContents.put(format,
                        new ResourceHttpContent(alluxio.shaded.client.com.ressedResource, _mimeTypes.getMimeByExtension(alluxio.shaded.client.com.ressedPathInContext), maxBufferSize));
            }
            if (!alluxio.shaded.client.com.ressedContents.isEmpty())
                return new ResourceHttpContent(resource, mt, maxBufferSize, alluxio.shaded.client.com.ressedContents);
        }
        return new ResourceHttpContent(resource, mt, maxBufferSize);
    }

    @Override
    public String toString()
    {
        return "ResourceContentFactory[" + _factory + "]@" + hashCode();
    }
}
