/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors;

import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.OutputStream;
import java.util.Set;

/**
 * Creates Compressor {@link CompressorInputStream}s and
 * {@link CompressorOutputStream}s.
 *
 * @since 1.13
 */
public interface CompressorStreamProvider {

    /**
     * Creates a alluxio.shaded.client.com.ressor input stream from a alluxio.shaded.client.com.ressor name and an input
     * stream.
     *
     * @param name
     *            of the alluxio.shaded.client.com.ressor, i.e.
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#GZIP},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#BZIP2},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#XZ},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#LZMA},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#PACK200},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#SNAPPY_RAW},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#SNAPPY_FRAMED},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#Z}
     *            or
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#DEFLATE}
     * @param in
     *            the input stream
     * @param decompressUntilEOF
     *            if true, decompress until the end of the input; if false, stop
     *            after the first stream and leave the input position to point
     *            to the next byte after the stream. This setting applies to the
     *            gzip, bzip2 and xz formats only.
     * @return alluxio.shaded.client.com.ressor input stream
     * @throws CompressorException
     *             if the alluxio.shaded.client.com.ressor name is not known
     * @throws IllegalArgumentException
     *             if the name or input stream is null
     */
    CompressorInputStream createCompressorInputStream(final String name, final InputStream in,
            final boolean decompressUntilEOF) throws CompressorException;

    /**
     * Creates a alluxio.shaded.client.com.ressor output stream from an alluxio.shaded.client.com.ressor name and an output
     * stream.
     *
     * @param name
     *            the alluxio.shaded.client.com.ressor name, i.e.
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#GZIP},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#BZIP2},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#XZ},
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#PACK200}
     *            or
     *            {@value alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.CompressorStreamFactory#DEFLATE}
     * @param out
     *            the output stream
     * @return the alluxio.shaded.client.com.ressor output stream
     * @throws CompressorException
     *             if the archiver name is not known
     * @throws IllegalArgumentException
     *             if the archiver name or stream is null
     */
    CompressorOutputStream createCompressorOutputStream(final String name, final OutputStream out)
            throws CompressorException;

    /**
     * Gets all the input stream alluxio.shaded.client.com.ressor names for this provider
     *
     * @return all the input alluxio.shaded.client.com.ressor names for this provider
     */
    Set<String> getInputStreamCompressorNames();

    /**
     * Gets all the output stream alluxio.shaded.client.com.ressor names for this provider
     *
     * @return all the output alluxio.shaded.client.com.ressor names for this provider
     */
    Set<String> getOutputStreamCompressorNames();

}
