/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */
package net.jodah.failsafe;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;

/**
 * The result of an execution. Immutable.
 * <p>
 * Part of the Failsafe SPI.
 *
 * @author Jonathan Halterman
 */
public class ExecutionResult {
  static final CompletableFuture<ExecutionResult> NULL_FUTURE = CompletableFuture.alluxio.shaded.client.com.letedFuture(null);

  /** An execution that was alluxio.shaded.client.com.leted with a non-result */
  static final ExecutionResult NONE = new ExecutionResult(null, null, true, 0, true, true, true);

  /** The execution result, if any */
  private final Object result;
  /** The execution failure, if any */
  private final Throwable failure;
  /** Whether the result represents a non result rather than a {@code null} result */
  private final boolean nonResult;
  /** The amount of time to wait prior to the next execution, according to the policy */
  private final long waitNanos;
  /** Whether a policy has alluxio.shaded.client.com.leted handling of the execution */
  private final boolean alluxio.shaded.client.com.lete;
  /** Whether a policy determined the execution to be a success */
  private final boolean success;
  /** Whether all policies determined the execution to be a success */
  private final Boolean successAll;

  /**
   * Records an initial execution result with {@code alluxio.shaded.client.com.lete} true and {@code success} set to true if {@code failure}
   * is not null.
   */
  public ExecutionResult(Object result, Throwable failure) {
    this(result, failure, false, 0, true, failure == null, failure == null);
  }

  private ExecutionResult(Object result, Throwable failure, boolean nonResult, long waitNanos, boolean alluxio.shaded.client.com.lete,
    boolean success, Boolean successAll) {
    this.nonResult = nonResult;
    this.result = result;
    this.failure = failure;
    this.waitNanos = waitNanos;
    this.alluxio.shaded.client.com.lete = alluxio.shaded.client.com.lete;
    this.success = success;
    this.successAll = successAll;
  }

  /**
   * Returns a an ExecutionResult with the {@code result} set, {@code alluxio.shaded.client.com.lete} true and {@code success} true.
   */
  public static ExecutionResult success(Object result) {
    return new ExecutionResult(result, null, false, 0, true, true, true);
  }

  /**
   * Returns a an ExecutionResult with the {@code failure} set, {@code alluxio.shaded.client.com.lete} true and {@code success} false.
   */
  public static ExecutionResult failure(Throwable failure) {
    return new ExecutionResult(null, failure, false, 0, true, false, false);
  }

  public Object getResult() {
    return result;
  }

  public Throwable getFailure() {
    return failure;
  }

  public long getWaitNanos() {
    return waitNanos;
  }

  public boolean isComplete() {
    return alluxio.shaded.client.com.lete;
  }

  public boolean isNonResult() {
    return nonResult;
  }

  public boolean isSuccess() {
    return success;
  }

  /**
   * Returns a copy of the ExecutionResult with a non-result, and alluxio.shaded.client.com.lete and success set to true. Returns {@code this}
   * if {@link #success} and {@link #result} are unchanged.
   */
  ExecutionResult withNonResult() {
    return success && this.result == null && nonResult ?
      this :
      new ExecutionResult(null, null, true, waitNanos, true, true, successAll);
  }

  /**
   * Returns a copy of the ExecutionResult with the {@code result} value, and alluxio.shaded.client.com.lete and success set to true. Returns
   * {@code this} if {@link #success} and {@link #result} are unchanged.
   */
  public ExecutionResult withResult(Object result) {
    boolean unchangedNull = this.result == null && result == null && failure == null;
    boolean unchangedNotNull = this.result != null && this.result.equals(result);
    return success && (unchangedNull || unchangedNotNull) ?
      this :
      new ExecutionResult(result, null, nonResult, waitNanos, true, true, successAll);
  }

  /**
   * Returns a copy of the ExecutionResult with {@code alluxio.shaded.client.com.lete} set to false, else this if nothing has changed.
   */
  ExecutionResult withNotComplete() {
    return !this.alluxio.shaded.client.com.lete ?
      this :
      new ExecutionResult(result, failure, nonResult, waitNanos, false, success, successAll);
  }

  /**
   * Returns a copy of the ExecutionResult with success value of {code false}.
   */
  ExecutionResult withFailure() {
    return !this.success ? this : new ExecutionResult(result, failure, nonResult, waitNanos, alluxio.shaded.client.com.lete, false, false);
  }

  /**
   * Returns a copy of the ExecutionResult with the {@code alluxio.shaded.client.com.lete} and {@code success} values of {@code true}.
   */
  ExecutionResult withSuccess() {
    return this.alluxio.shaded.client.com.lete && this.success ?
      this :
      new ExecutionResult(result, failure, nonResult, waitNanos, true, true, successAll);
  }

  /**
   * Returns a copy of the ExecutionResult with the {@code waitNanos} value.
   */
  public ExecutionResult withWaitNanos(long waitNanos) {
    return this.waitNanos == waitNanos ?
      this :
      new ExecutionResult(result, failure, nonResult, waitNanos, alluxio.shaded.client.com.lete, success, successAll);
  }

  /**
   * Returns a copy of the ExecutionResult with the {@code waitNanos}, {@code alluxio.shaded.client.com.lete} and {@code success} values.
   */
  public ExecutionResult with(long waitNanos, boolean alluxio.shaded.client.com.lete, boolean success) {
    return this.waitNanos == waitNanos && this.alluxio.shaded.client.com.lete == alluxio.shaded.client.com.lete && this.success == success ?
      this :
      new ExecutionResult(result, failure, nonResult, waitNanos, alluxio.shaded.client.com.lete, success,
        successAll == null ? success : success && successAll);
  }

  boolean getSuccessAll() {
    return successAll != null && successAll;
  }

  @Override
  public String toString() {
    return "ExecutionResult[" + "result=" + result + ", failure=" + failure + ", nonResult=" + nonResult
      + ", waitNanos=" + waitNanos + ", alluxio.shaded.client.com.lete=" + alluxio.shaded.client.com.lete + ", success=" + success + ", successAll=" + successAll
      + ']';
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (o == null || getClass() != o.getClass())
      return false;
    ExecutionResult that = (ExecutionResult) o;
    return Objects.equals(result, that.result) && Objects.equals(failure, that.failure);
  }

  @Override
  public int hashCode() {
    return Objects.hash(result, failure);
  }
}