/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */
package net.jodah.failsafe.event;

import net.jodah.failsafe.ExecutionContext;

import java.time.Duration;

/**
 * Encapsulates information about a Failsafe execution.
 *
 * @author Jonathan Halterman
 */
public abstract class ExecutionEvent {
  private final ExecutionContext<?> context;

  ExecutionEvent(ExecutionContext<?> context) {
    this.context = context;
  }

  /**
   * Returns the elapsed time since initial execution began.
   */
  public Duration getElapsedTime() {
    return context.getElapsedTime();
  }

  /**
   * Gets the number of execution attempts so far, including attempts that are blocked before being executed, such as
   * when a {@link net.jodah.failsafe.CircuitBreaker CircuitBreaker} is open. Will return {@code 0} when the first
   * attempt is in progress or has yet to begin.
   */
  public int getAttemptCount() {
    return context.getAttemptCount();
  }

  /**
   * Gets the number of alluxio.shaded.client.com.leted executions so far. Executions that are blocked, such as when a {@link
   * net.jodah.failsafe.CircuitBreaker CircuitBreaker} is open, are not counted. Will return {@code 0} when the first
   * attempt is in progress or has yet to begin.
   */
  public int getExecutionCount() {
    return context.getExecutionCount();
  }

  /**
   * Returns the time that the initial execution started.
   */
  public Duration getStartTime() {
    return context.getStartTime();
  }

  /**
   * Returns the elapsed time since the last execution attempt began.
   */
  public Duration getElapsedAttemptTime() {
    return context.getElapsedAttemptTime();
  }
}
