/*
 * Decompiled with CFR 0.152.
 */
package alluxio.underfs.hdfs;

import alluxio.AlluxioURI;
import alluxio.collections.Pair;
import alluxio.conf.PropertyKey;
import alluxio.retry.CountingRetry;
import alluxio.security.authorization.AccessControlList;
import alluxio.security.authorization.AclEntry;
import alluxio.security.authorization.DefaultAccessControlList;
import alluxio.underfs.AtomicFileOutputStream;
import alluxio.underfs.AtomicFileOutputStreamCallback;
import alluxio.underfs.ChecksumType;
import alluxio.underfs.ConsistentUnderFileSystem;
import alluxio.underfs.UfsDirectoryStatus;
import alluxio.underfs.UfsFileStatus;
import alluxio.underfs.UfsStatus;
import alluxio.underfs.UnderFileSystem;
import alluxio.underfs.UnderFileSystemConfiguration;
import alluxio.underfs.hdfs.AlluxioHdfsException;
import alluxio.underfs.hdfs.HdfsAclProvider;
import alluxio.underfs.hdfs.HdfsPositionedUnderFileInputStream;
import alluxio.underfs.hdfs.HdfsUfsStatusIterator;
import alluxio.underfs.hdfs.HdfsUnderFileInputStream;
import alluxio.underfs.hdfs.HdfsUnderFileOutputStream;
import alluxio.underfs.hdfs.NoopHdfsAclProvider;
import alluxio.underfs.options.CreateOptions;
import alluxio.underfs.options.DeleteOptions;
import alluxio.underfs.options.FileLocationOptions;
import alluxio.underfs.options.GetStatusOptions;
import alluxio.underfs.options.ListOptions;
import alluxio.underfs.options.MkdirsOptions;
import alluxio.underfs.options.OpenOptions;
import alluxio.util.CommonUtils;
import alluxio.util.UnderFileSystemUtils;
import alluxio.util.network.NetworkAddressUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class HdfsUnderFileSystem
extends ConsistentUnderFileSystem
implements AtomicFileOutputStreamCallback {
    private static final Logger LOG = LoggerFactory.getLogger(HdfsUnderFileSystem.class);
    protected static final int MAX_TRY = 5;
    protected static final String HDFS_USER = "";
    protected static final String HDFS_ACL_PROVIDER_CLASS = "alluxio.underfs.hdfs.acl.SupportedHdfsAclProvider";
    protected static final String HDFS_EC_MIN_VERSION = "3.0.0";
    protected static final String HDFS_EC_CODEC_REGISTRY_CLASS = "org.apache.hadoop.io.erasurecode.CodecRegistry";
    protected static final String JAVAX_WS_RS_CORE_MEDIA_TYPE = "javax.ws.rs.core.MediaType";
    protected static final String HADOOP_AUTH_METHOD = "hadoop.security.authentication";
    protected static final String KRB5_CONF_FILE = "java.security.krb5.conf";
    protected static final String KRB_KEYTAB_LOGIN_AUTO_RENEW = "hadoop.kerberos.keytab.login.autorenewal.enabled";
    protected static final String CHECKSUM_COMBINE_MODE = "dfs.checksum.combine.mode";
    protected static final String USER_NAMESPACE_PREFIX = "user.";
    private final LoadingCache<String, FileSystem> mUserFs;
    protected final HdfsAclProvider mHdfsAclProvider;
    private final boolean mTrashEnable = alluxio.conf.Configuration.getBoolean((PropertyKey)PropertyKey.UNDERFS_HDFS_TRASH_ENABLED);
    private final LoadingCache<FileSystem, Trash> mFsTrash;

    public static HdfsUnderFileSystem createInstance(AlluxioURI ufsUri, UnderFileSystemConfiguration conf) {
        Configuration hdfsConf = HdfsUnderFileSystem.createConfiguration(conf);
        return new HdfsUnderFileSystem(ufsUri, conf, hdfsConf);
    }

    public HdfsUnderFileSystem(AlluxioURI ufsUri, UnderFileSystemConfiguration conf, Configuration hdfsConf) {
        this(ufsUri, conf, hdfsConf, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected HdfsUnderFileSystem(AlluxioURI ufsUri, UnderFileSystemConfiguration conf, final Configuration hdfsConf, boolean useLoadingCache) {
        super(ufsUri, conf);
        LOG.info(PropertyKey.UNDERFS_HDFS_TRASH_ENABLED.getName() + " is set to {}", (Object)this.mTrashEnable);
        this.mFsTrash = CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<FileSystem, Trash>(){

            public Trash load(FileSystem fs) throws Exception {
                return new Trash(fs, fs.getConf());
            }
        });
        HdfsAclProvider hdfsAclProvider = new NoopHdfsAclProvider();
        try {
            Object o = Class.forName(HDFS_ACL_PROVIDER_CLASS).newInstance();
            if (o instanceof HdfsAclProvider) {
                hdfsAclProvider = (HdfsAclProvider)o;
            } else {
                LOG.warn("SupportedHdfsAclProvider is not instance of HdfsAclProvider. HDFS ACLs will not be supported.");
            }
        }
        catch (Exception e) {
            LOG.warn("Cannot create SupportedHdfsAclProvider. HDFS ACLs is not supported, Please upgrade to an HDFS version > 2.4 to enable support for ACL");
            LOG.debug("Exception:", (Throwable)e);
        }
        this.mHdfsAclProvider = hdfsAclProvider;
        final Path path = new Path(ufsUri.toString());
        final ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(hdfsConf.getClassLoader());
            if (this.mUfsConf.isSet(PropertyKey.HADOOP_SECURITY_AUTHENTICATION)) {
                hdfsConf.set(HADOOP_AUTH_METHOD, this.mUfsConf.getString(PropertyKey.HADOOP_SECURITY_AUTHENTICATION));
            }
            if (this.mUfsConf.isSet(PropertyKey.HADOOP_KRB5_CONF_FILE)) {
                System.setProperty(KRB5_CONF_FILE, this.mUfsConf.getString(PropertyKey.HADOOP_KRB5_CONF_FILE));
            }
            if (this.mUfsConf.isSet(PropertyKey.HADOOP_KERBEROS_KEYTAB_LOGIN_AUTORENEWAL)) {
                hdfsConf.setBoolean(KRB_KEYTAB_LOGIN_AUTO_RENEW, this.mUfsConf.getBoolean(PropertyKey.HADOOP_KERBEROS_KEYTAB_LOGIN_AUTORENEWAL));
            }
            if (((ChecksumType)this.mUfsConf.getEnum(PropertyKey.UNDERFS_CHECKSUM_TYPE, ChecksumType.class)).equals((Object)ChecksumType.CRC32C)) {
                hdfsConf.set(CHECKSUM_COMBINE_MODE, "COMPOSITE_CRC");
            }
            UserGroupInformation.setConfiguration((Configuration)hdfsConf);
            if ("3.3.1".compareTo(HDFS_EC_MIN_VERSION) >= 0) {
                try {
                    Class.forName(HDFS_EC_CODEC_REGISTRY_CLASS);
                }
                catch (ClassNotFoundException e) {
                    LOG.warn("Cannot initialize HDFS EC CodecRegistry. HDFS EC will not be supported:", (Throwable)e);
                }
            }
            if (StringUtils.equals((CharSequence)"hdfs", (CharSequence)ufsUri.getScheme())) {
                try {
                    Class.forName(JAVAX_WS_RS_CORE_MEDIA_TYPE);
                }
                catch (ClassNotFoundException e) {
                    LOG.warn("Cannot initialize javax.ws.rs.MediaType.", (Throwable)e);
                }
            }
        }
        finally {
            Thread.currentThread().setContextClassLoader(currentClassLoader);
        }
        this.mUserFs = useLoadingCache ? CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<String, FileSystem>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public FileSystem load(String userKey) throws Exception {
                ClassLoader previousClassLoader = Thread.currentThread().getContextClassLoader();
                try {
                    Thread.currentThread().setContextClassLoader(currentClassLoader);
                    FileSystem fileSystem = path.getFileSystem(hdfsConf);
                    return fileSystem;
                }
                finally {
                    Thread.currentThread().setContextClassLoader(previousClassLoader);
                }
            }
        }) : null;
    }

    public String getUnderFSType() {
        return "hdfs";
    }

    public static Configuration createConfiguration(UnderFileSystemConfiguration conf) {
        Preconditions.checkNotNull((Object)conf, (Object)"conf");
        Configuration hdfsConf = new Configuration();
        for (String path : conf.getString(PropertyKey.UNDERFS_HDFS_CONFIGURATION).split(":")) {
            if (path.isEmpty()) continue;
            hdfsConf.addResource(new Path(path));
        }
        String ufsHdfsImpl = conf.getString(PropertyKey.UNDERFS_HDFS_IMPL);
        if (!StringUtils.isEmpty((CharSequence)ufsHdfsImpl)) {
            hdfsConf.set("fs.hdfs.impl", ufsHdfsImpl);
        }
        hdfsConf.set("fs.hdfs.impl.disable.cache", System.getProperty("fs.hdfs.impl.disable.cache", "true"));
        for (Map.Entry entry : conf.getMountSpecificConf().entrySet()) {
            hdfsConf.set((String)entry.getKey(), entry.getValue() == null ? null : entry.getValue().toString());
        }
        return hdfsConf;
    }

    public void cleanup() {
    }

    public void close() {
    }

    public OutputStream create(String path, CreateOptions options) throws IOException {
        if (!options.isEnsureAtomic()) {
            return this.createDirect(path, options);
        }
        return new AtomicFileOutputStream(path, (AtomicFileOutputStreamCallback)this, options);
    }

    public OutputStream createDirect(String path, CreateOptions options) throws IOException {
        IOException te = null;
        FileSystem hdfs = this.getFs();
        CountingRetry retryPolicy = new CountingRetry(5);
        while (retryPolicy.attempt()) {
            try {
                HdfsUnderFileOutputStream outputStream = new HdfsUnderFileOutputStream(hdfs, path, FileSystem.create((FileSystem)hdfs, (Path)new Path(path), (FsPermission)new FsPermission(options.getMode().toShort())));
                if (options.getAcl() != null) {
                    this.setAclEntries(path, options.getAcl().getEntries());
                }
                return outputStream;
            }
            catch (IOException e) {
                LOG.warn("Attempt count {} : {} ", (Object)retryPolicy.getAttemptCount(), (Object)e.toString());
                te = e;
            }
        }
        throw te;
    }

    public boolean deleteDirectory(String path, DeleteOptions options) throws IOException {
        return this.isDirectory(path) && this.delete(path, options.isRecursive(), true);
    }

    public boolean deleteFile(String path) throws IOException {
        return this.isFile(path) && this.delete(path, false, false);
    }

    public boolean exists(String path) throws IOException {
        FileSystem hdfs = this.getFs();
        return hdfs.exists(new Path(path));
    }

    public Pair<AccessControlList, DefaultAccessControlList> getAclPair(String path) throws IOException {
        return this.mHdfsAclProvider.getAcl(this.getFs(), path);
    }

    public void setAclEntries(String path, List<AclEntry> aclEntries) throws IOException {
        this.mHdfsAclProvider.setAclEntries(this.getFs(), path, aclEntries);
    }

    public long getBlockSizeByte(String path) throws IOException {
        Path tPath = new Path(path);
        FileSystem hdfs = this.getFs();
        if (!hdfs.exists(tPath)) {
            throw new FileNotFoundException(path);
        }
        FileStatus fs = hdfs.getFileStatus(tPath);
        return fs.getBlockSize();
    }

    public UfsDirectoryStatus getDirectoryStatus(String path) throws IOException {
        Path tPath = new Path(path);
        FileSystem hdfs = this.getFs();
        FileStatus fs = hdfs.getFileStatus(tPath);
        return new UfsDirectoryStatus(path, fs.getOwner(), fs.getGroup(), fs.getPermission().toShort(), Long.valueOf(fs.getModificationTime()));
    }

    public List<String> getFileLocations(String path) throws IOException {
        return this.getFileLocations(path, FileLocationOptions.defaults());
    }

    @Nullable
    public List<String> getFileLocations(String path, FileLocationOptions options) throws IOException {
        if (this.mUfsConf.getBoolean(PropertyKey.UNDERFS_HDFS_REMOTE)) {
            return null;
        }
        FileSystem hdfs = this.getFs();
        ArrayList<String> ret = new ArrayList<String>();
        try {
            FileStatus fileStatus = new FileStatus(0L, false, 0, 0L, 0L, 0L, null, null, null, new Path(path));
            BlockLocation[] bLocations = hdfs.getFileBlockLocations(fileStatus, options.getOffset(), 1L);
            if (bLocations.length > 0) {
                String[] names = bLocations[0].getHosts();
                Collections.addAll(ret, names);
            }
        }
        catch (IOException e) {
            LOG.debug("Unable to get file location for {}", (Object)path, (Object)e);
        }
        return ret;
    }

    public UfsFileStatus getFileStatus(String path, GetStatusOptions options) throws IOException {
        Path tPath = new Path(path);
        FileSystem hdfs = this.getFs();
        FileStatus fs = hdfs.getFileStatus(tPath);
        String contentHash = options.isIncludeRealContentHash() ? Base64.encodeBase64String((byte[])hdfs.getFileChecksum(tPath).getBytes()) : UnderFileSystemUtils.approximateContentHash((long)fs.getLen(), (long)fs.getModificationTime());
        return new UfsFileStatus(path, contentHash, fs.getLen(), Long.valueOf(fs.getModificationTime()), fs.getOwner(), fs.getGroup(), fs.getPermission().toShort(), fs.getBlockSize());
    }

    public long getSpace(String path, UnderFileSystem.SpaceType type) throws IOException {
        FileSystem hdfs = this.getFs();
        long space = -1L;
        if (hdfs instanceof DistributedFileSystem) {
            switch (type) {
                case SPACE_TOTAL: {
                    space = hdfs.getStatus().getCapacity();
                    break;
                }
                case SPACE_USED: {
                    space = hdfs.getStatus().getUsed();
                    break;
                }
                case SPACE_FREE: {
                    space = hdfs.getStatus().getRemaining();
                    break;
                }
                default: {
                    throw new IOException("Unknown space type: " + type);
                }
            }
        }
        return space;
    }

    public UfsStatus getStatus(String path, GetStatusOptions options) throws IOException {
        Path tPath = new Path(path);
        FileSystem hdfs = this.getFs();
        FileStatus fs = hdfs.getFileStatus(tPath);
        if (!fs.isDir()) {
            String contentHash = options.isIncludeRealContentHash() ? Base64.encodeBase64String((byte[])hdfs.getFileChecksum(tPath).getBytes()) : UnderFileSystemUtils.approximateContentHash((long)fs.getLen(), (long)fs.getModificationTime());
            return new UfsFileStatus(path, contentHash, fs.getLen(), Long.valueOf(fs.getModificationTime()), fs.getOwner(), fs.getGroup(), fs.getPermission().toShort(), fs.getBlockSize());
        }
        return new UfsDirectoryStatus(path, fs.getOwner(), fs.getGroup(), fs.getPermission().toShort(), Long.valueOf(fs.getModificationTime()));
    }

    public boolean isDirectory(String path) throws IOException {
        FileSystem hdfs = this.getFs();
        return hdfs.isDirectory(new Path(path));
    }

    public boolean isFile(String path) throws IOException {
        FileSystem hdfs = this.getFs();
        return hdfs.isFile(new Path(path));
    }

    @Nullable
    public Iterator<UfsStatus> listStatusIterable(String path, ListOptions options, String startAfter, int batchSize) throws IOException {
        FileSystem hdfs = this.getFs();
        return new HdfsUfsStatusIterator(path, hdfs);
    }

    @Nullable
    public UfsStatus[] listStatus(String path) throws IOException {
        FileStatus[] files = this.listStatusInternal(path);
        if (files == null) {
            return null;
        }
        UfsStatus[] rtn = new UfsStatus[files.length];
        int i = 0;
        for (FileStatus status : files) {
            UfsDirectoryStatus retStatus;
            if (!status.isDir()) {
                String contentHash = UnderFileSystemUtils.approximateContentHash((long)status.getLen(), (long)status.getModificationTime());
                retStatus = new UfsFileStatus(status.getPath().getName(), contentHash, status.getLen(), Long.valueOf(status.getModificationTime()), status.getOwner(), status.getGroup(), status.getPermission().toShort(), status.getBlockSize());
            } else {
                retStatus = new UfsDirectoryStatus(status.getPath().getName(), status.getOwner(), status.getGroup(), status.getPermission().toShort(), Long.valueOf(status.getModificationTime()));
            }
            rtn[i++] = retStatus;
        }
        return rtn;
    }

    public void connectFromMaster(String host) throws IOException {
        if (!this.mUfsConf.isSet(PropertyKey.MASTER_KEYTAB_KEY_FILE) || !this.mUfsConf.isSet(PropertyKey.MASTER_PRINCIPAL)) {
            return;
        }
        String masterKeytab = this.mUfsConf.getString(PropertyKey.MASTER_KEYTAB_KEY_FILE);
        String masterPrincipal = this.mUfsConf.getString(PropertyKey.MASTER_PRINCIPAL);
        this.login(PropertyKey.MASTER_KEYTAB_KEY_FILE, masterKeytab, PropertyKey.MASTER_PRINCIPAL, masterPrincipal, host);
    }

    public void connectFromWorker(String host) throws IOException {
        if (!this.mUfsConf.isSet(PropertyKey.WORKER_KEYTAB_FILE) || !this.mUfsConf.isSet(PropertyKey.WORKER_PRINCIPAL)) {
            return;
        }
        String workerKeytab = this.mUfsConf.getString(PropertyKey.WORKER_KEYTAB_FILE);
        String workerPrincipal = this.mUfsConf.getString(PropertyKey.WORKER_PRINCIPAL);
        this.login(PropertyKey.WORKER_KEYTAB_FILE, workerKeytab, PropertyKey.WORKER_PRINCIPAL, workerPrincipal, host);
    }

    private void login(PropertyKey keytabFileKey, String keytabFile, PropertyKey principalKey, String principal, String hostname) throws IOException {
        Configuration conf = new Configuration();
        conf.set(keytabFileKey.toString(), keytabFile);
        conf.set(principalKey.toString(), principal);
        SecurityUtil.login((Configuration)conf, (String)keytabFileKey.toString(), (String)principalKey.toString(), (String)hostname);
    }

    public boolean mkdirs(String path, MkdirsOptions options) throws IOException {
        IOException te = null;
        FileSystem hdfs = this.getFs();
        CountingRetry retryPolicy = new CountingRetry(5);
        while (retryPolicy.attempt()) {
            try {
                Path hdfsPath = new Path(path);
                if (hdfs.exists(hdfsPath)) {
                    LOG.debug("Trying to create existing directory at {}", (Object)path);
                    return false;
                }
                if (!hdfs.mkdirs(hdfsPath, new FsPermission(options.getMode().toShort()))) {
                    return false;
                }
                try {
                    this.setOwner(hdfsPath.toString(), options.getOwner(), options.getGroup());
                }
                catch (IOException e) {
                    LOG.warn("Failed to update the ufs dir ownership, default values will be used. " + e);
                }
                return true;
            }
            catch (IOException e) {
                LOG.debug("{} try to make directory for {} : {}", new Object[]{retryPolicy.getAttemptCount(), path, e.toString()});
                te = e;
            }
        }
        throw te;
    }

    private boolean isReadLocal(FileSystem fs, Path filePath, OpenOptions options) {
        String localHost = NetworkAddressUtils.getLocalHostName((int)((int)this.mUfsConf.getMs(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS)));
        try {
            BlockLocation[] blockLocations = fs.getFileBlockLocations(filePath, options.getOffset(), options.getLength());
            if (blockLocations == null) {
                return true;
            }
            for (BlockLocation loc : blockLocations) {
                if (!Arrays.stream(loc.getHosts()).noneMatch(localHost::equals)) continue;
                return false;
            }
        }
        catch (IOException e) {
            return true;
        }
        return true;
    }

    public InputStream open(String path, OpenOptions options) throws IOException {
        boolean remote;
        IOException te = null;
        FileSystem hdfs = this.getFs();
        CountingRetry retryPolicy = new CountingRetry(5);
        DistributedFileSystem dfs = null;
        if (hdfs instanceof DistributedFileSystem) {
            dfs = (DistributedFileSystem)hdfs;
        }
        Path filePath = new Path(path);
        boolean bl = remote = options.getPositionShort() || this.mUfsConf.getBoolean(PropertyKey.UNDERFS_HDFS_REMOTE) || !this.isReadLocal(hdfs, filePath, options);
        while (retryPolicy.attempt()) {
            try {
                FSDataInputStream inputStream = hdfs.open(filePath);
                if (remote) {
                    LOG.debug("Using pread API to HDFS");
                    return new HdfsPositionedUnderFileInputStream(inputStream, options.getOffset());
                }
                try {
                    inputStream.seek(options.getOffset());
                }
                catch (IOException e) {
                    inputStream.close();
                    throw e;
                }
                LOG.debug("Using original API to HDFS");
                return new HdfsUnderFileInputStream(inputStream);
            }
            catch (IOException e) {
                LOG.debug("{} try to open {} : {}", new Object[]{retryPolicy.getAttemptCount(), path, e.toString()});
                te = e;
                if (!options.getRecoverFailedOpen() || dfs == null || !e.getMessage().toLowerCase().startsWith("cannot obtain block length for")) continue;
                try {
                    if (dfs.recoverLease(new Path(path))) {
                        LOG.warn("HDFS recoverLease-1 success for: {}", (Object)path);
                        continue;
                    }
                    CommonUtils.sleepMs((long)5000L);
                    if (dfs.recoverLease(new Path(path))) {
                        LOG.warn("HDFS recoverLease-2 success for: {}", (Object)path);
                        continue;
                    }
                    LOG.warn("HDFS recoverLease: path not closed: {}", (Object)path);
                }
                catch (IOException e1) {
                    LOG.warn("HDFS recoverLease failed for: {} error: {}", (Object)path, (Object)e1.getMessage());
                }
            }
        }
        if (te != null) {
            LOG.error("{} failed attempts to open \"{}\" with last error:", new Object[]{retryPolicy.getAttemptCount(), path, te});
            throw te;
        }
        throw new IllegalStateException("Exceeded the number of retry attempts with no exception");
    }

    public HdfsPositionedUnderFileInputStream openPositionRead(String path, long fileLength) {
        try {
            FSDataInputStream inputStream = this.getFs().open(new Path(path));
            return new HdfsPositionedUnderFileInputStream(inputStream, 0L);
        }
        catch (IOException e) {
            throw AlluxioHdfsException.from((IOException)e);
        }
    }

    public boolean renameDirectory(String src, String dst) throws IOException {
        if (!this.isDirectory(src)) {
            LOG.warn("Unable to rename {} to {} because source does not exist or is a file", (Object)src, (Object)dst);
            return false;
        }
        return this.rename(src, dst);
    }

    public boolean renameFile(String src, String dst) throws IOException {
        if (!this.isFile(src)) {
            LOG.warn("Unable to rename {} to {} because source does not exist or is a directory", (Object)src, (Object)dst);
            return false;
        }
        return this.rename(src, dst);
    }

    public void setOwner(String path, String user, String group) throws IOException {
        if (user == null && group == null) {
            return;
        }
        FileSystem hdfs = this.getFs();
        try {
            FileStatus fileStatus = hdfs.getFileStatus(new Path(path));
            hdfs.setOwner(fileStatus.getPath(), user, group);
        }
        catch (IOException e) {
            LOG.debug("Exception: ", (Throwable)e);
            if (!this.mUfsConf.getBoolean(PropertyKey.UNDERFS_ALLOW_SET_OWNER_FAILURE)) {
                LOG.warn("Failed to set owner for {} with user: {}, group: {}: {}. Running Alluxio as superuser is required to modify ownership of local files", new Object[]{path, user, group, e.toString()});
                throw e;
            }
            LOG.warn("Failed to set owner for {} with user: {}, group: {}: {}. This failure is ignored but may cause permission inconsistency between Alluxio and local under file system", new Object[]{path, user, group, e.toString()});
        }
    }

    public void setAttribute(String path, String name, byte[] value) throws IOException {
        if (StringUtils.isEmpty((CharSequence)name)) {
            LOG.warn("Try to set Xattr to an empty file name for path {}", (Object)path);
            return;
        }
        FileSystem hdfs = this.getFs();
        try {
            FileStatus fileStatus = hdfs.getFileStatus(new Path(path));
            hdfs.setXAttr(fileStatus.getPath(), USER_NAMESPACE_PREFIX + name, value);
        }
        catch (IOException e) {
            LOG.warn("Failed to set XAttr for {} with name: {}, value: {}: {}. Running Alluxio as superuser is required to modify attributes of local files", new Object[]{path, name, new String(value), e.toString()});
            throw e;
        }
    }

    public Map<String, String> getAttributes(String path) throws IOException {
        FileSystem hdfs = this.getFs();
        try {
            Map attrMap = hdfs.getXAttrs(new Path(path));
            HashMap resMap = new HashMap(attrMap.size());
            attrMap.entrySet().stream().filter(entry -> ((String)entry.getKey()).startsWith(USER_NAMESPACE_PREFIX)).forEach(entry -> resMap.put(((String)entry.getKey()).substring(((String)entry.getKey()).indexOf(".") + 1), new String((byte[])entry.getValue())));
            return Collections.unmodifiableMap(resMap);
        }
        catch (IOException e) {
            LOG.warn("Failed to get XAttr for {}.", (Object)path);
            throw e;
        }
    }

    public void setMode(String path, short mode) throws IOException {
        FileSystem hdfs = this.getFs();
        try {
            FileStatus fileStatus = hdfs.getFileStatus(new Path(path));
            hdfs.setPermission(fileStatus.getPath(), new FsPermission(mode));
        }
        catch (IOException e) {
            LOG.warn("Fail to set permission for {} with perm {} : {}", new Object[]{path, mode, e.toString()});
            throw e;
        }
    }

    public boolean supportsFlush() {
        return true;
    }

    private boolean delete(String path, boolean recursive, boolean isDirectory) throws IOException {
        IOException te = null;
        FileSystem hdfs = this.getFs();
        CountingRetry retryPolicy = new CountingRetry(5);
        while (retryPolicy.attempt()) {
            try {
                Path hdfsPath = new Path(path);
                if (!this.mTrashEnable) {
                    return hdfs.delete(hdfsPath, recursive);
                }
                Trash trash = this.getTrash(hdfs);
                if (isDirectory && !recursive && hdfs.listStatus(hdfsPath).length != 0) {
                    return false;
                }
                if (trash.moveToTrash(hdfsPath)) {
                    return true;
                }
                LOG.debug("Failed to move '{}' to trash. Now delete it.", (Object)path);
                return hdfs.delete(hdfsPath, recursive);
            }
            catch (IOException e) {
                LOG.warn("Attempt count {} : {}", (Object)retryPolicy.getAttemptCount(), (Object)e.toString());
                te = e;
            }
        }
        throw te;
    }

    @Nullable
    private FileStatus[] listStatusInternal(String path) throws IOException {
        FileStatus[] files;
        FileSystem hdfs = this.getFs();
        Path thePath = new Path(path);
        try {
            files = hdfs.listStatus(thePath);
        }
        catch (FileNotFoundException e) {
            return null;
        }
        if (files != null && files.length == 1 && files[0].getPath().toUri().getPath().equals(thePath.toUri().getPath())) {
            return null;
        }
        return files;
    }

    private boolean rename(String src, String dst) throws IOException {
        IOException te = null;
        FileSystem hdfs = this.getFs();
        CountingRetry retryPolicy = new CountingRetry(5);
        while (retryPolicy.attempt()) {
            try {
                return hdfs.rename(new Path(src), new Path(dst));
            }
            catch (IOException e) {
                LOG.warn("{} try to rename {} to {} : {}", new Object[]{retryPolicy.getAttemptCount(), src, dst, e.toString()});
                te = e;
            }
        }
        throw te;
    }

    public boolean isSeekable() {
        return true;
    }

    @VisibleForTesting
    public FileSystem getFs() throws IOException {
        try {
            return (FileSystem)this.mUserFs.get((Object)HDFS_USER);
        }
        catch (ExecutionException e) {
            throw new IOException("Failed get FileSystem for " + this.mUri, e.getCause());
        }
    }

    private Trash getTrash(FileSystem fs) throws IOException {
        try {
            return (Trash)this.mFsTrash.get((Object)fs);
        }
        catch (ExecutionException e) {
            throw new IOException("Failed get Trash for " + fs.getUri(), e.getCause());
        }
    }
}

