/*
 * This file is part of Ammonium Commons, licensed under the permissive MIT License
 *
 * Copyright (c) 2020 Ammonium Studios
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package org.ammonium.common.util.gson;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.ammonium.common.util.gson.adapter.UuidTypeAdapter;
import org.ammonium.common.util.gson.adapter.ZonedDateTimeTypeAdapter;

import java.time.ZonedDateTime;
import java.util.UUID;

/**
 * Represents a class which holds {@link Gson} objects.
 */
public final class GsonProvider {

    /**
     * Creates a new {@link Gson} object without pretty printing
     */
    private static final Gson NORMAL = new GsonBuilder()
            .registerTypeAdapter(UUID.class, new UuidTypeAdapter())
            .registerTypeAdapter(ZonedDateTime.class, new ZonedDateTimeTypeAdapter())
            .create();

    /**
     * Creates a new {@link Gson} object with pretty printing
     */
    private static final Gson PRETTY_PRINTING = new GsonBuilder().setPrettyPrinting()
            .registerTypeAdapter(UUID.class, new UuidTypeAdapter())
            .registerTypeAdapter(ZonedDateTime.class, new ZonedDateTimeTypeAdapter())
            .create();

    /**
     * Represents a {@link Gson} object without pretty printing.
     *
     * @return {@link #NORMAL}
     * @see GsonProvider#NORMAL
     */
    public static Gson normal() {
        return GsonProvider.NORMAL;
    }

    /**
     * Represents a {@link Gson} object with pretty printing.
     *
     * @return {@link #PRETTY_PRINTING}
     * @see GsonProvider#PRETTY_PRINTING
     */
    public static Gson prettyPrinting() {
        return GsonProvider.PRETTY_PRINTING;
    }

    private GsonProvider() {
    }
}
