/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  The ASF licenses this file to You
 * under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.  For additional information regarding
 * copyright in this work, please see the NOTICE file in the top level
 * directory of this distribution.
 */
package org.apache.abdera.i18n.text.data;

import java.util.Arrays;

import org.apache.abdera.i18n.text.CharUtils;

public class UnicodeCharacterDatabase {

    public static final String UNICODE_VERSION = "5.0";

    // inversion set
    private static final int[] compat =
        {160, 161, 168, 169, 170, 171, 175, 176, 178, 182, 184, 187, 188, 191, 306, 308, 319, 321, 329, 330, 383, 384,
         452, 461, 497, 500, 688, 697, 728, 734, 736, 741, 890, 891, 900, 901, 976, 979, 981, 983, 1008, 1011, 1012,
         1014, 1017, 1018, 1415, 1416, 1653, 1657, 3635, 3636, 3763, 3764, 3804, 3806, 3852, 3853, 3959, 3960, 3961,
         3962, 4348, 4349, 7468, 7471, 7472, 7483, 7484, 7502, 7503, 7531, 7544, 7545, 7579, 7616, 7834, 7835, 8125,
         8126, 8127, 8129, 8190, 8191, 8194, 8203, 8209, 8210, 8215, 8216, 8228, 8231, 8239, 8240, 8243, 8245, 8246,
         8248, 8252, 8253, 8254, 8255, 8263, 8266, 8279, 8280, 8287, 8288, 8304, 8306, 8308, 8335, 8336, 8341, 8360,
         8361, 8448, 8452, 8453, 8456, 8457, 8468, 8469, 8471, 8473, 8478, 8480, 8483, 8484, 8485, 8488, 8489, 8492,
         8494, 8495, 8498, 8499, 8506, 8507, 8513, 8517, 8522, 8531, 8576, 8748, 8750, 8751, 8753, 9312, 9451, 10764,
         10765, 10868, 10871, 11631, 11632, 11935, 11936, 12019, 12020, 12032, 12246, 12288, 12289, 12342, 12343,
         12344, 12347, 12443, 12445, 12447, 12448, 12543, 12544, 12593, 12687, 12690, 12704, 12800, 12831, 12832,
         12868, 12880, 12927, 12928, 13055, 13056, 13312, 64256, 64263, 64275, 64280, 64288, 64298, 64335, 64434,
         64467, 64830, 64848, 64912, 64914, 65021, 65040, 65050, 65072, 65093, 65095, 65107, 65108, 65127, 65128,
         65132, 65136, 65139, 65140, 65141, 65142, 65277, 65281, 65471, 65474, 65480, 65482, 65488, 65490, 65496,
         65498, 65501, 65504, 65511, 65512, 65519, 119808, 119893, 119894, 119965, 119966, 119968, 119970, 119971,
         119973, 119975, 119977, 119981, 119982, 119994, 119995, 119996, 119997, 120004, 120005, 120069, 120071,
         120075, 120077, 120085, 120086, 120093, 120094, 120122, 120123, 120127, 120128, 120133, 120134, 120135,
         120138, 120145, 120146, 120486, 120488, 120780, 120782, 120832};

    private static final int[] decomp_idx = getDecompIdx();
    private static final int[][] decomp_data = getDecompData();
    private static final int[] comp_idx = getComposeIdx();
    private static final int[] comp_data = getComposeData();
    private static final int[] hangul_pairs = getHangulPairs();
    private static final int[] hangul_codepoints = getHangulCodepoints();

    // inversion sets
    private static enum CC {
        _1(new int[] {820, 825, 8402, 8404, 8408, 8411, 8421, 8423, 8426, 8428, 68153, 68154, 119143, 119146}), _7(
            new int[] {2364, 2365, 2492, 2493, 2620, 2621, 2748, 2749, 2876, 2877, 3260, 3261, 4151, 4152, 6964, 6965}), _8(
            new int[] {12441, 12443}), _9(new int[] {2381, 2382, 2509, 2510, 2637, 2638, 2765, 2766, 2893, 2894, 3021,
                                                     3022, 3149, 3150, 3277, 3278, 3405, 3406, 3530, 3531, 3642, 3643,
                                                     3972, 3973, 4153, 4154, 5908, 5909, 5940, 5941, 6098, 6099, 6980,
                                                     6981, 43014, 43015, 68159, 68160}), _10(new int[] {1456, 1457}), _11(
            new int[] {1457, 1458}), _12(new int[] {1458, 1459}), _13(new int[] {1459, 1460}), _14(new int[] {1460,
                                                                                                              1461}), _15(
            new int[] {1461, 1462}), _16(new int[] {1462, 1463}), _17(new int[] {1463, 1464}), _18(new int[] {1464,
                                                                                                              1465,
                                                                                                              1479,
                                                                                                              1480}), _19(
            new int[] {1465, 1467}), _20(new int[] {1467, 1468}), _21(new int[] {1468, 1469}), _22(new int[] {1469,
                                                                                                              1470}), _23(
            new int[] {1471, 1472}), _24(new int[] {1473, 1474}), _25(new int[] {1474, 1475}), _26(new int[] {64286,
                                                                                                              64287}), _27(
            new int[] {1611, 1612}), _28(new int[] {1612, 1613}), _29(new int[] {1613, 1614}), _31(new int[] {1615,
                                                                                                              1616}), _32(
            new int[] {1616, 1617}), _33(new int[] {1617, 1618}), _34(new int[] {1618, 1619}), _35(new int[] {1648,
                                                                                                              1649}), _36(
            new int[] {1809, 1810}), _30(new int[] {1614, 1615}), _84(new int[] {3157, 3158}), _91(new int[] {3158,
                                                                                                              3159}), _103(
            new int[] {3640, 3642}), _107(new int[] {3656, 3660}), _118(new int[] {3768, 3770}), _122(new int[] {3784,
                                                                                                                 3788}), _129(
            new int[] {3953, 3954}), _130(new int[] {3954, 3955, 3962, 3965, 3966, 3967, 3968, 3969}), _132(
            new int[] {3956, 3957}), _202(new int[] {801, 803, 807, 809}), _216(new int[] {795, 796, 3897, 3898,
                                                                                           119141, 119143, 119150,
                                                                                           119155}), _218(
            new int[] {12330, 12331}), _220(new int[] {790, 794, 796, 801, 803, 807, 809, 820, 825, 829, 839, 842, 845,
                                                       847, 851, 855, 857, 859, 1425, 1426, 1430, 1431, 1435, 1436,
                                                       1442, 1448, 1450, 1451, 1477, 1478, 1621, 1623, 1628, 1629,
                                                       1763, 1764, 1770, 1771, 1773, 1774, 1841, 1842, 1844, 1845,
                                                       1847, 1850, 1851, 1853, 1854, 1855, 1858, 1859, 1860, 1861,
                                                       1862, 1863, 1864, 1865, 2034, 2035, 2386, 2387, 3864, 3866,
                                                       3893, 3894, 3895, 3896, 4038, 4039, 6459, 6460, 6680, 6681,
                                                       7020, 7021, 7618, 7619, 7626, 7627, 7679, 7680, 8424, 8425,
                                                       8428, 8432, 68109, 68110, 68154, 68155, 119163, 119171, 119178,
                                                       119180}), _222(new int[] {1434, 1435, 1453, 1454, 6457, 6458,
                                                                                 12333, 12334}), _224(new int[] {12334,
                                                                                                                 12336}), _226(
            new int[] {119149, 119150}), _228(new int[] {1454, 1455, 6313, 6314, 12331, 12332}), _230(
            new int[] {768, 789, 829, 837, 838, 839, 842, 845, 848, 851, 855, 856, 859, 860, 867, 880, 1155, 1159,
                       1426, 1430, 1431, 1434, 1436, 1442, 1448, 1450, 1451, 1453, 1455, 1456, 1476, 1477, 1552, 1558,
                       1619, 1621, 1623, 1628, 1629, 1631, 1750, 1757, 1759, 1763, 1764, 1765, 1767, 1769, 1771, 1773,
                       1840, 1841, 1842, 1844, 1845, 1847, 1850, 1851, 1853, 1854, 1855, 1858, 1859, 1860, 1861, 1862,
                       1863, 1864, 1865, 1867, 2027, 2034, 2035, 2036, 2385, 2386, 2387, 2389, 3970, 3972, 3974, 3976,
                       4959, 4960, 6109, 6110, 6458, 6459, 6679, 6680, 7019, 7020, 7021, 7028, 7616, 7618, 7619, 7626,
                       7678, 7679, 8400, 8402, 8404, 8408, 8411, 8413, 8417, 8418, 8423, 8424, 8425, 8426, 65056,
                       65060, 68111, 68112, 68152, 68153, 119173, 119178, 119210, 119214, 119362, 119365}), _232(
            new int[] {789, 790, 794, 795, 856, 857, 12332, 12333}), _233(new int[] {860, 861, 863, 864, 866, 867}), _234(
            new int[] {861, 863, 864, 866}), _240(new int[] {837, 838});
        private final int[] set;
        private final int cc;

        CC(int[] set) {
            this.set = set;
            this.cc = Integer.parseInt(this.name().substring(1));
        }

        static int lookup(int c) {
            for (CC cc : CC.values()) {
                if (CharUtils.invset_contains(cc.set, c)) {
                    return cc.cc;
                }
            }
            return 0;
        }
    }

    private static int[] getComposeIdx() {
        return new int[] {59, 75, 96, 180, 183, 197, 290, 697, 768, 769, 787, 901, 902, 904, 905, 906, 908, 910, 911,
                          912, 937, 940, 941, 942, 943, 944, 953, 972, 973, 974, 1308, 1317, 1355, 1594, 2052, 2270,
                          2604, 2915, 5348, 5800, 5866, 6600, 6936, 7435, 7652, 7654, 8194, 8195, 8579, 8607, 9009,
                          9009, 9940, 10308, 10314, 11020, 11249, 12296, 12297, 12298, 12984, 13151, 13203, 13212,
                          13251, 13269, 13421, 13470, 13497, 13499, 13535, 13589, 13987, 14062, 14076, 14209, 14383,
                          14434, 14460, 14503, 14535, 14563, 14620, 14650, 14894, 14956, 14989, 15076, 15098, 15112,
                          15129, 15177, 15261, 15261, 15384, 15438, 15548, 15646, 15667, 15766, 16044, 16056, 16056,
                          16081, 16155, 16222, 16270, 16380, 16392, 16408, 16441, 16441, 16454, 16534, 16611, 16687,
                          16898, 16935, 16995, 17056, 17153, 17204, 17241, 17323, 17365, 17369, 17419, 17515, 17707,
                          17757, 17761, 17771, 17879, 17913, 17928, 17973, 18110, 18119, 18229, 18452, 18837, 18918,
                          19054, 19062, 19122, 19251, 19406, 19510, 19602, 19662, 19693, 19704, 19798, 19981, 20006,
                          20018, 20024, 20025, 20029, 20033, 20098, 20102, 20142, 20160, 20172, 20196, 20320, 20352,
                          20358, 20363, 20385, 20398, 20398, 20408, 20411, 20415, 20482, 20523, 20548, 20602, 20633,
                          20687, 20698, 20711, 20711, 20722, 20723, 20761, 20787, 20800, 20805, 20813, 20813, 20820,
                          20836, 20839, 20840, 20841, 20845, 20855, 20864, 20877, 20882, 20885, 20887, 20900, 20908,
                          20917, 20917, 20919, 20937, 20940, 20956, 20958, 20981, 20995, 20999, 20999, 21015, 21033,
                          21050, 21051, 21062, 21065, 21106, 21111, 21129, 21147, 21155, 21171, 21191, 21191, 21193,
                          21193, 21202, 21214, 21220, 21220, 21237, 21242, 21242, 21253, 21254, 21271, 21271, 21311,
                          21321, 21329, 21329, 21338, 21363, 21365, 21373, 21375, 21375, 21375, 21443, 21450, 21471,
                          21477, 21483, 21489, 21510, 21519, 21533, 21533, 21560, 21570, 21576, 21608, 21662, 21666,
                          21693, 21750, 21776, 21843, 21845, 21859, 21892, 21892, 21895, 21913, 21913, 21917, 21917,
                          21931, 21939, 21952, 21954, 21986, 22022, 22022, 22054, 22097, 22120, 22132, 22170, 22213,
                          22265, 22294, 22295, 22411, 22478, 22516, 22541, 22577, 22578, 22592, 22618, 22618, 22622,
                          22696, 22700, 22707, 22744, 22751, 22766, 22770, 22775, 22790, 22810, 22818, 22852, 22856,
                          22865, 22868, 22882, 22899, 22908, 23000, 23020, 23067, 23079, 23138, 23142, 23207, 23207,
                          23221, 23304, 23336, 23358, 23358, 23429, 23467, 23491, 23512, 23527, 23527, 23527, 23534,
                          23539, 23551, 23558, 23586, 23615, 23648, 23650, 23652, 23653, 23662, 23662, 23680, 23693,
                          23744, 23760, 23833, 23875, 23888, 23915, 23918, 23932, 23986, 23994, 24033, 24034, 24061,
                          24104, 24125, 24169, 24180, 24230, 24240, 24243, 24246, 24265, 24266, 24266, 24274, 24275,
                          24281, 24300, 24318, 24324, 24354, 24354, 24403, 24418, 24425, 24427, 24454, 24459, 24474,
                          24489, 24493, 24525, 24535, 24565, 24569, 24594, 24604, 24705, 24724, 24724, 24775, 24792,
                          24801, 24840, 24900, 24904, 24908, 24908, 24910, 24910, 24928, 24936, 24954, 24974, 24974,
                          24974, 24976, 24996, 25007, 25010, 25050, 25054, 25074, 25074, 25074, 25078, 25078, 25088,
                          25104, 25115, 25128, 25134, 25140, 25159, 25181, 25265, 25289, 25295, 25299, 25300, 25305,
                          25340, 25342, 25405, 25406, 25424, 25448, 25467, 25475, 25504, 25513, 25540, 25541, 25572,
                          25628, 25634, 25682, 25705, 25719, 25726, 25754, 25757, 25796, 25818, 25891, 25935, 25935,
                          25942, 25964, 25976, 26009, 26024, 26053, 26082, 26083, 26131, 26185, 26228, 26228, 26248,
                          26257, 26257, 26268, 26292, 26310, 26356, 26360, 26368, 26391, 26391, 26391, 26395, 26395,
                          26401, 26446, 26451, 26454, 26462, 26491, 26501, 26519, 26535, 26549, 26611, 26618, 26647,
                          26655, 26706, 26753, 26757, 26757, 26766, 26792, 26900, 26946, 27043, 27114, 27138, 27138,
                          27138, 27155, 27304, 27347, 27355, 27396, 27425, 27452, 27476, 27506, 27511, 27513, 27551,
                          27566, 27578, 27578, 27578, 27579, 27702, 27726, 27751, 27784, 27839, 27852, 27853, 27861,
                          27877, 27926, 27931, 27934, 27956, 27966, 27969, 27969, 27969, 28009, 28010, 28011, 28023,
                          28023, 28024, 28037, 28107, 28122, 28138, 28153, 28186, 28207, 28270, 28316, 28346, 28359,
                          28363, 28363, 28369, 28379, 28431, 28450, 28450, 28451, 28460, 28526, 28593, 28614, 28651,
                          28670, 28699, 28702, 28702, 28729, 28746, 28784, 28791, 28797, 28825, 28845, 28872, 28882,
                          28889, 28997, 29001, 29038, 29038, 29084, 29134, 29136, 29200, 29211, 29224, 29227, 29237,
                          29237, 29264, 29282, 29312, 29333, 29359, 29376, 29436, 29482, 29482, 29557, 29562, 29575,
                          29575, 29579, 29605, 29618, 29642, 29662, 29702, 29705, 29730, 29767, 29788, 29801, 29809,
                          29809, 29829, 29833, 29848, 29898, 29958, 29988, 30011, 30014, 30041, 30053, 30064, 30064,
                          30178, 30224, 30237, 30239, 30274, 30311, 30313, 30410, 30410, 30427, 30439, 30452, 30452,
                          30465, 30494, 30495, 30495, 30528, 30538, 30538, 30603, 30631, 30798, 30827, 30860, 30860,
                          30865, 30894, 30922, 30924, 30924, 30971, 31018, 31036, 31038, 31048, 31049, 31056, 31062,
                          31062, 31069, 31070, 31077, 31078, 31103, 31117, 31118, 31119, 31119, 31150, 31178, 31211,
                          31260, 31296, 31296, 31306, 31311, 31361, 31409, 31435, 31470, 31520, 31680, 31680, 31686,
                          31689, 31806, 31840, 31867, 31890, 31912, 31934, 31954, 31958, 31971, 31975, 31976, 32000,
                          32016, 32034, 32047, 32091, 32099, 32160, 32190, 32199, 32244, 32244, 32244, 32258, 32265,
                          32311, 32321, 32325, 32467, 32559, 32574, 32626, 32633, 32634, 32645, 32661, 32666, 32701,
                          32762, 32769, 32773, 32773, 32773, 32838, 32864, 32879, 32880, 32894, 32907, 32941, 32946,
                          33027, 33086, 33240, 33256, 33261, 33281, 33281, 33284, 33391, 33401, 33401, 33419, 33425,
                          33437, 33457, 33459, 33469, 33509, 33509, 33510, 33565, 33571, 33590, 33618, 33619, 33635,
                          33709, 33725, 33737, 33738, 33740, 33756, 33767, 33775, 33777, 33853, 33865, 33879, 33879,
                          34030, 34033, 34035, 34044, 34070, 34148, 34253, 34258, 34285, 34298, 34310, 34322, 34349,
                          34367, 34384, 34396, 34396, 34407, 34409, 34440, 34473, 34530, 34574, 34600, 34606, 34667,
                          34681, 34681, 34694, 34746, 34785, 34817, 34847, 34892, 34912, 34915, 35010, 35023, 35031,
                          35038, 35041, 35064, 35066, 35088, 35137, 35172, 35206, 35211, 35222, 35222, 35488, 35498,
                          35498, 35519, 35531, 35538, 35542, 35565, 35565, 35576, 35576, 35582, 35582, 35585, 35585,
                          35641, 35641, 35672, 35712, 35722, 35722, 35834, 35912, 35925, 36011, 36033, 36034, 36040,
                          36051, 36104, 36104, 36123, 36215, 36215, 36284, 36299, 36335, 36336, 36554, 36564, 36646,
                          36650, 36664, 36664, 36667, 36706, 36766, 36784, 36790, 36899, 36920, 36920, 36978, 36988,
                          37007, 37012, 37070, 37105, 37117, 37137, 37147, 37189, 37226, 37273, 37300, 37324, 37327,
                          37329, 37343, 37402, 37428, 37432, 37494, 37500, 37591, 37592, 37636, 37706, 37881, 37898,
                          37909, 38038, 38283, 38317, 38326, 38327, 38446, 38475, 38477, 38517, 38520, 38524, 38534,
                          38563, 38584, 38595, 38626, 38627, 38627, 38646, 38647, 38691, 38706, 38728, 38742, 38742,
                          38875, 38880, 38911, 38911, 38923, 38923, 38923, 38936, 38953, 38971, 38971, 39006, 39138,
                          39151, 39164, 39208, 39209, 39335, 39362, 39409, 39422, 39530, 39698, 39698, 39728, 39791,
                          40000, 40023, 40189, 40295, 40372, 40442, 40478, 40575, 40599, 40607, 40635, 40654, 40697,
                          40702, 40709, 40719, 40726, 40763, 40771, 40845, 40846, 40860, 40860, 40860, 41166, 41221,
                          41486, 41617, 41874, 42496, 3932984, 3998520, 4064056, 4260608, 4260609, 4260610, 4260611,
                          4260612, 4260614, 4260615, 4260616, 4260617, 4260618, 4260620, 4260623, 4260625, 4260643,
                          4260645, 4260648, 4326151, 4326179, 4326193, 4391681, 4391682, 4391687, 4391692, 4391719,
                          4457223, 4457228, 4457251, 4457255, 4457261, 4457265, 4522752, 4522753, 4522754, 4522755,
                          4522756, 4522758, 4522759, 4522760, 4522761, 4522764, 4522767, 4522769, 4522787, 4522791,
                          4522792, 4522797, 4522800, 4588295, 4653825, 4653826, 4653828, 4653830, 4653831, 4653836,
                          4653863, 4719362, 4719367, 4719368, 4719372, 4719395, 4719399, 4719406, 4784896, 4784897,
                          4784898, 4784899, 4784900, 4784902, 4784903, 4784904, 4784905, 4784908, 4784911, 4784913,
                          4784931, 4784936, 4784944, 4850434, 4915969, 4915980, 4916003, 4916007, 4916017, 4981505,
                          4981516, 4981539, 4981543, 4981549, 4981553, 5047041, 5047047, 5047075, 5112576, 5112577,
                          5112579, 5112583, 5112588, 5112611, 5112615, 5112621, 5112625, 5178112, 5178113, 5178114,
                          5178115, 5178116, 5178118, 5178119, 5178120, 5178121, 5178123, 5178124, 5178127, 5178129,
                          5178139, 5178147, 5178152, 5243649, 5243655, 5374721, 5374727, 5374732, 5374735, 5374737,
                          5374755, 5374759, 5374769, 5440257, 5440258, 5440263, 5440268, 5440291, 5440294, 5440295,
                          5505799, 5505804, 5505827, 5505830, 5505831, 5505837, 5505841, 5571328, 5571329, 5571330,
                          5571331, 5571332, 5571334, 5571336, 5571337, 5571338, 5571339, 5571340, 5571343, 5571345,
                          5571355, 5571363, 5571364, 5571368, 5571373, 5571376, 5636867, 5636899, 5702400, 5702401,
                          5702402, 5702407, 5702408, 5702435, 5767943, 5767944, 5833472, 5833473, 5833474, 5833475,
                          5833476, 5833479, 5833480, 5833481, 5833507, 5899009, 5899010, 5899015, 5899020, 5899043,
                          5899057, 6357760, 6357761, 6357762, 6357763, 6357764, 6357766, 6357767, 6357768, 6357769,
                          6357770, 6357772, 6357775, 6357777, 6357795, 6357797, 6357800, 6423303, 6423331, 6423345,
                          6488833, 6488834, 6488839, 6488844, 6488871, 6554375, 6554380, 6554403, 6554407, 6554413,
                          6554417, 6619904, 6619905, 6619906, 6619907, 6619908, 6619910, 6619911, 6619912, 6619913,
                          6619916, 6619919, 6619921, 6619939, 6619943, 6619944, 6619949, 6619952, 6685447, 6750977,
                          6750978, 6750980, 6750982, 6750983, 6750988, 6751015, 6816514, 6816519, 6816520, 6816524,
                          6816547, 6816551, 6816558, 6816561, 6882048, 6882049, 6882050, 6882051, 6882052, 6882054,
                          6882056, 6882057, 6882060, 6882063, 6882065, 6882083, 6882088, 6882096, 6947586, 6947596,
                          7013121, 7013132, 7013155, 7013159, 7013169, 7078657, 7078668, 7078691, 7078695, 7078701,
                          7078705, 7144193, 7144199, 7144227, 7209728, 7209729, 7209731, 7209735, 7209740, 7209763,
                          7209767, 7209773, 7209777, 7275264, 7275265, 7275266, 7275267, 7275268, 7275270, 7275271,
                          7275272, 7275273, 7275275, 7275276, 7275279, 7275281, 7275291, 7275299, 7275304, 7340801,
                          7340807, 7471873, 7471879, 7471884, 7471887, 7471889, 7471907, 7471911, 7471921, 7537409,
                          7537410, 7537415, 7537420, 7537443, 7537446, 7537447, 7602951, 7602952, 7602956, 7602979,
                          7602982, 7602983, 7602989, 7602993, 7668480, 7668481, 7668482, 7668483, 7668484, 7668486,
                          7668488, 7668489, 7668490, 7668491, 7668492, 7668495, 7668497, 7668507, 7668515, 7668516,
                          7668520, 7668525, 7668528, 7734019, 7734051, 7799552, 7799553, 7799554, 7799559, 7799560,
                          7799562, 7799587, 7865095, 7865096, 7930624, 7930625, 7930626, 7930627, 7930628, 7930631,
                          7930632, 7930633, 7930634, 7930659, 7996161, 7996162, 7996167, 7996172, 7996195, 7996209,
                          11010816, 11010817, 11010882, 12714752, 12714753, 12714755, 12714761, 12845828, 12911361,
                          12976897, 12976900, 13042433, 13239040, 13239041, 13239043, 13239049, 13566721, 13894400,
                          13894401, 13894403, 13894409, 13959937, 13959940, 13959944, 14025476, 14156545, 14418688,
                          14418689, 14418692, 14418700, 14811904, 14811905, 14811907, 14811913, 14942980, 15008513,
                          15074049, 15074052, 15139585, 15336192, 15336193, 15336195, 15336201, 15663873, 15991552,
                          15991553, 15991555, 15991561, 16057089, 16057092, 16057096, 16122628, 16253697, 16515840,
                          16515841, 16515844, 16515852, 16909056, 16909057, 16909059, 16909065, 16974592, 16974593,
                          16974595, 16974601, 17957632, 17957633, 18023168, 18023169, 21758720, 21758721, 21824256,
                          21824257, 22676231, 22741767, 23069447, 23134983, 23593729, 23659265, 23724808, 23790344,
                          25101063, 27263744, 27263745, 27263747, 27263753, 27263779, 27329280, 27329281, 27329283,
                          27329289, 27329315, 28246784, 28246785, 28246787, 28246793, 28246819, 28312320, 28312321,
                          28312323, 28312329, 28312355, 28771084, 32113412, 32178948, 36045572, 36111108, 36176646,
                          36242182, 36569860, 36635396, 43123468, 50856705, 59835136, 59835137, 59835140, 59835142,
                          59835155, 59835156, 59835205, 60097280, 60097281, 60097299, 60097300, 60228352, 60228353,
                          60228371, 60228372, 60228421, 60359424, 60359425, 60359428, 60359430, 60359432, 60359443,
                          60359444, 60752640, 60752641, 60752659, 60752660, 60883732, 61145856, 61145857, 61145860,
                          61145862, 61145864, 61145876, 61408000, 61408001, 61408019, 61408020, 61408069, 61604677,
                          61735749, 61932288, 61932289, 61932292, 61932294, 61932307, 61932308, 61932354, 61932357,
                          62194432, 62194433, 62194451, 62194452, 62325504, 62325505, 62325523, 62325524, 62325570,
                          62325573, 62456576, 62456577, 62456580, 62456582, 62456584, 62456595, 62456596, 62456642,
                          62849792, 62849793, 62849811, 62849812, 62980883, 62980884, 63243008, 63243009, 63243012,
                          63243014, 63243016, 63243027, 63243028, 63243074, 63505152, 63505153, 63505171, 63505172,
                          63505218, 63505221, 63570688, 63570689, 63570754, 63636224, 63636225, 63636290, 63832901,
                          64094977, 64094984, 67502856, 68158214, 68158216, 68354817, 68485888, 68485894, 68485896,
                          68551430, 68551432, 68616968, 68682496, 68682500, 68682502, 68682504, 68813569, 69075720,
                          69403396, 69403398, 69403400, 69403403, 69665544, 69927688, 70058760, 70255366, 70255368,
                          70451969, 70583040, 70583046, 70583048, 70648582, 70648584, 70714120, 70779648, 70779652,
                          70779654, 70779656, 70910721, 71172872, 71500548, 71500550, 71500552, 71500555, 71762696,
                          72024840, 72155912, 72745736, 74711823, 74777359, 81265416, 81330952, 82313992, 82379528,
                          103220819, 103220820, 103220821, 105383508, 105514580, 113313364, 114427476, 114624084,
                          153618748, 154143036, 154339644, 164039102, 164039127, 189205310, 189205334, 189205335,
                          194120663, 197528510, 197528535, 197594046, 205917270, 213847253, 214305986, 214306005,
                          214306006, 214568149, 222694718, 222694743, 222760254, 232328650, 232328655, 232328671,
                          232525258, 259067762, 259067764, 259067776, 270864430, 453319477, 453450549, 453581621,
                          453712693, 453843765, 454105909, 456792885, 456923957, 457055029, 457120565, 457317173,
                          506856196, 506921732, 509215492, 509281028, 509739783, 509805319, 513803010, 513803014,
                          513868546, 513868550, 515375874, 515441410, 516686594, 516752130, 520094464, 520094465,
                          520094530, 520094533, 520160000, 520160001, 520160066, 520160069, 520225605, 520291141,
                          520356677, 520422213, 520487749, 520553285, 520618752, 520618753, 520618818, 520618821,
                          520684288, 520684289, 520684354, 520684357, 520749893, 520815429, 520880965, 520946501,
                          521012037, 521077573, 521143040, 521143041, 521208576, 521208577, 521667328, 521667329,
                          521732864, 521732865, 522191616, 522191617, 522191682, 522191685, 522257152, 522257153,
                          522257218, 522257221, 522322757, 522388293, 522453829, 522519365, 522584901, 522650437,
                          522715904, 522715905, 522715970, 522715973, 522781440, 522781441, 522781506, 522781509,
                          522847045, 522912581, 522978117, 523043653, 523109189, 523174725, 523240192, 523240193,
                          523240258, 523305728, 523305729, 523305794, 523764480, 523764481, 523764546, 523830016,
                          523830017, 523830082, 524288768, 524288769, 524354304, 524354305, 524813056, 524813057,
                          524878592, 524878593, 525337344, 525337345, 525337410, 525402880, 525402881, 525402946,
                          525927168, 525927169, 525927234, 526385920, 526385921, 526385986, 526385989, 526451456,
                          526451457, 526451522, 526451525, 526517061, 526582597, 526648133, 526713669, 526779205,
                          526844741, 526910208, 526910209, 526910274, 526910277, 526975744, 526975745, 526975810,
                          526975813, 527041349, 527106885, 527172421, 527237957, 527303493, 527369029, 527434565,
                          527696709, 528220997, 532022085, 532611840, 532611841, 532611906, 533070661, 536216389,
                          536740608, 536740609, 536740674, 563086136, 563217208, 563348280, 567280440, 567411512,
                          567542584, 570622776, 570950456, 571147064, 572719928, 572851000, 574358328, 574817080,
                          574948152, 575144760, 575472440, 576783160, 576979768, 577045304, 577897272, 577962808,
                          578159416, 578224952, 578421560, 578487096, 578552632, 578618168, 578945848, 579011384,
                          579207992, 579273528, 579928888, 579994424, 581043000, 581436216, 581501752, 581632824,
                          582091576, 582157112, 582222648, 582288184, 809906329, 810234009, 810365081, 810496153,
                          810627225, 810758297, 810889369, 811020441, 811151513, 811282585, 811413657, 811544729,
                          811675801, 811872409, 812003481, 812134553, 812593305, 812593306, 812789913, 812789914,
                          812986521, 812986522, 813183129, 813183130, 813379737, 813379738, 815607961, 816197785,
                          816525465, 816656537, 816787609, 816918681, 817049753, 817180825, 817311897, 817442969,
                          817574041, 817705113, 817836185, 817967257, 818163865, 818294937, 818426009, 818884761,
                          818884762, 819081369, 819081370, 819277977, 819277978, 819474585, 819474586, 819671193,
                          819671194, 820981913, 821047449, 821112985, 821178521, 821899417};
    }

    private static int[] getComposeData() {
        return new int[] {894, 8490, 8175, 8189, 903, 8491, 194563, 884, 832, 833, 835, 8174, 8123, 8137, 8139, 8155,
                          8185, 8171, 8187, 8147, 8486, 8049, 8051, 8053, 8055, 8163, 8126, 8057, 8059, 8061, 194578,
                          194843, 194582, 194573, 195033, 195037, 194612, 194616, 194649, 194656, 194657, 194668,
                          194673, 194808, 194683, 194685, 8192, 8193, 194697, 194873, 194705, 194706, 194724, 64208,
                          64207, 194744, 194750, 9001, 9002, 194762, 194711, 194944, 194953, 194954, 194781, 64209,
                          194787, 194572, 194579, 195018, 194591, 194596, 194796, 194663, 194664, 194678, 194691,
                          194696, 194698, 194800, 194710, 194715, 194722, 194721, 194754, 194759, 194807, 194769,
                          194809, 194768, 194766, 194782, 64210, 194791, 194798, 194802, 194811, 194822, 194826,
                          194838, 194858, 194860, 194861, 194829, 194867, 194832, 194833, 194878, 194879, 64211, 64212,
                          194889, 194891, 194892, 194897, 194904, 194912, 194916, 194845, 194919, 194925, 194929,
                          194932, 194847, 194945, 194775, 194948, 194958, 194983, 194990, 194991, 194994, 195007,
                          195010, 194851, 195016, 195021, 195022, 194854, 194855, 195055, 195058, 195064, 195065,
                          195068, 195075, 195080, 194869, 194871, 195085, 195086, 195089, 195094, 63847, 64112, 63749,
                          194561, 63838, 194560, 194562, 63771, 63930, 63863, 63997, 194585, 63912, 194564, 64115,
                          63789, 63925, 194875, 64048, 194565, 194876, 194566, 63845, 194567, 63956, 194877, 194568,
                          194569, 194571, 63931, 64049, 194570, 194882, 194881, 194883, 194884, 64012, 64116, 64050,
                          194574, 194575, 194576, 194580, 64114, 63864, 63953, 194577, 64117, 194581, 194770, 194771,
                          194583, 194584, 194586, 64113, 194587, 63790, 63865, 63829, 63828, 64021, 194589, 194590,
                          64000, 194640, 63900, 63965, 63999, 194592, 194593, 64213, 194594, 194595, 63943, 63882,
                          63901, 194962, 64118, 194597, 64051, 194598, 63826, 63791, 64052, 194599, 63871, 64119,
                          194600, 194601, 194602, 63843, 194603, 63979, 194604, 64053, 194605, 194606, 194607, 63772,
                          194608, 194609, 194610, 194611, 63851, 194614, 194615, 63750, 194617, 194618, 194619, 63966,
                          63981, 194893, 194621, 63872, 194622, 194623, 194620, 194624, 63902, 194625, 194626, 194627,
                          64121, 194628, 194629, 194630, 63755, 64122, 194631, 64054, 64120, 194632, 194633, 64013,
                          194634, 64123, 64055, 194636, 194898, 194638, 64056, 194639, 194900, 194901, 63913, 194635,
                          194637, 194645, 194642, 194643, 194644, 194647, 194646, 64057, 64016, 64124, 63852, 64058,
                          194648, 64125, 63818, 63810, 194641, 194650, 194651, 194652, 194653, 194654, 64126, 63756,
                          63753, 64127, 194655, 63873, 194908, 194661, 194658, 194659, 194660, 64128, 194662, 194909,
                          194910, 194950, 194665, 64129, 194666, 194667, 64004, 194913, 194669, 194670, 63839, 63914,
                          194671, 63932, 194672, 194674, 194675, 194677, 63933, 194679, 63819, 64059, 63967, 64060,
                          194680, 194917, 194682, 194681, 64214, 63957, 194684, 63777, 194687, 194686, 194688, 195060,
                          63915, 194689, 194690, 194692, 194693, 194694, 194695, 63886, 64001, 194699, 194700, 194701,
                          63906, 63784, 194702, 64130, 64011, 64131, 63874, 194704, 63811, 194708, 194709, 194676,
                          194713, 64132, 194714, 194923, 63960, 194716, 63846, 64133, 194717, 194718, 63907, 194719,
                          63840, 63916, 194720, 64061, 194723, 194725, 64134, 63929, 64136, 63961, 194726, 194727,
                          194729, 64135, 194728, 64138, 64062, 194730, 64063, 64137, 194731, 63887, 194733, 194734,
                          194732, 194712, 194735, 64064, 64139, 194736, 63757, 194737, 63888, 194738, 194739, 194930,
                          63954, 64140, 194931, 194740, 194741, 63781, 63835, 64002, 194742, 194933, 194746, 63859,
                          194745, 194935, 194743, 194747, 63908, 194748, 63861, 194753, 64141, 194752, 194749, 64142,
                          194751, 64143, 194755, 194758, 194756, 63889, 194757, 63792, 194939, 194940, 64065, 194760,
                          64144, 194761, 63849, 63934, 194942, 63875, 64066, 194763, 63968, 194765, 64018, 64145,
                          63941, 64067, 194767, 194773, 64006, 63883, 63745, 194764, 194772, 63785, 64146, 194776,
                          64147, 194777, 194778, 63969, 194780, 64148, 194779, 63944, 194784, 63988, 194951, 194952,
                          63945, 194783, 63962, 194789, 194785, 63866, 64068, 194786, 194788, 63970, 194790, 194792,
                          194793, 194794, 63764, 63836, 63935, 63820, 194795, 63793, 194797, 63773, 194799, 194967,
                          194801, 194803, 63884, 64149, 194804, 63909, 63856, 64150, 194805, 194806, 194980, 194810,
                          194814, 63858, 194812, 63848, 194813, 194982, 63971, 194815, 63765, 64005, 194823, 194816,
                          63946, 64151, 194818, 194819, 63786, 194981, 64069, 194817, 194820, 194821, 63989, 63821,
                          63958, 194830, 64070, 194824, 194825, 63947, 63980, 194828, 64153, 194827, 63748, 64152,
                          63822, 64071, 64154, 63890, 194989, 194831, 194992, 194834, 63778, 63876, 194837, 64155,
                          194836, 194835, 194839, 194613, 194841, 194840, 63995, 194842, 63903, 194993, 63766, 194844,
                          63891, 64072, 64156, 194846, 63936, 63982, 63794, 63774, 194848, 64073, 64158, 194849,
                          194850, 63814, 194852, 194853, 64159, 63994, 63787, 64022, 64160, 63911, 194856, 63841,
                          63963, 194857, 194859, 63917, 194987, 63767, 63972, 63948, 64074, 194862, 194863, 63918,
                          64161, 194864, 194865, 63892, 63983, 194866, 64162, 194868, 64163, 194870, 63949, 63862,
                          63842, 194872, 63973, 194874, 64164, 64165, 63937, 195013, 63758, 64023, 64166, 64167, 63795,
                          64168, 194880, 63853, 194885, 194886, 194887, 64170, 64169, 194888, 194890, 64157, 194894,
                          63950, 63803, 194895, 64075, 195019, 63815, 64171, 194896, 63844, 63877, 64024, 64076, 64078,
                          64077, 64079, 64080, 194899, 64081, 64025, 64026, 195020, 63804, 64082, 64083, 64027, 194902,
                          63926, 63893, 194903, 63830, 64084, 194905, 194906, 194907, 64085, 64172, 63991, 194911,
                          63992, 64086, 64173, 194914, 194915, 63910, 63812, 64174, 63993, 195027, 64029, 194918,
                          64003, 194921, 63867, 194920, 194922, 63951, 63850, 63823, 64175, 194924, 63805, 63831,
                          194926, 63894, 64087, 64176, 194927, 64088, 63824, 64089, 194928, 64215, 195032, 64177,
                          64090, 63974, 194934, 63759, 194936, 63919, 64030, 194937, 63796, 64091, 64178, 194938,
                          63920, 194941, 63895, 194943, 63813, 63827, 194774, 194946, 194947, 194949, 63782, 63990,
                          64092, 194707, 194955, 194956, 63868, 64093, 64094, 194960, 194959, 194961, 194963, 194964,
                          194965, 63860, 194968, 194966, 194969, 194972, 63998, 64179, 194976, 194970, 194971, 194973,
                          63806, 194977, 194978, 194979, 194974, 64180, 63832, 63768, 63854, 64095, 194975, 63897,
                          194984, 194985, 63938, 194986, 194988, 63779, 195040, 195041, 63984, 63797, 64032, 63775,
                          63760, 194995, 63798, 194996, 194997, 194998, 195000, 194999, 195002, 195001, 195004, 195045,
                          195005, 64181, 195003, 195006, 63761, 195008, 195009, 63783, 64008, 195011, 195012, 63904,
                          63975, 195014, 195015, 63976, 63762, 195017, 64096, 64182, 63780, 64183, 64010, 64097, 64184,
                          195023, 63855, 63905, 64185, 64187, 63869, 63809, 64190, 195024, 64034, 64186, 63837, 64189,
                          64098, 64188, 64099, 64191, 63996, 63834, 64192, 195025, 195053, 63744, 195026, 195028,
                          195029, 63816, 63747, 64100, 64101, 64193, 195030, 195031, 195057, 195035, 195034, 63799,
                          195036, 63746, 195038, 63896, 63959, 64194, 195039, 64007, 63885, 194957, 63857, 64102,
                          63898, 64037, 64103, 64195, 63939, 63763, 195042, 63788, 195043, 64038, 195044, 195046,
                          195062, 63769, 64196, 63927, 63977, 63870, 63754, 194588, 195063, 63921, 195047, 64197,
                          195050, 195048, 195049, 63807, 63899, 195051, 195067, 195052, 195069, 195054, 63878, 195073,
                          195056, 63942, 63825, 64009, 63833, 63955, 64198, 63964, 63985, 63928, 195059, 63978, 64104,
                          64199, 63922, 63817, 195061, 63800, 63923, 64028, 64200, 64201, 195066, 64105, 64202, 64203,
                          195070, 195071, 63924, 195072, 64106, 64204, 63952, 195074, 64042, 64043, 64044, 195076,
                          195077, 195078, 63770, 195079, 63879, 64205, 195082, 195081, 63801, 195083, 63986, 195084,
                          195087, 64045, 63802, 63776, 63808, 63880, 63987, 195093, 63881, 195095, 195096, 195097,
                          195098, 195099, 195100, 64216, 63940, 64217, 63751, 63752, 64206, 195088, 195090, 195091,
                          195092, 194703, 195101, 8814, 8800, 8815, 192, 193, 194, 195, 256, 258, 550, 196, 7842, 197,
                          461, 512, 514, 7840, 7680, 260, 7682, 7684, 7686, 262, 264, 266, 268, 199, 7690, 270, 7692,
                          7696, 7698, 7694, 200, 201, 202, 7868, 274, 276, 278, 203, 7866, 282, 516, 518, 7864, 552,
                          280, 7704, 7706, 7710, 500, 284, 7712, 286, 288, 486, 290, 292, 7714, 7718, 542, 7716, 7720,
                          7722, 204, 205, 206, 296, 298, 300, 304, 207, 7880, 463, 520, 522, 7882, 302, 7724, 308,
                          7728, 488, 7730, 310, 7732, 313, 317, 7734, 315, 7740, 7738, 7742, 7744, 7746, 504, 323, 209,
                          7748, 327, 7750, 325, 7754, 7752, 210, 211, 212, 213, 332, 334, 558, 214, 7886, 336, 465,
                          524, 526, 416, 7884, 490, 7764, 7766, 340, 7768, 344, 528, 530, 7770, 342, 7774, 346, 348,
                          7776, 352, 7778, 536, 350, 7786, 356, 7788, 538, 354, 7792, 7790, 217, 218, 219, 360, 362,
                          364, 220, 7910, 366, 368, 467, 532, 534, 431, 7908, 7794, 370, 7798, 7796, 7804, 7806, 7808,
                          7810, 372, 7814, 7812, 7816, 7818, 7820, 7922, 221, 374, 7928, 562, 7822, 376, 7926, 7924,
                          377, 7824, 379, 381, 7826, 7828, 224, 225, 226, 227, 257, 259, 551, 228, 7843, 229, 462, 513,
                          515, 7841, 7681, 261, 7683, 7685, 7687, 263, 265, 267, 269, 231, 7691, 271, 7693, 7697, 7699,
                          7695, 232, 233, 234, 7869, 275, 277, 279, 235, 7867, 283, 517, 519, 7865, 553, 281, 7705,
                          7707, 7711, 501, 285, 7713, 287, 289, 487, 291, 293, 7715, 7719, 543, 7717, 7721, 7723, 7830,
                          236, 237, 238, 297, 299, 301, 239, 7881, 464, 521, 523, 7883, 303, 7725, 309, 496, 7729, 489,
                          7731, 311, 7733, 314, 318, 7735, 316, 7741, 7739, 7743, 7745, 7747, 505, 324, 241, 7749, 328,
                          7751, 326, 7755, 7753, 242, 243, 244, 245, 333, 335, 559, 246, 7887, 337, 466, 525, 527, 417,
                          7885, 491, 7765, 7767, 341, 7769, 345, 529, 531, 7771, 343, 7775, 347, 349, 7777, 353, 7779,
                          537, 351, 7787, 7831, 357, 7789, 539, 355, 7793, 7791, 249, 250, 251, 361, 363, 365, 252,
                          7911, 367, 369, 468, 533, 535, 432, 7909, 7795, 371, 7799, 7797, 7805, 7807, 7809, 7811, 373,
                          7815, 7813, 7832, 7817, 7819, 7821, 7923, 253, 375, 7929, 563, 7823, 255, 7927, 7833, 7925,
                          378, 7825, 380, 382, 7827, 7829, 8173, 901, 8129, 7846, 7844, 7850, 7848, 478, 506, 508, 482,
                          7688, 7872, 7870, 7876, 7874, 7726, 7890, 7888, 7894, 7892, 7756, 556, 7758, 554, 510, 475,
                          471, 469, 473, 7847, 7845, 7851, 7849, 479, 507, 509, 483, 7689, 7873, 7871, 7877, 7875,
                          7727, 7891, 7889, 7895, 7893, 7757, 557, 7759, 555, 511, 476, 472, 470, 474, 7856, 7854,
                          7860, 7858, 7857, 7855, 7861, 7859, 7700, 7702, 7701, 7703, 7760, 7762, 7761, 7763, 7780,
                          7781, 7782, 7783, 7800, 7801, 7802, 7803, 7835, 7900, 7898, 7904, 7902, 7906, 7901, 7899,
                          7905, 7903, 7907, 7914, 7912, 7918, 7916, 7920, 7915, 7913, 7919, 7917, 7921, 494, 492, 493,
                          480, 481, 7708, 7709, 560, 561, 495, 836, 8122, 902, 8121, 8120, 7944, 7945, 8124, 8136, 904,
                          7960, 7961, 8138, 905, 7976, 7977, 8140, 8154, 906, 8153, 8152, 938, 7992, 7993, 8184, 908,
                          8008, 8009, 8172, 8170, 910, 8169, 8168, 939, 8025, 8186, 911, 8040, 8041, 8188, 8116, 8132,
                          8048, 940, 8113, 8112, 7936, 7937, 8118, 8115, 8050, 941, 7952, 7953, 8052, 942, 7968, 7969,
                          8134, 8131, 8054, 943, 8145, 8144, 970, 7984, 7985, 8150, 8056, 972, 8000, 8001, 8164, 8165,
                          8058, 973, 8161, 8160, 971, 8016, 8017, 8166, 8060, 974, 8032, 8033, 8182, 8179, 8146, 912,
                          8151, 8162, 944, 8167, 8180, 979, 980, 1031, 1232, 1234, 1027, 1024, 1238, 1025, 1217, 1244,
                          1246, 1037, 1250, 1049, 1252, 1036, 1254, 1262, 1038, 1264, 1266, 1268, 1272, 1260, 1233,
                          1235, 1107, 1104, 1239, 1105, 1218, 1245, 1247, 1117, 1251, 1081, 1253, 1116, 1255, 1263,
                          1118, 1265, 1267, 1269, 1273, 1261, 1111, 1142, 1143, 1242, 1243, 1258, 1259, 1570, 1571,
                          1573, 1572, 1574, 1730, 1747, 1728, 2345, 2353, 2356, 2507, 2508, 2891, 2888, 2892, 2964,
                          3018, 3020, 3019, 3144, 3264, 3274, 3271, 3272, 3275, 3402, 3404, 3403, 3546, 3548, 3550,
                          3549, 3955, 3957, 3969, 4134, 6918, 6920, 6922, 6924, 6926, 6930, 6971, 6973, 6976, 6977,
                          6979, 7736, 7737, 7772, 7773, 7784, 7785, 7852, 7862, 7853, 7863, 7878, 7879, 7896, 7897,
                          7938, 7940, 7942, 8064, 7939, 7941, 7943, 8065, 8066, 8067, 8068, 8069, 8070, 8071, 7946,
                          7948, 7950, 8072, 7947, 7949, 7951, 8073, 8074, 8075, 8076, 8077, 8078, 8079, 7954, 7956,
                          7955, 7957, 7962, 7964, 7963, 7965, 7970, 7972, 7974, 8080, 7971, 7973, 7975, 8081, 8082,
                          8083, 8084, 8085, 8086, 8087, 7978, 7980, 7982, 8088, 7979, 7981, 7983, 8089, 8090, 8091,
                          8092, 8093, 8094, 8095, 7986, 7988, 7990, 7987, 7989, 7991, 7994, 7996, 7998, 7995, 7997,
                          7999, 8002, 8004, 8003, 8005, 8010, 8012, 8011, 8013, 8018, 8020, 8022, 8019, 8021, 8023,
                          8027, 8029, 8031, 8034, 8036, 8038, 8096, 8035, 8037, 8039, 8097, 8098, 8099, 8100, 8101,
                          8102, 8103, 8042, 8044, 8046, 8104, 8043, 8045, 8047, 8105, 8106, 8107, 8108, 8109, 8110,
                          8111, 8114, 8130, 8178, 8119, 8141, 8142, 8143, 8135, 8183, 8157, 8158, 8159, 8602, 8603,
                          8622, 8653, 8655, 8654, 8708, 8713, 8716, 8740, 8742, 8769, 8772, 8775, 8777, 8813, 8802,
                          8816, 8817, 8820, 8821, 8824, 8825, 8832, 8833, 8928, 8929, 8836, 8837, 8840, 8841, 8930,
                          8931, 8876, 8877, 8878, 8879, 8938, 8939, 8940, 8941, 12436, 12364, 12366, 12368, 12370,
                          12372, 12374, 12376, 12378, 12380, 12382, 12384, 12386, 12389, 12391, 12393, 12400, 12401,
                          12403, 12404, 12406, 12407, 12409, 12410, 12412, 12413, 12446, 12532, 12460, 12462, 12464,
                          12466, 12468, 12470, 12472, 12474, 12476, 12478, 12480, 12482, 12485, 12487, 12489, 12496,
                          12497, 12499, 12500, 12502, 12503, 12505, 12506, 12508, 12509, 12535, 12536, 12537, 12538,
                          12542};
    }

    private static int[] getDecompIdx() {
        return new int[] {160, 168, 170, 175, 178, 179, 180, 181, 184, 185, 186, 188, 189, 190, 192, 193, 194, 195,
                          196, 197, 199, 200, 201, 202, 203, 204, 205, 206, 207, 209, 210, 211, 212, 213, 214, 217,
                          218, 219, 220, 221, 224, 225, 226, 227, 228, 229, 231, 232, 233, 234, 235, 236, 237, 238,
                          239, 241, 242, 243, 244, 245, 246, 249, 250, 251, 252, 253, 255, 256, 257, 258, 259, 260,
                          261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 279, 280,
                          281, 282, 283, 284, 285, 286, 287, 288, 289, 290, 291, 292, 293, 296, 297, 298, 299, 300,
                          301, 302, 303, 304, 306, 307, 308, 309, 310, 311, 313, 314, 315, 316, 317, 318, 319, 320,
                          323, 324, 325, 326, 327, 328, 329, 332, 333, 334, 335, 336, 337, 340, 341, 342, 343, 344,
                          345, 346, 347, 348, 349, 350, 351, 352, 353, 354, 355, 356, 357, 360, 361, 362, 363, 364,
                          365, 366, 367, 368, 369, 370, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382,
                          383, 416, 417, 431, 432, 452, 453, 454, 455, 456, 457, 458, 459, 460, 461, 462, 463, 464,
                          465, 466, 467, 468, 469, 470, 471, 472, 473, 474, 475, 476, 478, 479, 480, 481, 482, 483,
                          486, 487, 488, 489, 490, 491, 492, 493, 494, 495, 496, 497, 498, 499, 500, 501, 504, 505,
                          506, 507, 508, 509, 510, 511, 512, 513, 514, 515, 516, 517, 518, 519, 520, 521, 522, 523,
                          524, 525, 526, 527, 528, 529, 530, 531, 532, 533, 534, 535, 536, 537, 538, 539, 542, 543,
                          550, 551, 552, 553, 554, 555, 556, 557, 558, 559, 560, 561, 562, 563, 688, 689, 690, 691,
                          692, 693, 694, 695, 696, 728, 729, 730, 731, 732, 733, 736, 737, 738, 739, 740, 832, 833,
                          835, 836, 884, 890, 894, 900, 901, 902, 903, 904, 905, 906, 908, 910, 911, 912, 938, 939,
                          940, 941, 942, 943, 944, 970, 971, 972, 973, 974, 976, 977, 978, 979, 980, 981, 982, 1008,
                          1009, 1010, 1012, 1013, 1017, 1024, 1025, 1027, 1031, 1036, 1037, 1038, 1049, 1081, 1104,
                          1105, 1107, 1111, 1116, 1117, 1118, 1142, 1143, 1217, 1218, 1232, 1233, 1234, 1235, 1238,
                          1239, 1242, 1243, 1244, 1245, 1246, 1247, 1250, 1251, 1252, 1253, 1254, 1255, 1258, 1259,
                          1260, 1261, 1262, 1263, 1264, 1265, 1266, 1267, 1268, 1269, 1272, 1273, 1415, 1570, 1571,
                          1572, 1573, 1574, 1653, 1654, 1655, 1656, 1728, 1730, 1747, 2345, 2353, 2356, 2392, 2393,
                          2394, 2395, 2396, 2397, 2398, 2399, 2507, 2508, 2524, 2525, 2527, 2611, 2614, 2649, 2650,
                          2651, 2654, 2888, 2891, 2892, 2908, 2909, 2964, 3018, 3019, 3020, 3144, 3264, 3271, 3272,
                          3274, 3275, 3402, 3403, 3404, 3546, 3548, 3549, 3550, 3635, 3763, 3804, 3805, 3852, 3907,
                          3917, 3922, 3927, 3932, 3945, 3955, 3957, 3958, 3959, 3960, 3961, 3969, 3987, 3997, 4002,
                          4007, 4012, 4025, 4134, 4348, 6918, 6920, 6922, 6924, 6926, 6930, 6971, 6973, 6976, 6977,
                          6979, 7468, 7469, 7470, 7472, 7473, 7474, 7475, 7476, 7477, 7478, 7479, 7480, 7481, 7482,
                          7484, 7485, 7486, 7487, 7488, 7489, 7490, 7491, 7492, 7493, 7494, 7495, 7496, 7497, 7498,
                          7499, 7500, 7501, 7503, 7504, 7505, 7506, 7507, 7508, 7509, 7510, 7511, 7512, 7513, 7514,
                          7515, 7516, 7517, 7518, 7519, 7520, 7521, 7522, 7523, 7524, 7525, 7526, 7527, 7528, 7529,
                          7530, 7544, 7579, 7580, 7581, 7582, 7583, 7584, 7585, 7586, 7587, 7588, 7589, 7590, 7591,
                          7592, 7593, 7594, 7595, 7596, 7597, 7598, 7599, 7600, 7601, 7602, 7603, 7604, 7605, 7606,
                          7607, 7608, 7609, 7610, 7611, 7612, 7613, 7614, 7615, 7680, 7681, 7682, 7683, 7684, 7685,
                          7686, 7687, 7688, 7689, 7690, 7691, 7692, 7693, 7694, 7695, 7696, 7697, 7698, 7699, 7700,
                          7701, 7702, 7703, 7704, 7705, 7706, 7707, 7708, 7709, 7710, 7711, 7712, 7713, 7714, 7715,
                          7716, 7717, 7718, 7719, 7720, 7721, 7722, 7723, 7724, 7725, 7726, 7727, 7728, 7729, 7730,
                          7731, 7732, 7733, 7734, 7735, 7736, 7737, 7738, 7739, 7740, 7741, 7742, 7743, 7744, 7745,
                          7746, 7747, 7748, 7749, 7750, 7751, 7752, 7753, 7754, 7755, 7756, 7757, 7758, 7759, 7760,
                          7761, 7762, 7763, 7764, 7765, 7766, 7767, 7768, 7769, 7770, 7771, 7772, 7773, 7774, 7775,
                          7776, 7777, 7778, 7779, 7780, 7781, 7782, 7783, 7784, 7785, 7786, 7787, 7788, 7789, 7790,
                          7791, 7792, 7793, 7794, 7795, 7796, 7797, 7798, 7799, 7800, 7801, 7802, 7803, 7804, 7805,
                          7806, 7807, 7808, 7809, 7810, 7811, 7812, 7813, 7814, 7815, 7816, 7817, 7818, 7819, 7820,
                          7821, 7822, 7823, 7824, 7825, 7826, 7827, 7828, 7829, 7830, 7831, 7832, 7833, 7834, 7835,
                          7840, 7841, 7842, 7843, 7844, 7845, 7846, 7847, 7848, 7849, 7850, 7851, 7852, 7853, 7854,
                          7855, 7856, 7857, 7858, 7859, 7860, 7861, 7862, 7863, 7864, 7865, 7866, 7867, 7868, 7869,
                          7870, 7871, 7872, 7873, 7874, 7875, 7876, 7877, 7878, 7879, 7880, 7881, 7882, 7883, 7884,
                          7885, 7886, 7887, 7888, 7889, 7890, 7891, 7892, 7893, 7894, 7895, 7896, 7897, 7898, 7899,
                          7900, 7901, 7902, 7903, 7904, 7905, 7906, 7907, 7908, 7909, 7910, 7911, 7912, 7913, 7914,
                          7915, 7916, 7917, 7918, 7919, 7920, 7921, 7922, 7923, 7924, 7925, 7926, 7927, 7928, 7929,
                          7936, 7937, 7938, 7939, 7940, 7941, 7942, 7943, 7944, 7945, 7946, 7947, 7948, 7949, 7950,
                          7951, 7952, 7953, 7954, 7955, 7956, 7957, 7960, 7961, 7962, 7963, 7964, 7965, 7968, 7969,
                          7970, 7971, 7972, 7973, 7974, 7975, 7976, 7977, 7978, 7979, 7980, 7981, 7982, 7983, 7984,
                          7985, 7986, 7987, 7988, 7989, 7990, 7991, 7992, 7993, 7994, 7995, 7996, 7997, 7998, 7999,
                          8000, 8001, 8002, 8003, 8004, 8005, 8008, 8009, 8010, 8011, 8012, 8013, 8016, 8017, 8018,
                          8019, 8020, 8021, 8022, 8023, 8025, 8027, 8029, 8031, 8032, 8033, 8034, 8035, 8036, 8037,
                          8038, 8039, 8040, 8041, 8042, 8043, 8044, 8045, 8046, 8047, 8048, 8049, 8050, 8051, 8052,
                          8053, 8054, 8055, 8056, 8057, 8058, 8059, 8060, 8061, 8064, 8065, 8066, 8067, 8068, 8069,
                          8070, 8071, 8072, 8073, 8074, 8075, 8076, 8077, 8078, 8079, 8080, 8081, 8082, 8083, 8084,
                          8085, 8086, 8087, 8088, 8089, 8090, 8091, 8092, 8093, 8094, 8095, 8096, 8097, 8098, 8099,
                          8100, 8101, 8102, 8103, 8104, 8105, 8106, 8107, 8108, 8109, 8110, 8111, 8112, 8113, 8114,
                          8115, 8116, 8118, 8119, 8120, 8121, 8122, 8123, 8124, 8125, 8126, 8127, 8128, 8129, 8130,
                          8131, 8132, 8134, 8135, 8136, 8137, 8138, 8139, 8140, 8141, 8142, 8143, 8144, 8145, 8146,
                          8147, 8150, 8151, 8152, 8153, 8154, 8155, 8157, 8158, 8159, 8160, 8161, 8162, 8163, 8164,
                          8165, 8166, 8167, 8168, 8169, 8170, 8171, 8172, 8173, 8174, 8175, 8178, 8179, 8180, 8182,
                          8183, 8184, 8185, 8186, 8187, 8188, 8189, 8190, 8192, 8193, 8194, 8195, 8196, 8197, 8198,
                          8199, 8200, 8201, 8202, 8209, 8215, 8228, 8229, 8230, 8239, 8243, 8244, 8246, 8247, 8252,
                          8254, 8263, 8264, 8265, 8279, 8287, 8304, 8305, 8308, 8309, 8310, 8311, 8312, 8313, 8314,
                          8315, 8316, 8317, 8318, 8319, 8320, 8321, 8322, 8323, 8324, 8325, 8326, 8327, 8328, 8329,
                          8330, 8331, 8332, 8333, 8334, 8336, 8337, 8338, 8339, 8340, 8360, 8448, 8449, 8450, 8451,
                          8453, 8454, 8455, 8457, 8458, 8459, 8460, 8461, 8462, 8463, 8464, 8465, 8466, 8467, 8469,
                          8470, 8473, 8474, 8475, 8476, 8477, 8480, 8481, 8482, 8484, 8486, 8488, 8490, 8491, 8492,
                          8493, 8495, 8496, 8497, 8499, 8500, 8501, 8502, 8503, 8504, 8505, 8507, 8508, 8509, 8510,
                          8511, 8512, 8517, 8518, 8519, 8520, 8521, 8531, 8532, 8533, 8534, 8535, 8536, 8537, 8538,
                          8539, 8540, 8541, 8542, 8543, 8544, 8545, 8546, 8547, 8548, 8549, 8550, 8551, 8552, 8553,
                          8554, 8555, 8556, 8557, 8558, 8559, 8560, 8561, 8562, 8563, 8564, 8565, 8566, 8567, 8568,
                          8569, 8570, 8571, 8572, 8573, 8574, 8575, 8602, 8603, 8622, 8653, 8654, 8655, 8708, 8713,
                          8716, 8740, 8742, 8748, 8749, 8751, 8752, 8769, 8772, 8775, 8777, 8800, 8802, 8813, 8814,
                          8815, 8816, 8817, 8820, 8821, 8824, 8825, 8832, 8833, 8836, 8837, 8840, 8841, 8876, 8877,
                          8878, 8879, 8928, 8929, 8930, 8931, 8938, 8939, 8940, 8941, 9001, 9002, 9312, 9313, 9314,
                          9315, 9316, 9317, 9318, 9319, 9320, 9321, 9322, 9323, 9324, 9325, 9326, 9327, 9328, 9329,
                          9330, 9331, 9332, 9333, 9334, 9335, 9336, 9337, 9338, 9339, 9340, 9341, 9342, 9343, 9344,
                          9345, 9346, 9347, 9348, 9349, 9350, 9351, 9352, 9353, 9354, 9355, 9356, 9357, 9358, 9359,
                          9360, 9361, 9362, 9363, 9364, 9365, 9366, 9367, 9368, 9369, 9370, 9371, 9372, 9373, 9374,
                          9375, 9376, 9377, 9378, 9379, 9380, 9381, 9382, 9383, 9384, 9385, 9386, 9387, 9388, 9389,
                          9390, 9391, 9392, 9393, 9394, 9395, 9396, 9397, 9398, 9399, 9400, 9401, 9402, 9403, 9404,
                          9405, 9406, 9407, 9408, 9409, 9410, 9411, 9412, 9413, 9414, 9415, 9416, 9417, 9418, 9419,
                          9420, 9421, 9422, 9423, 9424, 9425, 9426, 9427, 9428, 9429, 9430, 9431, 9432, 9433, 9434,
                          9435, 9436, 9437, 9438, 9439, 9440, 9441, 9442, 9443, 9444, 9445, 9446, 9447, 9448, 9449,
                          9450, 10764, 10868, 10869, 10870, 10972, 11631, 11935, 12019, 12032, 12033, 12034, 12035,
                          12036, 12037, 12038, 12039, 12040, 12041, 12042, 12043, 12044, 12045, 12046, 12047, 12048,
                          12049, 12050, 12051, 12052, 12053, 12054, 12055, 12056, 12057, 12058, 12059, 12060, 12061,
                          12062, 12063, 12064, 12065, 12066, 12067, 12068, 12069, 12070, 12071, 12072, 12073, 12074,
                          12075, 12076, 12077, 12078, 12079, 12080, 12081, 12082, 12083, 12084, 12085, 12086, 12087,
                          12088, 12089, 12090, 12091, 12092, 12093, 12094, 12095, 12096, 12097, 12098, 12099, 12100,
                          12101, 12102, 12103, 12104, 12105, 12106, 12107, 12108, 12109, 12110, 12111, 12112, 12113,
                          12114, 12115, 12116, 12117, 12118, 12119, 12120, 12121, 12122, 12123, 12124, 12125, 12126,
                          12127, 12128, 12129, 12130, 12131, 12132, 12133, 12134, 12135, 12136, 12137, 12138, 12139,
                          12140, 12141, 12142, 12143, 12144, 12145, 12146, 12147, 12148, 12149, 12150, 12151, 12152,
                          12153, 12154, 12155, 12156, 12157, 12158, 12159, 12160, 12161, 12162, 12163, 12164, 12165,
                          12166, 12167, 12168, 12169, 12170, 12171, 12172, 12173, 12174, 12175, 12176, 12177, 12178,
                          12179, 12180, 12181, 12182, 12183, 12184, 12185, 12186, 12187, 12188, 12189, 12190, 12191,
                          12192, 12193, 12194, 12195, 12196, 12197, 12198, 12199, 12200, 12201, 12202, 12203, 12204,
                          12205, 12206, 12207, 12208, 12209, 12210, 12211, 12212, 12213, 12214, 12215, 12216, 12217,
                          12218, 12219, 12220, 12221, 12222, 12223, 12224, 12225, 12226, 12227, 12228, 12229, 12230,
                          12231, 12232, 12233, 12234, 12235, 12236, 12237, 12238, 12239, 12240, 12241, 12242, 12243,
                          12244, 12245, 12288, 12342, 12344, 12345, 12346, 12364, 12366, 12368, 12370, 12372, 12374,
                          12376, 12378, 12380, 12382, 12384, 12386, 12389, 12391, 12393, 12400, 12401, 12403, 12404,
                          12406, 12407, 12409, 12410, 12412, 12413, 12436, 12443, 12444, 12446, 12447, 12460, 12462,
                          12464, 12466, 12468, 12470, 12472, 12474, 12476, 12478, 12480, 12482, 12485, 12487, 12489,
                          12496, 12497, 12499, 12500, 12502, 12503, 12505, 12506, 12508, 12509, 12532, 12535, 12536,
                          12537, 12538, 12542, 12543, 12593, 12594, 12595, 12596, 12597, 12598, 12599, 12600, 12601,
                          12602, 12603, 12604, 12605, 12606, 12607, 12608, 12609, 12610, 12611, 12612, 12613, 12614,
                          12615, 12616, 12617, 12618, 12619, 12620, 12621, 12622, 12623, 12624, 12625, 12626, 12627,
                          12628, 12629, 12630, 12631, 12632, 12633, 12634, 12635, 12636, 12637, 12638, 12639, 12640,
                          12641, 12642, 12643, 12644, 12645, 12646, 12647, 12648, 12649, 12650, 12651, 12652, 12653,
                          12654, 12655, 12656, 12657, 12658, 12659, 12660, 12661, 12662, 12663, 12664, 12665, 12666,
                          12667, 12668, 12669, 12670, 12671, 12672, 12673, 12674, 12675, 12676, 12677, 12678, 12679,
                          12680, 12681, 12682, 12683, 12684, 12685, 12686, 12690, 12691, 12692, 12693, 12694, 12695,
                          12696, 12697, 12698, 12699, 12700, 12701, 12702, 12703, 12800, 12801, 12802, 12803, 12804,
                          12805, 12806, 12807, 12808, 12809, 12810, 12811, 12812, 12813, 12814, 12815, 12816, 12817,
                          12818, 12819, 12820, 12821, 12822, 12823, 12824, 12825, 12826, 12827, 12828, 12829, 12830,
                          12832, 12833, 12834, 12835, 12836, 12837, 12838, 12839, 12840, 12841, 12842, 12843, 12844,
                          12845, 12846, 12847, 12848, 12849, 12850, 12851, 12852, 12853, 12854, 12855, 12856, 12857,
                          12858, 12859, 12860, 12861, 12862, 12863, 12864, 12865, 12866, 12867, 12880, 12881, 12882,
                          12883, 12884, 12885, 12886, 12887, 12888, 12889, 12890, 12891, 12892, 12893, 12894, 12895,
                          12896, 12897, 12898, 12899, 12900, 12901, 12902, 12903, 12904, 12905, 12906, 12907, 12908,
                          12909, 12910, 12911, 12912, 12913, 12914, 12915, 12916, 12917, 12918, 12919, 12920, 12921,
                          12922, 12923, 12924, 12925, 12926, 12928, 12929, 12930, 12931, 12932, 12933, 12934, 12935,
                          12936, 12937, 12938, 12939, 12940, 12941, 12942, 12943, 12944, 12945, 12946, 12947, 12948,
                          12949, 12950, 12951, 12952, 12953, 12954, 12955, 12956, 12957, 12958, 12959, 12960, 12961,
                          12962, 12963, 12964, 12965, 12966, 12967, 12968, 12969, 12970, 12971, 12972, 12973, 12974,
                          12975, 12976, 12977, 12978, 12979, 12980, 12981, 12982, 12983, 12984, 12985, 12986, 12987,
                          12988, 12989, 12990, 12991, 12992, 12993, 12994, 12995, 12996, 12997, 12998, 12999, 13000,
                          13001, 13002, 13003, 13004, 13005, 13006, 13007, 13008, 13009, 13010, 13011, 13012, 13013,
                          13014, 13015, 13016, 13017, 13018, 13019, 13020, 13021, 13022, 13023, 13024, 13025, 13026,
                          13027, 13028, 13029, 13030, 13031, 13032, 13033, 13034, 13035, 13036, 13037, 13038, 13039,
                          13040, 13041, 13042, 13043, 13044, 13045, 13046, 13047, 13048, 13049, 13050, 13051, 13052,
                          13053, 13054, 13056, 13057, 13058, 13059, 13060, 13061, 13062, 13063, 13064, 13065, 13066,
                          13067, 13068, 13069, 13070, 13071, 13072, 13073, 13074, 13075, 13076, 13077, 13078, 13079,
                          13080, 13081, 13082, 13083, 13084, 13085, 13086, 13087, 13088, 13089, 13090, 13091, 13092,
                          13093, 13094, 13095, 13096, 13097, 13098, 13099, 13100, 13101, 13102, 13103, 13104, 13105,
                          13106, 13107, 13108, 13109, 13110, 13111, 13112, 13113, 13114, 13115, 13116, 13117, 13118,
                          13119, 13120, 13121, 13122, 13123, 13124, 13125, 13126, 13127, 13128, 13129, 13130, 13131,
                          13132, 13133, 13134, 13135, 13136, 13137, 13138, 13139, 13140, 13141, 13142, 13143, 13144,
                          13145, 13146, 13147, 13148, 13149, 13150, 13151, 13152, 13153, 13154, 13155, 13156, 13157,
                          13158, 13159, 13160, 13161, 13162, 13163, 13164, 13165, 13166, 13167, 13168, 13169, 13170,
                          13171, 13172, 13173, 13174, 13175, 13176, 13177, 13178, 13179, 13180, 13181, 13182, 13183,
                          13184, 13185, 13186, 13187, 13188, 13189, 13190, 13191, 13192, 13193, 13194, 13195, 13196,
                          13197, 13198, 13199, 13200, 13201, 13202, 13203, 13204, 13205, 13206, 13207, 13208, 13209,
                          13210, 13211, 13212, 13213, 13214, 13215, 13216, 13217, 13218, 13219, 13220, 13221, 13222,
                          13223, 13224, 13225, 13226, 13227, 13228, 13229, 13230, 13231, 13232, 13233, 13234, 13235,
                          13236, 13237, 13238, 13239, 13240, 13241, 13242, 13243, 13244, 13245, 13246, 13247, 13248,
                          13249, 13250, 13251, 13252, 13253, 13254, 13255, 13256, 13257, 13258, 13259, 13260, 13261,
                          13262, 13263, 13264, 13265, 13266, 13267, 13268, 13269, 13270, 13271, 13272, 13273, 13274,
                          13275, 13276, 13277, 13278, 13279, 13280, 13281, 13282, 13283, 13284, 13285, 13286, 13287,
                          13288, 13289, 13290, 13291, 13292, 13293, 13294, 13295, 13296, 13297, 13298, 13299, 13300,
                          13301, 13302, 13303, 13304, 13305, 13306, 13307, 13308, 13309, 13310, 13311, 63744, 63745,
                          63746, 63747, 63748, 63749, 63750, 63751, 63752, 63753, 63754, 63755, 63756, 63757, 63758,
                          63759, 63760, 63761, 63762, 63763, 63764, 63765, 63766, 63767, 63768, 63769, 63770, 63771,
                          63772, 63773, 63774, 63775, 63776, 63777, 63778, 63779, 63780, 63781, 63782, 63783, 63784,
                          63785, 63786, 63787, 63788, 63789, 63790, 63791, 63792, 63793, 63794, 63795, 63796, 63797,
                          63798, 63799, 63800, 63801, 63802, 63803, 63804, 63805, 63806, 63807, 63808, 63809, 63810,
                          63811, 63812, 63813, 63814, 63815, 63816, 63817, 63818, 63819, 63820, 63821, 63822, 63823,
                          63824, 63825, 63826, 63827, 63828, 63829, 63830, 63831, 63832, 63833, 63834, 63835, 63836,
                          63837, 63838, 63839, 63840, 63841, 63842, 63843, 63844, 63845, 63846, 63847, 63848, 63849,
                          63850, 63851, 63852, 63853, 63854, 63855, 63856, 63857, 63858, 63859, 63860, 63861, 63862,
                          63863, 63864, 63865, 63866, 63867, 63868, 63869, 63870, 63871, 63872, 63873, 63874, 63875,
                          63876, 63877, 63878, 63879, 63880, 63881, 63882, 63883, 63884, 63885, 63886, 63887, 63888,
                          63889, 63890, 63891, 63892, 63893, 63894, 63895, 63896, 63897, 63898, 63899, 63900, 63901,
                          63902, 63903, 63904, 63905, 63906, 63907, 63908, 63909, 63910, 63911, 63912, 63913, 63914,
                          63915, 63916, 63917, 63918, 63919, 63920, 63921, 63922, 63923, 63924, 63925, 63926, 63927,
                          63928, 63929, 63930, 63931, 63932, 63933, 63934, 63935, 63936, 63937, 63938, 63939, 63940,
                          63941, 63942, 63943, 63944, 63945, 63946, 63947, 63948, 63949, 63950, 63951, 63952, 63953,
                          63954, 63955, 63956, 63957, 63958, 63959, 63960, 63961, 63962, 63963, 63964, 63965, 63966,
                          63967, 63968, 63969, 63970, 63971, 63972, 63973, 63974, 63975, 63976, 63977, 63978, 63979,
                          63980, 63981, 63982, 63983, 63984, 63985, 63986, 63987, 63988, 63989, 63990, 63991, 63992,
                          63993, 63994, 63995, 63996, 63997, 63998, 63999, 64000, 64001, 64002, 64003, 64004, 64005,
                          64006, 64007, 64008, 64009, 64010, 64011, 64012, 64013, 64016, 64018, 64021, 64022, 64023,
                          64024, 64025, 64026, 64027, 64028, 64029, 64030, 64032, 64034, 64037, 64038, 64042, 64043,
                          64044, 64045, 64048, 64049, 64050, 64051, 64052, 64053, 64054, 64055, 64056, 64057, 64058,
                          64059, 64060, 64061, 64062, 64063, 64064, 64065, 64066, 64067, 64068, 64069, 64070, 64071,
                          64072, 64073, 64074, 64075, 64076, 64077, 64078, 64079, 64080, 64081, 64082, 64083, 64084,
                          64085, 64086, 64087, 64088, 64089, 64090, 64091, 64092, 64093, 64094, 64095, 64096, 64097,
                          64098, 64099, 64100, 64101, 64102, 64103, 64104, 64105, 64106, 64112, 64113, 64114, 64115,
                          64116, 64117, 64118, 64119, 64120, 64121, 64122, 64123, 64124, 64125, 64126, 64127, 64128,
                          64129, 64130, 64131, 64132, 64133, 64134, 64135, 64136, 64137, 64138, 64139, 64140, 64141,
                          64142, 64143, 64144, 64145, 64146, 64147, 64148, 64149, 64150, 64151, 64152, 64153, 64154,
                          64155, 64156, 64157, 64158, 64159, 64160, 64161, 64162, 64163, 64164, 64165, 64166, 64167,
                          64168, 64169, 64170, 64171, 64172, 64173, 64174, 64175, 64176, 64177, 64178, 64179, 64180,
                          64181, 64182, 64183, 64184, 64185, 64186, 64187, 64188, 64189, 64190, 64191, 64192, 64193,
                          64194, 64195, 64196, 64197, 64198, 64199, 64200, 64201, 64202, 64203, 64204, 64205, 64206,
                          64207, 64208, 64209, 64210, 64211, 64212, 64213, 64214, 64215, 64216, 64217, 64256, 64257,
                          64258, 64259, 64260, 64261, 64262, 64275, 64276, 64277, 64278, 64279, 64285, 64287, 64288,
                          64289, 64290, 64291, 64292, 64293, 64294, 64295, 64296, 64297, 64298, 64299, 64300, 64301,
                          64302, 64303, 64304, 64305, 64306, 64307, 64308, 64309, 64310, 64312, 64313, 64314, 64315,
                          64316, 64318, 64320, 64321, 64323, 64324, 64326, 64327, 64328, 64329, 64330, 64331, 64332,
                          64333, 64334, 64335, 64336, 64337, 64338, 64339, 64340, 64341, 64342, 64343, 64344, 64345,
                          64346, 64347, 64348, 64349, 64350, 64351, 64352, 64353, 64354, 64355, 64356, 64357, 64358,
                          64359, 64360, 64361, 64362, 64363, 64364, 64365, 64366, 64367, 64368, 64369, 64370, 64371,
                          64372, 64373, 64374, 64375, 64376, 64377, 64378, 64379, 64380, 64381, 64382, 64383, 64384,
                          64385, 64386, 64387, 64388, 64389, 64390, 64391, 64392, 64393, 64394, 64395, 64396, 64397,
                          64398, 64399, 64400, 64401, 64402, 64403, 64404, 64405, 64406, 64407, 64408, 64409, 64410,
                          64411, 64412, 64413, 64414, 64415, 64416, 64417, 64418, 64419, 64420, 64421, 64422, 64423,
                          64424, 64425, 64426, 64427, 64428, 64429, 64430, 64431, 64432, 64433, 64467, 64468, 64469,
                          64470, 64471, 64472, 64473, 64474, 64475, 64476, 64477, 64478, 64479, 64480, 64481, 64482,
                          64483, 64484, 64485, 64486, 64487, 64488, 64489, 64490, 64491, 64492, 64493, 64494, 64495,
                          64496, 64497, 64498, 64499, 64500, 64501, 64502, 64503, 64504, 64505, 64506, 64507, 64508,
                          64509, 64510, 64511, 64512, 64513, 64514, 64515, 64516, 64517, 64518, 64519, 64520, 64521,
                          64522, 64523, 64524, 64525, 64526, 64527, 64528, 64529, 64530, 64531, 64532, 64533, 64534,
                          64535, 64536, 64537, 64538, 64539, 64540, 64541, 64542, 64543, 64544, 64545, 64546, 64547,
                          64548, 64549, 64550, 64551, 64552, 64553, 64554, 64555, 64556, 64557, 64558, 64559, 64560,
                          64561, 64562, 64563, 64564, 64565, 64566, 64567, 64568, 64569, 64570, 64571, 64572, 64573,
                          64574, 64575, 64576, 64577, 64578, 64579, 64580, 64581, 64582, 64583, 64584, 64585, 64586,
                          64587, 64588, 64589, 64590, 64591, 64592, 64593, 64594, 64595, 64596, 64597, 64598, 64599,
                          64600, 64601, 64602, 64603, 64604, 64605, 64606, 64607, 64608, 64609, 64610, 64611, 64612,
                          64613, 64614, 64615, 64616, 64617, 64618, 64619, 64620, 64621, 64622, 64623, 64624, 64625,
                          64626, 64627, 64628, 64629, 64630, 64631, 64632, 64633, 64634, 64635, 64636, 64637, 64638,
                          64639, 64640, 64641, 64642, 64643, 64644, 64645, 64646, 64647, 64648, 64649, 64650, 64651,
                          64652, 64653, 64654, 64655, 64656, 64657, 64658, 64659, 64660, 64661, 64662, 64663, 64664,
                          64665, 64666, 64667, 64668, 64669, 64670, 64671, 64672, 64673, 64674, 64675, 64676, 64677,
                          64678, 64679, 64680, 64681, 64682, 64683, 64684, 64685, 64686, 64687, 64688, 64689, 64690,
                          64691, 64692, 64693, 64694, 64695, 64696, 64697, 64698, 64699, 64700, 64701, 64702, 64703,
                          64704, 64705, 64706, 64707, 64708, 64709, 64710, 64711, 64712, 64713, 64714, 64715, 64716,
                          64717, 64718, 64719, 64720, 64721, 64722, 64723, 64724, 64725, 64726, 64727, 64728, 64729,
                          64730, 64731, 64732, 64733, 64734, 64735, 64736, 64737, 64738, 64739, 64740, 64741, 64742,
                          64743, 64744, 64745, 64746, 64747, 64748, 64749, 64750, 64751, 64752, 64753, 64754, 64755,
                          64756, 64757, 64758, 64759, 64760, 64761, 64762, 64763, 64764, 64765, 64766, 64767, 64768,
                          64769, 64770, 64771, 64772, 64773, 64774, 64775, 64776, 64777, 64778, 64779, 64780, 64781,
                          64782, 64783, 64784, 64785, 64786, 64787, 64788, 64789, 64790, 64791, 64792, 64793, 64794,
                          64795, 64796, 64797, 64798, 64799, 64800, 64801, 64802, 64803, 64804, 64805, 64806, 64807,
                          64808, 64809, 64810, 64811, 64812, 64813, 64814, 64815, 64816, 64817, 64818, 64819, 64820,
                          64821, 64822, 64823, 64824, 64825, 64826, 64827, 64828, 64829, 64848, 64849, 64850, 64851,
                          64852, 64853, 64854, 64855, 64856, 64857, 64858, 64859, 64860, 64861, 64862, 64863, 64864,
                          64865, 64866, 64867, 64868, 64869, 64870, 64871, 64872, 64873, 64874, 64875, 64876, 64877,
                          64878, 64879, 64880, 64881, 64882, 64883, 64884, 64885, 64886, 64887, 64888, 64889, 64890,
                          64891, 64892, 64893, 64894, 64895, 64896, 64897, 64898, 64899, 64900, 64901, 64902, 64903,
                          64904, 64905, 64906, 64907, 64908, 64909, 64910, 64911, 64914, 64915, 64916, 64917, 64918,
                          64919, 64920, 64921, 64922, 64923, 64924, 64925, 64926, 64927, 64928, 64929, 64930, 64931,
                          64932, 64933, 64934, 64935, 64936, 64937, 64938, 64939, 64940, 64941, 64942, 64943, 64944,
                          64945, 64946, 64947, 64948, 64949, 64950, 64951, 64952, 64953, 64954, 64955, 64956, 64957,
                          64958, 64959, 64960, 64961, 64962, 64963, 64964, 64965, 64966, 64967, 65008, 65009, 65010,
                          65011, 65012, 65013, 65014, 65015, 65016, 65017, 65018, 65019, 65020, 65040, 65041, 65042,
                          65043, 65044, 65045, 65046, 65047, 65048, 65049, 65072, 65073, 65074, 65075, 65076, 65077,
                          65078, 65079, 65080, 65081, 65082, 65083, 65084, 65085, 65086, 65087, 65088, 65089, 65090,
                          65091, 65092, 65095, 65096, 65097, 65098, 65099, 65100, 65101, 65102, 65103, 65104, 65105,
                          65106, 65108, 65109, 65110, 65111, 65112, 65113, 65114, 65115, 65116, 65117, 65118, 65119,
                          65120, 65121, 65122, 65123, 65124, 65125, 65126, 65128, 65129, 65130, 65131, 65136, 65137,
                          65138, 65140, 65142, 65143, 65144, 65145, 65146, 65147, 65148, 65149, 65150, 65151, 65152,
                          65153, 65154, 65155, 65156, 65157, 65158, 65159, 65160, 65161, 65162, 65163, 65164, 65165,
                          65166, 65167, 65168, 65169, 65170, 65171, 65172, 65173, 65174, 65175, 65176, 65177, 65178,
                          65179, 65180, 65181, 65182, 65183, 65184, 65185, 65186, 65187, 65188, 65189, 65190, 65191,
                          65192, 65193, 65194, 65195, 65196, 65197, 65198, 65199, 65200, 65201, 65202, 65203, 65204,
                          65205, 65206, 65207, 65208, 65209, 65210, 65211, 65212, 65213, 65214, 65215, 65216, 65217,
                          65218, 65219, 65220, 65221, 65222, 65223, 65224, 65225, 65226, 65227, 65228, 65229, 65230,
                          65231, 65232, 65233, 65234, 65235, 65236, 65237, 65238, 65239, 65240, 65241, 65242, 65243,
                          65244, 65245, 65246, 65247, 65248, 65249, 65250, 65251, 65252, 65253, 65254, 65255, 65256,
                          65257, 65258, 65259, 65260, 65261, 65262, 65263, 65264, 65265, 65266, 65267, 65268, 65269,
                          65270, 65271, 65272, 65273, 65274, 65275, 65276, 65281, 65282, 65283, 65284, 65285, 65286,
                          65287, 65288, 65289, 65290, 65291, 65292, 65293, 65294, 65295, 65296, 65297, 65298, 65299,
                          65300, 65301, 65302, 65303, 65304, 65305, 65306, 65307, 65308, 65309, 65310, 65311, 65312,
                          65313, 65314, 65315, 65316, 65317, 65318, 65319, 65320, 65321, 65322, 65323, 65324, 65325,
                          65326, 65327, 65328, 65329, 65330, 65331, 65332, 65333, 65334, 65335, 65336, 65337, 65338,
                          65339, 65340, 65341, 65342, 65343, 65344, 65345, 65346, 65347, 65348, 65349, 65350, 65351,
                          65352, 65353, 65354, 65355, 65356, 65357, 65358, 65359, 65360, 65361, 65362, 65363, 65364,
                          65365, 65366, 65367, 65368, 65369, 65370, 65371, 65372, 65373, 65374, 65375, 65376, 65377,
                          65378, 65379, 65380, 65381, 65382, 65383, 65384, 65385, 65386, 65387, 65388, 65389, 65390,
                          65391, 65392, 65393, 65394, 65395, 65396, 65397, 65398, 65399, 65400, 65401, 65402, 65403,
                          65404, 65405, 65406, 65407, 65408, 65409, 65410, 65411, 65412, 65413, 65414, 65415, 65416,
                          65417, 65418, 65419, 65420, 65421, 65422, 65423, 65424, 65425, 65426, 65427, 65428, 65429,
                          65430, 65431, 65432, 65433, 65434, 65435, 65436, 65437, 65438, 65439, 65440, 65441, 65442,
                          65443, 65444, 65445, 65446, 65447, 65448, 65449, 65450, 65451, 65452, 65453, 65454, 65455,
                          65456, 65457, 65458, 65459, 65460, 65461, 65462, 65463, 65464, 65465, 65466, 65467, 65468,
                          65469, 65470, 65474, 65475, 65476, 65477, 65478, 65479, 65482, 65483, 65484, 65485, 65486,
                          65487, 65490, 65491, 65492, 65493, 65494, 65495, 65498, 65499, 65500, 65504, 65505, 65506,
                          65507, 65508, 65509, 65510, 65512, 65513, 65514, 65515, 65516, 65517, 65518, 119134, 119135,
                          119136, 119137, 119138, 119139, 119140, 119227, 119228, 119229, 119230, 119231, 119232,
                          119808, 119809, 119810, 119811, 119812, 119813, 119814, 119815, 119816, 119817, 119818,
                          119819, 119820, 119821, 119822, 119823, 119824, 119825, 119826, 119827, 119828, 119829,
                          119830, 119831, 119832, 119833, 119834, 119835, 119836, 119837, 119838, 119839, 119840,
                          119841, 119842, 119843, 119844, 119845, 119846, 119847, 119848, 119849, 119850, 119851,
                          119852, 119853, 119854, 119855, 119856, 119857, 119858, 119859, 119860, 119861, 119862,
                          119863, 119864, 119865, 119866, 119867, 119868, 119869, 119870, 119871, 119872, 119873,
                          119874, 119875, 119876, 119877, 119878, 119879, 119880, 119881, 119882, 119883, 119884,
                          119885, 119886, 119887, 119888, 119889, 119890, 119891, 119892, 119894, 119895, 119896,
                          119897, 119898, 119899, 119900, 119901, 119902, 119903, 119904, 119905, 119906, 119907,
                          119908, 119909, 119910, 119911, 119912, 119913, 119914, 119915, 119916, 119917, 119918,
                          119919, 119920, 119921, 119922, 119923, 119924, 119925, 119926, 119927, 119928, 119929,
                          119930, 119931, 119932, 119933, 119934, 119935, 119936, 119937, 119938, 119939, 119940,
                          119941, 119942, 119943, 119944, 119945, 119946, 119947, 119948, 119949, 119950, 119951,
                          119952, 119953, 119954, 119955, 119956, 119957, 119958, 119959, 119960, 119961, 119962,
                          119963, 119964, 119966, 119967, 119970, 119973, 119974, 119977, 119978, 119979, 119980,
                          119982, 119983, 119984, 119985, 119986, 119987, 119988, 119989, 119990, 119991, 119992,
                          119993, 119995, 119997, 119998, 119999, 120000, 120001, 120002, 120003, 120005, 120006,
                          120007, 120008, 120009, 120010, 120011, 120012, 120013, 120014, 120015, 120016, 120017,
                          120018, 120019, 120020, 120021, 120022, 120023, 120024, 120025, 120026, 120027, 120028,
                          120029, 120030, 120031, 120032, 120033, 120034, 120035, 120036, 120037, 120038, 120039,
                          120040, 120041, 120042, 120043, 120044, 120045, 120046, 120047, 120048, 120049, 120050,
                          120051, 120052, 120053, 120054, 120055, 120056, 120057, 120058, 120059, 120060, 120061,
                          120062, 120063, 120064, 120065, 120066, 120067, 120068, 120069, 120071, 120072, 120073,
                          120074, 120077, 120078, 120079, 120080, 120081, 120082, 120083, 120084, 120086, 120087,
                          120088, 120089, 120090, 120091, 120092, 120094, 120095, 120096, 120097, 120098, 120099,
                          120100, 120101, 120102, 120103, 120104, 120105, 120106, 120107, 120108, 120109, 120110,
                          120111, 120112, 120113, 120114, 120115, 120116, 120117, 120118, 120119, 120120, 120121,
                          120123, 120124, 120125, 120126, 120128, 120129, 120130, 120131, 120132, 120134, 120138,
                          120139, 120140, 120141, 120142, 120143, 120144, 120146, 120147, 120148, 120149, 120150,
                          120151, 120152, 120153, 120154, 120155, 120156, 120157, 120158, 120159, 120160, 120161,
                          120162, 120163, 120164, 120165, 120166, 120167, 120168, 120169, 120170, 120171, 120172,
                          120173, 120174, 120175, 120176, 120177, 120178, 120179, 120180, 120181, 120182, 120183,
                          120184, 120185, 120186, 120187, 120188, 120189, 120190, 120191, 120192, 120193, 120194,
                          120195, 120196, 120197, 120198, 120199, 120200, 120201, 120202, 120203, 120204, 120205,
                          120206, 120207, 120208, 120209, 120210, 120211, 120212, 120213, 120214, 120215, 120216,
                          120217, 120218, 120219, 120220, 120221, 120222, 120223, 120224, 120225, 120226, 120227,
                          120228, 120229, 120230, 120231, 120232, 120233, 120234, 120235, 120236, 120237, 120238,
                          120239, 120240, 120241, 120242, 120243, 120244, 120245, 120246, 120247, 120248, 120249,
                          120250, 120251, 120252, 120253, 120254, 120255, 120256, 120257, 120258, 120259, 120260,
                          120261, 120262, 120263, 120264, 120265, 120266, 120267, 120268, 120269, 120270, 120271,
                          120272, 120273, 120274, 120275, 120276, 120277, 120278, 120279, 120280, 120281, 120282,
                          120283, 120284, 120285, 120286, 120287, 120288, 120289, 120290, 120291, 120292, 120293,
                          120294, 120295, 120296, 120297, 120298, 120299, 120300, 120301, 120302, 120303, 120304,
                          120305, 120306, 120307, 120308, 120309, 120310, 120311, 120312, 120313, 120314, 120315,
                          120316, 120317, 120318, 120319, 120320, 120321, 120322, 120323, 120324, 120325, 120326,
                          120327, 120328, 120329, 120330, 120331, 120332, 120333, 120334, 120335, 120336, 120337,
                          120338, 120339, 120340, 120341, 120342, 120343, 120344, 120345, 120346, 120347, 120348,
                          120349, 120350, 120351, 120352, 120353, 120354, 120355, 120356, 120357, 120358, 120359,
                          120360, 120361, 120362, 120363, 120364, 120365, 120366, 120367, 120368, 120369, 120370,
                          120371, 120372, 120373, 120374, 120375, 120376, 120377, 120378, 120379, 120380, 120381,
                          120382, 120383, 120384, 120385, 120386, 120387, 120388, 120389, 120390, 120391, 120392,
                          120393, 120394, 120395, 120396, 120397, 120398, 120399, 120400, 120401, 120402, 120403,
                          120404, 120405, 120406, 120407, 120408, 120409, 120410, 120411, 120412, 120413, 120414,
                          120415, 120416, 120417, 120418, 120419, 120420, 120421, 120422, 120423, 120424, 120425,
                          120426, 120427, 120428, 120429, 120430, 120431, 120432, 120433, 120434, 120435, 120436,
                          120437, 120438, 120439, 120440, 120441, 120442, 120443, 120444, 120445, 120446, 120447,
                          120448, 120449, 120450, 120451, 120452, 120453, 120454, 120455, 120456, 120457, 120458,
                          120459, 120460, 120461, 120462, 120463, 120464, 120465, 120466, 120467, 120468, 120469,
                          120470, 120471, 120472, 120473, 120474, 120475, 120476, 120477, 120478, 120479, 120480,
                          120481, 120482, 120483, 120484, 120485, 120488, 120489, 120490, 120491, 120492, 120493,
                          120494, 120495, 120496, 120497, 120498, 120499, 120500, 120501, 120502, 120503, 120504,
                          120505, 120506, 120507, 120508, 120509, 120510, 120511, 120512, 120513, 120514, 120515,
                          120516, 120517, 120518, 120519, 120520, 120521, 120522, 120523, 120524, 120525, 120526,
                          120527, 120528, 120529, 120530, 120531, 120532, 120533, 120534, 120535, 120536, 120537,
                          120538, 120539, 120540, 120541, 120542, 120543, 120544, 120545, 120546, 120547, 120548,
                          120549, 120550, 120551, 120552, 120553, 120554, 120555, 120556, 120557, 120558, 120559,
                          120560, 120561, 120562, 120563, 120564, 120565, 120566, 120567, 120568, 120569, 120570,
                          120571, 120572, 120573, 120574, 120575, 120576, 120577, 120578, 120579, 120580, 120581,
                          120582, 120583, 120584, 120585, 120586, 120587, 120588, 120589, 120590, 120591, 120592,
                          120593, 120594, 120595, 120596, 120597, 120598, 120599, 120600, 120601, 120602, 120603,
                          120604, 120605, 120606, 120607, 120608, 120609, 120610, 120611, 120612, 120613, 120614,
                          120615, 120616, 120617, 120618, 120619, 120620, 120621, 120622, 120623, 120624, 120625,
                          120626, 120627, 120628, 120629, 120630, 120631, 120632, 120633, 120634, 120635, 120636,
                          120637, 120638, 120639, 120640, 120641, 120642, 120643, 120644, 120645, 120646, 120647,
                          120648, 120649, 120650, 120651, 120652, 120653, 120654, 120655, 120656, 120657, 120658,
                          120659, 120660, 120661, 120662, 120663, 120664, 120665, 120666, 120667, 120668, 120669,
                          120670, 120671, 120672, 120673, 120674, 120675, 120676, 120677, 120678, 120679, 120680,
                          120681, 120682, 120683, 120684, 120685, 120686, 120687, 120688, 120689, 120690, 120691,
                          120692, 120693, 120694, 120695, 120696, 120697, 120698, 120699, 120700, 120701, 120702,
                          120703, 120704, 120705, 120706, 120707, 120708, 120709, 120710, 120711, 120712, 120713,
                          120714, 120715, 120716, 120717, 120718, 120719, 120720, 120721, 120722, 120723, 120724,
                          120725, 120726, 120727, 120728, 120729, 120730, 120731, 120732, 120733, 120734, 120735,
                          120736, 120737, 120738, 120739, 120740, 120741, 120742, 120743, 120744, 120745, 120746,
                          120747, 120748, 120749, 120750, 120751, 120752, 120753, 120754, 120755, 120756, 120757,
                          120758, 120759, 120760, 120761, 120762, 120763, 120764, 120765, 120766, 120767, 120768,
                          120769, 120770, 120771, 120772, 120773, 120774, 120775, 120776, 120777, 120778, 120779,
                          120782, 120783, 120784, 120785, 120786, 120787, 120788, 120789, 120790, 120791, 120792,
                          120793, 120794, 120795, 120796, 120797, 120798, 120799, 120800, 120801, 120802, 120803,
                          120804, 120805, 120806, 120807, 120808, 120809, 120810, 120811, 120812, 120813, 120814,
                          120815, 120816, 120817, 120818, 120819, 120820, 120821, 120822, 120823, 120824, 120825,
                          120826, 120827, 120828, 120829, 120830, 120831, 194560, 194561, 194562, 194563, 194564,
                          194565, 194566, 194567, 194568, 194569, 194570, 194571, 194572, 194573, 194574, 194575,
                          194576, 194577, 194578, 194579, 194580, 194581, 194582, 194583, 194584, 194585, 194586,
                          194587, 194588, 194589, 194590, 194591, 194592, 194593, 194594, 194595, 194596, 194597,
                          194598, 194599, 194600, 194601, 194602, 194603, 194604, 194605, 194606, 194607, 194608,
                          194609, 194610, 194611, 194612, 194613, 194614, 194615, 194616, 194617, 194618, 194619,
                          194620, 194621, 194622, 194623, 194624, 194625, 194626, 194627, 194628, 194629, 194630,
                          194631, 194632, 194633, 194634, 194635, 194636, 194637, 194638, 194639, 194640, 194641,
                          194642, 194643, 194644, 194645, 194646, 194647, 194648, 194649, 194650, 194651, 194652,
                          194653, 194654, 194655, 194656, 194657, 194658, 194659, 194660, 194661, 194662, 194663,
                          194664, 194665, 194666, 194667, 194668, 194669, 194670, 194671, 194672, 194673, 194674,
                          194675, 194676, 194677, 194678, 194679, 194680, 194681, 194682, 194683, 194684, 194685,
                          194686, 194687, 194688, 194689, 194690, 194691, 194692, 194693, 194694, 194695, 194696,
                          194697, 194698, 194699, 194700, 194701, 194702, 194703, 194704, 194705, 194706, 194707,
                          194708, 194709, 194710, 194711, 194712, 194713, 194714, 194715, 194716, 194717, 194718,
                          194719, 194720, 194721, 194722, 194723, 194724, 194725, 194726, 194727, 194728, 194729,
                          194730, 194731, 194732, 194733, 194734, 194735, 194736, 194737, 194738, 194739, 194740,
                          194741, 194742, 194743, 194744, 194745, 194746, 194747, 194748, 194749, 194750, 194751,
                          194752, 194753, 194754, 194755, 194756, 194757, 194758, 194759, 194760, 194761, 194762,
                          194763, 194764, 194765, 194766, 194767, 194768, 194769, 194770, 194771, 194772, 194773,
                          194774, 194775, 194776, 194777, 194778, 194779, 194780, 194781, 194782, 194783, 194784,
                          194785, 194786, 194787, 194788, 194789, 194790, 194791, 194792, 194793, 194794, 194795,
                          194796, 194797, 194798, 194799, 194800, 194801, 194802, 194803, 194804, 194805, 194806,
                          194807, 194808, 194809, 194810, 194811, 194812, 194813, 194814, 194815, 194816, 194817,
                          194818, 194819, 194820, 194821, 194822, 194823, 194824, 194825, 194826, 194827, 194828,
                          194829, 194830, 194831, 194832, 194833, 194834, 194835, 194836, 194837, 194838, 194839,
                          194840, 194841, 194842, 194843, 194844, 194845, 194846, 194847, 194848, 194849, 194850,
                          194851, 194852, 194853, 194854, 194855, 194856, 194857, 194858, 194859, 194860, 194861,
                          194862, 194863, 194864, 194865, 194866, 194867, 194868, 194869, 194870, 194871, 194872,
                          194873, 194874, 194875, 194876, 194877, 194878, 194879, 194880, 194881, 194882, 194883,
                          194884, 194885, 194886, 194887, 194888, 194889, 194890, 194891, 194892, 194893, 194894,
                          194895, 194896, 194897, 194898, 194899, 194900, 194901, 194902, 194903, 194904, 194905,
                          194906, 194907, 194908, 194909, 194910, 194911, 194912, 194913, 194914, 194915, 194916,
                          194917, 194918, 194919, 194920, 194921, 194922, 194923, 194924, 194925, 194926, 194927,
                          194928, 194929, 194930, 194931, 194932, 194933, 194934, 194935, 194936, 194937, 194938,
                          194939, 194940, 194941, 194942, 194943, 194944, 194945, 194946, 194947, 194948, 194949,
                          194950, 194951, 194952, 194953, 194954, 194955, 194956, 194957, 194958, 194959, 194960,
                          194961, 194962, 194963, 194964, 194965, 194966, 194967, 194968, 194969, 194970, 194971,
                          194972, 194973, 194974, 194975, 194976, 194977, 194978, 194979, 194980, 194981, 194982,
                          194983, 194984, 194985, 194986, 194987, 194988, 194989, 194990, 194991, 194992, 194993,
                          194994, 194995, 194996, 194997, 194998, 194999, 195000, 195001, 195002, 195003, 195004,
                          195005, 195006, 195007, 195008, 195009, 195010, 195011, 195012, 195013, 195014, 195015,
                          195016, 195017, 195018, 195019, 195020, 195021, 195022, 195023, 195024, 195025, 195026,
                          195027, 195028, 195029, 195030, 195031, 195032, 195033, 195034, 195035, 195036, 195037,
                          195038, 195039, 195040, 195041, 195042, 195043, 195044, 195045, 195046, 195047, 195048,
                          195049, 195050, 195051, 195052, 195053, 195054, 195055, 195056, 195057, 195058, 195059,
                          195060, 195061, 195062, 195063, 195064, 195065, 195066, 195067, 195068, 195069, 195070,
                          195071, 195072, 195073, 195074, 195075, 195076, 195077, 195078, 195079, 195080, 195081,
                          195082, 195083, 195084, 195085, 195086, 195087, 195088, 195089, 195090, 195091, 195092,
                          195093, 195094, 195095, 195096, 195097, 195098, 195099, 195100, 195101};
    }

    private static int[][] getDecompData1() {
        return new int[][] { {32}, {32, 776}, {97}, {32, 772}, {50}, {51}, {32, 769}, {956}, {32, 807}, {49}, {111},
                            {49, 8260, 52}, {49, 8260, 50}, {51, 8260, 52}, {65, 768}, {65, 769}, {65, 770}, {65, 771},
                            {65, 776}, {65, 778}, {67, 807}, {69, 768}, {69, 769}, {69, 770}, {69, 776}, {73, 768},
                            {73, 769}, {73, 770}, {73, 776}, {78, 771}, {79, 768}, {79, 769}, {79, 770}, {79, 771},
                            {79, 776}, {85, 768}, {85, 769}, {85, 770}, {85, 776}, {89, 769}, {97, 768}, {97, 769},
                            {97, 770}, {97, 771}, {97, 776}, {97, 778}, {99, 807}, {101, 768}, {101, 769}, {101, 770},
                            {101, 776}, {105, 768}, {105, 769}, {105, 770}, {105, 776}, {110, 771}, {111, 768},
                            {111, 769}, {111, 770}, {111, 771}, {111, 776}, {117, 768}, {117, 769}, {117, 770},
                            {117, 776}, {121, 769}, {121, 776}, {65, 772}, {97, 772}, {65, 774}, {97, 774}, {65, 808},
                            {97, 808}, {67, 769}, {99, 769}, {67, 770}, {99, 770}, {67, 775}, {99, 775}, {67, 780},
                            {99, 780}, {68, 780}, {100, 780}, {69, 772}, {101, 772}, {69, 774}, {101, 774}, {69, 775},
                            {101, 775}, {69, 808}, {101, 808}, {69, 780}, {101, 780}, {71, 770}, {103, 770}, {71, 774},
                            {103, 774}, {71, 775}, {103, 775}, {71, 807}, {103, 807}, {72, 770}, {104, 770}, {73, 771},
                            {105, 771}, {73, 772}, {105, 772}, {73, 774}, {105, 774}, {73, 808}, {105, 808}, {73, 775},
                            {73, 74}, {105, 106}, {74, 770}, {106, 770}, {75, 807}, {107, 807}, {76, 769}, {108, 769},
                            {76, 807}, {108, 807}, {76, 780}, {108, 780}, {76, 183}, {108, 183}, {78, 769}, {110, 769},
                            {78, 807}, {110, 807}, {78, 780}, {110, 780}, {700, 110}, {79, 772}, {111, 772}, {79, 774},
                            {111, 774}, {79, 779}, {111, 779}, {82, 769}, {114, 769}, {82, 807}, {114, 807}, {82, 780},
                            {114, 780}, {83, 769}, {115, 769}, {83, 770}, {115, 770}, {83, 807}, {115, 807}, {83, 780},
                            {115, 780}, {84, 807}, {116, 807}, {84, 780}, {116, 780}, {85, 771}, {117, 771}, {85, 772},
                            {117, 772}, {85, 774}, {117, 774}, {85, 778}, {117, 778}, {85, 779}, {117, 779}, {85, 808},
                            {117, 808}, {87, 770}, {119, 770}, {89, 770}, {121, 770}, {89, 776}, {90, 769}, {122, 769},
                            {90, 775}, {122, 775}, {90, 780}, {122, 780}, {115}, {79, 795}, {111, 795}, {85, 795},
                            {117, 795}, {68, 381}, {68, 382}, {100, 382}, {76, 74}, {76, 106}, {108, 106}, {78, 74},
                            {78, 106}, {110, 106}, {65, 780}, {97, 780}, {73, 780}, {105, 780}, {79, 780}, {111, 780},
                            {85, 780}, {117, 780}, {220, 772}, {252, 772}, {220, 769}, {252, 769}, {220, 780},
                            {252, 780}, {220, 768}, {252, 768}, {196, 772}, {228, 772}, {550, 772}, {551, 772},
                            {198, 772}, {230, 772}, {71, 780}, {103, 780}, {75, 780}, {107, 780}, {79, 808},
                            {111, 808}, {490, 772}, {491, 772}, {439, 780}, {658, 780}, {106, 780}, {68, 90},
                            {68, 122}, {100, 122}, {71, 769}, {103, 769}, {78, 768}, {110, 768}, {197, 769},
                            {229, 769}, {198, 769}, {230, 769}, {216, 769}, {248, 769}, {65, 783}, {97, 783},
                            {65, 785}, {97, 785}, {69, 783}, {101, 783}, {69, 785}, {101, 785}, {73, 783}, {105, 783},
                            {73, 785}, {105, 785}, {79, 783}, {111, 783}, {79, 785}, {111, 785}, {82, 783}, {114, 783},
                            {82, 785}, {114, 785}, {85, 783}, {117, 783}, {85, 785}, {117, 785}, {83, 806}, {115, 806},
                            {84, 806}, {116, 806}, {72, 780}, {104, 780}, {65, 775}, {97, 775}, {69, 807}, {101, 807},
                            {214, 772}, {246, 772}, {213, 772}, {245, 772}, {79, 775}, {111, 775}, {558, 772},
                            {559, 772}, {89, 772}, {121, 772}, {104}, {614}, {106}, {114}, {633}, {635}, {641}, {119},
                            {121}, {32, 774}, {32, 775}, {32, 778}, {32, 808}, {32, 771}, {32, 779}, {611}, {108},
                            {115}, {120}, {661}, {768}, {769}, {787}, {776, 769}, {697}, {32, 837}, {59}, {32, 769},
                            {168, 769}, {913, 769}, {183}, {917, 769}, {919, 769}, {921, 769}, {927, 769}, {933, 769},
                            {937, 769}, {970, 769}, {921, 776}, {933, 776}, {945, 769}, {949, 769}, {951, 769},
                            {953, 769}, {971, 769}, {953, 776}, {965, 776}, {959, 769}, {965, 769}, {969, 769}, {946},
                            {952}, {933}, {978, 769}, {978, 776}, {966}, {960}, {954}, {961}, {962}, {920}, {949},
                            {931}, {1045, 768}, {1045, 776}, {1043, 769}, {1030, 776}, {1050, 769}, {1048, 768},
                            {1059, 774}, {1048, 774}, {1080, 774}, {1077, 768}, {1077, 776}, {1075, 769}, {1110, 776},
                            {1082, 769}, {1080, 768}, {1091, 774}, {1140, 783}, {1141, 783}, {1046, 774}, {1078, 774},
                            {1040, 774}, {1072, 774}, {1040, 776}, {1072, 776}, {1045, 774}, {1077, 774}, {1240, 776},
                            {1241, 776}, {1046, 776}, {1078, 776}, {1047, 776}, {1079, 776}, {1048, 772}, {1080, 772},
                            {1048, 776}, {1080, 776}, {1054, 776}, {1086, 776}, {1256, 776}, {1257, 776}, {1069, 776},
                            {1101, 776}, {1059, 772}, {1091, 772}, {1059, 776}, {1091, 776}, {1059, 779}, {1091, 779},
                            {1063, 776}, {1095, 776}, {1067, 776}, {1099, 776}, {1381, 1410}, {1575, 1619},
                            {1575, 1620}, {1608, 1620}, {1575, 1621}, {1610, 1620}, {1575, 1652}, {1608, 1652},
                            {1735, 1652}, {1610, 1652}, {1749, 1620}, {1729, 1620}, {1746, 1620}, {2344, 2364},
                            {2352, 2364}, {2355, 2364}, {2325, 2364}, {2326, 2364}, {2327, 2364}, {2332, 2364},
                            {2337, 2364}, {2338, 2364}, {2347, 2364}, {2351, 2364}, {2503, 2494}, {2503, 2519},
                            {2465, 2492}, {2466, 2492}, {2479, 2492}, {2610, 2620}, {2616, 2620}, {2582, 2620},
                            {2583, 2620}, {2588, 2620}, {2603, 2620}, {2887, 2902}, {2887, 2878}, {2887, 2903},
                            {2849, 2876}, {2850, 2876}, {2962, 3031}, {3014, 3006}, {3015, 3006}, {3014, 3031},
                            {3142, 3158}, {3263, 3285}, {3270, 3285}, {3270, 3286}, {3270, 3266}, {3274, 3285},
                            {3398, 3390}, {3399, 3390}, {3398, 3415}, {3545, 3530}, {3545, 3535}, {3548, 3530},
                            {3545, 3551}, {3661, 3634}, {3789, 3762}, {3755, 3737}, {3755, 3745}, {3851}, {3906, 4023},
                            {3916, 4023}, {3921, 4023}, {3926, 4023}, {3931, 4023}, {3904, 4021}, {3953, 3954},
                            {3953, 3956}, {4018, 3968}, {4018, 3969}, {4019, 3968}, {4019, 3969}, {3953, 3968},
                            {3986, 4023}, {3996, 4023}, {4001, 4023}, {4006, 4023}, {4011, 4023}, {3984, 4021},
                            {4133, 4142}, {4316}, {6917, 6965}, {6919, 6965}, {6921, 6965}, {6923, 6965}, {6925, 6965},
                            {6929, 6965}, {6970, 6965}, {6972, 6965}, {6974, 6965}, {6975, 6965}, {6978, 6965}, {65},
                            {198}, {66}, {68}, {69}, {398}, {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79},
                            {546}, {80}, {82}, {84}, {85}, {87}, {97}, {592}, {593}, {7426}, {98}, {100}, {101}, {601},
                            {603}, {604}, {103}, {107}, {109}, {331}, {111}, {596}, {7446}, {7447}, {112}, {116},
                            {117}, {7453}, {623}, {118}, {7461}, {946}, {947}, {948}, {966}, {967}, {105}, {114},
                            {117}, {118}, {946}, {947}, {961}, {966}, {967}, {1085}, {594}, {99}, {597}, {240}, {604},
                            {102}, {607}, {609}, {613}, {616}, {617}, {618}, {7547}, {669}, {621}, {7557}, {671},
                            {625}, {624}, {626}, {627}, {628}, {629}, {632}, {642}, {643}, {427}, {649}, {650}, {7452},
                            {651}, {652}, {122}, {656}, {657}, {658}, {952}, {65, 805}, {97, 805}, {66, 775},
                            {98, 775}, {66, 803}, {98, 803}, {66, 817}, {98, 817}, {199, 769}, {231, 769}, {68, 775},
                            {100, 775}, {68, 803}, {100, 803}, {68, 817}, {100, 817}, {68, 807}, {100, 807}, {68, 813},
                            {100, 813}, {274, 768}, {275, 768}, {274, 769}, {275, 769}, {69, 813}, {101, 813},
                            {69, 816}, {101, 816}, {552, 774}, {553, 774}, {70, 775}, {102, 775}, {71, 772},
                            {103, 772}, {72, 775}, {104, 775}, {72, 803}, {104, 803}, {72, 776}, {104, 776}, {72, 807},
                            {104, 807}, {72, 814}, {104, 814}, {73, 816}, {105, 816}, {207, 769}, {239, 769},
                            {75, 769}, {107, 769}, {75, 803}, {107, 803}, {75, 817}, {107, 817}, {76, 803}, {108, 803},
                            {7734, 772}, {7735, 772}, {76, 817}, {108, 817}, {76, 813}, {108, 813}, {77, 769},
                            {109, 769}, {77, 775}, {109, 775}, {77, 803}, {109, 803}, {78, 775}, {110, 775}, {78, 803},
                            {110, 803}, {78, 817}, {110, 817}, {78, 813}, {110, 813}, {213, 769}, {245, 769},
                            {213, 776}, {245, 776}, {332, 768}, {333, 768}, {332, 769}, {333, 769}, {80, 769},
                            {112, 769}, {80, 775}, {112, 775}, {82, 775}, {114, 775}, {82, 803}, {114, 803},
                            {7770, 772}, {7771, 772}, {82, 817}, {114, 817}, {83, 775}, {115, 775}, {83, 803},
                            {115, 803}, {346, 775}, {347, 775}, {352, 775}, {353, 775}, {7778, 775}, {7779, 775},
                            {84, 775}, {116, 775}, {84, 803}, {116, 803}, {84, 817}, {116, 817}, {84, 813}, {116, 813},
                            {85, 804}, {117, 804}, {85, 816}, {117, 816}, {85, 813}, {117, 813}, {360, 769},
                            {361, 769}, {362, 776}, {363, 776}, {86, 771}, {118, 771}, {86, 803}, {118, 803},
                            {87, 768}, {119, 768}, {87, 769}, {119, 769}, {87, 776}, {119, 776}, {87, 775}, {119, 775},
                            {87, 803}, {119, 803}, {88, 775}, {120, 775}, {88, 776}, {120, 776}, {89, 775}, {121, 775},
                            {90, 770}, {122, 770}, {90, 803}, {122, 803}, {90, 817}, {122, 817}, {104, 817},
                            {116, 776}, {119, 778}, {121, 778}, {97, 702}, {383, 775}, {65, 803}, {97, 803}, {65, 777},
                            {97, 777}, {194, 769}, {226, 769}, {194, 768}, {226, 768}, {194, 777}, {226, 777},
                            {194, 771}, {226, 771}, {7840, 770}, {7841, 770}, {258, 769}, {259, 769}, {258, 768},
                            {259, 768}, {258, 777}, {259, 777}, {258, 771}, {259, 771}, {7840, 774}, {7841, 774},
                            {69, 803}, {101, 803}, {69, 777}, {101, 777}, {69, 771}, {101, 771}, {202, 769},
                            {234, 769}, {202, 768}, {234, 768}, {202, 777}, {234, 777}, {202, 771}, {234, 771},
                            {7864, 770}, {7865, 770}, {73, 777}, {105, 777}, {73, 803}, {105, 803}, {79, 803},
                            {111, 803}, {79, 777}, {111, 777}, {212, 769}, {244, 769}, {212, 768}, {244, 768},
                            {212, 777}, {244, 777}, {212, 771}, {244, 771}, {7884, 770}, {7885, 770}, {416, 769},
                            {417, 769}, {416, 768}, {417, 768}, {416, 777}, {417, 777}, {416, 771}, {417, 771},
                            {416, 803}, {417, 803}, {85, 803}, {117, 803}, {85, 777}, {117, 777}, {431, 769},
                            {432, 769}, {431, 768}, {432, 768}, {431, 777}, {432, 777}, {431, 771}, {432, 771},
                            {431, 803}, {432, 803}, {89, 768}, {121, 768}, {89, 803}, {121, 803}, {89, 777},
                            {121, 777}, {89, 771}, {121, 771}, {945, 787}, {945, 788}, {7936, 768}, {7937, 768},
                            {7936, 769}, {7937, 769}, {7936, 834}, {7937, 834}, {913, 787}, {913, 788}, {7944, 768},
                            {7945, 768}, {7944, 769}, {7945, 769}, {7944, 834}, {7945, 834}, {949, 787}, {949, 788},
                            {7952, 768}, {7953, 768}, {7952, 769}, {7953, 769}, {917, 787}, {917, 788}, {7960, 768},
                            {7961, 768}, {7960, 769}, {7961, 769}, {951, 787}, {951, 788}, {7968, 768}, {7969, 768},
                            {7968, 769}, {7969, 769}, {7968, 834}, {7969, 834}, {919, 787}, {919, 788}, {7976, 768},
                            {7977, 768}, {7976, 769}, {7977, 769}, {7976, 834}, {7977, 834}, {953, 787}, {953, 788},
                            {7984, 768}, {7985, 768}, {7984, 769}, {7985, 769}, {7984, 834}, {7985, 834}, {921, 787},
                            {921, 788}, {7992, 768}, {7993, 768}, {7992, 769}, {7993, 769}, {7992, 834}, {7993, 834},
                            {959, 787}, {959, 788}, {8000, 768}, {8001, 768}, {8000, 769}, {8001, 769}, {927, 787},
                            {927, 788}, {8008, 768}, {8009, 768}, {8008, 769}, {8009, 769}, {965, 787}, {965, 788},
                            {8016, 768}, {8017, 768}, {8016, 769}, {8017, 769}, {8016, 834}, {8017, 834}, {933, 788},
                            {8025, 768}, {8025, 769}, {8025, 834}, {969, 787}, {969, 788}, {8032, 768}, {8033, 768},
                            {8032, 769}, {8033, 769}, {8032, 834}, {8033, 834}, {937, 787}, {937, 788}, {8040, 768},
                            {8041, 768}, {8040, 769}, {8041, 769}, {8040, 834}, {8041, 834}, {945, 768}, {940},
                            {949, 768}, {941}, {951, 768}, {942}, {953, 768}, {943}, {959, 768}, {972}, {965, 768},
                            {973}, {969, 768}, {974}, {7936, 837}, {7937, 837}, {7938, 837}, {7939, 837}, {7940, 837},
                            {7941, 837}, {7942, 837}, {7943, 837}, {7944, 837}, {7945, 837}, {7946, 837}, {7947, 837},
                            {7948, 837}, {7949, 837}, {7950, 837}, {7951, 837}, {7968, 837}, {7969, 837}, {7970, 837},
                            {7971, 837}, {7972, 837}, {7973, 837}, {7974, 837}, {7975, 837}, {7976, 837}, {7977, 837},
                            {7978, 837}, {7979, 837}, {7980, 837}, {7981, 837}, {7982, 837}, {7983, 837}, {8032, 837},
                            {8033, 837}, {8034, 837}, {8035, 837}, {8036, 837}, {8037, 837}, {8038, 837}, {8039, 837},
                            {8040, 837}, {8041, 837}, {8042, 837}, {8043, 837}, {8044, 837}, {8045, 837}, {8046, 837},
                            {8047, 837}, {945, 774}, {945, 772}, {8048, 837}, {945, 837}, {940, 837}, {945, 834},
                            {8118, 837}, {913, 774}, {913, 772}, {913, 768}, {902}, {913, 837}, {32, 787}, {953},
                            {32, 787}, {32, 834}, {168, 834}, {8052, 837}, {951, 837}, {942, 837}, {951, 834},
                            {8134, 837}, {917, 768}, {904}, {919, 768}, {905}, {919, 837}, {8127, 768}, {8127, 769},
                            {8127, 834}, {953, 774}, {953, 772}, {970, 768}, {912}, {953, 834}, {970, 834}, {921, 774},
                            {921, 772}, {921, 768}, {906}, {8190, 768}, {8190, 769}, {8190, 834}, {965, 774},
                            {965, 772}, {971, 768}, {944}, {961, 787}, {961, 788}, {965, 834}, {971, 834}, {933, 774},
                            {933, 772}, {933, 768}, {910}, {929, 788}, {168, 768}, {901}, {96}, {8060, 837},
                            {969, 837}, {974, 837}, {969, 834}, {8182, 837}, {927, 768}, {908}, {937, 768}, {911},
                            {937, 837}, {180}, {32, 788}, {8194}, {8195}, {32}, {32}, {32}, {32}, {32}, {32}, {32},
                            {32}, {32}, {8208}, {32, 819}, {46}, {46, 46}, {46, 46, 46}, {32}, {8242, 8242},
                            {8242, 8242, 8242}, {8245, 8245}, {8245, 8245, 8245}, {33, 33}, {32, 773}, {63, 63},
                            {63, 33}, {33, 63}, {8242, 8242, 8242, 8242}, {32}, {48}, {105}, {52}, {53}, {54}, {55},
                            {56}, {57}, {43}, {8722}, {61}, {40}, {41}, {110}, {48}, {49}, {50}, {51}, {52}, {53},
                            {54}, {55}, {56}, {57}, {43}, {8722}, {61}, {40}, {41}, {97}, {101}, {111}, {120}, {601},
                            {82, 115}, {97, 47, 99}, {97, 47, 115}, {67}, {176, 67}, {99, 47, 111}, {99, 47, 117},
                            {400}, {176, 70}, {103}, {72}, {72}, {72}, {104}, {295}, {73}, {73}, {76}, {108}, {78},
                            {78, 111}, {80}, {81}, {82}, {82}, {82}, {83, 77}, {84, 69, 76}, {84, 77}, {90}, {937},
                            {90}, {75}, {197}, {66}, {67}, {101}, {69}, {70}, {77}, {111}, {1488}, {1489}, {1490},
                            {1491}, {105}, {70, 65, 88}, {960}, {947}, {915}, {928}, {8721}, {68}, {100}, {101}, {105},
                            {106}, {49, 8260, 51}, {50, 8260, 51}, {49, 8260, 53}, {50, 8260, 53}, {51, 8260, 53},
                            {52, 8260, 53}, {49, 8260, 54}, {53, 8260, 54}, {49, 8260, 56}, {51, 8260, 56},
                            {53, 8260, 56}, {55, 8260, 56}, {49, 8260}, {73}, {73, 73}, {73, 73, 73}, {73, 86}, {86},
                            {86, 73}, {86, 73, 73}, {86, 73, 73, 73}, {73, 88}, {88}, {88, 73}, {88, 73, 73}, {76},
                            {67}, {68}, {77}, {105}, {105, 105}, {105, 105, 105}, {105, 118}, {118}, {118, 105},
                            {118, 105, 105}, {118, 105, 105, 105}, {105, 120}, {120}, {120, 105}, {120, 105, 105},
                            {108}, {99}, {100}, {109}, {8592, 824}, {8594, 824}, {8596, 824}, {8656, 824}, {8660, 824},
                            {8658, 824}, {8707, 824}, {8712, 824}, {8715, 824}, {8739, 824}, {8741, 824}, {8747, 8747},
                            {8747, 8747, 8747}, {8750, 8750}, {8750, 8750, 8750}, {8764, 824}, {8771, 824},
                            {8773, 824}, {8776, 824}, {61, 824}, {8801, 824}, {8781, 824}, {60, 824}, {62, 824},
                            {8804, 824}, {8805, 824}, {8818, 824}, {8819, 824}, {8822, 824}, {8823, 824}, {8826, 824},
                            {8827, 824}, {8834, 824}, {8835, 824}, {8838, 824}, {8839, 824}, {8866, 824}, {8872, 824},
                            {8873, 824}, {8875, 824}, {8828, 824}, {8829, 824}, {8849, 824}, {8850, 824}, {8882, 824},
                            {8883, 824}, {8884, 824}, {8885, 824}, {12296}, {12297}, {49}, {50}, {51}, {52}, {53},
                            {54}, {55}, {56}, {57}, {49, 48}, {49, 49}, {49, 50}, {49, 51}, {49, 52}, {49, 53},
                            {49, 54}, {49, 55}, {49, 56}, {49, 57}, {50, 48}, {40, 49, 41}, {40, 50, 41}, {40, 51, 41},
                            {40, 52, 41}, {40, 53, 41}, {40, 54, 41}, {40, 55, 41}, {40, 56, 41}, {40, 57, 41},
                            {40, 49, 48, 41}, {40, 49, 49, 41}, {40, 49, 50, 41}, {40, 49, 51, 41}, {40, 49, 52, 41},
                            {40, 49, 53, 41}, {40, 49, 54, 41}, {40, 49, 55, 41}, {40, 49, 56, 41}, {40, 49, 57, 41},
                            {40, 50, 48, 41}, {49, 46}, {50, 46}, {51, 46}, {52, 46}, {53, 46}, {54, 46}, {55, 46},
                            {56, 46}, {57, 46}, {49, 48, 46}, {49, 49, 46}, {49, 50, 46}, {49, 51, 46}, {49, 52, 46},
                            {49, 53, 46}, {49, 54, 46}, {49, 55, 46}, {49, 56, 46}, {49, 57, 46}, {50, 48, 46},
                            {40, 97, 41}, {40, 98, 41}, {40, 99, 41}, {40, 100, 41}, {40, 101, 41}, {40, 102, 41},
                            {40, 103, 41}, {40, 104, 41}, {40, 105, 41}, {40, 106, 41}, {40, 107, 41}, {40, 108, 41},
                            {40, 109, 41}, {40, 110, 41}, {40, 111, 41}, {40, 112, 41}, {40, 113, 41}, {40, 114, 41},
                            {40, 115, 41}, {40, 116, 41}, {40, 117, 41}, {40, 118, 41}, {40, 119, 41}, {40, 120, 41},
                            {40, 121, 41}, {40, 122, 41}, {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74},
                            {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89},
                            {90}, {97}, {98}, {99}, {100}, {101}, {102}, {103}, {104}, {105}, {106}, {107}, {108},
                            {109}, {110}, {111}, {112}, {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121},
                            {122}, {48}, {8747, 8747, 8747, 8747}, {58, 58, 61}, {61, 61}, {61, 61, 61}, {10973, 824},
                            {11617}, {27597}, {40863}, {19968}, {20008}, {20022}, {20031}, {20057}, {20101}, {20108},
                            {20128}, {20154}, {20799}, {20837}, {20843}, {20866}, {20886}, {20907}, {20960}, {20981},
                            {20992}, {21147}, {21241}, {21269}, {21274}, {21304}, {21313}, {21340}, {21353}, {21378},
                            {21430}, {21448}, {21475}, {22231}, {22303}, {22763}, {22786}, {22794}, {22805}, {22823},
                            {22899}, {23376}, {23424}, {23544}, {23567}, {23586}, {23608}, {23662}, {23665}, {24027},
                            {24037}, {24049}, {24062}, {24178}, {24186}, {24191}, {24308}, {24318}, {24331}, {24339},
                            {24400}, {24417}, {24435}, {24515}, {25096}, {25142}, {25163}, {25903}, {25908}, {25991},
                            {26007}, {26020}, {26041}, {26080}, {26085}, {26352}, {26376}, {26408}, {27424}, {27490},
                            {27513}, {27571}, {27595}, {27604}, {27611}, {27663}, {27668}, {27700}, {28779}, {29226},
                            {29238}, {29243}, {29247}, {29255}, {29273}, {29275}, {29356}, {29572}, {29577}, {29916},
                            {29926}, {29976}, {29983}, {29992}, {30000}, {30091}, {30098}, {30326}, {30333}, {30382},
                            {30399}, {30446}, {30683}, {30690}, {30707}, {31034}, {31160}, {31166}, {31348}, {31435},
                            {31481}, {31859}, {31992}, {32566}, {32593}, {32650}, {32701}, {32769}, {32780}, {32786},
                            {32819}, {32895}, {32905}, {33251}, {33258}, {33267}, {33276}, {33292}, {33307}, {33311},
                            {33390}, {33394}, {33400}, {34381}, {34411}, {34880}, {34892}, {34915}, {35198}, {35211},
                            {35282}, {35328}, {35895}, {35910}, {35925}, {35960}, {35997}, {36196}, {36208}, {36275},
                            {36523}, {36554}, {36763}, {36784}, {36789}, {37009}, {37193}, {37318}, {37324}, {37329},
                            {38263}, {38272}, {38428}, {38582}, {38585}, {38632}, {38737}, {38750}, {38754}, {38761},
                            {38859}, {38893}, {38899}, {38913}, {39080}, {39131}, {39135}, {39318}, {39321}, {39340},
                            {39592}, {39640}, {39647}, {39717}, {39727}, {39730}, {39740}, {39770}, {40165}, {40565},
                            {40575}, {40613}, {40635}, {40643}, {40653}, {40657}, {40697}, {40701}, {40718}, {40723},
                            {40736}, {40763}, {40778}, {40786}, {40845}, {40860}, {40864}, {32}, {12306}, {21313},
                            {21316}, {21317}, {12363, 12441}, {12365, 12441}, {12367, 12441}, {12369, 12441},
                            {12371, 12441}, {12373, 12441}, {12375, 12441}, {12377, 12441}, {12379, 12441},
                            {12381, 12441}, {12383, 12441}, {12385, 12441}, {12388, 12441}, {12390, 12441},
                            {12392, 12441}, {12399, 12441}, {12399, 12442}, {12402, 12441}, {12402, 12442},
                            {12405, 12441}, {12405, 12442}, {12408, 12441}, {12408, 12442}, {12411, 12441},
                            {12411, 12442}, {12358, 12441}, {32, 12441}, {32, 12442}, {12445, 12441}, {12424, 12426},
                            {12459, 12441}, {12461, 12441}, {12463, 12441}, {12465, 12441}, {12467, 12441},
                            {12469, 12441}, {12471, 12441}, {12473, 12441}, {12475, 12441}, {12477, 12441},
                            {12479, 12441}, {12481, 12441}, {12484, 12441}, {12486, 12441}, {12488, 12441},
                            {12495, 12441}, {12495, 12442}, {12498, 12441}, {12498, 12442}, {12501, 12441},
                            {12501, 12442}, {12504, 12441}, {12504, 12442}, {12507, 12441}, {12507, 12442},
                            {12454, 12441}, {12527, 12441}, {12528, 12441}, {12529, 12441}, {12530, 12441},
                            {12541, 12441}, {12467, 12488}, {4352}, {4353}, {4522}, {4354}, {4524}, {4525}, {4355},
                            {4356}, {4357}, {4528}, {4529}, {4530}, {4531}, {4532}, {4533}, {4378}, {4358}, {4359},
                            {4360}, {4385}, {4361}, {4362}, {4363}, {4364}, {4365}, {4366}, {4367}, {4368}, {4369},
                            {4370}, {4449}, {4450}, {4451}, {4452}, {4453}, {4454}, {4455}, {4456}, {4457}, {4458},
                            {4459}, {4460}, {4461}, {4462}, {4463}, {4464}, {4465}, {4466}, {4467}, {4468}, {4469},
                            {4448}, {4372}, {4373}, {4551}, {4552}, {4556}, {4558}, {4563}, {4567}, {4569}, {4380},
                            {4573}, {4575}, {4381}, {4382}, {4384}, {4386}, {4387}, {4391}, {4393}, {4395}, {4396},
                            {4397}, {4398}, {4399}, {4402}, {4406}, {4416}, {4423}, {4428}, {4593}, {4594}, {4439},
                            {4440}, {4441}, {4484}, {4485}, {4488}, {4497}, {4498}, {4500}, {4510}, {4513}, {19968},
                            {20108}, {19977}, {22235}, {19978}, {20013}, {19979}, {30002}, {20057}, {19993}, {19969},
                            {22825}, {22320}, {20154}, {40, 4352, 41}, {40, 4354, 41}, {40, 4355, 41}, {40, 4357, 41},
                            {40, 4358, 41}, {40, 4359, 41}, {40, 4361, 41}, {40, 4363, 41}, {40, 4364, 41},
                            {40, 4366, 41}, {40, 4367, 41}, {40, 4368, 41}, {40, 4369, 41}, {40, 4370, 41},
                            {40, 4352, 4449, 41}, {40, 4354, 4449, 41}, {40, 4355, 4449, 41}, {40, 4357, 4449, 41},
                            {40, 4358, 4449, 41}, {40, 4359, 4449, 41}, {40, 4361, 4449, 41}, {40, 4363, 4449, 41},
                            {40, 4364, 4449, 41}, {40, 4366, 4449, 41}, {40, 4367, 4449, 41}, {40, 4368, 4449, 41},
                            {40, 4369, 4449, 41}, {40, 4370, 4449, 41}, {40, 4364, 4462, 41},
                            {40, 4363, 4457, 4364, 4453, 4523, 41}, {40, 4363, 4457, 4370, 4462, 41}, {40, 19968, 41},
                            {40, 20108, 41}, {40, 19977, 41}, {40, 22235, 41}, {40, 20116, 41}, {40, 20845, 41},
                            {40, 19971, 41}, {40, 20843, 41}, {40, 20061, 41}, {40, 21313, 41}, {40, 26376, 41},
                            {40, 28779, 41}, {40, 27700, 41}, {40, 26408, 41}, {40, 37329, 41}, {40, 22303, 41},
                            {40, 26085, 41}, {40, 26666, 41}, {40, 26377, 41}, {40, 31038, 41}, {40, 21517, 41},
                            {40, 29305, 41}, {40, 36001, 41}, {40, 31069, 41}, {40, 21172, 41}, {40, 20195, 41},
                            {40, 21628, 41}, {40, 23398, 41}, {40, 30435, 41}, {40, 20225, 41}, {40, 36039, 41},
                            {40, 21332, 41}, {40, 31085, 41}, {40, 20241, 41}, {40, 33258, 41}, {40, 33267, 41},
                            {80, 84, 69}, {50, 49}, {50, 50}, {50, 51}, {50, 52}, {50, 53}, {50, 54}, {50, 55},
                            {50, 56}, {50, 57}, {51, 48}, {51, 49}, {51, 50}, {51, 51}, {51, 52}, {51, 53}, {4352},
                            {4354}, {4355}, {4357}, {4358}, {4359}, {4361}, {4363}, {4364}, {4366}, {4367}, {4368},
                            {4369}, {4370}, {4352, 4449}, {4354, 4449}, {4355, 4449}, {4357, 4449}, {4358, 4449},
                            {4359, 4449}, {4361, 4449}, {4363, 4449}, {4364, 4449}, {4366, 4449}, {4367, 4449},
                            {4368, 4449}, {4369, 4449}, {4370, 4449}, {4366, 4449, 4535, 4352, 4457},
                            {4364, 4462, 4363, 4468}, {4363, 4462}, {19968}, {20108}, {19977}, {22235}, {20116},
                            {20845}, {19971}, {20843}, {20061}, {21313}, {26376}, {28779}, {27700}, {26408}, {37329},
                            {22303}, {26085}, {26666}, {26377}, {31038}, {21517}, {29305}, {36001}, {31069}, {21172},
                            {31192}, {30007}, {22899}, {36969}, {20778}, {21360}, {27880}, {38917}, {20241}, {20889},
                            {27491}, {19978}, {20013}, {19979}, {24038}, {21491}, {21307}, {23447}, {23398}, {30435},
                            {20225}, {36039}, {21332}, {22812}, {51, 54}, {51, 55}, {51, 56}, {51, 57}, {52, 48},
                            {52, 49}, {52, 50}, {52, 51}, {52, 52}, {52, 53}, {52, 54}, {52, 55}, {52, 56}, {52, 57},
                            {53, 48}, {49, 26376}, {50, 26376}, {51, 26376}, {52, 26376}, {53, 26376}, {54, 26376},
                            {55, 26376}, {56, 26376}, {57, 26376}, {49, 48, 26376}, {49, 49, 26376}, {49, 50, 26376},
                            {72, 103}, {101, 114, 103}, {101, 86}, {76, 84, 68}, {12450}, {12452}, {12454}, {12456},
                            {12458}, {12459}, {12461}, {12463}, {12465}, {12467}, {12469}, {12471}, {12473}, {12475},
                            {12477}, {12479}, {12481}, {12484}, {12486}, {12488}, {12490}, {12491}, {12492}, {12493},
                            {12494}, {12495}, {12498}, {12501}, {12504}, {12507}, {12510}, {12511}, {12512}, {12513},
                            {12514}, {12516}, {12518}, {12520}, {12521}, {12522}, {12523}, {12524}, {12525}, {12527},
                            {12528}, {12529}, {12530}, {12450, 12497, 12540, 12488}, {12450, 12523, 12501, 12449},
                            {12450, 12531, 12506, 12450}, {12450, 12540, 12523}, {12452, 12491, 12531, 12464},
                            {12452, 12531, 12481}, {12454, 12457, 12531}, {12456, 12473, 12463, 12540, 12489},
                            {12456, 12540, 12459, 12540}, {12458, 12531, 12473}, {12458, 12540, 12512},
                            {12459, 12452, 12522}, {12459, 12521, 12483, 12488}, {12459, 12525, 12522, 12540},
                            {12460, 12525, 12531}, {12460, 12531, 12510}, {12462, 12460}, {12462, 12491, 12540},
                            {12461, 12517, 12522, 12540}, {12462, 12523, 12480, 12540}, {12461, 12525},
                            {12461, 12525, 12464, 12521, 12512}, {12461, 12525, 12513, 12540, 12488, 12523},
                            {12461, 12525, 12527, 12483, 12488}, {12464, 12521, 12512},
                            {12464, 12521, 12512, 12488, 12531}, {12463, 12523, 12476, 12452, 12525},
                            {12463, 12525, 12540, 12493}, {12465, 12540, 12473}, {12467, 12523, 12490},
                            {12467, 12540, 12509}, {12469, 12452, 12463, 12523}, {12469, 12531, 12481, 12540, 12512},
                            {12471, 12522, 12531, 12464}, {12475, 12531, 12481}, {12475, 12531, 12488},
                            {12480, 12540, 12473}, {12487, 12471}, {12489, 12523}, {12488, 12531}, {12490, 12494},
                            {12494, 12483, 12488}, {12495, 12452, 12484}, {12497, 12540, 12475, 12531, 12488},
                            {12497, 12540, 12484}, {12496, 12540, 12524, 12523}, {12500, 12450, 12473, 12488, 12523},
                            {12500, 12463, 12523}, {12500, 12467}, {12499, 12523}, {12501, 12449, 12521, 12483, 12489},
                            {12501, 12451, 12540, 12488}, {12502, 12483, 12471, 12455, 12523}, {12501, 12521, 12531},
                            {12504, 12463, 12479, 12540, 12523}, {12506, 12477}, {12506, 12491, 12498},
                            {12504, 12523, 12484}, {12506, 12531, 12473}, {12506, 12540, 12472}, {12505, 12540, 12479},
                            {12509, 12452, 12531, 12488}, {12508, 12523, 12488}, {12507, 12531}, {12509, 12531, 12489},
                            {12507, 12540, 12523}, {12507, 12540, 12531}, {12510, 12452, 12463, 12525},
                            {12510, 12452, 12523}, {12510, 12483, 12495}, {12510, 12523, 12463},
                            {12510, 12531, 12471, 12519, 12531}, {12511, 12463, 12525, 12531}, {12511, 12522},
                            {12511, 12522, 12496, 12540, 12523}, {12513, 12460}, {12513, 12460, 12488, 12531},
                            {12513, 12540, 12488, 12523}, {12516, 12540, 12489}, {12516, 12540, 12523},
                            {12518, 12450, 12531}, {12522, 12483, 12488, 12523}, {12522, 12521}, {12523, 12500, 12540},
                            {12523, 12540, 12502, 12523}, {12524, 12512}, {12524, 12531, 12488, 12466, 12531},
                            {12527, 12483, 12488}, {48, 28857}, {49, 28857}, {50, 28857}, {51, 28857}, {52, 28857},
                            {53, 28857}, {54, 28857}, {55, 28857}, {56, 28857}, {57, 28857}, {49, 48, 28857},
                            {49, 49, 28857}, {49, 50, 28857}, {49, 51, 28857}, {49, 52, 28857}, {49, 53, 28857},
                            {49, 54, 28857}, {49, 55, 28857}, {49, 56, 28857}, {49, 57, 28857}, {50, 48, 28857},
                            {50, 49, 28857}, {50, 50, 28857}, {50, 51, 28857}, {50, 52, 28857}, {104, 80, 97},
                            {100, 97}, {65, 85}, {98, 97, 114}, {111, 86}, {112, 99}, {100, 109}, {100, 109, 178},
                            {100, 109, 179}, {73, 85}, {24179, 25104}, {26157, 21644}, {22823, 27491}, {26126, 27835},
                            {26666, 24335, 20250, 31038}, {112, 65}, {110, 65}, {956, 65}, {109, 65}, {107, 65},
                            {75, 66}, {77, 66}, {71, 66}, {99, 97, 108}, {107, 99, 97, 108}, {112, 70}, {110, 70},
                            {956, 70}, {956, 103}, {109, 103}, {107, 103}, {72, 122}, {107, 72, 122}, {77, 72, 122},
                            {71, 72, 122}, {84, 72, 122}, {956, 8467}, {109, 8467}, {100, 8467}, {107, 8467},
                            {102, 109}, {110, 109}, {956, 109}, {109, 109}, {99, 109}, {107, 109}, {109, 109, 178},
                            {99, 109, 178}, {109, 178}, {107, 109, 178}, {109, 109, 179}, {99, 109, 179}, {109, 179},
                            {107, 109, 179}, {109, 8725, 115}, {109, 8725, 115, 178}, {80, 97}, {107, 80, 97},
                            {77, 80, 97}, {71, 80, 97}, {114, 97, 100}, {114, 97, 100, 8725, 115},
                            {114, 97, 100, 8725, 115, 178}, {112, 115}, {110, 115}, {956, 115}, {109, 115}, {112, 86},
                            {110, 86}, {956, 86}, {109, 86}, {107, 86}, {77, 86}, {112, 87}, {110, 87}, {956, 87},
                            {109, 87}, {107, 87}, {77, 87}, {107, 937}, {77, 937}, {97, 46, 109, 46}, {66, 113},
                            {99, 99}, {99, 100}, {67, 8725, 107, 103}, {67, 111, 46}, {100, 66}, {71, 121}, {104, 97},
                            {72, 80}, {105, 110}, {75, 75}, {75, 77}, {107, 116}, {108, 109}, {108, 110},
                            {108, 111, 103}, {108, 120}, {109, 98}, {109, 105, 108}, {109, 111, 108}, {80, 72},
                            {112, 46, 109, 46}, {80, 80, 77}, {80, 82}, {115, 114}, {83, 118}, {87, 98},
                            {86, 8725, 109}, {65, 8725, 109}, {49, 26085}, {50, 26085}, {51, 26085}, {52, 26085},
                            {53, 26085}, {54, 26085}, {55, 26085}, {56, 26085}, {57, 26085}, {49, 48, 26085},
                            {49, 49, 26085}, {49, 50, 26085}, {49, 51, 26085}, {49, 52, 26085}, {49, 53, 26085},
                            {49, 54, 26085}, {49, 55, 26085}, {49, 56, 26085}, {49, 57, 26085}, {50, 48, 26085},
                            {50, 49, 26085}, {50, 50, 26085}, {50, 51, 26085}, {50, 52, 26085}, {50, 53, 26085},
                            {50, 54, 26085}, {50, 55, 26085}, {50, 56, 26085}, {50, 57, 26085}, {51, 48, 26085},
                            {51, 49, 26085}, {103, 97, 108}, {35912}, {26356}, {36554}, {36040}, {28369}, {20018},
                            {21477}, {40860}, {40860}, {22865}, {37329}, {21895}, {22856}, {25078}, {30313}, {32645},
                            {34367}, {34746}, {35064}, {37007}, {27138}, {27931}, {28889}, {29662}, {33853}, {37226},
                            {39409}, {20098}, {21365}, {27396}, {29211}, {34349}, {40478}, {23888}, {28651}, {34253},
                            {35172}, {25289}, {33240}, {34847}, {24266}, {26391}, {28010}, {29436}, {37070}, {20358},
                            {20919}, {21214}, {25796}, {27347}, {29200}, {30439}, {32769}, {34310}, {34396}, {36335},
                            {38706}, {39791}, {40442}, {30860}, {31103}, {32160}, {33737}, {37636}, {40575}, {35542},
                            {22751}, {24324}, {31840}, {32894}, {29282}, {30922}, {36034}, {38647}, {22744}, {23650},
                            {27155}, {28122}, {28431}, {32047}, {32311}, {38475}, {21202}, {32907}, {20956}, {20940},
                            {31260}, {32190}, {33777}, {38517}, {35712}, {25295}, {27138}, {35582}, {20025}, {23527},
                            {24594}, {29575}, {30064}, {21271}, {30971}, {20415}, {24489}, {19981}, {27852}, {25976},
                            {32034}, {21443}, {22622}, {30465}, {33865}, {35498}, {27578}, {36784}, {27784}, {25342},
                            {33509}, {25504}, {30053}, {20142}, {20841}, {20937}, {26753}, {31975}, {33391}, {35538},
                            {37327}, {21237}, {21570}, {22899}, {24300}, {26053}, {28670}, {31018}, {38317}, {39530},
                            {40599}, {40654}, {21147}, {26310}, {27511}, {36706}, {24180}, {24976}, {25088}, {25754},
                            {28451}, {29001}, {29833}, {31178}, {32244}, {32879}, {36646}, {34030}, {36899}, {37706},
                            {21015}, {21155}, {21693}, {28872}, {35010}, {35498}, {24265}, {24565}, {25467}, {27566},
                            {31806}, {29557}, {20196}, {22265}, {23527}, {23994}, {24604}, {29618}, {29801}, {32666},
                            {32838}, {37428}, {38646}, {38728}, {38936}, {20363}, {31150}, {37300}, {38584}, {24801},
                            {20102}, {20698}, {23534}, {23615}, {26009}, {27138}, {29134}, {30274}, {34044}, {36988},
                            {40845}, {26248}, {38446}, {21129}, {26491}, {26611}, {27969}, {28316}, {29705}, {30041},
                            {30827}, {32016}, {39006}, {20845}, {25134}, {38520}, {20523}, {23833}, {28138}, {36650},
                            {24459}, {24900}, {26647}, {29575}, {38534}, {21033}, {21519}, {23653}, {26131}, {26446},
                            {26792}, {27877}, {29702}, {30178}, {32633}, {35023}, {35041}, {37324}, {38626}, {21311},
                            {28346}, {21533}, {29136}, {29848}, {34298}, {38563}, {40023}, {40607}, {26519}, {28107},
                            {33256}, {31435}, {31520}, {31890}, {29376}, {28825}, {35672}, {20160}, {33590}, {21050},
                            {20999}, {24230}, {25299}, {31958}, {23429}, {27934}, {26292}, {36667}, {34892}, {38477},
                            {35211}, {24275}, {20800}, {21952}, {22618}, {26228}, {20958}, {29482}, {30410}, {31036},
                            {31070}, {31077}, {31119}, {38742}, {31934}, {32701}, {34322}, {35576}, {36920}, {37117},
                            {39151}, {39164}, {39208}, {40372}, {20398}, {20711}, {20813}, {21193}, {21220}, {21329},
                            {21917}, {22022}, {22120}, {22592}, {22696}, {23652}, {23662}, {24724}, {24936}, {24974},
                            {25074}, {25935}, {26082}, {26257}, {26757}, {28023}, {28186}, {28450}, {29038}, {29227},
                            {29730}, {30865}, {31038}, {31049}, {31048}, {31056}, {31062}, {31069}, {31117}, {31118},
                            {31296}, {31361}, {31680}, {32244}, {32265}, {32321}, {32626}, {32773}, {33261}, {33401},
                            {33401}, {33879}, {35088}, {35222}, {35585}, {35641}, {36051}, {36104}, {36790}, {36920},
                            {38627}, {38911}, {38971}, {20006}, {20917}, {20840}, {20352}, {20805}, {20864}, {21191},
                            {21242}, {21917}, {21845}, {21913}, {21986}, {22618}, {22707}, {22852}, {22868}, {23138},
                            {23336}, {24274}, {24281}, {24425}, {24493}, {24792}, {24910}, {24840}, {24974}, {24928},
                            {25074}, {25140}, {25540}, {25628}, {25682}, {25942}, {26228}, {26391},};
    }

    private static int[][] getDecompData2() {
        return new int[][] {
                            {26395},
                            {26454},
                            {27513},
                            {27578},
                            {27969},
                            {28379},
                            {28363},
                            {28450},
                            {28702},
                            {29038},
                            {30631},
                            {29237},
                            {29359},
                            {29482},
                            {29809},
                            {29958},
                            {30011},
                            {30237},
                            {30239},
                            {30410},
                            {30427},
                            {30452},
                            {30538},
                            {30528},
                            {30924},
                            {31409},
                            {31680},
                            {31867},
                            {32091},
                            {32244},
                            {32574},
                            {32773},
                            {33618},
                            {33775},
                            {34681},
                            {35137},
                            {35206},
                            {35222},
                            {35519},
                            {35576},
                            {35531},
                            {35585},
                            {35582},
                            {35565},
                            {35641},
                            {35722},
                            {36104},
                            {36664},
                            {36978},
                            {37273},
                            {37494},
                            {38524},
                            {38627},
                            {38742},
                            {38875},
                            {38911},
                            {38923},
                            {38971},
                            {39698},
                            {40860},
                            {141386},
                            {141380},
                            {144341},
                            {15261},
                            {16408},
                            {16441},
                            {152137},
                            {154832},
                            {163539},
                            {40771},
                            {40846},
                            {102, 102},
                            {102, 105},
                            {102, 108},
                            {102, 102, 105},
                            {102, 102, 108},
                            {383, 116},
                            {115, 116},
                            {1396, 1398},
                            {1396, 1381},
                            {1396, 1387},
                            {1406, 1398},
                            {1396, 1389},
                            {1497, 1460},
                            {1522, 1463},
                            {1506},
                            {1488},
                            {1491},
                            {1492},
                            {1499},
                            {1500},
                            {1501},
                            {1512},
                            {1514},
                            {43},
                            {1513, 1473},
                            {1513, 1474},
                            {64329, 1473},
                            {64329, 1474},
                            {1488, 1463},
                            {1488, 1464},
                            {1488, 1468},
                            {1489, 1468},
                            {1490, 1468},
                            {1491, 1468},
                            {1492, 1468},
                            {1493, 1468},
                            {1494, 1468},
                            {1496, 1468},
                            {1497, 1468},
                            {1498, 1468},
                            {1499, 1468},
                            {1500, 1468},
                            {1502, 1468},
                            {1504, 1468},
                            {1505, 1468},
                            {1507, 1468},
                            {1508, 1468},
                            {1510, 1468},
                            {1511, 1468},
                            {1512, 1468},
                            {1513, 1468},
                            {1514, 1468},
                            {1493, 1465},
                            {1489, 1471},
                            {1499, 1471},
                            {1508, 1471},
                            {1488, 1500},
                            {1649},
                            {1649},
                            {1659},
                            {1659},
                            {1659},
                            {1659},
                            {1662},
                            {1662},
                            {1662},
                            {1662},
                            {1664},
                            {1664},
                            {1664},
                            {1664},
                            {1658},
                            {1658},
                            {1658},
                            {1658},
                            {1663},
                            {1663},
                            {1663},
                            {1663},
                            {1657},
                            {1657},
                            {1657},
                            {1657},
                            {1700},
                            {1700},
                            {1700},
                            {1700},
                            {1702},
                            {1702},
                            {1702},
                            {1702},
                            {1668},
                            {1668},
                            {1668},
                            {1668},
                            {1667},
                            {1667},
                            {1667},
                            {1667},
                            {1670},
                            {1670},
                            {1670},
                            {1670},
                            {1671},
                            {1671},
                            {1671},
                            {1671},
                            {1677},
                            {1677},
                            {1676},
                            {1676},
                            {1678},
                            {1678},
                            {1672},
                            {1672},
                            {1688},
                            {1688},
                            {1681},
                            {1681},
                            {1705},
                            {1705},
                            {1705},
                            {1705},
                            {1711},
                            {1711},
                            {1711},
                            {1711},
                            {1715},
                            {1715},
                            {1715},
                            {1715},
                            {1713},
                            {1713},
                            {1713},
                            {1713},
                            {1722},
                            {1722},
                            {1723},
                            {1723},
                            {1723},
                            {1723},
                            {1728},
                            {1728},
                            {1729},
                            {1729},
                            {1729},
                            {1729},
                            {1726},
                            {1726},
                            {1726},
                            {1726},
                            {1746},
                            {1746},
                            {1747},
                            {1747},
                            {1709},
                            {1709},
                            {1709},
                            {1709},
                            {1735},
                            {1735},
                            {1734},
                            {1734},
                            {1736},
                            {1736},
                            {1655},
                            {1739},
                            {1739},
                            {1733},
                            {1733},
                            {1737},
                            {1737},
                            {1744},
                            {1744},
                            {1744},
                            {1744},
                            {1609},
                            {1609},
                            {1574, 1575},
                            {1574, 1575},
                            {1574, 1749},
                            {1574, 1749},
                            {1574, 1608},
                            {1574, 1608},
                            {1574, 1735},
                            {1574, 1735},
                            {1574, 1734},
                            {1574, 1734},
                            {1574, 1736},
                            {1574, 1736},
                            {1574, 1744},
                            {1574, 1744},
                            {1574, 1744},
                            {1574, 1609},
                            {1574, 1609},
                            {1574, 1609},
                            {1740},
                            {1740},
                            {1740},
                            {1740},
                            {1574, 1580},
                            {1574, 1581},
                            {1574, 1605},
                            {1574, 1609},
                            {1574, 1610},
                            {1576, 1580},
                            {1576, 1581},
                            {1576, 1582},
                            {1576, 1605},
                            {1576, 1609},
                            {1576, 1610},
                            {1578, 1580},
                            {1578, 1581},
                            {1578, 1582},
                            {1578, 1605},
                            {1578, 1609},
                            {1578, 1610},
                            {1579, 1580},
                            {1579, 1605},
                            {1579, 1609},
                            {1579, 1610},
                            {1580, 1581},
                            {1580, 1605},
                            {1581, 1580},
                            {1581, 1605},
                            {1582, 1580},
                            {1582, 1581},
                            {1582, 1605},
                            {1587, 1580},
                            {1587, 1581},
                            {1587, 1582},
                            {1587, 1605},
                            {1589, 1581},
                            {1589, 1605},
                            {1590, 1580},
                            {1590, 1581},
                            {1590, 1582},
                            {1590, 1605},
                            {1591, 1581},
                            {1591, 1605},
                            {1592, 1605},
                            {1593, 1580},
                            {1593, 1605},
                            {1594, 1580},
                            {1594, 1605},
                            {1601, 1580},
                            {1601, 1581},
                            {1601, 1582},
                            {1601, 1605},
                            {1601, 1609},
                            {1601, 1610},
                            {1602, 1581},
                            {1602, 1605},
                            {1602, 1609},
                            {1602, 1610},
                            {1603, 1575},
                            {1603, 1580},
                            {1603, 1581},
                            {1603, 1582},
                            {1603, 1604},
                            {1603, 1605},
                            {1603, 1609},
                            {1603, 1610},
                            {1604, 1580},
                            {1604, 1581},
                            {1604, 1582},
                            {1604, 1605},
                            {1604, 1609},
                            {1604, 1610},
                            {1605, 1580},
                            {1605, 1581},
                            {1605, 1582},
                            {1605, 1605},
                            {1605, 1609},
                            {1605, 1610},
                            {1606, 1580},
                            {1606, 1581},
                            {1606, 1582},
                            {1606, 1605},
                            {1606, 1609},
                            {1606, 1610},
                            {1607, 1580},
                            {1607, 1605},
                            {1607, 1609},
                            {1607, 1610},
                            {1610, 1580},
                            {1610, 1581},
                            {1610, 1582},
                            {1610, 1605},
                            {1610, 1609},
                            {1610, 1610},
                            {1584, 1648},
                            {1585, 1648},
                            {1609, 1648},
                            {32, 1612, 1617},
                            {32, 1613, 1617},
                            {32, 1614, 1617},
                            {32, 1615, 1617},
                            {32, 1616, 1617},
                            {32, 1617, 1648},
                            {1574, 1585},
                            {1574, 1586},
                            {1574, 1605},
                            {1574, 1606},
                            {1574, 1609},
                            {1574, 1610},
                            {1576, 1585},
                            {1576, 1586},
                            {1576, 1605},
                            {1576, 1606},
                            {1576, 1609},
                            {1576, 1610},
                            {1578, 1585},
                            {1578, 1586},
                            {1578, 1605},
                            {1578, 1606},
                            {1578, 1609},
                            {1578, 1610},
                            {1579, 1585},
                            {1579, 1586},
                            {1579, 1605},
                            {1579, 1606},
                            {1579, 1609},
                            {1579, 1610},
                            {1601, 1609},
                            {1601, 1610},
                            {1602, 1609},
                            {1602, 1610},
                            {1603, 1575},
                            {1603, 1604},
                            {1603, 1605},
                            {1603, 1609},
                            {1603, 1610},
                            {1604, 1605},
                            {1604, 1609},
                            {1604, 1610},
                            {1605, 1575},
                            {1605, 1605},
                            {1606, 1585},
                            {1606, 1586},
                            {1606, 1605},
                            {1606, 1606},
                            {1606, 1609},
                            {1606, 1610},
                            {1609, 1648},
                            {1610, 1585},
                            {1610, 1586},
                            {1610, 1605},
                            {1610, 1606},
                            {1610, 1609},
                            {1610, 1610},
                            {1574, 1580},
                            {1574, 1581},
                            {1574, 1582},
                            {1574, 1605},
                            {1574, 1607},
                            {1576, 1580},
                            {1576, 1581},
                            {1576, 1582},
                            {1576, 1605},
                            {1576, 1607},
                            {1578, 1580},
                            {1578, 1581},
                            {1578, 1582},
                            {1578, 1605},
                            {1578, 1607},
                            {1579, 1605},
                            {1580, 1581},
                            {1580, 1605},
                            {1581, 1580},
                            {1581, 1605},
                            {1582, 1580},
                            {1582, 1605},
                            {1587, 1580},
                            {1587, 1581},
                            {1587, 1582},
                            {1587, 1605},
                            {1589, 1581},
                            {1589, 1582},
                            {1589, 1605},
                            {1590, 1580},
                            {1590, 1581},
                            {1590, 1582},
                            {1590, 1605},
                            {1591, 1581},
                            {1592, 1605},
                            {1593, 1580},
                            {1593, 1605},
                            {1594, 1580},
                            {1594, 1605},
                            {1601, 1580},
                            {1601, 1581},
                            {1601, 1582},
                            {1601, 1605},
                            {1602, 1581},
                            {1602, 1605},
                            {1603, 1580},
                            {1603, 1581},
                            {1603, 1582},
                            {1603, 1604},
                            {1603, 1605},
                            {1604, 1580},
                            {1604, 1581},
                            {1604, 1582},
                            {1604, 1605},
                            {1604, 1607},
                            {1605, 1580},
                            {1605, 1581},
                            {1605, 1582},
                            {1605, 1605},
                            {1606, 1580},
                            {1606, 1581},
                            {1606, 1582},
                            {1606, 1605},
                            {1606, 1607},
                            {1607, 1580},
                            {1607, 1605},
                            {1607, 1648},
                            {1610, 1580},
                            {1610, 1581},
                            {1610, 1582},
                            {1610, 1605},
                            {1610, 1607},
                            {1574, 1605},
                            {1574, 1607},
                            {1576, 1605},
                            {1576, 1607},
                            {1578, 1605},
                            {1578, 1607},
                            {1579, 1605},
                            {1579, 1607},
                            {1587, 1605},
                            {1587, 1607},
                            {1588, 1605},
                            {1588, 1607},
                            {1603, 1604},
                            {1603, 1605},
                            {1604, 1605},
                            {1606, 1605},
                            {1606, 1607},
                            {1610, 1605},
                            {1610, 1607},
                            {1600, 1614, 1617},
                            {1600, 1615, 1617},
                            {1600, 1616, 1617},
                            {1591, 1609},
                            {1591, 1610},
                            {1593, 1609},
                            {1593, 1610},
                            {1594, 1609},
                            {1594, 1610},
                            {1587, 1609},
                            {1587, 1610},
                            {1588, 1609},
                            {1588, 1610},
                            {1581, 1609},
                            {1581, 1610},
                            {1580, 1609},
                            {1580, 1610},
                            {1582, 1609},
                            {1582, 1610},
                            {1589, 1609},
                            {1589, 1610},
                            {1590, 1609},
                            {1590, 1610},
                            {1588, 1580},
                            {1588, 1581},
                            {1588, 1582},
                            {1588, 1605},
                            {1588, 1585},
                            {1587, 1585},
                            {1589, 1585},
                            {1590, 1585},
                            {1591, 1609},
                            {1591, 1610},
                            {1593, 1609},
                            {1593, 1610},
                            {1594, 1609},
                            {1594, 1610},
                            {1587, 1609},
                            {1587, 1610},
                            {1588, 1609},
                            {1588, 1610},
                            {1581, 1609},
                            {1581, 1610},
                            {1580, 1609},
                            {1580, 1610},
                            {1582, 1609},
                            {1582, 1610},
                            {1589, 1609},
                            {1589, 1610},
                            {1590, 1609},
                            {1590, 1610},
                            {1588, 1580},
                            {1588, 1581},
                            {1588, 1582},
                            {1588, 1605},
                            {1588, 1585},
                            {1587, 1585},
                            {1589, 1585},
                            {1590, 1585},
                            {1588, 1580},
                            {1588, 1581},
                            {1588, 1582},
                            {1588, 1605},
                            {1587, 1607},
                            {1588, 1607},
                            {1591, 1605},
                            {1587, 1580},
                            {1587, 1581},
                            {1587, 1582},
                            {1588, 1580},
                            {1588, 1581},
                            {1588, 1582},
                            {1591, 1605},
                            {1592, 1605},
                            {1575, 1611},
                            {1575, 1611},
                            {1578, 1580, 1605},
                            {1578, 1581, 1580},
                            {1578, 1581, 1580},
                            {1578, 1581, 1605},
                            {1578, 1582, 1605},
                            {1578, 1605, 1580},
                            {1578, 1605, 1581},
                            {1578, 1605, 1582},
                            {1580, 1605, 1581},
                            {1580, 1605, 1581},
                            {1581, 1605, 1610},
                            {1581, 1605, 1609},
                            {1587, 1581, 1580},
                            {1587, 1580, 1581},
                            {1587, 1580, 1609},
                            {1587, 1605, 1581},
                            {1587, 1605, 1581},
                            {1587, 1605, 1580},
                            {1587, 1605, 1605},
                            {1587, 1605, 1605},
                            {1589, 1581, 1581},
                            {1589, 1581, 1581},
                            {1589, 1605, 1605},
                            {1588, 1581, 1605},
                            {1588, 1581, 1605},
                            {1588, 1580, 1610},
                            {1588, 1605, 1582},
                            {1588, 1605, 1582},
                            {1588, 1605, 1605},
                            {1588, 1605, 1605},
                            {1590, 1581, 1609},
                            {1590, 1582, 1605},
                            {1590, 1582, 1605},
                            {1591, 1605, 1581},
                            {1591, 1605, 1581},
                            {1591, 1605, 1605},
                            {1591, 1605, 1610},
                            {1593, 1580, 1605},
                            {1593, 1605, 1605},
                            {1593, 1605, 1605},
                            {1593, 1605, 1609},
                            {1594, 1605, 1605},
                            {1594, 1605, 1610},
                            {1594, 1605, 1609},
                            {1601, 1582, 1605},
                            {1601, 1582, 1605},
                            {1602, 1605, 1581},
                            {1602, 1605, 1605},
                            {1604, 1581, 1605},
                            {1604, 1581, 1610},
                            {1604, 1581, 1609},
                            {1604, 1580, 1580},
                            {1604, 1580, 1580},
                            {1604, 1582, 1605},
                            {1604, 1582, 1605},
                            {1604, 1605, 1581},
                            {1604, 1605, 1581},
                            {1605, 1581, 1580},
                            {1605, 1581, 1605},
                            {1605, 1581, 1610},
                            {1605, 1580, 1581},
                            {1605, 1580, 1605},
                            {1605, 1582, 1580},
                            {1605, 1582, 1605},
                            {1605, 1580, 1582},
                            {1607, 1605, 1580},
                            {1607, 1605, 1605},
                            {1606, 1581, 1605},
                            {1606, 1581, 1609},
                            {1606, 1580, 1605},
                            {1606, 1580, 1605},
                            {1606, 1580, 1609},
                            {1606, 1605, 1610},
                            {1606, 1605, 1609},
                            {1610, 1605, 1605},
                            {1610, 1605, 1605},
                            {1576, 1582, 1610},
                            {1578, 1580, 1610},
                            {1578, 1580, 1609},
                            {1578, 1582, 1610},
                            {1578, 1582, 1609},
                            {1578, 1605, 1610},
                            {1578, 1605, 1609},
                            {1580, 1605, 1610},
                            {1580, 1581, 1609},
                            {1580, 1605, 1609},
                            {1587, 1582, 1609},
                            {1589, 1581, 1610},
                            {1588, 1581, 1610},
                            {1590, 1581, 1610},
                            {1604, 1580, 1610},
                            {1604, 1605, 1610},
                            {1610, 1581, 1610},
                            {1610, 1580, 1610},
                            {1610, 1605, 1610},
                            {1605, 1605, 1610},
                            {1602, 1605, 1610},
                            {1606, 1581, 1610},
                            {1602, 1605, 1581},
                            {1604, 1581, 1605},
                            {1593, 1605, 1610},
                            {1603, 1605, 1610},
                            {1606, 1580, 1581},
                            {1605, 1582, 1610},
                            {1604, 1580, 1605},
                            {1603, 1605, 1605},
                            {1604, 1580, 1605},
                            {1606, 1580, 1581},
                            {1580, 1581, 1610},
                            {1581, 1580, 1610},
                            {1605, 1580, 1610},
                            {1601, 1605, 1610},
                            {1576, 1581, 1610},
                            {1603, 1605, 1605},
                            {1593, 1580, 1605},
                            {1589, 1605, 1605},
                            {1587, 1582, 1610},
                            {1606, 1580, 1610},
                            {1589, 1604, 1746},
                            {1602, 1604, 1746},
                            {1575, 1604, 1604, 1607},
                            {1575, 1603, 1576, 1585},
                            {1605, 1581, 1605, 1583},
                            {1589, 1604, 1593, 1605},
                            {1585, 1587, 1608, 1604},
                            {1593, 1604, 1610, 1607},
                            {1608, 1587, 1604, 1605},
                            {1589, 1604, 1609},
                            {1589, 1604, 1609, 32, 1575, 1604, 1604, 1607, 32, 1593, 1604, 1610, 1607, 32, 1608, 1587,
                             1604, 1605}, {1580, 1604, 32, 1580, 1604, 1575, 1604, 1607}, {1585, 1740, 1575, 1604},
                            {44}, {12289}, {12290}, {58}, {59}, {33}, {63}, {12310}, {12311}, {8230}, {8229}, {8212},
                            {8211}, {95}, {95}, {40}, {41}, {123}, {125}, {12308}, {12309}, {12304}, {12305}, {12298},
                            {12299}, {12296}, {12297}, {12300}, {12301}, {12302}, {12303}, {91}, {93}, {8254}, {8254},
                            {8254}, {8254}, {95}, {95}, {95}, {44}, {12289}, {46}, {59}, {58}, {63}, {33}, {8212},
                            {40}, {41}, {123}, {125}, {12308}, {12309}, {35}, {38}, {42}, {43}, {45}, {60}, {62}, {61},
                            {92}, {36}, {37}, {64}, {32, 1611}, {1600, 1611}, {32, 1612}, {32, 1613}, {32, 1614},
                            {1600, 1614}, {32, 1615}, {1600, 1615}, {32, 1616}, {1600, 1616}, {32, 1617}, {1600, 1617},
                            {32, 1618}, {1600, 1618}, {1569}, {1570}, {1570}, {1571}, {1571}, {1572}, {1572}, {1573},
                            {1573}, {1574}, {1574}, {1574}, {1574}, {1575}, {1575}, {1576}, {1576}, {1576}, {1576},
                            {1577}, {1577}, {1578}, {1578}, {1578}, {1578}, {1579}, {1579}, {1579}, {1579}, {1580},
                            {1580}, {1580}, {1580}, {1581}, {1581}, {1581}, {1581}, {1582}, {1582}, {1582}, {1582},
                            {1583}, {1583}, {1584}, {1584}, {1585}, {1585}, {1586}, {1586}, {1587}, {1587}, {1587},
                            {1587}, {1588}, {1588}, {1588}, {1588}, {1589}, {1589}, {1589}, {1589}, {1590}, {1590},
                            {1590}, {1590}, {1591}, {1591}, {1591}, {1591}, {1592}, {1592}, {1592}, {1592}, {1593},
                            {1593}, {1593}, {1593}, {1594}, {1594}, {1594}, {1594}, {1601}, {1601}, {1601}, {1601},
                            {1602}, {1602}, {1602}, {1602}, {1603}, {1603}, {1603}, {1603}, {1604}, {1604}, {1604},
                            {1604}, {1605}, {1605}, {1605}, {1605}, {1606}, {1606}, {1606}, {1606}, {1607}, {1607},
                            {1607}, {1607}, {1608}, {1608}, {1609}, {1609}, {1610}, {1610}, {1610}, {1610},
                            {1604, 1570}, {1604, 1570}, {1604, 1571}, {1604, 1571}, {1604, 1573}, {1604, 1573},
                            {1604, 1575}, {1604, 1575}, {33}, {34}, {35}, {36}, {37}, {38}, {39}, {40}, {41}, {42},
                            {43}, {44}, {45}, {46}, {47}, {48}, {49}, {50}, {51}, {52}, {53}, {54}, {55}, {56}, {57},
                            {58}, {59}, {60}, {61}, {62}, {63}, {64}, {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72},
                            {73}, {74}, {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87},
                            {88}, {89}, {90}, {91}, {92}, {93}, {94}, {95}, {96}, {97}, {98}, {99}, {100}, {101},
                            {102}, {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113}, {114},
                            {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {123}, {124}, {125}, {126},
                            {10629}, {10630}, {12290}, {12300}, {12301}, {12289}, {12539}, {12530}, {12449}, {12451},
                            {12453}, {12455}, {12457}, {12515}, {12517}, {12519}, {12483}, {12540}, {12450}, {12452},
                            {12454}, {12456}, {12458}, {12459}, {12461}, {12463}, {12465}, {12467}, {12469}, {12471},
                            {12473}, {12475}, {12477}, {12479}, {12481}, {12484}, {12486}, {12488}, {12490}, {12491},
                            {12492}, {12493}, {12494}, {12495}, {12498}, {12501}, {12504}, {12507}, {12510}, {12511},
                            {12512}, {12513}, {12514}, {12516}, {12518}, {12520}, {12521}, {12522}, {12523}, {12524},
                            {12525}, {12527}, {12531}, {12441}, {12442}, {12644}, {12593}, {12594}, {12595}, {12596},
                            {12597}, {12598}, {12599}, {12600}, {12601}, {12602}, {12603}, {12604}, {12605}, {12606},
                            {12607}, {12608}, {12609}, {12610}, {12611}, {12612}, {12613}, {12614}, {12615}, {12616},
                            {12617}, {12618}, {12619}, {12620}, {12621}, {12622}, {12623}, {12624}, {12625}, {12626},
                            {12627}, {12628}, {12629}, {12630}, {12631}, {12632}, {12633}, {12634}, {12635}, {12636},
                            {12637}, {12638}, {12639}, {12640}, {12641}, {12642}, {12643}, {162}, {163}, {172}, {175},
                            {166}, {165}, {8361}, {9474}, {8592}, {8593}, {8594}, {8595}, {9632}, {9675},
                            {119127, 119141}, {119128, 119141}, {119135, 119150}, {119135, 119151}, {119135, 119152},
                            {119135, 119153}, {119135, 119154}, {119225, 119141}, {119226, 119141}, {119227, 119150},
                            {119228, 119150}, {119227, 119151}, {119228, 119151}, {65}, {66}, {67}, {68}, {69}, {70},
                            {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85},
                            {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99}, {100}, {101}, {102}, {103}, {104}, {105},
                            {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113}, {114}, {115}, {116}, {117}, {118},
                            {119}, {120}, {121}, {122}, {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74},
                            {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89},
                            {90}, {97}, {98}, {99}, {100}, {101}, {102}, {103}, {105}, {106}, {107}, {108}, {109},
                            {110}, {111}, {112}, {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122},
                            {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79},
                            {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99}, {100},
                            {101}, {102}, {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113},
                            {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {65}, {67}, {68}, {71},
                            {74}, {75}, {78}, {79}, {80}, {81}, {83}, {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97},
                            {98}, {99}, {100}, {102}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {112}, {113},
                            {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {65}, {66}, {67}, {68},
                            {69}, {70}, {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83},
                            {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99}, {100}, {101}, {102}, {103},
                            {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113}, {114}, {115}, {116},
                            {117}, {118}, {119}, {120}, {121}, {122}, {65}, {66}, {68}, {69}, {70}, {71}, {74}, {75},
                            {76}, {77}, {78}, {79}, {80}, {81}, {83}, {84}, {85}, {86}, {87}, {88}, {89}, {97}, {98},
                            {99}, {100}, {101}, {102}, {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111},
                            {112}, {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {65}, {66},
                            {68}, {69}, {70}, {71}, {73}, {74}, {75}, {76}, {77}, {79}, {83}, {84}, {85}, {86}, {87},
                            {88}, {89}, {97}, {98}, {99}, {100}, {101}, {102}, {103}, {104}, {105}, {106}, {107},
                            {108}, {109}, {110}, {111}, {112}, {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120},
                            {121}, {122}, {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74}, {75}, {76}, {77},
                            {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97}, {98},
                            {99}, {100}, {101}, {102}, {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111},
                            {112}, {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {65}, {66},
                            {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79}, {80}, {81},
                            {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99}, {100}, {101},
                            {102}, {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113}, {114},
                            {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {65}, {66}, {67}, {68}, {69}, {70},
                            {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85},
                            {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99}, {100}, {101}, {102}, {103}, {104}, {105},
                            {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113}, {114}, {115}, {116}, {117}, {118},
                            {119}, {120}, {121}, {122}, {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74},
                            {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89},
                            {90}, {97}, {98}, {99}, {100}, {101}, {102}, {103}, {104}, {105}, {106}, {107}, {108},
                            {109}, {110}, {111}, {112}, {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121},
                            {122}, {65}, {66}, {67}, {68}, {69}, {70}, {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78},
                            {79}, {80}, {81}, {82}, {83}, {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99},
                            {100}, {101}, {102}, {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111}, {112},
                            {113}, {114}, {115}, {116}, {117}, {118}, {119}, {120}, {121}, {122}, {65}, {66}, {67},
                            {68}, {69}, {70}, {71}, {72}, {73}, {74}, {75}, {76}, {77}, {78}, {79}, {80}, {81}, {82},
                            {83}, {84}, {85}, {86}, {87}, {88}, {89}, {90}, {97}, {98}, {99}, {100}, {101}, {102},
                            {103}, {104}, {105}, {106}, {107}, {108}, {109}, {110}, {111}, {112}, {113}, {114}, {115},
                            {116}, {117}, {118}, {119}, {120}, {121}, {122}, {305}, {567}, {913}, {914}, {915}, {916},
                            {917}, {918}, {919}, {920}, {921}, {922}, {923}, {924}, {925}, {926}, {927}, {928}, {929},
                            {1012}, {931}, {932}, {933}, {934}, {935}, {936}, {937}, {8711}, {945}, {946}, {947},
                            {948}, {949}, {950}, {951}, {952}, {953}, {954}, {955}, {956}, {957}, {958}, {959}, {960},
                            {961}, {962}, {963}, {964}, {965}, {966}, {967}, {968}, {969}, {8706}, {1013}, {977},
                            {1008}, {981}, {1009}, {982}, {913}, {914}, {915}, {916}, {917}, {918}, {919}, {920},
                            {921}, {922}, {923}, {924}, {925}, {926}, {927}, {928}, {929}, {1012}, {931}, {932}, {933},
                            {934}, {935}, {936}, {937}, {8711}, {945}, {946}, {947}, {948}, {949}, {950}, {951}, {952},
                            {953}, {954}, {955}, {956}, {957}, {958}, {959}, {960}, {961}, {962}, {963}, {964}, {965},
                            {966}, {967}, {968}, {969}, {8706}, {1013}, {977}, {1008}, {981}, {1009}, {982}, {913},
                            {914}, {915}, {916}, {917}, {918}, {919}, {920}, {921}, {922}, {923}, {924}, {925}, {926},
                            {927}, {928}, {929}, {1012}, {931}, {932}, {933}, {934}, {935}, {936}, {937}, {8711},
                            {945}, {946}, {947}, {948}, {949}, {950}, {951}, {952}, {953}, {954}, {955}, {956}, {957},
                            {958}, {959}, {960}, {961}, {962}, {963}, {964}, {965}, {966}, {967}, {968}, {969}, {8706},
                            {1013}, {977}, {1008}, {981}, {1009}, {982}, {913}, {914}, {915}, {916}, {917}, {918},
                            {919}, {920}, {921}, {922}, {923}, {924}, {925}, {926}, {927}, {928}, {929}, {1012}, {931},
                            {932}, {933}, {934}, {935}, {936}, {937}, {8711}, {945}, {946}, {947}, {948}, {949}, {950},
                            {951}, {952}, {953}, {954}, {955}, {956}, {957}, {958}, {959}, {960}, {961}, {962}, {963},
                            {964}, {965}, {966}, {967}, {968}, {969}, {8706}, {1013}, {977}, {1008}, {981}, {1009},
                            {982}, {913}, {914}, {915}, {916}, {917}, {918}, {919}, {920}, {921}, {922}, {923}, {924},
                            {925}, {926}, {927}, {928}, {929}, {1012}, {931}, {932}, {933}, {934}, {935}, {936}, {937},
                            {8711}, {945}, {946}, {947}, {948}, {949}, {950}, {951}, {952}, {953}, {954}, {955}, {956},
                            {957}, {958}, {959}, {960}, {961}, {962}, {963}, {964}, {965}, {966}, {967}, {968}, {969},
                            {8706}, {1013}, {977}, {1008}, {981}, {1009}, {982}, {988}, {989}, {48}, {49}, {50}, {51},
                            {52}, {53}, {54}, {55}, {56}, {57}, {48}, {49}, {50}, {51}, {52}, {53}, {54}, {55}, {56},
                            {57}, {48}, {49}, {50}, {51}, {52}, {53}, {54}, {55}, {56}, {57}, {48}, {49}, {50}, {51},
                            {52}, {53}, {54}, {55}, {56}, {57}, {48}, {49}, {50}, {51}, {52}, {53}, {54}, {55}, {56},
                            {57}, {20029}, {20024}, {20033}, {131362}, {20320}, {20398}, {20411}, {20482}, {20602},
                            {20633}, {20711}, {20687}, {13470}, {132666}, {20813}, {20820}, {20836}, {20855}, {132380},
                            {13497}, {20839}, {20877}, {132427}, {20887}, {20900}, {20172}, {20908}, {20917}, {168415},
                            {20981}, {20995}, {13535}, {21051}, {21062}, {21106}, {21111}, {13589}, {21191}, {21193},
                            {21220}, {21242}, {21253}, {21254}, {21271}, {21321}, {21329}, {21338}, {21363}, {21373},
                            {21375}, {21375}, {21375}, {133676}, {28784}, {21450}, {21471}, {133987}, {21483}, {21489},
                            {21510}, {21662}, {21560}, {21576}, {21608}, {21666}, {21750}, {21776}, {21843}, {21859},
                            {21892}, {21892}, {21913}, {21931}, {21939}, {21954}, {22294}, {22022}, {22295}, {22097},
                            {22132}, {20999}, {22766}, {22478}, {22516}, {22541}, {22411}, {22578}, {22577}, {22700},
                            {136420}, {22770}, {22775}, {22790}, {22810}, {22818}, {22882}, {136872}, {136938},
                            {23020}, {23067}, {23079}, {23000}, {23142}, {14062}, {14076}, {23304}, {23358}, {23358},
                            {137672}, {23491}, {23512}, {23527}, {23539}, {138008}, {23551}, {23558}, {24403}, {23586},
                            {14209}, {23648}, {23662}, {23744}, {23693}, {138724}, {23875}, {138726}, {23918}, {23915},
                            {23932}, {24033}, {24034}, {14383}, {24061}, {24104}, {24125}, {24169}, {14434}, {139651},
                            {14460}, {24240}, {24243}, {24246}, {24266}, {172946}, {24318}, {140081}, {140081},
                            {33281}, {24354}, {24354}, {14535}, {144056}, {156122}, {24418}, {24427}, {14563}, {24474},
                            {24525}, {24535}, {24569}, {24705}, {14650}, {14620}, {24724}, {141012}, {24775}, {24904},
                            {24908}, {24910}, {24908}, {24954}, {24974}, {25010}, {24996}, {25007}, {25054}, {25074},
                            {25078}, {25104}, {25115}, {25181}, {25265}, {25300}, {25424}, {142092}, {25405}, {25340},
                            {25448}, {25475}, {25572}, {142321}, {25634}, {25541}, {25513}, {14894}, {25705}, {25726},
                            {25757}, {25719}, {14956}, {25935}, {25964}, {143370}, {26083}, {26360}, {26185}, {15129},
                            {26257}, {15112}, {15076}, {20882}, {20885}, {26368}, {26268}, {32941}, {17369}, {26391},
                            {26395}, {26401}, {26462}, {26451}, {144323}, {15177}, {26618}, {26501}, {26706}, {26757},
                            {144493}, {26766}, {26655}, {26900}, {15261}, {26946}, {27043}, {27114}, {27304}, {145059},
                            {27355}, {15384}, {27425}, {145575}, {27476}, {15438}, {27506}, {27551}, {27578}, {27579},
                            {146061}, {138507}, {146170}, {27726}, {146620}, {27839}, {27853}, {27751}, {27926},
                            {27966}, {28023}, {27969}, {28009}, {28024}, {28037}, {146718}, {27956}, {28207}, {28270},
                            {15667}, {28363}, {28359}, {147153}, {28153}, {28526}, {147294}, {147342}, {28614},
                            {28729}, {28702}, {28699}, {15766}, {28746}, {28797}, {28791}, {28845}, {132389}, {28997},
                            {148067}, {29084}, {148395}, {29224}, {29237}, {29264}, {149000}, {29312}, {29333},
                            {149301}, {149524}, {29562}, {29579}, {16044}, {29605}, {16056}, {16056}, {29767}, {29788},
                            {29809}, {29829}, {29898}, {16155}, {29988}, {150582}, {30014}, {150674}, {30064},
                            {139679}, {30224}, {151457}, {151480}, {151620}, {16380}, {16392}, {30452}, {151795},
                            {151794}, {151833}, {151859}, {30494}, {30495}, {30495}, {30538}, {16441}, {30603},
                            {16454}, {16534}, {152605}, {30798}, {30860}, {30924}, {16611}, {153126}, {31062},
                            {153242}, {153285}, {31119}, {31211}, {16687}, {31296}, {31306}, {31311}, {153980},
                            {154279}, {154279}, {31470}, {16898}, {154539}, {31686}, {31689}, {16935}, {154752},
                            {31954}, {17056}, {31976}, {31971}, {32000}, {155526}, {32099}, {17153}, {32199}, {32258},
                            {32325}, {17204}, {156200}, {156231}, {17241}, {156377}, {32634}, {156478}, {32661},
                            {32762}, {32773}, {156890}, {156963}, {32864}, {157096}, {32880}, {144223}, {17365},
                            {32946}, {33027}, {17419}, {33086}, {23221}, {157607}, {157621}, {144275}, {144284},
                            {33281}, {33284}, {36766}, {17515}, {33425}, {33419}, {33437}, {21171}, {33457}, {33459},
                            {33469}, {33510}, {158524}, {33509}, {33565}, {33635}, {33709}, {33571}, {33725}, {33767},
                            {33879}, {33619}, {33738}, {33740}, {33756}, {158774}, {159083}, {158933}, {17707},
                            {34033}, {34035}, {34070}, {160714}, {34148}, {159532}, {17757}, {17761}, {159665},
                            {159954}, {17771}, {34384}, {34396}, {34407}, {34409}, {34473}, {34440}, {34574}, {34530},
                            {34681}, {34600}, {34667}, {34694}, {17879}, {34785}, {34817}, {17913}, {34912}, {34915},
                            {161383}, {35031}, {35038}, {17973}, {35066}, {13499}, {161966}, {162150}, {18110},
                            {18119}, {35488}, {35565}, {35722}, {35925}, {162984}, {36011}, {36033}, {36123}, {36215},
                            {163631}, {133124}, {36299}, {36284}, {36336}, {133342}, {36564}, {36664}, {165330},
                            {165357}, {37012}, {37105}, {37137}, {165678}, {37147}, {37432}, {37591}, {37592}, {37500},
                            {37881}, {37909}, {166906}, {38283}, {18837}, {38327}, {167287}, {18918}, {38595}, {23986},
                            {38691}, {168261}, {168474}, {19054}, {19062}, {38880}, {168970}, {19122}, {169110},
                            {38923}, {38923}, {38953}, {169398}, {39138}, {19251}, {39209}, {39335}, {39362}, {39422},
                            {19406}, {170800}, {39698}, {40000}, {40189}, {19662}, {19693}, {40295}, {172238}, {19704},
                            {172293}, {172558}, {172689}, {40635}, {19798}, {40697}, {40702}, {40709}, {40719},
                            {40726}, {40763}, {173568}};
    }

    private static int[][] getDecompData() {
        int[][] d1 = getDecompData1();
        int[][] d2 = getDecompData2();
        int[][] d = new int[d1.length + d2.length][];
        System.arraycopy(d1, 0, d, 0, d1.length);
        System.arraycopy(d2, 0, d, d1.length, d2.length);
        return d;
    }

    private static int[] getHangulPairs1() {
        return new int[] {-1409281624, -1409281623, -1409281622, -1409281621, -1409281620, -1409281619, -1409281618,
                          -1409281617, -1409281616, -1409281615, -1409281614, -1409281613, -1409281612, -1409281611,
                          -1409281610, -1409281609, -1409281608, -1409281607, -1409281606, -1409281605, -1409281604,
                          -1409281603, -1409281602, -1409281601, -1409281600, -1409281599, -1409281598, -1407446616,
                          -1407446615, -1407446614, -1407446613, -1407446612, -1407446611, -1407446610, -1407446609,
                          -1407446608, -1407446607, -1407446606, -1407446605, -1407446604, -1407446603, -1407446602,
                          -1407446601, -1407446600, -1407446599, -1407446598, -1407446597, -1407446596, -1407446595,
                          -1407446594, -1407446593, -1407446592, -1407446591, -1407446590, -1405611608, -1405611607,
                          -1405611606, -1405611605, -1405611604, -1405611603, -1405611602, -1405611601, -1405611600,
                          -1405611599, -1405611598, -1405611597, -1405611596, -1405611595, -1405611594, -1405611593,
                          -1405611592, -1405611591, -1405611590, -1405611589, -1405611588, -1405611587, -1405611586,
                          -1405611585, -1405611584, -1405611583, -1405611582, -1403776600, -1403776599, -1403776598,
                          -1403776597, -1403776596, -1403776595, -1403776594, -1403776593, -1403776592, -1403776591,
                          -1403776590, -1403776589, -1403776588, -1403776587, -1403776586, -1403776585, -1403776584,
                          -1403776583, -1403776582, -1403776581, -1403776580, -1403776579, -1403776578, -1403776577,
                          -1403776576, -1403776575, -1403776574, -1401941592, -1401941591, -1401941590, -1401941589,
                          -1401941588, -1401941587, -1401941586, -1401941585, -1401941584, -1401941583, -1401941582,
                          -1401941581, -1401941580, -1401941579, -1401941578, -1401941577, -1401941576, -1401941575,
                          -1401941574, -1401941573, -1401941572, -1401941571, -1401941570, -1401941569, -1401941568,
                          -1401941567, -1401941566, -1400106584, -1400106583, -1400106582, -1400106581, -1400106580,
                          -1400106579, -1400106578, -1400106577, -1400106576, -1400106575, -1400106574, -1400106573,
                          -1400106572, -1400106571, -1400106570, -1400106569, -1400106568, -1400106567, -1400106566,
                          -1400106565, -1400106564, -1400106563, -1400106562, -1400106561, -1400106560, -1400106559,
                          -1400106558, -1398271576, -1398271575, -1398271574, -1398271573, -1398271572, -1398271571,
                          -1398271570, -1398271569, -1398271568, -1398271567, -1398271566, -1398271565, -1398271564,
                          -1398271563, -1398271562, -1398271561, -1398271560, -1398271559, -1398271558, -1398271557,
                          -1398271556, -1398271555, -1398271554, -1398271553, -1398271552, -1398271551, -1398271550,
                          -1396436568, -1396436567, -1396436566, -1396436565, -1396436564, -1396436563, -1396436562,
                          -1396436561, -1396436560, -1396436559, -1396436558, -1396436557, -1396436556, -1396436555,
                          -1396436554, -1396436553, -1396436552, -1396436551, -1396436550, -1396436549, -1396436548,
                          -1396436547, -1396436546, -1396436545, -1396436544, -1396436543, -1396436542, -1394601560,
                          -1394601559, -1394601558, -1394601557, -1394601556, -1394601555, -1394601554, -1394601553,
                          -1394601552, -1394601551, -1394601550, -1394601549, -1394601548, -1394601547, -1394601546,
                          -1394601545, -1394601544, -1394601543, -1394601542, -1394601541, -1394601540, -1394601539,
                          -1394601538, -1394601537, -1394601536, -1394601535, -1394601534, -1392766552, -1392766551,
                          -1392766550, -1392766549, -1392766548, -1392766547, -1392766546, -1392766545, -1392766544,
                          -1392766543, -1392766542, -1392766541, -1392766540, -1392766539, -1392766538, -1392766537,
                          -1392766536, -1392766535, -1392766534, -1392766533, -1392766532, -1392766531, -1392766530,
                          -1392766529, -1392766528, -1392766527, -1392766526, -1390931544, -1390931543, -1390931542,
                          -1390931541, -1390931540, -1390931539, -1390931538, -1390931537, -1390931536, -1390931535,
                          -1390931534, -1390931533, -1390931532, -1390931531, -1390931530, -1390931529, -1390931528,
                          -1390931527, -1390931526, -1390931525, -1390931524, -1390931523, -1390931522, -1390931521,
                          -1390931520, -1390931519, -1390931518, -1389096536, -1389096535, -1389096534, -1389096533,
                          -1389096532, -1389096531, -1389096530, -1389096529, -1389096528, -1389096527, -1389096526,
                          -1389096525, -1389096524, -1389096523, -1389096522, -1389096521, -1389096520, -1389096519,
                          -1389096518, -1389096517, -1389096516, -1389096515, -1389096514, -1389096513, -1389096512,
                          -1389096511, -1389096510, -1387261528, -1387261527, -1387261526, -1387261525, -1387261524,
                          -1387261523, -1387261522, -1387261521, -1387261520, -1387261519, -1387261518, -1387261517,
                          -1387261516, -1387261515, -1387261514, -1387261513, -1387261512, -1387261511, -1387261510,
                          -1387261509, -1387261508, -1387261507, -1387261506, -1387261505, -1387261504, -1387261503,
                          -1387261502, -1385426520, -1385426519, -1385426518, -1385426517, -1385426516, -1385426515,
                          -1385426514, -1385426513, -1385426512, -1385426511, -1385426510, -1385426509, -1385426508,
                          -1385426507, -1385426506, -1385426505, -1385426504, -1385426503, -1385426502, -1385426501,
                          -1385426500, -1385426499, -1385426498, -1385426497, -1385426496, -1385426495, -1385426494,
                          -1383591512, -1383591511, -1383591510, -1383591509, -1383591508, -1383591507, -1383591506,
                          -1383591505, -1383591504, -1383591503, -1383591502, -1383591501, -1383591500, -1383591499,
                          -1383591498, -1383591497, -1383591496, -1383591495, -1383591494, -1383591493, -1383591492,
                          -1383591491, -1383591490, -1383591489, -1383591488, -1383591487, -1383591486, -1381756504,
                          -1381756503, -1381756502, -1381756501, -1381756500, -1381756499, -1381756498, -1381756497,
                          -1381756496, -1381756495, -1381756494, -1381756493, -1381756492, -1381756491, -1381756490,
                          -1381756489, -1381756488, -1381756487, -1381756486, -1381756485, -1381756484, -1381756483,
                          -1381756482, -1381756481, -1381756480, -1381756479, -1381756478, -1379921496, -1379921495,
                          -1379921494, -1379921493, -1379921492, -1379921491, -1379921490, -1379921489, -1379921488,
                          -1379921487, -1379921486, -1379921485, -1379921484, -1379921483, -1379921482, -1379921481,
                          -1379921480, -1379921479, -1379921478, -1379921477, -1379921476, -1379921475, -1379921474,
                          -1379921473, -1379921472, -1379921471, -1379921470, -1378086488, -1378086487, -1378086486,
                          -1378086485, -1378086484, -1378086483, -1378086482, -1378086481, -1378086480, -1378086479,
                          -1378086478, -1378086477, -1378086476, -1378086475, -1378086474, -1378086473, -1378086472,
                          -1378086471, -1378086470, -1378086469, -1378086468, -1378086467, -1378086466, -1378086465,
                          -1378086464, -1378086463, -1378086462, -1376251480, -1376251479, -1376251478, -1376251477,
                          -1376251476, -1376251475, -1376251474, -1376251473, -1376251472, -1376251471, -1376251470,
                          -1376251469, -1376251468, -1376251467, -1376251466, -1376251465, -1376251464, -1376251463,
                          -1376251462, -1376251461, -1376251460, -1376251459, -1376251458, -1376251457, -1376251456,
                          -1376251455, -1376251454, -1374416472, -1374416471, -1374416470, -1374416469, -1374416468,
                          -1374416467, -1374416466, -1374416465, -1374416464, -1374416463, -1374416462, -1374416461,
                          -1374416460, -1374416459, -1374416458, -1374416457, -1374416456, -1374416455, -1374416454,
                          -1374416453, -1374416452, -1374416451, -1374416450, -1374416449, -1374416448, -1374416447,
                          -1374416446, -1372581464, -1372581463, -1372581462, -1372581461, -1372581460, -1372581459,
                          -1372581458, -1372581457, -1372581456, -1372581455, -1372581454, -1372581453, -1372581452,
                          -1372581451, -1372581450, -1372581449, -1372581448, -1372581447, -1372581446, -1372581445,
                          -1372581444, -1372581443, -1372581442, -1372581441, -1372581440, -1372581439, -1372581438,
                          -1370746456, -1370746455, -1370746454, -1370746453, -1370746452, -1370746451, -1370746450,
                          -1370746449, -1370746448, -1370746447, -1370746446, -1370746445, -1370746444, -1370746443,
                          -1370746442, -1370746441, -1370746440, -1370746439, -1370746438, -1370746437, -1370746436,
                          -1370746435, -1370746434, -1370746433, -1370746432, -1370746431, -1370746430, -1368911448,
                          -1368911447, -1368911446, -1368911445, -1368911444, -1368911443, -1368911442, -1368911441,
                          -1368911440, -1368911439, -1368911438, -1368911437, -1368911436, -1368911435, -1368911434,
                          -1368911433, -1368911432, -1368911431, -1368911430, -1368911429, -1368911428, -1368911427,
                          -1368911426, -1368911425, -1368911424, -1368911423, -1368911422, -1367076440, -1367076439,
                          -1367076438, -1367076437, -1367076436, -1367076435, -1367076434, -1367076433, -1367076432,
                          -1367076431, -1367076430, -1367076429, -1367076428, -1367076427, -1367076426, -1367076425,
                          -1367076424, -1367076423, -1367076422, -1367076421, -1367076420, -1367076419, -1367076418,
                          -1367076417, -1367076416, -1367076415, -1367076414, -1365241432, -1365241431, -1365241430,
                          -1365241429, -1365241428, -1365241427, -1365241426, -1365241425, -1365241424, -1365241423,
                          -1365241422, -1365241421, -1365241420, -1365241419, -1365241418, -1365241417, -1365241416,
                          -1365241415, -1365241414, -1365241413, -1365241412, -1365241411, -1365241410, -1365241409,
                          -1365241408, -1365241407, -1365241406, -1363406424, -1363406423, -1363406422, -1363406421,
                          -1363406420, -1363406419, -1363406418, -1363406417, -1363406416, -1363406415, -1363406414,
                          -1363406413, -1363406412, -1363406411, -1363406410, -1363406409, -1363406408, -1363406407,
                          -1363406406, -1363406405, -1363406404, -1363406403, -1363406402, -1363406401, -1363406400,
                          -1363406399, -1363406398, -1361571416, -1361571415, -1361571414, -1361571413, -1361571412,
                          -1361571411, -1361571410, -1361571409, -1361571408, -1361571407, -1361571406, -1361571405,
                          -1361571404, -1361571403, -1361571402, -1361571401, -1361571400, -1361571399, -1361571398,
                          -1361571397, -1361571396, -1361571395, -1361571394, -1361571393, -1361571392, -1361571391,
                          -1361571390, -1359736408, -1359736407, -1359736406, -1359736405, -1359736404, -1359736403,
                          -1359736402, -1359736401, -1359736400, -1359736399, -1359736398, -1359736397, -1359736396,
                          -1359736395, -1359736394, -1359736393, -1359736392, -1359736391, -1359736390, -1359736389,
                          -1359736388, -1359736387, -1359736386, -1359736385, -1359736384, -1359736383, -1359736382,
                          -1357901400, -1357901399, -1357901398, -1357901397, -1357901396, -1357901395, -1357901394,
                          -1357901393, -1357901392, -1357901391, -1357901390, -1357901389, -1357901388, -1357901387,
                          -1357901386, -1357901385, -1357901384, -1357901383, -1357901382, -1357901381, -1357901380,
                          -1357901379, -1357901378, -1357901377, -1357901376, -1357901375, -1357901374, -1356066392,
                          -1356066391, -1356066390, -1356066389, -1356066388, -1356066387, -1356066386, -1356066385,
                          -1356066384, -1356066383, -1356066382, -1356066381, -1356066380, -1356066379, -1356066378,
                          -1356066377, -1356066376, -1356066375, -1356066374, -1356066373, -1356066372, -1356066371,
                          -1356066370, -1356066369, -1356066368, -1356066367, -1356066366, -1354231384, -1354231383,
                          -1354231382, -1354231381, -1354231380, -1354231379, -1354231378, -1354231377, -1354231376,
                          -1354231375, -1354231374, -1354231373, -1354231372, -1354231371, -1354231370, -1354231369,
                          -1354231368, -1354231367, -1354231366, -1354231365, -1354231364, -1354231363, -1354231362,
                          -1354231361, -1354231360, -1354231359, -1354231358, -1352396376, -1352396375, -1352396374,
                          -1352396373, -1352396372, -1352396371, -1352396370, -1352396369, -1352396368, -1352396367,
                          -1352396366, -1352396365, -1352396364, -1352396363, -1352396362, -1352396361, -1352396360,
                          -1352396359, -1352396358, -1352396357, -1352396356, -1352396355, -1352396354, -1352396353,
                          -1352396352, -1352396351, -1352396350, -1350561368, -1350561367, -1350561366, -1350561365,
                          -1350561364, -1350561363, -1350561362, -1350561361, -1350561360, -1350561359, -1350561358,
                          -1350561357, -1350561356, -1350561355, -1350561354, -1350561353, -1350561352, -1350561351,
                          -1350561350, -1350561349, -1350561348, -1350561347, -1350561346, -1350561345, -1350561344,
                          -1350561343, -1350561342, -1348726360, -1348726359, -1348726358, -1348726357, -1348726356,
                          -1348726355, -1348726354, -1348726353, -1348726352, -1348726351, -1348726350, -1348726349,
                          -1348726348, -1348726347, -1348726346, -1348726345, -1348726344, -1348726343, -1348726342,
                          -1348726341, -1348726340, -1348726339, -1348726338, -1348726337, -1348726336, -1348726335,
                          -1348726334, -1346891352, -1346891351, -1346891350, -1346891349, -1346891348, -1346891347,
                          -1346891346, -1346891345, -1346891344, -1346891343, -1346891342, -1346891341, -1346891340,
                          -1346891339, -1346891338, -1346891337, -1346891336, -1346891335, -1346891334, -1346891333,
                          -1346891332, -1346891331, -1346891330, -1346891329, -1346891328, -1346891327, -1346891326,
                          -1345056344, -1345056343, -1345056342, -1345056341, -1345056340, -1345056339, -1345056338,
                          -1345056337, -1345056336, -1345056335, -1345056334, -1345056333, -1345056332, -1345056331,
                          -1345056330, -1345056329, -1345056328, -1345056327, -1345056326, -1345056325, -1345056324,
                          -1345056323, -1345056322, -1345056321, -1345056320, -1345056319, -1345056318, -1343221336,
                          -1343221335, -1343221334, -1343221333, -1343221332, -1343221331, -1343221330, -1343221329,
                          -1343221328, -1343221327, -1343221326, -1343221325, -1343221324, -1343221323, -1343221322,
                          -1343221321, -1343221320, -1343221319, -1343221318, -1343221317, -1343221316, -1343221315,
                          -1343221314, -1343221313, -1343221312, -1343221311, -1343221310, -1341386328, -1341386327,
                          -1341386326, -1341386325, -1341386324, -1341386323, -1341386322, -1341386321, -1341386320,
                          -1341386319, -1341386318, -1341386317, -1341386316, -1341386315, -1341386314, -1341386313,
                          -1341386312, -1341386311, -1341386310, -1341386309, -1341386308, -1341386307, -1341386306,
                          -1341386305, -1341386304, -1341386303, -1341386302, -1339551320, -1339551319, -1339551318,
                          -1339551317, -1339551316, -1339551315, -1339551314, -1339551313, -1339551312, -1339551311,
                          -1339551310, -1339551309, -1339551308, -1339551307, -1339551306, -1339551305, -1339551304,
                          -1339551303, -1339551302, -1339551301, -1339551300, -1339551299, -1339551298, -1339551297,
                          -1339551296, -1339551295, -1339551294, -1337716312, -1337716311, -1337716310, -1337716309,
                          -1337716308, -1337716307, -1337716306, -1337716305, -1337716304, -1337716303, -1337716302,
                          -1337716301, -1337716300, -1337716299, -1337716298, -1337716297, -1337716296, -1337716295,
                          -1337716294, -1337716293, -1337716292, -1337716291, -1337716290, -1337716289, -1337716288,
                          -1337716287, -1337716286, -1335881304, -1335881303, -1335881302, -1335881301, -1335881300,
                          -1335881299, -1335881298, -1335881297, -1335881296, -1335881295, -1335881294, -1335881293,
                          -1335881292, -1335881291, -1335881290, -1335881289, -1335881288, -1335881287, -1335881286,
                          -1335881285, -1335881284, -1335881283, -1335881282, -1335881281, -1335881280, -1335881279,
                          -1335881278, -1334046296, -1334046295, -1334046294, -1334046293, -1334046292, -1334046291,
                          -1334046290, -1334046289, -1334046288, -1334046287, -1334046286, -1334046285, -1334046284,
                          -1334046283, -1334046282, -1334046281, -1334046280, -1334046279, -1334046278, -1334046277,
                          -1334046276, -1334046275, -1334046274, -1334046273, -1334046272, -1334046271, -1334046270,
                          -1332211288, -1332211287, -1332211286, -1332211285, -1332211284, -1332211283, -1332211282,
                          -1332211281, -1332211280, -1332211279, -1332211278, -1332211277, -1332211276, -1332211275,
                          -1332211274, -1332211273, -1332211272, -1332211271, -1332211270, -1332211269, -1332211268,
                          -1332211267, -1332211266, -1332211265, -1332211264, -1332211263, -1332211262, -1330376280,
                          -1330376279, -1330376278, -1330376277, -1330376276, -1330376275, -1330376274, -1330376273,
                          -1330376272, -1330376271, -1330376270, -1330376269, -1330376268, -1330376267, -1330376266,
                          -1330376265, -1330376264, -1330376263, -1330376262, -1330376261, -1330376260, -1330376259,
                          -1330376258, -1330376257, -1330376256, -1330376255, -1330376254, -1328541272, -1328541271,
                          -1328541270, -1328541269, -1328541268, -1328541267, -1328541266, -1328541265, -1328541264,
                          -1328541263, -1328541262, -1328541261, -1328541260, -1328541259, -1328541258, -1328541257,
                          -1328541256, -1328541255, -1328541254, -1328541253, -1328541252, -1328541251, -1328541250,
                          -1328541249, -1328541248, -1328541247, -1328541246, -1326706264, -1326706263, -1326706262,
                          -1326706261, -1326706260, -1326706259, -1326706258, -1326706257, -1326706256, -1326706255,
                          -1326706254, -1326706253, -1326706252, -1326706251, -1326706250, -1326706249, -1326706248,
                          -1326706247, -1326706246, -1326706245, -1326706244, -1326706243, -1326706242, -1326706241,
                          -1326706240, -1326706239, -1326706238, -1324871256, -1324871255, -1324871254, -1324871253,
                          -1324871252, -1324871251, -1324871250, -1324871249, -1324871248, -1324871247, -1324871246,
                          -1324871245, -1324871244, -1324871243, -1324871242, -1324871241, -1324871240, -1324871239,
                          -1324871238, -1324871237, -1324871236, -1324871235, -1324871234, -1324871233, -1324871232,
                          -1324871231, -1324871230, -1323036248, -1323036247, -1323036246, -1323036245, -1323036244,
                          -1323036243, -1323036242, -1323036241, -1323036240, -1323036239, -1323036238, -1323036237,
                          -1323036236, -1323036235, -1323036234, -1323036233, -1323036232, -1323036231, -1323036230,
                          -1323036229, -1323036228, -1323036227, -1323036226, -1323036225, -1323036224, -1323036223,
                          -1323036222, -1321201240, -1321201239, -1321201238, -1321201237, -1321201236, -1321201235,
                          -1321201234, -1321201233, -1321201232, -1321201231, -1321201230, -1321201229, -1321201228,
                          -1321201227, -1321201226, -1321201225, -1321201224, -1321201223, -1321201222, -1321201221,
                          -1321201220, -1321201219, -1321201218, -1321201217, -1321201216, -1321201215, -1321201214,
                          -1319366232, -1319366231, -1319366230, -1319366229, -1319366228, -1319366227, -1319366226,
                          -1319366225, -1319366224, -1319366223, -1319366222, -1319366221, -1319366220, -1319366219,
                          -1319366218, -1319366217, -1319366216, -1319366215, -1319366214, -1319366213, -1319366212,
                          -1319366211, -1319366210, -1319366209, -1319366208, -1319366207, -1319366206, -1317531224,
                          -1317531223, -1317531222, -1317531221, -1317531220, -1317531219, -1317531218, -1317531217,
                          -1317531216, -1317531215, -1317531214, -1317531213, -1317531212, -1317531211, -1317531210,
                          -1317531209, -1317531208, -1317531207, -1317531206, -1317531205, -1317531204, -1317531203,
                          -1317531202, -1317531201, -1317531200, -1317531199, -1317531198, -1315696216, -1315696215,
                          -1315696214, -1315696213, -1315696212, -1315696211, -1315696210, -1315696209, -1315696208,
                          -1315696207, -1315696206, -1315696205, -1315696204, -1315696203, -1315696202, -1315696201,
                          -1315696200, -1315696199, -1315696198, -1315696197, -1315696196, -1315696195, -1315696194,
                          -1315696193, -1315696192, -1315696191, -1315696190, -1313861208, -1313861207, -1313861206,
                          -1313861205, -1313861204, -1313861203, -1313861202, -1313861201, -1313861200, -1313861199,
                          -1313861198, -1313861197, -1313861196, -1313861195, -1313861194, -1313861193, -1313861192,
                          -1313861191, -1313861190, -1313861189, -1313861188, -1313861187, -1313861186, -1313861185,
                          -1313861184, -1313861183, -1313861182, -1312026200, -1312026199, -1312026198, -1312026197,
                          -1312026196, -1312026195, -1312026194, -1312026193, -1312026192, -1312026191, -1312026190,
                          -1312026189, -1312026188, -1312026187, -1312026186, -1312026185, -1312026184, -1312026183,
                          -1312026182, -1312026181, -1312026180, -1312026179, -1312026178, -1312026177, -1312026176,
                          -1312026175, -1312026174, -1310191192, -1310191191, -1310191190, -1310191189, -1310191188,
                          -1310191187, -1310191186, -1310191185, -1310191184, -1310191183, -1310191182, -1310191181,
                          -1310191180, -1310191179, -1310191178, -1310191177, -1310191176, -1310191175, -1310191174,
                          -1310191173, -1310191172, -1310191171, -1310191170, -1310191169, -1310191168, -1310191167,
                          -1310191166, -1308356184, -1308356183, -1308356182, -1308356181, -1308356180, -1308356179,
                          -1308356178, -1308356177, -1308356176, -1308356175, -1308356174, -1308356173, -1308356172,
                          -1308356171, -1308356170, -1308356169, -1308356168, -1308356167, -1308356166, -1308356165,
                          -1308356164, -1308356163, -1308356162, -1308356161, -1308356160, -1308356159, -1308356158,
                          -1306521176, -1306521175, -1306521174, -1306521173, -1306521172, -1306521171, -1306521170,
                          -1306521169, -1306521168, -1306521167, -1306521166, -1306521165, -1306521164, -1306521163,
                          -1306521162, -1306521161, -1306521160, -1306521159, -1306521158, -1306521157, -1306521156,
                          -1306521155, -1306521154, -1306521153, -1306521152, -1306521151, -1306521150, -1304686168,
                          -1304686167, -1304686166, -1304686165, -1304686164, -1304686163, -1304686162, -1304686161,
                          -1304686160, -1304686159, -1304686158, -1304686157, -1304686156, -1304686155, -1304686154,
                          -1304686153, -1304686152, -1304686151, -1304686150, -1304686149, -1304686148, -1304686147,
                          -1304686146, -1304686145, -1304686144, -1304686143, -1304686142, -1302851160, -1302851159,
                          -1302851158, -1302851157, -1302851156, -1302851155, -1302851154, -1302851153, -1302851152,
                          -1302851151, -1302851150, -1302851149, -1302851148, -1302851147, -1302851146, -1302851145,
                          -1302851144, -1302851143, -1302851142, -1302851141, -1302851140, -1302851139, -1302851138,
                          -1302851137, -1302851136, -1302851135, -1302851134, -1301016152, -1301016151, -1301016150,
                          -1301016149, -1301016148, -1301016147, -1301016146, -1301016145, -1301016144, -1301016143,
                          -1301016142, -1301016141, -1301016140, -1301016139, -1301016138, -1301016137, -1301016136,
                          -1301016135, -1301016134, -1301016133, -1301016132, -1301016131, -1301016130, -1301016129,
                          -1301016128, -1301016127, -1301016126, -1299181144, -1299181143, -1299181142, -1299181141,
                          -1299181140, -1299181139, -1299181138, -1299181137, -1299181136, -1299181135, -1299181134,
                          -1299181133, -1299181132, -1299181131, -1299181130, -1299181129, -1299181128, -1299181127,
                          -1299181126, -1299181125, -1299181124, -1299181123, -1299181122, -1299181121, -1299181120,
                          -1299181119, -1299181118, -1297346136, -1297346135, -1297346134, -1297346133, -1297346132,
                          -1297346131, -1297346130, -1297346129, -1297346128, -1297346127, -1297346126, -1297346125,
                          -1297346124, -1297346123, -1297346122, -1297346121, -1297346120, -1297346119, -1297346118,
                          -1297346117, -1297346116, -1297346115, -1297346114, -1297346113, -1297346112, -1297346111,
                          -1297346110, -1295511128, -1295511127, -1295511126, -1295511125, -1295511124, -1295511123,
                          -1295511122, -1295511121, -1295511120, -1295511119, -1295511118, -1295511117, -1295511116,
                          -1295511115, -1295511114, -1295511113, -1295511112, -1295511111, -1295511110, -1295511109,
                          -1295511108, -1295511107, -1295511106, -1295511105, -1295511104, -1295511103, -1295511102,
                          -1293676120, -1293676119, -1293676118, -1293676117, -1293676116, -1293676115, -1293676114,
                          -1293676113, -1293676112, -1293676111, -1293676110, -1293676109, -1293676108, -1293676107,
                          -1293676106, -1293676105, -1293676104, -1293676103, -1293676102, -1293676101, -1293676100,
                          -1293676099, -1293676098, -1293676097, -1293676096, -1293676095, -1293676094, -1291841112,
                          -1291841111, -1291841110, -1291841109, -1291841108, -1291841107, -1291841106, -1291841105,
                          -1291841104, -1291841103, -1291841102, -1291841101, -1291841100, -1291841099, -1291841098,
                          -1291841097, -1291841096, -1291841095, -1291841094, -1291841093, -1291841092, -1291841091,
                          -1291841090, -1291841089, -1291841088, -1291841087, -1291841086, -1290006104, -1290006103,
                          -1290006102, -1290006101, -1290006100, -1290006099, -1290006098, -1290006097, -1290006096,
                          -1290006095, -1290006094, -1290006093, -1290006092, -1290006091, -1290006090, -1290006089,
                          -1290006088, -1290006087, -1290006086, -1290006085, -1290006084, -1290006083, -1290006082,
                          -1290006081, -1290006080, -1290006079, -1290006078, -1288171096, -1288171095, -1288171094,
                          -1288171093, -1288171092, -1288171091, -1288171090, -1288171089, -1288171088, -1288171087,
                          -1288171086, -1288171085, -1288171084, -1288171083, -1288171082, -1288171081, -1288171080,
                          -1288171079, -1288171078, -1288171077, -1288171076, -1288171075, -1288171074, -1288171073,
                          -1288171072, -1288171071, -1288171070, -1286336088, -1286336087, -1286336086, -1286336085,
                          -1286336084, -1286336083, -1286336082, -1286336081, -1286336080, -1286336079, -1286336078,
                          -1286336077, -1286336076, -1286336075, -1286336074, -1286336073, -1286336072, -1286336071,
                          -1286336070, -1286336069, -1286336068, -1286336067, -1286336066, -1286336065, -1286336064,
                          -1286336063, -1286336062, -1284501080, -1284501079, -1284501078, -1284501077, -1284501076,
                          -1284501075, -1284501074, -1284501073, -1284501072, -1284501071, -1284501070, -1284501069,
                          -1284501068, -1284501067, -1284501066, -1284501065, -1284501064, -1284501063, -1284501062,
                          -1284501061, -1284501060, -1284501059, -1284501058, -1284501057, -1284501056, -1284501055,
                          -1284501054, -1282666072, -1282666071, -1282666070, -1282666069, -1282666068, -1282666067,
                          -1282666066, -1282666065, -1282666064, -1282666063, -1282666062, -1282666061, -1282666060,
                          -1282666059, -1282666058, -1282666057, -1282666056, -1282666055, -1282666054, -1282666053,
                          -1282666052, -1282666051, -1282666050, -1282666049, -1282666048, -1282666047, -1282666046,
                          -1280831064, -1280831063, -1280831062, -1280831061, -1280831060, -1280831059, -1280831058,
                          -1280831057, -1280831056, -1280831055, -1280831054, -1280831053, -1280831052, -1280831051,
                          -1280831050, -1280831049, -1280831048, -1280831047, -1280831046, -1280831045, -1280831044,
                          -1280831043, -1280831042, -1280831041, -1280831040, -1280831039, -1280831038, -1278996056,
                          -1278996055, -1278996054, -1278996053, -1278996052, -1278996051, -1278996050, -1278996049,
                          -1278996048, -1278996047, -1278996046, -1278996045, -1278996044, -1278996043, -1278996042,
                          -1278996041, -1278996040, -1278996039, -1278996038, -1278996037, -1278996036, -1278996035,
                          -1278996034, -1278996033, -1278996032, -1278996031, -1278996030, -1277161048, -1277161047,
                          -1277161046, -1277161045, -1277161044, -1277161043, -1277161042, -1277161041, -1277161040,
                          -1277161039, -1277161038, -1277161037, -1277161036, -1277161035, -1277161034, -1277161033,
                          -1277161032, -1277161031, -1277161030, -1277161029, -1277161028, -1277161027, -1277161026,
                          -1277161025, -1277161024, -1277161023, -1277161022, -1275326040, -1275326039, -1275326038,
                          -1275326037, -1275326036, -1275326035, -1275326034, -1275326033, -1275326032, -1275326031,
                          -1275326030, -1275326029, -1275326028, -1275326027, -1275326026, -1275326025, -1275326024,
                          -1275326023, -1275326022, -1275326021, -1275326020, -1275326019, -1275326018, -1275326017,
                          -1275326016, -1275326015, -1275326014, -1273491032, -1273491031, -1273491030, -1273491029,
                          -1273491028, -1273491027, -1273491026, -1273491025, -1273491024, -1273491023, -1273491022,
                          -1273491021, -1273491020, -1273491019, -1273491018, -1273491017, -1273491016, -1273491015,
                          -1273491014, -1273491013, -1273491012, -1273491011, -1273491010, -1273491009, -1273491008,
                          -1273491007, -1273491006, -1271656024, -1271656023, -1271656022, -1271656021, -1271656020,
                          -1271656019, -1271656018, -1271656017, -1271656016, -1271656015, -1271656014, -1271656013,
                          -1271656012, -1271656011, -1271656010, -1271656009, -1271656008, -1271656007, -1271656006,
                          -1271656005, -1271656004, -1271656003, -1271656002, -1271656001, -1271656000, -1271655999,
                          -1271655998, -1269821016, -1269821015, -1269821014, -1269821013, -1269821012, -1269821011,
                          -1269821010, -1269821009, -1269821008, -1269821007, -1269821006, -1269821005, -1269821004,
                          -1269821003, -1269821002, -1269821001, -1269821000, -1269820999, -1269820998, -1269820997,
                          -1269820996, -1269820995, -1269820994, -1269820993, -1269820992, -1269820991, -1269820990,
                          -1267986008, -1267986007, -1267986006, -1267986005, -1267986004, -1267986003, -1267986002,
                          -1267986001, -1267986000, -1267985999, -1267985998, -1267985997, -1267985996, -1267985995,
                          -1267985994, -1267985993, -1267985992, -1267985991, -1267985990, -1267985989, -1267985988,
                          -1267985987, -1267985986, -1267985985, -1267985984, -1267985983, -1267985982, -1266151000,
                          -1266150999, -1266150998, -1266150997, -1266150996, -1266150995, -1266150994, -1266150993,
                          -1266150992, -1266150991, -1266150990, -1266150989, -1266150988, -1266150987, -1266150986,
                          -1266150985, -1266150984, -1266150983, -1266150982, -1266150981, -1266150980, -1266150979,
                          -1266150978, -1266150977, -1266150976, -1266150975, -1266150974, -1264315992, -1264315991,
                          -1264315990, -1264315989, -1264315988, -1264315987, -1264315986, -1264315985, -1264315984,
                          -1264315983, -1264315982, -1264315981, -1264315980, -1264315979, -1264315978, -1264315977,
                          -1264315976, -1264315975, -1264315974, -1264315973, -1264315972, -1264315971, -1264315970,
                          -1264315969, -1264315968, -1264315967, -1264315966, -1262480984, -1262480983, -1262480982,
                          -1262480981, -1262480980, -1262480979, -1262480978, -1262480977, -1262480976, -1262480975,
                          -1262480974, -1262480973, -1262480972, -1262480971, -1262480970, -1262480969, -1262480968,
                          -1262480967, -1262480966, -1262480965, -1262480964, -1262480963, -1262480962, -1262480961,
                          -1262480960, -1262480959, -1262480958, -1260645976, -1260645975, -1260645974, -1260645973,
                          -1260645972, -1260645971, -1260645970, -1260645969, -1260645968, -1260645967, -1260645966,
                          -1260645965, -1260645964, -1260645963, -1260645962, -1260645961, -1260645960, -1260645959,
                          -1260645958, -1260645957, -1260645956, -1260645955, -1260645954, -1260645953, -1260645952,
                          -1260645951, -1260645950, -1258810968, -1258810967, -1258810966, -1258810965, -1258810964,
                          -1258810963, -1258810962, -1258810961, -1258810960, -1258810959, -1258810958, -1258810957,
                          -1258810956, -1258810955, -1258810954, -1258810953, -1258810952, -1258810951, -1258810950,
                          -1258810949, -1258810948, -1258810947, -1258810946, -1258810945, -1258810944, -1258810943,
                          -1258810942, -1256975960, -1256975959, -1256975958, -1256975957, -1256975956, -1256975955,
                          -1256975954, -1256975953, -1256975952, -1256975951, -1256975950, -1256975949, -1256975948,
                          -1256975947, -1256975946, -1256975945, -1256975944, -1256975943, -1256975942, -1256975941,
                          -1256975940, -1256975939, -1256975938, -1256975937, -1256975936, -1256975935, -1256975934,
                          -1255140952, -1255140951, -1255140950, -1255140949, -1255140948, -1255140947, -1255140946,
                          -1255140945, -1255140944, -1255140943, -1255140942, -1255140941, -1255140940, -1255140939,
                          -1255140938, -1255140937, -1255140936, -1255140935, -1255140934, -1255140933, -1255140932,
                          -1255140931, -1255140930, -1255140929, -1255140928, -1255140927, -1255140926, -1253305944,
                          -1253305943, -1253305942, -1253305941, -1253305940, -1253305939, -1253305938, -1253305937,
                          -1253305936, -1253305935, -1253305934, -1253305933, -1253305932, -1253305931, -1253305930,
                          -1253305929, -1253305928, -1253305927, -1253305926, -1253305925, -1253305924, -1253305923,
                          -1253305922, -1253305921, -1253305920, -1253305919, -1253305918, -1251470936, -1251470935,
                          -1251470934, -1251470933, -1251470932, -1251470931, -1251470930, -1251470929, -1251470928,
                          -1251470927, -1251470926, -1251470925, -1251470924, -1251470923, -1251470922, -1251470921,
                          -1251470920, -1251470919, -1251470918, -1251470917, -1251470916, -1251470915, -1251470914,
                          -1251470913, -1251470912, -1251470911, -1251470910, -1249635928, -1249635927, -1249635926,
                          -1249635925, -1249635924, -1249635923, -1249635922, -1249635921, -1249635920, -1249635919,
                          -1249635918, -1249635917, -1249635916, -1249635915, -1249635914, -1249635913, -1249635912,
                          -1249635911, -1249635910, -1249635909, -1249635908, -1249635907, -1249635906, -1249635905,
                          -1249635904, -1249635903, -1249635902, -1247800920, -1247800919, -1247800918, -1247800917,
                          -1247800916, -1247800915, -1247800914, -1247800913, -1247800912, -1247800911, -1247800910,
                          -1247800909, -1247800908, -1247800907, -1247800906, -1247800905, -1247800904, -1247800903,
                          -1247800902, -1247800901, -1247800900, -1247800899, -1247800898, -1247800897, -1247800896,
                          -1247800895, -1247800894, -1245965912, -1245965911, -1245965910, -1245965909, -1245965908,
                          -1245965907, -1245965906, -1245965905, -1245965904, -1245965903, -1245965902, -1245965901,
                          -1245965900, -1245965899, -1245965898, -1245965897, -1245965896, -1245965895, -1245965894,
                          -1245965893, -1245965892, -1245965891, -1245965890, -1245965889, -1245965888, -1245965887,
                          -1245965886, -1244130904, -1244130903, -1244130902, -1244130901, -1244130900, -1244130899,
                          -1244130898, -1244130897, -1244130896, -1244130895, -1244130894, -1244130893, -1244130892,
                          -1244130891, -1244130890, -1244130889, -1244130888, -1244130887, -1244130886, -1244130885,
                          -1244130884, -1244130883, -1244130882, -1244130881, -1244130880, -1244130879, -1244130878,
                          -1242295896, -1242295895, -1242295894, -1242295893, -1242295892, -1242295891, -1242295890,
                          -1242295889, -1242295888, -1242295887, -1242295886, -1242295885, -1242295884, -1242295883,
                          -1242295882, -1242295881, -1242295880, -1242295879, -1242295878, -1242295877, -1242295876,
                          -1242295875, -1242295874, -1242295873, -1242295872, -1242295871, -1242295870, -1240460888,
                          -1240460887, -1240460886, -1240460885, -1240460884, -1240460883, -1240460882, -1240460881,
                          -1240460880, -1240460879, -1240460878, -1240460877, -1240460876, -1240460875, -1240460874,
                          -1240460873, -1240460872, -1240460871, -1240460870, -1240460869, -1240460868, -1240460867,
                          -1240460866, -1240460865, -1240460864, -1240460863, -1240460862, -1238625880, -1238625879,
                          -1238625878, -1238625877, -1238625876, -1238625875, -1238625874, -1238625873, -1238625872,
                          -1238625871, -1238625870, -1238625869, -1238625868, -1238625867, -1238625866, -1238625865,
                          -1238625864, -1238625863, -1238625862, -1238625861, -1238625860, -1238625859, -1238625858,
                          -1238625857, -1238625856, -1238625855, -1238625854, -1236790872, -1236790871, -1236790870,
                          -1236790869, -1236790868, -1236790867, -1236790866, -1236790865, -1236790864, -1236790863,
                          -1236790862, -1236790861, -1236790860, -1236790859, -1236790858, -1236790857, -1236790856,
                          -1236790855, -1236790854, -1236790853, -1236790852, -1236790851, -1236790850, -1236790849,
                          -1236790848, -1236790847, -1236790846, -1234955864, -1234955863, -1234955862, -1234955861,
                          -1234955860, -1234955859, -1234955858, -1234955857, -1234955856, -1234955855, -1234955854,
                          -1234955853, -1234955852, -1234955851, -1234955850, -1234955849, -1234955848, -1234955847,
                          -1234955846, -1234955845, -1234955844, -1234955843, -1234955842, -1234955841, -1234955840,
                          -1234955839, -1234955838, -1233120856, -1233120855, -1233120854, -1233120853, -1233120852,
                          -1233120851, -1233120850, -1233120849, -1233120848, -1233120847, -1233120846, -1233120845,
                          -1233120844, -1233120843, -1233120842, -1233120841, -1233120840, -1233120839, -1233120838,
                          -1233120837, -1233120836, -1233120835, -1233120834, -1233120833, -1233120832, -1233120831,
                          -1233120830, -1231285848, -1231285847, -1231285846, -1231285845, -1231285844, -1231285843,
                          -1231285842, -1231285841, -1231285840, -1231285839, -1231285838, -1231285837, -1231285836,
                          -1231285835, -1231285834, -1231285833, -1231285832, -1231285831, -1231285830, -1231285829,
                          -1231285828, -1231285827, -1231285826, -1231285825, -1231285824, -1231285823, -1231285822,
                          -1229450840, -1229450839, -1229450838, -1229450837, -1229450836, -1229450835, -1229450834,
                          -1229450833, -1229450832, -1229450831, -1229450830, -1229450829, -1229450828, -1229450827,
                          -1229450826, -1229450825, -1229450824, -1229450823, -1229450822, -1229450821, -1229450820,
                          -1229450819, -1229450818, -1229450817, -1229450816, -1229450815, -1229450814, -1227615832,
                          -1227615831, -1227615830, -1227615829, -1227615828, -1227615827, -1227615826, -1227615825,
                          -1227615824, -1227615823, -1227615822, -1227615821, -1227615820, -1227615819, -1227615818,
                          -1227615817, -1227615816, -1227615815, -1227615814, -1227615813, -1227615812, -1227615811,
                          -1227615810, -1227615809, -1227615808, -1227615807, -1227615806, -1225780824, -1225780823,
                          -1225780822, -1225780821, -1225780820, -1225780819, -1225780818, -1225780817, -1225780816,
                          -1225780815, -1225780814, -1225780813, -1225780812, -1225780811, -1225780810, -1225780809,
                          -1225780808, -1225780807, -1225780806, -1225780805, -1225780804, -1225780803, -1225780802,
                          -1225780801, -1225780800, -1225780799, -1225780798, -1223945816, -1223945815, -1223945814,
                          -1223945813, -1223945812, -1223945811, -1223945810, -1223945809, -1223945808, -1223945807,
                          -1223945806, -1223945805, -1223945804, -1223945803, -1223945802, -1223945801, -1223945800,
                          -1223945799, -1223945798, -1223945797, -1223945796, -1223945795, -1223945794, -1223945793,
                          -1223945792, -1223945791, -1223945790, -1222110808, -1222110807, -1222110806, -1222110805,
                          -1222110804, -1222110803, -1222110802, -1222110801, -1222110800, -1222110799, -1222110798,
                          -1222110797, -1222110796, -1222110795, -1222110794, -1222110793, -1222110792, -1222110791,
                          -1222110790, -1222110789, -1222110788, -1222110787, -1222110786, -1222110785, -1222110784,
                          -1222110783, -1222110782, -1220275800, -1220275799, -1220275798, -1220275797, -1220275796,
                          -1220275795, -1220275794, -1220275793, -1220275792, -1220275791, -1220275790, -1220275789,
                          -1220275788, -1220275787, -1220275786, -1220275785, -1220275784, -1220275783, -1220275782,
                          -1220275781, -1220275780, -1220275779, -1220275778, -1220275777, -1220275776, -1220275775,
                          -1220275774, -1218440792, -1218440791, -1218440790, -1218440789, -1218440788, -1218440787,
                          -1218440786, -1218440785, -1218440784, -1218440783, -1218440782, -1218440781, -1218440780,
                          -1218440779, -1218440778, -1218440777, -1218440776, -1218440775, -1218440774, -1218440773,
                          -1218440772, -1218440771, -1218440770, -1218440769, -1218440768, -1218440767, -1218440766,
                          -1216605784, -1216605783, -1216605782, -1216605781, -1216605780, -1216605779, -1216605778,
                          -1216605777, -1216605776, -1216605775, -1216605774, -1216605773, -1216605772, -1216605771,
                          -1216605770, -1216605769, -1216605768, -1216605767, -1216605766, -1216605765, -1216605764,
                          -1216605763, -1216605762, -1216605761, -1216605760, -1216605759, -1216605758, -1214770776,
                          -1214770775, -1214770774, -1214770773, -1214770772, -1214770771, -1214770770, -1214770769,
                          -1214770768, -1214770767, -1214770766, -1214770765, -1214770764, -1214770763, -1214770762,
                          -1214770761, -1214770760, -1214770759, -1214770758, -1214770757, -1214770756, -1214770755,
                          -1214770754, -1214770753, -1214770752, -1214770751, -1214770750, -1212935768, -1212935767,
                          -1212935766, -1212935765, -1212935764, -1212935763, -1212935762, -1212935761, -1212935760,
                          -1212935759, -1212935758, -1212935757, -1212935756, -1212935755, -1212935754, -1212935753,
                          -1212935752, -1212935751, -1212935750, -1212935749, -1212935748, -1212935747, -1212935746,
                          -1212935745, -1212935744, -1212935743, -1212935742, -1211100760, -1211100759, -1211100758,
                          -1211100757, -1211100756, -1211100755, -1211100754, -1211100753, -1211100752, -1211100751,
                          -1211100750, -1211100749, -1211100748, -1211100747, -1211100746, -1211100745, -1211100744,
                          -1211100743, -1211100742, -1211100741, -1211100740, -1211100739, -1211100738, -1211100737,
                          -1211100736, -1211100735, -1211100734, -1209265752, -1209265751, -1209265750, -1209265749,
                          -1209265748, -1209265747, -1209265746, -1209265745, -1209265744, -1209265743, -1209265742,
                          -1209265741, -1209265740, -1209265739, -1209265738, -1209265737, -1209265736, -1209265735,
                          -1209265734, -1209265733, -1209265732, -1209265731, -1209265730, -1209265729, -1209265728,
                          -1209265727, -1209265726, -1207430744, -1207430743, -1207430742, -1207430741, -1207430740,
                          -1207430739, -1207430738, -1207430737, -1207430736, -1207430735, -1207430734, -1207430733,
                          -1207430732, -1207430731, -1207430730, -1207430729, -1207430728, -1207430727, -1207430726,
                          -1207430725, -1207430724, -1207430723, -1207430722, -1207430721, -1207430720, -1207430719,
                          -1207430718, -1205595736, -1205595735, -1205595734, -1205595733, -1205595732, -1205595731,
                          -1205595730, -1205595729, -1205595728, -1205595727, -1205595726, -1205595725, -1205595724,
                          -1205595723, -1205595722, -1205595721, -1205595720, -1205595719, -1205595718, -1205595717,
                          -1205595716, -1205595715, -1205595714, -1205595713, -1205595712, -1205595711, -1205595710,
                          -1203760728, -1203760727, -1203760726, -1203760725, -1203760724, -1203760723, -1203760722,
                          -1203760721, -1203760720, -1203760719, -1203760718, -1203760717, -1203760716, -1203760715,
                          -1203760714, -1203760713, -1203760712, -1203760711, -1203760710, -1203760709, -1203760708,
                          -1203760707, -1203760706, -1203760705, -1203760704, -1203760703, -1203760702, -1201925720,
                          -1201925719, -1201925718, -1201925717, -1201925716, -1201925715, -1201925714, -1201925713,
                          -1201925712, -1201925711, -1201925710, -1201925709, -1201925708, -1201925707, -1201925706,
                          -1201925705, -1201925704, -1201925703, -1201925702, -1201925701, -1201925700, -1201925699,
                          -1201925698, -1201925697, -1201925696, -1201925695, -1201925694, -1200090712, -1200090711,
                          -1200090710, -1200090709, -1200090708, -1200090707, -1200090706, -1200090705, -1200090704,
                          -1200090703, -1200090702, -1200090701, -1200090700, -1200090699, -1200090698, -1200090697,
                          -1200090696, -1200090695, -1200090694, -1200090693, -1200090692, -1200090691, -1200090690,
                          -1200090689, -1200090688, -1200090687, -1200090686, -1198255704, -1198255703, -1198255702,
                          -1198255701, -1198255700, -1198255699, -1198255698, -1198255697, -1198255696, -1198255695,
                          -1198255694, -1198255693, -1198255692, -1198255691, -1198255690, -1198255689, -1198255688,
                          -1198255687, -1198255686, -1198255685, -1198255684, -1198255683, -1198255682, -1198255681,
                          -1198255680, -1198255679, -1198255678, -1196420696, -1196420695, -1196420694, -1196420693,
                          -1196420692, -1196420691, -1196420690, -1196420689, -1196420688, -1196420687, -1196420686,
                          -1196420685, -1196420684, -1196420683, -1196420682, -1196420681, -1196420680, -1196420679,
                          -1196420678, -1196420677, -1196420676, -1196420675, -1196420674, -1196420673, -1196420672,
                          -1196420671, -1196420670, -1194585688, -1194585687, -1194585686, -1194585685, -1194585684,
                          -1194585683, -1194585682, -1194585681, -1194585680, -1194585679, -1194585678, -1194585677,
                          -1194585676, -1194585675, -1194585674, -1194585673, -1194585672, -1194585671, -1194585670,
                          -1194585669, -1194585668, -1194585667, -1194585666, -1194585665, -1194585664, -1194585663,
                          -1194585662, -1192750680, -1192750679, -1192750678, -1192750677, -1192750676, -1192750675,
                          -1192750674, -1192750673, -1192750672, -1192750671, -1192750670, -1192750669, -1192750668,
                          -1192750667, -1192750666, -1192750665, -1192750664, -1192750663, -1192750662, -1192750661,
                          -1192750660, -1192750659, -1192750658, -1192750657, -1192750656, -1192750655, -1192750654,
                          -1190915672, -1190915671, -1190915670, -1190915669, -1190915668, -1190915667, -1190915666,
                          -1190915665, -1190915664, -1190915663, -1190915662, -1190915661, -1190915660, -1190915659,
                          -1190915658, -1190915657, -1190915656, -1190915655, -1190915654, -1190915653, -1190915652,
                          -1190915651, -1190915650, -1190915649, -1190915648, -1190915647, -1190915646, -1189080664,
                          -1189080663, -1189080662, -1189080661, -1189080660, -1189080659, -1189080658, -1189080657,
                          -1189080656, -1189080655, -1189080654, -1189080653, -1189080652, -1189080651, -1189080650,
                          -1189080649, -1189080648, -1189080647, -1189080646, -1189080645, -1189080644, -1189080643,
                          -1189080642, -1189080641, -1189080640, -1189080639, -1189080638, -1187245656, -1187245655,
                          -1187245654, -1187245653, -1187245652, -1187245651, -1187245650, -1187245649, -1187245648,
                          -1187245647, -1187245646, -1187245645, -1187245644, -1187245643, -1187245642, -1187245641,
                          -1187245640, -1187245639, -1187245638, -1187245637, -1187245636, -1187245635, -1187245634,
                          -1187245633, -1187245632, -1187245631, -1187245630, -1185410648, -1185410647, -1185410646,
                          -1185410645, -1185410644, -1185410643, -1185410642, -1185410641, -1185410640, -1185410639,
                          -1185410638, -1185410637, -1185410636, -1185410635, -1185410634, -1185410633, -1185410632,
                          -1185410631, -1185410630, -1185410629, -1185410628, -1185410627, -1185410626, -1185410625,
                          -1185410624, -1185410623, -1185410622, -1183575640, -1183575639, -1183575638, -1183575637,
                          -1183575636, -1183575635, -1183575634, -1183575633, -1183575632, -1183575631, -1183575630,
                          -1183575629, -1183575628, -1183575627, -1183575626, -1183575625, -1183575624, -1183575623,
                          -1183575622, -1183575621, -1183575620, -1183575619, -1183575618, -1183575617, -1183575616,
                          -1183575615, -1183575614, -1181740632, -1181740631, -1181740630, -1181740629, -1181740628,
                          -1181740627, -1181740626, -1181740625, -1181740624, -1181740623, -1181740622, -1181740621,
                          -1181740620, -1181740619, -1181740618, -1181740617, -1181740616, -1181740615, -1181740614,
                          -1181740613, -1181740612, -1181740611, -1181740610, -1181740609, -1181740608, -1181740607,
                          -1181740606, -1179905624, -1179905623, -1179905622, -1179905621, -1179905620, -1179905619,
                          -1179905618, -1179905617, -1179905616, -1179905615, -1179905614, -1179905613, -1179905612,
                          -1179905611, -1179905610, -1179905609, -1179905608, -1179905607, -1179905606, -1179905605,
                          -1179905604, -1179905603, -1179905602, -1179905601, -1179905600, -1179905599, -1179905598,
                          -1178070616, -1178070615, -1178070614, -1178070613, -1178070612, -1178070611, -1178070610,
                          -1178070609, -1178070608, -1178070607, -1178070606, -1178070605, -1178070604, -1178070603,
                          -1178070602, -1178070601, -1178070600, -1178070599, -1178070598, -1178070597, -1178070596,
                          -1178070595, -1178070594, -1178070593, -1178070592, -1178070591, -1178070590, -1176235608,
                          -1176235607, -1176235606, -1176235605, -1176235604, -1176235603, -1176235602, -1176235601,
                          -1176235600, -1176235599, -1176235598, -1176235597, -1176235596, -1176235595, -1176235594,
                          -1176235593, -1176235592, -1176235591, -1176235590, -1176235589, -1176235588, -1176235587,
                          -1176235586, -1176235585, -1176235584, -1176235583, -1176235582, -1174400600, -1174400599,
                          -1174400598, -1174400597, -1174400596, -1174400595, -1174400594, -1174400593, -1174400592,
                          -1174400591, -1174400590, -1174400589, -1174400588, -1174400587, -1174400586, -1174400585,
                          -1174400584, -1174400583, -1174400582, -1174400581, -1174400580, -1174400579, -1174400578,
                          -1174400577, -1174400576, -1174400575, -1174400574, -1172565592, -1172565591, -1172565590,
                          -1172565589, -1172565588, -1172565587, -1172565586, -1172565585, -1172565584, -1172565583,
                          -1172565582, -1172565581, -1172565580, -1172565579, -1172565578, -1172565577, -1172565576,
                          -1172565575, -1172565574, -1172565573, -1172565572, -1172565571, -1172565570, -1172565569,
                          -1172565568, -1172565567, -1172565566, -1170730584, -1170730583, -1170730582, -1170730581,
                          -1170730580, -1170730579, -1170730578, -1170730577, -1170730576, -1170730575, -1170730574,
                          -1170730573, -1170730572, -1170730571, -1170730570, -1170730569, -1170730568, -1170730567,
                          -1170730566, -1170730565, -1170730564, -1170730563, -1170730562, -1170730561, -1170730560,
                          -1170730559, -1170730558, -1168895576, -1168895575, -1168895574, -1168895573, -1168895572,
                          -1168895571, -1168895570, -1168895569, -1168895568, -1168895567, -1168895566, -1168895565,
                          -1168895564, -1168895563, -1168895562, -1168895561, -1168895560, -1168895559, -1168895558,
                          -1168895557, -1168895556, -1168895555, -1168895554, -1168895553, -1168895552, -1168895551,
                          -1168895550, -1167060568, -1167060567, -1167060566, -1167060565, -1167060564, -1167060563,
                          -1167060562, -1167060561, -1167060560, -1167060559, -1167060558, -1167060557, -1167060556,
                          -1167060555, -1167060554, -1167060553, -1167060552, -1167060551, -1167060550, -1167060549,
                          -1167060548, -1167060547, -1167060546, -1167060545, -1167060544, -1167060543, -1167060542,
                          -1165225560, -1165225559, -1165225558, -1165225557, -1165225556, -1165225555, -1165225554,
                          -1165225553, -1165225552, -1165225551, -1165225550, -1165225549, -1165225548, -1165225547,
                          -1165225546, -1165225545, -1165225544, -1165225543, -1165225542, -1165225541, -1165225540,
                          -1165225539, -1165225538, -1165225537, -1165225536, -1165225535, -1165225534, -1163390552,
                          -1163390551, -1163390550, -1163390549, -1163390548, -1163390547, -1163390546, -1163390545,
                          -1163390544, -1163390543, -1163390542, -1163390541, -1163390540, -1163390539, -1163390538,
                          -1163390537, -1163390536, -1163390535, -1163390534, -1163390533, -1163390532, -1163390531,
                          -1163390530, -1163390529, -1163390528, -1163390527, -1163390526, -1161555544, -1161555543,
                          -1161555542, -1161555541, -1161555540, -1161555539, -1161555538, -1161555537, -1161555536,
                          -1161555535, -1161555534, -1161555533, -1161555532, -1161555531, -1161555530, -1161555529,
                          -1161555528, -1161555527, -1161555526, -1161555525, -1161555524, -1161555523, -1161555522,
                          -1161555521, -1161555520, -1161555519, -1161555518, -1159720536, -1159720535, -1159720534,
                          -1159720533, -1159720532, -1159720531, -1159720530, -1159720529, -1159720528, -1159720527,
                          -1159720526, -1159720525, -1159720524, -1159720523, -1159720522, -1159720521, -1159720520,
                          -1159720519, -1159720518, -1159720517, -1159720516, -1159720515, -1159720514, -1159720513,
                          -1159720512, -1159720511, -1159720510, -1157885528, -1157885527, -1157885526, -1157885525,
                          -1157885524, -1157885523, -1157885522, -1157885521, -1157885520, -1157885519, -1157885518,
                          -1157885517, -1157885516, -1157885515, -1157885514, -1157885513, -1157885512, -1157885511,
                          -1157885510, -1157885509, -1157885508, -1157885507, -1157885506, -1157885505, -1157885504,
                          -1157885503, -1157885502, -1156050520, -1156050519, -1156050518, -1156050517, -1156050516,
                          -1156050515, -1156050514, -1156050513, -1156050512, -1156050511, -1156050510, -1156050509,
                          -1156050508, -1156050507, -1156050506, -1156050505, -1156050504, -1156050503, -1156050502,
                          -1156050501, -1156050500, -1156050499, -1156050498, -1156050497, -1156050496, -1156050495,
                          -1156050494, -1154215512, -1154215511, -1154215510, -1154215509, -1154215508, -1154215507,
                          -1154215506, -1154215505, -1154215504, -1154215503, -1154215502, -1154215501, -1154215500,
                          -1154215499, -1154215498, -1154215497, -1154215496, -1154215495, -1154215494, -1154215493,
                          -1154215492, -1154215491, -1154215490, -1154215489, -1154215488, -1154215487, -1154215486,
                          -1152380504, -1152380503, -1152380502, -1152380501, -1152380500, -1152380499, -1152380498,
                          -1152380497, -1152380496, -1152380495, -1152380494, -1152380493, -1152380492, -1152380491,
                          -1152380490, -1152380489, -1152380488, -1152380487, -1152380486, -1152380485, -1152380484,
                          -1152380483, -1152380482, -1152380481, -1152380480, -1152380479, -1152380478, -1150545496,
                          -1150545495, -1150545494, -1150545493, -1150545492, -1150545491, -1150545490, -1150545489,
                          -1150545488, -1150545487, -1150545486, -1150545485, -1150545484, -1150545483, -1150545482,
                          -1150545481, -1150545480, -1150545479, -1150545478, -1150545477, -1150545476, -1150545475,
                          -1150545474, -1150545473, -1150545472, -1150545471, -1150545470, -1148710488, -1148710487,
                          -1148710486, -1148710485, -1148710484, -1148710483, -1148710482, -1148710481, -1148710480,
                          -1148710479, -1148710478, -1148710477, -1148710476, -1148710475, -1148710474, -1148710473,
                          -1148710472, -1148710471, -1148710470, -1148710469, -1148710468, -1148710467, -1148710466,
                          -1148710465, -1148710464, -1148710463, -1148710462, -1146875480, -1146875479, -1146875478,
                          -1146875477, -1146875476, -1146875475, -1146875474, -1146875473, -1146875472, -1146875471,
                          -1146875470, -1146875469, -1146875468, -1146875467, -1146875466, -1146875465, -1146875464,
                          -1146875463, -1146875462, -1146875461, -1146875460, -1146875459, -1146875458, -1146875457,
                          -1146875456, -1146875455, -1146875454, -1145040472, -1145040471, -1145040470, -1145040469,
                          -1145040468, -1145040467, -1145040466, -1145040465, -1145040464, -1145040463, -1145040462,
                          -1145040461, -1145040460, -1145040459, -1145040458, -1145040457, -1145040456, -1145040455,
                          -1145040454, -1145040453, -1145040452, -1145040451, -1145040450, -1145040449, -1145040448,
                          -1145040447, -1145040446, -1143205464, -1143205463, -1143205462, -1143205461, -1143205460,
                          -1143205459, -1143205458, -1143205457, -1143205456, -1143205455, -1143205454, -1143205453,
                          -1143205452, -1143205451, -1143205450, -1143205449, -1143205448, -1143205447, -1143205446,
                          -1143205445, -1143205444, -1143205443, -1143205442, -1143205441, -1143205440, -1143205439,
                          -1143205438, -1141370456, -1141370455, -1141370454, -1141370453, -1141370452, -1141370451,
                          -1141370450, -1141370449, -1141370448, -1141370447, -1141370446, -1141370445, -1141370444,
                          -1141370443, -1141370442, -1141370441, -1141370440, -1141370439, -1141370438, -1141370437,
                          -1141370436, -1141370435, -1141370434, -1141370433, -1141370432, -1141370431, -1141370430,
                          -1139535448, -1139535447, -1139535446, -1139535445, -1139535444, -1139535443, -1139535442,
                          -1139535441, -1139535440, -1139535439, -1139535438, -1139535437, -1139535436, -1139535435,
                          -1139535434, -1139535433, -1139535432, -1139535431, -1139535430, -1139535429, -1139535428,
                          -1139535427, -1139535426, -1139535425, -1139535424, -1139535423, -1139535422, -1137700440,
                          -1137700439, -1137700438, -1137700437, -1137700436, -1137700435, -1137700434, -1137700433,
                          -1137700432, -1137700431, -1137700430, -1137700429, -1137700428, -1137700427, -1137700426,
                          -1137700425, -1137700424, -1137700423, -1137700422, -1137700421, -1137700420, -1137700419,
                          -1137700418, -1137700417, -1137700416, -1137700415, -1137700414, -1135865432, -1135865431,
                          -1135865430, -1135865429, -1135865428, -1135865427, -1135865426, -1135865425, -1135865424,
                          -1135865423, -1135865422, -1135865421, -1135865420, -1135865419, -1135865418, -1135865417,
                          -1135865416, -1135865415, -1135865414, -1135865413, -1135865412, -1135865411, -1135865410,
                          -1135865409, -1135865408, -1135865407, -1135865406, -1134030424, -1134030423, -1134030422,
                          -1134030421, -1134030420, -1134030419, -1134030418, -1134030417, -1134030416, -1134030415,
                          -1134030414, -1134030413, -1134030412, -1134030411, -1134030410, -1134030409, -1134030408,
                          -1134030407, -1134030406, -1134030405, -1134030404, -1134030403, -1134030402, -1134030401,
                          -1134030400, -1134030399, -1134030398, -1132195416, -1132195415, -1132195414, -1132195413,
                          -1132195412, -1132195411, -1132195410, -1132195409, -1132195408, -1132195407, -1132195406,
                          -1132195405, -1132195404, -1132195403, -1132195402, -1132195401, -1132195400, -1132195399,
                          -1132195398, -1132195397, -1132195396, -1132195395, -1132195394, -1132195393, -1132195392,
                          -1132195391, -1132195390, -1130360408, -1130360407, -1130360406, -1130360405, -1130360404,
                          -1130360403, -1130360402, -1130360401, -1130360400, -1130360399, -1130360398, -1130360397,
                          -1130360396, -1130360395, -1130360394, -1130360393, -1130360392, -1130360391, -1130360390,
                          -1130360389, -1130360388, -1130360387, -1130360386, -1130360385, -1130360384, -1130360383,
                          -1130360382, -1128525400, -1128525399, -1128525398, -1128525397, -1128525396, -1128525395,
                          -1128525394, -1128525393, -1128525392, -1128525391, -1128525390, -1128525389, -1128525388,
                          -1128525387, -1128525386, -1128525385, -1128525384, -1128525383, -1128525382, -1128525381,
                          -1128525380, -1128525379, -1128525378, -1128525377, -1128525376, -1128525375, -1128525374,
                          -1126690392, -1126690391, -1126690390, -1126690389, -1126690388, -1126690387, -1126690386,
                          -1126690385, -1126690384, -1126690383, -1126690382, -1126690381, -1126690380, -1126690379,
                          -1126690378, -1126690377, -1126690376, -1126690375, -1126690374, -1126690373, -1126690372,
                          -1126690371, -1126690370, -1126690369, -1126690368, -1126690367, -1126690366, -1124855384,
                          -1124855383, -1124855382, -1124855381, -1124855380, -1124855379, -1124855378, -1124855377,
                          -1124855376, -1124855375, -1124855374, -1124855373, -1124855372, -1124855371, -1124855370,
                          -1124855369, -1124855368, -1124855367, -1124855366, -1124855365, -1124855364, -1124855363,
                          -1124855362, -1124855361, -1124855360, -1124855359, -1124855358, -1123020376, -1123020375,
                          -1123020374, -1123020373, -1123020372, -1123020371, -1123020370, -1123020369, -1123020368,
                          -1123020367, -1123020366, -1123020365, -1123020364, -1123020363, -1123020362, -1123020361,
                          -1123020360, -1123020359, -1123020358, -1123020357, -1123020356, -1123020355, -1123020354,
                          -1123020353, -1123020352, -1123020351, -1123020350, -1121185368, -1121185367, -1121185366,
                          -1121185365, -1121185364, -1121185363, -1121185362, -1121185361, -1121185360, -1121185359,
                          -1121185358, -1121185357, -1121185356, -1121185355, -1121185354, -1121185353, -1121185352,
                          -1121185351, -1121185350, -1121185349, -1121185348, -1121185347, -1121185346, -1121185345,
                          -1121185344, -1121185343, -1121185342, -1119350360, -1119350359, -1119350358, -1119350357,
                          -1119350356, -1119350355, -1119350354, -1119350353, -1119350352, -1119350351, -1119350350,
                          -1119350349, -1119350348, -1119350347, -1119350346, -1119350345, -1119350344, -1119350343,
                          -1119350342, -1119350341, -1119350340, -1119350339, -1119350338, -1119350337, -1119350336,
                          -1119350335, -1119350334, -1117515352, -1117515351, -1117515350, -1117515349, -1117515348,
                          -1117515347, -1117515346, -1117515345, -1117515344, -1117515343, -1117515342, -1117515341,
                          -1117515340, -1117515339, -1117515338, -1117515337, -1117515336, -1117515335, -1117515334,
                          -1117515333, -1117515332, -1117515331, -1117515330, -1117515329, -1117515328, -1117515327,
                          -1117515326, -1115680344, -1115680343, -1115680342, -1115680341, -1115680340, -1115680339,
                          -1115680338, -1115680337, -1115680336, -1115680335, -1115680334, -1115680333, -1115680332,
                          -1115680331, -1115680330, -1115680329, -1115680328, -1115680327, -1115680326, -1115680325,
                          -1115680324, -1115680323, -1115680322, -1115680321, -1115680320, -1115680319, -1115680318,
                          -1113845336, -1113845335, -1113845334, -1113845333, -1113845332, -1113845331, -1113845330,
                          -1113845329, -1113845328, -1113845327, -1113845326, -1113845325, -1113845324, -1113845323,
                          -1113845322, -1113845321, -1113845320, -1113845319, -1113845318, -1113845317, -1113845316,
                          -1113845315, -1113845314, -1113845313, -1113845312, -1113845311, -1113845310, -1112010328,
                          -1112010327, -1112010326, -1112010325, -1112010324, -1112010323, -1112010322, -1112010321,
                          -1112010320, -1112010319, -1112010318, -1112010317, -1112010316, -1112010315, -1112010314,
                          -1112010313, -1112010312, -1112010311, -1112010310, -1112010309, -1112010308, -1112010307,
                          -1112010306, -1112010305, -1112010304, -1112010303, -1112010302, -1110175320, -1110175319,
                          -1110175318, -1110175317, -1110175316, -1110175315, -1110175314, -1110175313, -1110175312,
                          -1110175311, -1110175310, -1110175309, -1110175308, -1110175307, -1110175306, -1110175305,
                          -1110175304, -1110175303, -1110175302, -1110175301, -1110175300, -1110175299, -1110175298,
                          -1110175297, -1110175296, -1110175295, -1110175294, -1108340312, -1108340311, -1108340310,
                          -1108340309, -1108340308, -1108340307, -1108340306, -1108340305, -1108340304, -1108340303,
                          -1108340302, -1108340301, -1108340300, -1108340299, -1108340298, -1108340297, -1108340296,
                          -1108340295, -1108340294, -1108340293, -1108340292, -1108340291, -1108340290, -1108340289,
                          -1108340288, -1108340287, -1108340286, -1106505304, -1106505303, -1106505302, -1106505301,
                          -1106505300, -1106505299, -1106505298, -1106505297, -1106505296, -1106505295, -1106505294,
                          -1106505293, -1106505292, -1106505291, -1106505290, -1106505289, -1106505288, -1106505287,
                          -1106505286, -1106505285, -1106505284, -1106505283, -1106505282, -1106505281, -1106505280,
                          -1106505279, -1106505278, -1104670296, -1104670295, -1104670294, -1104670293, -1104670292,
                          -1104670291, -1104670290, -1104670289, -1104670288, -1104670287, -1104670286, -1104670285,
                          -1104670284, -1104670283, -1104670282, -1104670281, -1104670280, -1104670279, -1104670278,
                          -1104670277, -1104670276, -1104670275, -1104670274, -1104670273, -1104670272, -1104670271,
                          -1104670270, -1102835288, -1102835287, -1102835286, -1102835285, -1102835284, -1102835283,
                          -1102835282, -1102835281, -1102835280, -1102835279, -1102835278, -1102835277, -1102835276,
                          -1102835275, -1102835274, -1102835273, -1102835272, -1102835271, -1102835270, -1102835269,
                          -1102835268, -1102835267, -1102835266, -1102835265, -1102835264, -1102835263, -1102835262,
                          -1101000280, -1101000279, -1101000278, -1101000277, -1101000276, -1101000275, -1101000274,
                          -1101000273, -1101000272, -1101000271, -1101000270, -1101000269, -1101000268, -1101000267,
                          -1101000266, -1101000265, -1101000264, -1101000263, -1101000262, -1101000261, -1101000260,
                          -1101000259, -1101000258, -1101000257, -1101000256, -1101000255, -1101000254, -1099165272,
                          -1099165271, -1099165270, -1099165269, -1099165268, -1099165267, -1099165266, -1099165265,
                          -1099165264, -1099165263, -1099165262, -1099165261, -1099165260, -1099165259, -1099165258,
                          -1099165257, -1099165256, -1099165255, -1099165254, -1099165253, -1099165252, -1099165251,
                          -1099165250, -1099165249, -1099165248, -1099165247, -1099165246, -1097330264, -1097330263,
                          -1097330262, -1097330261, -1097330260, -1097330259, -1097330258, -1097330257, -1097330256,
                          -1097330255, -1097330254, -1097330253, -1097330252, -1097330251, -1097330250, -1097330249,
                          -1097330248, -1097330247, -1097330246, -1097330245, -1097330244, -1097330243, -1097330242,
                          -1097330241, -1097330240, -1097330239, -1097330238, -1095495256, -1095495255, -1095495254,
                          -1095495253, -1095495252, -1095495251, -1095495250, -1095495249, -1095495248, -1095495247,
                          -1095495246, -1095495245, -1095495244, -1095495243, -1095495242, -1095495241, -1095495240,
                          -1095495239, -1095495238, -1095495237, -1095495236, -1095495235, -1095495234, -1095495233,
                          -1095495232, -1095495231, -1095495230, -1093660248, -1093660247, -1093660246, -1093660245,
                          -1093660244, -1093660243, -1093660242, -1093660241, -1093660240, -1093660239, -1093660238,
                          -1093660237, -1093660236, -1093660235, -1093660234, -1093660233, -1093660232, -1093660231,
                          -1093660230, -1093660229, -1093660228, -1093660227, -1093660226, -1093660225, -1093660224,
                          -1093660223, -1093660222, -1091825240, -1091825239, -1091825238, -1091825237, -1091825236,
                          -1091825235, -1091825234, -1091825233, -1091825232, -1091825231, -1091825230, -1091825229,
                          -1091825228, -1091825227, -1091825226, -1091825225, -1091825224, -1091825223, -1091825222,
                          -1091825221, -1091825220, -1091825219, -1091825218, -1091825217, -1091825216, -1091825215,
                          -1091825214, -1089990232, -1089990231, -1089990230, -1089990229, -1089990228, -1089990227,
                          -1089990226, -1089990225, -1089990224, -1089990223, -1089990222, -1089990221, -1089990220,
                          -1089990219, -1089990218, -1089990217, -1089990216, -1089990215, -1089990214, -1089990213,
                          -1089990212, -1089990211, -1089990210, -1089990209, -1089990208, -1089990207, -1089990206,
                          -1088155224, -1088155223, -1088155222, -1088155221, -1088155220, -1088155219, -1088155218,
                          -1088155217, -1088155216, -1088155215, -1088155214, -1088155213, -1088155212, -1088155211,
                          -1088155210, -1088155209, -1088155208, -1088155207, -1088155206, -1088155205, -1088155204,
                          -1088155203, -1088155202, -1088155201, -1088155200, -1088155199, -1088155198, -1086320216,
                          -1086320215, -1086320214, -1086320213, -1086320212, -1086320211, -1086320210, -1086320209,
                          -1086320208, -1086320207, -1086320206, -1086320205, -1086320204, -1086320203, -1086320202,
                          -1086320201, -1086320200, -1086320199, -1086320198, -1086320197, -1086320196, -1086320195,
                          -1086320194, -1086320193, -1086320192, -1086320191, -1086320190, -1084485208, -1084485207,
                          -1084485206, -1084485205, -1084485204, -1084485203, -1084485202, -1084485201, -1084485200,
                          -1084485199, -1084485198, -1084485197, -1084485196, -1084485195, -1084485194, -1084485193,
                          -1084485192, -1084485191, -1084485190, -1084485189, -1084485188, -1084485187, -1084485186,
                          -1084485185, -1084485184, -1084485183, -1084485182, -1082650200, -1082650199, -1082650198,
                          -1082650197, -1082650196, -1082650195, -1082650194, -1082650193, -1082650192, -1082650191,
                          -1082650190, -1082650189, -1082650188, -1082650187, -1082650186, -1082650185, -1082650184,
                          -1082650183, -1082650182, -1082650181, -1082650180, -1082650179, -1082650178, -1082650177,
                          -1082650176, -1082650175, -1082650174, -1080815192, -1080815191, -1080815190, -1080815189,
                          -1080815188, -1080815187, -1080815186, -1080815185, -1080815184, -1080815183, -1080815182,
                          -1080815181, -1080815180, -1080815179, -1080815178, -1080815177, -1080815176, -1080815175,
                          -1080815174, -1080815173, -1080815172, -1080815171, -1080815170, -1080815169, -1080815168,
                          -1080815167, -1080815166, -1078980184, -1078980183, -1078980182, -1078980181, -1078980180,
                          -1078980179, -1078980178, -1078980177, -1078980176, -1078980175, -1078980174, -1078980173,
                          -1078980172, -1078980171, -1078980170, -1078980169, -1078980168, -1078980167, -1078980166,
                          -1078980165, -1078980164, -1078980163, -1078980162, -1078980161, -1078980160, -1078980159,
                          -1078980158, -1077145176, -1077145175, -1077145174, -1077145173, -1077145172, -1077145171,
                          -1077145170, -1077145169, -1077145168, -1077145167, -1077145166, -1077145165, -1077145164,
                          -1077145163, -1077145162, -1077145161, -1077145160, -1077145159, -1077145158, -1077145157,
                          -1077145156, -1077145155, -1077145154, -1077145153, -1077145152, -1077145151, -1077145150,
                          -1075310168, -1075310167, -1075310166, -1075310165, -1075310164, -1075310163, -1075310162,
                          -1075310161, -1075310160, -1075310159, -1075310158, -1075310157, -1075310156, -1075310155,
                          -1075310154, -1075310153, -1075310152, -1075310151, -1075310150, -1075310149, -1075310148,
                          -1075310147, -1075310146, -1075310145, -1075310144, -1075310143, -1075310142, -1073475160,
                          -1073475159, -1073475158, -1073475157, -1073475156, -1073475155, -1073475154, -1073475153,
                          -1073475152, -1073475151, -1073475150, -1073475149, -1073475148, -1073475147, -1073475146,
                          -1073475145, -1073475144, -1073475143, -1073475142, -1073475141, -1073475140, -1073475139,
                          -1073475138, -1073475137, -1073475136, -1073475135, -1073475134, -1071640152, -1071640151,
                          -1071640150, -1071640149, -1071640148, -1071640147, -1071640146, -1071640145, -1071640144,
                          -1071640143, -1071640142, -1071640141, -1071640140, -1071640139, -1071640138, -1071640137,
                          -1071640136, -1071640135, -1071640134, -1071640133, -1071640132, -1071640131, -1071640130,
                          -1071640129, -1071640128, -1071640127, -1071640126, -1069805144, -1069805143, -1069805142,
                          -1069805141, -1069805140, -1069805139, -1069805138, -1069805137, -1069805136, -1069805135,
                          -1069805134, -1069805133, -1069805132, -1069805131, -1069805130, -1069805129, -1069805128,
                          -1069805127, -1069805126, -1069805125, -1069805124, -1069805123, -1069805122, -1069805121,
                          -1069805120, -1069805119, -1069805118, -1067970136, -1067970135, -1067970134, -1067970133,
                          -1067970132, -1067970131, -1067970130, -1067970129, -1067970128, -1067970127, -1067970126,
                          -1067970125, -1067970124, -1067970123, -1067970122, -1067970121, -1067970120, -1067970119,
                          -1067970118, -1067970117, -1067970116, -1067970115, -1067970114, -1067970113, -1067970112,
                          -1067970111, -1067970110, -1066135128, -1066135127, -1066135126, -1066135125, -1066135124,
                          -1066135123, -1066135122, -1066135121, -1066135120, -1066135119, -1066135118, -1066135117,
                          -1066135116, -1066135115, -1066135114, -1066135113, -1066135112, -1066135111, -1066135110,
                          -1066135109, -1066135108, -1066135107, -1066135106, -1066135105, -1066135104, -1066135103,
                          -1066135102, -1064300120, -1064300119, -1064300118, -1064300117, -1064300116, -1064300115,
                          -1064300114, -1064300113, -1064300112, -1064300111, -1064300110, -1064300109, -1064300108,
                          -1064300107, -1064300106, -1064300105, -1064300104, -1064300103, -1064300102, -1064300101,
                          -1064300100, -1064300099, -1064300098, -1064300097, -1064300096, -1064300095, -1064300094,
                          -1062465112, -1062465111, -1062465110, -1062465109, -1062465108, -1062465107, -1062465106,
                          -1062465105, -1062465104, -1062465103, -1062465102, -1062465101, -1062465100, -1062465099,
                          -1062465098, -1062465097, -1062465096, -1062465095, -1062465094, -1062465093, -1062465092,
                          -1062465091, -1062465090, -1062465089, -1062465088, -1062465087, -1062465086, -1060630104,
                          -1060630103, -1060630102, -1060630101, -1060630100, -1060630099, -1060630098, -1060630097,
                          -1060630096, -1060630095, -1060630094, -1060630093, -1060630092, -1060630091, -1060630090,
                          -1060630089, -1060630088, -1060630087, -1060630086, -1060630085, -1060630084, -1060630083,
                          -1060630082, -1060630081, -1060630080, -1060630079, -1060630078, -1058795096, -1058795095,
                          -1058795094, -1058795093, -1058795092, -1058795091, -1058795090, -1058795089, -1058795088,
                          -1058795087, -1058795086, -1058795085, -1058795084, -1058795083, -1058795082, -1058795081,
                          -1058795080, -1058795079, -1058795078, -1058795077, -1058795076, -1058795075, -1058795074,
                          -1058795073, -1058795072, -1058795071, -1058795070, -1056960088, -1056960087, -1056960086,
                          -1056960085, -1056960084, -1056960083, -1056960082, -1056960081, -1056960080, -1056960079,
                          -1056960078, -1056960077, -1056960076, -1056960075, -1056960074, -1056960073, -1056960072,
                          -1056960071, -1056960070, -1056960069, -1056960068, -1056960067, -1056960066, -1056960065,
                          -1056960064, -1056960063, -1056960062, -1055125080, -1055125079, -1055125078, -1055125077,
                          -1055125076, -1055125075, -1055125074, -1055125073, -1055125072, -1055125071, -1055125070,
                          -1055125069, -1055125068, -1055125067, -1055125066, -1055125065, -1055125064, -1055125063,
                          -1055125062, -1055125061, -1055125060, -1055125059, -1055125058, -1055125057, -1055125056,
                          -1055125055, -1055125054, -1053290072, -1053290071, -1053290070, -1053290069, -1053290068,
                          -1053290067, -1053290066, -1053290065, -1053290064, -1053290063, -1053290062, -1053290061,
                          -1053290060, -1053290059, -1053290058, -1053290057, -1053290056, -1053290055, -1053290054,
                          -1053290053, -1053290052, -1053290051, -1053290050, -1053290049, -1053290048, -1053290047,
                          -1053290046, -1051455064, -1051455063, -1051455062, -1051455061, -1051455060, -1051455059,
                          -1051455058, -1051455057, -1051455056, -1051455055, -1051455054, -1051455053, -1051455052,
                          -1051455051, -1051455050, -1051455049, -1051455048, -1051455047, -1051455046, -1051455045,
                          -1051455044, -1051455043, -1051455042, -1051455041, -1051455040, -1051455039, -1051455038,
                          -1049620056, -1049620055, -1049620054, -1049620053, -1049620052, -1049620051, -1049620050,
                          -1049620049, -1049620048, -1049620047, -1049620046, -1049620045, -1049620044, -1049620043,
                          -1049620042, -1049620041, -1049620040, -1049620039, -1049620038, -1049620037, -1049620036,
                          -1049620035, -1049620034, -1049620033, -1049620032, -1049620031, -1049620030, -1047785048,
                          -1047785047, -1047785046, -1047785045, -1047785044, -1047785043, -1047785042, -1047785041,
                          -1047785040, -1047785039, -1047785038, -1047785037, -1047785036, -1047785035, -1047785034,
                          -1047785033, -1047785032, -1047785031, -1047785030, -1047785029, -1047785028, -1047785027,
                          -1047785026, -1047785025, -1047785024, -1047785023, -1047785022, -1045950040, -1045950039,
                          -1045950038, -1045950037, -1045950036, -1045950035, -1045950034, -1045950033, -1045950032,
                          -1045950031, -1045950030, -1045950029, -1045950028, -1045950027, -1045950026, -1045950025,
                          -1045950024, -1045950023, -1045950022, -1045950021, -1045950020, -1045950019, -1045950018,
                          -1045950017, -1045950016, -1045950015, -1045950014, -1044115032, -1044115031, -1044115030,
                          -1044115029, -1044115028, -1044115027, -1044115026, -1044115025, -1044115024, -1044115023,
                          -1044115022, -1044115021, -1044115020, -1044115019, -1044115018, -1044115017, -1044115016,
                          -1044115015, -1044115014, -1044115013, -1044115012, -1044115011, -1044115010, -1044115009,
                          -1044115008, -1044115007, -1044115006, -1042280024, -1042280023, -1042280022, -1042280021,
                          -1042280020, -1042280019, -1042280018, -1042280017, -1042280016, -1042280015, -1042280014,
                          -1042280013, -1042280012, -1042280011, -1042280010, -1042280009, -1042280008, -1042280007,
                          -1042280006, -1042280005, -1042280004, -1042280003, -1042280002, -1042280001, -1042280000,
                          -1042279999, -1042279998, -1040445016, -1040445015, -1040445014, -1040445013, -1040445012,
                          -1040445011, -1040445010, -1040445009, -1040445008, -1040445007, -1040445006, -1040445005,
                          -1040445004, -1040445003, -1040445002, -1040445001, -1040445000, -1040444999, -1040444998,
                          -1040444997, -1040444996, -1040444995, -1040444994, -1040444993, -1040444992, -1040444991,
                          -1040444990, -1038610008, -1038610007, -1038610006, -1038610005, -1038610004, -1038610003,
                          -1038610002, -1038610001, -1038610000, -1038609999, -1038609998, -1038609997, -1038609996,
                          -1038609995, -1038609994, -1038609993, -1038609992, -1038609991, -1038609990, -1038609989,
                          -1038609988, -1038609987, -1038609986, -1038609985, -1038609984, -1038609983, -1038609982,
                          -1036775000, -1036774999, -1036774998, -1036774997, -1036774996, -1036774995, -1036774994,
                          -1036774993, -1036774992, -1036774991, -1036774990, -1036774989, -1036774988, -1036774987,
                          -1036774986, -1036774985, -1036774984, -1036774983, -1036774982, -1036774981, -1036774980,
                          -1036774979, -1036774978, -1036774977, -1036774976, -1036774975, -1036774974, -1034939992,
                          -1034939991, -1034939990, -1034939989, -1034939988, -1034939987, -1034939986, -1034939985,
                          -1034939984, -1034939983, -1034939982, -1034939981, -1034939980, -1034939979, -1034939978,
                          -1034939977, -1034939976, -1034939975, -1034939974, -1034939973, -1034939972, -1034939971,
                          -1034939970, -1034939969, -1034939968, -1034939967, -1034939966, -1033104984, -1033104983,
                          -1033104982, -1033104981, -1033104980, -1033104979, -1033104978, -1033104977, -1033104976,
                          -1033104975, -1033104974, -1033104973, -1033104972, -1033104971, -1033104970, -1033104969,
                          -1033104968, -1033104967, -1033104966, -1033104965, -1033104964, -1033104963, -1033104962,
                          -1033104961, -1033104960, -1033104959, -1033104958, -1031269976, -1031269975, -1031269974,
                          -1031269973, -1031269972, -1031269971, -1031269970, -1031269969, -1031269968, -1031269967,
                          -1031269966, -1031269965, -1031269964, -1031269963, -1031269962, -1031269961, -1031269960,
                          -1031269959, -1031269958, -1031269957, -1031269956, -1031269955, -1031269954, -1031269953,};
    }

    private static int[] getHangulCodepoints1() {
        return new int[] {44033, 44034, 44035, 44036, 44037, 44038, 44039, 44040, 44041, 44042, 44043, 44044, 44045,
                          44046, 44047, 44048, 44049, 44050, 44051, 44052, 44053, 44054, 44055, 44056, 44057, 44058,
                          44059, 44061, 44062, 44063, 44064, 44065, 44066, 44067, 44068, 44069, 44070, 44071, 44072,
                          44073, 44074, 44075, 44076, 44077, 44078, 44079, 44080, 44081, 44082, 44083, 44084, 44085,
                          44086, 44087, 44089, 44090, 44091, 44092, 44093, 44094, 44095, 44096, 44097, 44098, 44099,
                          44100, 44101, 44102, 44103, 44104, 44105, 44106, 44107, 44108, 44109, 44110, 44111, 44112,
                          44113, 44114, 44115, 44117, 44118, 44119, 44120, 44121, 44122, 44123, 44124, 44125, 44126,
                          44127, 44128, 44129, 44130, 44131, 44132, 44133, 44134, 44135, 44136, 44137, 44138, 44139,
                          44140, 44141, 44142, 44143, 44145, 44146, 44147, 44148, 44149, 44150, 44151, 44152, 44153,
                          44154, 44155, 44156, 44157, 44158, 44159, 44160, 44161, 44162, 44163, 44164, 44165, 44166,
                          44167, 44168, 44169, 44170, 44171, 44173, 44174, 44175, 44176, 44177, 44178, 44179, 44180,
                          44181, 44182, 44183, 44184, 44185, 44186, 44187, 44188, 44189, 44190, 44191, 44192, 44193,
                          44194, 44195, 44196, 44197, 44198, 44199, 44201, 44202, 44203, 44204, 44205, 44206, 44207,
                          44208, 44209, 44210, 44211, 44212, 44213, 44214, 44215, 44216, 44217, 44218, 44219, 44220,
                          44221, 44222, 44223, 44224, 44225, 44226, 44227, 44229, 44230, 44231, 44232, 44233, 44234,
                          44235, 44236, 44237, 44238, 44239, 44240, 44241, 44242, 44243, 44244, 44245, 44246, 44247,
                          44248, 44249, 44250, 44251, 44252, 44253, 44254, 44255, 44257, 44258, 44259, 44260, 44261,
                          44262, 44263, 44264, 44265, 44266, 44267, 44268, 44269, 44270, 44271, 44272, 44273, 44274,
                          44275, 44276, 44277, 44278, 44279, 44280, 44281, 44282, 44283, 44285, 44286, 44287, 44288,
                          44289, 44290, 44291, 44292, 44293, 44294, 44295, 44296, 44297, 44298, 44299, 44300, 44301,
                          44302, 44303, 44304, 44305, 44306, 44307, 44308, 44309, 44310, 44311, 44313, 44314, 44315,
                          44316, 44317, 44318, 44319, 44320, 44321, 44322, 44323, 44324, 44325, 44326, 44327, 44328,
                          44329, 44330, 44331, 44332, 44333, 44334, 44335, 44336, 44337, 44338, 44339, 44341, 44342,
                          44343, 44344, 44345, 44346, 44347, 44348, 44349, 44350, 44351, 44352, 44353, 44354, 44355,
                          44356, 44357, 44358, 44359, 44360, 44361, 44362, 44363, 44364, 44365, 44366, 44367, 44369,
                          44370, 44371, 44372, 44373, 44374, 44375, 44376, 44377, 44378, 44379, 44380, 44381, 44382,
                          44383, 44384, 44385, 44386, 44387, 44388, 44389, 44390, 44391, 44392, 44393, 44394, 44395,
                          44397, 44398, 44399, 44400, 44401, 44402, 44403, 44404, 44405, 44406, 44407, 44408, 44409,
                          44410, 44411, 44412, 44413, 44414, 44415, 44416, 44417, 44418, 44419, 44420, 44421, 44422,
                          44423, 44425, 44426, 44427, 44428, 44429, 44430, 44431, 44432, 44433, 44434, 44435, 44436,
                          44437, 44438, 44439, 44440, 44441, 44442, 44443, 44444, 44445, 44446, 44447, 44448, 44449,
                          44450, 44451, 44453, 44454, 44455, 44456, 44457, 44458, 44459, 44460, 44461, 44462, 44463,
                          44464, 44465, 44466, 44467, 44468, 44469, 44470, 44471, 44472, 44473, 44474, 44475, 44476,
                          44477, 44478, 44479, 44481, 44482, 44483, 44484, 44485, 44486, 44487, 44488, 44489, 44490,
                          44491, 44492, 44493, 44494, 44495, 44496, 44497, 44498, 44499, 44500, 44501, 44502, 44503,
                          44504, 44505, 44506, 44507, 44509, 44510, 44511, 44512, 44513, 44514, 44515, 44516, 44517,
                          44518, 44519, 44520, 44521, 44522, 44523, 44524, 44525, 44526, 44527, 44528, 44529, 44530,
                          44531, 44532, 44533, 44534, 44535, 44537, 44538, 44539, 44540, 44541, 44542, 44543, 44544,
                          44545, 44546, 44547, 44548, 44549, 44550, 44551, 44552, 44553, 44554, 44555, 44556, 44557,
                          44558, 44559, 44560, 44561, 44562, 44563, 44565, 44566, 44567, 44568, 44569, 44570, 44571,
                          44572, 44573, 44574, 44575, 44576, 44577, 44578, 44579, 44580, 44581, 44582, 44583, 44584,
                          44585, 44586, 44587, 44588, 44589, 44590, 44591, 44593, 44594, 44595, 44596, 44597, 44598,
                          44599, 44600, 44601, 44602, 44603, 44604, 44605, 44606, 44607, 44608, 44609, 44610, 44611,
                          44612, 44613, 44614, 44615, 44616, 44617, 44618, 44619, 44621, 44622, 44623, 44624, 44625,
                          44626, 44627, 44628, 44629, 44630, 44631, 44632, 44633, 44634, 44635, 44636, 44637, 44638,
                          44639, 44640, 44641, 44642, 44643, 44644, 44645, 44646, 44647, 44649, 44650, 44651, 44652,
                          44653, 44654, 44655, 44656, 44657, 44658, 44659, 44660, 44661, 44662, 44663, 44664, 44665,
                          44666, 44667, 44668, 44669, 44670, 44671, 44672, 44673, 44674, 44675, 44677, 44678, 44679,
                          44680, 44681, 44682, 44683, 44684, 44685, 44686, 44687, 44688, 44689, 44690, 44691, 44692,
                          44693, 44694, 44695, 44696, 44697, 44698, 44699, 44700, 44701, 44702, 44703, 44705, 44706,
                          44707, 44708, 44709, 44710, 44711, 44712, 44713, 44714, 44715, 44716, 44717, 44718, 44719,
                          44720, 44721, 44722, 44723, 44724, 44725, 44726, 44727, 44728, 44729, 44730, 44731, 44733,
                          44734, 44735, 44736, 44737, 44738, 44739, 44740, 44741, 44742, 44743, 44744, 44745, 44746,
                          44747, 44748, 44749, 44750, 44751, 44752, 44753, 44754, 44755, 44756, 44757, 44758, 44759,
                          44761, 44762, 44763, 44764, 44765, 44766, 44767, 44768, 44769, 44770, 44771, 44772, 44773,
                          44774, 44775, 44776, 44777, 44778, 44779, 44780, 44781, 44782, 44783, 44784, 44785, 44786,
                          44787, 44789, 44790, 44791, 44792, 44793, 44794, 44795, 44796, 44797, 44798, 44799, 44800,
                          44801, 44802, 44803, 44804, 44805, 44806, 44807, 44808, 44809, 44810, 44811, 44812, 44813,
                          44814, 44815, 44817, 44818, 44819, 44820, 44821, 44822, 44823, 44824, 44825, 44826, 44827,
                          44828, 44829, 44830, 44831, 44832, 44833, 44834, 44835, 44836, 44837, 44838, 44839, 44840,
                          44841, 44842, 44843, 44845, 44846, 44847, 44848, 44849, 44850, 44851, 44852, 44853, 44854,
                          44855, 44856, 44857, 44858, 44859, 44860, 44861, 44862, 44863, 44864, 44865, 44866, 44867,
                          44868, 44869, 44870, 44871, 44873, 44874, 44875, 44876, 44877, 44878, 44879, 44880, 44881,
                          44882, 44883, 44884, 44885, 44886, 44887, 44888, 44889, 44890, 44891, 44892, 44893, 44894,
                          44895, 44896, 44897, 44898, 44899, 44901, 44902, 44903, 44904, 44905, 44906, 44907, 44908,
                          44909, 44910, 44911, 44912, 44913, 44914, 44915, 44916, 44917, 44918, 44919, 44920, 44921,
                          44922, 44923, 44924, 44925, 44926, 44927, 44929, 44930, 44931, 44932, 44933, 44934, 44935,
                          44936, 44937, 44938, 44939, 44940, 44941, 44942, 44943, 44944, 44945, 44946, 44947, 44948,
                          44949, 44950, 44951, 44952, 44953, 44954, 44955, 44957, 44958, 44959, 44960, 44961, 44962,
                          44963, 44964, 44965, 44966, 44967, 44968, 44969, 44970, 44971, 44972, 44973, 44974, 44975,
                          44976, 44977, 44978, 44979, 44980, 44981, 44982, 44983, 44985, 44986, 44987, 44988, 44989,
                          44990, 44991, 44992, 44993, 44994, 44995, 44996, 44997, 44998, 44999, 45000, 45001, 45002,
                          45003, 45004, 45005, 45006, 45007, 45008, 45009, 45010, 45011, 45013, 45014, 45015, 45016,
                          45017, 45018, 45019, 45020, 45021, 45022, 45023, 45024, 45025, 45026, 45027, 45028, 45029,
                          45030, 45031, 45032, 45033, 45034, 45035, 45036, 45037, 45038, 45039, 45041, 45042, 45043,
                          45044, 45045, 45046, 45047, 45048, 45049, 45050, 45051, 45052, 45053, 45054, 45055, 45056,
                          45057, 45058, 45059, 45060, 45061, 45062, 45063, 45064, 45065, 45066, 45067, 45069, 45070,
                          45071, 45072, 45073, 45074, 45075, 45076, 45077, 45078, 45079, 45080, 45081, 45082, 45083,
                          45084, 45085, 45086, 45087, 45088, 45089, 45090, 45091, 45092, 45093, 45094, 45095, 45097,
                          45098, 45099, 45100, 45101, 45102, 45103, 45104, 45105, 45106, 45107, 45108, 45109, 45110,
                          45111, 45112, 45113, 45114, 45115, 45116, 45117, 45118, 45119, 45120, 45121, 45122, 45123,
                          45125, 45126, 45127, 45128, 45129, 45130, 45131, 45132, 45133, 45134, 45135, 45136, 45137,
                          45138, 45139, 45140, 45141, 45142, 45143, 45144, 45145, 45146, 45147, 45148, 45149, 45150,
                          45151, 45153, 45154, 45155, 45156, 45157, 45158, 45159, 45160, 45161, 45162, 45163, 45164,
                          45165, 45166, 45167, 45168, 45169, 45170, 45171, 45172, 45173, 45174, 45175, 45176, 45177,
                          45178, 45179, 45181, 45182, 45183, 45184, 45185, 45186, 45187, 45188, 45189, 45190, 45191,
                          45192, 45193, 45194, 45195, 45196, 45197, 45198, 45199, 45200, 45201, 45202, 45203, 45204,
                          45205, 45206, 45207, 45209, 45210, 45211, 45212, 45213, 45214, 45215, 45216, 45217, 45218,
                          45219, 45220, 45221, 45222, 45223, 45224, 45225, 45226, 45227, 45228, 45229, 45230, 45231,
                          45232, 45233, 45234, 45235, 45237, 45238, 45239, 45240, 45241, 45242, 45243, 45244, 45245,
                          45246, 45247, 45248, 45249, 45250, 45251, 45252, 45253, 45254, 45255, 45256, 45257, 45258,
                          45259, 45260, 45261, 45262, 45263, 45265, 45266, 45267, 45268, 45269, 45270, 45271, 45272,
                          45273, 45274, 45275, 45276, 45277, 45278, 45279, 45280, 45281, 45282, 45283, 45284, 45285,
                          45286, 45287, 45288, 45289, 45290, 45291, 45293, 45294, 45295, 45296, 45297, 45298, 45299,
                          45300, 45301, 45302, 45303, 45304, 45305, 45306, 45307, 45308, 45309, 45310, 45311, 45312,
                          45313, 45314, 45315, 45316, 45317, 45318, 45319, 45321, 45322, 45323, 45324, 45325, 45326,
                          45327, 45328, 45329, 45330, 45331, 45332, 45333, 45334, 45335, 45336, 45337, 45338, 45339,
                          45340, 45341, 45342, 45343, 45344, 45345, 45346, 45347, 45349, 45350, 45351, 45352, 45353,
                          45354, 45355, 45356, 45357, 45358, 45359, 45360, 45361, 45362, 45363, 45364, 45365, 45366,
                          45367, 45368, 45369, 45370, 45371, 45372, 45373, 45374, 45375, 45377, 45378, 45379, 45380,
                          45381, 45382, 45383, 45384, 45385, 45386, 45387, 45388, 45389, 45390, 45391, 45392, 45393,
                          45394, 45395, 45396, 45397, 45398, 45399, 45400, 45401, 45402, 45403, 45405, 45406, 45407,
                          45408, 45409, 45410, 45411, 45412, 45413, 45414, 45415, 45416, 45417, 45418, 45419, 45420,
                          45421, 45422, 45423, 45424, 45425, 45426, 45427, 45428, 45429, 45430, 45431, 45433, 45434,
                          45435, 45436, 45437, 45438, 45439, 45440, 45441, 45442, 45443, 45444, 45445, 45446, 45447,
                          45448, 45449, 45450, 45451, 45452, 45453, 45454, 45455, 45456, 45457, 45458, 45459, 45461,
                          45462, 45463, 45464, 45465, 45466, 45467, 45468, 45469, 45470, 45471, 45472, 45473, 45474,
                          45475, 45476, 45477, 45478, 45479, 45480, 45481, 45482, 45483, 45484, 45485, 45486, 45487,
                          45489, 45490, 45491, 45492, 45493, 45494, 45495, 45496, 45497, 45498, 45499, 45500, 45501,
                          45502, 45503, 45504, 45505, 45506, 45507, 45508, 45509, 45510, 45511, 45512, 45513, 45514,
                          45515, 45517, 45518, 45519, 45520, 45521, 45522, 45523, 45524, 45525, 45526, 45527, 45528,
                          45529, 45530, 45531, 45532, 45533, 45534, 45535, 45536, 45537, 45538, 45539, 45540, 45541,
                          45542, 45543, 45545, 45546, 45547, 45548, 45549, 45550, 45551, 45552, 45553, 45554, 45555,
                          45556, 45557, 45558, 45559, 45560, 45561, 45562, 45563, 45564, 45565, 45566, 45567, 45568,
                          45569, 45570, 45571, 45573, 45574, 45575, 45576, 45577, 45578, 45579, 45580, 45581, 45582,
                          45583, 45584, 45585, 45586, 45587, 45588, 45589, 45590, 45591, 45592, 45593, 45594, 45595,
                          45596, 45597, 45598, 45599, 45601, 45602, 45603, 45604, 45605, 45606, 45607, 45608, 45609,
                          45610, 45611, 45612, 45613, 45614, 45615, 45616, 45617, 45618, 45619, 45620, 45621, 45622,
                          45623, 45624, 45625, 45626, 45627, 45629, 45630, 45631, 45632, 45633, 45634, 45635, 45636,
                          45637, 45638, 45639, 45640, 45641, 45642, 45643, 45644, 45645, 45646, 45647, 45648, 45649,
                          45650, 45651, 45652, 45653, 45654, 45655, 45657, 45658, 45659, 45660, 45661, 45662, 45663,
                          45664, 45665, 45666, 45667, 45668, 45669, 45670, 45671, 45672, 45673, 45674, 45675, 45676,
                          45677, 45678, 45679, 45680, 45681, 45682, 45683, 45685, 45686, 45687, 45688, 45689, 45690,
                          45691, 45692, 45693, 45694, 45695, 45696, 45697, 45698, 45699, 45700, 45701, 45702, 45703,
                          45704, 45705, 45706, 45707, 45708, 45709, 45710, 45711, 45713, 45714, 45715, 45716, 45717,
                          45718, 45719, 45720, 45721, 45722, 45723, 45724, 45725, 45726, 45727, 45728, 45729, 45730,
                          45731, 45732, 45733, 45734, 45735, 45736, 45737, 45738, 45739, 45741, 45742, 45743, 45744,
                          45745, 45746, 45747, 45748, 45749, 45750, 45751, 45752, 45753, 45754, 45755, 45756, 45757,
                          45758, 45759, 45760, 45761, 45762, 45763, 45764, 45765, 45766, 45767, 45769, 45770, 45771,
                          45772, 45773, 45774, 45775, 45776, 45777, 45778, 45779, 45780, 45781, 45782, 45783, 45784,
                          45785, 45786, 45787, 45788, 45789, 45790, 45791, 45792, 45793, 45794, 45795, 45797, 45798,
                          45799, 45800, 45801, 45802, 45803, 45804, 45805, 45806, 45807, 45808, 45809, 45810, 45811,
                          45812, 45813, 45814, 45815, 45816, 45817, 45818, 45819, 45820, 45821, 45822, 45823, 45825,
                          45826, 45827, 45828, 45829, 45830, 45831, 45832, 45833, 45834, 45835, 45836, 45837, 45838,
                          45839, 45840, 45841, 45842, 45843, 45844, 45845, 45846, 45847, 45848, 45849, 45850, 45851,
                          45853, 45854, 45855, 45856, 45857, 45858, 45859, 45860, 45861, 45862, 45863, 45864, 45865,
                          45866, 45867, 45868, 45869, 45870, 45871, 45872, 45873, 45874, 45875, 45876, 45877, 45878,
                          45879, 45881, 45882, 45883, 45884, 45885, 45886, 45887, 45888, 45889, 45890, 45891, 45892,
                          45893, 45894, 45895, 45896, 45897, 45898, 45899, 45900, 45901, 45902, 45903, 45904, 45905,
                          45906, 45907, 45909, 45910, 45911, 45912, 45913, 45914, 45915, 45916, 45917, 45918, 45919,
                          45920, 45921, 45922, 45923, 45924, 45925, 45926, 45927, 45928, 45929, 45930, 45931, 45932,
                          45933, 45934, 45935, 45937, 45938, 45939, 45940, 45941, 45942, 45943, 45944, 45945, 45946,
                          45947, 45948, 45949, 45950, 45951, 45952, 45953, 45954, 45955, 45956, 45957, 45958, 45959,
                          45960, 45961, 45962, 45963, 45965, 45966, 45967, 45968, 45969, 45970, 45971, 45972, 45973,
                          45974, 45975, 45976, 45977, 45978, 45979, 45980, 45981, 45982, 45983, 45984, 45985, 45986,
                          45987, 45988, 45989, 45990, 45991, 45993, 45994, 45995, 45996, 45997, 45998, 45999, 46000,
                          46001, 46002, 46003, 46004, 46005, 46006, 46007, 46008, 46009, 46010, 46011, 46012, 46013,
                          46014, 46015, 46016, 46017, 46018, 46019, 46021, 46022, 46023, 46024, 46025, 46026, 46027,
                          46028, 46029, 46030, 46031, 46032, 46033, 46034, 46035, 46036, 46037, 46038, 46039, 46040,
                          46041, 46042, 46043, 46044, 46045, 46046, 46047, 46049, 46050, 46051, 46052, 46053, 46054,
                          46055, 46056, 46057, 46058, 46059, 46060, 46061, 46062, 46063, 46064, 46065, 46066, 46067,
                          46068, 46069, 46070, 46071, 46072, 46073, 46074, 46075, 46077, 46078, 46079, 46080, 46081,
                          46082, 46083, 46084, 46085, 46086, 46087, 46088, 46089, 46090, 46091, 46092, 46093, 46094,
                          46095, 46096, 46097, 46098, 46099, 46100, 46101, 46102, 46103, 46105, 46106, 46107, 46108,
                          46109, 46110, 46111, 46112, 46113, 46114, 46115, 46116, 46117, 46118, 46119, 46120, 46121,
                          46122, 46123, 46124, 46125, 46126, 46127, 46128, 46129, 46130, 46131, 46133, 46134, 46135,
                          46136, 46137, 46138, 46139, 46140, 46141, 46142, 46143, 46144, 46145, 46146, 46147, 46148,
                          46149, 46150, 46151, 46152, 46153, 46154, 46155, 46156, 46157, 46158, 46159, 46161, 46162,
                          46163, 46164, 46165, 46166, 46167, 46168, 46169, 46170, 46171, 46172, 46173, 46174, 46175,
                          46176, 46177, 46178, 46179, 46180, 46181, 46182, 46183, 46184, 46185, 46186, 46187, 46189,
                          46190, 46191, 46192, 46193, 46194, 46195, 46196, 46197, 46198, 46199, 46200, 46201, 46202,
                          46203, 46204, 46205, 46206, 46207, 46208, 46209, 46210, 46211, 46212, 46213, 46214, 46215,
                          46217, 46218, 46219, 46220, 46221, 46222, 46223, 46224, 46225, 46226, 46227, 46228, 46229,
                          46230, 46231, 46232, 46233, 46234, 46235, 46236, 46237, 46238, 46239, 46240, 46241, 46242,
                          46243, 46245, 46246, 46247, 46248, 46249, 46250, 46251, 46252, 46253, 46254, 46255, 46256,
                          46257, 46258, 46259, 46260, 46261, 46262, 46263, 46264, 46265, 46266, 46267, 46268, 46269,
                          46270, 46271, 46273, 46274, 46275, 46276, 46277, 46278, 46279, 46280, 46281, 46282, 46283,
                          46284, 46285, 46286, 46287, 46288, 46289, 46290, 46291, 46292, 46293, 46294, 46295, 46296,
                          46297, 46298, 46299, 46301, 46302, 46303, 46304, 46305, 46306, 46307, 46308, 46309, 46310,
                          46311, 46312, 46313, 46314, 46315, 46316, 46317, 46318, 46319, 46320, 46321, 46322, 46323,
                          46324, 46325, 46326, 46327, 46329, 46330, 46331, 46332, 46333, 46334, 46335, 46336, 46337,
                          46338, 46339, 46340, 46341, 46342, 46343, 46344, 46345, 46346, 46347, 46348, 46349, 46350,
                          46351, 46352, 46353, 46354, 46355, 46357, 46358, 46359, 46360, 46361, 46362, 46363, 46364,
                          46365, 46366, 46367, 46368, 46369, 46370, 46371, 46372, 46373, 46374, 46375, 46376, 46377,
                          46378, 46379, 46380, 46381, 46382, 46383, 46385, 46386, 46387, 46388, 46389, 46390, 46391,
                          46392, 46393, 46394, 46395, 46396, 46397, 46398, 46399, 46400, 46401, 46402, 46403, 46404,
                          46405, 46406, 46407, 46408, 46409, 46410, 46411, 46413, 46414, 46415, 46416, 46417, 46418,
                          46419, 46420, 46421, 46422, 46423, 46424, 46425, 46426, 46427, 46428, 46429, 46430, 46431,
                          46432, 46433, 46434, 46435, 46436, 46437, 46438, 46439, 46441, 46442, 46443, 46444, 46445,
                          46446, 46447, 46448, 46449, 46450, 46451, 46452, 46453, 46454, 46455, 46456, 46457, 46458,
                          46459, 46460, 46461, 46462, 46463, 46464, 46465, 46466, 46467, 46469, 46470, 46471, 46472,
                          46473, 46474, 46475, 46476, 46477, 46478, 46479, 46480, 46481, 46482, 46483, 46484, 46485,
                          46486, 46487, 46488, 46489, 46490, 46491, 46492, 46493, 46494, 46495, 46497, 46498, 46499,
                          46500, 46501, 46502, 46503, 46504, 46505, 46506, 46507, 46508, 46509, 46510, 46511, 46512,
                          46513, 46514, 46515, 46516, 46517, 46518, 46519, 46520, 46521, 46522, 46523, 46525, 46526,
                          46527, 46528, 46529, 46530, 46531, 46532, 46533, 46534, 46535, 46536, 46537, 46538, 46539,
                          46540, 46541, 46542, 46543, 46544, 46545, 46546, 46547, 46548, 46549, 46550, 46551, 46553,
                          46554, 46555, 46556, 46557, 46558, 46559, 46560, 46561, 46562, 46563, 46564, 46565, 46566,
                          46567, 46568, 46569, 46570, 46571, 46572, 46573, 46574, 46575, 46576, 46577, 46578, 46579,
                          46581, 46582, 46583, 46584, 46585, 46586, 46587, 46588, 46589, 46590, 46591, 46592, 46593,
                          46594, 46595, 46596, 46597, 46598, 46599, 46600, 46601, 46602, 46603, 46604, 46605, 46606,
                          46607, 46609, 46610, 46611, 46612, 46613, 46614, 46615, 46616, 46617, 46618, 46619, 46620,
                          46621, 46622, 46623, 46624, 46625, 46626, 46627, 46628, 46629, 46630, 46631, 46632, 46633,
                          46634, 46635, 46637, 46638, 46639, 46640, 46641, 46642, 46643, 46644, 46645, 46646, 46647,
                          46648, 46649, 46650, 46651, 46652, 46653, 46654, 46655, 46656, 46657, 46658, 46659, 46660,
                          46661, 46662, 46663, 46665, 46666, 46667, 46668, 46669, 46670, 46671, 46672, 46673, 46674,
                          46675, 46676, 46677, 46678, 46679, 46680, 46681, 46682, 46683, 46684, 46685, 46686, 46687,
                          46688, 46689, 46690, 46691, 46693, 46694, 46695, 46696, 46697, 46698, 46699, 46700, 46701,
                          46702, 46703, 46704, 46705, 46706, 46707, 46708, 46709, 46710, 46711, 46712, 46713, 46714,
                          46715, 46716, 46717, 46718, 46719, 46721, 46722, 46723, 46724, 46725, 46726, 46727, 46728,
                          46729, 46730, 46731, 46732, 46733, 46734, 46735, 46736, 46737, 46738, 46739, 46740, 46741,
                          46742, 46743, 46744, 46745, 46746, 46747, 46749, 46750, 46751, 46752, 46753, 46754, 46755,
                          46756, 46757, 46758, 46759, 46760, 46761, 46762, 46763, 46764, 46765, 46766, 46767, 46768,
                          46769, 46770, 46771, 46772, 46773, 46774, 46775, 46777, 46778, 46779, 46780, 46781, 46782,
                          46783, 46784, 46785, 46786, 46787, 46788, 46789, 46790, 46791, 46792, 46793, 46794, 46795,
                          46796, 46797, 46798, 46799, 46800, 46801, 46802, 46803, 46805, 46806, 46807, 46808, 46809,
                          46810, 46811, 46812, 46813, 46814, 46815, 46816, 46817, 46818, 46819, 46820, 46821, 46822,
                          46823, 46824, 46825, 46826, 46827, 46828, 46829, 46830, 46831, 46833, 46834, 46835, 46836,
                          46837, 46838, 46839, 46840, 46841, 46842, 46843, 46844, 46845, 46846, 46847, 46848, 46849,
                          46850, 46851, 46852, 46853, 46854, 46855, 46856, 46857, 46858, 46859, 46861, 46862, 46863,
                          46864, 46865, 46866, 46867, 46868, 46869, 46870, 46871, 46872, 46873, 46874, 46875, 46876,
                          46877, 46878, 46879, 46880, 46881, 46882, 46883, 46884, 46885, 46886, 46887, 46889, 46890,
                          46891, 46892, 46893, 46894, 46895, 46896, 46897, 46898, 46899, 46900, 46901, 46902, 46903,
                          46904, 46905, 46906, 46907, 46908, 46909, 46910, 46911, 46912, 46913, 46914, 46915, 46917,
                          46918, 46919, 46920, 46921, 46922, 46923, 46924, 46925, 46926, 46927, 46928, 46929, 46930,
                          46931, 46932, 46933, 46934, 46935, 46936, 46937, 46938, 46939, 46940, 46941, 46942, 46943,
                          46945, 46946, 46947, 46948, 46949, 46950, 46951, 46952, 46953, 46954, 46955, 46956, 46957,
                          46958, 46959, 46960, 46961, 46962, 46963, 46964, 46965, 46966, 46967, 46968, 46969, 46970,
                          46971, 46973, 46974, 46975, 46976, 46977, 46978, 46979, 46980, 46981, 46982, 46983, 46984,
                          46985, 46986, 46987, 46988, 46989, 46990, 46991, 46992, 46993, 46994, 46995, 46996, 46997,
                          46998, 46999, 47001, 47002, 47003, 47004, 47005, 47006, 47007, 47008, 47009, 47010, 47011,
                          47012, 47013, 47014, 47015, 47016, 47017, 47018, 47019, 47020, 47021, 47022, 47023, 47024,
                          47025, 47026, 47027, 47029, 47030, 47031, 47032, 47033, 47034, 47035, 47036, 47037, 47038,
                          47039, 47040, 47041, 47042, 47043, 47044, 47045, 47046, 47047, 47048, 47049, 47050, 47051,
                          47052, 47053, 47054, 47055, 47057, 47058, 47059, 47060, 47061, 47062, 47063, 47064, 47065,
                          47066, 47067, 47068, 47069, 47070, 47071, 47072, 47073, 47074, 47075, 47076, 47077, 47078,
                          47079, 47080, 47081, 47082, 47083, 47085, 47086, 47087, 47088, 47089, 47090, 47091, 47092,
                          47093, 47094, 47095, 47096, 47097, 47098, 47099, 47100, 47101, 47102, 47103, 47104, 47105,
                          47106, 47107, 47108, 47109, 47110, 47111, 47113, 47114, 47115, 47116, 47117, 47118, 47119,
                          47120, 47121, 47122, 47123, 47124, 47125, 47126, 47127, 47128, 47129, 47130, 47131, 47132,
                          47133, 47134, 47135, 47136, 47137, 47138, 47139, 47141, 47142, 47143, 47144, 47145, 47146,
                          47147, 47148, 47149, 47150, 47151, 47152, 47153, 47154, 47155, 47156, 47157, 47158, 47159,
                          47160, 47161, 47162, 47163, 47164, 47165, 47166, 47167, 47169, 47170, 47171, 47172, 47173,
                          47174, 47175, 47176, 47177, 47178, 47179, 47180, 47181, 47182, 47183, 47184, 47185, 47186,
                          47187, 47188, 47189, 47190, 47191, 47192, 47193, 47194, 47195, 47197, 47198, 47199, 47200,
                          47201, 47202, 47203, 47204, 47205, 47206, 47207, 47208, 47209, 47210, 47211, 47212, 47213,
                          47214, 47215, 47216, 47217, 47218, 47219, 47220, 47221, 47222, 47223, 47225, 47226, 47227,
                          47228, 47229, 47230, 47231, 47232, 47233, 47234, 47235, 47236, 47237, 47238, 47239, 47240,
                          47241, 47242, 47243, 47244, 47245, 47246, 47247, 47248, 47249, 47250, 47251, 47253, 47254,
                          47255, 47256, 47257, 47258, 47259, 47260, 47261, 47262, 47263, 47264, 47265, 47266, 47267,
                          47268, 47269, 47270, 47271, 47272, 47273, 47274, 47275, 47276, 47277, 47278, 47279, 47281,
                          47282, 47283, 47284, 47285, 47286, 47287, 47288, 47289, 47290, 47291, 47292, 47293, 47294,
                          47295, 47296, 47297, 47298, 47299, 47300, 47301, 47302, 47303, 47304, 47305, 47306, 47307,
                          47309, 47310, 47311, 47312, 47313, 47314, 47315, 47316, 47317, 47318, 47319, 47320, 47321,
                          47322, 47323, 47324, 47325, 47326, 47327, 47328, 47329, 47330, 47331, 47332, 47333, 47334,
                          47335, 47337, 47338, 47339, 47340, 47341, 47342, 47343, 47344, 47345, 47346, 47347, 47348,
                          47349, 47350, 47351, 47352, 47353, 47354, 47355, 47356, 47357, 47358, 47359, 47360, 47361,
                          47362, 47363, 47365, 47366, 47367, 47368, 47369, 47370, 47371, 47372, 47373, 47374, 47375,
                          47376, 47377, 47378, 47379, 47380, 47381, 47382, 47383, 47384, 47385, 47386, 47387, 47388,
                          47389, 47390, 47391, 47393, 47394, 47395, 47396, 47397, 47398, 47399, 47400, 47401, 47402,
                          47403, 47404, 47405, 47406, 47407, 47408, 47409, 47410, 47411, 47412, 47413, 47414, 47415,
                          47416, 47417, 47418, 47419, 47421, 47422, 47423, 47424, 47425, 47426, 47427, 47428, 47429,
                          47430, 47431, 47432, 47433, 47434, 47435, 47436, 47437, 47438, 47439, 47440, 47441, 47442,
                          47443, 47444, 47445, 47446, 47447, 47449, 47450, 47451, 47452, 47453, 47454, 47455, 47456,
                          47457, 47458, 47459, 47460, 47461, 47462, 47463, 47464, 47465, 47466, 47467, 47468, 47469,
                          47470, 47471, 47472, 47473, 47474, 47475, 47477, 47478, 47479, 47480, 47481, 47482, 47483,
                          47484, 47485, 47486, 47487, 47488, 47489, 47490, 47491, 47492, 47493, 47494, 47495, 47496,
                          47497, 47498, 47499, 47500, 47501, 47502, 47503, 47505, 47506, 47507, 47508, 47509, 47510,
                          47511, 47512, 47513, 47514, 47515, 47516, 47517, 47518, 47519, 47520, 47521, 47522, 47523,
                          47524, 47525, 47526, 47527, 47528, 47529, 47530, 47531, 47533, 47534, 47535, 47536, 47537,
                          47538, 47539, 47540, 47541, 47542, 47543, 47544, 47545, 47546, 47547, 47548, 47549, 47550,
                          47551, 47552, 47553, 47554, 47555, 47556, 47557, 47558, 47559, 47561, 47562, 47563, 47564,
                          47565, 47566, 47567, 47568, 47569, 47570, 47571, 47572, 47573, 47574, 47575, 47576, 47577,
                          47578, 47579, 47580, 47581, 47582, 47583, 47584, 47585, 47586, 47587, 47589, 47590, 47591,
                          47592, 47593, 47594, 47595, 47596, 47597, 47598, 47599, 47600, 47601, 47602, 47603, 47604,
                          47605, 47606, 47607, 47608, 47609, 47610, 47611, 47612, 47613, 47614, 47615, 47617, 47618,
                          47619, 47620, 47621, 47622, 47623, 47624, 47625, 47626, 47627, 47628, 47629, 47630, 47631,
                          47632, 47633, 47634, 47635, 47636, 47637, 47638, 47639, 47640, 47641, 47642, 47643, 47645,
                          47646, 47647, 47648, 47649, 47650, 47651, 47652, 47653, 47654, 47655, 47656, 47657, 47658,
                          47659, 47660, 47661, 47662, 47663, 47664, 47665, 47666, 47667, 47668, 47669, 47670, 47671,
                          47673, 47674, 47675, 47676, 47677, 47678, 47679, 47680, 47681, 47682, 47683, 47684, 47685,
                          47686, 47687, 47688, 47689, 47690, 47691, 47692, 47693, 47694, 47695, 47696, 47697, 47698,
                          47699, 47701, 47702, 47703, 47704, 47705, 47706, 47707, 47708, 47709, 47710, 47711, 47712,
                          47713, 47714, 47715, 47716, 47717, 47718, 47719, 47720, 47721, 47722, 47723, 47724, 47725,
                          47726, 47727, 47729, 47730, 47731, 47732, 47733, 47734, 47735, 47736, 47737, 47738, 47739,
                          47740, 47741, 47742, 47743, 47744, 47745, 47746, 47747, 47748, 47749, 47750, 47751, 47752,
                          47753, 47754, 47755, 47757, 47758, 47759, 47760, 47761, 47762, 47763, 47764, 47765, 47766,
                          47767, 47768, 47769, 47770, 47771, 47772, 47773, 47774, 47775, 47776, 47777, 47778, 47779,
                          47780, 47781, 47782, 47783, 47785, 47786, 47787, 47788, 47789, 47790, 47791, 47792, 47793,
                          47794, 47795, 47796, 47797, 47798, 47799, 47800, 47801, 47802, 47803, 47804, 47805, 47806,
                          47807, 47808, 47809, 47810, 47811, 47813, 47814, 47815, 47816, 47817, 47818, 47819, 47820,
                          47821, 47822, 47823, 47824, 47825, 47826, 47827, 47828, 47829, 47830, 47831, 47832, 47833,
                          47834, 47835, 47836, 47837, 47838, 47839, 47841, 47842, 47843, 47844, 47845, 47846, 47847,
                          47848, 47849, 47850, 47851, 47852, 47853, 47854, 47855, 47856, 47857, 47858, 47859, 47860,
                          47861, 47862, 47863, 47864, 47865, 47866, 47867, 47869, 47870, 47871, 47872, 47873, 47874,
                          47875, 47876, 47877, 47878, 47879, 47880, 47881, 47882, 47883, 47884, 47885, 47886, 47887,
                          47888, 47889, 47890, 47891, 47892, 47893, 47894, 47895, 47897, 47898, 47899, 47900, 47901,
                          47902, 47903, 47904, 47905, 47906, 47907, 47908, 47909, 47910, 47911, 47912, 47913, 47914,
                          47915, 47916, 47917, 47918, 47919, 47920, 47921, 47922, 47923, 47925, 47926, 47927, 47928,
                          47929, 47930, 47931, 47932, 47933, 47934, 47935, 47936, 47937, 47938, 47939, 47940, 47941,
                          47942, 47943, 47944, 47945, 47946, 47947, 47948, 47949, 47950, 47951, 47953, 47954, 47955,
                          47956, 47957, 47958, 47959, 47960, 47961, 47962, 47963, 47964, 47965, 47966, 47967, 47968,
                          47969, 47970, 47971, 47972, 47973, 47974, 47975, 47976, 47977, 47978, 47979, 47981, 47982,
                          47983, 47984, 47985, 47986, 47987, 47988, 47989, 47990, 47991, 47992, 47993, 47994, 47995,
                          47996, 47997, 47998, 47999, 48000, 48001, 48002, 48003, 48004, 48005, 48006, 48007, 48009,
                          48010, 48011, 48012, 48013, 48014, 48015, 48016, 48017, 48018, 48019, 48020, 48021, 48022,
                          48023, 48024, 48025, 48026, 48027, 48028, 48029, 48030, 48031, 48032, 48033, 48034, 48035,
                          48037, 48038, 48039, 48040, 48041, 48042, 48043, 48044, 48045, 48046, 48047, 48048, 48049,
                          48050, 48051, 48052, 48053, 48054, 48055, 48056, 48057, 48058, 48059, 48060, 48061, 48062,
                          48063, 48065, 48066, 48067, 48068, 48069, 48070, 48071, 48072, 48073, 48074, 48075, 48076,
                          48077, 48078, 48079, 48080, 48081, 48082, 48083, 48084, 48085, 48086, 48087, 48088, 48089,
                          48090, 48091, 48093, 48094, 48095, 48096, 48097, 48098, 48099, 48100, 48101, 48102, 48103,
                          48104, 48105, 48106, 48107, 48108, 48109, 48110, 48111, 48112, 48113, 48114, 48115, 48116,
                          48117, 48118, 48119, 48121, 48122, 48123, 48124, 48125, 48126, 48127, 48128, 48129, 48130,
                          48131, 48132, 48133, 48134, 48135, 48136, 48137, 48138, 48139, 48140, 48141, 48142, 48143,
                          48144, 48145, 48146, 48147, 48149, 48150, 48151, 48152, 48153, 48154, 48155, 48156, 48157,
                          48158, 48159, 48160, 48161, 48162, 48163, 48164, 48165, 48166, 48167, 48168, 48169, 48170,
                          48171, 48172, 48173, 48174, 48175, 48177, 48178, 48179, 48180, 48181, 48182, 48183, 48184,
                          48185, 48186, 48187, 48188, 48189, 48190, 48191, 48192, 48193, 48194, 48195, 48196, 48197,
                          48198, 48199, 48200, 48201, 48202, 48203, 48205, 48206, 48207, 48208, 48209, 48210, 48211,
                          48212, 48213, 48214, 48215, 48216, 48217, 48218, 48219, 48220, 48221, 48222, 48223, 48224,
                          48225, 48226, 48227, 48228, 48229, 48230, 48231, 48233, 48234, 48235, 48236, 48237, 48238,
                          48239, 48240, 48241, 48242, 48243, 48244, 48245, 48246, 48247, 48248, 48249, 48250, 48251,
                          48252, 48253, 48254, 48255, 48256, 48257, 48258, 48259, 48261, 48262, 48263, 48264, 48265,
                          48266, 48267, 48268, 48269, 48270, 48271, 48272, 48273, 48274, 48275, 48276, 48277, 48278,
                          48279, 48280, 48281, 48282, 48283, 48284, 48285, 48286, 48287, 48289, 48290, 48291, 48292,
                          48293, 48294, 48295, 48296, 48297, 48298, 48299, 48300, 48301, 48302, 48303, 48304, 48305,
                          48306, 48307, 48308, 48309, 48310, 48311, 48312, 48313, 48314, 48315, 48317, 48318, 48319,
                          48320, 48321, 48322, 48323, 48324, 48325, 48326, 48327, 48328, 48329, 48330, 48331, 48332,
                          48333, 48334, 48335, 48336, 48337, 48338, 48339, 48340, 48341, 48342, 48343, 48345, 48346,
                          48347, 48348, 48349, 48350, 48351, 48352, 48353, 48354, 48355, 48356, 48357, 48358, 48359,
                          48360, 48361, 48362, 48363, 48364, 48365, 48366, 48367, 48368, 48369, 48370, 48371, 48373,
                          48374, 48375, 48376, 48377, 48378, 48379, 48380, 48381, 48382, 48383, 48384, 48385, 48386,
                          48387, 48388, 48389, 48390, 48391, 48392, 48393, 48394, 48395, 48396, 48397, 48398, 48399,
                          48401, 48402, 48403, 48404, 48405, 48406, 48407, 48408, 48409, 48410, 48411, 48412, 48413,
                          48414, 48415, 48416, 48417, 48418, 48419, 48420, 48421, 48422, 48423, 48424, 48425, 48426,
                          48427, 48429, 48430, 48431, 48432, 48433, 48434, 48435, 48436, 48437, 48438, 48439, 48440,
                          48441, 48442, 48443, 48444, 48445, 48446, 48447, 48448, 48449, 48450, 48451, 48452, 48453,
                          48454, 48455, 48457, 48458, 48459, 48460, 48461, 48462, 48463, 48464, 48465, 48466, 48467,
                          48468, 48469, 48470, 48471, 48472, 48473, 48474, 48475, 48476, 48477, 48478, 48479, 48480,
                          48481, 48482, 48483, 48485, 48486, 48487, 48488, 48489, 48490, 48491, 48492, 48493, 48494,
                          48495, 48496, 48497, 48498, 48499, 48500, 48501, 48502, 48503, 48504, 48505, 48506, 48507,
                          48508, 48509, 48510, 48511, 48513, 48514, 48515, 48516, 48517, 48518, 48519, 48520, 48521,
                          48522, 48523, 48524, 48525, 48526, 48527, 48528, 48529, 48530, 48531, 48532, 48533, 48534,
                          48535, 48536, 48537, 48538, 48539, 48541, 48542, 48543, 48544, 48545, 48546, 48547, 48548,
                          48549, 48550, 48551, 48552, 48553, 48554, 48555, 48556, 48557, 48558, 48559, 48560, 48561,
                          48562, 48563, 48564, 48565, 48566, 48567, 48569, 48570, 48571, 48572, 48573, 48574, 48575,
                          48576, 48577, 48578, 48579, 48580, 48581, 48582, 48583, 48584, 48585, 48586, 48587, 48588,
                          48589, 48590, 48591, 48592, 48593, 48594, 48595, 48597, 48598, 48599, 48600, 48601, 48602,
                          48603, 48604, 48605, 48606, 48607, 48608, 48609, 48610, 48611, 48612, 48613, 48614, 48615,
                          48616, 48617, 48618, 48619, 48620, 48621, 48622, 48623, 48625, 48626, 48627, 48628, 48629,
                          48630, 48631, 48632, 48633, 48634, 48635, 48636, 48637, 48638, 48639, 48640, 48641, 48642,
                          48643, 48644, 48645, 48646, 48647, 48648, 48649, 48650, 48651, 48653, 48654, 48655, 48656,
                          48657, 48658, 48659, 48660, 48661, 48662, 48663, 48664, 48665, 48666, 48667, 48668, 48669,
                          48670, 48671, 48672, 48673, 48674, 48675, 48676, 48677, 48678, 48679, 48681, 48682, 48683,
                          48684, 48685, 48686, 48687, 48688, 48689, 48690, 48691, 48692, 48693, 48694, 48695, 48696,
                          48697, 48698, 48699, 48700, 48701, 48702, 48703, 48704, 48705, 48706, 48707, 48709, 48710,
                          48711, 48712, 48713, 48714, 48715, 48716, 48717, 48718, 48719, 48720, 48721, 48722, 48723,
                          48724, 48725, 48726, 48727, 48728, 48729, 48730, 48731, 48732, 48733, 48734, 48735, 48737,
                          48738, 48739, 48740, 48741, 48742, 48743, 48744, 48745, 48746, 48747, 48748, 48749, 48750,
                          48751, 48752, 48753, 48754, 48755, 48756, 48757, 48758, 48759, 48760, 48761, 48762, 48763,
                          48765, 48766, 48767, 48768, 48769, 48770, 48771, 48772, 48773, 48774, 48775, 48776, 48777,
                          48778, 48779, 48780, 48781, 48782, 48783, 48784, 48785, 48786, 48787, 48788, 48789, 48790,
                          48791, 48793, 48794, 48795, 48796, 48797, 48798, 48799, 48800, 48801, 48802, 48803, 48804,
                          48805, 48806, 48807, 48808, 48809, 48810, 48811, 48812, 48813, 48814, 48815, 48816, 48817,
                          48818, 48819, 48821, 48822, 48823, 48824, 48825, 48826, 48827, 48828, 48829, 48830, 48831,
                          48832, 48833, 48834, 48835, 48836, 48837, 48838, 48839, 48840, 48841, 48842, 48843, 48844,
                          48845, 48846, 48847, 48849, 48850, 48851, 48852, 48853, 48854, 48855, 48856, 48857, 48858,
                          48859, 48860, 48861, 48862, 48863, 48864, 48865, 48866, 48867, 48868, 48869, 48870, 48871,
                          48872, 48873, 48874, 48875, 48877, 48878, 48879, 48880, 48881, 48882, 48883, 48884, 48885,
                          48886, 48887, 48888, 48889, 48890, 48891, 48892, 48893, 48894, 48895, 48896, 48897, 48898,
                          48899, 48900, 48901, 48902, 48903, 48905, 48906, 48907, 48908, 48909, 48910, 48911, 48912,
                          48913, 48914, 48915, 48916, 48917, 48918, 48919, 48920, 48921, 48922, 48923, 48924, 48925,
                          48926, 48927, 48928, 48929, 48930, 48931, 48933, 48934, 48935, 48936, 48937, 48938, 48939,
                          48940, 48941, 48942, 48943, 48944, 48945, 48946, 48947, 48948, 48949, 48950, 48951, 48952,
                          48953, 48954, 48955, 48956, 48957, 48958, 48959, 48961, 48962, 48963, 48964, 48965, 48966,
                          48967, 48968, 48969, 48970, 48971, 48972, 48973, 48974, 48975, 48976, 48977, 48978, 48979,
                          48980, 48981, 48982, 48983, 48984, 48985, 48986, 48987, 48989, 48990, 48991, 48992, 48993,
                          48994, 48995, 48996, 48997, 48998, 48999, 49000, 49001, 49002, 49003, 49004, 49005, 49006,
                          49007, 49008, 49009, 49010, 49011, 49012, 49013, 49014, 49015, 49017, 49018, 49019, 49020,
                          49021, 49022, 49023, 49024, 49025, 49026, 49027, 49028, 49029, 49030, 49031, 49032, 49033,
                          49034, 49035, 49036, 49037, 49038, 49039, 49040, 49041, 49042, 49043, 49045, 49046, 49047,
                          49048, 49049, 49050, 49051, 49052, 49053, 49054, 49055, 49056, 49057, 49058, 49059, 49060,
                          49061, 49062, 49063, 49064, 49065, 49066, 49067, 49068, 49069, 49070, 49071, 49073, 49074,
                          49075, 49076, 49077, 49078, 49079, 49080, 49081, 49082, 49083, 49084, 49085, 49086, 49087,
                          49088, 49089, 49090, 49091, 49092, 49093, 49094, 49095, 49096, 49097, 49098, 49099, 49101,
                          49102, 49103, 49104, 49105, 49106, 49107, 49108, 49109, 49110, 49111, 49112, 49113, 49114,
                          49115, 49116, 49117, 49118, 49119, 49120, 49121, 49122, 49123, 49124, 49125, 49126, 49127,
                          49129, 49130, 49131, 49132, 49133, 49134, 49135, 49136, 49137, 49138, 49139, 49140, 49141,
                          49142, 49143, 49144, 49145, 49146, 49147, 49148, 49149, 49150, 49151, 49152, 49153, 49154,
                          49155, 49157, 49158, 49159, 49160, 49161, 49162, 49163, 49164, 49165, 49166, 49167, 49168,
                          49169, 49170, 49171, 49172, 49173, 49174, 49175, 49176, 49177, 49178, 49179, 49180, 49181,
                          49182, 49183, 49185, 49186, 49187, 49188, 49189, 49190, 49191, 49192, 49193, 49194, 49195,
                          49196, 49197, 49198, 49199, 49200, 49201, 49202, 49203, 49204, 49205, 49206, 49207, 49208,
                          49209, 49210, 49211, 49213, 49214, 49215, 49216, 49217, 49218, 49219, 49220, 49221, 49222,
                          49223, 49224, 49225, 49226, 49227, 49228, 49229, 49230, 49231, 49232, 49233, 49234, 49235,
                          49236, 49237, 49238, 49239, 49241, 49242, 49243, 49244, 49245, 49246, 49247, 49248, 49249,
                          49250, 49251, 49252, 49253, 49254, 49255, 49256, 49257, 49258, 49259, 49260, 49261, 49262,
                          49263, 49264, 49265, 49266, 49267, 49269, 49270, 49271, 49272, 49273, 49274, 49275, 49276,
                          49277, 49278, 49279, 49280, 49281, 49282, 49283, 49284, 49285, 49286, 49287, 49288, 49289,
                          49290, 49291, 49292, 49293, 49294, 49295, 49297, 49298, 49299, 49300, 49301, 49302, 49303,
                          49304, 49305, 49306, 49307, 49308, 49309, 49310, 49311, 49312, 49313, 49314, 49315, 49316,
                          49317, 49318, 49319, 49320, 49321, 49322, 49323, 49325, 49326, 49327, 49328, 49329, 49330,
                          49331, 49332, 49333, 49334, 49335, 49336, 49337, 49338, 49339, 49340, 49341, 49342, 49343,
                          49344, 49345, 49346, 49347, 49348, 49349, 49350, 49351, 49353, 49354, 49355, 49356, 49357,
                          49358, 49359, 49360, 49361, 49362, 49363, 49364, 49365, 49366, 49367, 49368, 49369, 49370,
                          49371, 49372, 49373, 49374, 49375, 49376, 49377, 49378, 49379, 49381, 49382, 49383, 49384,
                          49385, 49386, 49387, 49388, 49389, 49390, 49391, 49392, 49393, 49394, 49395, 49396, 49397,
                          49398, 49399, 49400, 49401, 49402, 49403, 49404, 49405, 49406, 49407, 49409, 49410, 49411,
                          49412, 49413, 49414, 49415, 49416, 49417, 49418, 49419, 49420, 49421, 49422, 49423, 49424,
                          49425, 49426, 49427, 49428, 49429, 49430, 49431, 49432, 49433, 49434, 49435, 49437, 49438,
                          49439, 49440, 49441, 49442, 49443, 49444, 49445, 49446, 49447, 49448, 49449, 49450, 49451,
                          49452, 49453, 49454, 49455, 49456, 49457, 49458, 49459, 49460, 49461, 49462, 49463, 49465,
                          49466, 49467, 49468, 49469, 49470, 49471, 49472, 49473, 49474, 49475, 49476, 49477, 49478,
                          49479, 49480, 49481, 49482, 49483, 49484, 49485, 49486, 49487, 49488, 49489, 49490, 49491,
                          49493, 49494, 49495, 49496, 49497, 49498, 49499, 49500, 49501, 49502, 49503, 49504, 49505,
                          49506, 49507, 49508, 49509, 49510, 49511, 49512, 49513, 49514, 49515, 49516, 49517, 49518,
                          49519, 49521, 49522, 49523, 49524, 49525, 49526, 49527, 49528, 49529, 49530, 49531, 49532,
                          49533, 49534, 49535, 49536, 49537, 49538, 49539, 49540, 49541, 49542, 49543, 49544, 49545,
                          49546, 49547, 49549, 49550, 49551, 49552, 49553, 49554, 49555, 49556, 49557, 49558, 49559,
                          49560, 49561, 49562, 49563, 49564, 49565, 49566, 49567, 49568, 49569, 49570, 49571, 49572,
                          49573, 49574, 49575, 49577, 49578, 49579, 49580, 49581, 49582, 49583, 49584, 49585, 49586,
                          49587, 49588, 49589, 49590, 49591, 49592, 49593, 49594, 49595, 49596, 49597, 49598, 49599,
                          49600, 49601, 49602, 49603, 49605, 49606, 49607, 49608, 49609, 49610, 49611, 49612, 49613,
                          49614, 49615, 49616, 49617, 49618, 49619, 49620, 49621, 49622, 49623, 49624, 49625, 49626,
                          49627, 49628, 49629, 49630, 49631, 49633, 49634, 49635, 49636, 49637, 49638, 49639, 49640,
                          49641, 49642, 49643, 49644, 49645, 49646, 49647, 49648, 49649, 49650, 49651, 49652, 49653,
                          49654, 49655, 49656, 49657, 49658, 49659, 49661, 49662, 49663, 49664, 49665, 49666, 49667,
                          49668, 49669, 49670, 49671, 49672, 49673, 49674, 49675, 49676, 49677, 49678, 49679, 49680,
                          49681, 49682, 49683, 49684, 49685, 49686, 49687, 49689, 49690, 49691, 49692, 49693, 49694,
                          49695, 49696, 49697, 49698, 49699, 49700, 49701, 49702, 49703, 49704, 49705, 49706, 49707,
                          49708, 49709, 49710, 49711, 49712, 49713, 49714, 49715, 49717, 49718, 49719, 49720, 49721,
                          49722, 49723, 49724, 49725, 49726, 49727, 49728, 49729, 49730, 49731, 49732, 49733, 49734,
                          49735, 49736, 49737, 49738, 49739, 49740, 49741, 49742, 49743, 49745, 49746, 49747, 49748,
                          49749, 49750, 49751, 49752, 49753, 49754, 49755, 49756, 49757, 49758, 49759, 49760, 49761,
                          49762, 49763, 49764, 49765, 49766, 49767, 49768, 49769, 49770, 49771, 49773, 49774, 49775,
                          49776, 49777, 49778, 49779, 49780, 49781, 49782, 49783, 49784, 49785, 49786, 49787, 49788,
                          49789, 49790, 49791, 49792, 49793, 49794, 49795, 49796, 49797, 49798, 49799, 49801, 49802,
                          49803, 49804, 49805, 49806, 49807, 49808, 49809, 49810, 49811, 49812, 49813, 49814, 49815,
                          49816, 49817, 49818, 49819, 49820, 49821, 49822, 49823, 49824,};
    }

    private static int[] getHangulPairs2() {
        return new int[] {-1031269952, -1031269951, -1031269950, -1029434968, -1029434967, -1029434966, -1029434965,
                          -1029434964, -1029434963, -1029434962, -1029434961, -1029434960, -1029434959, -1029434958,
                          -1029434957, -1029434956, -1029434955, -1029434954, -1029434953, -1029434952, -1029434951,
                          -1029434950, -1029434949, -1029434948, -1029434947, -1029434946, -1029434945, -1029434944,
                          -1029434943, -1029434942, -1027599960, -1027599959, -1027599958, -1027599957, -1027599956,
                          -1027599955, -1027599954, -1027599953, -1027599952, -1027599951, -1027599950, -1027599949,
                          -1027599948, -1027599947, -1027599946, -1027599945, -1027599944, -1027599943, -1027599942,
                          -1027599941, -1027599940, -1027599939, -1027599938, -1027599937, -1027599936, -1027599935,
                          -1027599934, -1025764952, -1025764951, -1025764950, -1025764949, -1025764948, -1025764947,
                          -1025764946, -1025764945, -1025764944, -1025764943, -1025764942, -1025764941, -1025764940,
                          -1025764939, -1025764938, -1025764937, -1025764936, -1025764935, -1025764934, -1025764933,
                          -1025764932, -1025764931, -1025764930, -1025764929, -1025764928, -1025764927, -1025764926,
                          -1023929944, -1023929943, -1023929942, -1023929941, -1023929940, -1023929939, -1023929938,
                          -1023929937, -1023929936, -1023929935, -1023929934, -1023929933, -1023929932, -1023929931,
                          -1023929930, -1023929929, -1023929928, -1023929927, -1023929926, -1023929925, -1023929924,
                          -1023929923, -1023929922, -1023929921, -1023929920, -1023929919, -1023929918, -1022094936,
                          -1022094935, -1022094934, -1022094933, -1022094932, -1022094931, -1022094930, -1022094929,
                          -1022094928, -1022094927, -1022094926, -1022094925, -1022094924, -1022094923, -1022094922,
                          -1022094921, -1022094920, -1022094919, -1022094918, -1022094917, -1022094916, -1022094915,
                          -1022094914, -1022094913, -1022094912, -1022094911, -1022094910, -1020259928, -1020259927,
                          -1020259926, -1020259925, -1020259924, -1020259923, -1020259922, -1020259921, -1020259920,
                          -1020259919, -1020259918, -1020259917, -1020259916, -1020259915, -1020259914, -1020259913,
                          -1020259912, -1020259911, -1020259910, -1020259909, -1020259908, -1020259907, -1020259906,
                          -1020259905, -1020259904, -1020259903, -1020259902, -1018424920, -1018424919, -1018424918,
                          -1018424917, -1018424916, -1018424915, -1018424914, -1018424913, -1018424912, -1018424911,
                          -1018424910, -1018424909, -1018424908, -1018424907, -1018424906, -1018424905, -1018424904,
                          -1018424903, -1018424902, -1018424901, -1018424900, -1018424899, -1018424898, -1018424897,
                          -1018424896, -1018424895, -1018424894, -1016589912, -1016589911, -1016589910, -1016589909,
                          -1016589908, -1016589907, -1016589906, -1016589905, -1016589904, -1016589903, -1016589902,
                          -1016589901, -1016589900, -1016589899, -1016589898, -1016589897, -1016589896, -1016589895,
                          -1016589894, -1016589893, -1016589892, -1016589891, -1016589890, -1016589889, -1016589888,
                          -1016589887, -1016589886, -1014754904, -1014754903, -1014754902, -1014754901, -1014754900,
                          -1014754899, -1014754898, -1014754897, -1014754896, -1014754895, -1014754894, -1014754893,
                          -1014754892, -1014754891, -1014754890, -1014754889, -1014754888, -1014754887, -1014754886,
                          -1014754885, -1014754884, -1014754883, -1014754882, -1014754881, -1014754880, -1014754879,
                          -1014754878, -1012919896, -1012919895, -1012919894, -1012919893, -1012919892, -1012919891,
                          -1012919890, -1012919889, -1012919888, -1012919887, -1012919886, -1012919885, -1012919884,
                          -1012919883, -1012919882, -1012919881, -1012919880, -1012919879, -1012919878, -1012919877,
                          -1012919876, -1012919875, -1012919874, -1012919873, -1012919872, -1012919871, -1012919870,
                          -1011084888, -1011084887, -1011084886, -1011084885, -1011084884, -1011084883, -1011084882,
                          -1011084881, -1011084880, -1011084879, -1011084878, -1011084877, -1011084876, -1011084875,
                          -1011084874, -1011084873, -1011084872, -1011084871, -1011084870, -1011084869, -1011084868,
                          -1011084867, -1011084866, -1011084865, -1011084864, -1011084863, -1011084862, -1009249880,
                          -1009249879, -1009249878, -1009249877, -1009249876, -1009249875, -1009249874, -1009249873,
                          -1009249872, -1009249871, -1009249870, -1009249869, -1009249868, -1009249867, -1009249866,
                          -1009249865, -1009249864, -1009249863, -1009249862, -1009249861, -1009249860, -1009249859,
                          -1009249858, -1009249857, -1009249856, -1009249855, -1009249854, -1007414872, -1007414871,
                          -1007414870, -1007414869, -1007414868, -1007414867, -1007414866, -1007414865, -1007414864,
                          -1007414863, -1007414862, -1007414861, -1007414860, -1007414859, -1007414858, -1007414857,
                          -1007414856, -1007414855, -1007414854, -1007414853, -1007414852, -1007414851, -1007414850,
                          -1007414849, -1007414848, -1007414847, -1007414846, -1005579864, -1005579863, -1005579862,
                          -1005579861, -1005579860, -1005579859, -1005579858, -1005579857, -1005579856, -1005579855,
                          -1005579854, -1005579853, -1005579852, -1005579851, -1005579850, -1005579849, -1005579848,
                          -1005579847, -1005579846, -1005579845, -1005579844, -1005579843, -1005579842, -1005579841,
                          -1005579840, -1005579839, -1005579838, -1003744856, -1003744855, -1003744854, -1003744853,
                          -1003744852, -1003744851, -1003744850, -1003744849, -1003744848, -1003744847, -1003744846,
                          -1003744845, -1003744844, -1003744843, -1003744842, -1003744841, -1003744840, -1003744839,
                          -1003744838, -1003744837, -1003744836, -1003744835, -1003744834, -1003744833, -1003744832,
                          -1003744831, -1003744830, -1001909848, -1001909847, -1001909846, -1001909845, -1001909844,
                          -1001909843, -1001909842, -1001909841, -1001909840, -1001909839, -1001909838, -1001909837,
                          -1001909836, -1001909835, -1001909834, -1001909833, -1001909832, -1001909831, -1001909830,
                          -1001909829, -1001909828, -1001909827, -1001909826, -1001909825, -1001909824, -1001909823,
                          -1001909822, -1000074840, -1000074839, -1000074838, -1000074837, -1000074836, -1000074835,
                          -1000074834, -1000074833, -1000074832, -1000074831, -1000074830, -1000074829, -1000074828,
                          -1000074827, -1000074826, -1000074825, -1000074824, -1000074823, -1000074822, -1000074821,
                          -1000074820, -1000074819, -1000074818, -1000074817, -1000074816, -1000074815, -1000074814,
                          -998239832, -998239831, -998239830, -998239829, -998239828, -998239827, -998239826,
                          -998239825, -998239824, -998239823, -998239822, -998239821, -998239820, -998239819,
                          -998239818, -998239817, -998239816, -998239815, -998239814, -998239813, -998239812,
                          -998239811, -998239810, -998239809, -998239808, -998239807, -998239806, -996404824,
                          -996404823, -996404822, -996404821, -996404820, -996404819, -996404818, -996404817,
                          -996404816, -996404815, -996404814, -996404813, -996404812, -996404811, -996404810,
                          -996404809, -996404808, -996404807, -996404806, -996404805, -996404804, -996404803,
                          -996404802, -996404801, -996404800, -996404799, -996404798, -994569816, -994569815,
                          -994569814, -994569813, -994569812, -994569811, -994569810, -994569809, -994569808,
                          -994569807, -994569806, -994569805, -994569804, -994569803, -994569802, -994569801,
                          -994569800, -994569799, -994569798, -994569797, -994569796, -994569795, -994569794,
                          -994569793, -994569792, -994569791, -994569790, -992734808, -992734807, -992734806,
                          -992734805, -992734804, -992734803, -992734802, -992734801, -992734800, -992734799,
                          -992734798, -992734797, -992734796, -992734795, -992734794, -992734793, -992734792,
                          -992734791, -992734790, -992734789, -992734788, -992734787, -992734786, -992734785,
                          -992734784, -992734783, -992734782, -990899800, -990899799, -990899798, -990899797,
                          -990899796, -990899795, -990899794, -990899793, -990899792, -990899791, -990899790,
                          -990899789, -990899788, -990899787, -990899786, -990899785, -990899784, -990899783,
                          -990899782, -990899781, -990899780, -990899779, -990899778, -990899777, -990899776,
                          -990899775, -990899774, -989064792, -989064791, -989064790, -989064789, -989064788,
                          -989064787, -989064786, -989064785, -989064784, -989064783, -989064782, -989064781,
                          -989064780, -989064779, -989064778, -989064777, -989064776, -989064775, -989064774,
                          -989064773, -989064772, -989064771, -989064770, -989064769, -989064768, -989064767,
                          -989064766, -987229784, -987229783, -987229782, -987229781, -987229780, -987229779,
                          -987229778, -987229777, -987229776, -987229775, -987229774, -987229773, -987229772,
                          -987229771, -987229770, -987229769, -987229768, -987229767, -987229766, -987229765,
                          -987229764, -987229763, -987229762, -987229761, -987229760, -987229759, -987229758,
                          -985394776, -985394775, -985394774, -985394773, -985394772, -985394771, -985394770,
                          -985394769, -985394768, -985394767, -985394766, -985394765, -985394764, -985394763,
                          -985394762, -985394761, -985394760, -985394759, -985394758, -985394757, -985394756,
                          -985394755, -985394754, -985394753, -985394752, -985394751, -985394750, -983559768,
                          -983559767, -983559766, -983559765, -983559764, -983559763, -983559762, -983559761,
                          -983559760, -983559759, -983559758, -983559757, -983559756, -983559755, -983559754,
                          -983559753, -983559752, -983559751, -983559750, -983559749, -983559748, -983559747,
                          -983559746, -983559745, -983559744, -983559743, -983559742, -981724760, -981724759,
                          -981724758, -981724757, -981724756, -981724755, -981724754, -981724753, -981724752,
                          -981724751, -981724750, -981724749, -981724748, -981724747, -981724746, -981724745,
                          -981724744, -981724743, -981724742, -981724741, -981724740, -981724739, -981724738,
                          -981724737, -981724736, -981724735, -981724734, -979889752, -979889751, -979889750,
                          -979889749, -979889748, -979889747, -979889746, -979889745, -979889744, -979889743,
                          -979889742, -979889741, -979889740, -979889739, -979889738, -979889737, -979889736,
                          -979889735, -979889734, -979889733, -979889732, -979889731, -979889730, -979889729,
                          -979889728, -979889727, -979889726, -978054744, -978054743, -978054742, -978054741,
                          -978054740, -978054739, -978054738, -978054737, -978054736, -978054735, -978054734,
                          -978054733, -978054732, -978054731, -978054730, -978054729, -978054728, -978054727,
                          -978054726, -978054725, -978054724, -978054723, -978054722, -978054721, -978054720,
                          -978054719, -978054718, -976219736, -976219735, -976219734, -976219733, -976219732,
                          -976219731, -976219730, -976219729, -976219728, -976219727, -976219726, -976219725,
                          -976219724, -976219723, -976219722, -976219721, -976219720, -976219719, -976219718,
                          -976219717, -976219716, -976219715, -976219714, -976219713, -976219712, -976219711,
                          -976219710, -974384728, -974384727, -974384726, -974384725, -974384724, -974384723,
                          -974384722, -974384721, -974384720, -974384719, -974384718, -974384717, -974384716,
                          -974384715, -974384714, -974384713, -974384712, -974384711, -974384710, -974384709,
                          -974384708, -974384707, -974384706, -974384705, -974384704, -974384703, -974384702,
                          -972549720, -972549719, -972549718, -972549717, -972549716, -972549715, -972549714,
                          -972549713, -972549712, -972549711, -972549710, -972549709, -972549708, -972549707,
                          -972549706, -972549705, -972549704, -972549703, -972549702, -972549701, -972549700,
                          -972549699, -972549698, -972549697, -972549696, -972549695, -972549694, -970714712,
                          -970714711, -970714710, -970714709, -970714708, -970714707, -970714706, -970714705,
                          -970714704, -970714703, -970714702, -970714701, -970714700, -970714699, -970714698,
                          -970714697, -970714696, -970714695, -970714694, -970714693, -970714692, -970714691,
                          -970714690, -970714689, -970714688, -970714687, -970714686, -968879704, -968879703,
                          -968879702, -968879701, -968879700, -968879699, -968879698, -968879697, -968879696,
                          -968879695, -968879694, -968879693, -968879692, -968879691, -968879690, -968879689,
                          -968879688, -968879687, -968879686, -968879685, -968879684, -968879683, -968879682,
                          -968879681, -968879680, -968879679, -968879678, -967044696, -967044695, -967044694,
                          -967044693, -967044692, -967044691, -967044690, -967044689, -967044688, -967044687,
                          -967044686, -967044685, -967044684, -967044683, -967044682, -967044681, -967044680,
                          -967044679, -967044678, -967044677, -967044676, -967044675, -967044674, -967044673,
                          -967044672, -967044671, -967044670, -965209688, -965209687, -965209686, -965209685,
                          -965209684, -965209683, -965209682, -965209681, -965209680, -965209679, -965209678,
                          -965209677, -965209676, -965209675, -965209674, -965209673, -965209672, -965209671,
                          -965209670, -965209669, -965209668, -965209667, -965209666, -965209665, -965209664,
                          -965209663, -965209662, -963374680, -963374679, -963374678, -963374677, -963374676,
                          -963374675, -963374674, -963374673, -963374672, -963374671, -963374670, -963374669,
                          -963374668, -963374667, -963374666, -963374665, -963374664, -963374663, -963374662,
                          -963374661, -963374660, -963374659, -963374658, -963374657, -963374656, -963374655,
                          -963374654, -961539672, -961539671, -961539670, -961539669, -961539668, -961539667,
                          -961539666, -961539665, -961539664, -961539663, -961539662, -961539661, -961539660,
                          -961539659, -961539658, -961539657, -961539656, -961539655, -961539654, -961539653,
                          -961539652, -961539651, -961539650, -961539649, -961539648, -961539647, -961539646,
                          -959704664, -959704663, -959704662, -959704661, -959704660, -959704659, -959704658,
                          -959704657, -959704656, -959704655, -959704654, -959704653, -959704652, -959704651,
                          -959704650, -959704649, -959704648, -959704647, -959704646, -959704645, -959704644,
                          -959704643, -959704642, -959704641, -959704640, -959704639, -959704638, -957869656,
                          -957869655, -957869654, -957869653, -957869652, -957869651, -957869650, -957869649,
                          -957869648, -957869647, -957869646, -957869645, -957869644, -957869643, -957869642,
                          -957869641, -957869640, -957869639, -957869638, -957869637, -957869636, -957869635,
                          -957869634, -957869633, -957869632, -957869631, -957869630, -956034648, -956034647,
                          -956034646, -956034645, -956034644, -956034643, -956034642, -956034641, -956034640,
                          -956034639, -956034638, -956034637, -956034636, -956034635, -956034634, -956034633,
                          -956034632, -956034631, -956034630, -956034629, -956034628, -956034627, -956034626,
                          -956034625, -956034624, -956034623, -956034622, -954199640, -954199639, -954199638,
                          -954199637, -954199636, -954199635, -954199634, -954199633, -954199632, -954199631,
                          -954199630, -954199629, -954199628, -954199627, -954199626, -954199625, -954199624,
                          -954199623, -954199622, -954199621, -954199620, -954199619, -954199618, -954199617,
                          -954199616, -954199615, -954199614, -952364632, -952364631, -952364630, -952364629,
                          -952364628, -952364627, -952364626, -952364625, -952364624, -952364623, -952364622,
                          -952364621, -952364620, -952364619, -952364618, -952364617, -952364616, -952364615,
                          -952364614, -952364613, -952364612, -952364611, -952364610, -952364609, -952364608,
                          -952364607, -952364606, -950529624, -950529623, -950529622, -950529621, -950529620,
                          -950529619, -950529618, -950529617, -950529616, -950529615, -950529614, -950529613,
                          -950529612, -950529611, -950529610, -950529609, -950529608, -950529607, -950529606,
                          -950529605, -950529604, -950529603, -950529602, -950529601, -950529600, -950529599,
                          -950529598, -948694616, -948694615, -948694614, -948694613, -948694612, -948694611,
                          -948694610, -948694609, -948694608, -948694607, -948694606, -948694605, -948694604,
                          -948694603, -948694602, -948694601, -948694600, -948694599, -948694598, -948694597,
                          -948694596, -948694595, -948694594, -948694593, -948694592, -948694591, -948694590,
                          -946859608, -946859607, -946859606, -946859605, -946859604, -946859603, -946859602,
                          -946859601, -946859600, -946859599, -946859598, -946859597, -946859596, -946859595,
                          -946859594, -946859593, -946859592, -946859591, -946859590, -946859589, -946859588,
                          -946859587, -946859586, -946859585, -946859584, -946859583, -946859582, -945024600,
                          -945024599, -945024598, -945024597, -945024596, -945024595, -945024594, -945024593,
                          -945024592, -945024591, -945024590, -945024589, -945024588, -945024587, -945024586,
                          -945024585, -945024584, -945024583, -945024582, -945024581, -945024580, -945024579,
                          -945024578, -945024577, -945024576, -945024575, -945024574, -943189592, -943189591,
                          -943189590, -943189589, -943189588, -943189587, -943189586, -943189585, -943189584,
                          -943189583, -943189582, -943189581, -943189580, -943189579, -943189578, -943189577,
                          -943189576, -943189575, -943189574, -943189573, -943189572, -943189571, -943189570,
                          -943189569, -943189568, -943189567, -943189566, -941354584, -941354583, -941354582,
                          -941354581, -941354580, -941354579, -941354578, -941354577, -941354576, -941354575,
                          -941354574, -941354573, -941354572, -941354571, -941354570, -941354569, -941354568,
                          -941354567, -941354566, -941354565, -941354564, -941354563, -941354562, -941354561,
                          -941354560, -941354559, -941354558, -939519576, -939519575, -939519574, -939519573,
                          -939519572, -939519571, -939519570, -939519569, -939519568, -939519567, -939519566,
                          -939519565, -939519564, -939519563, -939519562, -939519561, -939519560, -939519559,
                          -939519558, -939519557, -939519556, -939519555, -939519554, -939519553, -939519552,
                          -939519551, -939519550, -937684568, -937684567, -937684566, -937684565, -937684564,
                          -937684563, -937684562, -937684561, -937684560, -937684559, -937684558, -937684557,
                          -937684556, -937684555, -937684554, -937684553, -937684552, -937684551, -937684550,
                          -937684549, -937684548, -937684547, -937684546, -937684545, -937684544, -937684543,
                          -937684542, -935849560, -935849559, -935849558, -935849557, -935849556, -935849555,
                          -935849554, -935849553, -935849552, -935849551, -935849550, -935849549, -935849548,
                          -935849547, -935849546, -935849545, -935849544, -935849543, -935849542, -935849541,
                          -935849540, -935849539, -935849538, -935849537, -935849536, -935849535, -935849534,
                          -934014552, -934014551, -934014550, -934014549, -934014548, -934014547, -934014546,
                          -934014545, -934014544, -934014543, -934014542, -934014541, -934014540, -934014539,
                          -934014538, -934014537, -934014536, -934014535, -934014534, -934014533, -934014532,
                          -934014531, -934014530, -934014529, -934014528, -934014527, -934014526, -932179544,
                          -932179543, -932179542, -932179541, -932179540, -932179539, -932179538, -932179537,
                          -932179536, -932179535, -932179534, -932179533, -932179532, -932179531, -932179530,
                          -932179529, -932179528, -932179527, -932179526, -932179525, -932179524, -932179523,
                          -932179522, -932179521, -932179520, -932179519, -932179518, -930344536, -930344535,
                          -930344534, -930344533, -930344532, -930344531, -930344530, -930344529, -930344528,
                          -930344527, -930344526, -930344525, -930344524, -930344523, -930344522, -930344521,
                          -930344520, -930344519, -930344518, -930344517, -930344516, -930344515, -930344514,
                          -930344513, -930344512, -930344511, -930344510, -928509528, -928509527, -928509526,
                          -928509525, -928509524, -928509523, -928509522, -928509521, -928509520, -928509519,
                          -928509518, -928509517, -928509516, -928509515, -928509514, -928509513, -928509512,
                          -928509511, -928509510, -928509509, -928509508, -928509507, -928509506, -928509505,
                          -928509504, -928509503, -928509502, -926674520, -926674519, -926674518, -926674517,
                          -926674516, -926674515, -926674514, -926674513, -926674512, -926674511, -926674510,
                          -926674509, -926674508, -926674507, -926674506, -926674505, -926674504, -926674503,
                          -926674502, -926674501, -926674500, -926674499, -926674498, -926674497, -926674496,
                          -926674495, -926674494, -924839512, -924839511, -924839510, -924839509, -924839508,
                          -924839507, -924839506, -924839505, -924839504, -924839503, -924839502, -924839501,
                          -924839500, -924839499, -924839498, -924839497, -924839496, -924839495, -924839494,
                          -924839493, -924839492, -924839491, -924839490, -924839489, -924839488, -924839487,
                          -924839486, -923004504, -923004503, -923004502, -923004501, -923004500, -923004499,
                          -923004498, -923004497, -923004496, -923004495, -923004494, -923004493, -923004492,
                          -923004491, -923004490, -923004489, -923004488, -923004487, -923004486, -923004485,
                          -923004484, -923004483, -923004482, -923004481, -923004480, -923004479, -923004478,
                          -921169496, -921169495, -921169494, -921169493, -921169492, -921169491, -921169490,
                          -921169489, -921169488, -921169487, -921169486, -921169485, -921169484, -921169483,
                          -921169482, -921169481, -921169480, -921169479, -921169478, -921169477, -921169476,
                          -921169475, -921169474, -921169473, -921169472, -921169471, -921169470, -919334488,
                          -919334487, -919334486, -919334485, -919334484, -919334483, -919334482, -919334481,
                          -919334480, -919334479, -919334478, -919334477, -919334476, -919334475, -919334474,
                          -919334473, -919334472, -919334471, -919334470, -919334469, -919334468, -919334467,
                          -919334466, -919334465, -919334464, -919334463, -919334462, -917499480, -917499479,
                          -917499478, -917499477, -917499476, -917499475, -917499474, -917499473, -917499472,
                          -917499471, -917499470, -917499469, -917499468, -917499467, -917499466, -917499465,
                          -917499464, -917499463, -917499462, -917499461, -917499460, -917499459, -917499458,
                          -917499457, -917499456, -917499455, -917499454, -915664472, -915664471, -915664470,
                          -915664469, -915664468, -915664467, -915664466, -915664465, -915664464, -915664463,
                          -915664462, -915664461, -915664460, -915664459, -915664458, -915664457, -915664456,
                          -915664455, -915664454, -915664453, -915664452, -915664451, -915664450, -915664449,
                          -915664448, -915664447, -915664446, -913829464, -913829463, -913829462, -913829461,
                          -913829460, -913829459, -913829458, -913829457, -913829456, -913829455, -913829454,
                          -913829453, -913829452, -913829451, -913829450, -913829449, -913829448, -913829447,
                          -913829446, -913829445, -913829444, -913829443, -913829442, -913829441, -913829440,
                          -913829439, -913829438, -911994456, -911994455, -911994454, -911994453, -911994452,
                          -911994451, -911994450, -911994449, -911994448, -911994447, -911994446, -911994445,
                          -911994444, -911994443, -911994442, -911994441, -911994440, -911994439, -911994438,
                          -911994437, -911994436, -911994435, -911994434, -911994433, -911994432, -911994431,
                          -911994430, -910159448, -910159447, -910159446, -910159445, -910159444, -910159443,
                          -910159442, -910159441, -910159440, -910159439, -910159438, -910159437, -910159436,
                          -910159435, -910159434, -910159433, -910159432, -910159431, -910159430, -910159429,
                          -910159428, -910159427, -910159426, -910159425, -910159424, -910159423, -910159422,
                          -908324440, -908324439, -908324438, -908324437, -908324436, -908324435, -908324434,
                          -908324433, -908324432, -908324431, -908324430, -908324429, -908324428, -908324427,
                          -908324426, -908324425, -908324424, -908324423, -908324422, -908324421, -908324420,
                          -908324419, -908324418, -908324417, -908324416, -908324415, -908324414, -906489432,
                          -906489431, -906489430, -906489429, -906489428, -906489427, -906489426, -906489425,
                          -906489424, -906489423, -906489422, -906489421, -906489420, -906489419, -906489418,
                          -906489417, -906489416, -906489415, -906489414, -906489413, -906489412, -906489411,
                          -906489410, -906489409, -906489408, -906489407, -906489406, -904654424, -904654423,
                          -904654422, -904654421, -904654420, -904654419, -904654418, -904654417, -904654416,
                          -904654415, -904654414, -904654413, -904654412, -904654411, -904654410, -904654409,
                          -904654408, -904654407, -904654406, -904654405, -904654404, -904654403, -904654402,
                          -904654401, -904654400, -904654399, -904654398, -902819416, -902819415, -902819414,
                          -902819413, -902819412, -902819411, -902819410, -902819409, -902819408, -902819407,
                          -902819406, -902819405, -902819404, -902819403, -902819402, -902819401, -902819400,
                          -902819399, -902819398, -902819397, -902819396, -902819395, -902819394, -902819393,
                          -902819392, -902819391, -902819390, -900984408, -900984407, -900984406, -900984405,
                          -900984404, -900984403, -900984402, -900984401, -900984400, -900984399, -900984398,
                          -900984397, -900984396, -900984395, -900984394, -900984393, -900984392, -900984391,
                          -900984390, -900984389, -900984388, -900984387, -900984386, -900984385, -900984384,
                          -900984383, -900984382, -899149400, -899149399, -899149398, -899149397, -899149396,
                          -899149395, -899149394, -899149393, -899149392, -899149391, -899149390, -899149389,
                          -899149388, -899149387, -899149386, -899149385, -899149384, -899149383, -899149382,
                          -899149381, -899149380, -899149379, -899149378, -899149377, -899149376, -899149375,
                          -899149374, -897314392, -897314391, -897314390, -897314389, -897314388, -897314387,
                          -897314386, -897314385, -897314384, -897314383, -897314382, -897314381, -897314380,
                          -897314379, -897314378, -897314377, -897314376, -897314375, -897314374, -897314373,
                          -897314372, -897314371, -897314370, -897314369, -897314368, -897314367, -897314366,
                          -895479384, -895479383, -895479382, -895479381, -895479380, -895479379, -895479378,
                          -895479377, -895479376, -895479375, -895479374, -895479373, -895479372, -895479371,
                          -895479370, -895479369, -895479368, -895479367, -895479366, -895479365, -895479364,
                          -895479363, -895479362, -895479361, -895479360, -895479359, -895479358, -893644376,
                          -893644375, -893644374, -893644373, -893644372, -893644371, -893644370, -893644369,
                          -893644368, -893644367, -893644366, -893644365, -893644364, -893644363, -893644362,
                          -893644361, -893644360, -893644359, -893644358, -893644357, -893644356, -893644355,
                          -893644354, -893644353, -893644352, -893644351, -893644350, -891809368, -891809367,
                          -891809366, -891809365, -891809364, -891809363, -891809362, -891809361, -891809360,
                          -891809359, -891809358, -891809357, -891809356, -891809355, -891809354, -891809353,
                          -891809352, -891809351, -891809350, -891809349, -891809348, -891809347, -891809346,
                          -891809345, -891809344, -891809343, -891809342, -889974360, -889974359, -889974358,
                          -889974357, -889974356, -889974355, -889974354, -889974353, -889974352, -889974351,
                          -889974350, -889974349, -889974348, -889974347, -889974346, -889974345, -889974344,
                          -889974343, -889974342, -889974341, -889974340, -889974339, -889974338, -889974337,
                          -889974336, -889974335, -889974334, -888139352, -888139351, -888139350, -888139349,
                          -888139348, -888139347, -888139346, -888139345, -888139344, -888139343, -888139342,
                          -888139341, -888139340, -888139339, -888139338, -888139337, -888139336, -888139335,
                          -888139334, -888139333, -888139332, -888139331, -888139330, -888139329, -888139328,
                          -888139327, -888139326, -886304344, -886304343, -886304342, -886304341, -886304340,
                          -886304339, -886304338, -886304337, -886304336, -886304335, -886304334, -886304333,
                          -886304332, -886304331, -886304330, -886304329, -886304328, -886304327, -886304326,
                          -886304325, -886304324, -886304323, -886304322, -886304321, -886304320, -886304319,
                          -886304318, -884469336, -884469335, -884469334, -884469333, -884469332, -884469331,
                          -884469330, -884469329, -884469328, -884469327, -884469326, -884469325, -884469324,
                          -884469323, -884469322, -884469321, -884469320, -884469319, -884469318, -884469317,
                          -884469316, -884469315, -884469314, -884469313, -884469312, -884469311, -884469310,
                          -882634328, -882634327, -882634326, -882634325, -882634324, -882634323, -882634322,
                          -882634321, -882634320, -882634319, -882634318, -882634317, -882634316, -882634315,
                          -882634314, -882634313, -882634312, -882634311, -882634310, -882634309, -882634308,
                          -882634307, -882634306, -882634305, -882634304, -882634303, -882634302, -880799320,
                          -880799319, -880799318, -880799317, -880799316, -880799315, -880799314, -880799313,
                          -880799312, -880799311, -880799310, -880799309, -880799308, -880799307, -880799306,
                          -880799305, -880799304, -880799303, -880799302, -880799301, -880799300, -880799299,
                          -880799298, -880799297, -880799296, -880799295, -880799294, -878964312, -878964311,
                          -878964310, -878964309, -878964308, -878964307, -878964306, -878964305, -878964304,
                          -878964303, -878964302, -878964301, -878964300, -878964299, -878964298, -878964297,
                          -878964296, -878964295, -878964294, -878964293, -878964292, -878964291, -878964290,
                          -878964289, -878964288, -878964287, -878964286, -877129304, -877129303, -877129302,
                          -877129301, -877129300, -877129299, -877129298, -877129297, -877129296, -877129295,
                          -877129294, -877129293, -877129292, -877129291, -877129290, -877129289, -877129288,
                          -877129287, -877129286, -877129285, -877129284, -877129283, -877129282, -877129281,
                          -877129280, -877129279, -877129278, -875294296, -875294295, -875294294, -875294293,
                          -875294292, -875294291, -875294290, -875294289, -875294288, -875294287, -875294286,
                          -875294285, -875294284, -875294283, -875294282, -875294281, -875294280, -875294279,
                          -875294278, -875294277, -875294276, -875294275, -875294274, -875294273, -875294272,
                          -875294271, -875294270, -873459288, -873459287, -873459286, -873459285, -873459284,
                          -873459283, -873459282, -873459281, -873459280, -873459279, -873459278, -873459277,
                          -873459276, -873459275, -873459274, -873459273, -873459272, -873459271, -873459270,
                          -873459269, -873459268, -873459267, -873459266, -873459265, -873459264, -873459263,
                          -873459262, -871624280, -871624279, -871624278, -871624277, -871624276, -871624275,
                          -871624274, -871624273, -871624272, -871624271, -871624270, -871624269, -871624268,
                          -871624267, -871624266, -871624265, -871624264, -871624263, -871624262, -871624261,
                          -871624260, -871624259, -871624258, -871624257, -871624256, -871624255, -871624254,
                          -869789272, -869789271, -869789270, -869789269, -869789268, -869789267, -869789266,
                          -869789265, -869789264, -869789263, -869789262, -869789261, -869789260, -869789259,
                          -869789258, -869789257, -869789256, -869789255, -869789254, -869789253, -869789252,
                          -869789251, -869789250, -869789249, -869789248, -869789247, -869789246, -867954264,
                          -867954263, -867954262, -867954261, -867954260, -867954259, -867954258, -867954257,
                          -867954256, -867954255, -867954254, -867954253, -867954252, -867954251, -867954250,
                          -867954249, -867954248, -867954247, -867954246, -867954245, -867954244, -867954243,
                          -867954242, -867954241, -867954240, -867954239, -867954238, -866119256, -866119255,
                          -866119254, -866119253, -866119252, -866119251, -866119250, -866119249, -866119248,
                          -866119247, -866119246, -866119245, -866119244, -866119243, -866119242, -866119241,
                          -866119240, -866119239, -866119238, -866119237, -866119236, -866119235, -866119234,
                          -866119233, -866119232, -866119231, -866119230, -864284248, -864284247, -864284246,
                          -864284245, -864284244, -864284243, -864284242, -864284241, -864284240, -864284239,
                          -864284238, -864284237, -864284236, -864284235, -864284234, -864284233, -864284232,
                          -864284231, -864284230, -864284229, -864284228, -864284227, -864284226, -864284225,
                          -864284224, -864284223, -864284222, -862449240, -862449239, -862449238, -862449237,
                          -862449236, -862449235, -862449234, -862449233, -862449232, -862449231, -862449230,
                          -862449229, -862449228, -862449227, -862449226, -862449225, -862449224, -862449223,
                          -862449222, -862449221, -862449220, -862449219, -862449218, -862449217, -862449216,
                          -862449215, -862449214, -860614232, -860614231, -860614230, -860614229, -860614228,
                          -860614227, -860614226, -860614225, -860614224, -860614223, -860614222, -860614221,
                          -860614220, -860614219, -860614218, -860614217, -860614216, -860614215, -860614214,
                          -860614213, -860614212, -860614211, -860614210, -860614209, -860614208, -860614207,
                          -860614206, -858779224, -858779223, -858779222, -858779221, -858779220, -858779219,
                          -858779218, -858779217, -858779216, -858779215, -858779214, -858779213, -858779212,
                          -858779211, -858779210, -858779209, -858779208, -858779207, -858779206, -858779205,
                          -858779204, -858779203, -858779202, -858779201, -858779200, -858779199, -858779198,
                          -856944216, -856944215, -856944214, -856944213, -856944212, -856944211, -856944210,
                          -856944209, -856944208, -856944207, -856944206, -856944205, -856944204, -856944203,
                          -856944202, -856944201, -856944200, -856944199, -856944198, -856944197, -856944196,
                          -856944195, -856944194, -856944193, -856944192, -856944191, -856944190, -855109208,
                          -855109207, -855109206, -855109205, -855109204, -855109203, -855109202, -855109201,
                          -855109200, -855109199, -855109198, -855109197, -855109196, -855109195, -855109194,
                          -855109193, -855109192, -855109191, -855109190, -855109189, -855109188, -855109187,
                          -855109186, -855109185, -855109184, -855109183, -855109182, -853274200, -853274199,
                          -853274198, -853274197, -853274196, -853274195, -853274194, -853274193, -853274192,
                          -853274191, -853274190, -853274189, -853274188, -853274187, -853274186, -853274185,
                          -853274184, -853274183, -853274182, -853274181, -853274180, -853274179, -853274178,
                          -853274177, -853274176, -853274175, -853274174, -851439192, -851439191, -851439190,
                          -851439189, -851439188, -851439187, -851439186, -851439185, -851439184, -851439183,
                          -851439182, -851439181, -851439180, -851439179, -851439178, -851439177, -851439176,
                          -851439175, -851439174, -851439173, -851439172, -851439171, -851439170, -851439169,
                          -851439168, -851439167, -851439166, -849604184, -849604183, -849604182, -849604181,
                          -849604180, -849604179, -849604178, -849604177, -849604176, -849604175, -849604174,
                          -849604173, -849604172, -849604171, -849604170, -849604169, -849604168, -849604167,
                          -849604166, -849604165, -849604164, -849604163, -849604162, -849604161, -849604160,
                          -849604159, -849604158, -847769176, -847769175, -847769174, -847769173, -847769172,
                          -847769171, -847769170, -847769169, -847769168, -847769167, -847769166, -847769165,
                          -847769164, -847769163, -847769162, -847769161, -847769160, -847769159, -847769158,
                          -847769157, -847769156, -847769155, -847769154, -847769153, -847769152, -847769151,
                          -847769150, -845934168, -845934167, -845934166, -845934165, -845934164, -845934163,
                          -845934162, -845934161, -845934160, -845934159, -845934158, -845934157, -845934156,
                          -845934155, -845934154, -845934153, -845934152, -845934151, -845934150, -845934149,
                          -845934148, -845934147, -845934146, -845934145, -845934144, -845934143, -845934142,
                          -844099160, -844099159, -844099158, -844099157, -844099156, -844099155, -844099154,
                          -844099153, -844099152, -844099151, -844099150, -844099149, -844099148, -844099147,
                          -844099146, -844099145, -844099144, -844099143, -844099142, -844099141, -844099140,
                          -844099139, -844099138, -844099137, -844099136, -844099135, -844099134, -842264152,
                          -842264151, -842264150, -842264149, -842264148, -842264147, -842264146, -842264145,
                          -842264144, -842264143, -842264142, -842264141, -842264140, -842264139, -842264138,
                          -842264137, -842264136, -842264135, -842264134, -842264133, -842264132, -842264131,
                          -842264130, -842264129, -842264128, -842264127, -842264126, -840429144, -840429143,
                          -840429142, -840429141, -840429140, -840429139, -840429138, -840429137, -840429136,
                          -840429135, -840429134, -840429133, -840429132, -840429131, -840429130, -840429129,
                          -840429128, -840429127, -840429126, -840429125, -840429124, -840429123, -840429122,
                          -840429121, -840429120, -840429119, -840429118, -838594136, -838594135, -838594134,
                          -838594133, -838594132, -838594131, -838594130, -838594129, -838594128, -838594127,
                          -838594126, -838594125, -838594124, -838594123, -838594122, -838594121, -838594120,
                          -838594119, -838594118, -838594117, -838594116, -838594115, -838594114, -838594113,
                          -838594112, -838594111, -838594110, -836759128, -836759127, -836759126, -836759125,
                          -836759124, -836759123, -836759122, -836759121, -836759120, -836759119, -836759118,
                          -836759117, -836759116, -836759115, -836759114, -836759113, -836759112, -836759111,
                          -836759110, -836759109, -836759108, -836759107, -836759106, -836759105, -836759104,
                          -836759103, -836759102, -834924120, -834924119, -834924118, -834924117, -834924116,
                          -834924115, -834924114, -834924113, -834924112, -834924111, -834924110, -834924109,
                          -834924108, -834924107, -834924106, -834924105, -834924104, -834924103, -834924102,
                          -834924101, -834924100, -834924099, -834924098, -834924097, -834924096, -834924095,
                          -834924094, -833089112, -833089111, -833089110, -833089109, -833089108, -833089107,
                          -833089106, -833089105, -833089104, -833089103, -833089102, -833089101, -833089100,
                          -833089099, -833089098, -833089097, -833089096, -833089095, -833089094, -833089093,
                          -833089092, -833089091, -833089090, -833089089, -833089088, -833089087, -833089086,
                          -831254104, -831254103, -831254102, -831254101, -831254100, -831254099, -831254098,
                          -831254097, -831254096, -831254095, -831254094, -831254093, -831254092, -831254091,
                          -831254090, -831254089, -831254088, -831254087, -831254086, -831254085, -831254084,
                          -831254083, -831254082, -831254081, -831254080, -831254079, -831254078, -829419096,
                          -829419095, -829419094, -829419093, -829419092, -829419091, -829419090, -829419089,
                          -829419088, -829419087, -829419086, -829419085, -829419084, -829419083, -829419082,
                          -829419081, -829419080, -829419079, -829419078, -829419077, -829419076, -829419075,
                          -829419074, -829419073, -829419072, -829419071, -829419070, -827584088, -827584087,
                          -827584086, -827584085, -827584084, -827584083, -827584082, -827584081, -827584080,
                          -827584079, -827584078, -827584077, -827584076, -827584075, -827584074, -827584073,
                          -827584072, -827584071, -827584070, -827584069, -827584068, -827584067, -827584066,
                          -827584065, -827584064, -827584063, -827584062, -825749080, -825749079, -825749078,
                          -825749077, -825749076, -825749075, -825749074, -825749073, -825749072, -825749071,
                          -825749070, -825749069, -825749068, -825749067, -825749066, -825749065, -825749064,
                          -825749063, -825749062, -825749061, -825749060, -825749059, -825749058, -825749057,
                          -825749056, -825749055, -825749054, -823914072, -823914071, -823914070, -823914069,
                          -823914068, -823914067, -823914066, -823914065, -823914064, -823914063, -823914062,
                          -823914061, -823914060, -823914059, -823914058, -823914057, -823914056, -823914055,
                          -823914054, -823914053, -823914052, -823914051, -823914050, -823914049, -823914048,
                          -823914047, -823914046, -822079064, -822079063, -822079062, -822079061, -822079060,
                          -822079059, -822079058, -822079057, -822079056, -822079055, -822079054, -822079053,
                          -822079052, -822079051, -822079050, -822079049, -822079048, -822079047, -822079046,
                          -822079045, -822079044, -822079043, -822079042, -822079041, -822079040, -822079039,
                          -822079038, -820244056, -820244055, -820244054, -820244053, -820244052, -820244051,
                          -820244050, -820244049, -820244048, -820244047, -820244046, -820244045, -820244044,
                          -820244043, -820244042, -820244041, -820244040, -820244039, -820244038, -820244037,
                          -820244036, -820244035, -820244034, -820244033, -820244032, -820244031, -820244030,
                          -818409048, -818409047, -818409046, -818409045, -818409044, -818409043, -818409042,
                          -818409041, -818409040, -818409039, -818409038, -818409037, -818409036, -818409035,
                          -818409034, -818409033, -818409032, -818409031, -818409030, -818409029, -818409028,
                          -818409027, -818409026, -818409025, -818409024, -818409023, -818409022, -816574040,
                          -816574039, -816574038, -816574037, -816574036, -816574035, -816574034, -816574033,
                          -816574032, -816574031, -816574030, -816574029, -816574028, -816574027, -816574026,
                          -816574025, -816574024, -816574023, -816574022, -816574021, -816574020, -816574019,
                          -816574018, -816574017, -816574016, -816574015, -816574014, -814739032, -814739031,
                          -814739030, -814739029, -814739028, -814739027, -814739026, -814739025, -814739024,
                          -814739023, -814739022, -814739021, -814739020, -814739019, -814739018, -814739017,
                          -814739016, -814739015, -814739014, -814739013, -814739012, -814739011, -814739010,
                          -814739009, -814739008, -814739007, -814739006, -812904024, -812904023, -812904022,
                          -812904021, -812904020, -812904019, -812904018, -812904017, -812904016, -812904015,
                          -812904014, -812904013, -812904012, -812904011, -812904010, -812904009, -812904008,
                          -812904007, -812904006, -812904005, -812904004, -812904003, -812904002, -812904001,
                          -812904000, -812903999, -812903998, -811069016, -811069015, -811069014, -811069013,
                          -811069012, -811069011, -811069010, -811069009, -811069008, -811069007, -811069006,
                          -811069005, -811069004, -811069003, -811069002, -811069001, -811069000, -811068999,
                          -811068998, -811068997, -811068996, -811068995, -811068994, -811068993, -811068992,
                          -811068991, -811068990, -809234008, -809234007, -809234006, -809234005, -809234004,
                          -809234003, -809234002, -809234001, -809234000, -809233999, -809233998, -809233997,
                          -809233996, -809233995, -809233994, -809233993, -809233992, -809233991, -809233990,
                          -809233989, -809233988, -809233987, -809233986, -809233985, -809233984, -809233983,
                          -809233982, -807399000, -807398999, -807398998, -807398997, -807398996, -807398995,
                          -807398994, -807398993, -807398992, -807398991, -807398990, -807398989, -807398988,
                          -807398987, -807398986, -807398985, -807398984, -807398983, -807398982, -807398981,
                          -807398980, -807398979, -807398978, -807398977, -807398976, -807398975, -807398974,
                          -805563992, -805563991, -805563990, -805563989, -805563988, -805563987, -805563986,
                          -805563985, -805563984, -805563983, -805563982, -805563981, -805563980, -805563979,
                          -805563978, -805563977, -805563976, -805563975, -805563974, -805563973, -805563972,
                          -805563971, -805563970, -805563969, -805563968, -805563967, -805563966, -803728984,
                          -803728983, -803728982, -803728981, -803728980, -803728979, -803728978, -803728977,
                          -803728976, -803728975, -803728974, -803728973, -803728972, -803728971, -803728970,
                          -803728969, -803728968, -803728967, -803728966, -803728965, -803728964, -803728963,
                          -803728962, -803728961, -803728960, -803728959, -803728958, -801893976, -801893975,
                          -801893974, -801893973, -801893972, -801893971, -801893970, -801893969, -801893968,
                          -801893967, -801893966, -801893965, -801893964, -801893963, -801893962, -801893961,
                          -801893960, -801893959, -801893958, -801893957, -801893956, -801893955, -801893954,
                          -801893953, -801893952, -801893951, -801893950, -800058968, -800058967, -800058966,
                          -800058965, -800058964, -800058963, -800058962, -800058961, -800058960, -800058959,
                          -800058958, -800058957, -800058956, -800058955, -800058954, -800058953, -800058952,
                          -800058951, -800058950, -800058949, -800058948, -800058947, -800058946, -800058945,
                          -800058944, -800058943, -800058942, -798223960, -798223959, -798223958, -798223957,
                          -798223956, -798223955, -798223954, -798223953, -798223952, -798223951, -798223950,
                          -798223949, -798223948, -798223947, -798223946, -798223945, -798223944, -798223943,
                          -798223942, -798223941, -798223940, -798223939, -798223938, -798223937, -798223936,
                          -798223935, -798223934, -796388952, -796388951, -796388950, -796388949, -796388948,
                          -796388947, -796388946, -796388945, -796388944, -796388943, -796388942, -796388941,
                          -796388940, -796388939, -796388938, -796388937, -796388936, -796388935, -796388934,
                          -796388933, -796388932, -796388931, -796388930, -796388929, -796388928, -796388927,
                          -796388926, -794553944, -794553943, -794553942, -794553941, -794553940, -794553939,
                          -794553938, -794553937, -794553936, -794553935, -794553934, -794553933, -794553932,
                          -794553931, -794553930, -794553929, -794553928, -794553927, -794553926, -794553925,
                          -794553924, -794553923, -794553922, -794553921, -794553920, -794553919, -794553918,
                          -792718936, -792718935, -792718934, -792718933, -792718932, -792718931, -792718930,
                          -792718929, -792718928, -792718927, -792718926, -792718925, -792718924, -792718923,
                          -792718922, -792718921, -792718920, -792718919, -792718918, -792718917, -792718916,
                          -792718915, -792718914, -792718913, -792718912, -792718911, -792718910, -790883928,
                          -790883927, -790883926, -790883925, -790883924, -790883923, -790883922, -790883921,
                          -790883920, -790883919, -790883918, -790883917, -790883916, -790883915, -790883914,
                          -790883913, -790883912, -790883911, -790883910, -790883909, -790883908, -790883907,
                          -790883906, -790883905, -790883904, -790883903, -790883902, -789048920, -789048919,
                          -789048918, -789048917, -789048916, -789048915, -789048914, -789048913, -789048912,
                          -789048911, -789048910, -789048909, -789048908, -789048907, -789048906, -789048905,
                          -789048904, -789048903, -789048902, -789048901, -789048900, -789048899, -789048898,
                          -789048897, -789048896, -789048895, -789048894, -787213912, -787213911, -787213910,
                          -787213909, -787213908, -787213907, -787213906, -787213905, -787213904, -787213903,
                          -787213902, -787213901, -787213900, -787213899, -787213898, -787213897, -787213896,
                          -787213895, -787213894, -787213893, -787213892, -787213891, -787213890, -787213889,
                          -787213888, -787213887, -787213886, -785378904, -785378903, -785378902, -785378901,
                          -785378900, -785378899, -785378898, -785378897, -785378896, -785378895, -785378894,
                          -785378893, -785378892, -785378891, -785378890, -785378889, -785378888, -785378887,
                          -785378886, -785378885, -785378884, -785378883, -785378882, -785378881, -785378880,
                          -785378879, -785378878, -783543896, -783543895, -783543894, -783543893, -783543892,
                          -783543891, -783543890, -783543889, -783543888, -783543887, -783543886, -783543885,
                          -783543884, -783543883, -783543882, -783543881, -783543880, -783543879, -783543878,
                          -783543877, -783543876, -783543875, -783543874, -783543873, -783543872, -783543871,
                          -783543870, -781708888, -781708887, -781708886, -781708885, -781708884, -781708883,
                          -781708882, -781708881, -781708880, -781708879, -781708878, -781708877, -781708876,
                          -781708875, -781708874, -781708873, -781708872, -781708871, -781708870, -781708869,
                          -781708868, -781708867, -781708866, -781708865, -781708864, -781708863, -781708862,
                          -779873880, -779873879, -779873878, -779873877, -779873876, -779873875, -779873874,
                          -779873873, -779873872, -779873871, -779873870, -779873869, -779873868, -779873867,
                          -779873866, -779873865, -779873864, -779873863, -779873862, -779873861, -779873860,
                          -779873859, -779873858, -779873857, -779873856, -779873855, -779873854, -778038872,
                          -778038871, -778038870, -778038869, -778038868, -778038867, -778038866, -778038865,
                          -778038864, -778038863, -778038862, -778038861, -778038860, -778038859, -778038858,
                          -778038857, -778038856, -778038855, -778038854, -778038853, -778038852, -778038851,
                          -778038850, -778038849, -778038848, -778038847, -778038846, -776203864, -776203863,
                          -776203862, -776203861, -776203860, -776203859, -776203858, -776203857, -776203856,
                          -776203855, -776203854, -776203853, -776203852, -776203851, -776203850, -776203849,
                          -776203848, -776203847, -776203846, -776203845, -776203844, -776203843, -776203842,
                          -776203841, -776203840, -776203839, -776203838, -774368856, -774368855, -774368854,
                          -774368853, -774368852, -774368851, -774368850, -774368849, -774368848, -774368847,
                          -774368846, -774368845, -774368844, -774368843, -774368842, -774368841, -774368840,
                          -774368839, -774368838, -774368837, -774368836, -774368835, -774368834, -774368833,
                          -774368832, -774368831, -774368830, -772533848, -772533847, -772533846, -772533845,
                          -772533844, -772533843, -772533842, -772533841, -772533840, -772533839, -772533838,
                          -772533837, -772533836, -772533835, -772533834, -772533833, -772533832, -772533831,
                          -772533830, -772533829, -772533828, -772533827, -772533826, -772533825, -772533824,
                          -772533823, -772533822, -770698840, -770698839, -770698838, -770698837, -770698836,
                          -770698835, -770698834, -770698833, -770698832, -770698831, -770698830, -770698829,
                          -770698828, -770698827, -770698826, -770698825, -770698824, -770698823, -770698822,
                          -770698821, -770698820, -770698819, -770698818, -770698817, -770698816, -770698815,
                          -770698814, -768863832, -768863831, -768863830, -768863829, -768863828, -768863827,
                          -768863826, -768863825, -768863824, -768863823, -768863822, -768863821, -768863820,
                          -768863819, -768863818, -768863817, -768863816, -768863815, -768863814, -768863813,
                          -768863812, -768863811, -768863810, -768863809, -768863808, -768863807, -768863806,
                          -767028824, -767028823, -767028822, -767028821, -767028820, -767028819, -767028818,
                          -767028817, -767028816, -767028815, -767028814, -767028813, -767028812, -767028811,
                          -767028810, -767028809, -767028808, -767028807, -767028806, -767028805, -767028804,
                          -767028803, -767028802, -767028801, -767028800, -767028799, -767028798, -765193816,
                          -765193815, -765193814, -765193813, -765193812, -765193811, -765193810, -765193809,
                          -765193808, -765193807, -765193806, -765193805, -765193804, -765193803, -765193802,
                          -765193801, -765193800, -765193799, -765193798, -765193797, -765193796, -765193795,
                          -765193794, -765193793, -765193792, -765193791, -765193790, -763358808, -763358807,
                          -763358806, -763358805, -763358804, -763358803, -763358802, -763358801, -763358800,
                          -763358799, -763358798, -763358797, -763358796, -763358795, -763358794, -763358793,
                          -763358792, -763358791, -763358790, -763358789, -763358788, -763358787, -763358786,
                          -763358785, -763358784, -763358783, -763358782, -761523800, -761523799, -761523798,
                          -761523797, -761523796, -761523795, -761523794, -761523793, -761523792, -761523791,
                          -761523790, -761523789, -761523788, -761523787, -761523786, -761523785, -761523784,
                          -761523783, -761523782, -761523781, -761523780, -761523779, -761523778, -761523777,
                          -761523776, -761523775, -761523774, -759688792, -759688791, -759688790, -759688789,
                          -759688788, -759688787, -759688786, -759688785, -759688784, -759688783, -759688782,
                          -759688781, -759688780, -759688779, -759688778, -759688777, -759688776, -759688775,
                          -759688774, -759688773, -759688772, -759688771, -759688770, -759688769, -759688768,
                          -759688767, -759688766, -757853784, -757853783, -757853782, -757853781, -757853780,
                          -757853779, -757853778, -757853777, -757853776, -757853775, -757853774, -757853773,
                          -757853772, -757853771, -757853770, -757853769, -757853768, -757853767, -757853766,
                          -757853765, -757853764, -757853763, -757853762, -757853761, -757853760, -757853759,
                          -757853758, -756018776, -756018775, -756018774, -756018773, -756018772, -756018771,
                          -756018770, -756018769, -756018768, -756018767, -756018766, -756018765, -756018764,
                          -756018763, -756018762, -756018761, -756018760, -756018759, -756018758, -756018757,
                          -756018756, -756018755, -756018754, -756018753, -756018752, -756018751, -756018750,
                          -754183768, -754183767, -754183766, -754183765, -754183764, -754183763, -754183762,
                          -754183761, -754183760, -754183759, -754183758, -754183757, -754183756, -754183755,
                          -754183754, -754183753, -754183752, -754183751, -754183750, -754183749, -754183748,
                          -754183747, -754183746, -754183745, -754183744, -754183743, -754183742, -752348760,
                          -752348759, -752348758, -752348757, -752348756, -752348755, -752348754, -752348753,
                          -752348752, -752348751, -752348750, -752348749, -752348748, -752348747, -752348746,
                          -752348745, -752348744, -752348743, -752348742, -752348741, -752348740, -752348739,
                          -752348738, -752348737, -752348736, -752348735, -752348734, -750513752, -750513751,
                          -750513750, -750513749, -750513748, -750513747, -750513746, -750513745, -750513744,
                          -750513743, -750513742, -750513741, -750513740, -750513739, -750513738, -750513737,
                          -750513736, -750513735, -750513734, -750513733, -750513732, -750513731, -750513730,
                          -750513729, -750513728, -750513727, -750513726, -748678744, -748678743, -748678742,
                          -748678741, -748678740, -748678739, -748678738, -748678737, -748678736, -748678735,
                          -748678734, -748678733, -748678732, -748678731, -748678730, -748678729, -748678728,
                          -748678727, -748678726, -748678725, -748678724, -748678723, -748678722, -748678721,
                          -748678720, -748678719, -748678718, -746843736, -746843735, -746843734, -746843733,
                          -746843732, -746843731, -746843730, -746843729, -746843728, -746843727, -746843726,
                          -746843725, -746843724, -746843723, -746843722, -746843721, -746843720, -746843719,
                          -746843718, -746843717, -746843716, -746843715, -746843714, -746843713, -746843712,
                          -746843711, -746843710, -745008728, -745008727, -745008726, -745008725, -745008724,
                          -745008723, -745008722, -745008721, -745008720, -745008719, -745008718, -745008717,
                          -745008716, -745008715, -745008714, -745008713, -745008712, -745008711, -745008710,
                          -745008709, -745008708, -745008707, -745008706, -745008705, -745008704, -745008703,
                          -745008702, -743173720, -743173719, -743173718, -743173717, -743173716, -743173715,
                          -743173714, -743173713, -743173712, -743173711, -743173710, -743173709, -743173708,
                          -743173707, -743173706, -743173705, -743173704, -743173703, -743173702, -743173701,
                          -743173700, -743173699, -743173698, -743173697, -743173696, -743173695, -743173694,
                          -741338712, -741338711, -741338710, -741338709, -741338708, -741338707, -741338706,
                          -741338705, -741338704, -741338703, -741338702, -741338701, -741338700, -741338699,
                          -741338698, -741338697, -741338696, -741338695, -741338694, -741338693, -741338692,
                          -741338691, -741338690, -741338689, -741338688, -741338687, -741338686, -739503704,
                          -739503703, -739503702, -739503701, -739503700, -739503699, -739503698, -739503697,
                          -739503696, -739503695, -739503694, -739503693, -739503692, -739503691, -739503690,
                          -739503689, -739503688, -739503687, -739503686, -739503685, -739503684, -739503683,
                          -739503682, -739503681, -739503680, -739503679, -739503678, -737668696, -737668695,
                          -737668694, -737668693, -737668692, -737668691, -737668690, -737668689, -737668688,
                          -737668687, -737668686, -737668685, -737668684, -737668683, -737668682, -737668681,
                          -737668680, -737668679, -737668678, -737668677, -737668676, -737668675, -737668674,
                          -737668673, -737668672, -737668671, -737668670, -735833688, -735833687, -735833686,
                          -735833685, -735833684, -735833683, -735833682, -735833681, -735833680, -735833679,
                          -735833678, -735833677, -735833676, -735833675, -735833674, -735833673, -735833672,
                          -735833671, -735833670, -735833669, -735833668, -735833667, -735833666, -735833665,
                          -735833664, -735833663, -735833662, -733998680, -733998679, -733998678, -733998677,
                          -733998676, -733998675, -733998674, -733998673, -733998672, -733998671, -733998670,
                          -733998669, -733998668, -733998667, -733998666, -733998665, -733998664, -733998663,
                          -733998662, -733998661, -733998660, -733998659, -733998658, -733998657, -733998656,
                          -733998655, -733998654, -732163672, -732163671, -732163670, -732163669, -732163668,
                          -732163667, -732163666, -732163665, -732163664, -732163663, -732163662, -732163661,
                          -732163660, -732163659, -732163658, -732163657, -732163656, -732163655, -732163654,
                          -732163653, -732163652, -732163651, -732163650, -732163649, -732163648, -732163647,
                          -732163646, -730328664, -730328663, -730328662, -730328661, -730328660, -730328659,
                          -730328658, -730328657, -730328656, -730328655, -730328654, -730328653, -730328652,
                          -730328651, -730328650, -730328649, -730328648, -730328647, -730328646, -730328645,
                          -730328644, -730328643, -730328642, -730328641, -730328640, -730328639, -730328638,
                          -728493656, -728493655, -728493654, -728493653, -728493652, -728493651, -728493650,
                          -728493649, -728493648, -728493647, -728493646, -728493645, -728493644, -728493643,
                          -728493642, -728493641, -728493640, -728493639, -728493638, -728493637, -728493636,
                          -728493635, -728493634, -728493633, -728493632, -728493631, -728493630, -726658648,
                          -726658647, -726658646, -726658645, -726658644, -726658643, -726658642, -726658641,
                          -726658640, -726658639, -726658638, -726658637, -726658636, -726658635, -726658634,
                          -726658633, -726658632, -726658631, -726658630, -726658629, -726658628, -726658627,
                          -726658626, -726658625, -726658624, -726658623, -726658622, -724823640, -724823639,
                          -724823638, -724823637, -724823636, -724823635, -724823634, -724823633, -724823632,
                          -724823631, -724823630, -724823629, -724823628, -724823627, -724823626, -724823625,
                          -724823624, -724823623, -724823622, -724823621, -724823620, -724823619, -724823618,
                          -724823617, -724823616, -724823615, -724823614, -722988632, -722988631, -722988630,
                          -722988629, -722988628, -722988627, -722988626, -722988625, -722988624, -722988623,
                          -722988622, -722988621, -722988620, -722988619, -722988618, -722988617, -722988616,
                          -722988615, -722988614, -722988613, -722988612, -722988611, -722988610, -722988609,
                          -722988608, -722988607, -722988606, -721153624, -721153623, -721153622, -721153621,
                          -721153620, -721153619, -721153618, -721153617, -721153616, -721153615, -721153614,
                          -721153613, -721153612, -721153611, -721153610, -721153609, -721153608, -721153607,
                          -721153606, -721153605, -721153604, -721153603, -721153602, -721153601, -721153600,
                          -721153599, -721153598, -719318616, -719318615, -719318614, -719318613, -719318612,
                          -719318611, -719318610, -719318609, -719318608, -719318607, -719318606, -719318605,
                          -719318604, -719318603, -719318602, -719318601, -719318600, -719318599, -719318598,
                          -719318597, -719318596, -719318595, -719318594, -719318593, -719318592, -719318591,
                          -719318590, -717483608, -717483607, -717483606, -717483605, -717483604, -717483603,
                          -717483602, -717483601, -717483600, -717483599, -717483598, -717483597, -717483596,
                          -717483595, -717483594, -717483593, -717483592, -717483591, -717483590, -717483589,
                          -717483588, -717483587, -717483586, -717483585, -717483584, -717483583, -717483582,
                          -715648600, -715648599, -715648598, -715648597, -715648596, -715648595, -715648594,
                          -715648593, -715648592, -715648591, -715648590, -715648589, -715648588, -715648587,
                          -715648586, -715648585, -715648584, -715648583, -715648582, -715648581, -715648580,
                          -715648579, -715648578, -715648577, -715648576, -715648575, -715648574, -713813592,
                          -713813591, -713813590, -713813589, -713813588, -713813587, -713813586, -713813585,
                          -713813584, -713813583, -713813582, -713813581, -713813580, -713813579, -713813578,
                          -713813577, -713813576, -713813575, -713813574, -713813573, -713813572, -713813571,
                          -713813570, -713813569, -713813568, -713813567, -713813566, -711978584, -711978583,
                          -711978582, -711978581, -711978580, -711978579, -711978578, -711978577, -711978576,
                          -711978575, -711978574, -711978573, -711978572, -711978571, -711978570, -711978569,
                          -711978568, -711978567, -711978566, -711978565, -711978564, -711978563, -711978562,
                          -711978561, -711978560, -711978559, -711978558, -710143576, -710143575, -710143574,
                          -710143573, -710143572, -710143571, -710143570, -710143569, -710143568, -710143567,
                          -710143566, -710143565, -710143564, -710143563, -710143562, -710143561, -710143560,
                          -710143559, -710143558, -710143557, -710143556, -710143555, -710143554, -710143553,
                          -710143552, -710143551, -710143550, -708308568, -708308567, -708308566, -708308565,
                          -708308564, -708308563, -708308562, -708308561, -708308560, -708308559, -708308558,
                          -708308557, -708308556, -708308555, -708308554, -708308553, -708308552, -708308551,
                          -708308550, -708308549, -708308548, -708308547, -708308546, -708308545, -708308544,
                          -708308543, -708308542, -706473560, -706473559, -706473558, -706473557, -706473556,
                          -706473555, -706473554, -706473553, -706473552, -706473551, -706473550, -706473549,
                          -706473548, -706473547, -706473546, -706473545, -706473544, -706473543, -706473542,
                          -706473541, -706473540, -706473539, -706473538, -706473537, -706473536, -706473535,
                          -706473534, -704638552, -704638551, -704638550, -704638549, -704638548, -704638547,
                          -704638546, -704638545, -704638544, -704638543, -704638542, -704638541, -704638540,
                          -704638539, -704638538, -704638537, -704638536, -704638535, -704638534, -704638533,
                          -704638532, -704638531, -704638530, -704638529, -704638528, -704638527, -704638526,
                          -702803544, -702803543, -702803542, -702803541, -702803540, -702803539, -702803538,
                          -702803537, -702803536, -702803535, -702803534, -702803533, -702803532, -702803531,
                          -702803530, -702803529, -702803528, -702803527, -702803526, -702803525, -702803524,
                          -702803523, -702803522, -702803521, -702803520, -702803519, -702803518, -700968536,
                          -700968535, -700968534, -700968533, -700968532, -700968531, -700968530, -700968529,
                          -700968528, -700968527, -700968526, -700968525, -700968524, -700968523, -700968522,
                          -700968521, -700968520, -700968519, -700968518, -700968517, -700968516, -700968515,
                          -700968514, -700968513, -700968512, -700968511, -700968510, -699133528, -699133527,
                          -699133526, -699133525, -699133524, -699133523, -699133522, -699133521, -699133520,
                          -699133519, -699133518, -699133517, -699133516, -699133515, -699133514, -699133513,
                          -699133512, -699133511, -699133510, -699133509, -699133508, -699133507, -699133506,
                          -699133505, -699133504, -699133503, -699133502, -697298520, -697298519, -697298518,
                          -697298517, -697298516, -697298515, -697298514, -697298513, -697298512, -697298511,
                          -697298510, -697298509, -697298508, -697298507, -697298506, -697298505, -697298504,
                          -697298503, -697298502, -697298501, -697298500, -697298499, -697298498, -697298497,
                          -697298496, -697298495, -697298494, -695463512, -695463511, -695463510, -695463509,
                          -695463508, -695463507, -695463506, -695463505, -695463504, -695463503, -695463502,
                          -695463501, -695463500, -695463499, -695463498, -695463497, -695463496, -695463495,
                          -695463494, -695463493, -695463492, -695463491, -695463490, -695463489, -695463488,
                          -695463487, -695463486, -693628504, -693628503, -693628502, -693628501, -693628500,
                          -693628499, -693628498, -693628497, -693628496, -693628495, -693628494, -693628493,
                          -693628492, -693628491, -693628490, -693628489, -693628488, -693628487, -693628486,
                          -693628485, -693628484, -693628483, -693628482, -693628481, -693628480, -693628479,
                          -693628478, -691793496, -691793495, -691793494, -691793493, -691793492, -691793491,
                          -691793490, -691793489, -691793488, -691793487, -691793486, -691793485, -691793484,
                          -691793483, -691793482, -691793481, -691793480, -691793479, -691793478, -691793477,
                          -691793476, -691793475, -691793474, -691793473, -691793472, -691793471, -691793470,
                          -689958488, -689958487, -689958486, -689958485, -689958484, -689958483, -689958482,
                          -689958481, -689958480, -689958479, -689958478, -689958477, -689958476, -689958475,
                          -689958474, -689958473, -689958472, -689958471, -689958470, -689958469, -689958468,
                          -689958467, -689958466, -689958465, -689958464, -689958463, -689958462, -688123480,
                          -688123479, -688123478, -688123477, -688123476, -688123475, -688123474, -688123473,
                          -688123472, -688123471, -688123470, -688123469, -688123468, -688123467, -688123466,
                          -688123465, -688123464, -688123463, -688123462, -688123461, -688123460, -688123459,
                          -688123458, -688123457, -688123456, -688123455, -688123454, -686288472, -686288471,
                          -686288470, -686288469, -686288468, -686288467, -686288466, -686288465, -686288464,
                          -686288463, -686288462, -686288461, -686288460, -686288459, -686288458, -686288457,
                          -686288456, -686288455, -686288454, -686288453, -686288452, -686288451, -686288450,
                          -686288449, -686288448, -686288447, -686288446, -684453464, -684453463, -684453462,
                          -684453461, -684453460, -684453459, -684453458, -684453457, -684453456, -684453455,
                          -684453454, -684453453, -684453452, -684453451, -684453450, -684453449, -684453448,
                          -684453447, -684453446, -684453445, -684453444, -684453443, -684453442, -684453441,
                          -684453440, -684453439, -684453438, -682618456, -682618455, -682618454, -682618453,
                          -682618452, -682618451, -682618450, -682618449, -682618448, -682618447, -682618446,
                          -682618445, -682618444, -682618443, -682618442, -682618441, -682618440, -682618439,
                          -682618438, -682618437, -682618436, -682618435, -682618434, -682618433, -682618432,
                          -682618431, -682618430, -680783448, -680783447, -680783446, -680783445, -680783444,
                          -680783443, -680783442, -680783441, -680783440, -680783439, -680783438, -680783437,
                          -680783436, -680783435, -680783434, -680783433, -680783432, -680783431, -680783430,
                          -680783429, -680783428, -680783427, -680783426, -680783425, -680783424, -680783423,
                          -680783422, -678948440, -678948439, -678948438, -678948437, -678948436, -678948435,
                          -678948434, -678948433, -678948432, -678948431, -678948430, -678948429, -678948428,
                          -678948427, -678948426, -678948425, -678948424, -678948423, -678948422, -678948421,
                          -678948420, -678948419, -678948418, -678948417, -678948416, -678948415, -678948414,
                          285217121, 285217122, 285217123, 285217124, 285217125, 285217126, 285217127, 285217128,
                          285217129, 285217130, 285217131, 285217132, 285217133, 285217134, 285217135, 285217136,
                          285217137, 285217138, 285217139, 285217140, 285217141, 285282657, 285282658, 285282659,
                          285282660, 285282661, 285282662, 285282663, 285282664, 285282665, 285282666, 285282667,
                          285282668, 285282669, 285282670, 285282671, 285282672, 285282673, 285282674, 285282675,
                          285282676, 285282677, 285348193, 285348194, 285348195, 285348196, 285348197, 285348198,
                          285348199, 285348200, 285348201, 285348202, 285348203, 285348204, 285348205, 285348206,
                          285348207, 285348208, 285348209, 285348210, 285348211, 285348212, 285348213, 285413729,
                          285413730, 285413731, 285413732, 285413733, 285413734, 285413735, 285413736, 285413737,
                          285413738, 285413739, 285413740, 285413741, 285413742, 285413743, 285413744, 285413745,
                          285413746, 285413747, 285413748, 285413749, 285479265, 285479266, 285479267, 285479268,
                          285479269, 285479270, 285479271, 285479272, 285479273, 285479274, 285479275, 285479276,
                          285479277, 285479278, 285479279, 285479280, 285479281, 285479282, 285479283, 285479284,
                          285479285, 285544801, 285544802, 285544803, 285544804, 285544805, 285544806, 285544807,
                          285544808, 285544809, 285544810, 285544811, 285544812, 285544813, 285544814, 285544815,
                          285544816, 285544817, 285544818, 285544819, 285544820, 285544821, 285610337, 285610338,
                          285610339, 285610340, 285610341, 285610342, 285610343, 285610344, 285610345, 285610346,
                          285610347, 285610348, 285610349, 285610350, 285610351, 285610352, 285610353, 285610354,
                          285610355, 285610356, 285610357, 285675873, 285675874, 285675875, 285675876, 285675877,
                          285675878, 285675879, 285675880, 285675881, 285675882, 285675883, 285675884, 285675885,
                          285675886, 285675887, 285675888, 285675889, 285675890, 285675891, 285675892, 285675893,
                          285741409, 285741410, 285741411, 285741412, 285741413, 285741414, 285741415, 285741416,
                          285741417, 285741418, 285741419, 285741420, 285741421, 285741422, 285741423, 285741424,
                          285741425, 285741426, 285741427, 285741428, 285741429, 285806945, 285806946, 285806947,
                          285806948, 285806949, 285806950, 285806951, 285806952, 285806953, 285806954, 285806955,
                          285806956, 285806957, 285806958, 285806959, 285806960, 285806961, 285806962, 285806963,
                          285806964, 285806965, 285872481, 285872482, 285872483, 285872484, 285872485, 285872486,
                          285872487, 285872488, 285872489, 285872490, 285872491, 285872492, 285872493, 285872494,
                          285872495, 285872496, 285872497, 285872498, 285872499, 285872500, 285872501, 285938017,
                          285938018, 285938019, 285938020, 285938021, 285938022, 285938023, 285938024, 285938025,
                          285938026, 285938027, 285938028, 285938029, 285938030, 285938031, 285938032, 285938033,
                          285938034, 285938035, 285938036, 285938037, 286003553, 286003554, 286003555, 286003556,
                          286003557, 286003558, 286003559, 286003560, 286003561, 286003562, 286003563, 286003564,
                          286003565, 286003566, 286003567, 286003568, 286003569, 286003570, 286003571, 286003572,
                          286003573, 286069089, 286069090, 286069091, 286069092, 286069093, 286069094, 286069095,
                          286069096, 286069097, 286069098, 286069099, 286069100, 286069101, 286069102, 286069103,
                          286069104, 286069105, 286069106, 286069107, 286069108, 286069109, 286134625, 286134626,
                          286134627, 286134628, 286134629, 286134630, 286134631, 286134632, 286134633, 286134634,
                          286134635, 286134636, 286134637, 286134638, 286134639, 286134640, 286134641, 286134642,
                          286134643, 286134644, 286134645, 286200161, 286200162, 286200163, 286200164, 286200165,
                          286200166, 286200167, 286200168, 286200169, 286200170, 286200171, 286200172, 286200173,
                          286200174, 286200175, 286200176, 286200177, 286200178, 286200179, 286200180, 286200181,
                          286265697, 286265698, 286265699, 286265700, 286265701, 286265702, 286265703, 286265704,
                          286265705, 286265706, 286265707, 286265708, 286265709, 286265710, 286265711, 286265712,
                          286265713, 286265714, 286265715, 286265716, 286265717, 286331233, 286331234, 286331235,
                          286331236, 286331237, 286331238, 286331239, 286331240, 286331241, 286331242, 286331243,
                          286331244, 286331245, 286331246, 286331247, 286331248, 286331249, 286331250, 286331251,
                          286331252, 286331253, 286396769, 286396770, 286396771, 286396772, 286396773, 286396774,
                          286396775, 286396776, 286396777, 286396778, 286396779, 286396780, 286396781, 286396782,
                          286396783, 286396784, 286396785, 286396786, 286396787, 286396788, 286396789};
    }

    private static int[] getHangulCodepoints2() {
        return new int[] {49825, 49826, 49827, 49829, 49830, 49831, 49832, 49833, 49834, 49835, 49836, 49837, 49838,
                          49839, 49840, 49841, 49842, 49843, 49844, 49845, 49846, 49847, 49848, 49849, 49850, 49851,
                          49852, 49853, 49854, 49855, 49857, 49858, 49859, 49860, 49861, 49862, 49863, 49864, 49865,
                          49866, 49867, 49868, 49869, 49870, 49871, 49872, 49873, 49874, 49875, 49876, 49877, 49878,
                          49879, 49880, 49881, 49882, 49883, 49885, 49886, 49887, 49888, 49889, 49890, 49891, 49892,
                          49893, 49894, 49895, 49896, 49897, 49898, 49899, 49900, 49901, 49902, 49903, 49904, 49905,
                          49906, 49907, 49908, 49909, 49910, 49911, 49913, 49914, 49915, 49916, 49917, 49918, 49919,
                          49920, 49921, 49922, 49923, 49924, 49925, 49926, 49927, 49928, 49929, 49930, 49931, 49932,
                          49933, 49934, 49935, 49936, 49937, 49938, 49939, 49941, 49942, 49943, 49944, 49945, 49946,
                          49947, 49948, 49949, 49950, 49951, 49952, 49953, 49954, 49955, 49956, 49957, 49958, 49959,
                          49960, 49961, 49962, 49963, 49964, 49965, 49966, 49967, 49969, 49970, 49971, 49972, 49973,
                          49974, 49975, 49976, 49977, 49978, 49979, 49980, 49981, 49982, 49983, 49984, 49985, 49986,
                          49987, 49988, 49989, 49990, 49991, 49992, 49993, 49994, 49995, 49997, 49998, 49999, 50000,
                          50001, 50002, 50003, 50004, 50005, 50006, 50007, 50008, 50009, 50010, 50011, 50012, 50013,
                          50014, 50015, 50016, 50017, 50018, 50019, 50020, 50021, 50022, 50023, 50025, 50026, 50027,
                          50028, 50029, 50030, 50031, 50032, 50033, 50034, 50035, 50036, 50037, 50038, 50039, 50040,
                          50041, 50042, 50043, 50044, 50045, 50046, 50047, 50048, 50049, 50050, 50051, 50053, 50054,
                          50055, 50056, 50057, 50058, 50059, 50060, 50061, 50062, 50063, 50064, 50065, 50066, 50067,
                          50068, 50069, 50070, 50071, 50072, 50073, 50074, 50075, 50076, 50077, 50078, 50079, 50081,
                          50082, 50083, 50084, 50085, 50086, 50087, 50088, 50089, 50090, 50091, 50092, 50093, 50094,
                          50095, 50096, 50097, 50098, 50099, 50100, 50101, 50102, 50103, 50104, 50105, 50106, 50107,
                          50109, 50110, 50111, 50112, 50113, 50114, 50115, 50116, 50117, 50118, 50119, 50120, 50121,
                          50122, 50123, 50124, 50125, 50126, 50127, 50128, 50129, 50130, 50131, 50132, 50133, 50134,
                          50135, 50137, 50138, 50139, 50140, 50141, 50142, 50143, 50144, 50145, 50146, 50147, 50148,
                          50149, 50150, 50151, 50152, 50153, 50154, 50155, 50156, 50157, 50158, 50159, 50160, 50161,
                          50162, 50163, 50165, 50166, 50167, 50168, 50169, 50170, 50171, 50172, 50173, 50174, 50175,
                          50176, 50177, 50178, 50179, 50180, 50181, 50182, 50183, 50184, 50185, 50186, 50187, 50188,
                          50189, 50190, 50191, 50193, 50194, 50195, 50196, 50197, 50198, 50199, 50200, 50201, 50202,
                          50203, 50204, 50205, 50206, 50207, 50208, 50209, 50210, 50211, 50212, 50213, 50214, 50215,
                          50216, 50217, 50218, 50219, 50221, 50222, 50223, 50224, 50225, 50226, 50227, 50228, 50229,
                          50230, 50231, 50232, 50233, 50234, 50235, 50236, 50237, 50238, 50239, 50240, 50241, 50242,
                          50243, 50244, 50245, 50246, 50247, 50249, 50250, 50251, 50252, 50253, 50254, 50255, 50256,
                          50257, 50258, 50259, 50260, 50261, 50262, 50263, 50264, 50265, 50266, 50267, 50268, 50269,
                          50270, 50271, 50272, 50273, 50274, 50275, 50277, 50278, 50279, 50280, 50281, 50282, 50283,
                          50284, 50285, 50286, 50287, 50288, 50289, 50290, 50291, 50292, 50293, 50294, 50295, 50296,
                          50297, 50298, 50299, 50300, 50301, 50302, 50303, 50305, 50306, 50307, 50308, 50309, 50310,
                          50311, 50312, 50313, 50314, 50315, 50316, 50317, 50318, 50319, 50320, 50321, 50322, 50323,
                          50324, 50325, 50326, 50327, 50328, 50329, 50330, 50331, 50333, 50334, 50335, 50336, 50337,
                          50338, 50339, 50340, 50341, 50342, 50343, 50344, 50345, 50346, 50347, 50348, 50349, 50350,
                          50351, 50352, 50353, 50354, 50355, 50356, 50357, 50358, 50359, 50361, 50362, 50363, 50364,
                          50365, 50366, 50367, 50368, 50369, 50370, 50371, 50372, 50373, 50374, 50375, 50376, 50377,
                          50378, 50379, 50380, 50381, 50382, 50383, 50384, 50385, 50386, 50387, 50389, 50390, 50391,
                          50392, 50393, 50394, 50395, 50396, 50397, 50398, 50399, 50400, 50401, 50402, 50403, 50404,
                          50405, 50406, 50407, 50408, 50409, 50410, 50411, 50412, 50413, 50414, 50415, 50417, 50418,
                          50419, 50420, 50421, 50422, 50423, 50424, 50425, 50426, 50427, 50428, 50429, 50430, 50431,
                          50432, 50433, 50434, 50435, 50436, 50437, 50438, 50439, 50440, 50441, 50442, 50443, 50445,
                          50446, 50447, 50448, 50449, 50450, 50451, 50452, 50453, 50454, 50455, 50456, 50457, 50458,
                          50459, 50460, 50461, 50462, 50463, 50464, 50465, 50466, 50467, 50468, 50469, 50470, 50471,
                          50473, 50474, 50475, 50476, 50477, 50478, 50479, 50480, 50481, 50482, 50483, 50484, 50485,
                          50486, 50487, 50488, 50489, 50490, 50491, 50492, 50493, 50494, 50495, 50496, 50497, 50498,
                          50499, 50501, 50502, 50503, 50504, 50505, 50506, 50507, 50508, 50509, 50510, 50511, 50512,
                          50513, 50514, 50515, 50516, 50517, 50518, 50519, 50520, 50521, 50522, 50523, 50524, 50525,
                          50526, 50527, 50529, 50530, 50531, 50532, 50533, 50534, 50535, 50536, 50537, 50538, 50539,
                          50540, 50541, 50542, 50543, 50544, 50545, 50546, 50547, 50548, 50549, 50550, 50551, 50552,
                          50553, 50554, 50555, 50557, 50558, 50559, 50560, 50561, 50562, 50563, 50564, 50565, 50566,
                          50567, 50568, 50569, 50570, 50571, 50572, 50573, 50574, 50575, 50576, 50577, 50578, 50579,
                          50580, 50581, 50582, 50583, 50585, 50586, 50587, 50588, 50589, 50590, 50591, 50592, 50593,
                          50594, 50595, 50596, 50597, 50598, 50599, 50600, 50601, 50602, 50603, 50604, 50605, 50606,
                          50607, 50608, 50609, 50610, 50611, 50613, 50614, 50615, 50616, 50617, 50618, 50619, 50620,
                          50621, 50622, 50623, 50624, 50625, 50626, 50627, 50628, 50629, 50630, 50631, 50632, 50633,
                          50634, 50635, 50636, 50637, 50638, 50639, 50641, 50642, 50643, 50644, 50645, 50646, 50647,
                          50648, 50649, 50650, 50651, 50652, 50653, 50654, 50655, 50656, 50657, 50658, 50659, 50660,
                          50661, 50662, 50663, 50664, 50665, 50666, 50667, 50669, 50670, 50671, 50672, 50673, 50674,
                          50675, 50676, 50677, 50678, 50679, 50680, 50681, 50682, 50683, 50684, 50685, 50686, 50687,
                          50688, 50689, 50690, 50691, 50692, 50693, 50694, 50695, 50697, 50698, 50699, 50700, 50701,
                          50702, 50703, 50704, 50705, 50706, 50707, 50708, 50709, 50710, 50711, 50712, 50713, 50714,
                          50715, 50716, 50717, 50718, 50719, 50720, 50721, 50722, 50723, 50725, 50726, 50727, 50728,
                          50729, 50730, 50731, 50732, 50733, 50734, 50735, 50736, 50737, 50738, 50739, 50740, 50741,
                          50742, 50743, 50744, 50745, 50746, 50747, 50748, 50749, 50750, 50751, 50753, 50754, 50755,
                          50756, 50757, 50758, 50759, 50760, 50761, 50762, 50763, 50764, 50765, 50766, 50767, 50768,
                          50769, 50770, 50771, 50772, 50773, 50774, 50775, 50776, 50777, 50778, 50779, 50781, 50782,
                          50783, 50784, 50785, 50786, 50787, 50788, 50789, 50790, 50791, 50792, 50793, 50794, 50795,
                          50796, 50797, 50798, 50799, 50800, 50801, 50802, 50803, 50804, 50805, 50806, 50807, 50809,
                          50810, 50811, 50812, 50813, 50814, 50815, 50816, 50817, 50818, 50819, 50820, 50821, 50822,
                          50823, 50824, 50825, 50826, 50827, 50828, 50829, 50830, 50831, 50832, 50833, 50834, 50835,
                          50837, 50838, 50839, 50840, 50841, 50842, 50843, 50844, 50845, 50846, 50847, 50848, 50849,
                          50850, 50851, 50852, 50853, 50854, 50855, 50856, 50857, 50858, 50859, 50860, 50861, 50862,
                          50863, 50865, 50866, 50867, 50868, 50869, 50870, 50871, 50872, 50873, 50874, 50875, 50876,
                          50877, 50878, 50879, 50880, 50881, 50882, 50883, 50884, 50885, 50886, 50887, 50888, 50889,
                          50890, 50891, 50893, 50894, 50895, 50896, 50897, 50898, 50899, 50900, 50901, 50902, 50903,
                          50904, 50905, 50906, 50907, 50908, 50909, 50910, 50911, 50912, 50913, 50914, 50915, 50916,
                          50917, 50918, 50919, 50921, 50922, 50923, 50924, 50925, 50926, 50927, 50928, 50929, 50930,
                          50931, 50932, 50933, 50934, 50935, 50936, 50937, 50938, 50939, 50940, 50941, 50942, 50943,
                          50944, 50945, 50946, 50947, 50949, 50950, 50951, 50952, 50953, 50954, 50955, 50956, 50957,
                          50958, 50959, 50960, 50961, 50962, 50963, 50964, 50965, 50966, 50967, 50968, 50969, 50970,
                          50971, 50972, 50973, 50974, 50975, 50977, 50978, 50979, 50980, 50981, 50982, 50983, 50984,
                          50985, 50986, 50987, 50988, 50989, 50990, 50991, 50992, 50993, 50994, 50995, 50996, 50997,
                          50998, 50999, 51000, 51001, 51002, 51003, 51005, 51006, 51007, 51008, 51009, 51010, 51011,
                          51012, 51013, 51014, 51015, 51016, 51017, 51018, 51019, 51020, 51021, 51022, 51023, 51024,
                          51025, 51026, 51027, 51028, 51029, 51030, 51031, 51033, 51034, 51035, 51036, 51037, 51038,
                          51039, 51040, 51041, 51042, 51043, 51044, 51045, 51046, 51047, 51048, 51049, 51050, 51051,
                          51052, 51053, 51054, 51055, 51056, 51057, 51058, 51059, 51061, 51062, 51063, 51064, 51065,
                          51066, 51067, 51068, 51069, 51070, 51071, 51072, 51073, 51074, 51075, 51076, 51077, 51078,
                          51079, 51080, 51081, 51082, 51083, 51084, 51085, 51086, 51087, 51089, 51090, 51091, 51092,
                          51093, 51094, 51095, 51096, 51097, 51098, 51099, 51100, 51101, 51102, 51103, 51104, 51105,
                          51106, 51107, 51108, 51109, 51110, 51111, 51112, 51113, 51114, 51115, 51117, 51118, 51119,
                          51120, 51121, 51122, 51123, 51124, 51125, 51126, 51127, 51128, 51129, 51130, 51131, 51132,
                          51133, 51134, 51135, 51136, 51137, 51138, 51139, 51140, 51141, 51142, 51143, 51145, 51146,
                          51147, 51148, 51149, 51150, 51151, 51152, 51153, 51154, 51155, 51156, 51157, 51158, 51159,
                          51160, 51161, 51162, 51163, 51164, 51165, 51166, 51167, 51168, 51169, 51170, 51171, 51173,
                          51174, 51175, 51176, 51177, 51178, 51179, 51180, 51181, 51182, 51183, 51184, 51185, 51186,
                          51187, 51188, 51189, 51190, 51191, 51192, 51193, 51194, 51195, 51196, 51197, 51198, 51199,
                          51201, 51202, 51203, 51204, 51205, 51206, 51207, 51208, 51209, 51210, 51211, 51212, 51213,
                          51214, 51215, 51216, 51217, 51218, 51219, 51220, 51221, 51222, 51223, 51224, 51225, 51226,
                          51227, 51229, 51230, 51231, 51232, 51233, 51234, 51235, 51236, 51237, 51238, 51239, 51240,
                          51241, 51242, 51243, 51244, 51245, 51246, 51247, 51248, 51249, 51250, 51251, 51252, 51253,
                          51254, 51255, 51257, 51258, 51259, 51260, 51261, 51262, 51263, 51264, 51265, 51266, 51267,
                          51268, 51269, 51270, 51271, 51272, 51273, 51274, 51275, 51276, 51277, 51278, 51279, 51280,
                          51281, 51282, 51283, 51285, 51286, 51287, 51288, 51289, 51290, 51291, 51292, 51293, 51294,
                          51295, 51296, 51297, 51298, 51299, 51300, 51301, 51302, 51303, 51304, 51305, 51306, 51307,
                          51308, 51309, 51310, 51311, 51313, 51314, 51315, 51316, 51317, 51318, 51319, 51320, 51321,
                          51322, 51323, 51324, 51325, 51326, 51327, 51328, 51329, 51330, 51331, 51332, 51333, 51334,
                          51335, 51336, 51337, 51338, 51339, 51341, 51342, 51343, 51344, 51345, 51346, 51347, 51348,
                          51349, 51350, 51351, 51352, 51353, 51354, 51355, 51356, 51357, 51358, 51359, 51360, 51361,
                          51362, 51363, 51364, 51365, 51366, 51367, 51369, 51370, 51371, 51372, 51373, 51374, 51375,
                          51376, 51377, 51378, 51379, 51380, 51381, 51382, 51383, 51384, 51385, 51386, 51387, 51388,
                          51389, 51390, 51391, 51392, 51393, 51394, 51395, 51397, 51398, 51399, 51400, 51401, 51402,
                          51403, 51404, 51405, 51406, 51407, 51408, 51409, 51410, 51411, 51412, 51413, 51414, 51415,
                          51416, 51417, 51418, 51419, 51420, 51421, 51422, 51423, 51425, 51426, 51427, 51428, 51429,
                          51430, 51431, 51432, 51433, 51434, 51435, 51436, 51437, 51438, 51439, 51440, 51441, 51442,
                          51443, 51444, 51445, 51446, 51447, 51448, 51449, 51450, 51451, 51453, 51454, 51455, 51456,
                          51457, 51458, 51459, 51460, 51461, 51462, 51463, 51464, 51465, 51466, 51467, 51468, 51469,
                          51470, 51471, 51472, 51473, 51474, 51475, 51476, 51477, 51478, 51479, 51481, 51482, 51483,
                          51484, 51485, 51486, 51487, 51488, 51489, 51490, 51491, 51492, 51493, 51494, 51495, 51496,
                          51497, 51498, 51499, 51500, 51501, 51502, 51503, 51504, 51505, 51506, 51507, 51509, 51510,
                          51511, 51512, 51513, 51514, 51515, 51516, 51517, 51518, 51519, 51520, 51521, 51522, 51523,
                          51524, 51525, 51526, 51527, 51528, 51529, 51530, 51531, 51532, 51533, 51534, 51535, 51537,
                          51538, 51539, 51540, 51541, 51542, 51543, 51544, 51545, 51546, 51547, 51548, 51549, 51550,
                          51551, 51552, 51553, 51554, 51555, 51556, 51557, 51558, 51559, 51560, 51561, 51562, 51563,
                          51565, 51566, 51567, 51568, 51569, 51570, 51571, 51572, 51573, 51574, 51575, 51576, 51577,
                          51578, 51579, 51580, 51581, 51582, 51583, 51584, 51585, 51586, 51587, 51588, 51589, 51590,
                          51591, 51593, 51594, 51595, 51596, 51597, 51598, 51599, 51600, 51601, 51602, 51603, 51604,
                          51605, 51606, 51607, 51608, 51609, 51610, 51611, 51612, 51613, 51614, 51615, 51616, 51617,
                          51618, 51619, 51621, 51622, 51623, 51624, 51625, 51626, 51627, 51628, 51629, 51630, 51631,
                          51632, 51633, 51634, 51635, 51636, 51637, 51638, 51639, 51640, 51641, 51642, 51643, 51644,
                          51645, 51646, 51647, 51649, 51650, 51651, 51652, 51653, 51654, 51655, 51656, 51657, 51658,
                          51659, 51660, 51661, 51662, 51663, 51664, 51665, 51666, 51667, 51668, 51669, 51670, 51671,
                          51672, 51673, 51674, 51675, 51677, 51678, 51679, 51680, 51681, 51682, 51683, 51684, 51685,
                          51686, 51687, 51688, 51689, 51690, 51691, 51692, 51693, 51694, 51695, 51696, 51697, 51698,
                          51699, 51700, 51701, 51702, 51703, 51705, 51706, 51707, 51708, 51709, 51710, 51711, 51712,
                          51713, 51714, 51715, 51716, 51717, 51718, 51719, 51720, 51721, 51722, 51723, 51724, 51725,
                          51726, 51727, 51728, 51729, 51730, 51731, 51733, 51734, 51735, 51736, 51737, 51738, 51739,
                          51740, 51741, 51742, 51743, 51744, 51745, 51746, 51747, 51748, 51749, 51750, 51751, 51752,
                          51753, 51754, 51755, 51756, 51757, 51758, 51759, 51761, 51762, 51763, 51764, 51765, 51766,
                          51767, 51768, 51769, 51770, 51771, 51772, 51773, 51774, 51775, 51776, 51777, 51778, 51779,
                          51780, 51781, 51782, 51783, 51784, 51785, 51786, 51787, 51789, 51790, 51791, 51792, 51793,
                          51794, 51795, 51796, 51797, 51798, 51799, 51800, 51801, 51802, 51803, 51804, 51805, 51806,
                          51807, 51808, 51809, 51810, 51811, 51812, 51813, 51814, 51815, 51817, 51818, 51819, 51820,
                          51821, 51822, 51823, 51824, 51825, 51826, 51827, 51828, 51829, 51830, 51831, 51832, 51833,
                          51834, 51835, 51836, 51837, 51838, 51839, 51840, 51841, 51842, 51843, 51845, 51846, 51847,
                          51848, 51849, 51850, 51851, 51852, 51853, 51854, 51855, 51856, 51857, 51858, 51859, 51860,
                          51861, 51862, 51863, 51864, 51865, 51866, 51867, 51868, 51869, 51870, 51871, 51873, 51874,
                          51875, 51876, 51877, 51878, 51879, 51880, 51881, 51882, 51883, 51884, 51885, 51886, 51887,
                          51888, 51889, 51890, 51891, 51892, 51893, 51894, 51895, 51896, 51897, 51898, 51899, 51901,
                          51902, 51903, 51904, 51905, 51906, 51907, 51908, 51909, 51910, 51911, 51912, 51913, 51914,
                          51915, 51916, 51917, 51918, 51919, 51920, 51921, 51922, 51923, 51924, 51925, 51926, 51927,
                          51929, 51930, 51931, 51932, 51933, 51934, 51935, 51936, 51937, 51938, 51939, 51940, 51941,
                          51942, 51943, 51944, 51945, 51946, 51947, 51948, 51949, 51950, 51951, 51952, 51953, 51954,
                          51955, 51957, 51958, 51959, 51960, 51961, 51962, 51963, 51964, 51965, 51966, 51967, 51968,
                          51969, 51970, 51971, 51972, 51973, 51974, 51975, 51976, 51977, 51978, 51979, 51980, 51981,
                          51982, 51983, 51985, 51986, 51987, 51988, 51989, 51990, 51991, 51992, 51993, 51994, 51995,
                          51996, 51997, 51998, 51999, 52000, 52001, 52002, 52003, 52004, 52005, 52006, 52007, 52008,
                          52009, 52010, 52011, 52013, 52014, 52015, 52016, 52017, 52018, 52019, 52020, 52021, 52022,
                          52023, 52024, 52025, 52026, 52027, 52028, 52029, 52030, 52031, 52032, 52033, 52034, 52035,
                          52036, 52037, 52038, 52039, 52041, 52042, 52043, 52044, 52045, 52046, 52047, 52048, 52049,
                          52050, 52051, 52052, 52053, 52054, 52055, 52056, 52057, 52058, 52059, 52060, 52061, 52062,
                          52063, 52064, 52065, 52066, 52067, 52069, 52070, 52071, 52072, 52073, 52074, 52075, 52076,
                          52077, 52078, 52079, 52080, 52081, 52082, 52083, 52084, 52085, 52086, 52087, 52088, 52089,
                          52090, 52091, 52092, 52093, 52094, 52095, 52097, 52098, 52099, 52100, 52101, 52102, 52103,
                          52104, 52105, 52106, 52107, 52108, 52109, 52110, 52111, 52112, 52113, 52114, 52115, 52116,
                          52117, 52118, 52119, 52120, 52121, 52122, 52123, 52125, 52126, 52127, 52128, 52129, 52130,
                          52131, 52132, 52133, 52134, 52135, 52136, 52137, 52138, 52139, 52140, 52141, 52142, 52143,
                          52144, 52145, 52146, 52147, 52148, 52149, 52150, 52151, 52153, 52154, 52155, 52156, 52157,
                          52158, 52159, 52160, 52161, 52162, 52163, 52164, 52165, 52166, 52167, 52168, 52169, 52170,
                          52171, 52172, 52173, 52174, 52175, 52176, 52177, 52178, 52179, 52181, 52182, 52183, 52184,
                          52185, 52186, 52187, 52188, 52189, 52190, 52191, 52192, 52193, 52194, 52195, 52196, 52197,
                          52198, 52199, 52200, 52201, 52202, 52203, 52204, 52205, 52206, 52207, 52209, 52210, 52211,
                          52212, 52213, 52214, 52215, 52216, 52217, 52218, 52219, 52220, 52221, 52222, 52223, 52224,
                          52225, 52226, 52227, 52228, 52229, 52230, 52231, 52232, 52233, 52234, 52235, 52237, 52238,
                          52239, 52240, 52241, 52242, 52243, 52244, 52245, 52246, 52247, 52248, 52249, 52250, 52251,
                          52252, 52253, 52254, 52255, 52256, 52257, 52258, 52259, 52260, 52261, 52262, 52263, 52265,
                          52266, 52267, 52268, 52269, 52270, 52271, 52272, 52273, 52274, 52275, 52276, 52277, 52278,
                          52279, 52280, 52281, 52282, 52283, 52284, 52285, 52286, 52287, 52288, 52289, 52290, 52291,
                          52293, 52294, 52295, 52296, 52297, 52298, 52299, 52300, 52301, 52302, 52303, 52304, 52305,
                          52306, 52307, 52308, 52309, 52310, 52311, 52312, 52313, 52314, 52315, 52316, 52317, 52318,
                          52319, 52321, 52322, 52323, 52324, 52325, 52326, 52327, 52328, 52329, 52330, 52331, 52332,
                          52333, 52334, 52335, 52336, 52337, 52338, 52339, 52340, 52341, 52342, 52343, 52344, 52345,
                          52346, 52347, 52349, 52350, 52351, 52352, 52353, 52354, 52355, 52356, 52357, 52358, 52359,
                          52360, 52361, 52362, 52363, 52364, 52365, 52366, 52367, 52368, 52369, 52370, 52371, 52372,
                          52373, 52374, 52375, 52377, 52378, 52379, 52380, 52381, 52382, 52383, 52384, 52385, 52386,
                          52387, 52388, 52389, 52390, 52391, 52392, 52393, 52394, 52395, 52396, 52397, 52398, 52399,
                          52400, 52401, 52402, 52403, 52405, 52406, 52407, 52408, 52409, 52410, 52411, 52412, 52413,
                          52414, 52415, 52416, 52417, 52418, 52419, 52420, 52421, 52422, 52423, 52424, 52425, 52426,
                          52427, 52428, 52429, 52430, 52431, 52433, 52434, 52435, 52436, 52437, 52438, 52439, 52440,
                          52441, 52442, 52443, 52444, 52445, 52446, 52447, 52448, 52449, 52450, 52451, 52452, 52453,
                          52454, 52455, 52456, 52457, 52458, 52459, 52461, 52462, 52463, 52464, 52465, 52466, 52467,
                          52468, 52469, 52470, 52471, 52472, 52473, 52474, 52475, 52476, 52477, 52478, 52479, 52480,
                          52481, 52482, 52483, 52484, 52485, 52486, 52487, 52489, 52490, 52491, 52492, 52493, 52494,
                          52495, 52496, 52497, 52498, 52499, 52500, 52501, 52502, 52503, 52504, 52505, 52506, 52507,
                          52508, 52509, 52510, 52511, 52512, 52513, 52514, 52515, 52517, 52518, 52519, 52520, 52521,
                          52522, 52523, 52524, 52525, 52526, 52527, 52528, 52529, 52530, 52531, 52532, 52533, 52534,
                          52535, 52536, 52537, 52538, 52539, 52540, 52541, 52542, 52543, 52545, 52546, 52547, 52548,
                          52549, 52550, 52551, 52552, 52553, 52554, 52555, 52556, 52557, 52558, 52559, 52560, 52561,
                          52562, 52563, 52564, 52565, 52566, 52567, 52568, 52569, 52570, 52571, 52573, 52574, 52575,
                          52576, 52577, 52578, 52579, 52580, 52581, 52582, 52583, 52584, 52585, 52586, 52587, 52588,
                          52589, 52590, 52591, 52592, 52593, 52594, 52595, 52596, 52597, 52598, 52599, 52601, 52602,
                          52603, 52604, 52605, 52606, 52607, 52608, 52609, 52610, 52611, 52612, 52613, 52614, 52615,
                          52616, 52617, 52618, 52619, 52620, 52621, 52622, 52623, 52624, 52625, 52626, 52627, 52629,
                          52630, 52631, 52632, 52633, 52634, 52635, 52636, 52637, 52638, 52639, 52640, 52641, 52642,
                          52643, 52644, 52645, 52646, 52647, 52648, 52649, 52650, 52651, 52652, 52653, 52654, 52655,
                          52657, 52658, 52659, 52660, 52661, 52662, 52663, 52664, 52665, 52666, 52667, 52668, 52669,
                          52670, 52671, 52672, 52673, 52674, 52675, 52676, 52677, 52678, 52679, 52680, 52681, 52682,
                          52683, 52685, 52686, 52687, 52688, 52689, 52690, 52691, 52692, 52693, 52694, 52695, 52696,
                          52697, 52698, 52699, 52700, 52701, 52702, 52703, 52704, 52705, 52706, 52707, 52708, 52709,
                          52710, 52711, 52713, 52714, 52715, 52716, 52717, 52718, 52719, 52720, 52721, 52722, 52723,
                          52724, 52725, 52726, 52727, 52728, 52729, 52730, 52731, 52732, 52733, 52734, 52735, 52736,
                          52737, 52738, 52739, 52741, 52742, 52743, 52744, 52745, 52746, 52747, 52748, 52749, 52750,
                          52751, 52752, 52753, 52754, 52755, 52756, 52757, 52758, 52759, 52760, 52761, 52762, 52763,
                          52764, 52765, 52766, 52767, 52769, 52770, 52771, 52772, 52773, 52774, 52775, 52776, 52777,
                          52778, 52779, 52780, 52781, 52782, 52783, 52784, 52785, 52786, 52787, 52788, 52789, 52790,
                          52791, 52792, 52793, 52794, 52795, 52797, 52798, 52799, 52800, 52801, 52802, 52803, 52804,
                          52805, 52806, 52807, 52808, 52809, 52810, 52811, 52812, 52813, 52814, 52815, 52816, 52817,
                          52818, 52819, 52820, 52821, 52822, 52823, 52825, 52826, 52827, 52828, 52829, 52830, 52831,
                          52832, 52833, 52834, 52835, 52836, 52837, 52838, 52839, 52840, 52841, 52842, 52843, 52844,
                          52845, 52846, 52847, 52848, 52849, 52850, 52851, 52853, 52854, 52855, 52856, 52857, 52858,
                          52859, 52860, 52861, 52862, 52863, 52864, 52865, 52866, 52867, 52868, 52869, 52870, 52871,
                          52872, 52873, 52874, 52875, 52876, 52877, 52878, 52879, 52881, 52882, 52883, 52884, 52885,
                          52886, 52887, 52888, 52889, 52890, 52891, 52892, 52893, 52894, 52895, 52896, 52897, 52898,
                          52899, 52900, 52901, 52902, 52903, 52904, 52905, 52906, 52907, 52909, 52910, 52911, 52912,
                          52913, 52914, 52915, 52916, 52917, 52918, 52919, 52920, 52921, 52922, 52923, 52924, 52925,
                          52926, 52927, 52928, 52929, 52930, 52931, 52932, 52933, 52934, 52935, 52937, 52938, 52939,
                          52940, 52941, 52942, 52943, 52944, 52945, 52946, 52947, 52948, 52949, 52950, 52951, 52952,
                          52953, 52954, 52955, 52956, 52957, 52958, 52959, 52960, 52961, 52962, 52963, 52965, 52966,
                          52967, 52968, 52969, 52970, 52971, 52972, 52973, 52974, 52975, 52976, 52977, 52978, 52979,
                          52980, 52981, 52982, 52983, 52984, 52985, 52986, 52987, 52988, 52989, 52990, 52991, 52993,
                          52994, 52995, 52996, 52997, 52998, 52999, 53000, 53001, 53002, 53003, 53004, 53005, 53006,
                          53007, 53008, 53009, 53010, 53011, 53012, 53013, 53014, 53015, 53016, 53017, 53018, 53019,
                          53021, 53022, 53023, 53024, 53025, 53026, 53027, 53028, 53029, 53030, 53031, 53032, 53033,
                          53034, 53035, 53036, 53037, 53038, 53039, 53040, 53041, 53042, 53043, 53044, 53045, 53046,
                          53047, 53049, 53050, 53051, 53052, 53053, 53054, 53055, 53056, 53057, 53058, 53059, 53060,
                          53061, 53062, 53063, 53064, 53065, 53066, 53067, 53068, 53069, 53070, 53071, 53072, 53073,
                          53074, 53075, 53077, 53078, 53079, 53080, 53081, 53082, 53083, 53084, 53085, 53086, 53087,
                          53088, 53089, 53090, 53091, 53092, 53093, 53094, 53095, 53096, 53097, 53098, 53099, 53100,
                          53101, 53102, 53103, 53105, 53106, 53107, 53108, 53109, 53110, 53111, 53112, 53113, 53114,
                          53115, 53116, 53117, 53118, 53119, 53120, 53121, 53122, 53123, 53124, 53125, 53126, 53127,
                          53128, 53129, 53130, 53131, 53133, 53134, 53135, 53136, 53137, 53138, 53139, 53140, 53141,
                          53142, 53143, 53144, 53145, 53146, 53147, 53148, 53149, 53150, 53151, 53152, 53153, 53154,
                          53155, 53156, 53157, 53158, 53159, 53161, 53162, 53163, 53164, 53165, 53166, 53167, 53168,
                          53169, 53170, 53171, 53172, 53173, 53174, 53175, 53176, 53177, 53178, 53179, 53180, 53181,
                          53182, 53183, 53184, 53185, 53186, 53187, 53189, 53190, 53191, 53192, 53193, 53194, 53195,
                          53196, 53197, 53198, 53199, 53200, 53201, 53202, 53203, 53204, 53205, 53206, 53207, 53208,
                          53209, 53210, 53211, 53212, 53213, 53214, 53215, 53217, 53218, 53219, 53220, 53221, 53222,
                          53223, 53224, 53225, 53226, 53227, 53228, 53229, 53230, 53231, 53232, 53233, 53234, 53235,
                          53236, 53237, 53238, 53239, 53240, 53241, 53242, 53243, 53245, 53246, 53247, 53248, 53249,
                          53250, 53251, 53252, 53253, 53254, 53255, 53256, 53257, 53258, 53259, 53260, 53261, 53262,
                          53263, 53264, 53265, 53266, 53267, 53268, 53269, 53270, 53271, 53273, 53274, 53275, 53276,
                          53277, 53278, 53279, 53280, 53281, 53282, 53283, 53284, 53285, 53286, 53287, 53288, 53289,
                          53290, 53291, 53292, 53293, 53294, 53295, 53296, 53297, 53298, 53299, 53301, 53302, 53303,
                          53304, 53305, 53306, 53307, 53308, 53309, 53310, 53311, 53312, 53313, 53314, 53315, 53316,
                          53317, 53318, 53319, 53320, 53321, 53322, 53323, 53324, 53325, 53326, 53327, 53329, 53330,
                          53331, 53332, 53333, 53334, 53335, 53336, 53337, 53338, 53339, 53340, 53341, 53342, 53343,
                          53344, 53345, 53346, 53347, 53348, 53349, 53350, 53351, 53352, 53353, 53354, 53355, 53357,
                          53358, 53359, 53360, 53361, 53362, 53363, 53364, 53365, 53366, 53367, 53368, 53369, 53370,
                          53371, 53372, 53373, 53374, 53375, 53376, 53377, 53378, 53379, 53380, 53381, 53382, 53383,
                          53385, 53386, 53387, 53388, 53389, 53390, 53391, 53392, 53393, 53394, 53395, 53396, 53397,
                          53398, 53399, 53400, 53401, 53402, 53403, 53404, 53405, 53406, 53407, 53408, 53409, 53410,
                          53411, 53413, 53414, 53415, 53416, 53417, 53418, 53419, 53420, 53421, 53422, 53423, 53424,
                          53425, 53426, 53427, 53428, 53429, 53430, 53431, 53432, 53433, 53434, 53435, 53436, 53437,
                          53438, 53439, 53441, 53442, 53443, 53444, 53445, 53446, 53447, 53448, 53449, 53450, 53451,
                          53452, 53453, 53454, 53455, 53456, 53457, 53458, 53459, 53460, 53461, 53462, 53463, 53464,
                          53465, 53466, 53467, 53469, 53470, 53471, 53472, 53473, 53474, 53475, 53476, 53477, 53478,
                          53479, 53480, 53481, 53482, 53483, 53484, 53485, 53486, 53487, 53488, 53489, 53490, 53491,
                          53492, 53493, 53494, 53495, 53497, 53498, 53499, 53500, 53501, 53502, 53503, 53504, 53505,
                          53506, 53507, 53508, 53509, 53510, 53511, 53512, 53513, 53514, 53515, 53516, 53517, 53518,
                          53519, 53520, 53521, 53522, 53523, 53525, 53526, 53527, 53528, 53529, 53530, 53531, 53532,
                          53533, 53534, 53535, 53536, 53537, 53538, 53539, 53540, 53541, 53542, 53543, 53544, 53545,
                          53546, 53547, 53548, 53549, 53550, 53551, 53553, 53554, 53555, 53556, 53557, 53558, 53559,
                          53560, 53561, 53562, 53563, 53564, 53565, 53566, 53567, 53568, 53569, 53570, 53571, 53572,
                          53573, 53574, 53575, 53576, 53577, 53578, 53579, 53581, 53582, 53583, 53584, 53585, 53586,
                          53587, 53588, 53589, 53590, 53591, 53592, 53593, 53594, 53595, 53596, 53597, 53598, 53599,
                          53600, 53601, 53602, 53603, 53604, 53605, 53606, 53607, 53609, 53610, 53611, 53612, 53613,
                          53614, 53615, 53616, 53617, 53618, 53619, 53620, 53621, 53622, 53623, 53624, 53625, 53626,
                          53627, 53628, 53629, 53630, 53631, 53632, 53633, 53634, 53635, 53637, 53638, 53639, 53640,
                          53641, 53642, 53643, 53644, 53645, 53646, 53647, 53648, 53649, 53650, 53651, 53652, 53653,
                          53654, 53655, 53656, 53657, 53658, 53659, 53660, 53661, 53662, 53663, 53665, 53666, 53667,
                          53668, 53669, 53670, 53671, 53672, 53673, 53674, 53675, 53676, 53677, 53678, 53679, 53680,
                          53681, 53682, 53683, 53684, 53685, 53686, 53687, 53688, 53689, 53690, 53691, 53693, 53694,
                          53695, 53696, 53697, 53698, 53699, 53700, 53701, 53702, 53703, 53704, 53705, 53706, 53707,
                          53708, 53709, 53710, 53711, 53712, 53713, 53714, 53715, 53716, 53717, 53718, 53719, 53721,
                          53722, 53723, 53724, 53725, 53726, 53727, 53728, 53729, 53730, 53731, 53732, 53733, 53734,
                          53735, 53736, 53737, 53738, 53739, 53740, 53741, 53742, 53743, 53744, 53745, 53746, 53747,
                          53749, 53750, 53751, 53752, 53753, 53754, 53755, 53756, 53757, 53758, 53759, 53760, 53761,
                          53762, 53763, 53764, 53765, 53766, 53767, 53768, 53769, 53770, 53771, 53772, 53773, 53774,
                          53775, 53777, 53778, 53779, 53780, 53781, 53782, 53783, 53784, 53785, 53786, 53787, 53788,
                          53789, 53790, 53791, 53792, 53793, 53794, 53795, 53796, 53797, 53798, 53799, 53800, 53801,
                          53802, 53803, 53805, 53806, 53807, 53808, 53809, 53810, 53811, 53812, 53813, 53814, 53815,
                          53816, 53817, 53818, 53819, 53820, 53821, 53822, 53823, 53824, 53825, 53826, 53827, 53828,
                          53829, 53830, 53831, 53833, 53834, 53835, 53836, 53837, 53838, 53839, 53840, 53841, 53842,
                          53843, 53844, 53845, 53846, 53847, 53848, 53849, 53850, 53851, 53852, 53853, 53854, 53855,
                          53856, 53857, 53858, 53859, 53861, 53862, 53863, 53864, 53865, 53866, 53867, 53868, 53869,
                          53870, 53871, 53872, 53873, 53874, 53875, 53876, 53877, 53878, 53879, 53880, 53881, 53882,
                          53883, 53884, 53885, 53886, 53887, 53889, 53890, 53891, 53892, 53893, 53894, 53895, 53896,
                          53897, 53898, 53899, 53900, 53901, 53902, 53903, 53904, 53905, 53906, 53907, 53908, 53909,
                          53910, 53911, 53912, 53913, 53914, 53915, 53917, 53918, 53919, 53920, 53921, 53922, 53923,
                          53924, 53925, 53926, 53927, 53928, 53929, 53930, 53931, 53932, 53933, 53934, 53935, 53936,
                          53937, 53938, 53939, 53940, 53941, 53942, 53943, 53945, 53946, 53947, 53948, 53949, 53950,
                          53951, 53952, 53953, 53954, 53955, 53956, 53957, 53958, 53959, 53960, 53961, 53962, 53963,
                          53964, 53965, 53966, 53967, 53968, 53969, 53970, 53971, 53973, 53974, 53975, 53976, 53977,
                          53978, 53979, 53980, 53981, 53982, 53983, 53984, 53985, 53986, 53987, 53988, 53989, 53990,
                          53991, 53992, 53993, 53994, 53995, 53996, 53997, 53998, 53999, 54001, 54002, 54003, 54004,
                          54005, 54006, 54007, 54008, 54009, 54010, 54011, 54012, 54013, 54014, 54015, 54016, 54017,
                          54018, 54019, 54020, 54021, 54022, 54023, 54024, 54025, 54026, 54027, 54029, 54030, 54031,
                          54032, 54033, 54034, 54035, 54036, 54037, 54038, 54039, 54040, 54041, 54042, 54043, 54044,
                          54045, 54046, 54047, 54048, 54049, 54050, 54051, 54052, 54053, 54054, 54055, 54057, 54058,
                          54059, 54060, 54061, 54062, 54063, 54064, 54065, 54066, 54067, 54068, 54069, 54070, 54071,
                          54072, 54073, 54074, 54075, 54076, 54077, 54078, 54079, 54080, 54081, 54082, 54083, 54085,
                          54086, 54087, 54088, 54089, 54090, 54091, 54092, 54093, 54094, 54095, 54096, 54097, 54098,
                          54099, 54100, 54101, 54102, 54103, 54104, 54105, 54106, 54107, 54108, 54109, 54110, 54111,
                          54113, 54114, 54115, 54116, 54117, 54118, 54119, 54120, 54121, 54122, 54123, 54124, 54125,
                          54126, 54127, 54128, 54129, 54130, 54131, 54132, 54133, 54134, 54135, 54136, 54137, 54138,
                          54139, 54141, 54142, 54143, 54144, 54145, 54146, 54147, 54148, 54149, 54150, 54151, 54152,
                          54153, 54154, 54155, 54156, 54157, 54158, 54159, 54160, 54161, 54162, 54163, 54164, 54165,
                          54166, 54167, 54169, 54170, 54171, 54172, 54173, 54174, 54175, 54176, 54177, 54178, 54179,
                          54180, 54181, 54182, 54183, 54184, 54185, 54186, 54187, 54188, 54189, 54190, 54191, 54192,
                          54193, 54194, 54195, 54197, 54198, 54199, 54200, 54201, 54202, 54203, 54204, 54205, 54206,
                          54207, 54208, 54209, 54210, 54211, 54212, 54213, 54214, 54215, 54216, 54217, 54218, 54219,
                          54220, 54221, 54222, 54223, 54225, 54226, 54227, 54228, 54229, 54230, 54231, 54232, 54233,
                          54234, 54235, 54236, 54237, 54238, 54239, 54240, 54241, 54242, 54243, 54244, 54245, 54246,
                          54247, 54248, 54249, 54250, 54251, 54253, 54254, 54255, 54256, 54257, 54258, 54259, 54260,
                          54261, 54262, 54263, 54264, 54265, 54266, 54267, 54268, 54269, 54270, 54271, 54272, 54273,
                          54274, 54275, 54276, 54277, 54278, 54279, 54281, 54282, 54283, 54284, 54285, 54286, 54287,
                          54288, 54289, 54290, 54291, 54292, 54293, 54294, 54295, 54296, 54297, 54298, 54299, 54300,
                          54301, 54302, 54303, 54304, 54305, 54306, 54307, 54309, 54310, 54311, 54312, 54313, 54314,
                          54315, 54316, 54317, 54318, 54319, 54320, 54321, 54322, 54323, 54324, 54325, 54326, 54327,
                          54328, 54329, 54330, 54331, 54332, 54333, 54334, 54335, 54337, 54338, 54339, 54340, 54341,
                          54342, 54343, 54344, 54345, 54346, 54347, 54348, 54349, 54350, 54351, 54352, 54353, 54354,
                          54355, 54356, 54357, 54358, 54359, 54360, 54361, 54362, 54363, 54365, 54366, 54367, 54368,
                          54369, 54370, 54371, 54372, 54373, 54374, 54375, 54376, 54377, 54378, 54379, 54380, 54381,
                          54382, 54383, 54384, 54385, 54386, 54387, 54388, 54389, 54390, 54391, 54393, 54394, 54395,
                          54396, 54397, 54398, 54399, 54400, 54401, 54402, 54403, 54404, 54405, 54406, 54407, 54408,
                          54409, 54410, 54411, 54412, 54413, 54414, 54415, 54416, 54417, 54418, 54419, 54421, 54422,
                          54423, 54424, 54425, 54426, 54427, 54428, 54429, 54430, 54431, 54432, 54433, 54434, 54435,
                          54436, 54437, 54438, 54439, 54440, 54441, 54442, 54443, 54444, 54445, 54446, 54447, 54449,
                          54450, 54451, 54452, 54453, 54454, 54455, 54456, 54457, 54458, 54459, 54460, 54461, 54462,
                          54463, 54464, 54465, 54466, 54467, 54468, 54469, 54470, 54471, 54472, 54473, 54474, 54475,
                          54477, 54478, 54479, 54480, 54481, 54482, 54483, 54484, 54485, 54486, 54487, 54488, 54489,
                          54490, 54491, 54492, 54493, 54494, 54495, 54496, 54497, 54498, 54499, 54500, 54501, 54502,
                          54503, 54505, 54506, 54507, 54508, 54509, 54510, 54511, 54512, 54513, 54514, 54515, 54516,
                          54517, 54518, 54519, 54520, 54521, 54522, 54523, 54524, 54525, 54526, 54527, 54528, 54529,
                          54530, 54531, 54533, 54534, 54535, 54536, 54537, 54538, 54539, 54540, 54541, 54542, 54543,
                          54544, 54545, 54546, 54547, 54548, 54549, 54550, 54551, 54552, 54553, 54554, 54555, 54556,
                          54557, 54558, 54559, 54561, 54562, 54563, 54564, 54565, 54566, 54567, 54568, 54569, 54570,
                          54571, 54572, 54573, 54574, 54575, 54576, 54577, 54578, 54579, 54580, 54581, 54582, 54583,
                          54584, 54585, 54586, 54587, 54589, 54590, 54591, 54592, 54593, 54594, 54595, 54596, 54597,
                          54598, 54599, 54600, 54601, 54602, 54603, 54604, 54605, 54606, 54607, 54608, 54609, 54610,
                          54611, 54612, 54613, 54614, 54615, 54617, 54618, 54619, 54620, 54621, 54622, 54623, 54624,
                          54625, 54626, 54627, 54628, 54629, 54630, 54631, 54632, 54633, 54634, 54635, 54636, 54637,
                          54638, 54639, 54640, 54641, 54642, 54643, 54645, 54646, 54647, 54648, 54649, 54650, 54651,
                          54652, 54653, 54654, 54655, 54656, 54657, 54658, 54659, 54660, 54661, 54662, 54663, 54664,
                          54665, 54666, 54667, 54668, 54669, 54670, 54671, 54673, 54674, 54675, 54676, 54677, 54678,
                          54679, 54680, 54681, 54682, 54683, 54684, 54685, 54686, 54687, 54688, 54689, 54690, 54691,
                          54692, 54693, 54694, 54695, 54696, 54697, 54698, 54699, 54701, 54702, 54703, 54704, 54705,
                          54706, 54707, 54708, 54709, 54710, 54711, 54712, 54713, 54714, 54715, 54716, 54717, 54718,
                          54719, 54720, 54721, 54722, 54723, 54724, 54725, 54726, 54727, 54729, 54730, 54731, 54732,
                          54733, 54734, 54735, 54736, 54737, 54738, 54739, 54740, 54741, 54742, 54743, 54744, 54745,
                          54746, 54747, 54748, 54749, 54750, 54751, 54752, 54753, 54754, 54755, 54757, 54758, 54759,
                          54760, 54761, 54762, 54763, 54764, 54765, 54766, 54767, 54768, 54769, 54770, 54771, 54772,
                          54773, 54774, 54775, 54776, 54777, 54778, 54779, 54780, 54781, 54782, 54783, 54785, 54786,
                          54787, 54788, 54789, 54790, 54791, 54792, 54793, 54794, 54795, 54796, 54797, 54798, 54799,
                          54800, 54801, 54802, 54803, 54804, 54805, 54806, 54807, 54808, 54809, 54810, 54811, 54813,
                          54814, 54815, 54816, 54817, 54818, 54819, 54820, 54821, 54822, 54823, 54824, 54825, 54826,
                          54827, 54828, 54829, 54830, 54831, 54832, 54833, 54834, 54835, 54836, 54837, 54838, 54839,
                          54841, 54842, 54843, 54844, 54845, 54846, 54847, 54848, 54849, 54850, 54851, 54852, 54853,
                          54854, 54855, 54856, 54857, 54858, 54859, 54860, 54861, 54862, 54863, 54864, 54865, 54866,
                          54867, 54869, 54870, 54871, 54872, 54873, 54874, 54875, 54876, 54877, 54878, 54879, 54880,
                          54881, 54882, 54883, 54884, 54885, 54886, 54887, 54888, 54889, 54890, 54891, 54892, 54893,
                          54894, 54895, 54897, 54898, 54899, 54900, 54901, 54902, 54903, 54904, 54905, 54906, 54907,
                          54908, 54909, 54910, 54911, 54912, 54913, 54914, 54915, 54916, 54917, 54918, 54919, 54920,
                          54921, 54922, 54923, 54925, 54926, 54927, 54928, 54929, 54930, 54931, 54932, 54933, 54934,
                          54935, 54936, 54937, 54938, 54939, 54940, 54941, 54942, 54943, 54944, 54945, 54946, 54947,
                          54948, 54949, 54950, 54951, 54953, 54954, 54955, 54956, 54957, 54958, 54959, 54960, 54961,
                          54962, 54963, 54964, 54965, 54966, 54967, 54968, 54969, 54970, 54971, 54972, 54973, 54974,
                          54975, 54976, 54977, 54978, 54979, 54981, 54982, 54983, 54984, 54985, 54986, 54987, 54988,
                          54989, 54990, 54991, 54992, 54993, 54994, 54995, 54996, 54997, 54998, 54999, 55000, 55001,
                          55002, 55003, 55004, 55005, 55006, 55007, 55009, 55010, 55011, 55012, 55013, 55014, 55015,
                          55016, 55017, 55018, 55019, 55020, 55021, 55022, 55023, 55024, 55025, 55026, 55027, 55028,
                          55029, 55030, 55031, 55032, 55033, 55034, 55035, 55037, 55038, 55039, 55040, 55041, 55042,
                          55043, 55044, 55045, 55046, 55047, 55048, 55049, 55050, 55051, 55052, 55053, 55054, 55055,
                          55056, 55057, 55058, 55059, 55060, 55061, 55062, 55063, 55065, 55066, 55067, 55068, 55069,
                          55070, 55071, 55072, 55073, 55074, 55075, 55076, 55077, 55078, 55079, 55080, 55081, 55082,
                          55083, 55084, 55085, 55086, 55087, 55088, 55089, 55090, 55091, 55093, 55094, 55095, 55096,
                          55097, 55098, 55099, 55100, 55101, 55102, 55103, 55104, 55105, 55106, 55107, 55108, 55109,
                          55110, 55111, 55112, 55113, 55114, 55115, 55116, 55117, 55118, 55119, 55121, 55122, 55123,
                          55124, 55125, 55126, 55127, 55128, 55129, 55130, 55131, 55132, 55133, 55134, 55135, 55136,
                          55137, 55138, 55139, 55140, 55141, 55142, 55143, 55144, 55145, 55146, 55147, 55149, 55150,
                          55151, 55152, 55153, 55154, 55155, 55156, 55157, 55158, 55159, 55160, 55161, 55162, 55163,
                          55164, 55165, 55166, 55167, 55168, 55169, 55170, 55171, 55172, 55173, 55174, 55175, 55177,
                          55178, 55179, 55180, 55181, 55182, 55183, 55184, 55185, 55186, 55187, 55188, 55189, 55190,
                          55191, 55192, 55193, 55194, 55195, 55196, 55197, 55198, 55199, 55200, 55201, 55202, 55203,
                          44032, 44060, 44088, 44116, 44144, 44172, 44200, 44228, 44256, 44284, 44312, 44340, 44368,
                          44396, 44424, 44452, 44480, 44508, 44536, 44564, 44592, 44620, 44648, 44676, 44704, 44732,
                          44760, 44788, 44816, 44844, 44872, 44900, 44928, 44956, 44984, 45012, 45040, 45068, 45096,
                          45124, 45152, 45180, 45208, 45236, 45264, 45292, 45320, 45348, 45376, 45404, 45432, 45460,
                          45488, 45516, 45544, 45572, 45600, 45628, 45656, 45684, 45712, 45740, 45768, 45796, 45824,
                          45852, 45880, 45908, 45936, 45964, 45992, 46020, 46048, 46076, 46104, 46132, 46160, 46188,
                          46216, 46244, 46272, 46300, 46328, 46356, 46384, 46412, 46440, 46468, 46496, 46524, 46552,
                          46580, 46608, 46636, 46664, 46692, 46720, 46748, 46776, 46804, 46832, 46860, 46888, 46916,
                          46944, 46972, 47000, 47028, 47056, 47084, 47112, 47140, 47168, 47196, 47224, 47252, 47280,
                          47308, 47336, 47364, 47392, 47420, 47448, 47476, 47504, 47532, 47560, 47588, 47616, 47644,
                          47672, 47700, 47728, 47756, 47784, 47812, 47840, 47868, 47896, 47924, 47952, 47980, 48008,
                          48036, 48064, 48092, 48120, 48148, 48176, 48204, 48232, 48260, 48288, 48316, 48344, 48372,
                          48400, 48428, 48456, 48484, 48512, 48540, 48568, 48596, 48624, 48652, 48680, 48708, 48736,
                          48764, 48792, 48820, 48848, 48876, 48904, 48932, 48960, 48988, 49016, 49044, 49072, 49100,
                          49128, 49156, 49184, 49212, 49240, 49268, 49296, 49324, 49352, 49380, 49408, 49436, 49464,
                          49492, 49520, 49548, 49576, 49604, 49632, 49660, 49688, 49716, 49744, 49772, 49800, 49828,
                          49856, 49884, 49912, 49940, 49968, 49996, 50024, 50052, 50080, 50108, 50136, 50164, 50192,
                          50220, 50248, 50276, 50304, 50332, 50360, 50388, 50416, 50444, 50472, 50500, 50528, 50556,
                          50584, 50612, 50640, 50668, 50696, 50724, 50752, 50780, 50808, 50836, 50864, 50892, 50920,
                          50948, 50976, 51004, 51032, 51060, 51088, 51116, 51144, 51172, 51200, 51228, 51256, 51284,
                          51312, 51340, 51368, 51396, 51424, 51452, 51480, 51508, 51536, 51564, 51592, 51620, 51648,
                          51676, 51704, 51732, 51760, 51788, 51816, 51844, 51872, 51900, 51928, 51956, 51984, 52012,
                          52040, 52068, 52096, 52124, 52152, 52180, 52208, 52236, 52264, 52292, 52320, 52348, 52376,
                          52404, 52432, 52460, 52488, 52516, 52544, 52572, 52600, 52628, 52656, 52684, 52712, 52740,
                          52768, 52796, 52824, 52852, 52880, 52908, 52936, 52964, 52992, 53020, 53048, 53076, 53104,
                          53132, 53160, 53188, 53216, 53244, 53272, 53300, 53328, 53356, 53384, 53412, 53440, 53468,
                          53496, 53524, 53552, 53580, 53608, 53636, 53664, 53692, 53720, 53748, 53776, 53804, 53832,
                          53860, 53888, 53916, 53944, 53972, 54000, 54028, 54056, 54084, 54112, 54140, 54168, 54196,
                          54224, 54252, 54280, 54308, 54336, 54364, 54392, 54420, 54448, 54476, 54504, 54532, 54560,
                          54588, 54616, 54644, 54672, 54700, 54728, 54756, 54784, 54812, 54840, 54868, 54896, 54924,
                          54952, 54980, 55008, 55036, 55064, 55092, 55120, 55148, 55176};
    }

    private static int[] getHangulPairs() {
        int[] d1 = getHangulPairs1();
        int[] d2 = getHangulPairs2();
        int[] d = new int[d1.length + d2.length];
        System.arraycopy(d1, 0, d, 0, d1.length);
        System.arraycopy(d2, 0, d, d1.length, d2.length);
        return d;
    }

    private static int[] getHangulCodepoints() {
        int[] d1 = getHangulCodepoints1();
        int[] d2 = getHangulCodepoints2();
        int[] d = new int[d1.length + d2.length];
        System.arraycopy(d1, 0, d, 0, d1.length);
        System.arraycopy(d2, 0, d, d1.length, d2.length);
        return d;
    }

    private UnicodeCharacterDatabase() {
    }

    private static int index(int[] index, int value) {
        return Arrays.binarySearch(index, value);
    }

    private static boolean compatibility(int codepoint) {
        return CharUtils.invset_contains(compat, codepoint);
    }

    public static int getCanonicalClass(int c) {
        return CC.lookup(c);
    }

    public static boolean isComposite(int f, int s) {
        return !(f < 0 || f > 0x10FFFF || s < 0 || s > 0x10FFFF);
    }

    public static char getPairComposition(int f, int s) {
        if (!isComposite(f, s))
            return '\uFFFF';
        int chk = (f << 16) | s;
        int ndx = index(hangul_pairs, chk);
        if (ndx > -1)
            return (char)hangul_codepoints[ndx];
        ndx = index(comp_idx, chk);
        return ndx > -1 ? (char)comp_data[ndx] : '\uFFFF';
    }

    public static void decompose(int c, boolean canonical, StringBuilder buf) {
        if (c >= 44032 && c <= 55203) {
            int z = c - 0xAC00;
            int t = z % 0x001C;
            char f = (t != 0) ? (char)(0xAC00 + z - t) : (char)(0x1100 + z / 0x024C);
            char e = (t != 0) ? (char)(0x11A7 + t) : (char)(0x1161 + (z % 0x024C) / 0x001C);
            CharUtils.append(buf, f);
            CharUtils.append(buf, e);
        } else {
            int ndx = index(decomp_idx, c);
            if (ndx > -1 && !(canonical && compatibility(c))) {
                int[] points = decomp_data[ndx];
                for (int point : points)
                    decompose(point, canonical, buf);
            } else
                CharUtils.append(buf, c);
        }
    }

}
