/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.manager.state;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Scope;
import java.io.DataInput;
import java.io.IOException;
import java.util.Map;
import org.apache.accumulo.core.client.Accumulo;
import org.apache.accumulo.core.client.AccumuloClient;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.ScannerBase;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.data.InstanceId;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.fate.zookeeper.ZooReaderWriter;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.metadata.MetadataTable;
import org.apache.accumulo.core.metadata.RootTable;
import org.apache.accumulo.core.metadata.TabletLocationState;
import org.apache.accumulo.core.metadata.TabletState;
import org.apache.accumulo.core.metadata.schema.MetadataSchema;
import org.apache.accumulo.core.security.Authorizations;
import org.apache.accumulo.core.trace.TraceUtil;
import org.apache.accumulo.server.cli.ServerUtilOpts;
import org.apache.accumulo.server.manager.state.CurrentState;
import org.apache.accumulo.server.manager.state.MergeInfo;
import org.apache.accumulo.server.manager.state.MergeState;
import org.apache.accumulo.server.manager.state.MetaDataTableScanner;
import org.apache.hadoop.io.BinaryComparable;
import org.apache.hadoop.io.DataInputBuffer;
import org.apache.hadoop.io.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MergeStats {
    private static final Logger log = LoggerFactory.getLogger(MergeStats.class);
    private final MergeInfo info;
    private int hosted = 0;
    private int unassigned = 0;
    private int chopped = 0;
    private int needsToBeChopped = 0;
    private int total = 0;
    private boolean lowerSplit = false;
    private boolean upperSplit = false;

    public MergeStats(MergeInfo info) {
        this.info = info;
        if (info.getState().equals((Object)MergeState.NONE)) {
            return;
        }
        if (info.getExtent().endRow() == null) {
            this.upperSplit = true;
        }
        if (info.getExtent().prevEndRow() == null) {
            this.lowerSplit = true;
        }
    }

    public MergeInfo getMergeInfo() {
        return this.info;
    }

    public void update(KeyExtent ke, TabletState state, boolean chopped, boolean hasWALs) {
        if (this.info.getState().equals((Object)MergeState.NONE)) {
            return;
        }
        if (!this.upperSplit && this.info.getExtent().endRow().equals((Object)ke.prevEndRow())) {
            log.info("Upper split found: {}", (Object)ke.prevEndRow());
            this.upperSplit = true;
        }
        if (!this.lowerSplit && this.info.getExtent().prevEndRow().equals((Object)ke.endRow())) {
            log.info("Lower split found: {}", (Object)ke.endRow());
            this.lowerSplit = true;
        }
        if (!this.info.overlaps(ke)) {
            return;
        }
        if (this.info.needsToBeChopped(ke)) {
            ++this.needsToBeChopped;
            if (chopped && (state.equals((Object)TabletState.HOSTED) || !hasWALs)) {
                ++this.chopped;
            }
        }
        ++this.total;
        if (state.equals((Object)TabletState.HOSTED)) {
            ++this.hosted;
        }
        if (state.equals((Object)TabletState.UNASSIGNED) || state.equals((Object)TabletState.SUSPENDED)) {
            ++this.unassigned;
        }
    }

    public MergeState nextMergeState(AccumuloClient accumuloClient, CurrentState manager) throws Exception {
        MergeState state = this.info.getState();
        if (state == MergeState.NONE) {
            return state;
        }
        if (this.total == 0) {
            log.trace("failed to see any tablets for this range, ignoring {}", (Object)this.info.getExtent());
            return state;
        }
        log.info("Computing next merge state for {} which is presently {} isDelete : {}", new Object[]{this.info.getExtent(), state, this.info.isDelete()});
        if (state == MergeState.STARTED) {
            state = MergeState.SPLITTING;
        }
        if (state == MergeState.SPLITTING) {
            log.info("{} are hosted, total {}", (Object)this.hosted, (Object)this.total);
            if (!this.info.isDelete() && this.total == 1) {
                log.info("Merge range is already contained in a single tablet {}", (Object)this.info.getExtent());
                state = MergeState.COMPLETE;
            } else if (this.hosted == this.total) {
                if (this.info.isDelete()) {
                    if (!this.lowerSplit) {
                        log.info("Waiting for {} lower split to occur {}", (Object)this.info, (Object)this.info.getExtent());
                    } else if (!this.upperSplit) {
                        log.info("Waiting for {} upper split to occur {}", (Object)this.info, (Object)this.info.getExtent());
                    } else {
                        state = MergeState.WAITING_FOR_CHOPPED;
                    }
                } else {
                    state = MergeState.WAITING_FOR_CHOPPED;
                }
            } else {
                log.info("Waiting for {} hosted tablets to be {} {}", new Object[]{this.hosted, this.total, this.info.getExtent()});
            }
        }
        if (state == MergeState.WAITING_FOR_CHOPPED) {
            log.info("{} tablets are chopped {}", (Object)this.chopped, (Object)this.info.getExtent());
            if (this.chopped == this.needsToBeChopped) {
                state = MergeState.WAITING_FOR_OFFLINE;
            } else {
                log.info("Waiting for {} chopped tablets to be {} {}", new Object[]{this.chopped, this.needsToBeChopped, this.info.getExtent()});
            }
        }
        if (state == MergeState.WAITING_FOR_OFFLINE) {
            if (this.chopped == this.needsToBeChopped) {
                log.info("{} tablets are chopped, {} are offline {}", new Object[]{this.chopped, this.unassigned, this.info.getExtent()});
                if (this.unassigned == this.total) {
                    if (this.verifyMergeConsistency(accumuloClient, manager)) {
                        state = MergeState.MERGING;
                    } else {
                        log.info("Merge consistency check failed {}", (Object)this.info.getExtent());
                    }
                } else {
                    log.info("Waiting for {} unassigned tablets to be {} {}", new Object[]{this.unassigned, this.total, this.info.getExtent()});
                }
            } else {
                log.warn("Unexpected state: chopped tablets should be {} was {} merge {}", new Object[]{this.needsToBeChopped, this.chopped, this.info.getExtent()});
                state = MergeState.WAITING_FOR_CHOPPED;
            }
        }
        if (state == MergeState.MERGING) {
            if (this.hosted != 0) {
                log.error("Unexpected state: hosted tablets should be zero {} merge {}", (Object)this.hosted, (Object)this.info.getExtent());
                state = MergeState.WAITING_FOR_OFFLINE;
            }
            if (this.unassigned != this.total) {
                log.error("Unexpected state: unassigned tablets should be {} was {} merge {}", new Object[]{this.total, this.unassigned, this.info.getExtent()});
                state = MergeState.WAITING_FOR_CHOPPED;
            }
            log.info("{} tablets are unassigned {}", (Object)this.unassigned, (Object)this.info.getExtent());
        }
        return state;
    }

    private boolean verifyMergeConsistency(AccumuloClient accumuloClient, CurrentState manager) throws TableNotFoundException, IOException {
        this.verifyState(this.info, MergeState.WAITING_FOR_OFFLINE);
        MergeStats verify = new MergeStats(this.info);
        KeyExtent extent = this.info.getExtent();
        Scanner scanner = accumuloClient.createScanner(extent.isMeta() ? RootTable.NAME : MetadataTable.NAME, Authorizations.EMPTY);
        MetaDataTableScanner.configureScanner((ScannerBase)scanner, (CurrentState)manager);
        Text start = extent.prevEndRow();
        if (start == null) {
            start = new Text();
        }
        TableId tableId = extent.tableId();
        Text first = MetadataSchema.TabletsSection.encodeRow((TableId)tableId, (Text)start);
        Range range = new Range(first, false, null, true);
        scanner.setRange(range.clip(MetadataSchema.TabletsSection.getRange()));
        KeyExtent prevExtent = null;
        log.debug("Scanning range {}", (Object)range);
        for (Map.Entry entry : scanner) {
            TabletLocationState tls;
            try {
                tls = MetaDataTableScanner.createTabletLocationState((Key)((Key)entry.getKey()), (Value)((Value)entry.getValue()));
            }
            catch (TabletLocationState.BadLocationStateException e) {
                log.error("{}", (Object)e.getMessage(), (Object)e);
                return false;
            }
            log.debug("consistency check: {} walogs {}", (Object)tls, (Object)tls.walogs.size());
            if (!tls.extent.tableId().equals((Object)tableId)) break;
            if (!this.verifyWalogs(tls)) {
                log.debug("failing consistency: {} has walogs {}", (Object)tls.extent, (Object)tls.walogs.size());
                return false;
            }
            if (prevExtent == null) {
                if (tls.extent.prevEndRow() != null && tls.extent.prevEndRow().compareTo((BinaryComparable)start) > 0) {
                    log.debug("failing consistency: prev row is too high {}", (Object)start);
                    return false;
                }
                if (tls.getState(manager.onlineTabletServers()) != TabletState.UNASSIGNED && tls.getState(manager.onlineTabletServers()) != TabletState.SUSPENDED) {
                    log.debug("failing consistency: assigned or hosted {}", (Object)tls);
                    return false;
                }
            } else if (!tls.extent.isPreviousExtent(prevExtent)) {
                log.debug("hole in {}", (Object)MetadataTable.NAME);
                return false;
            }
            prevExtent = tls.extent;
            verify.update(tls.extent, tls.getState(manager.onlineTabletServers()), tls.chopped, !tls.walogs.isEmpty());
            if (tls.extent.prevEndRow() == null || extent.endRow() == null || tls.extent.prevEndRow().compareTo((BinaryComparable)extent.endRow()) <= 0) continue;
            break;
        }
        log.debug("chopped {} v.chopped {} unassigned {} v.unassigned {} verify.total {}", new Object[]{this.chopped, verify.chopped, this.unassigned, verify.unassigned, verify.total});
        return this.chopped == verify.chopped && this.unassigned == verify.unassigned && this.unassigned == verify.total;
    }

    @VisibleForTesting
    void verifyState(MergeInfo info, MergeState expectedState) {
        Preconditions.checkState((info.getState() == expectedState ? 1 : 0) != 0, (String)"Unexpected merge state %s", (Object)info.getState());
    }

    @VisibleForTesting
    boolean verifyWalogs(TabletLocationState tls) {
        return tls.walogs.isEmpty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws Exception {
        ServerUtilOpts opts = new ServerUtilOpts();
        opts.parseArgs(MergeStats.class.getName(), args, new Object[0]);
        Span span = TraceUtil.startSpan(MergeStats.class, (String)"main");
        try (Scope scope = span.makeCurrent();
             AccumuloClient client = (AccumuloClient)Accumulo.newClient().from(opts.getClientProps()).build();){
            Map tableIdMap = client.tableOperations().tableIdMap();
            ZooReaderWriter zooReaderWriter = opts.getServerContext().getZooReaderWriter();
            for (Map.Entry entry : tableIdMap.entrySet()) {
                String table = (String)entry.getKey();
                String tableId = (String)entry.getValue();
                String path = ZooUtil.getRoot((InstanceId)client.instanceOperations().getInstanceId()) + "/tables/" + tableId + "/merge";
                MergeInfo info = new MergeInfo();
                if (zooReaderWriter.exists(path)) {
                    byte[] data = zooReaderWriter.getData(path);
                    DataInputBuffer in = new DataInputBuffer();
                    in.reset(data, data.length);
                    info.readFields((DataInput)in);
                }
                System.out.printf("%25s  %10s %10s %s%n", table, info.getState(), info.getOperation(), info.getExtent());
            }
        }
        finally {
            span.end();
        }
    }
}

