/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.tables;

import java.nio.charset.StandardCharsets;
import java.security.SecurityPermission;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.apache.accumulo.core.client.Instance;
import org.apache.accumulo.core.client.NamespaceNotFoundException;
import org.apache.accumulo.core.client.impl.Tables;
import org.apache.accumulo.core.master.state.tables.TableState;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.zookeeper.ZooUtil;
import org.apache.accumulo.fate.zookeeper.IZooReaderWriter;
import org.apache.accumulo.fate.zookeeper.ZooUtil;
import org.apache.accumulo.server.client.HdfsZooInstance;
import org.apache.accumulo.server.tables.TableObserver;
import org.apache.accumulo.server.util.TablePropUtil;
import org.apache.accumulo.server.zookeeper.ZooCache;
import org.apache.accumulo.server.zookeeper.ZooReaderWriter;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableManager {
    private static SecurityPermission TABLE_MANAGER_PERMISSION = new SecurityPermission("tableManagerPermission");
    private static final Logger log = LoggerFactory.getLogger(TableManager.class);
    private static final Set<TableObserver> observers = Collections.synchronizedSet(new HashSet());
    private static final Map<String, TableState> tableStateCache = Collections.synchronizedMap(new HashMap());
    private static final byte[] ZERO_BYTE = new byte[]{48};
    private static TableManager tableManager = null;
    private final Instance instance = HdfsZooInstance.getInstance();
    private ZooCache zooStateCache = new ZooCache(new TableStateWatcher());

    public static void prepareNewNamespaceState(String instanceId, String namespaceId, String namespace, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        log.debug("Creating ZooKeeper entries for new namespace " + namespace + " (ID: " + namespaceId + ")");
        String zPath = "/accumulo/" + instanceId + "/namespaces" + "/" + namespaceId;
        ZooReaderWriter zoo = ZooReaderWriter.getInstance();
        zoo.putPersistentData(zPath, new byte[0], existsPolicy);
        zoo.putPersistentData(zPath + "/name", namespace.getBytes(StandardCharsets.UTF_8), existsPolicy);
        zoo.putPersistentData(zPath + "/conf", new byte[0], existsPolicy);
    }

    public static void prepareNewTableState(String instanceId, String tableId, String namespaceId, String tableName, TableState state, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        log.debug("Creating ZooKeeper entries for new table " + tableName + " (ID: " + tableId + ") in namespace (ID: " + namespaceId + ")");
        Pair qualifiedTableName = Tables.qualify((String)tableName);
        tableName = (String)qualifiedTableName.getSecond();
        String zTablePath = "/accumulo/" + instanceId + "/tables" + "/" + tableId;
        ZooReaderWriter zoo = ZooReaderWriter.getInstance();
        zoo.putPersistentData(zTablePath, new byte[0], existsPolicy);
        zoo.putPersistentData(zTablePath + "/conf", new byte[0], existsPolicy);
        zoo.putPersistentData(zTablePath + "/namespace", namespaceId.getBytes(StandardCharsets.UTF_8), existsPolicy);
        zoo.putPersistentData(zTablePath + "/name", tableName.getBytes(StandardCharsets.UTF_8), existsPolicy);
        zoo.putPersistentData(zTablePath + "/flush-id", ZERO_BYTE, existsPolicy);
        zoo.putPersistentData(zTablePath + "/compact-id", ZERO_BYTE, existsPolicy);
        zoo.putPersistentData(zTablePath + "/compact-cancel-id", ZERO_BYTE, existsPolicy);
        zoo.putPersistentData(zTablePath + "/state", state.name().getBytes(StandardCharsets.UTF_8), existsPolicy);
    }

    public static synchronized TableManager getInstance() {
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            sm.checkPermission(TABLE_MANAGER_PERMISSION);
        }
        if (tableManager == null) {
            tableManager = new TableManager();
        }
        return tableManager;
    }

    private TableManager() {
        this.updateTableStateCache();
    }

    public TableState getTableState(String tableId) {
        return tableStateCache.get(tableId);
    }

    public synchronized void transitionTableState(final String tableId, final TableState newState) {
        String statePath = ZooUtil.getRoot((Instance)HdfsZooInstance.getInstance()) + "/tables" + "/" + tableId + "/state";
        try {
            ZooReaderWriter.getInstance().mutate(statePath, newState.name().getBytes(StandardCharsets.UTF_8), ZooUtil.PUBLIC, new IZooReaderWriter.Mutator(){

                public byte[] mutate(byte[] oldData) throws Exception {
                    TableState oldState = TableState.UNKNOWN;
                    if (oldData != null) {
                        oldState = TableState.valueOf((String)new String(oldData, StandardCharsets.UTF_8));
                    }
                    boolean transition = true;
                    switch (oldState) {
                        case NEW: {
                            transition = newState == TableState.OFFLINE || newState == TableState.ONLINE;
                            break;
                        }
                        case ONLINE: 
                        case UNKNOWN: 
                        case OFFLINE: {
                            transition = newState != TableState.NEW;
                            break;
                        }
                        case DELETING: {
                            transition = false;
                        }
                    }
                    if (!transition) {
                        throw new IllegalTableTransitionException(oldState, newState);
                    }
                    log.debug("Transitioning state for table " + tableId + " from " + oldState + " to " + newState);
                    return newState.name().getBytes(StandardCharsets.UTF_8);
                }
            });
        }
        catch (Exception e) {
            log.error("FATAL Failed to transition table to state " + newState);
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateTableStateCache() {
        Map<String, TableState> map = tableStateCache;
        synchronized (map) {
            for (String tableId : this.zooStateCache.getChildren(ZooUtil.getRoot((Instance)this.instance) + "/tables")) {
                if (this.zooStateCache.get(ZooUtil.getRoot((Instance)this.instance) + "/tables" + "/" + tableId + "/state") == null) continue;
                this.updateTableStateCache(tableId);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TableState updateTableStateCache(String tableId) {
        Map<String, TableState> map = tableStateCache;
        synchronized (map) {
            TableState tState = TableState.UNKNOWN;
            byte[] data = this.zooStateCache.get(ZooUtil.getRoot((Instance)this.instance) + "/tables" + "/" + tableId + "/state");
            if (data != null) {
                String sState = new String(data, StandardCharsets.UTF_8);
                try {
                    tState = TableState.valueOf((String)sState);
                }
                catch (IllegalArgumentException e) {
                    log.error("Unrecognized state for table with tableId=" + tableId + ": " + sState);
                }
                tableStateCache.put(tableId, tState);
            }
            return tState;
        }
    }

    public void addTable(String tableId, String namespaceId, String tableName, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException, NamespaceNotFoundException {
        TableManager.prepareNewTableState(this.instance.getInstanceID(), tableId, namespaceId, tableName, TableState.NEW, existsPolicy);
        this.updateTableStateCache(tableId);
    }

    public void cloneTable(String srcTable, String tableId, String tableName, String namespaceId, Map<String, String> propertiesToSet, Set<String> propertiesToExclude, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        TableManager.prepareNewTableState(this.instance.getInstanceID(), tableId, namespaceId, tableName, TableState.NEW, existsPolicy);
        String srcTablePath = "/accumulo/" + this.instance.getInstanceID() + "/tables" + "/" + srcTable + "/conf";
        String newTablePath = "/accumulo/" + this.instance.getInstanceID() + "/tables" + "/" + tableId + "/conf";
        ZooReaderWriter.getInstance().recursiveCopyPersistent(srcTablePath, newTablePath, ZooUtil.NodeExistsPolicy.OVERWRITE);
        for (Map.Entry<String, String> entry : propertiesToSet.entrySet()) {
            TablePropUtil.setTableProperty(tableId, entry.getKey(), entry.getValue());
        }
        for (String prop : propertiesToExclude) {
            ZooReaderWriter.getInstance().recursiveDelete("/accumulo/" + this.instance.getInstanceID() + "/tables" + "/" + tableId + "/conf" + "/" + prop, ZooUtil.NodeMissingPolicy.SKIP);
        }
        this.updateTableStateCache(tableId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeTable(String tableId) throws KeeperException, InterruptedException {
        Map<String, TableState> map = tableStateCache;
        synchronized (map) {
            tableStateCache.remove(tableId);
            ZooReaderWriter.getInstance().recursiveDelete(ZooUtil.getRoot((Instance)this.instance) + "/tables" + "/" + tableId + "/state", ZooUtil.NodeMissingPolicy.SKIP);
            ZooReaderWriter.getInstance().recursiveDelete(ZooUtil.getRoot((Instance)this.instance) + "/tables" + "/" + tableId, ZooUtil.NodeMissingPolicy.SKIP);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addObserver(TableObserver to) {
        Set<TableObserver> set = observers;
        synchronized (set) {
            Map<String, TableState> map = tableStateCache;
            synchronized (map) {
                to.initialize(Collections.unmodifiableMap(tableStateCache));
                return observers.add(to);
            }
        }
    }

    public void removeNamespace(String namespaceId) throws KeeperException, InterruptedException {
        ZooReaderWriter.getInstance().recursiveDelete(ZooUtil.getRoot((Instance)this.instance) + "/namespaces" + "/" + namespaceId, ZooUtil.NodeMissingPolicy.SKIP);
    }

    private class TableStateWatcher
    implements Watcher {
        private TableStateWatcher() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void process(WatchedEvent event) {
            if (log.isTraceEnabled()) {
                log.trace("{}", (Object)event);
            }
            String zPath = event.getPath();
            Watcher.Event.EventType zType = event.getType();
            String tablesPrefix = ZooUtil.getRoot((Instance)TableManager.this.instance) + "/tables";
            String tableId = null;
            if (zPath != null && zPath.startsWith(tablesPrefix + "/")) {
                String suffix = zPath.substring(tablesPrefix.length() + 1);
                if (suffix.contains("/")) {
                    String[] sa = suffix.split("/", 2);
                    if ("/state".equals("/" + sa[1])) {
                        tableId = sa[0];
                    }
                }
                if (tableId == null) {
                    log.warn("Unknown path in " + event);
                    return;
                }
            }
            block4 : switch (zType) {
                case NodeChildrenChanged: {
                    if (zPath != null && zPath.equals(tablesPrefix)) {
                        TableManager.this.updateTableStateCache();
                        break;
                    }
                    log.warn("Unexpected path " + zPath);
                    break;
                }
                case NodeCreated: 
                case NodeDataChanged: {
                    TableState tState = TableManager.this.updateTableStateCache(tableId);
                    log.debug("State transition to " + tState + " @ " + event);
                    Set set = observers;
                    synchronized (set) {
                        for (TableObserver to : observers) {
                            to.stateChanged(tableId, tState);
                        }
                        break;
                    }
                }
                case NodeDeleted: {
                    if (zPath == null || tableId == null || !zPath.equals(tablesPrefix + "/" + tableId + "/state") && !zPath.equals(tablesPrefix + "/" + tableId + "/conf") && !zPath.equals(tablesPrefix + "/" + tableId + "/name")) break;
                    tableStateCache.remove(tableId);
                    break;
                }
                case None: {
                    Set set;
                    switch (event.getState()) {
                        case Expired: {
                            if (log.isTraceEnabled()) {
                                log.trace("Session expired " + event);
                            }
                            set = observers;
                            synchronized (set) {
                                for (TableObserver to : observers) {
                                    to.sessionExpired();
                                }
                                break block4;
                            }
                        }
                        default: {
                            if (!log.isTraceEnabled()) break block4;
                        }
                    }
                    log.trace("Ignored " + event);
                    break;
                }
                default: {
                    log.warn("Unandled " + event);
                }
            }
        }
    }

    public static class IllegalTableTransitionException
    extends Exception {
        private static final long serialVersionUID = 1L;
        final TableState oldState;
        final TableState newState;

        public IllegalTableTransitionException(TableState oldState, TableState newState) {
            this.oldState = oldState;
            this.newState = newState;
        }

        public TableState getOldState() {
            return this.oldState;
        }

        public TableState getNewState() {
            return this.newState;
        }
    }
}

