/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.Instance;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.util.AddressUtil;
import org.apache.accumulo.core.util.UtilWaitThread;
import org.apache.accumulo.core.volume.Volume;
import org.apache.accumulo.core.zookeeper.ZooUtil;
import org.apache.accumulo.fate.ReadOnlyStore;
import org.apache.accumulo.fate.TStore;
import org.apache.accumulo.fate.ZooStore;
import org.apache.accumulo.fate.zookeeper.IZooReaderWriter;
import org.apache.accumulo.server.ServerConstants;
import org.apache.accumulo.server.client.HdfsZooInstance;
import org.apache.accumulo.server.conf.ServerConfigurationFactory;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.accumulo.server.util.time.SimpleTimer;
import org.apache.accumulo.server.watcher.Log4jConfiguration;
import org.apache.accumulo.server.watcher.MonitorLog4jWatcher;
import org.apache.accumulo.server.zookeeper.ZooReaderWriter;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;
import org.apache.log4j.helpers.LogLog;
import org.apache.zookeeper.KeeperException;

public class Accumulo {
    private static final Logger log = Logger.getLogger(Accumulo.class);

    public static synchronized void updateAccumuloVersion(VolumeManager fs, int oldVersion) {
        for (Volume volume : fs.getVolumes()) {
            try {
                if (Accumulo.getAccumuloPersistentVersion(volume) != oldVersion) continue;
                log.debug((Object)("Attempting to upgrade " + volume));
                Path dataVersionLocation = ServerConstants.getDataVersionLocation(volume);
                fs.create(new Path(dataVersionLocation, Integer.toString(7))).close();
                Path prevDataVersionLoc = new Path(dataVersionLocation, Integer.toString(oldVersion));
                if (fs.delete(prevDataVersionLoc)) continue;
                throw new RuntimeException("Could not delete previous data version location (" + prevDataVersionLoc + ") for " + volume);
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to set accumulo version: an error occurred.", e);
            }
        }
    }

    public static synchronized int getAccumuloPersistentVersion(FileSystem fs, Path path) {
        try {
            FileStatus[] files = fs.listStatus(path);
            int dataVersion = files == null || files.length == 0 ? -1 : Integer.parseInt(files[0].getPath().getName());
            return dataVersion;
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to read accumulo version: an error occurred.", e);
        }
    }

    public static synchronized int getAccumuloPersistentVersion(Volume v) {
        Path path = ServerConstants.getDataVersionLocation(v);
        return Accumulo.getAccumuloPersistentVersion(v.getFileSystem(), path);
    }

    public static synchronized int getAccumuloPersistentVersion(VolumeManager fs) {
        return Accumulo.getAccumuloPersistentVersion(fs.getVolumes().iterator().next());
    }

    public static synchronized Path getAccumuloInstanceIdPath(VolumeManager fs) {
        Volume v = fs.getVolumes().iterator().next();
        return ServerConstants.getInstanceIdLocation(v);
    }

    static String locateLogConfig(String confDir, String application) {
        String explicitConfigFile = System.getProperty("log4j.configuration");
        if (explicitConfigFile != null) {
            return explicitConfigFile;
        }
        String[] configFiles = new String[]{String.format("%s/%s_logger.xml", confDir, application), String.format("%s/%s_logger.properties", confDir, application), String.format("%s/generic_logger.xml", confDir), String.format("%s/generic_logger.properties", confDir)};
        String defaultConfigFile = configFiles[2];
        for (String f : configFiles) {
            if (!new File(f).exists()) continue;
            return f;
        }
        return defaultConfigFile;
    }

    public static void setupLogging(String application) throws UnknownHostException {
        System.setProperty("org.apache.accumulo.core.application", application);
        if (System.getenv("ACCUMULO_LOG_DIR") != null) {
            System.setProperty("org.apache.accumulo.core.dir.log", System.getenv("ACCUMULO_LOG_DIR"));
        } else {
            System.setProperty("org.apache.accumulo.core.dir.log", System.getenv("ACCUMULO_HOME") + "/logs/");
        }
        String localhost = InetAddress.getLocalHost().getHostName();
        System.setProperty("org.apache.accumulo.core.ip.localhost.hostname", localhost);
        String logConfigFile = Accumulo.locateLogConfig(System.getenv("ACCUMULO_CONF_DIR"), application);
        LogLog.setQuietMode((boolean)true);
        Log4jConfiguration logConf = new Log4jConfiguration(logConfigFile);
        logConf.resetLogger();
    }

    public static void init(VolumeManager fs, ServerConfigurationFactory serverConfig, String application) throws IOException {
        AccumuloConfiguration conf = serverConfig.getConfiguration();
        Instance instance = serverConfig.getInstance();
        String logConfigFile = Accumulo.locateLogConfig(System.getenv("ACCUMULO_CONF_DIR"), application);
        MonitorLog4jWatcher logConfigWatcher = new MonitorLog4jWatcher(instance.getInstanceID(), logConfigFile);
        logConfigWatcher.setDelay(5000L);
        logConfigWatcher.start();
        int logPort = conf.getPort(Property.MONITOR_LOG4J_PORT);
        System.setProperty("org.apache.accumulo.core.host.log.port", Integer.toString(logPort));
        log.info((Object)(application + " starting"));
        log.info((Object)("Instance " + serverConfig.getInstance().getInstanceID()));
        int dataVersion = Accumulo.getAccumuloPersistentVersion(fs);
        log.info((Object)("Data Version " + dataVersion));
        Accumulo.waitForZookeeperAndHdfs(fs);
        if (!Accumulo.canUpgradeFromDataVersion(dataVersion)) {
            throw new RuntimeException("This version of accumulo (1.7.4) is not compatible with files stored using data version " + dataVersion);
        }
        TreeMap sortedProps = new TreeMap();
        for (Map.Entry entry : conf) {
            sortedProps.put(entry.getKey(), entry.getValue());
        }
        for (Map.Entry entry : sortedProps.entrySet()) {
            String key = (String)entry.getKey();
            log.info((Object)(key + " = " + (Property.isSensitive((String)key) ? "<hidden>" : (String)entry.getValue())));
        }
        Accumulo.monitorSwappiness(conf);
        String SSL = "SSL";
        for (Property sslProtocolProperty : Arrays.asList(Property.RPC_SSL_CLIENT_PROTOCOL, Property.RPC_SSL_ENABLED_PROTOCOLS, Property.MONITOR_SSL_INCLUDE_PROTOCOLS)) {
            String value = conf.get(sslProtocolProperty);
            if (!value.contains("SSL")) continue;
            log.warn((Object)("It is recommended that " + sslProtocolProperty + " only allow TLS"));
        }
    }

    public static boolean canUpgradeFromDataVersion(int dataVersion) {
        return ServerConstants.CAN_UPGRADE.get(dataVersion);
    }

    public static boolean persistentVersionNeedsUpgrade(int accumuloPersistentVersion) {
        return ServerConstants.NEEDS_UPGRADE.get(accumuloPersistentVersion);
    }

    public static void monitorSwappiness(AccumuloConfiguration config) {
        SimpleTimer.getInstance(config).schedule(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                block6: {
                    try {
                        String procFile = "/proc/sys/vm/swappiness";
                        File swappiness = new File(procFile);
                        if (!swappiness.exists() || !swappiness.canRead()) break block6;
                        try (FileInputStream is = new FileInputStream(procFile);){
                            byte[] buffer = new byte[10];
                            int bytes = ((InputStream)is).read(buffer);
                            String setting = new String(buffer, 0, bytes, StandardCharsets.UTF_8);
                            setting = setting.trim();
                            if (bytes > 0 && Integer.parseInt(setting) > 10) {
                                log.warn((Object)("System swappiness setting is greater than ten (" + setting + ") which can cause time-sensitive operations to be delayed.  Accumulo is time sensitive because it needs to maintain distributed lock agreement."));
                            }
                        }
                    }
                    catch (Throwable t) {
                        log.error((Object)t, t);
                    }
                }
            }
        }, 1000L, 600000L);
    }

    public static void waitForZookeeperAndHdfs(VolumeManager fs) {
        log.info((Object)"Attempting to talk to zookeeper");
        while (true) {
            try {
                ZooReaderWriter.getInstance().getChildren("/accumulo");
            }
            catch (InterruptedException interruptedException) {
                continue;
            }
            catch (KeeperException ex) {
                log.info((Object)"Waiting for accumulo to be initialized");
                UtilWaitThread.sleep((long)1000L);
                continue;
            }
            break;
        }
        log.info((Object)"ZooKeeper connected and initialized, attempting to talk to HDFS");
        long sleep = 1000L;
        int unknownHostTries = 3;
        while (true) {
            try {
                if (fs.isReady()) break;
                log.warn((Object)"Waiting for the NameNode to leave safemode");
            }
            catch (IOException ex) {
                log.warn((Object)"Unable to connect to HDFS", (Throwable)ex);
            }
            catch (IllegalArgumentException exception) {
                if (exception.getCause() instanceof UnknownHostException) {
                    if (unknownHostTries <= 0) {
                        log.error((Object)"Unable to connect to HDFS and have exceeded the maximum number of retries.", (Throwable)exception);
                        throw exception;
                    }
                    log.warn((Object)("Unable to connect to HDFS, will retry. cause: " + exception.getCause()));
                    sleep = Math.max(sleep, (long)((AddressUtil.getAddressCacheNegativeTtl((UnknownHostException)((UnknownHostException)exception.getCause())) + 1) * 1000));
                    --unknownHostTries;
                }
                throw exception;
            }
            log.info((Object)("Backing off due to failure; current sleep period is " + (double)sleep / 1000.0 + " seconds"));
            UtilWaitThread.sleep((long)sleep);
            sleep = Math.min(60000L, sleep * 2L);
        }
        log.info((Object)"Connected to HDFS");
    }

    public static void abortIfFateTransactions() {
        try {
            ReadOnlyStore fate = new ReadOnlyStore((TStore)new ZooStore(ZooUtil.getRoot((Instance)HdfsZooInstance.getInstance()) + "/fate", (IZooReaderWriter)ZooReaderWriter.getInstance()));
            if (!fate.list().isEmpty()) {
                throw new AccumuloException("Aborting upgrade because there are outstanding FATE transactions from a previous Accumulo version. Please see the README document for instructions on what to do under your previous version.");
            }
        }
        catch (Exception exception) {
            log.fatal((Object)"Problem verifying Fate readiness", (Throwable)exception);
            System.exit(1);
        }
    }
}

