/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.conf.store.impl;

import com.github.benmanes.caffeine.cache.Ticker;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import org.apache.accumulo.core.data.InstanceId;
import org.apache.accumulo.core.fate.zookeeper.ZooReader;
import org.apache.accumulo.core.fate.zookeeper.ZooReaderWriter;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.metrics.MetricsProducer;
import org.apache.accumulo.core.metrics.MetricsUtil;
import org.apache.accumulo.server.conf.codec.VersionedPropCodec;
import org.apache.accumulo.server.conf.codec.VersionedProperties;
import org.apache.accumulo.server.conf.store.PropCache;
import org.apache.accumulo.server.conf.store.PropChangeListener;
import org.apache.accumulo.server.conf.store.PropStore;
import org.apache.accumulo.server.conf.store.PropStoreKey;
import org.apache.accumulo.server.conf.store.SystemPropKey;
import org.apache.accumulo.server.conf.store.impl.PropCacheCaffeineImpl;
import org.apache.accumulo.server.conf.store.impl.PropStoreMetrics;
import org.apache.accumulo.server.conf.store.impl.PropStoreWatcher;
import org.apache.accumulo.server.conf.store.impl.ReadyMonitor;
import org.apache.accumulo.server.conf.store.impl.ZooPropLoader;
import org.apache.accumulo.server.conf.util.ConfigTransformer;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooPropStore
implements PropStore,
PropChangeListener {
    private static final Logger log = LoggerFactory.getLogger(ZooPropStore.class);
    private static final VersionedPropCodec codec = VersionedPropCodec.getDefault();
    private final ZooReaderWriter zrw;
    private final PropStoreWatcher propStoreWatcher;
    private final PropCacheCaffeineImpl cache;
    private final PropStoreMetrics cacheMetrics = new PropStoreMetrics();
    private final ReadyMonitor zkReadyMon;

    private ZooPropStore(InstanceId instanceId, ZooReaderWriter zrw) {
        this(instanceId, zrw, null, null, null);
    }

    @SuppressFBWarnings(value={"PREDICTABLE_RANDOM"}, justification="random number not used in secure context")
    ZooPropStore(InstanceId instanceId, ZooReaderWriter zrw, ReadyMonitor monitor, PropStoreWatcher watcher, Ticker ticker) {
        this.zrw = zrw;
        this.zkReadyMon = Objects.requireNonNullElseGet(monitor, () -> new ReadyMonitor("prop-store", Math.round((double)zrw.getSessionTimeout() * 1.75)));
        this.propStoreWatcher = Objects.requireNonNullElseGet(watcher, () -> new PropStoreWatcher(this.zkReadyMon));
        ZooPropLoader propLoader = new ZooPropLoader(zrw, codec, this.propStoreWatcher, this.cacheMetrics);
        this.cache = ticker == null ? new PropCacheCaffeineImpl.Builder(propLoader, this.cacheMetrics).build() : new PropCacheCaffeineImpl.Builder(propLoader, this.cacheMetrics).forTests(ticker).build();
        MetricsUtil.initializeProducers((MetricsProducer[])new MetricsProducer[]{this.cacheMetrics});
        try {
            String path = ZooUtil.getRoot((InstanceId)instanceId);
            if (!zrw.exists(path, (Watcher)this.propStoreWatcher)) {
                throw new IllegalStateException("Instance may not have been initialized, root node: " + path + " does not exist in ZooKeeper");
            }
            log.debug("Have a ZooKeeper connection and found instance node: {}", (Object)instanceId);
            this.zkReadyMon.setReady();
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Interrupted trying to read root node " + instanceId + " from ZooKeeper", ex);
        }
        catch (KeeperException ex) {
            throw new IllegalStateException("Failed to read root node " + instanceId + " from ZooKeeper", ex);
        }
    }

    public static ZooPropStore initialize(@NonNull InstanceId instanceId, @NonNull ZooReaderWriter zrw) {
        return new ZooPropStore(instanceId, zrw);
    }

    @Override
    public boolean exists(PropStoreKey<?> propStoreKey) {
        try {
            if (this.zrw.exists(propStoreKey.getPath())) {
                return true;
            }
        }
        catch (KeeperException keeperException) {
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Interrupted testing if node exists", ex);
        }
        return false;
    }

    public PropStoreMetrics getMetrics() {
        return this.cacheMetrics;
    }

    @Override
    public void create(PropStoreKey<?> propStoreKey, Map<String, String> props) {
        try {
            VersionedProperties vProps = new VersionedProperties(props);
            String path = propStoreKey.getPath();
            this.zrw.putPrivatePersistentData(path, codec.toBytes(vProps), ZooUtil.NodeExistsPolicy.FAIL);
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            throw new IllegalStateException("Failed to serialize properties for " + propStoreKey, ex);
        }
    }

    @Override
    public @NonNull VersionedProperties get(PropStoreKey<?> propStoreKey) {
        this.checkZkConnection();
        this.propStoreWatcher.registerListener(propStoreKey, this);
        VersionedProperties props = this.cache.get(propStoreKey);
        if (props != null) {
            return props;
        }
        if (propStoreKey instanceof SystemPropKey) {
            return new ConfigTransformer(this.zrw, codec, this.propStoreWatcher).transform(propStoreKey, propStoreKey.getPath(), false);
        }
        throw new IllegalStateException("Invalid request for " + propStoreKey + ", the property node does not exist");
    }

    public static @Nullable VersionedProperties readFromZk(PropStoreKey<?> propStoreKey, PropStoreWatcher watcher, ZooReader zooReader) throws IOException, KeeperException, InterruptedException {
        try {
            Stat stat = new Stat();
            byte[] bytes = zooReader.getData(propStoreKey.getPath(), (Watcher)watcher, stat);
            if (stat.getDataLength() == 0) {
                return null;
            }
            return codec.fromBytes(stat.getVersion(), bytes);
        }
        catch (KeeperException.NoNodeException ex) {
            return null;
        }
    }

    @Override
    public void putAll(@NonNull PropStoreKey<?> propStoreKey, @NonNull Map<String, String> props) {
        if (props.isEmpty()) {
            return;
        }
        this.mutateVersionedProps(propStoreKey, VersionedProperties::addOrUpdate, props);
    }

    @Override
    public void replaceAll(@NonNull PropStoreKey<?> propStoreKey, long version, @NonNull Map<String, String> props) {
        this.mutateVersionedProps(propStoreKey, VersionedProperties::replaceAll, version, props);
    }

    @Override
    public void removeProperties(@NonNull PropStoreKey<?> propStoreKey, @NonNull Collection<String> keys) {
        if (keys.isEmpty()) {
            return;
        }
        this.mutateVersionedProps(propStoreKey, VersionedProperties::remove, keys);
    }

    @Override
    public void delete(@NonNull PropStoreKey<?> propStoreKey) {
        Objects.requireNonNull(propStoreKey, "prop store delete() - Must provide propCacheId");
        try {
            log.trace("called delete() for: {}", propStoreKey);
            String path = propStoreKey.getPath();
            this.zrw.delete(path);
            this.cache.remove(propStoreKey);
        }
        catch (InterruptedException | KeeperException ex) {
            throw new IllegalStateException("Failed to delete properties for propCacheId " + propStoreKey, ex);
        }
    }

    private <T> void mutateVersionedProps(PropStoreKey<?> propStoreKey, BiFunction<VersionedProperties, T, VersionedProperties> action, T changes) {
        log.trace("mutateVersionedProps called for: {}", propStoreKey);
        try {
            VersionedProperties vProps = this.cache.getIfCached(propStoreKey);
            if (vProps == null) {
                vProps = this.readPropsFromZk(propStoreKey);
            }
            for (int attempts = 3; attempts > 0; --attempts) {
                VersionedProperties updates = action.apply(vProps, (VersionedProperties)changes);
                if (this.zrw.overwritePersistentData(propStoreKey.getPath(), codec.toBytes(updates), (int)updates.getDataVersion())) {
                    return;
                }
                Thread.sleep(20L);
                vProps = this.readPropsFromZk(propStoreKey);
            }
            throw new IllegalStateException("failed to remove properties to zooKeeper for " + propStoreKey, null);
        }
        catch (IOException | IllegalArgumentException ex) {
            throw new IllegalStateException("Codec failed to decode / encode properties for " + propStoreKey, ex);
        }
        catch (InterruptedException | KeeperException ex) {
            if (ex instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new IllegalStateException("failed to remove properties to zooKeeper for " + propStoreKey, ex);
        }
    }

    private <T> void mutateVersionedProps(PropStoreKey<?> propStoreKey, BiFunction<VersionedProperties, T, VersionedProperties> action, long existingVersion, T changes) {
        log.trace("mutateVersionedProps called for: {}", propStoreKey);
        try {
            VersionedProperties vProps = this.cache.getIfCached(propStoreKey);
            if (vProps == null) {
                vProps = this.readPropsFromZk(propStoreKey);
            }
            if (vProps.getDataVersion() != existingVersion) {
                throw new ConcurrentModificationException("Failed to modify properties to zooKeeper for " + propStoreKey + ", properties changed since reading.", null);
            }
            VersionedProperties updates = action.apply(vProps, (VersionedProperties)changes);
            if (!this.zrw.overwritePersistentData(propStoreKey.getPath(), codec.toBytes(updates), (int)updates.getDataVersion())) {
                throw new ConcurrentModificationException("Failed to modify properties to zooKeeper for " + propStoreKey + ", properties changed since reading.", null);
            }
        }
        catch (IOException | IllegalArgumentException ex) {
            throw new IllegalStateException("Codec failed to decode / encode properties for " + propStoreKey, ex);
        }
        catch (InterruptedException | KeeperException ex) {
            if (ex instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new IllegalStateException("failed to modify properties to zooKeeper for " + propStoreKey, ex);
        }
    }

    @Override
    public void registerAsListener(PropStoreKey<?> propStoreKey, PropChangeListener listener) {
        this.propStoreWatcher.registerListener(propStoreKey, listener);
    }

    private void checkZkConnection() {
        if (this.zkReadyMon.test()) {
            return;
        }
        this.cache.removeAll();
        this.zkReadyMon.isReady();
    }

    @Override
    public void zkChangeEvent(PropStoreKey<?> propStoreKey) {
        log.trace("Received change event from ZooKeeper for: {} removed from cache", propStoreKey);
        this.cache.remove(propStoreKey);
    }

    @Override
    public void cacheChangeEvent(PropStoreKey<?> propStoreKey) {
        log.trace("zkChangeEvent: {}", propStoreKey);
    }

    @Override
    public void deleteEvent(PropStoreKey<?> propStoreKey) {
        log.trace("deleteEvent: {}", propStoreKey);
        this.cache.remove(propStoreKey);
    }

    @Override
    public void connectionEvent() {
        log.trace("connectionEvent");
        this.cache.removeAll();
    }

    private VersionedProperties readPropsFromZk(PropStoreKey<?> propStoreKey) throws KeeperException, IOException {
        try {
            Stat stat = new Stat();
            byte[] bytes = this.zrw.getData(propStoreKey.getPath(), stat);
            if (stat.getDataLength() == 0) {
                return new VersionedProperties();
            }
            return codec.fromBytes(stat.getVersion(), bytes);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Interrupt received during ZooKeeper read", ex);
        }
    }

    @Override
    public PropCache getCache() {
        return this.cache;
    }

    @Override
    public @Nullable VersionedProperties getIfCached(PropStoreKey<?> propStoreKey) {
        return this.cache.getIfCached(propStoreKey);
    }

    @Override
    public boolean validateDataVersion(PropStoreKey<?> storeKey, long expectedVersion) {
        try {
            Stat stat = this.zrw.getStatus(storeKey.getPath());
            log.trace("data version sync: stat returned: {} for {}", (Object)stat, storeKey);
            if (stat == null || expectedVersion != (long)stat.getVersion()) {
                this.propStoreWatcher.signalZkChangeEvent(storeKey);
                return false;
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException(ex);
        }
        catch (KeeperException.NoNodeException ex) {
            this.propStoreWatcher.signalZkChangeEvent(storeKey);
            return false;
        }
        catch (KeeperException ex) {
            log.debug("exception occurred verifying data version for {}", storeKey);
            return false;
        }
        return true;
    }
}

