/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.security.delegation;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import javax.crypto.SecretKey;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.admin.DelegationTokenConfig;
import org.apache.accumulo.core.clientImpl.AuthenticationTokenIdentifier;
import org.apache.accumulo.core.data.InstanceId;
import org.apache.accumulo.core.securityImpl.thrift.TAuthenticationTokenIdentifier;
import org.apache.accumulo.server.security.delegation.AuthenticationKey;
import org.apache.accumulo.server.security.delegation.ZooAuthenticationKeyDistributor;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuthenticationTokenSecretManager
extends SecretManager<AuthenticationTokenIdentifier> {
    private static final Logger log = LoggerFactory.getLogger(AuthenticationTokenSecretManager.class);
    private final InstanceId instanceID;
    private final long tokenMaxLifetime;
    private final ConcurrentHashMap<Integer, AuthenticationKey> allKeys = new ConcurrentHashMap();
    private AuthenticationKey currentKey;

    public AuthenticationTokenSecretManager(InstanceId instanceID, long tokenMaxLifetime) {
        Objects.requireNonNull(instanceID);
        Preconditions.checkArgument((tokenMaxLifetime > 0L ? 1 : 0) != 0, (Object)"Max lifetime must be positive");
        this.instanceID = instanceID;
        this.tokenMaxLifetime = tokenMaxLifetime;
    }

    private byte[] createPassword(AuthenticationTokenIdentifier identifier, DelegationTokenConfig cfg) {
        long requestedLifetime;
        long now = System.currentTimeMillis();
        identifier.setIssueDate(now);
        identifier.setExpirationDate(this.calculateExpirationDate());
        if (cfg != null && (requestedLifetime = cfg.getTokenLifetime(TimeUnit.MILLISECONDS)) > 0L) {
            long requestedExpirationDate = identifier.getIssueDate() + requestedLifetime;
            if (requestedExpirationDate < identifier.getIssueDate()) {
                requestedExpirationDate = Long.MAX_VALUE;
            }
            if (requestedExpirationDate > identifier.getExpirationDate()) {
                throw new RuntimeException("Requested token lifetime exceeds configured maximum");
            }
            log.trace("Overriding token expiration date from {} to {}", (Object)identifier.getExpirationDate(), (Object)requestedExpirationDate);
            identifier.setExpirationDate(requestedExpirationDate);
        }
        return this.createPassword(identifier);
    }

    private long calculateExpirationDate() {
        long now = System.currentTimeMillis();
        long expiration = now + this.tokenMaxLifetime;
        if (expiration < now) {
            expiration = Long.MAX_VALUE;
        }
        return expiration;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected byte[] createPassword(AuthenticationTokenIdentifier identifier) {
        AuthenticationKey secretKey;
        AuthenticationTokenSecretManager authenticationTokenSecretManager = this;
        synchronized (authenticationTokenSecretManager) {
            secretKey = this.currentKey;
        }
        identifier.setKeyId(secretKey.getKeyId());
        identifier.setInstanceId(this.instanceID);
        if (!identifier.isSetIssueDate()) {
            identifier.setIssueDate(System.currentTimeMillis());
        }
        if (!identifier.isSetExpirationDate()) {
            identifier.setExpirationDate(this.calculateExpirationDate());
        }
        return AuthenticationTokenSecretManager.createPassword((byte[])identifier.getBytes(), (SecretKey)secretKey.getKey());
    }

    public byte[] retrievePassword(AuthenticationTokenIdentifier identifier) throws SecretManager.InvalidToken {
        long now = System.currentTimeMillis();
        if (identifier.getExpirationDate() < now) {
            throw new SecretManager.InvalidToken("Token has expired");
        }
        if (identifier.getIssueDate() > now) {
            throw new SecretManager.InvalidToken("Token issued in the future");
        }
        AuthenticationKey managerKey = this.allKeys.get(identifier.getKeyId());
        if (managerKey == null) {
            throw new SecretManager.InvalidToken("Unknown manager key for token (id=" + identifier.getKeyId() + ")");
        }
        return AuthenticationTokenSecretManager.createPassword((byte[])identifier.getBytes(), (SecretKey)managerKey.getKey());
    }

    public AuthenticationTokenIdentifier createIdentifier() {
        return new AuthenticationTokenIdentifier(new TAuthenticationTokenIdentifier());
    }

    public Map.Entry<Token<AuthenticationTokenIdentifier>, AuthenticationTokenIdentifier> generateToken(String username, DelegationTokenConfig cfg) throws AccumuloException {
        byte[] password;
        Objects.requireNonNull(username);
        Objects.requireNonNull(cfg);
        AuthenticationTokenIdentifier id = new AuthenticationTokenIdentifier(new TAuthenticationTokenIdentifier(username));
        StringBuilder svcName = new StringBuilder("AccumuloDelegationToken");
        if (id.getInstanceId() != null) {
            svcName.append("-").append(id.getInstanceId());
        }
        try {
            password = this.createPassword(id, cfg);
        }
        catch (RuntimeException e) {
            throw new AccumuloException(e.getMessage());
        }
        Token token = new Token(id.getBytes(), password, id.getKind(), new Text(svcName.toString()));
        return Maps.immutableEntry((Object)token, (Object)id);
    }

    public synchronized void addKey(AuthenticationKey key) {
        Objects.requireNonNull(key);
        log.debug("Adding AuthenticationKey with keyId {}", (Object)key.getKeyId());
        this.allKeys.put(key.getKeyId(), key);
        if (this.currentKey == null || key.getKeyId() > this.currentKey.getKeyId()) {
            this.currentKey = key;
        }
    }

    synchronized boolean removeKey(Integer keyId) {
        Objects.requireNonNull(keyId);
        log.debug("Removing AuthenticationKey with keyId {}", (Object)keyId);
        return this.allKeys.remove(keyId) != null;
    }

    @VisibleForTesting
    AuthenticationKey getCurrentKey() {
        return this.currentKey;
    }

    @VisibleForTesting
    Map<Integer, AuthenticationKey> getKeys() {
        return this.allKeys;
    }

    synchronized int removeExpiredKeys(ZooAuthenticationKeyDistributor keyDistributor) {
        long now = System.currentTimeMillis();
        int keysRemoved = 0;
        Iterator<Map.Entry<Integer, AuthenticationKey>> iter = this.allKeys.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<Integer, AuthenticationKey> entry = iter.next();
            AuthenticationKey key = entry.getValue();
            if (key.getExpirationDate() >= now) continue;
            log.debug("Removing expired delegation token key {}", (Object)key.getKeyId());
            iter.remove();
            ++keysRemoved;
            try {
                keyDistributor.remove(key);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Failed to remove AuthenticationKey from ZooKeeper. Exiting", e);
                throw new RuntimeException(e);
            }
        }
        return keysRemoved;
    }

    synchronized boolean isCurrentKeySet() {
        return this.currentKey != null;
    }

    public synchronized void removeAllKeys() {
        this.allKeys.clear();
        this.currentKey = null;
    }

    protected SecretKey generateSecret() {
        return super.generateSecret();
    }

    public static SecretKey createSecretKey(byte[] raw) {
        return SecretManager.createSecretKey((byte[])raw);
    }
}

