/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package org.apache.activemq.artemis.shaded.io.netty.example.http2.helloworld.server;

import static org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpHeaderNames.CONNECTION;
import static org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_LENGTH;
import static org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpResponseStatus.CONTINUE;
import static org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpResponseStatus.OK;
import static org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpVersion.HTTP_1_1;
import static org.apache.activemq.artemis.shaded.io.netty.util.internal.ObjectUtil.checkNotNull;

import org.apache.activemq.artemis.shaded.io.netty.buffer.ByteBuf;
import org.apache.activemq.artemis.shaded.io.netty.buffer.ByteBufUtil;
import org.apache.activemq.artemis.shaded.io.netty.channel.ChannelFutureListener;
import org.apache.activemq.artemis.shaded.io.netty.channel.ChannelHandlerContext;
import org.apache.activemq.artemis.shaded.io.netty.channel.SimpleChannelInboundHandler;
import org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.DefaultFullHttpResponse;
import org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.FullHttpRequest;
import org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.FullHttpResponse;
import org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpUtil;
import org.apache.activemq.artemis.shaded.io.netty.handler.codec.http.HttpHeaderValues;

/**
 * HTTP handler that responds with a "Hello World"
 */
public class HelloWorldHttp1Handler extends SimpleChannelInboundHandler<FullHttpRequest> {
    private final String establishApproach;

    public HelloWorldHttp1Handler(String establishApproach) {
        this.establishApproach = checkNotNull(establishApproach, "establishApproach");
    }

    @Override
    public void channelRead0(ChannelHandlerContext ctx, FullHttpRequest req) throws Exception {
        if (HttpUtil.is100ContinueExpected(req)) {
            ctx.write(new DefaultFullHttpResponse(HTTP_1_1, CONTINUE));
        }
        boolean keepAlive = HttpUtil.isKeepAlive(req);

        ByteBuf content = ctx.alloc().buffer();
        content.writeBytes(HelloWorldHttp2Handler.RESPONSE_BYTES.duplicate());
        ByteBufUtil.writeAscii(content, " - via " + req.protocolVersion() + " (" + establishApproach + ")");

        FullHttpResponse response = new DefaultFullHttpResponse(HTTP_1_1, OK, content);
        response.headers().set(CONTENT_TYPE, "text/plain; charset=UTF-8");
        response.headers().setInt(CONTENT_LENGTH, response.content().readableBytes());

        if (!keepAlive) {
            ctx.write(response).addListener(ChannelFutureListener.CLOSE);
        } else {
            response.headers().set(CONNECTION, HttpHeaderValues.KEEP_ALIVE);
            ctx.write(response);
        }
    }

    @Override
    public void channelReadComplete(ChannelHandlerContext ctx) throws Exception {
        ctx.flush();
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        cause.printStackTrace();
        ctx.close();
    }
}
