/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.remoting.impl.netty;

import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelPromise;
import io.netty.channel.EventLoop;
import java.net.SocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import org.apache.activemq.artemis.api.core.ActiveMQBuffer;
import org.apache.activemq.artemis.api.core.ActiveMQInterruptedException;
import org.apache.activemq.artemis.api.core.TransportConfiguration;
import org.apache.activemq.artemis.core.buffers.impl.ChannelBufferWrapper;
import org.apache.activemq.artemis.core.client.ActiveMQClientLogger;
import org.apache.activemq.artemis.core.remoting.impl.netty.NettyConnectorFactory;
import org.apache.activemq.artemis.core.security.ActiveMQPrincipal;
import org.apache.activemq.artemis.shaded.org.jboss.logging.Logger;
import org.apache.activemq.artemis.spi.core.protocol.RemotingConnection;
import org.apache.activemq.artemis.spi.core.remoting.BaseConnectionLifeCycleListener;
import org.apache.activemq.artemis.spi.core.remoting.Connection;
import org.apache.activemq.artemis.spi.core.remoting.ReadyListener;
import org.apache.activemq.artemis.utils.Env;
import org.apache.activemq.artemis.utils.IPV6Util;

public class NettyConnection
implements Connection {
    private static final Logger logger = Logger.getLogger(NettyConnection.class);
    private static final int DEFAULT_BATCH_BYTES = Integer.getInteger("io.netty.batch.bytes", 8192);
    private static final int DEFAULT_WAIT_MILLIS = 10000;
    protected final Channel channel;
    private final BaseConnectionLifeCycleListener<?> listener;
    private final boolean directDeliver;
    private final Map<String, Object> configuration;
    private final List<ReadyListener> readyListeners = new ArrayList<ReadyListener>();
    private final ThreadLocal<ArrayList<ReadyListener>> localListenersPool = new ThreadLocal();
    private final boolean batchingEnabled;
    private final int writeBufferHighWaterMark;
    private final int batchLimit;
    private boolean closed;
    private RemotingConnection protocolConnection;
    private boolean ready = true;

    public NettyConnection(Map<String, Object> configuration, Channel channel, BaseConnectionLifeCycleListener<?> listener, boolean batchingEnabled, boolean directDeliver) {
        this.configuration = configuration;
        this.channel = channel;
        this.listener = listener;
        this.directDeliver = directDeliver;
        this.batchingEnabled = batchingEnabled;
        this.writeBufferHighWaterMark = this.channel.config().getWriteBufferHighWaterMark();
        this.batchLimit = batchingEnabled ? Math.min(this.writeBufferHighWaterMark, DEFAULT_BATCH_BYTES) : 0;
    }

    private static void waitFor(ChannelPromise promise, long millis) {
        try {
            boolean completed = promise.await(millis);
            if (!completed) {
                ActiveMQClientLogger.LOGGER.timeoutFlushingPacket();
            }
        }
        catch (InterruptedException e) {
            throw new ActiveMQInterruptedException(e);
        }
    }

    private static int batchBufferSize(Channel channel, int writeBufferHighWaterMark) {
        int bytesBeforeUnwritable = (int)channel.bytesBeforeUnwritable();
        assert (bytesBeforeUnwritable >= 0);
        int writtenBytes = writeBufferHighWaterMark - bytesBeforeUnwritable;
        assert (writtenBytes >= 0);
        return writtenBytes;
    }

    public final int pendingWritesOnChannel() {
        return NettyConnection.batchBufferSize(this.channel, this.writeBufferHighWaterMark);
    }

    public final Channel getNettyChannel() {
        return this.channel;
    }

    @Override
    public final void setAutoRead(boolean autoRead) {
        this.channel.config().setAutoRead(autoRead);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final boolean isWritable(ReadyListener callback) {
        List<ReadyListener> list = this.readyListeners;
        synchronized (list) {
            if (!this.ready) {
                this.readyListeners.add(callback);
            }
            return this.ready;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final void fireReady(boolean ready) {
        ArrayList<ReadyListener> readyToCall = this.localListenersPool.get();
        if (readyToCall != null) {
            this.localListenersPool.set(null);
        }
        List<ReadyListener> list = this.readyListeners;
        synchronized (list) {
            this.ready = ready;
            if (ready) {
                int size = this.readyListeners.size();
                if (readyToCall != null) {
                    readyToCall.ensureCapacity(size);
                }
                try {
                    for (int i = 0; i < size; ++i) {
                        ReadyListener readyListener2 = this.readyListeners.get(i);
                        if (readyListener2 == null) {
                            break;
                        }
                        if (readyToCall == null) {
                            readyToCall = new ArrayList(size);
                        }
                        readyToCall.add(readyListener2);
                    }
                }
                finally {
                    this.readyListeners.clear();
                }
            }
        }
        if (readyToCall != null) {
            try {
                readyToCall.forEach(readyListener -> {
                    try {
                        readyListener.readyForWriting();
                    }
                    catch (Throwable logOnly) {
                        ActiveMQClientLogger.LOGGER.failedToSetChannelReadyForWriting(logOnly);
                    }
                });
            }
            catch (Throwable t) {
                ActiveMQClientLogger.LOGGER.failedToSetChannelReadyForWriting(t);
            }
            finally {
                readyToCall.clear();
                if (this.localListenersPool.get() != null) {
                    this.localListenersPool.set(readyToCall);
                }
            }
        }
    }

    @Override
    public final void forceClose() {
        if (this.channel != null) {
            try {
                this.channel.close();
            }
            catch (Throwable e) {
                ActiveMQClientLogger.LOGGER.failedForceClose(e);
            }
        }
    }

    public final Channel getChannel() {
        return this.channel;
    }

    @Override
    public final RemotingConnection getProtocolConnection() {
        return this.protocolConnection;
    }

    @Override
    public final void setProtocolConnection(RemotingConnection protocolConnection) {
        this.protocolConnection = protocolConnection;
    }

    @Override
    public final void close() {
        if (this.closed) {
            return;
        }
        EventLoop eventLoop = this.channel.eventLoop();
        boolean inEventLoop = eventLoop.inEventLoop();
        if (!inEventLoop) {
            this.closeChannel(this.channel, false);
        } else {
            eventLoop.execute(() -> this.closeChannel(this.channel, true));
        }
        this.closed = true;
        this.listener.connectionDestroyed(this.getID());
    }

    @Override
    public ActiveMQBuffer createTransportBuffer(int size) {
        try {
            return new ChannelBufferWrapper(this.channel.alloc().directBuffer(size), true);
        }
        catch (OutOfMemoryError oom) {
            long totalPendingWriteBytes = NettyConnection.batchBufferSize(this.channel, this.writeBufferHighWaterMark);
            logger.warn((Object)("Trying to allocate " + size + " bytes, System is throwing OutOfMemoryError on NettyConnection " + this + ", there are currently " + "pendingWrites: [NETTY] -> " + totalPendingWriteBytes + " causes: " + oom.getMessage()), oom);
            throw oom;
        }
    }

    @Override
    public final Object getID() {
        return this.channel.id();
    }

    @Override
    public final void checkFlushBatchBuffer() {
        int batchBufferSize;
        if (this.batchingEnabled && (batchBufferSize = NettyConnection.batchBufferSize(this.channel, this.writeBufferHighWaterMark)) > 0) {
            this.channel.flush();
        }
    }

    @Override
    public final void write(ActiveMQBuffer buffer) {
        this.write(buffer, false, false);
    }

    @Override
    public final void write(ActiveMQBuffer buffer, boolean flush, boolean batched) {
        this.write(buffer, flush, batched, null);
    }

    private void checkConnectionState() {
        if (this.closed || !this.channel.isActive()) {
            throw new IllegalStateException("Connection " + this.getID() + " closed or disconnected");
        }
    }

    @Override
    public final boolean blockUntilWritable(int requiredCapacity, long timeout, TimeUnit timeUnit) {
        boolean canWrite;
        this.checkConnectionState();
        boolean isAllowedToBlock = this.isAllowedToBlock();
        if (!isAllowedToBlock) {
            if (Env.isTestEnv()) {
                logger.warn((Object)"FAILURE! The code is using blockUntilWritable inside a Netty worker, which would block. The code will probably need fixing!", new Exception("trace"));
            }
            if (logger.isDebugEnabled()) {
                logger.debug("Calling blockUntilWritable using a thread where it's not allowed");
            }
            return this.canWrite(requiredCapacity);
        }
        long timeoutNanos = timeUnit.toNanos(timeout);
        long deadline = System.nanoTime() + timeoutNanos;
        long parkNanos = timeoutNanos >= 1000000L ? 100000L : 1000L;
        while (!(canWrite = this.canWrite(requiredCapacity)) && System.nanoTime() - deadline < 0L) {
            this.checkConnectionState();
            LockSupport.parkNanos(parkNanos);
        }
        return canWrite;
    }

    private boolean isAllowedToBlock() {
        EventLoop eventLoop = this.channel.eventLoop();
        boolean inEventLoop = eventLoop.inEventLoop();
        return !inEventLoop;
    }

    private boolean canWrite(int requiredCapacity) {
        long totalPendingWrites = this.pendingWritesOnChannel();
        boolean canWrite = requiredCapacity > this.writeBufferHighWaterMark ? totalPendingWrites == 0L : totalPendingWrites + (long)requiredCapacity <= (long)this.writeBufferHighWaterMark;
        return canWrite;
    }

    @Override
    public final void write(ActiveMQBuffer buffer, boolean flush, boolean batched, ChannelFutureListener futureListener) {
        int remainingBytes;
        int readableBytes = buffer.readableBytes();
        if (logger.isDebugEnabled() && (remainingBytes = this.writeBufferHighWaterMark - readableBytes) < 0) {
            logger.debug("a write request is exceeding by " + -remainingBytes + " bytes the writeBufferHighWaterMark size [ " + this.writeBufferHighWaterMark + " ] : consider to set it at least of " + readableBytes + " bytes");
        }
        Channel channel = this.channel;
        ChannelPromise promise = flush || futureListener != null ? channel.newPromise() : channel.voidPromise();
        ByteBuf bytes = buffer.byteBuf();
        assert (readableBytes >= 0);
        int writeBatchSize = this.batchLimit;
        boolean batchingEnabled = this.batchingEnabled;
        ChannelFuture future = batchingEnabled && batched && !flush && readableBytes < writeBatchSize ? this.writeBatch(bytes, readableBytes, promise) : channel.writeAndFlush(bytes, promise);
        if (futureListener != null) {
            future.addListener(futureListener);
        }
        if (flush) {
            NettyConnection.flushAndWait(channel, promise);
        }
    }

    private static void flushAndWait(Channel channel, ChannelPromise promise) {
        if (!channel.eventLoop().inEventLoop()) {
            NettyConnection.waitFor(promise, 10000L);
        } else if (logger.isDebugEnabled()) {
            logger.debug("Calling write with flush from a thread where it's not allowed");
        }
    }

    private ChannelFuture writeBatch(ByteBuf bytes, int readableBytes, ChannelPromise promise) {
        int batchBufferSize = NettyConnection.batchBufferSize(this.channel, this.writeBufferHighWaterMark);
        int nextBatchSize = batchBufferSize + readableBytes;
        if (nextBatchSize > this.batchLimit) {
            this.channel.flush();
            return this.channel.write(bytes, promise);
        }
        if (nextBatchSize == this.batchLimit) {
            return this.channel.writeAndFlush(bytes, promise);
        }
        return this.channel.write(bytes, promise);
    }

    @Override
    public final String getRemoteAddress() {
        SocketAddress address = this.channel.remoteAddress();
        if (address == null) {
            return null;
        }
        return address.toString();
    }

    @Override
    public final String getLocalAddress() {
        SocketAddress address = this.channel.localAddress();
        if (address == null) {
            return null;
        }
        return "tcp://" + IPV6Util.encloseHost(address.toString());
    }

    @Override
    public final boolean isDirectDeliver() {
        return this.directDeliver;
    }

    @Override
    public final ActiveMQPrincipal getDefaultActiveMQPrincipal() {
        return null;
    }

    @Override
    public final TransportConfiguration getConnectorConfig() {
        if (this.configuration != null) {
            return new TransportConfiguration(NettyConnectorFactory.class.getName(), this.configuration);
        }
        return null;
    }

    @Override
    public final boolean isUsingProtocolHandling() {
        return true;
    }

    @Override
    public boolean isSameTarget(TransportConfiguration ... configs) {
        boolean result = false;
        for (TransportConfiguration cfg : configs) {
            if (cfg == null || !NettyConnectorFactory.class.getName().equals(cfg.getFactoryClassName()) || !this.configuration.get("port").equals(cfg.getParams().get("port"))) continue;
            Object hostParam = this.configuration.get("host");
            if (hostParam != null) {
                if (hostParam.equals(cfg.getParams().get("host"))) {
                    result = true;
                    break;
                }
                if (!this.isLocalhost((String)this.configuration.get("host")) || !this.isLocalhost((String)cfg.getParams().get("host"))) continue;
                result = true;
                break;
            }
            if (cfg.getParams().get("host") != null) continue;
            result = true;
            break;
        }
        return result;
    }

    private boolean isLocalhost(String hostname) {
        return "127.0.0.1".equals(hostname) || "localhost".equals(hostname);
    }

    public final String toString() {
        return super.toString() + "[ID=" + this.getID() + ", local= " + this.channel.localAddress() + ", remote=" + this.channel.remoteAddress() + "]";
    }

    private void closeChannel(Channel channel, boolean inEventLoop) {
        this.checkFlushBatchBuffer();
        ChannelFuture closeFuture = channel.close();
        if (!inEventLoop && !closeFuture.awaitUninterruptibly(10000L)) {
            ActiveMQClientLogger.LOGGER.timeoutClosingNettyChannel();
        }
    }
}

