/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.io.buffer;

import io.netty.buffer.Unpooled;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import org.apache.activemq.artemis.api.core.ActiveMQBuffer;
import org.apache.activemq.artemis.api.core.ActiveMQInterruptedException;
import org.apache.activemq.artemis.core.buffers.impl.ChannelBufferWrapper;
import org.apache.activemq.artemis.core.io.IOCallback;
import org.apache.activemq.artemis.core.io.buffer.TimedBufferObserver;
import org.apache.activemq.artemis.core.journal.EncodingSupport;
import org.apache.activemq.artemis.journal.ActiveMQJournalLogger;

public final class TimedBuffer {
    private TimedBufferObserver bufferObserver;
    private final Semaphore spinLimiter = new Semaphore(1);
    private CheckTimer timerRunnable = new CheckTimer();
    private final int bufferSize;
    private final ActiveMQBuffer buffer;
    private int bufferLimit = 0;
    private List<IOCallback> callbacks;
    private final int timeout;
    private final AtomicLong pendingSyncs = new AtomicLong();
    private Thread timerThread;
    private volatile boolean started;
    private boolean delayFlush;
    private final boolean logRates;
    private long bytesFlushed = 0L;
    private final AtomicLong flushesDone = new AtomicLong(0L);
    private Timer logRatesTimer;
    private TimerTask logRatesTimerTask;
    private boolean spinning = false;

    public TimedBuffer(int size, int timeout, boolean logRates) {
        this.bufferSize = size;
        this.logRates = logRates;
        if (logRates) {
            this.logRatesTimer = new Timer(true);
        }
        this.buffer = new ChannelBufferWrapper(Unpooled.directBuffer((int)size, (int)size));
        this.buffer.clear();
        this.bufferLimit = 0;
        this.callbacks = null;
        this.timeout = timeout;
    }

    public synchronized void start() {
        if (this.started) {
            return;
        }
        try {
            this.spinLimiter.acquire();
        }
        catch (InterruptedException e) {
            throw new ActiveMQInterruptedException((Throwable)e);
        }
        this.timerRunnable = new CheckTimer();
        this.timerThread = new Thread((Runnable)this.timerRunnable, "activemq-buffer-timeout");
        this.timerThread.start();
        if (this.logRates) {
            this.logRatesTimerTask = new LogRatesTimerTask();
            this.logRatesTimer.scheduleAtFixedRate(this.logRatesTimerTask, 2000L, 2000L);
        }
        this.started = true;
    }

    public void stop() {
        if (!this.started) {
            return;
        }
        this.flush();
        this.bufferObserver = null;
        this.timerRunnable.close();
        this.spinLimiter.release();
        if (this.logRates) {
            this.logRatesTimerTask.cancel();
        }
        while (this.timerThread.isAlive()) {
            try {
                this.timerThread.join();
            }
            catch (InterruptedException e) {
                throw new ActiveMQInterruptedException((Throwable)e);
            }
        }
        this.started = false;
    }

    public synchronized void setObserver(TimedBufferObserver observer) {
        if (this.bufferObserver != null) {
            this.flush();
        }
        this.bufferObserver = observer;
    }

    public synchronized boolean checkSize(int sizeChecked) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        if (sizeChecked > this.bufferSize) {
            throw new IllegalStateException("Can't write records bigger than the bufferSize(" + this.bufferSize + ") on the journal");
        }
        if (this.bufferLimit == 0 || this.buffer.writerIndex() + sizeChecked > this.bufferLimit) {
            this.flush();
            this.delayFlush = true;
            int remainingInFile = this.bufferObserver.getRemainingBytes();
            if (sizeChecked > remainingInFile) {
                return false;
            }
            this.bufferLimit = Math.min(remainingInFile, this.bufferSize);
            return true;
        }
        this.delayFlush = true;
        return true;
    }

    public synchronized void addBytes(ActiveMQBuffer bytes, boolean sync, IOCallback callback) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        this.delayFlush = false;
        int readableBytes = bytes.readableBytes();
        int writerIndex = this.buffer.writerIndex();
        this.buffer.setBytes(writerIndex, bytes, bytes.readerIndex(), readableBytes);
        this.buffer.writerIndex(writerIndex + readableBytes);
        if (this.callbacks == null) {
            this.callbacks = new ArrayList<IOCallback>();
        }
        this.callbacks.add(callback);
        if (sync) {
            long currentPendingSyncs = this.pendingSyncs.get();
            this.pendingSyncs.lazySet(currentPendingSyncs + 1L);
            this.startSpin();
        }
    }

    public synchronized void addBytes(EncodingSupport bytes, boolean sync, IOCallback callback) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        this.delayFlush = false;
        bytes.encode(this.buffer);
        if (this.callbacks == null) {
            this.callbacks = new ArrayList<IOCallback>();
        }
        this.callbacks.add(callback);
        if (sync) {
            long currentPendingSyncs = this.pendingSyncs.get();
            this.pendingSyncs.lazySet(currentPendingSyncs + 1L);
            this.startSpin();
        }
    }

    public void flush() {
        this.flush(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flush(boolean force) {
        TimedBuffer timedBuffer = this;
        synchronized (timedBuffer) {
            if (!this.started) {
                throw new IllegalStateException("TimedBuffer is not started");
            }
            if ((force || !this.delayFlush) && this.buffer.writerIndex() > 0) {
                int pos = this.buffer.writerIndex();
                ByteBuffer bufferToFlush = this.bufferObserver.newBuffer(this.bufferSize, pos);
                bufferToFlush.limit(pos);
                this.buffer.getBytes(0, bufferToFlush);
                List<IOCallback> ioCallbacks = this.callbacks == null ? Collections.emptyList() : this.callbacks;
                this.bufferObserver.flushBuffer(bufferToFlush, this.pendingSyncs.get() > 0L, ioCallbacks);
                this.stopSpin();
                this.pendingSyncs.lazySet(0L);
                this.callbacks = null;
                this.buffer.clear();
                this.bufferLimit = 0;
                if (this.logRates) {
                    this.logFlushed(pos);
                }
            }
        }
    }

    private void logFlushed(int bytes) {
        this.bytesFlushed += (long)bytes;
        long currentFlushesDone = this.flushesDone.get();
        this.flushesDone.lazySet(currentFlushesDone + 1L);
    }

    private static int wait(int waitTimes, Semaphore spinLimiter) {
        if (waitTimes < 10) {
            Thread.yield();
            ++waitTimes;
        } else if (waitTimes < 20) {
            LockSupport.parkNanos(1L);
            ++waitTimes;
        } else if (waitTimes < 50) {
            LockSupport.parkNanos(10L);
            ++waitTimes;
        } else if (waitTimes < 100) {
            LockSupport.parkNanos(100L);
            ++waitTimes;
        } else if (waitTimes < 1000) {
            LockSupport.parkNanos(1000L);
            ++waitTimes;
        } else {
            LockSupport.parkNanos(100000L);
            try {
                spinLimiter.acquire();
                spinLimiter.release();
            }
            catch (InterruptedException e) {
                throw new ActiveMQInterruptedException((Throwable)e);
            }
        }
        return waitTimes;
    }

    protected void stopSpin() {
        if (this.spinning) {
            try {
                this.spinLimiter.acquire();
            }
            catch (InterruptedException e) {
                throw new ActiveMQInterruptedException((Throwable)e);
            }
            this.spinning = false;
        }
    }

    protected void startSpin() {
        if (!this.spinning) {
            this.spinLimiter.release();
            this.spinning = true;
        }
    }

    private class CheckTimer
    implements Runnable {
        private volatile boolean closed = false;

        private CheckTimer() {
        }

        @Override
        public void run() {
            int waitTimes = 0;
            long lastFlushTime = 0L;
            long estimatedOptimalBatch = Runtime.getRuntime().availableProcessors();
            Semaphore spinLimiter = TimedBuffer.this.spinLimiter;
            long timeout = TimedBuffer.this.timeout;
            while (!this.closed) {
                boolean flushed = false;
                long currentPendingSyncs = TimedBuffer.this.pendingSyncs.get();
                if (currentPendingSyncs > 0L && TimedBuffer.this.bufferObserver != null) {
                    boolean checkpoint;
                    boolean bl = checkpoint = System.nanoTime() > lastFlushTime + timeout;
                    if (checkpoint || currentPendingSyncs >= estimatedOptimalBatch) {
                        TimedBuffer.this.flush();
                        estimatedOptimalBatch = checkpoint ? currentPendingSyncs : Math.max(estimatedOptimalBatch, currentPendingSyncs);
                        lastFlushTime = System.nanoTime();
                        flushed = true;
                    }
                }
                if (flushed) {
                    waitTimes = 0;
                    continue;
                }
                waitTimes = TimedBuffer.wait(waitTimes, spinLimiter);
            }
        }

        public void close() {
            this.closed = true;
        }
    }

    private class LogRatesTimerTask
    extends TimerTask {
        private boolean closed;
        private long lastExecution;
        private long lastBytesFlushed;
        private long lastFlushesDone;

        private LogRatesTimerTask() {
        }

        @Override
        public synchronized void run() {
            if (!this.closed) {
                long now = System.currentTimeMillis();
                long flushesDone = TimedBuffer.this.flushesDone.get();
                long bytesFlushed = TimedBuffer.this.bytesFlushed;
                if (this.lastExecution != 0L) {
                    double rate = 1000.0 * (double)(bytesFlushed - this.lastBytesFlushed) / (double)(now - this.lastExecution);
                    ActiveMQJournalLogger.LOGGER.writeRate(rate, (long)(rate / 1048576.0));
                    double flushRate = 1000.0 * (double)(flushesDone - this.lastFlushesDone) / (double)(now - this.lastExecution);
                    ActiveMQJournalLogger.LOGGER.flushRate(flushRate);
                }
                this.lastExecution = now;
                this.lastBytesFlushed = bytesFlushed;
                this.lastFlushesDone = flushesDone;
            }
        }

        @Override
        public synchronized boolean cancel() {
            this.closed = true;
            return super.cancel();
        }
    }
}

