/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.io.aio;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.nio.ByteBuffer;
import java.util.PriorityQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.activemq.artemis.api.core.ActiveMQException;
import org.apache.activemq.artemis.api.core.ActiveMQNativeIOError;
import org.apache.activemq.artemis.core.io.AbstractSequentialFile;
import org.apache.activemq.artemis.core.io.DummyCallback;
import org.apache.activemq.artemis.core.io.IOCallback;
import org.apache.activemq.artemis.core.io.SequentialFile;
import org.apache.activemq.artemis.core.io.aio.AIOSequentialFileFactory;
import org.apache.activemq.artemis.core.journal.impl.SimpleWaitIOCallback;
import org.apache.activemq.artemis.jlibaio.LibaioFile;
import org.apache.activemq.artemis.jlibaio.SubmitInfo;
import org.apache.activemq.artemis.journal.ActiveMQJournalLogger;
import org.apache.activemq.artemis.utils.ReusableLatch;
import org.jboss.logging.Logger;

public class AIOSequentialFile
extends AbstractSequentialFile {
    private static final Logger logger = Logger.getLogger(AIOSequentialFileFactory.class);
    private boolean opened = false;
    private LibaioFile aioFile;
    private final AIOSequentialFileFactory aioFactory;
    private final ReusableLatch pendingCallbacks = new ReusableLatch();
    private final AtomicLong nextWritingSequence = new AtomicLong(0L);
    final PriorityQueue<AIOSequentialFileFactory.AIOSequentialCallback> pendingCallbackList = new PriorityQueue();
    private long nextReadSequence = 0L;

    public AIOSequentialFile(AIOSequentialFileFactory factory, int bufferSize, long bufferTimeoutMilliseconds, File directory, String fileName, Executor writerExecutor) {
        super(directory, fileName, factory, writerExecutor);
        this.aioFactory = factory;
    }

    @Override
    public boolean isOpen() {
        return this.opened;
    }

    @Override
    public int calculateBlockStart(int position) {
        int alignment = this.factory.getAlignment();
        int pos = (position / alignment + (position % alignment != 0 ? 1 : 0)) * alignment;
        return pos;
    }

    @Override
    public SequentialFile cloneFile() {
        return new AIOSequentialFile(this.aioFactory, -1, -1L, this.getFile().getParentFile(), this.getFile().getName(), null);
    }

    @Override
    public void close() throws IOException, InterruptedException, ActiveMQException {
        this.close(true);
    }

    @Override
    public synchronized void close(boolean waitSync) throws IOException, InterruptedException, ActiveMQException {
        if (!this.opened) {
            return;
        }
        super.close();
        if (waitSync) {
            String fileName = this.getFileName();
            try {
                int waitCount = 0;
                while (!this.pendingCallbacks.await(10L, TimeUnit.SECONDS)) {
                    if (++waitCount == 1) {
                        ThreadInfo[] threads;
                        for (ThreadInfo threadInfo : threads = ManagementFactory.getThreadMXBean().dumpAllThreads(true, true)) {
                            ActiveMQJournalLogger.LOGGER.warn(threadInfo.toString());
                        }
                        this.factory.onIOError(new IOException("Timeout on close"), "Timeout on close", this);
                    }
                    ActiveMQJournalLogger.LOGGER.warn("waiting pending callbacks on " + fileName + " from " + waitCount * 10 + " seconds!");
                }
            }
            catch (InterruptedException e) {
                ActiveMQJournalLogger.LOGGER.warn("interrupted while waiting pending callbacks on " + fileName, e);
                throw e;
            }
            finally {
                this.opened = false;
                this.timedBuffer = null;
                this.aioFile.close();
                this.aioFile = null;
            }
        }
    }

    @Override
    public synchronized void fill(int size) throws Exception {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Filling file: " + this.getFileName()));
        }
        this.checkOpened();
        this.aioFile.fill((long)size);
        this.fileSize = this.aioFile.getSize();
    }

    @Override
    public void open() throws Exception {
        this.open(this.aioFactory.getMaxIO(), true);
    }

    @Override
    public synchronized void open(int maxIO, boolean useExecutor) throws ActiveMQException {
        this.opened = true;
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Opening file: " + this.getFileName()));
        }
        try {
            this.aioFile = this.aioFactory.libaioContext.openFile(this.getFile(), this.factory.isDatasync());
        }
        catch (IOException e) {
            logger.error((Object)("Error opening file: " + this.getFileName()));
            this.factory.onIOError(e, e.getMessage(), this);
            throw new ActiveMQNativeIOError(e.getMessage(), (Throwable)e);
        }
        this.position.set(0L);
        this.fileSize = this.aioFile.getSize();
    }

    @Override
    public int read(ByteBuffer bytes, IOCallback callback) throws ActiveMQException {
        this.checkOpened();
        int bytesToRead = bytes.limit();
        long positionToRead = this.position.getAndAdd(bytesToRead);
        bytes.rewind();
        try {
            this.aioFile.read(positionToRead, bytesToRead, bytes, (SubmitInfo)this.getCallback(callback, null));
        }
        catch (IOException e) {
            logger.error((Object)("IOError reading file: " + this.getFileName()), (Throwable)e);
            this.factory.onIOError(e, e.getMessage(), this);
            throw new ActiveMQNativeIOError(e.getMessage(), (Throwable)e);
        }
        return bytesToRead;
    }

    @Override
    public int read(ByteBuffer bytes) throws Exception {
        SimpleWaitIOCallback waitCompletion = new SimpleWaitIOCallback();
        int bytesRead = this.read(bytes, waitCompletion);
        waitCompletion.waitCompletion();
        return bytesRead;
    }

    @Override
    public void writeDirect(ByteBuffer bytes, boolean sync) throws Exception {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Write Direct, Sync: " + sync + " File: " + this.getFileName()));
        }
        if (sync) {
            SimpleWaitIOCallback completion = new SimpleWaitIOCallback();
            this.writeDirect(bytes, true, completion);
            completion.waitCompletion();
        } else {
            this.writeDirect(bytes, false, DummyCallback.getInstance());
        }
    }

    @Override
    public void writeDirect(ByteBuffer bytes, boolean sync, IOCallback callback) {
        try {
            this.checkOpened();
        }
        catch (Exception e) {
            ActiveMQJournalLogger.LOGGER.warn(e.getMessage(), e);
            callback.onError(-1, e.getMessage());
            return;
        }
        int bytesToWrite = this.factory.calculateBlockSize(bytes.limit());
        long positionToWrite = this.position.getAndAdd(bytesToWrite);
        AIOSequentialFileFactory.AIOSequentialCallback runnableCallback = this.getCallback(callback, bytes);
        runnableCallback.initWrite(positionToWrite, bytesToWrite);
        runnableCallback.run();
    }

    AIOSequentialFileFactory.AIOSequentialCallback getCallback(IOCallback originalCallback, ByteBuffer buffer) {
        AIOSequentialFileFactory.AIOSequentialCallback callback = this.aioFactory.getCallback();
        callback.init(this.nextWritingSequence.getAndIncrement(), originalCallback, this.aioFile, this, buffer);
        this.pendingCallbacks.countUp();
        return callback;
    }

    void done(AIOSequentialFileFactory.AIOSequentialCallback callback) {
        if (callback.writeSequence == -1L) {
            callback.sequentialDone();
            this.pendingCallbacks.countDown();
        }
        if (callback.writeSequence == this.nextReadSequence) {
            ++this.nextReadSequence;
            callback.sequentialDone();
            this.pendingCallbacks.countDown();
            this.flushCallbacks();
        } else {
            this.pendingCallbackList.add(callback);
        }
    }

    private void flushCallbacks() {
        while (!this.pendingCallbackList.isEmpty() && this.pendingCallbackList.peek().writeSequence == this.nextReadSequence) {
            AIOSequentialFileFactory.AIOSequentialCallback callback = this.pendingCallbackList.poll();
            callback.sequentialDone();
            ++this.nextReadSequence;
            this.pendingCallbacks.countDown();
        }
    }

    @Override
    public void sync() {
        throw new UnsupportedOperationException("This method is not supported on AIO");
    }

    @Override
    public long size() throws Exception {
        if (this.aioFile == null) {
            return this.getFile().length();
        }
        return this.aioFile.getSize();
    }

    public String toString() {
        return "AIOSequentialFile:" + this.getFile().getAbsolutePath();
    }

    @Override
    protected ByteBuffer newBuffer(int size, int limit) {
        size = this.factory.calculateBlockSize(size);
        limit = this.factory.calculateBlockSize(limit);
        ByteBuffer buffer = this.factory.newBuffer(size);
        buffer.limit(limit);
        return buffer;
    }

    private void checkOpened() {
        if (this.aioFile == null || !this.opened) {
            throw new NullPointerException("File not opened, file=null");
        }
    }
}

