/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.activemq.artemis.core.management.impl;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanOperationInfo;

import java.util.Map;

import org.apache.activemq.artemis.api.core.JsonUtil;
import org.apache.activemq.artemis.api.core.management.DivertControl;
import org.apache.activemq.artemis.api.core.management.ResourceNames;
import org.apache.activemq.artemis.core.config.DivertConfiguration;
import org.apache.activemq.artemis.core.persistence.StorageManager;
import org.apache.activemq.artemis.core.server.Divert;
import org.apache.activemq.artemis.logs.AuditLogger;

public class DivertControlImpl extends AbstractControl implements DivertControl {

   // Constants -----------------------------------------------------

   // Attributes ----------------------------------------------------

   private final Divert divert;

   private final DivertConfiguration configuration;

   private final String internalNamingPrefix;

   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   // DivertControlMBean implementation ---------------------------

   public DivertControlImpl(final Divert divert,
                            final StorageManager storageManager,
                            final DivertConfiguration configuration,
                            final String internalNamingPrefix) throws Exception {
      super(DivertControl.class, storageManager);
      this.divert = divert;
      this.configuration = configuration;
      this.internalNamingPrefix = internalNamingPrefix;
   }

   @Override
   public String getAddress() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getAddress(this.divert);
      }
      clearIO();
      try {
         return configuration.getAddress();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getFilter() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getFilter(this.divert);
      }
      clearIO();
      try {
         return configuration.getFilterString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getForwardingAddress() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getForwardingAddress(this.divert);
      }
      clearIO();
      try {
         return configuration.getForwardingAddress();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getRoutingName() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getRoutingName(this.divert);
      }
      clearIO();
      try {
         return divert.getRoutingName().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getTransformerClassName() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTransformerClassName(this.divert);
      }
      clearIO();
      try {
         return configuration.getTransformerConfiguration() == null ? null : configuration.getTransformerConfiguration().getClassName();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getTransformerPropertiesAsJSON() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTransformerPropertiesAsJSON(this.divert);
      }
      return JsonUtil.toJsonObject(getTransformerProperties()).toString();
   }

   @Override
   public Map<String, String> getTransformerProperties() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTransformerProperties(this.divert);
      }
      clearIO();
      try {
         return configuration.getTransformerConfiguration() == null ? null : configuration.getTransformerConfiguration().getProperties();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getRoutingType() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getRoutingType(this.divert);
      }
      clearIO();
      try {
         return configuration.getRoutingType().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getUniqueName() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getUniqueName(this.divert);
      }
      clearIO();
      try {
         return divert.getUniqueName().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isExclusive() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isExclusive(this.divert);
      }
      clearIO();
      try {
         return divert.isExclusive();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isRetroactiveResource() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isRetroactiveResource(this.divert);
      }
      return ResourceNames.isRetroactiveResource(internalNamingPrefix, divert.getUniqueName());
   }

   @Override
   protected MBeanOperationInfo[] fillMBeanOperationInfo() {
      return MBeanInfoHelper.getMBeanOperationsInfo(DivertControl.class);
   }

   @Override
   protected MBeanAttributeInfo[] fillMBeanAttributeInfo() {
      return MBeanInfoHelper.getMBeanAttributesInfo(DivertControl.class);
   }

   // Public --------------------------------------------------------

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------

}
