/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.activemq.artemis.core.management.impl;

import javax.json.JsonArray;
import javax.json.JsonArrayBuilder;
import javax.json.JsonObject;
import javax.json.JsonObjectBuilder;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.Notification;
import javax.management.NotificationBroadcasterSupport;
import javax.management.NotificationEmitter;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.transaction.xa.Xid;
import java.net.URL;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import org.apache.activemq.artemis.api.config.ActiveMQDefaultConfiguration;
import org.apache.activemq.artemis.api.core.ActiveMQAddressDoesNotExistException;
import org.apache.activemq.artemis.api.core.ActiveMQException;
import org.apache.activemq.artemis.api.core.JsonUtil;
import org.apache.activemq.artemis.api.core.QueueConfiguration;
import org.apache.activemq.artemis.api.core.RoutingType;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.api.core.TransportConfiguration;
import org.apache.activemq.artemis.api.core.client.ActiveMQClient;
import org.apache.activemq.artemis.api.core.management.ActiveMQServerControl;
import org.apache.activemq.artemis.api.core.management.AddressControl;
import org.apache.activemq.artemis.api.core.management.BridgeControl;
import org.apache.activemq.artemis.api.core.management.CoreNotificationType;
import org.apache.activemq.artemis.api.core.management.DivertControl;
import org.apache.activemq.artemis.api.core.management.ManagementHelper;
import org.apache.activemq.artemis.api.core.management.Parameter;
import org.apache.activemq.artemis.api.core.management.QueueControl;
import org.apache.activemq.artemis.core.client.impl.Topology;
import org.apache.activemq.artemis.core.client.impl.TopologyMemberImpl;
import org.apache.activemq.artemis.core.config.BridgeConfiguration;
import org.apache.activemq.artemis.core.config.Configuration;
import org.apache.activemq.artemis.core.config.ConnectorServiceConfiguration;
import org.apache.activemq.artemis.core.config.DivertConfiguration;
import org.apache.activemq.artemis.core.config.TransformerConfiguration;
import org.apache.activemq.artemis.core.filter.Filter;
import org.apache.activemq.artemis.core.management.impl.view.AddressView;
import org.apache.activemq.artemis.core.management.impl.view.ConnectionView;
import org.apache.activemq.artemis.core.management.impl.view.ConsumerView;
import org.apache.activemq.artemis.core.management.impl.view.ProducerView;
import org.apache.activemq.artemis.core.management.impl.view.QueueView;
import org.apache.activemq.artemis.core.management.impl.view.SessionView;
import org.apache.activemq.artemis.core.messagecounter.MessageCounterManager;
import org.apache.activemq.artemis.core.messagecounter.impl.MessageCounterManagerImpl;
import org.apache.activemq.artemis.core.persistence.StorageManager;
import org.apache.activemq.artemis.core.persistence.config.PersistedAddressSetting;
import org.apache.activemq.artemis.core.persistence.config.PersistedRoles;
import org.apache.activemq.artemis.core.postoffice.Binding;
import org.apache.activemq.artemis.core.postoffice.Bindings;
import org.apache.activemq.artemis.core.postoffice.DuplicateIDCache;
import org.apache.activemq.artemis.core.postoffice.PostOffice;
import org.apache.activemq.artemis.core.postoffice.impl.LocalQueueBinding;
import org.apache.activemq.artemis.core.remoting.server.RemotingService;
import org.apache.activemq.artemis.core.security.CheckType;
import org.apache.activemq.artemis.core.security.Role;
import org.apache.activemq.artemis.core.server.ActiveMQMessageBundle;
import org.apache.activemq.artemis.core.server.ActiveMQServer;
import org.apache.activemq.artemis.core.server.ActiveMQServerLogger;
import org.apache.activemq.artemis.core.server.ConnectorServiceFactory;
import org.apache.activemq.artemis.core.server.Consumer;
import org.apache.activemq.artemis.core.server.ComponentConfigurationRoutingType;
import org.apache.activemq.artemis.core.server.Divert;
import org.apache.activemq.artemis.core.server.JournalType;
import org.apache.activemq.artemis.core.server.Queue;
import org.apache.activemq.artemis.core.server.ServerConsumer;
import org.apache.activemq.artemis.core.server.ServerProducer;
import org.apache.activemq.artemis.core.server.ServerSession;
import org.apache.activemq.artemis.core.server.cluster.ClusterConnection;
import org.apache.activemq.artemis.core.server.cluster.ClusterManager;
import org.apache.activemq.artemis.core.server.cluster.ha.HAPolicy;
import org.apache.activemq.artemis.core.server.cluster.ha.LiveOnlyPolicy;
import org.apache.activemq.artemis.core.server.cluster.ha.ScaleDownPolicy;
import org.apache.activemq.artemis.core.server.cluster.ha.SharedStoreSlavePolicy;
import org.apache.activemq.artemis.core.server.group.GroupingHandler;
import org.apache.activemq.artemis.core.server.impl.Activation;
import org.apache.activemq.artemis.core.server.impl.AddressInfo;
import org.apache.activemq.artemis.core.server.impl.SharedNothingLiveActivation;
import org.apache.activemq.artemis.core.settings.impl.AddressFullMessagePolicy;
import org.apache.activemq.artemis.core.settings.impl.AddressSettings;
import org.apache.activemq.artemis.core.settings.impl.DeletionPolicy;
import org.apache.activemq.artemis.core.settings.impl.SlowConsumerPolicy;
import org.apache.activemq.artemis.core.transaction.ResourceManager;
import org.apache.activemq.artemis.core.transaction.Transaction;
import org.apache.activemq.artemis.core.transaction.TransactionDetail;
import org.apache.activemq.artemis.core.transaction.TransactionDetailFactory;
import org.apache.activemq.artemis.core.transaction.impl.CoreTransactionDetail;
import org.apache.activemq.artemis.core.transaction.impl.XidImpl;
import org.apache.activemq.artemis.logs.AuditLogger;
import org.apache.activemq.artemis.spi.core.protocol.RemotingConnection;
import org.apache.activemq.artemis.spi.core.security.jaas.PropertiesLoginModuleConfigurator;
import org.apache.activemq.artemis.utils.JsonLoader;
import org.apache.activemq.artemis.utils.ListUtil;
import org.apache.activemq.artemis.utils.PasswordMaskingUtil;
import org.apache.activemq.artemis.utils.SecurityFormatter;
import org.apache.activemq.artemis.utils.collections.TypedProperties;
import org.jboss.logging.Logger;

public class ActiveMQServerControlImpl extends AbstractControl implements ActiveMQServerControl, NotificationEmitter, org.apache.activemq.artemis.core.server.management.NotificationListener {
   // Constants -----------------------------------------------------
   private static final Logger logger = Logger.getLogger(ActiveMQServerControlImpl.class);

   // Attributes ----------------------------------------------------

   private final PostOffice postOffice;

   private final Configuration configuration;

   private final ResourceManager resourceManager;

   private final RemotingService remotingService;

   private final ActiveMQServer server;

   private final MessageCounterManager messageCounterManager;

   private final NotificationBroadcasterSupport broadcaster;

   private final AtomicLong notifSeq = new AtomicLong(0);
   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   public ActiveMQServerControlImpl(final PostOffice postOffice,
                                    final Configuration configuration,
                                    final ResourceManager resourceManager,
                                    final RemotingService remotingService,
                                    final ActiveMQServer messagingServer,
                                    final MessageCounterManager messageCounterManager,
                                    final StorageManager storageManager,
                                    final NotificationBroadcasterSupport broadcaster) throws Exception {
      super(ActiveMQServerControl.class, storageManager);
      this.postOffice = postOffice;
      this.configuration = configuration;
      this.resourceManager = resourceManager;
      this.remotingService = remotingService;
      server = messagingServer;
      this.messageCounterManager = messageCounterManager;
      this.broadcaster = broadcaster;
      server.getManagementService().addNotificationListener(this);
   }

   // ActiveMQServerControlMBean implementation --------------------

   @Override
   public boolean isStarted() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isStarted(this.server);
      }
      clearIO();
      try {
         return server.isStarted();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getVersion() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getVersion(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getVersion().getFullVersion();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isActive() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getVersion(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.isActive();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isBackup() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isBackup(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getHAPolicy().isBackup();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isSharedStore() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isSharedStore(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getHAPolicy().isSharedStore();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getBindingsDirectory() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getBindingsDirectory(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getBindingsDirectory();
      } finally {
         blockOnIO();
      }
   }

   public String[] getInterceptorClassNames() {
      checkStarted();

      clearIO();
      try {
         return configuration.getIncomingInterceptorClassNames().toArray(new String[configuration.getIncomingInterceptorClassNames().size()]);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getIncomingInterceptorClassNames() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getIncomingInterceptorClassNames(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getIncomingInterceptorClassNames().toArray(new String[configuration.getIncomingInterceptorClassNames().size()]);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getOutgoingInterceptorClassNames() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getOutgoingInterceptorClassNames(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getOutgoingInterceptorClassNames().toArray(new String[configuration.getOutgoingInterceptorClassNames().size()]);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalBufferSize() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalBufferSize(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalType() == JournalType.ASYNCIO ? configuration.getJournalBufferSize_AIO() : configuration.getJournalBufferSize_NIO();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalBufferTimeout() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalBufferTimeout(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalType() == JournalType.ASYNCIO ? configuration.getJournalBufferTimeout_AIO() : configuration.getJournalBufferTimeout_NIO();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void setFailoverOnServerShutdown(boolean failoverOnServerShutdown) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.setFailoverOnServerShutdown(this.server, failoverOnServerShutdown);
      }
      checkStarted();

      clearIO();
      try {
         HAPolicy haPolicy = server.getHAPolicy();
         if (haPolicy instanceof SharedStoreSlavePolicy) {
            ((SharedStoreSlavePolicy) haPolicy).setFailoverOnServerShutdown(failoverOnServerShutdown);
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isFailoverOnServerShutdown() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isFailoverOnServerShutdown(this.server);
      }
      checkStarted();

      clearIO();
      try {
         HAPolicy haPolicy = server.getHAPolicy();
         if (haPolicy instanceof SharedStoreSlavePolicy) {
            return ((SharedStoreSlavePolicy) haPolicy).isFailoverOnServerShutdown();
         } else {
            return false;
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalMaxIO() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalMaxIO(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalType() == JournalType.ASYNCIO ? configuration.getJournalMaxIO_AIO() : configuration.getJournalMaxIO_NIO();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getJournalDirectory() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalDirectory(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalDirectory();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalFileSize() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalFileSize(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalFileSize();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalMinFiles() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalMinFiles(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalMinFiles();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalCompactMinFiles() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalCompactMinFiles(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalCompactMinFiles();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getJournalCompactPercentage() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalCompactPercentage(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalCompactPercentage();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isPersistenceEnabled() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isPersistenceEnabled(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isPersistenceEnabled();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getJournalType() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getJournalType(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getJournalType().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getPagingDirectory() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getPagingDirectory(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getPagingDirectory();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getScheduledThreadPoolMaxSize() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getScheduledThreadPoolMaxSize(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getScheduledThreadPoolMaxSize();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getThreadPoolMaxSize() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getThreadPoolMaxSize(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getThreadPoolMaxSize();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getSecurityInvalidationInterval() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getSecurityInvalidationInterval(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getSecurityInvalidationInterval();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isClustered() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isClustered(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isClustered();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isCreateBindingsDir() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isCreateBindingsDir(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isCreateBindingsDir();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isCreateJournalDir() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isCreateJournalDir(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isCreateJournalDir();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isJournalSyncNonTransactional() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isJournalSyncNonTransactional(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isJournalSyncNonTransactional();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isJournalSyncTransactional() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isJournalSyncTransactional(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isJournalSyncTransactional();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isSecurityEnabled() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isSecurityEnabled(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isSecurityEnabled();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isAsyncConnectionExecutionEnabled() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isAsyncConnectionExecutionEnabled(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isAsyncConnectionExecutionEnabled();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getDiskScanPeriod() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getDiskScanPeriod(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getDiskScanPeriod();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getMaxDiskUsage() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getMaxDiskUsage(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getMaxDiskUsage();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getGlobalMaxSize() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getGlobalMaxSize(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.getGlobalMaxSize();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getAddressMemoryUsage() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getAddressMemoryUsage(this.server);
      }
      checkStarted();
      clearIO();
      try {
         //this should not happen but if it does, return -1 to highlight it is not working
         if (server.getPagingManager() == null) {
            return -1L;
         }
         return server.getPagingManager().getGlobalSize();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public double getDiskStoreUsage() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getDiskStoreUsage(this.server);
      }
      checkStarted();
      clearIO();
      try {
         return server.getDiskStoreUsage();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getAddressMemoryUsagePercentage() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getAddressMemoryUsagePercentage(this.server);
      }
      long globalMaxSize = getGlobalMaxSize();
      // no max size set implies 0% used
      if (globalMaxSize <= 0) {
         return 0;
      }

      long memoryUsed = getAddressMemoryUsage();
      if (memoryUsed <= 0) {
         return 0;
      }
      double result = (100D * memoryUsed) / globalMaxSize;
      return (int) result;
   }

   @Override
   public boolean freezeReplication() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.freezeReplication(this.server);
      }
      Activation activation = server.getActivation();
      if (activation instanceof SharedNothingLiveActivation) {
         SharedNothingLiveActivation liveActivation = (SharedNothingLiveActivation) activation;
         liveActivation.freezeReplication();
         return true;
      }
      return false;
   }

   private enum AddressInfoTextFormatter {
      Long {
         @Override
         public StringBuilder format(AddressInfo addressInfo, StringBuilder output) {
            output.append("Address [name=").append(addressInfo.getName());
            output.append(", routingTypes={");
            final EnumSet<RoutingType> routingTypes = addressInfo.getRoutingTypes();
            if (!routingTypes.isEmpty()) {
               for (RoutingType routingType : routingTypes) {
                  output.append(routingType).append(',');
               }
               // delete hanging comma
               output.deleteCharAt(output.length() - 1);
            }
            output.append('}');
            output.append(", autoCreated=").append(addressInfo.isAutoCreated());
            output.append(']');
            return output;
         }
      };

      public abstract StringBuilder format(AddressInfo addressInfo, StringBuilder output);
   }

   public enum QueueTextFormatter {
      Long {
         @Override
         StringBuilder format(Queue queue, StringBuilder output) {
            output.append("Queue [name=").append(queue.getName());
            output.append(", address=").append(queue.getAddress());
            output.append(", routingType=").append(queue.getRoutingType());
            final Filter filter = queue.getFilter();
            if (filter != null) {
               output.append(", filter=").append(filter.getFilterString());
            }
            output.append(", durable=").append(queue.isDurable());
            final int maxConsumers = queue.getMaxConsumers();
            if (maxConsumers != Queue.MAX_CONSUMERS_UNLIMITED) {
               output.append(", maxConsumers=").append(queue.getMaxConsumers());
            }
            output.append(", purgeOnNoConsumers=").append(queue.isPurgeOnNoConsumers());
            output.append(", autoCreateAddress=").append(queue.isAutoCreated());
            output.append(", exclusive=").append(queue.isExclusive());
            output.append(", lastValue=").append(queue.isLastValue());
            output.append(", lastValueKey=").append(queue.getLastValueKey());
            output.append(", nonDestructive=").append(queue.isNonDestructive());
            output.append(", consumersBeforeDispatch=").append(queue.getConsumersBeforeDispatch());
            output.append(", delayBeforeDispatch=").append(queue.getDelayBeforeDispatch());
            output.append(", autoCreateAddress=").append(queue.isAutoCreated());

            output.append(']');
            return output;
         }
      };

      abstract StringBuilder format(Queue queue, StringBuilder output);
   }

   @Override
   public String createAddress(String name, String routingTypes) throws Exception {

      checkStarted();

      clearIO();
      try {
         EnumSet<RoutingType> set = EnumSet.noneOf(RoutingType.class);
         for (String routingType : ListUtil.toList(routingTypes)) {
            set.add(RoutingType.valueOf(routingType));
         }
         final AddressInfo addressInfo = new AddressInfo(new SimpleString(name), set);
         if (server.addAddressInfo(addressInfo)) {
            String result = AddressInfoTextFormatter.Long.format(addressInfo, new StringBuilder()).toString();
            if (AuditLogger.isResourceLoggingEnabled()) {
               AuditLogger.createAddressSuccess(name, routingTypes);
            }
            return result;
         } else {
            if (AuditLogger.isResourceLoggingEnabled()) {
               AuditLogger.createAddressFailure(name, routingTypes);
            }
            throw ActiveMQMessageBundle.BUNDLE.addressAlreadyExists(addressInfo.getName());
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String updateAddress(String name, String routingTypes) throws Exception {

      checkStarted();

      clearIO();
      try {
         final EnumSet<RoutingType> routingTypeSet;
         if (routingTypes == null) {
            routingTypeSet = null;
         } else {
            routingTypeSet = EnumSet.noneOf(RoutingType.class);
            final String[] routingTypeNames = routingTypes.split(",");
            for (String routingTypeName : routingTypeNames) {
               routingTypeSet.add(RoutingType.valueOf(routingTypeName));
            }
         }
         if (!server.updateAddressInfo(SimpleString.toSimpleString(name), routingTypeSet)) {
            if (AuditLogger.isResourceLoggingEnabled()) {
               AuditLogger.updateAddressFailure(name, routingTypes);
            }
            throw ActiveMQMessageBundle.BUNDLE.addressDoesNotExist(SimpleString.toSimpleString(name));
         }
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.updateAddressSuccess(name, routingTypes);
         }
         return AddressInfoTextFormatter.Long.format(server.getAddressInfo(SimpleString.toSimpleString(name)), new StringBuilder()).toString();
      } finally {
         blockOnIO();
      }
   }


   @Override
   public void deleteAddress(String name) throws Exception {
      deleteAddress(name, false);
   }

   @Override
   public void deleteAddress(String name, boolean force) throws Exception {
      checkStarted();

      clearIO();
      try {
         server.removeAddressInfo(new SimpleString(name), null, force);
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.deleteAddressSuccess(name);
         }
      } catch (ActiveMQException e) {
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.deleteAddressFailure(name);
         }
         throw new IllegalStateException(e.getMessage());
      } finally {
         blockOnIO();
      }
   }

   @Deprecated
   @Override
   public void deployQueue(final String address, final String name, final String filterString) throws Exception {
      deployQueue(address, name, filterString, true);
   }

   @Deprecated
   @Override
   public void deployQueue(final String address,
                           final String name,
                           final String filterStr,
                           final boolean durable) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.deployQueue(this.server, address, name, filterStr, durable);
      }
      checkStarted();

      clearIO();
      try {
         server.createQueue(new QueueConfiguration(name)
                            .setAddress(address)
                            .setFilterString(filterStr)
                            .setDurable(durable));
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void createQueue(final String address, final String name) throws Exception {
      createQueue(address, name, true);
   }

   @Override
   public void createQueue(final String address, final String name, final String routingType) throws Exception {
      createQueue(address, name, true, routingType);
   }

   @Override
   public void createQueue(final String address, final String name, final boolean durable) throws Exception {
      createQueue(address, name, null, durable);
   }

   @Override
   public void createQueue(final String address, final String name, final boolean durable, final String routingType) throws Exception {
      createQueue(address, name, null, durable, routingType);
   }

   @Override
   public void createQueue(final String address, final String name, final String filterStr, final boolean durable) throws Exception {
      createQueue(address, name, filterStr, durable, server.getAddressSettingsRepository().getMatch(address == null ? name : address).getDefaultQueueRoutingType().toString());
   }


   @Override
   public void createQueue(final String address, final String name, final String filterStr, final boolean durable, final String routingType) throws Exception {
      AddressSettings addressSettings = server.getAddressSettingsRepository().getMatch(address == null ? name : address);
      createQueue(address, routingType, name, filterStr, durable, addressSettings.getDefaultMaxConsumers(), addressSettings.isDefaultPurgeOnNoConsumers(), addressSettings.isAutoCreateAddresses());
   }

   @Override
   public String createQueue(String address,
                             String routingType,
                             String name,
                             String filterStr,
                             boolean durable,
                             int maxConsumers,
                             boolean purgeOnNoConsumers,
                             boolean autoCreateAddress) throws Exception {
      AddressSettings addressSettings = server.getAddressSettingsRepository().getMatch(address == null ? name : address);
      return createQueue(
              address,
              routingType,
              name,
              filterStr,
              durable,
              maxConsumers,
              purgeOnNoConsumers,
              addressSettings.isDefaultExclusiveQueue(),
              addressSettings.isDefaultGroupRebalance(),
              addressSettings.getDefaultGroupBuckets(),
              addressSettings.isDefaultLastValueQueue(),
              addressSettings.getDefaultLastValueKey() == null ? null : addressSettings.getDefaultLastValueKey().toString(),
              addressSettings.isDefaultNonDestructive(),
              addressSettings.getDefaultConsumersBeforeDispatch(),
              addressSettings.getDefaultDelayBeforeDispatch(),
              addressSettings.isAutoDeleteCreatedQueues(),
              addressSettings.getAutoDeleteQueuesDelay(),
              addressSettings.getAutoDeleteQueuesMessageCount(),
              autoCreateAddress
      );
   }

   @Override
   public String createQueue(String address,
                             String routingType,
                             String name,
                             String filterStr,
                             boolean durable,
                             int maxConsumers,
                             boolean purgeOnNoConsumers,
                             boolean exclusive,
                             boolean groupRebalance,
                             int groupBuckets,
                             boolean lastValue,
                             String lastValueKey,
                             boolean nonDestructive,
                             int consumersBeforeDispatch,
                             long delayBeforeDispatch,
                             boolean autoCreateAddress) throws Exception {
      AddressSettings addressSettings = server.getAddressSettingsRepository().getMatch(address == null ? name : address);
      return createQueue(
            address,
            routingType,
            name,
            filterStr,
            durable,
            maxConsumers,
            purgeOnNoConsumers,
            exclusive,
            groupRebalance,
            groupBuckets,
            lastValue,
            lastValueKey,
            nonDestructive,
            consumersBeforeDispatch,
            delayBeforeDispatch,
            addressSettings.isAutoDeleteCreatedQueues(),
            addressSettings.getAutoDeleteQueuesDelay(),
            addressSettings.getAutoDeleteQueuesMessageCount(),
            autoCreateAddress);
   }

   @Override
   public String createQueue(String address,
                             String routingType,
                             String name,
                             String filterStr,
                             boolean durable,
                             int maxConsumers,
                             boolean purgeOnNoConsumers,
                             boolean exclusive,
                             boolean groupRebalance,
                             int groupBuckets,
                             boolean lastValue,
                             String lastValueKey,
                             boolean nonDestructive,
                             int consumersBeforeDispatch,
                             long delayBeforeDispatch,
                             boolean autoDelete,
                             long autoDeleteDelay,
                             long autoDeleteMessageCount,
                             boolean autoCreateAddress) throws Exception {
      AddressSettings addressSettings = server.getAddressSettingsRepository().getMatch(address == null ? name : address);
      return createQueue(
            address,
            routingType,
            name,
            filterStr,
            durable,
            maxConsumers,
            purgeOnNoConsumers,
            exclusive,
            groupRebalance,
            groupBuckets,
            addressSettings.getDefaultGroupFirstKey() == null ? null : addressSettings.getDefaultGroupFirstKey().toString(),
            lastValue,
            lastValueKey,
            nonDestructive,
            consumersBeforeDispatch,
            delayBeforeDispatch,
            autoDelete,
            autoDeleteDelay,
            autoDeleteMessageCount,
            autoCreateAddress
      );
   }

   @Override
   public String createQueue(String address,
                             String routingType,
                             String name,
                             String filterStr,
                             boolean durable,
                             int maxConsumers,
                             boolean purgeOnNoConsumers,
                             boolean exclusive,
                             boolean groupRebalance,
                             int groupBuckets,
                             String groupFirstKey,
                             boolean lastValue,
                             String lastValueKey,
                             boolean nonDestructive,
                             int consumersBeforeDispatch,
                             long delayBeforeDispatch,
                             boolean autoDelete,
                             long autoDeleteDelay,
                             long autoDeleteMessageCount,
                             boolean autoCreateAddress) throws Exception {
      AddressSettings addressSettings = server.getAddressSettingsRepository().getMatch(address == null ? name : address);
      return createQueue(
         address,
         routingType,
         name,
         filterStr,
         durable,
         maxConsumers,
         purgeOnNoConsumers,
         exclusive,
         groupRebalance,
         groupBuckets,
         groupFirstKey,
         lastValue,
         lastValueKey,
         nonDestructive,
         consumersBeforeDispatch,
         delayBeforeDispatch,
         autoDelete,
         autoDeleteDelay,
         autoDeleteMessageCount,
         autoCreateAddress,
         addressSettings.getDefaultRingSize()
      );
   }

   @Override
   public String createQueue(String address,
                             String routingType,
                             String name,
                             String filterStr,
                             boolean durable,
                             int maxConsumers,
                             boolean purgeOnNoConsumers,
                             boolean exclusive,
                             boolean groupRebalance,
                             int groupBuckets,
                             String groupFirstKey,
                             boolean lastValue,
                             String lastValueKey,
                             boolean nonDestructive,
                             int consumersBeforeDispatch,
                             long delayBeforeDispatch,
                             boolean autoDelete,
                             long autoDeleteDelay,
                             long autoDeleteMessageCount,
                             boolean autoCreateAddress,
                             long ringSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.createQueue(this.server, null, address, routingType, name, filterStr, durable,
                  maxConsumers, purgeOnNoConsumers, exclusive, groupRebalance, groupBuckets, groupFirstKey,
                  lastValue, lastValueKey, nonDestructive, consumersBeforeDispatch, delayBeforeDispatch,
                  autoDelete, autoDeleteDelay, autoDeleteMessageCount, autoCreateAddress, ringSize);
      }
      checkStarted();

      clearIO();

      SimpleString filter = filterStr == null ? null : new SimpleString(filterStr);
      try {
         if (filterStr != null && !filterStr.trim().equals("")) {
            filter = new SimpleString(filterStr);
         }

         final Queue queue = server.createQueue(new QueueConfiguration(name)
                                                   .setAddress(address)
                                                   .setRoutingType(RoutingType.valueOf(routingType.toUpperCase()))
                                                   .setFilterString(filter)
                                                   .setDurable(durable)
                                                   .setMaxConsumers(maxConsumers)
                                                   .setPurgeOnNoConsumers(purgeOnNoConsumers)
                                                   .setExclusive(exclusive)
                                                   .setGroupRebalance(groupRebalance)
                                                   .setGroupBuckets(groupBuckets)
                                                   .setGroupFirstKey(groupFirstKey)
                                                   .setLastValue(lastValue)
                                                   .setLastValueKey(lastValueKey)
                                                   .setNonDestructive(nonDestructive)
                                                   .setConsumersBeforeDispatch(consumersBeforeDispatch)
                                                   .setDelayBeforeDispatch(delayBeforeDispatch)
                                                   .setAutoDelete(autoDelete)
                                                   .setAutoDeleteDelay(autoDeleteDelay)
                                                   .setAutoDeleteMessageCount(autoDeleteMessageCount)
                                                   .setAutoCreateAddress(autoCreateAddress)
                                                   .setRingSize(ringSize));
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.createQueueSuccess( name, address, routingType);
         }
         return QueueTextFormatter.Long.format(queue, new StringBuilder()).toString();
      } catch (ActiveMQException e) {
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.createQueueFailure( name, address, routingType);
         }
         throw new IllegalStateException(e.getMessage());
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String createQueue(String queueConfigurationAsJson) throws Exception {
      return createQueue(queueConfigurationAsJson, false);
   }

   @Override
   public String createQueue(String queueConfigurationAsJson, boolean ignoreIfExists) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.createQueue(this.server, null, queueConfigurationAsJson, ignoreIfExists);
      }
      checkStarted();

      clearIO();

      try {
         // when the QueueConfiguration is passed through createQueue all of its defaults get set which we return to the caller
         QueueConfiguration queueConfiguration = QueueConfiguration.fromJSON(queueConfigurationAsJson);
         if (queueConfiguration == null) {
            throw ActiveMQMessageBundle.BUNDLE.failedToParseJson(queueConfigurationAsJson);
         }
         server.createQueue(queueConfiguration, ignoreIfExists);
         return queueConfiguration.toJSON();
      } catch (ActiveMQException e) {
         throw new IllegalStateException(e.getMessage());
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String updateQueue(String queueConfigurationAsJson) throws Exception {

      if (AuditLogger.isEnabled()) {
         AuditLogger.updateQueue(this.server, queueConfigurationAsJson);
      }
      checkStarted();

      clearIO();

      try {
         QueueConfiguration queueConfiguration = QueueConfiguration.fromJSON(queueConfigurationAsJson);
         if (queueConfiguration == null) {
            throw ActiveMQMessageBundle.BUNDLE.failedToParseJson(queueConfigurationAsJson);
         }
         final Queue queue = server.updateQueue(queueConfiguration);
         if (queue == null) {
            throw ActiveMQMessageBundle.BUNDLE.noSuchQueue(queueConfiguration.getName());
         }
         return server.locateQueue(queueConfiguration.getName()).getQueueConfiguration().toJSON();
      } finally {
         blockOnIO();
      }
   }

   @Deprecated
   @Override
   public String updateQueue(String name,
                             String routingType,
                             Integer maxConsumers,
                             Boolean purgeOnNoConsumers) throws Exception {
      return updateQueue(name, routingType, maxConsumers, purgeOnNoConsumers, null);
   }

   @Deprecated
   @Override
   public String updateQueue(String name,
                             String routingType,
                             Integer maxConsumers,
                             Boolean purgeOnNoConsumers,
                             Boolean exclusive) throws Exception {
      return updateQueue(name, routingType, maxConsumers, purgeOnNoConsumers, exclusive, null);
   }

   @Override
   public String updateQueue(String name,
                             String routingType,
                             Integer maxConsumers,
                             Boolean purgeOnNoConsumers,
                             Boolean exclusive,
                             String user) throws Exception {
      return updateQueue(name, routingType, null, maxConsumers, purgeOnNoConsumers, exclusive, null, null, null, null, null, user);
   }

   @Override
   public String updateQueue(String name,
                             String routingType,
                             String filter,
                             Integer maxConsumers,
                             Boolean purgeOnNoConsumers,
                             Boolean exclusive,
                             Boolean groupRebalance,
                             Integer groupBuckets,
                             Boolean nonDestructive,
                             Integer consumersBeforeDispatch,
                             Long delayBeforeDispatch,
                             String user) throws Exception {
      return updateQueue(name, routingType, filter, maxConsumers, purgeOnNoConsumers, exclusive, groupRebalance, groupBuckets, null, nonDestructive, consumersBeforeDispatch, delayBeforeDispatch, user);
   }

   @Override
   public String updateQueue(String name,
                             String routingType,
                             String filter,
                             Integer maxConsumers,
                             Boolean purgeOnNoConsumers,
                             Boolean exclusive,
                             Boolean groupRebalance,
                             Integer groupBuckets,
                             String groupFirstKey,
                             Boolean nonDestructive,
                             Integer consumersBeforeDispatch,
                             Long delayBeforeDispatch,
                             String user) throws Exception {
      return updateQueue(name, routingType, filter, maxConsumers, purgeOnNoConsumers, exclusive, groupRebalance, groupBuckets, null, nonDestructive, consumersBeforeDispatch, delayBeforeDispatch, user, null);
   }

   @Override
   public String updateQueue(String name,
                             String routingType,
                             String filter,
                             Integer maxConsumers,
                             Boolean purgeOnNoConsumers,
                             Boolean exclusive,
                             Boolean groupRebalance,
                             Integer groupBuckets,
                             String groupFirstKey,
                             Boolean nonDestructive,
                             Integer consumersBeforeDispatch,
                             Long delayBeforeDispatch,
                             String user,
                             Long ringSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.updateQueue(this.server, name, routingType, filter, maxConsumers, purgeOnNoConsumers,
                  exclusive, groupRebalance, groupBuckets, groupFirstKey, nonDestructive, consumersBeforeDispatch, delayBeforeDispatch, user, ringSize);
      }
      checkStarted();

      clearIO();

      try {
         final Queue queue = server.updateQueue(name, routingType != null ? RoutingType.valueOf(routingType) : null, filter, maxConsumers, purgeOnNoConsumers, exclusive, groupRebalance, groupBuckets, groupFirstKey, nonDestructive, consumersBeforeDispatch, delayBeforeDispatch, user);
         if (queue == null) {
            if (AuditLogger.isResourceLoggingEnabled()) {
               AuditLogger.updateQueueFailure(name, routingType);
            }
            throw ActiveMQMessageBundle.BUNDLE.noSuchQueue(new SimpleString(name));
         }
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.updateQueueSuccess(name, routingType);
         }

         return QueueTextFormatter.Long.format(queue, new StringBuilder()).toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getQueueNames() {
      return getQueueNames(null);
   }

   @Override
   public String[] getQueueNames(String routingType) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getQueueNames(this.server, routingType);
      }
      checkStarted();

      clearIO();
      try {
         Object[] queueControls = server.getManagementService().getResources(QueueControl.class);
         List<String> names = new ArrayList<>();
         for (int i = 0; i < queueControls.length; i++) {
            QueueControl queueControl = (QueueControl) queueControls[i];
            if (routingType != null && queueControl.getRoutingType().equals(routingType.toUpperCase())) {
               names.add(queueControl.getName());
            } else if (routingType == null) {
               names.add(queueControl.getName());
            }
         }

         String[] result = new String[names.size()];
         return names.toArray(result);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getClusterConnectionNames() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getClusterConnectionNames(this.server);
      }
      checkStarted();

      clearIO();
      try {
         List<String> names = new ArrayList<>();
         for (ClusterConnection clusterConnection : server.getClusterManager().getClusterConnections()) {
            names.add(clusterConnection.getName().toString());
         }

         String[] result = new String[names.size()];
         return names.toArray(result);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getUptime() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getUptime(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getUptime();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getUptimeMillis() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getUptimeMillis(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getUptimeMillis();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isReplicaSync() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isReplicaSync(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.isReplicaSync();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getAddressNames() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getAddressNames(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Object[] addresses = server.getManagementService().getResources(AddressControl.class);
         String[] names = new String[addresses.length];
         for (int i = 0; i < addresses.length; i++) {
            AddressControl address = (AddressControl) addresses[i];
            names[i] = address.getAddress();
         }
         return names;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void destroyQueue(final String name, final boolean removeConsumers, final boolean autoDeleteAddress) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.destroyQueue(this.server, null, name, removeConsumers, autoDeleteAddress);
      }
      checkStarted();

      clearIO();
      try {
         SimpleString queueName = new SimpleString(name);
         try {
            server.destroyQueue(queueName, null, !removeConsumers, removeConsumers, autoDeleteAddress);
         } catch (Exception e) {
            if (AuditLogger.isResourceLoggingEnabled()) {
               AuditLogger.destroyQueueFailure(name);
            }
            throw e;
         }
         if (AuditLogger.isResourceLoggingEnabled()) {
            AuditLogger.destroyQueueSuccess(name);
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void destroyQueue(final String name, final boolean removeConsumers) throws Exception {
      destroyQueue(name, removeConsumers, false);
   }

   @Override
   public void destroyQueue(final String name) throws Exception {
      destroyQueue(name, false);
   }

   @Override
   public String getAddressInfo(String address) throws ActiveMQAddressDoesNotExistException {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getAddressInfo(this.server, address);
      }
      checkStarted();

      clearIO();
      try {
         final AddressInfo addressInfo = server.getAddressInfo(SimpleString.toSimpleString(address));
         if (addressInfo == null) {
            throw ActiveMQMessageBundle.BUNDLE.addressDoesNotExist(SimpleString.toSimpleString(address));
         } else {
            return AddressInfoTextFormatter.Long.format(addressInfo, new StringBuilder()).toString();
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listBindingsForAddress(String address) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listBindingsForAddress(this.server, address);
      }
      checkStarted();

      clearIO();
      try {
         final Bindings bindings = server.getPostOffice().lookupBindingsForAddress(new SimpleString(address));
         return bindings == null ? "" : bindings.getBindings().stream().map(Binding::toManagementString).collect(Collectors.joining(","));
      } finally {
         blockOnIO();
      }
   }


   @Override
   public String listAddresses(String separator) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listAddresses(this.server, separator);
      }
      checkStarted();

      clearIO();
      try {
         final Set<SimpleString> addresses = server.getPostOffice().getAddresses();
         TreeSet<SimpleString> sortAddress = new TreeSet<>(new Comparator<SimpleString>() {
            @Override
            public int compare(SimpleString o1, SimpleString o2) {
               return o1.toString().compareToIgnoreCase(o2.toString());
            }
         });

         sortAddress.addAll(addresses);

         StringBuilder result = new StringBuilder();
         for (SimpleString string : sortAddress) {
            if (result.length() > 0) {
               result.append(separator);
            }
            result.append(string);
         }

         return result.toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getConnectionCount() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getConnectionCount(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getConnectionCount();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getTotalConnectionCount() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTotalConnectionCount(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getTotalConnectionCount();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getTotalMessageCount() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTotalMessageCount(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getTotalMessageCount();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getTotalMessagesAdded() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTotalMessagesAdded(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getTotalMessagesAdded();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getTotalMessagesAcknowledged() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTotalMessagesAcknowledged(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getTotalMessagesAcknowledged();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public long getTotalConsumerCount() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTotalConsumerCount(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return server.getTotalConsumerCount();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void enableMessageCounters() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.enableMessageCounters(this.server);
      }
      checkStarted();

      clearIO();
      try {
         setMessageCounterEnabled(true);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void disableMessageCounters() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.disableMessageCounters(this.server);
      }
      checkStarted();

      clearIO();
      try {
         setMessageCounterEnabled(false);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void resetAllMessageCounters() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.resetAllMessageCounters(this.server);
      }
      checkStarted();

      clearIO();
      try {
         messageCounterManager.resetAllCounters();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void resetAllMessageCounterHistories() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.resetAllMessageCounterHistories(this.server);
      }
      checkStarted();

      clearIO();
      try {
         messageCounterManager.resetAllCounterHistories();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public boolean isMessageCounterEnabled() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isMessageCounterEnabled(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return configuration.isMessageCounterEnabled();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public synchronized long getMessageCounterSamplePeriod() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getMessageCounterSamplePeriod(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return messageCounterManager.getSamplePeriod();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public synchronized void setMessageCounterSamplePeriod(final long newPeriod) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.setMessageCounterSamplePeriod(this.server, newPeriod);
      }

      checkStarted();

      clearIO();
      try {
         if (newPeriod < MessageCounterManagerImpl.MIN_SAMPLE_PERIOD) {
            if (newPeriod <= 0) {
               throw ActiveMQMessageBundle.BUNDLE.periodMustGreaterThanZero(newPeriod);
            }
            ActiveMQServerLogger.LOGGER.invalidMessageCounterPeriod(newPeriod);
         }

         if (messageCounterManager != null && newPeriod != messageCounterManager.getSamplePeriod()) {
            messageCounterManager.reschedule(newPeriod);
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public int getMessageCounterMaxDayCount() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getMessageCounterMaxDayCount(this.server);
      }
      checkStarted();

      clearIO();
      try {
         return messageCounterManager.getMaxDayCount();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void setMessageCounterMaxDayCount(final int count) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.setMessageCounterMaxDayCount(this.server, count);
      }
      checkStarted();

      clearIO();
      try {
         if (count <= 0) {
            throw ActiveMQMessageBundle.BUNDLE.greaterThanZero(count);
         }
         messageCounterManager.setMaxDayCount(count);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] listPreparedTransactions() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listPreparedTransactions(this.server);
      }
      checkStarted();

      clearIO();
      try {
         DateFormat dateFormat = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.MEDIUM);

         Map<Xid, Long> xids = resourceManager.getPreparedTransactionsWithCreationTime();
         ArrayList<Entry<Xid, Long>> xidsSortedByCreationTime = new ArrayList<>(xids.entrySet());
         Collections.sort(xidsSortedByCreationTime, new Comparator<Entry<Xid, Long>>() {
            @Override
            public int compare(final Entry<Xid, Long> entry1, final Entry<Xid, Long> entry2) {
               // sort by creation time, oldest first
               return entry1.getValue().compareTo(entry2.getValue());
            }
         });
         String[] s = new String[xidsSortedByCreationTime.size()];
         int i = 0;
         for (Map.Entry<Xid, Long> entry : xidsSortedByCreationTime) {
            Date creation = new Date(entry.getValue());
            Xid xid = entry.getKey();
            s[i++] = dateFormat.format(creation) + " base64: " + XidImpl.toBase64String(xid) + " " + xid.toString();
         }
         return s;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listPreparedTransactionDetailsAsJSON() throws Exception {
      return listPreparedTransactionDetailsAsJSON((xid, tx, creation) -> new CoreTransactionDetail(xid, tx, creation));
   }

   public String listPreparedTransactionDetailsAsJSON(TransactionDetailFactory factory) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listPreparedTransactionDetailsAsJSON(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Map<Xid, Long> xids = resourceManager.getPreparedTransactionsWithCreationTime();
         if (xids == null || xids.size() == 0) {
            return "";
         }

         ArrayList<Entry<Xid, Long>> xidsSortedByCreationTime = new ArrayList<>(xids.entrySet());
         Collections.sort(xidsSortedByCreationTime, new Comparator<Entry<Xid, Long>>() {
            @Override
            public int compare(final Entry<Xid, Long> entry1, final Entry<Xid, Long> entry2) {
               // sort by creation time, oldest first
               return entry1.getValue().compareTo(entry2.getValue());
            }
         });

         JsonArrayBuilder txDetailListJson = JsonLoader.createArrayBuilder();
         for (Map.Entry<Xid, Long> entry : xidsSortedByCreationTime) {
            Xid xid = entry.getKey();

            Transaction tx = resourceManager.getTransaction(xid);

            if (tx == null) {
               continue;
            }

            TransactionDetail detail = factory.createTransactionDetail(xid, tx, entry.getValue());

            txDetailListJson.add(detail.toJSON());
         }
         return txDetailListJson.build().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listPreparedTransactionDetailsAsHTML() throws Exception {
      return listPreparedTransactionDetailsAsHTML((xid, tx, creation) -> new CoreTransactionDetail(xid, tx, creation));
   }

   public String listPreparedTransactionDetailsAsHTML(TransactionDetailFactory factory) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listPreparedTransactionDetailsAsHTML(this.server, factory);
      }
      checkStarted();

      clearIO();
      try {
         Map<Xid, Long> xids = resourceManager.getPreparedTransactionsWithCreationTime();
         if (xids == null || xids.size() == 0) {
            return "<h3>*** Prepared Transaction Details ***</h3><p>No entry.</p>";
         }

         ArrayList<Entry<Xid, Long>> xidsSortedByCreationTime = new ArrayList<>(xids.entrySet());
         Collections.sort(xidsSortedByCreationTime, new Comparator<Entry<Xid, Long>>() {
            @Override
            public int compare(final Entry<Xid, Long> entry1, final Entry<Xid, Long> entry2) {
               // sort by creation time, oldest first
               return entry1.getValue().compareTo(entry2.getValue());
            }
         });

         StringBuilder html = new StringBuilder();
         html.append("<h3>*** Prepared Transaction Details ***</h3>");

         for (Map.Entry<Xid, Long> entry : xidsSortedByCreationTime) {
            Xid xid = entry.getKey();

            Transaction tx = resourceManager.getTransaction(xid);

            if (tx == null) {
               continue;
            }

            TransactionDetail detail = factory.createTransactionDetail(xid, tx, entry.getValue());

            JsonObject txJson = detail.toJSON();

            html.append("<table border=\"1\">");
            html.append("<tr><th>creation_time</th>");
            html.append("<td>" + txJson.get(TransactionDetail.KEY_CREATION_TIME) + "</td>");
            html.append("<th>xid_as_base_64</th>");
            html.append("<td colspan=\"3\">" + txJson.get(TransactionDetail.KEY_XID_AS_BASE64) + "</td></tr>");
            html.append("<tr><th>xid_format_id</th>");
            html.append("<td>" + txJson.get(TransactionDetail.KEY_XID_FORMAT_ID) + "</td>");
            html.append("<th>xid_global_txid</th>");
            html.append("<td>" + txJson.get(TransactionDetail.KEY_XID_GLOBAL_TXID) + "</td>");
            html.append("<th>xid_branch_qual</th>");
            html.append("<td>" + txJson.get(TransactionDetail.KEY_XID_BRANCH_QUAL) + "</td></tr>");

            html.append("<tr><th colspan=\"6\">Message List</th></tr>");
            html.append("<tr><td colspan=\"6\">");
            html.append("<table border=\"1\" cellspacing=\"0\" cellpadding=\"0\">");

            JsonArray msgs = txJson.getJsonArray(TransactionDetail.KEY_TX_RELATED_MESSAGES);
            for (int i = 0; i < msgs.size(); i++) {
               JsonObject msgJson = msgs.getJsonObject(i);
               JsonObject props = msgJson.getJsonObject(TransactionDetail.KEY_MSG_PROPERTIES);
               StringBuilder propstr = new StringBuilder();
               Set<String> keys = props.keySet();
               for (String key : keys) {
                  propstr.append(key);
                  propstr.append("=");
                  propstr.append(props.get(key));
                  propstr.append(", ");
               }

               html.append("<th>operation_type</th>");
               html.append("<td>" + msgJson.get(TransactionDetail.KEY_MSG_OP_TYPE) + "</th>");
               html.append("<th>message_type</th>");
               html.append("<td>" + msgJson.get(TransactionDetail.KEY_MSG_TYPE) + "</td></tr>");
               html.append("<tr><th>properties</th>");
               html.append("<td colspan=\"3\">" + propstr.toString() + "</td></tr>");
            }
            html.append("</table></td></tr>");
            html.append("</table><br>");
         }

         return html.toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] listHeuristicCommittedTransactions() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listHeuristicCommittedTransactions(this.server);
      }
      checkStarted();

      clearIO();
      try {
         List<Xid> xids = resourceManager.getHeuristicCommittedTransactions();
         String[] s = new String[xids.size()];
         int i = 0;
         for (Xid xid : xids) {
            s[i++] = XidImpl.toBase64String(xid);
         }
         return s;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] listHeuristicRolledBackTransactions() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listHeuristicRolledBackTransactions(this.server);
      }
      checkStarted();

      clearIO();
      try {
         List<Xid> xids = resourceManager.getHeuristicRolledbackTransactions();
         String[] s = new String[xids.size()];
         int i = 0;
         for (Xid xid : xids) {
            s[i++] = XidImpl.toBase64String(xid);
         }
         return s;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public synchronized boolean commitPreparedTransaction(final String transactionAsBase64) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.commitPreparedTransaction(this.server, transactionAsBase64);
      }
      checkStarted();

      clearIO();
      try {
         List<Xid> xids = resourceManager.getPreparedTransactions();

         for (Xid xid : xids) {
            if (XidImpl.toBase64String(xid).equals(transactionAsBase64)) {
               Transaction transaction = resourceManager.removeTransaction(xid, null);
               transaction.commit(false);
               long recordID = server.getStorageManager().storeHeuristicCompletion(xid, true);
               storageManager.waitOnOperations();
               resourceManager.putHeuristicCompletion(recordID, xid, true);
               return true;
            }
         }
         return false;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public synchronized boolean rollbackPreparedTransaction(final String transactionAsBase64) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.rollbackPreparedTransaction(this.server, transactionAsBase64);
      }
      checkStarted();

      clearIO();
      try {

         List<Xid> xids = resourceManager.getPreparedTransactions();

         for (Xid xid : xids) {
            if (XidImpl.toBase64String(xid).equals(transactionAsBase64)) {
               Transaction transaction = resourceManager.removeTransaction(xid, null);
               transaction.rollback();
               long recordID = server.getStorageManager().storeHeuristicCompletion(xid, false);
               server.getStorageManager().waitOnOperations();
               resourceManager.putHeuristicCompletion(recordID, xid, false);
               return true;
            }
         }
         return false;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] listRemoteAddresses() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listRemoteAddresses(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Set<RemotingConnection> connections = remotingService.getConnections();

         String[] remoteAddresses = new String[connections.size()];
         int i = 0;
         for (RemotingConnection connection : connections) {
            remoteAddresses[i++] = connection.getRemoteAddress();
         }
         return remoteAddresses;
      } finally {
         blockOnIO();
      }

   }

   @Override
   public String[] listRemoteAddresses(final String ipAddress) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listRemoteAddresses(this.server, ipAddress);
      }
      checkStarted();

      clearIO();
      try {
         Set<RemotingConnection> connections = remotingService.getConnections();
         List<String> remoteConnections = new ArrayList<>();
         for (RemotingConnection connection : connections) {
            String remoteAddress = connection.getRemoteAddress();
            if (remoteAddress.contains(ipAddress)) {
               remoteConnections.add(connection.getRemoteAddress());
            }
         }
         return remoteConnections.toArray(new String[remoteConnections.size()]);
      } finally {
         blockOnIO();
      }

   }

   @Override
   public boolean closeConnectionsForAddress(final String ipAddress) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.closeConnectionsForAddress(this.server, ipAddress);
      }
      checkStarted();

      clearIO();
      try {
         boolean closed = false;
         Set<RemotingConnection> connections = remotingService.getConnections();
         for (RemotingConnection connection : connections) {
            String remoteAddress = connection.getRemoteAddress();
            if (remoteAddress.contains(ipAddress)) {
               connection.fail(ActiveMQMessageBundle.BUNDLE.connectionsClosedByManagement(ipAddress));
               remotingService.removeConnection(connection.getID());
               closed = true;
            }
         }

         return closed;
      } finally {
         blockOnIO();
      }

   }

   @Override
   public boolean closeConsumerConnectionsForAddress(final String address) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.closeConsumerConnectionsForAddress(this.server, address);
      }
      boolean closed = false;
      checkStarted();

      clearIO();
      try {
         for (Binding binding : postOffice.getMatchingBindings(SimpleString.toSimpleString(address))) {
            if (binding instanceof LocalQueueBinding) {
               Queue queue = ((LocalQueueBinding) binding).getQueue();
               for (Consumer consumer : queue.getConsumers()) {
                  if (consumer instanceof ServerConsumer) {
                     ServerConsumer serverConsumer = (ServerConsumer) consumer;
                     RemotingConnection connection = null;

                     for (RemotingConnection potentialConnection : remotingService.getConnections()) {
                        if (potentialConnection.getID().toString().equals(serverConsumer.getConnectionID())) {
                           connection = potentialConnection;
                        }
                     }

                     if (connection != null) {
                        remotingService.removeConnection(connection.getID());
                        connection.fail(ActiveMQMessageBundle.BUNDLE.consumerConnectionsClosedByManagement(address));
                        closed = true;
                     }
                  }
               }
            }
         }
      } catch (Exception e) {
         ActiveMQServerLogger.LOGGER.failedToCloseConsumerConnectionsForAddress(address, e);
      } finally {
         blockOnIO();
      }
      return closed;
   }

   @Override
   public boolean closeConnectionsForUser(final String userName) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.closeConnectionsForUser(this.server, userName);
      }
      boolean closed = false;
      checkStarted();

      clearIO();
      try {
         for (ServerSession serverSession : server.getSessions()) {
            if (serverSession.getUsername() != null && serverSession.getUsername().equals(userName)) {
               RemotingConnection connection = null;

               for (RemotingConnection potentialConnection : remotingService.getConnections()) {
                  if (potentialConnection.getID().toString().equals(serverSession.getConnectionID().toString())) {
                     connection = potentialConnection;
                  }
               }

               if (connection != null) {
                  remotingService.removeConnection(connection.getID());
                  connection.fail(ActiveMQMessageBundle.BUNDLE.connectionsForUserClosedByManagement(userName));
                  closed = true;
               }
            }
         }
      } finally {
         blockOnIO();
      }
      return closed;
   }

   @Override
   public boolean closeConnectionWithID(final String ID) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.closeConnectionWithID(this.server, ID);
      }
      checkStarted();

      clearIO();
      try {
         for (RemotingConnection connection : remotingService.getConnections()) {
            if (connection.getID().toString().equals(ID)) {
               remotingService.removeConnection(connection.getID());
               connection.fail(ActiveMQMessageBundle.BUNDLE.connectionWithIDClosedByManagement(ID));
               return true;
            }
         }
      } finally {
         blockOnIO();
      }
      return false;
   }

   @Override
   public boolean closeSessionWithID(final String connectionID, final String ID) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.closeSessionWithID(this.server, connectionID, ID);
      }
      checkStarted();

      clearIO();
      try {
         List<ServerSession> sessions = server.getSessions(connectionID);
         for (ServerSession session : sessions) {
            if (session.getName().equals(ID.toString())) {
               session.close(true);
               return true;
            }
         }

      } finally {
         blockOnIO();
      }
      return false;
   }

   @Override
   public boolean closeConsumerWithID(final String sessionID, final String ID) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.closeConsumerWithID(this.server, sessionID, ID);
      }
      checkStarted();

      clearIO();
      try {
         Set<ServerSession> sessions = server.getSessions();
         for (ServerSession session : sessions) {
            if (session.getName().equals(sessionID.toString())) {
               Set<ServerConsumer> serverConsumers = session.getServerConsumers();
               for (ServerConsumer serverConsumer : serverConsumers) {
                  if (serverConsumer.sequentialID() == Long.valueOf(ID)) {
                     serverConsumer.disconnect();
                     return true;
                  }
               }
            }
         }

      } finally {
         blockOnIO();
      }
      return false;
   }

   @Override
   public String[] listConnectionIDs() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listConnectionIDs(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Set<RemotingConnection> connections = remotingService.getConnections();
         String[] connectionIDs = new String[connections.size()];
         int i = 0;
         for (RemotingConnection connection : connections) {
            connectionIDs[i++] = connection.getID().toString();
         }
         return connectionIDs;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] listSessions(final String connectionID) {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listSessions(this.server, connectionID);
      }
      checkStarted();

      clearIO();
      try {
         List<ServerSession> sessions = server.getSessions(connectionID);
         String[] sessionIDs = new String[sessions.size()];
         int i = 0;
         for (ServerSession serverSession : sessions) {
            sessionIDs[i++] = serverSession.getName();
         }
         return sessionIDs;
      } finally {
         blockOnIO();
      }
   }

   /* (non-Javadoc)
    * @see org.apache.activemq.artemis.api.core.management.ActiveMQServerControl#listProducersInfoAsJSON()
    */
   @Override
   public String listProducersInfoAsJSON() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listProducersInfoAsJSON(this.server);
      }
      JsonArrayBuilder producers = JsonLoader.createArrayBuilder();

      for (ServerSession session : server.getSessions()) {
         session.describeProducersInfo(producers);
      }

      return producers.build().toString();
   }

   @Override
   public String listConnections(String options, int page, int pageSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listConnections(this.server, options, page, pageSize);
      }
      checkStarted();
      clearIO();
      try {
         server.getPostOffice().getAddresses();
         ConnectionView view = new ConnectionView(server);
         view.setCollection(server.getRemotingService().getConnections());
         view.setOptions(options);
         return view.getResultsAsJson(page, pageSize);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listSessions(String options, int page, int pageSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listSessions(this.server, options, page, pageSize);
      }
      checkStarted();
      clearIO();
      try {
         SessionView view = new SessionView();
         view.setCollection(server.getSessions());
         view.setOptions(options);
         return view.getResultsAsJson(page, pageSize);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listConsumers(String options, int page, int pageSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listConsumers(this.server, options, page, pageSize);
      }
      checkStarted();
      clearIO();
      try {
         Set<ServerConsumer> consumers = new HashSet();
         for (ServerSession session : server.getSessions()) {
            consumers.addAll(session.getServerConsumers());
         }
         ConsumerView view = new ConsumerView(server);
         view.setCollection(consumers);
         view.setOptions(options);
         return view.getResultsAsJson(page, pageSize);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listAddresses(String options, int page, int pageSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listAddresses(this.server, options, page, pageSize);
      }
      checkStarted();
      clearIO();
      try {
         final Set<SimpleString> addresses = server.getPostOffice().getAddresses();
         List<AddressInfo> addressInfo = new ArrayList<>();
         for (SimpleString address : addresses) {
            AddressInfo info = server.getPostOffice().getAddressInfo(address);
            //ignore if no longer available
            if (info != null) {
               addressInfo.add(info);
            }
         }
         AddressView view = new AddressView(server);
         view.setCollection(addressInfo);
         view.setOptions(options);
         return view.getResultsAsJson(page, pageSize);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listQueues(String options, int page, int pageSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listQueues(this.server, options, page, pageSize);
      }
      checkStarted();

      clearIO();
      try {
         List<QueueControl> queues = new ArrayList<>();
         Object[] qs = server.getManagementService().getResources(QueueControl.class);
         for (int i = 0; i < qs.length; i++) {
            queues.add((QueueControl) qs[i]);
         }
         QueueView view = new QueueView(server);
         view.setCollection(queues);
         view.setOptions(options);
         return view.getResultsAsJson(page, pageSize);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listProducers(@Parameter(name = "Options") String options,
                               @Parameter(name = "Page Number") int page,
                               @Parameter(name = "Page Size") int pageSize) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listProducers(this.server, options, page, pageSize);
      }
      checkStarted();
      clearIO();
      try {
         Set<ServerProducer> producers = new HashSet<>();
         for (ServerSession session : server.getSessions()) {
            producers.addAll(session.getServerProducers().values());
         }
         ProducerView view = new ProducerView(server);
         view.setCollection(producers);
         view.setOptions(options);
         return view.getResultsAsJson(page, pageSize);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listConnectionsAsJSON() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listConnectionsAsJSON(this.server);
      }
      checkStarted();

      clearIO();

      try {
         JsonArrayBuilder array = JsonLoader.createArrayBuilder();

         Set<RemotingConnection> connections = server.getRemotingService().getConnections();

         for (RemotingConnection connection : connections) {
            JsonObjectBuilder obj = JsonLoader.createObjectBuilder().add("connectionID", connection.getID().toString()).add("clientAddress", connection.getRemoteAddress()).add("creationTime", connection.getCreationTime()).add("implementation", connection.getClass().getSimpleName()).add("sessionCount", server.getSessions(connection.getID().toString()).size());
            array.add(obj);
         }
         return array.build().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listSessionsAsJSON(final String connectionID) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listSessionsAsJSON(this.server, connectionID);
      }
      checkStarted();

      clearIO();

      JsonArrayBuilder array = JsonLoader.createArrayBuilder();
      try {
         List<ServerSession> sessions = server.getSessions(connectionID);
         for (ServerSession sess : sessions) {
            buildSessionJSON(array, sess);
         }
      } finally {
         blockOnIO();
      }
      return array.build().toString();
   }

   @Override
   public String listAllSessionsAsJSON() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listAllSessionsAsJSON(this.server);
      }
      checkStarted();

      clearIO();

      JsonArrayBuilder array = JsonLoader.createArrayBuilder();
      try {
         Set<ServerSession> sessions = server.getSessions();
         for (ServerSession sess : sessions) {
            buildSessionJSON(array, sess);
         }
      } finally {
         blockOnIO();
      }
      return array.build().toString();
   }

   public void buildSessionJSON(JsonArrayBuilder array, ServerSession sess) {
      JsonObjectBuilder obj = JsonLoader.createObjectBuilder().add("sessionID", sess.getName()).add("creationTime", sess.getCreationTime()).add("consumerCount", sess.getServerConsumers().size());

      if (sess.getValidatedUser() != null) {
         obj.add("principal", sess.getValidatedUser());
      }

      if (sess.getMetaData() != null) {
         final JsonObjectBuilder metadata = JsonLoader.createObjectBuilder();
         for (Entry<String, String> entry : sess.getMetaData().entrySet()) {
            metadata.add(entry.getKey(), entry.getValue());
         }
         obj.add("metadata", metadata);
      }

      array.add(obj);
   }

   @Override
   public String listConsumersAsJSON(String connectionID) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listConsumersAsJSON(this.server, connectionID);
      }
      checkStarted();

      clearIO();

      try {
         JsonArrayBuilder array = JsonLoader.createArrayBuilder();

         Set<RemotingConnection> connections = server.getRemotingService().getConnections();
         for (RemotingConnection connection : connections) {
            if (connectionID.equals(connection.getID().toString())) {
               List<ServerSession> sessions = server.getSessions(connectionID);
               for (ServerSession session : sessions) {
                  Set<ServerConsumer> consumers = session.getServerConsumers();
                  for (ServerConsumer consumer : consumers) {
                     JsonObject obj = toJSONObject(consumer);
                     if (obj != null) {
                        array.add(obj);
                     }
                  }
               }
            }
         }
         return array.build().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String listAllConsumersAsJSON() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listAllConsumersAsJSON(this.server);
      }
      checkStarted();

      clearIO();

      try {
         JsonArrayBuilder array = JsonLoader.createArrayBuilder();

         Set<ServerSession> sessions = server.getSessions();
         for (ServerSession session : sessions) {
            Set<ServerConsumer> consumers = session.getServerConsumers();
            for (ServerConsumer consumer : consumers) {
               JsonObject obj = toJSONObject(consumer);
               if (obj != null) {
                  array.add(obj);
               }
            }
         }
         return array.build().toString();
      } finally {
         blockOnIO();
      }
   }

   private JsonObject toJSONObject(ServerConsumer consumer) throws Exception {
      JsonObjectBuilder obj = JsonLoader.createObjectBuilder().add("consumerID", consumer.getID()).add("connectionID", consumer.getConnectionID().toString()).add("sessionID", consumer.getSessionID()).add("queueName", consumer.getQueue().getName().toString()).add("browseOnly", consumer.isBrowseOnly()).add("creationTime", consumer.getCreationTime()).add("deliveringCount", consumer.getDeliveringMessages().size());
      if (consumer.getFilter() != null) {
         obj.add("filter", consumer.getFilter().getFilterString().toString());
      }

      return obj.build();
   }

   @Override
   public Object[] getConnectors() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getConnectors(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Collection<TransportConfiguration> connectorConfigurations = configuration.getConnectorConfigurations().values();

         Object[] ret = new Object[connectorConfigurations.size()];

         int i = 0;
         for (TransportConfiguration config : connectorConfigurations) {
            Object[] tc = new Object[3];

            tc[0] = config.getName();
            tc[1] = config.getFactoryClassName();
            tc[2] = config.getParams();

            ret[i++] = tc;
         }
         return ret;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getConnectorsAsJSON() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getConnectorsAsJSON(this.server);
      }
      checkStarted();

      clearIO();
      try {
         JsonArrayBuilder array = JsonLoader.createArrayBuilder();

         for (TransportConfiguration config : configuration.getConnectorConfigurations().values()) {
            array.add(config.toJson());
         }

         return array.build().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void addSecuritySettings(final String addressMatch,
                                   final String sendRoles,
                                   final String consumeRoles,
                                   final String createDurableQueueRoles,
                                   final String deleteDurableQueueRoles,
                                   final String createNonDurableQueueRoles,
                                   final String deleteNonDurableQueueRoles,
                                   final String manageRoles) throws Exception {
      addSecuritySettings(addressMatch, sendRoles, consumeRoles, createDurableQueueRoles, deleteDurableQueueRoles, createNonDurableQueueRoles, deleteNonDurableQueueRoles, manageRoles, "");
   }

   @Override
   public void addSecuritySettings(final String addressMatch,
                                   final String sendRoles,
                                   final String consumeRoles,
                                   final String createDurableQueueRoles,
                                   final String deleteDurableQueueRoles,
                                   final String createNonDurableQueueRoles,
                                   final String deleteNonDurableQueueRoles,
                                   final String manageRoles,
                                   final String browseRoles) throws Exception {
      addSecuritySettings(addressMatch, sendRoles, consumeRoles, createDurableQueueRoles, deleteDurableQueueRoles, createNonDurableQueueRoles, deleteNonDurableQueueRoles, manageRoles, browseRoles, "", "");
   }

   @Override
   public void addSecuritySettings(final String addressMatch,
                                   final String sendRoles,
                                   final String consumeRoles,
                                   final String createDurableQueueRoles,
                                   final String deleteDurableQueueRoles,
                                   final String createNonDurableQueueRoles,
                                   final String deleteNonDurableQueueRoles,
                                   final String manageRoles,
                                   final String browseRoles,
                                   final String createAddressRoles,
                                   final String deleteAddressRoles) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.addSecuritySettings(this.server, addressMatch, sendRoles, consumeRoles, createDurableQueueRoles,
                  deleteDurableQueueRoles, createNonDurableQueueRoles, deleteNonDurableQueueRoles, manageRoles,
                  browseRoles, createAddressRoles, deleteAddressRoles);
      }
      checkStarted();

      clearIO();
      try {
         Set<Role> roles = SecurityFormatter.createSecurity(sendRoles, consumeRoles, createDurableQueueRoles, deleteDurableQueueRoles, createNonDurableQueueRoles, deleteNonDurableQueueRoles, manageRoles, browseRoles, createAddressRoles, deleteAddressRoles);

         server.getSecurityRepository().addMatch(addressMatch, roles);

         PersistedRoles persistedRoles = new PersistedRoles(addressMatch, sendRoles, consumeRoles, createDurableQueueRoles, deleteDurableQueueRoles, createNonDurableQueueRoles, deleteNonDurableQueueRoles, manageRoles, browseRoles, createAddressRoles, deleteAddressRoles);

         storageManager.storeSecurityRoles(persistedRoles);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void removeSecuritySettings(final String addressMatch) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.removeSecuritySettings(this.server, addressMatch);
      }
      checkStarted();

      clearIO();
      try {
         server.getSecurityRepository().removeMatch(addressMatch);
         storageManager.deleteSecurityRoles(new SimpleString(addressMatch));
      } finally {
         blockOnIO();
      }
   }

   @Override
   public Object[] getRoles(final String addressMatch) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getRoles(this.server, addressMatch);
      }

      checkStarted();

      clearIO();
      try {
         Set<Role> roles = server.getSecurityRepository().getMatch(addressMatch);

         Object[] objRoles = new Object[roles.size()];

         int i = 0;
         for (Role role : roles) {
            objRoles[i++] = new Object[]{role.getName(), CheckType.SEND.hasRole(role), CheckType.CONSUME.hasRole(role), CheckType.CREATE_DURABLE_QUEUE.hasRole(role), CheckType.DELETE_DURABLE_QUEUE.hasRole(role), CheckType.CREATE_NON_DURABLE_QUEUE.hasRole(role), CheckType.DELETE_NON_DURABLE_QUEUE.hasRole(role), CheckType.MANAGE.hasRole(role)};
         }
         return objRoles;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getRolesAsJSON(final String addressMatch) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getRolesAsJSON(this.server, addressMatch);
      }
      checkStarted();

      clearIO();
      try {
         JsonArrayBuilder json = JsonLoader.createArrayBuilder();
         Set<Role> roles = server.getSecurityRepository().getMatch(addressMatch);

         for (Role role : roles) {
            json.add(role.toJson());
         }
         return json.build().toString();
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String getAddressSettingsAsJSON(final String address) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getAddressSettingsAsJSON(this.server, address);
      }
      checkStarted();

      AddressSettings addressSettings = server.getAddressSettingsRepository().getMatch(address);
      String policy = addressSettings.getAddressFullMessagePolicy() == AddressFullMessagePolicy.PAGE ? "PAGE" : addressSettings.getAddressFullMessagePolicy() == AddressFullMessagePolicy.BLOCK ? "BLOCK" : addressSettings.getAddressFullMessagePolicy() == AddressFullMessagePolicy.DROP ? "DROP" : "FAIL";
      String consumerPolicy = addressSettings.getSlowConsumerPolicy() == SlowConsumerPolicy.NOTIFY ? "NOTIFY" : "KILL";
      JsonObjectBuilder settings = JsonLoader.createObjectBuilder();
      if (addressSettings.getDeadLetterAddress() != null) {
         settings.add("DLA", addressSettings.getDeadLetterAddress().toString());
      }
      if (addressSettings.getExpiryAddress() != null) {
         settings.add("expiryAddress", addressSettings.getExpiryAddress().toString());
      }

      return settings.add("expiryDelay", addressSettings.getExpiryDelay())
            .add("minExpiryDelay", addressSettings.getMinExpiryDelay())
            .add("maxExpiryDelay", addressSettings.getMaxExpiryDelay())
            .add("maxDeliveryAttempts", addressSettings.getMaxDeliveryAttempts())
            .add("pageCacheMaxSize", addressSettings.getPageCacheMaxSize())
            .add("maxSizeBytes", addressSettings.getMaxSizeBytes())
            .add("pageSizeBytes", addressSettings.getPageSizeBytes())
            .add("redeliveryDelay", addressSettings.getRedeliveryDelay())
            .add("redeliveryMultiplier", addressSettings.getRedeliveryMultiplier())
            .add("maxRedeliveryDelay", addressSettings.getMaxRedeliveryDelay())
            .add("redistributionDelay", addressSettings.getRedistributionDelay())
            .add("lastValueQueue", addressSettings.isDefaultLastValueQueue())
            .add("sendToDLAOnNoRoute", addressSettings.isSendToDLAOnNoRoute())
            .add("addressFullMessagePolicy", policy)
            .add("slowConsumerThreshold", addressSettings.getSlowConsumerThreshold())
            .add("slowConsumerCheckPeriod", addressSettings.getSlowConsumerCheckPeriod())
            .add("slowConsumerPolicy", consumerPolicy)
            .add("autoCreateJmsQueues", addressSettings.isAutoCreateJmsQueues())
            .add("autoDeleteJmsQueues", addressSettings.isAutoDeleteJmsQueues())
            .add("autoCreateJmsTopics", addressSettings.isAutoCreateJmsTopics())
            .add("autoDeleteJmsTopics", addressSettings.isAutoDeleteJmsTopics())
            .add("autoCreateQueues", addressSettings.isAutoCreateQueues())
            .add("autoDeleteQueues", addressSettings.isAutoDeleteQueues())
            .add("autoCreateAddresses", addressSettings.isAutoCreateAddresses())
            .add("autoDeleteAddresses", addressSettings.isAutoDeleteAddresses())
            .add("configDeleteQueues", addressSettings.getConfigDeleteQueues().toString())
            .add("configDeleteAddresses", addressSettings.getConfigDeleteAddresses().toString())
            .add("maxSizeBytesRejectThreshold", addressSettings.getMaxSizeBytesRejectThreshold())
            .add("defaultLastValueKey", addressSettings.getDefaultLastValueKey() == null ? "" : addressSettings.getDefaultLastValueKey().toString())
            .add("defaultNonDestructive", addressSettings.isDefaultNonDestructive())
            .add("defaultExclusiveQueue", addressSettings.isDefaultExclusiveQueue())
            .add("defaultGroupRebalance", addressSettings.isDefaultGroupRebalance())
            .add("defaultGroupRebalancePauseDispatch", addressSettings.isDefaultGroupRebalancePauseDispatch())
            .add("defaultGroupBuckets", addressSettings.getDefaultGroupBuckets())
            .add("defaultGroupFirstKey", addressSettings.getDefaultGroupFirstKey() == null ? "" : addressSettings.getDefaultGroupFirstKey().toString())
            .add("defaultMaxConsumers", addressSettings.getDefaultMaxConsumers())
            .add("defaultPurgeOnNoConsumers", addressSettings.isDefaultPurgeOnNoConsumers())
            .add("defaultConsumersBeforeDispatch", addressSettings.getDefaultConsumersBeforeDispatch())
            .add("defaultDelayBeforeDispatch", addressSettings.getDefaultDelayBeforeDispatch())
            .add("defaultQueueRoutingType", addressSettings.getDefaultQueueRoutingType().toString())
            .add("defaultAddressRoutingType", addressSettings.getDefaultAddressRoutingType().toString())
            .add("defaultConsumerWindowSize", addressSettings.getDefaultConsumerWindowSize())
            .add("defaultRingSize", addressSettings.getDefaultRingSize())
            .add("autoDeleteCreatedQueues", addressSettings.isAutoDeleteCreatedQueues())
            .add("autoDeleteQueuesDelay", addressSettings.getAutoDeleteQueuesDelay())
            .add("autoDeleteQueuesMessageCount", addressSettings.getAutoDeleteQueuesMessageCount())
            .add("autoDeleteAddressesDelay", addressSettings.getAutoDeleteAddressesDelay())
            .add("redeliveryCollisionAvoidanceFactor", addressSettings.getRedeliveryCollisionAvoidanceFactor())
            .add("retroactiveMessageCount", addressSettings.getRetroactiveMessageCount())
            .add("autoCreateDeadLetterResources", addressSettings.isAutoCreateDeadLetterResources())
            .add("deadLetterQueuePrefix", addressSettings.getDeadLetterQueuePrefix().toString())
            .add("deadLetterQueueSuffix", addressSettings.getDeadLetterQueueSuffix().toString())
            .add("autoCreateExpiryResources", addressSettings.isAutoCreateExpiryResources())
            .add("expiryQueuePrefix", addressSettings.getExpiryQueuePrefix().toString())
            .add("expiryQueueSuffix", addressSettings.getExpiryQueueSuffix().toString())
            .add("enableMetrics", addressSettings.isEnableMetrics())
            .build()
            .toString();
   }

   @Override
   public void addAddressSettings(final String address,
                                  final String DLA,
                                  final String expiryAddress,
                                  final long expiryDelay,
                                  final boolean lastValueQueue,
                                  final int deliveryAttempts,
                                  final long maxSizeBytes,
                                  final int pageSizeBytes,
                                  final int pageMaxCacheSize,
                                  final long redeliveryDelay,
                                  final double redeliveryMultiplier,
                                  final long maxRedeliveryDelay,
                                  final long redistributionDelay,
                                  final boolean sendToDLAOnNoRoute,
                                  final String addressFullMessagePolicy,
                                  final long slowConsumerThreshold,
                                  final long slowConsumerCheckPeriod,
                                  final String slowConsumerPolicy,
                                  final boolean autoCreateJmsQueues,
                                  final boolean autoDeleteJmsQueues,
                                  final boolean autoCreateJmsTopics,
                                  final boolean autoDeleteJmsTopics) throws Exception {
      addAddressSettings(address,
                         DLA,
                         expiryAddress,
                         expiryDelay,
                         lastValueQueue,
                         deliveryAttempts,
                         maxSizeBytes,
                         pageSizeBytes,
                         pageMaxCacheSize,
                         redeliveryDelay,
                         redeliveryMultiplier,
                         maxRedeliveryDelay,
                         redistributionDelay,
                         sendToDLAOnNoRoute,
                         addressFullMessagePolicy,
                         slowConsumerThreshold,
                         slowConsumerCheckPeriod,
                         slowConsumerPolicy,
                         autoCreateJmsQueues,
                         autoDeleteJmsQueues,
                         autoCreateJmsTopics,
                         autoDeleteJmsTopics,
                         AddressSettings.DEFAULT_AUTO_CREATE_QUEUES,
                         AddressSettings.DEFAULT_AUTO_DELETE_QUEUES,
                         AddressSettings.DEFAULT_AUTO_CREATE_ADDRESSES,
                         AddressSettings.DEFAULT_AUTO_DELETE_ADDRESSES);
   }

   @Override
   public void addAddressSettings(final String address,
                                  final String DLA,
                                  final String expiryAddress,
                                  final long expiryDelay,
                                  final boolean defaultLastValueQueue,
                                  final int maxDeliveryAttempts,
                                  final long maxSizeBytes,
                                  final int pageSizeBytes,
                                  final int pageMaxCacheSize,
                                  final long redeliveryDelay,
                                  final double redeliveryMultiplier,
                                  final long maxRedeliveryDelay,
                                  final long redistributionDelay,
                                  final boolean sendToDLAOnNoRoute,
                                  final String addressFullMessagePolicy,
                                  final long slowConsumerThreshold,
                                  final long slowConsumerCheckPeriod,
                                  final String slowConsumerPolicy,
                                  final boolean autoCreateJmsQueues,
                                  final boolean autoDeleteJmsQueues,
                                  final boolean autoCreateJmsTopics,
                                  final boolean autoDeleteJmsTopics,
                                  final boolean autoCreateQueues,
                                  final boolean autoDeleteQueues,
                                  final boolean autoCreateAddresses,
                                  final boolean autoDeleteAddresses) throws Exception {
      addAddressSettings(address,
                         DLA,
                         expiryAddress,
                         expiryDelay,
                         defaultLastValueQueue,
                         maxDeliveryAttempts,
                         maxSizeBytes,
                         pageSizeBytes,
                         pageMaxCacheSize,
                         redeliveryDelay,
                         redeliveryMultiplier,
                         maxRedeliveryDelay,
                         redistributionDelay,
                         sendToDLAOnNoRoute,
                         addressFullMessagePolicy,
                         slowConsumerThreshold,
                         slowConsumerCheckPeriod,
                         slowConsumerPolicy,
                         autoCreateJmsQueues,
                         autoDeleteJmsQueues,
                         autoCreateJmsTopics,
                         autoDeleteJmsTopics,
                         autoCreateQueues,
                         autoDeleteQueues,
                         autoCreateAddresses,
                         autoDeleteAddresses,
                         AddressSettings.DEFAULT_CONFIG_DELETE_QUEUES.toString(),
                         AddressSettings.DEFAULT_CONFIG_DELETE_ADDRESSES.toString(),
                         AddressSettings.DEFAULT_ADDRESS_REJECT_THRESHOLD,
                         ActiveMQDefaultConfiguration.getDefaultLastValueKey() == null ? null : ActiveMQDefaultConfiguration.getDefaultLastValueKey().toString(),
                         ActiveMQDefaultConfiguration.getDefaultNonDestructive(),
                         ActiveMQDefaultConfiguration.getDefaultExclusive(),
                         ActiveMQDefaultConfiguration.getDefaultGroupRebalance(),
                         ActiveMQDefaultConfiguration.getDefaultGroupBuckets(),
                         ActiveMQDefaultConfiguration.getDefaultGroupFirstKey() == null ? null : ActiveMQDefaultConfiguration.getDefaultGroupFirstKey().toString(),
                         ActiveMQDefaultConfiguration.getDefaultMaxQueueConsumers(),
                         ActiveMQDefaultConfiguration.getDefaultPurgeOnNoConsumers(),
                         ActiveMQDefaultConfiguration.getDefaultConsumersBeforeDispatch(),
                         ActiveMQDefaultConfiguration.getDefaultDelayBeforeDispatch(),
                         ActiveMQDefaultConfiguration.getDefaultRoutingType().toString(),
                         ActiveMQDefaultConfiguration.getDefaultRoutingType().toString(),
                         ActiveMQClient.DEFAULT_CONSUMER_WINDOW_SIZE,
                         ActiveMQDefaultConfiguration.getDefaultRingSize(),
                         AddressSettings.DEFAULT_AUTO_DELETE_CREATED_QUEUES,
                         AddressSettings.DEFAULT_AUTO_DELETE_QUEUES_DELAY,
                         AddressSettings.DEFAULT_AUTO_DELETE_QUEUES_MESSAGE_COUNT,
                         AddressSettings.DEFAULT_AUTO_DELETE_ADDRESSES_DELAY,
                         AddressSettings.DEFAULT_REDELIVER_COLLISION_AVOIDANCE_FACTOR,
                         ActiveMQDefaultConfiguration.getDefaultRetroactiveMessageCount());
   }

   @Override
   public void addAddressSettings(final String address,
                                  final String DLA,
                                  final String expiryAddress,
                                  final long expiryDelay,
                                  final boolean defaultLastValueQueue,
                                  final int maxDeliveryAttempts,
                                  final long maxSizeBytes,
                                  final int pageSizeBytes,
                                  final int pageMaxCacheSize,
                                  final long redeliveryDelay,
                                  final double redeliveryMultiplier,
                                  final long maxRedeliveryDelay,
                                  final long redistributionDelay,
                                  final boolean sendToDLAOnNoRoute,
                                  final String addressFullMessagePolicy,
                                  final long slowConsumerThreshold,
                                  final long slowConsumerCheckPeriod,
                                  final String slowConsumerPolicy,
                                  final boolean autoCreateJmsQueues,
                                  final boolean autoDeleteJmsQueues,
                                  final boolean autoCreateJmsTopics,
                                  final boolean autoDeleteJmsTopics,
                                  final boolean autoCreateQueues,
                                  final boolean autoDeleteQueues,
                                  final boolean autoCreateAddresses,
                                  final boolean autoDeleteAddresses,
                                  final String configDeleteQueues,
                                  final String configDeleteAddresses,
                                  final long maxSizeBytesRejectThreshold,
                                  final String defaultLastValueKey,
                                  final boolean defaultNonDestructive,
                                  final boolean defaultExclusiveQueue,
                                  final boolean defaultGroupRebalance,
                                  final int defaultGroupBuckets,
                                  final String defaultGroupFirstKey,
                                  final int defaultMaxConsumers,
                                  final boolean defaultPurgeOnNoConsumers,
                                  final int defaultConsumersBeforeDispatch,
                                  final long defaultDelayBeforeDispatch,
                                  final String defaultQueueRoutingType,
                                  final String defaultAddressRoutingType,
                                  final int defaultConsumerWindowSize,
                                  final long defaultRingSize,
                                  final boolean autoDeleteCreatedQueues,
                                  final long autoDeleteQueuesDelay,
                                  final long autoDeleteQueuesMessageCount,
                                  final long autoDeleteAddressesDelay,
                                  final double redeliveryCollisionAvoidanceFactor,
                                  final long retroactiveMessageCount) throws Exception {
      addAddressSettings(address,
                         DLA,
                         expiryAddress,
                         expiryDelay,
                         defaultLastValueQueue,
                         maxDeliveryAttempts,
                         maxSizeBytes,
                         pageSizeBytes,
                         pageMaxCacheSize,
                         redeliveryDelay,
                         redeliveryMultiplier,
                         maxRedeliveryDelay,
                         redistributionDelay,
                         sendToDLAOnNoRoute,
                         addressFullMessagePolicy,
                         slowConsumerThreshold,
                         slowConsumerCheckPeriod,
                         slowConsumerPolicy,
                         autoCreateJmsQueues,
                         autoDeleteJmsQueues,
                         autoCreateJmsTopics,
                         autoDeleteJmsTopics,
                         autoCreateQueues,
                         autoDeleteQueues,
                         autoCreateAddresses,
                         autoDeleteAddresses,
                         configDeleteQueues,
                         configDeleteAddresses,
                         maxSizeBytesRejectThreshold,
                         defaultLastValueKey,
                         defaultNonDestructive,
                         defaultExclusiveQueue,
                         defaultGroupRebalance,
                         defaultGroupBuckets,
                         defaultGroupFirstKey,
                         defaultMaxConsumers,
                         defaultPurgeOnNoConsumers,
                         defaultConsumersBeforeDispatch,
                         defaultDelayBeforeDispatch,
                         defaultQueueRoutingType,
                         defaultAddressRoutingType,
                         defaultConsumerWindowSize,
                         defaultRingSize,
                         autoDeleteCreatedQueues,
                         autoDeleteQueuesDelay,
                         autoDeleteQueuesMessageCount,
                         autoDeleteAddressesDelay,
                         redeliveryCollisionAvoidanceFactor,
                         retroactiveMessageCount,
                         AddressSettings.DEFAULT_AUTO_CREATE_DEAD_LETTER_RESOURCES,
                         AddressSettings.DEFAULT_DEAD_LETTER_QUEUE_PREFIX.toString(),
                         AddressSettings.DEFAULT_DEAD_LETTER_QUEUE_SUFFIX.toString(),
                         AddressSettings.DEFAULT_AUTO_CREATE_EXPIRY_RESOURCES,
                         AddressSettings.DEFAULT_EXPIRY_QUEUE_PREFIX.toString(),
                         AddressSettings.DEFAULT_EXPIRY_QUEUE_SUFFIX.toString());
   }

   @Override
   public void addAddressSettings(final String address,
                                  final String DLA,
                                  final String expiryAddress,
                                  final long expiryDelay,
                                  final boolean defaultLastValueQueue,
                                  final int maxDeliveryAttempts,
                                  final long maxSizeBytes,
                                  final int pageSizeBytes,
                                  final int pageMaxCacheSize,
                                  final long redeliveryDelay,
                                  final double redeliveryMultiplier,
                                  final long maxRedeliveryDelay,
                                  final long redistributionDelay,
                                  final boolean sendToDLAOnNoRoute,
                                  final String addressFullMessagePolicy,
                                  final long slowConsumerThreshold,
                                  final long slowConsumerCheckPeriod,
                                  final String slowConsumerPolicy,
                                  final boolean autoCreateJmsQueues,
                                  final boolean autoDeleteJmsQueues,
                                  final boolean autoCreateJmsTopics,
                                  final boolean autoDeleteJmsTopics,
                                  final boolean autoCreateQueues,
                                  final boolean autoDeleteQueues,
                                  final boolean autoCreateAddresses,
                                  final boolean autoDeleteAddresses,
                                  final String configDeleteQueues,
                                  final String configDeleteAddresses,
                                  final long maxSizeBytesRejectThreshold,
                                  final String defaultLastValueKey,
                                  final boolean defaultNonDestructive,
                                  final boolean defaultExclusiveQueue,
                                  final boolean defaultGroupRebalance,
                                  final int defaultGroupBuckets,
                                  final String defaultGroupFirstKey,
                                  final int defaultMaxConsumers,
                                  final boolean defaultPurgeOnNoConsumers,
                                  final int defaultConsumersBeforeDispatch,
                                  final long defaultDelayBeforeDispatch,
                                  final String defaultQueueRoutingType,
                                  final String defaultAddressRoutingType,
                                  final int defaultConsumerWindowSize,
                                  final long defaultRingSize,
                                  final boolean autoDeleteCreatedQueues,
                                  final long autoDeleteQueuesDelay,
                                  final long autoDeleteQueuesMessageCount,
                                  final long autoDeleteAddressesDelay,
                                  final double redeliveryCollisionAvoidanceFactor,
                                  final long retroactiveMessageCount,
                                  final boolean autoCreateDeadLetterResources,
                                  final String deadLetterQueuePrefix,
                                  final String deadLetterQueueSuffix,
                                  final boolean autoCreateExpiryResources,
                                  final String expiryQueuePrefix,
                                  final String expiryQueueSuffix) throws Exception {
      addAddressSettings(address,
                         DLA,
                         expiryAddress,
                         expiryDelay,
                         defaultLastValueQueue,
                         maxDeliveryAttempts,
                         maxSizeBytes,
                         pageSizeBytes,
                         pageMaxCacheSize,
                         redeliveryDelay,
                         redeliveryMultiplier,
                         maxRedeliveryDelay,
                         redistributionDelay,
                         sendToDLAOnNoRoute,
                         addressFullMessagePolicy,
                         slowConsumerThreshold,
                         slowConsumerCheckPeriod,
                         slowConsumerPolicy,
                         autoCreateJmsQueues,
                         autoDeleteJmsQueues,
                         autoCreateJmsTopics,
                         autoDeleteJmsTopics,
                         autoCreateQueues,
                         autoDeleteQueues,
                         autoCreateAddresses,
                         autoDeleteAddresses,
                         configDeleteQueues,
                         configDeleteAddresses,
                         maxSizeBytesRejectThreshold,
                         defaultLastValueKey,
                         defaultNonDestructive,
                         defaultExclusiveQueue,
                         defaultGroupRebalance,
                         defaultGroupBuckets,
                         defaultGroupFirstKey,
                         defaultMaxConsumers,
                         defaultPurgeOnNoConsumers,
                         defaultConsumersBeforeDispatch,
                         defaultDelayBeforeDispatch,
                         defaultQueueRoutingType,
                         defaultAddressRoutingType,
                         defaultConsumerWindowSize,
                         defaultRingSize,
                         autoDeleteCreatedQueues,
                         autoDeleteQueuesDelay,
                         autoDeleteQueuesMessageCount,
                         autoDeleteAddressesDelay,
                         redeliveryCollisionAvoidanceFactor,
                         retroactiveMessageCount,
                         AddressSettings.DEFAULT_AUTO_CREATE_DEAD_LETTER_RESOURCES,
                         AddressSettings.DEFAULT_DEAD_LETTER_QUEUE_PREFIX.toString(),
                         AddressSettings.DEFAULT_DEAD_LETTER_QUEUE_SUFFIX.toString(),
                         AddressSettings.DEFAULT_AUTO_CREATE_EXPIRY_RESOURCES,
                         AddressSettings.DEFAULT_EXPIRY_QUEUE_PREFIX.toString(),
                         AddressSettings.DEFAULT_EXPIRY_QUEUE_SUFFIX.toString(),
                         AddressSettings.DEFAULT_MIN_EXPIRY_DELAY,
                         AddressSettings.DEFAULT_MAX_EXPIRY_DELAY,
                         AddressSettings.DEFAULT_ENABLE_METRICS);
   }

   @Override
   public void addAddressSettings(final String address,
                                  final String DLA,
                                  final String expiryAddress,
                                  final long expiryDelay,
                                  final boolean defaultLastValueQueue,
                                  final int maxDeliveryAttempts,
                                  final long maxSizeBytes,
                                  final int pageSizeBytes,
                                  final int pageMaxCacheSize,
                                  final long redeliveryDelay,
                                  final double redeliveryMultiplier,
                                  final long maxRedeliveryDelay,
                                  final long redistributionDelay,
                                  final boolean sendToDLAOnNoRoute,
                                  final String addressFullMessagePolicy,
                                  final long slowConsumerThreshold,
                                  final long slowConsumerCheckPeriod,
                                  final String slowConsumerPolicy,
                                  final boolean autoCreateJmsQueues,
                                  final boolean autoDeleteJmsQueues,
                                  final boolean autoCreateJmsTopics,
                                  final boolean autoDeleteJmsTopics,
                                  final boolean autoCreateQueues,
                                  final boolean autoDeleteQueues,
                                  final boolean autoCreateAddresses,
                                  final boolean autoDeleteAddresses,
                                  final String configDeleteQueues,
                                  final String configDeleteAddresses,
                                  final long maxSizeBytesRejectThreshold,
                                  final String defaultLastValueKey,
                                  final boolean defaultNonDestructive,
                                  final boolean defaultExclusiveQueue,
                                  final boolean defaultGroupRebalance,
                                  final int defaultGroupBuckets,
                                  final String defaultGroupFirstKey,
                                  final int defaultMaxConsumers,
                                  final boolean defaultPurgeOnNoConsumers,
                                  final int defaultConsumersBeforeDispatch,
                                  final long defaultDelayBeforeDispatch,
                                  final String defaultQueueRoutingType,
                                  final String defaultAddressRoutingType,
                                  final int defaultConsumerWindowSize,
                                  final long defaultRingSize,
                                  final boolean autoDeleteCreatedQueues,
                                  final long autoDeleteQueuesDelay,
                                  final long autoDeleteQueuesMessageCount,
                                  final long autoDeleteAddressesDelay,
                                  final double redeliveryCollisionAvoidanceFactor,
                                  final long retroactiveMessageCount,
                                  final boolean autoCreateDeadLetterResources,
                                  final String deadLetterQueuePrefix,
                                  final String deadLetterQueueSuffix,
                                  final boolean autoCreateExpiryResources,
                                  final String expiryQueuePrefix,
                                  final String expiryQueueSuffix,
                                  final long minExpiryDelay,
                                  final long maxExpiryDelay,
                                  final boolean enableMetrics) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.addAddressSettings(this.server, address, DLA, expiryAddress, expiryDelay, defaultLastValueQueue, maxDeliveryAttempts,
                  maxSizeBytes, pageSizeBytes, pageMaxCacheSize, redeliveryDelay, redeliveryMultiplier,
                  maxRedeliveryDelay, redistributionDelay, sendToDLAOnNoRoute, addressFullMessagePolicy,
                  slowConsumerThreshold, slowConsumerCheckPeriod, slowConsumerPolicy, autoCreateJmsQueues,
                  autoDeleteJmsQueues, autoCreateJmsTopics, autoDeleteJmsTopics, autoCreateQueues, autoDeleteQueues,
                  autoCreateAddresses, autoDeleteAddresses, configDeleteQueues, configDeleteAddresses, maxSizeBytesRejectThreshold,
                  defaultLastValueKey, defaultNonDestructive, defaultExclusiveQueue, defaultGroupRebalance, defaultGroupBuckets,
                  defaultGroupFirstKey, defaultMaxConsumers, defaultPurgeOnNoConsumers, defaultConsumersBeforeDispatch,
                  defaultDelayBeforeDispatch, defaultQueueRoutingType, defaultAddressRoutingType, defaultConsumerWindowSize,
                  defaultRingSize, autoDeleteCreatedQueues, autoDeleteQueuesDelay, autoDeleteQueuesMessageCount,
                  autoDeleteAddressesDelay, redeliveryCollisionAvoidanceFactor, retroactiveMessageCount, autoCreateDeadLetterResources,
                  deadLetterQueuePrefix, deadLetterQueueSuffix, autoCreateExpiryResources, expiryQueuePrefix,
                  expiryQueueSuffix, minExpiryDelay, maxExpiryDelay, enableMetrics);
      }
      checkStarted();

      // JBPAPP-6334 requested this to be pageSizeBytes > maxSizeBytes
      if (pageSizeBytes > maxSizeBytes && maxSizeBytes > 0) {
         throw new IllegalStateException("pageSize has to be lower than maxSizeBytes. Invalid argument (" + pageSizeBytes + " < " + maxSizeBytes + ")");
      }

      if (maxSizeBytes < -1) {
         throw new IllegalStateException("Invalid argument on maxSizeBytes");
      }

      AddressSettings addressSettings = new AddressSettings();
      addressSettings.setDeadLetterAddress(DLA == null ? null : new SimpleString(DLA));
      addressSettings.setExpiryAddress(expiryAddress == null ? null : new SimpleString(expiryAddress));
      addressSettings.setExpiryDelay(expiryDelay);
      addressSettings.setMinExpiryDelay(minExpiryDelay);
      addressSettings.setMaxExpiryDelay(maxExpiryDelay);
      addressSettings.setDefaultLastValueQueue(defaultLastValueQueue);
      addressSettings.setMaxDeliveryAttempts(maxDeliveryAttempts);
      addressSettings.setPageCacheMaxSize(pageMaxCacheSize);
      addressSettings.setMaxSizeBytes(maxSizeBytes);
      addressSettings.setPageSizeBytes(pageSizeBytes);
      addressSettings.setRedeliveryDelay(redeliveryDelay);
      addressSettings.setRedeliveryMultiplier(redeliveryMultiplier);
      addressSettings.setMaxRedeliveryDelay(maxRedeliveryDelay);
      addressSettings.setRedistributionDelay(redistributionDelay);
      addressSettings.setSendToDLAOnNoRoute(sendToDLAOnNoRoute);
      addressSettings.setAddressFullMessagePolicy(addressFullMessagePolicy == null ? AddressSettings.DEFAULT_ADDRESS_FULL_MESSAGE_POLICY : AddressFullMessagePolicy.valueOf(addressFullMessagePolicy.toUpperCase()));
      addressSettings.setSlowConsumerThreshold(slowConsumerThreshold);
      addressSettings.setSlowConsumerCheckPeriod(slowConsumerCheckPeriod);
      addressSettings.setSlowConsumerPolicy(slowConsumerPolicy == null ? AddressSettings.DEFAULT_SLOW_CONSUMER_POLICY : SlowConsumerPolicy.valueOf(slowConsumerPolicy.toUpperCase()));
      addressSettings.setAutoCreateJmsQueues(autoCreateJmsQueues);
      addressSettings.setAutoDeleteJmsQueues(autoDeleteJmsQueues);
      addressSettings.setAutoCreateJmsTopics(autoCreateJmsTopics);
      addressSettings.setAutoDeleteJmsTopics(autoDeleteJmsTopics);
      addressSettings.setAutoCreateQueues(autoCreateQueues);
      addressSettings.setAutoDeleteQueues(autoDeleteQueues);
      addressSettings.setAutoCreateAddresses(autoCreateAddresses);
      addressSettings.setAutoDeleteAddresses(autoDeleteAddresses);
      addressSettings.setConfigDeleteQueues(configDeleteQueues == null ? AddressSettings.DEFAULT_CONFIG_DELETE_QUEUES : DeletionPolicy.valueOf(configDeleteQueues.toUpperCase()));
      addressSettings.setConfigDeleteAddresses(configDeleteAddresses == null ? AddressSettings.DEFAULT_CONFIG_DELETE_ADDRESSES : DeletionPolicy.valueOf(configDeleteAddresses.toUpperCase()));
      addressSettings.setMaxSizeBytesRejectThreshold(maxSizeBytesRejectThreshold);
      addressSettings.setDefaultLastValueKey(defaultLastValueKey == null ? ActiveMQDefaultConfiguration.getDefaultLastValueKey() : new SimpleString(defaultLastValueKey));
      addressSettings.setDefaultNonDestructive(defaultNonDestructive);
      addressSettings.setDefaultExclusiveQueue(defaultExclusiveQueue);
      addressSettings.setDefaultGroupRebalance(defaultGroupRebalance);
      addressSettings.setDefaultGroupBuckets(defaultGroupBuckets);
      addressSettings.setDefaultGroupFirstKey(defaultGroupFirstKey == null ? ActiveMQDefaultConfiguration.getDefaultGroupFirstKey() : new SimpleString(defaultGroupFirstKey));
      addressSettings.setDefaultMaxConsumers(defaultMaxConsumers);
      addressSettings.setDefaultPurgeOnNoConsumers(defaultPurgeOnNoConsumers);
      addressSettings.setDefaultConsumersBeforeDispatch(defaultConsumersBeforeDispatch);
      addressSettings.setDefaultDelayBeforeDispatch(defaultDelayBeforeDispatch);
      addressSettings.setDefaultQueueRoutingType(defaultQueueRoutingType == null ? ActiveMQDefaultConfiguration.getDefaultRoutingType() : RoutingType.valueOf(defaultQueueRoutingType.toUpperCase()));
      addressSettings.setDefaultAddressRoutingType(defaultAddressRoutingType == null ? ActiveMQDefaultConfiguration.getDefaultRoutingType() : RoutingType.valueOf(defaultAddressRoutingType.toUpperCase()));
      addressSettings.setDefaultConsumerWindowSize(defaultConsumerWindowSize);
      addressSettings.setDefaultRingSize(defaultRingSize);
      addressSettings.setAutoDeleteCreatedQueues(autoDeleteCreatedQueues);
      addressSettings.setAutoDeleteQueuesDelay(autoDeleteQueuesDelay);
      addressSettings.setAutoDeleteQueuesMessageCount(autoDeleteQueuesMessageCount);
      addressSettings.setAutoDeleteAddressesDelay(autoDeleteAddressesDelay);
      addressSettings.setRedeliveryCollisionAvoidanceFactor(redeliveryCollisionAvoidanceFactor);
      addressSettings.setRetroactiveMessageCount(retroactiveMessageCount);
      addressSettings.setAutoCreateDeadLetterResources(autoCreateDeadLetterResources);
      addressSettings.setDeadLetterQueuePrefix(deadLetterQueuePrefix == null ? null : new SimpleString(deadLetterQueuePrefix));
      addressSettings.setDeadLetterQueueSuffix(deadLetterQueueSuffix == null ? null : new SimpleString(deadLetterQueueSuffix));
      addressSettings.setAutoCreateExpiryResources(autoCreateExpiryResources);
      addressSettings.setExpiryQueuePrefix(expiryQueuePrefix == null ? null : new SimpleString(expiryQueuePrefix));
      addressSettings.setExpiryQueueSuffix(expiryQueueSuffix == null ? null : new SimpleString(expiryQueueSuffix));
      addressSettings.setEnableMetrics(enableMetrics);

      server.getAddressSettingsRepository().addMatch(address, addressSettings);

      storageManager.storeAddressSetting(new PersistedAddressSetting(new SimpleString(address), addressSettings));

   }

   @Override
   public void removeAddressSettings(final String addressMatch) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.removeAddressSettings(this.server, addressMatch);
      }
      checkStarted();

      server.getAddressSettingsRepository().removeMatch(addressMatch);
      storageManager.deleteAddressSetting(new SimpleString(addressMatch));
   }

   public void sendQueueInfoToQueue(final String queueName, final String address) throws Exception {
      checkStarted();

      clearIO();
      try {
         postOffice.sendQueueInfoToQueue(new SimpleString(queueName), new SimpleString(address == null ? "" : address));

         GroupingHandler handler = server.getGroupingHandler();
         if (handler != null) {
            // the group handler would miss responses if the group was requested before the reset was done
            // on that case we ask the groupinghandler to replay its send in case it's waiting for the information
            handler.resendPending();
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getDivertNames() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getDivertNames(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Object[] diverts = server.getManagementService().getResources(DivertControl.class);
         String[] names = new String[diverts.length];
         for (int i = 0; i < diverts.length; i++) {
            DivertControl divert = (DivertControl) diverts[i];
            names[i] = divert.getUniqueName();
         }

         return names;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void createDivert(final String name,
                            final String routingName,
                            final String address,
                            final String forwardingAddress,
                            final boolean exclusive,
                            final String filterString,
                            final String transformerClassName) throws Exception {
      createDivert(name, routingName, address, forwardingAddress, exclusive, filterString, transformerClassName, ActiveMQDefaultConfiguration.getDefaultDivertRoutingType());
   }

   @Override
   public void createDivert(final String name,
                            final String routingName,
                            final String address,
                            final String forwardingAddress,
                            final boolean exclusive,
                            final String filterString,
                            final String transformerClassName,
                            final String routingType) throws Exception {
      createDivert(name, routingName, address, forwardingAddress, exclusive, filterString, transformerClassName, (String) null, routingType);
   }

   @Override
   public void createDivert(final String name,
                            final String routingName,
                            final String address,
                            final String forwardingAddress,
                            final boolean exclusive,
                            final String filterString,
                            final String transformerClassName,
                            final String transformerPropertiesAsJSON,
                            final String routingType) throws Exception {
      createDivert(name, routingName, address, forwardingAddress, exclusive, filterString, transformerClassName, JsonUtil.readJsonProperties(transformerPropertiesAsJSON), routingType);
   }

   @Override
   public void createDivert(final String name,
                            final String routingName,
                            final String address,
                            final String forwardingAddress,
                            final boolean exclusive,
                            final String filterString,
                            final String transformerClassName,
                            final Map<String, String> transformerProperties,
                            final String routingType) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.createDivert(this.server, name, routingName, address, forwardingAddress,
                  exclusive, filterString, transformerClassName, transformerProperties, routingType);
      }
      checkStarted();

      clearIO();
      try {
         TransformerConfiguration transformerConfiguration = transformerClassName == null ? null : new TransformerConfiguration(transformerClassName).setProperties(transformerProperties);
         DivertConfiguration config = new DivertConfiguration().setName(name).setRoutingName(routingName).setAddress(address).setForwardingAddress(forwardingAddress).setExclusive(exclusive).setFilterString(filterString).setTransformerConfiguration(transformerConfiguration).setRoutingType(ComponentConfigurationRoutingType.valueOf(routingType));
         server.deployDivert(config);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void updateDivert(final String name,
                            final String forwardingAddress,
                            final String filterString,
                            final String transformerClassName,
                            final Map<String, String> transformerProperties,
                            final String routingType) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.updateDivert(this.server, name, forwardingAddress, filterString,
                                  transformerClassName, transformerProperties, routingType);
      }
      checkStarted();

      clearIO();

      try {
         TransformerConfiguration transformerConfiguration = transformerClassName == null ? null :
            new TransformerConfiguration(transformerClassName).setProperties(transformerProperties);

         DivertConfiguration config = new DivertConfiguration().setName(name).setForwardingAddress(forwardingAddress).
            setFilterString(filterString).setTransformerConfiguration(transformerConfiguration).
            setRoutingType(ComponentConfigurationRoutingType.valueOf(routingType));

         final Divert divert = server.updateDivert(config);
         if (divert == null) {
            throw ActiveMQMessageBundle.BUNDLE.divertDoesNotExist(config.getName());
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void destroyDivert(final String name) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.destroyDivert(this.server, name);
      }
      checkStarted();

      clearIO();
      try {
         server.destroyDivert(SimpleString.toSimpleString(name));
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getBridgeNames() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getBridgeNames(this.server);
      }
      checkStarted();

      clearIO();
      try {
         Object[] bridges = server.getManagementService().getResources(BridgeControl.class);
         String[] names = new String[bridges.length];
         for (int i = 0; i < bridges.length; i++) {
            BridgeControl bridge = (BridgeControl) bridges[i];
            names[i] = bridge.getName();
         }

         return names;
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void createBridge(final String name,
                            final String queueName,
                            final String forwardingAddress,
                            final String filterString,
                            final String transformerClassName,
                            final long retryInterval,
                            final double retryIntervalMultiplier,
                            final int initialConnectAttempts,
                            final int reconnectAttempts,
                            final boolean useDuplicateDetection,
                            final int confirmationWindowSize,
                            final int producerWindowSize,
                            final long clientFailureCheckPeriod,
                            final String staticConnectorsOrDiscoveryGroup,
                            boolean useDiscoveryGroup,
                            final boolean ha,
                            final String user,
                            final String password) throws Exception {
      createBridge(name,
                   queueName,
                   forwardingAddress,
                   filterString,
                   transformerClassName,
                   (String) null,
                   retryInterval,
                   retryIntervalMultiplier,
                   initialConnectAttempts,
                   reconnectAttempts,
                   useDuplicateDetection,
                   confirmationWindowSize,
                   producerWindowSize,
                   clientFailureCheckPeriod,
                   staticConnectorsOrDiscoveryGroup,
                   useDiscoveryGroup,
                   ha,
                   user,
                   password);
   }

   @Override
   public void createBridge(final String name,
                            final String queueName,
                            final String forwardingAddress,
                            final String filterString,
                            final String transformerClassName,
                            final String transformerPropertiesAsJSON,
                            final long retryInterval,
                            final double retryIntervalMultiplier,
                            final int initialConnectAttempts,
                            final int reconnectAttempts,
                            final boolean useDuplicateDetection,
                            final int confirmationWindowSize,
                            final int producerWindowSize,
                            final long clientFailureCheckPeriod,
                            final String staticConnectorsOrDiscoveryGroup,
                            boolean useDiscoveryGroup,
                            final boolean ha,
                            final String user,
                            final String password) throws Exception {
      createBridge(name,
                   queueName,
                   forwardingAddress,
                   filterString,
                   transformerClassName,
                   JsonUtil.readJsonProperties(transformerPropertiesAsJSON),
                   retryInterval,
                   retryIntervalMultiplier,
                   initialConnectAttempts,
                   reconnectAttempts,
                   useDuplicateDetection,
                   confirmationWindowSize,
                   producerWindowSize,
                   clientFailureCheckPeriod,
                   staticConnectorsOrDiscoveryGroup,
                   useDiscoveryGroup,
                   ha,
                   user,
                   password);
   }

   @Override
   public void createBridge(final String name,
                            final String queueName,
                            final String forwardingAddress,
                            final String filterString,
                            final String transformerClassName,
                            final Map<String, String> transformerProperties,
                            final long retryInterval,
                            final double retryIntervalMultiplier,
                            final int initialConnectAttempts,
                            final int reconnectAttempts,
                            final boolean useDuplicateDetection,
                            final int confirmationWindowSize,
                            final int producerWindowSize,
                            final long clientFailureCheckPeriod,
                            final String staticConnectorsOrDiscoveryGroup,
                            boolean useDiscoveryGroup,
                            final boolean ha,
                            final String user,
                            final String password) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.createBridge(this.server, name, queueName, forwardingAddress, filterString,
                  transformerClassName, transformerProperties, retryInterval, retryIntervalMultiplier,
                  initialConnectAttempts, reconnectAttempts, useDuplicateDetection, confirmationWindowSize,
                  producerWindowSize, clientFailureCheckPeriod, staticConnectorsOrDiscoveryGroup,
                  useDiscoveryGroup, ha, user, "****");
      }
      checkStarted();

      clearIO();

      try {
         TransformerConfiguration transformerConfiguration = transformerClassName == null ? null : new TransformerConfiguration(transformerClassName).setProperties(transformerProperties);
         BridgeConfiguration config = new BridgeConfiguration().setName(name).setQueueName(queueName).setForwardingAddress(forwardingAddress).setFilterString(filterString).setTransformerConfiguration(transformerConfiguration).setClientFailureCheckPeriod(clientFailureCheckPeriod).setRetryInterval(retryInterval).setRetryIntervalMultiplier(retryIntervalMultiplier).setInitialConnectAttempts(initialConnectAttempts).setReconnectAttempts(reconnectAttempts).setUseDuplicateDetection(useDuplicateDetection).setConfirmationWindowSize(confirmationWindowSize).setProducerWindowSize(producerWindowSize).setHA(ha).setUser(user).setPassword(password);

         if (useDiscoveryGroup) {
            config.setDiscoveryGroupName(staticConnectorsOrDiscoveryGroup);
         } else {
            config.setStaticConnectors(ListUtil.toList(staticConnectorsOrDiscoveryGroup));
         }

         server.deployBridge(config);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void createBridge(final String name,
                            final String queueName,
                            final String forwardingAddress,
                            final String filterString,
                            final String transformerClassName,
                            final long retryInterval,
                            final double retryIntervalMultiplier,
                            final int initialConnectAttempts,
                            final int reconnectAttempts,
                            final boolean useDuplicateDetection,
                            final int confirmationWindowSize,
                            final long clientFailureCheckPeriod,
                            final String staticConnectorsOrDiscoveryGroup,
                            boolean useDiscoveryGroup,
                            final boolean ha,
                            final String user,
                            final String password) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.createBridge(this.server, name, queueName, forwardingAddress, filterString,
                  transformerClassName, retryInterval, retryIntervalMultiplier, initialConnectAttempts,
                  reconnectAttempts, useDuplicateDetection, confirmationWindowSize, clientFailureCheckPeriod,
                  staticConnectorsOrDiscoveryGroup, useDiscoveryGroup, ha, user, "****");
      }
      checkStarted();

      clearIO();

      try {
         TransformerConfiguration transformerConfiguration = transformerClassName == null ? null : new TransformerConfiguration(transformerClassName);
         BridgeConfiguration config = new BridgeConfiguration().setName(name).setQueueName(queueName).setForwardingAddress(forwardingAddress).setFilterString(filterString).setTransformerConfiguration(transformerConfiguration).setClientFailureCheckPeriod(clientFailureCheckPeriod).setRetryInterval(retryInterval).setRetryIntervalMultiplier(retryIntervalMultiplier).setInitialConnectAttempts(initialConnectAttempts).setReconnectAttempts(reconnectAttempts).setUseDuplicateDetection(useDuplicateDetection).setConfirmationWindowSize(confirmationWindowSize).setHA(ha).setUser(user).setPassword(password);

         if (useDiscoveryGroup) {
            config.setDiscoveryGroupName(staticConnectorsOrDiscoveryGroup);
         } else {
            config.setStaticConnectors(ListUtil.toList(staticConnectorsOrDiscoveryGroup));
         }

         server.deployBridge(config);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void destroyBridge(final String name) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.destroyBridge(this.server, name);
      }
      checkStarted();

      clearIO();
      try {
         server.destroyBridge(name);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void createConnectorService(final String name, final String factoryClass, final Map<String, Object> parameters) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.createConnectorService(this.server, name, factoryClass, parameters);
      }
      checkStarted();

      clearIO();

      try {
         final ConnectorServiceConfiguration config = new ConnectorServiceConfiguration().setName(name).setFactoryClassName(factoryClass).setParams(parameters);
         ConnectorServiceFactory factory = server.getServiceRegistry().getConnectorService(config);
         server.getConnectorsService().createService(config, factory);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void destroyConnectorService(final String name) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.destroyConnectorService(this.server, name);
      }
      checkStarted();

      clearIO();

      try {
         server.getConnectorsService().destroyService(name);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public String[] getConnectorServices() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getConnectorServices(this.server);
      }
      checkStarted();

      clearIO();

      try {
         return server.getConnectorsService().getConnectors().keySet().toArray(new String[0]);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void forceFailover() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.forceFailover(this.server);
      }
      checkStarted();

      clearIO();

      Thread t = new Thread() {
         @Override
         public void run() {
            try {
               server.stop(true, true);
            } catch (Throwable e) {
               logger.warn(e.getMessage(), e);
            }
         }
      };
      t.start();
   }

   public void updateDuplicateIdCache(String address, Object[] ids) throws Exception {
      clearIO();
      try {
         DuplicateIDCache duplicateIDCache = server.getPostOffice().getDuplicateIDCache(new SimpleString(address));
         for (Object id : ids) {
            duplicateIDCache.addToCache(((String) id).getBytes(), null);
         }
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void scaleDown(String connector) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.scaleDown(this.server, connector);
      }
      checkStarted();

      clearIO();
      HAPolicy haPolicy = server.getHAPolicy();
      if (haPolicy instanceof LiveOnlyPolicy) {
         LiveOnlyPolicy liveOnlyPolicy = (LiveOnlyPolicy) haPolicy;

         if (liveOnlyPolicy.getScaleDownPolicy() == null) {
            liveOnlyPolicy.setScaleDownPolicy(new ScaleDownPolicy());
         }

         liveOnlyPolicy.getScaleDownPolicy().setEnabled(true);

         if (connector != null) {
            liveOnlyPolicy.getScaleDownPolicy().getConnectors().add(0, connector);
         }

         server.fail(true);
      }
   }


   @Override
   public String listNetworkTopology() throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listNetworkTopology(this.server);
      }
      checkStarted();

      clearIO();
      try {
         JsonArrayBuilder brokers = JsonLoader.createArrayBuilder();
         ClusterManager clusterManager = server.getClusterManager();
         if (clusterManager != null) {
            Set<ClusterConnection> clusterConnections = clusterManager.getClusterConnections();
            for (ClusterConnection clusterConnection : clusterConnections) {
               Topology topology = clusterConnection.getTopology();
               Collection<TopologyMemberImpl> members = topology.getMembers();
               for (TopologyMemberImpl member : members) {

                  JsonObjectBuilder obj = JsonLoader.createObjectBuilder();
                  TransportConfiguration live = member.getLive();
                  if (live != null) {
                     obj.add("nodeID", member.getNodeId()).add("live", live.getParams().get("host") + ":" + live.getParams().get("port"));
                     TransportConfiguration backup = member.getBackup();
                     if (backup != null) {
                        obj.add("backup", backup.getParams().get("host") + ":" + backup.getParams().get("port"));
                     }
                  }
                  brokers.add(obj);
               }
            }
         }
         return brokers.build().toString();
      } finally {
         blockOnIO();
      }
   }


   // NotificationEmitter implementation ----------------------------

   @Override
   public void removeNotificationListener(final NotificationListener listener,
                                          final NotificationFilter filter,
                                          final Object handback) throws ListenerNotFoundException {
      if (AuditLogger.isEnabled()) {
         AuditLogger.removeNotificationListener(this.server, listener, filter, handback);
      }
      clearIO();
      try {
         broadcaster.removeNotificationListener(listener, filter, handback);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void removeNotificationListener(final NotificationListener listener) throws ListenerNotFoundException {
      if (AuditLogger.isEnabled()) {
         AuditLogger.removeNotificationListener(this.server, listener);
      }
      clearIO();
      try {
         broadcaster.removeNotificationListener(listener);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public void addNotificationListener(final NotificationListener listener,
                                       final NotificationFilter filter,
                                       final Object handback) throws IllegalArgumentException {
      if (AuditLogger.isEnabled()) {
         AuditLogger.addNotificationListener(this.server, listener, filter, handback);
      }
      clearIO();
      try {
         broadcaster.addNotificationListener(listener, filter, handback);
      } finally {
         blockOnIO();
      }
   }

   @Override
   public MBeanNotificationInfo[] getNotificationInfo() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getNotificationInfo(this.server);
      }
      CoreNotificationType[] values = CoreNotificationType.values();
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
         names[i] = values[i].toString();
      }
      return new MBeanNotificationInfo[]{new MBeanNotificationInfo(names, this.getClass().getName(), "Notifications emitted by a Core Server")};
   }

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   // Private -------------------------------------------------------

   private synchronized void setMessageCounterEnabled(final boolean enable) {
      if (isStarted()) {
         if (configuration.isMessageCounterEnabled() && !enable) {
            stopMessageCounters();
         } else if (!configuration.isMessageCounterEnabled() && enable) {
            startMessageCounters();
         }
      }
      configuration.setMessageCounterEnabled(enable);
   }

   private void startMessageCounters() {
      messageCounterManager.start();
   }

   private void stopMessageCounters() {
      messageCounterManager.stop();

      messageCounterManager.resetAllCounters();

      messageCounterManager.resetAllCounterHistories();
   }

   @Override
   public long getConnectionTTLOverride() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getConnectionTTLOverride(this.server);
      }
      return configuration.getConnectionTTLOverride();
   }

   @Override
   public int getIDCacheSize() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getIDCacheSize(this.server);
      }
      return configuration.getIDCacheSize();
   }

   @Override
   public String getLargeMessagesDirectory() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getLargeMessagesDirectory(this.server);
      }
      return configuration.getLargeMessagesDirectory();
   }

   @Override
   public String getManagementAddress() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getManagementAddress(this.server);
      }
      return configuration.getManagementAddress().toString();
   }

   @Override
   public String getNodeID() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getNodeID(this.server);
      }
      return server.getNodeID() == null ? null : server.getNodeID().toString();
   }

   @Override
   public String getManagementNotificationAddress() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getManagementNotificationAddress(this.server);
      }
      return configuration.getManagementNotificationAddress().toString();
   }

   @Override
   public long getMessageExpiryScanPeriod() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getMessageExpiryScanPeriod(this.server);
      }
      return configuration.getMessageExpiryScanPeriod();
   }

   @Override
   @Deprecated
   public long getMessageExpiryThreadPriority() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getMessageExpiryThreadPriority(this.server);
      }
      return configuration.getMessageExpiryThreadPriority();
   }

   @Override
   public long getTransactionTimeout() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTransactionTimeout(this.server);
      }
      return configuration.getTransactionTimeout();
   }

   @Override
   public long getTransactionTimeoutScanPeriod() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.getTransactionTimeoutScanPeriod(this.server);
      }
      return configuration.getTransactionTimeoutScanPeriod();
   }

   @Override
   public boolean isPersistDeliveryCountBeforeDelivery() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isPersistDeliveryCountBeforeDelivery(this.server);
      }
      return configuration.isPersistDeliveryCountBeforeDelivery();
   }

   @Override
   public boolean isPersistIDCache() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isPersistIDCache(this.server);
      }
      return configuration.isPersistIDCache();
   }

   @Override
   public boolean isWildcardRoutingEnabled() {
      if (AuditLogger.isEnabled()) {
         AuditLogger.isWildcardRoutingEnabled(this.server);
      }
      return configuration.isWildcardRoutingEnabled();
   }

   @Override
   protected MBeanOperationInfo[] fillMBeanOperationInfo() {
      return MBeanInfoHelper.getMBeanOperationsInfo(ActiveMQServerControl.class);
   }

   @Override
   protected MBeanAttributeInfo[] fillMBeanAttributeInfo() {
      return MBeanInfoHelper.getMBeanAttributesInfo(ActiveMQServerControl.class);
   }

   private void checkStarted() {
      if (!server.isStarted()) {
         throw new IllegalStateException("Broker is not started. It can not be managed yet");
      }
   }

   public String[] listTargetAddresses(final String sessionID) {
      ServerSession session = server.getSessionByID(sessionID);
      if (session != null) {
         return session.getTargetAddresses();
      }
      return new String[0];
   }

   @Override
   public void onNotification(org.apache.activemq.artemis.core.server.management.Notification notification) {
      if (!(notification.getType() instanceof CoreNotificationType))
         return;
      CoreNotificationType type = (CoreNotificationType) notification.getType();
      if (type == CoreNotificationType.SESSION_CREATED) {
         TypedProperties props = notification.getProperties();
         /*
          * If the SESSION_CREATED notification is received from another node in the cluster, no broadcast call is made.
          * To keep the original logic to avoid calling the broadcast multiple times for the same SESSION_CREATED notification in the cluster.
          */
         if (props.getIntProperty(ManagementHelper.HDR_DISTANCE) > 0) {
            return;
         }
      }

      this.broadcaster.sendNotification(new Notification(type.toString(), this, notifSeq.incrementAndGet(), notification.toString()));
   }

   @Override
   public void addUser(String username, String password, String roles, boolean plaintext) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.addUser(this.server, username, "****", roles, plaintext);
      }

      tcclInvoke(ActiveMQServerControlImpl.class.getClassLoader(), () -> internalAddUser(username, password, roles, plaintext));
   }

   private void internalAddUser(String username, String password, String roles, boolean plaintext) throws Exception {
      PropertiesLoginModuleConfigurator config = getPropertiesLoginModuleConfigurator();
      config.addNewUser(username, plaintext ? password : PasswordMaskingUtil.getHashProcessor().hash(password), roles.split(","));
      config.save();

   }

   private String getSecurityDomain() {
      return server.getSecurityManager().getDomain();
   }

   @Override
   public String listUser(String username) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.listUser(this.server, username);
      }

      return (String) tcclCall(ActiveMQServerControlImpl.class.getClassLoader(), () -> internaListUser(username));
   }

   private String internaListUser(String username) throws Exception {
      PropertiesLoginModuleConfigurator config = getPropertiesLoginModuleConfigurator();
      Map<String, Set<String>> info = config.listUser(username);
      JsonArrayBuilder users = JsonLoader.createArrayBuilder();
      for (Entry<String, Set<String>> entry : info.entrySet()) {
         JsonObjectBuilder user = JsonLoader.createObjectBuilder();
         user.add("username", entry.getKey());
         JsonArrayBuilder roles = JsonLoader.createArrayBuilder();
         for (String role : entry.getValue()) {
            roles.add(role);
         }
         user.add("roles", roles);
         users.add(user);
      }
      return users.build().toString();
   }

   @Override
   public void removeUser(String username) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.removeUser(this.server, username);
      }
      tcclInvoke(ActiveMQServerControlImpl.class.getClassLoader(), () -> internalRemoveUser(username));
   }

   private void internalRemoveUser(String username) throws Exception {
      PropertiesLoginModuleConfigurator config = getPropertiesLoginModuleConfigurator();
      config.removeUser(username);
      config.save();
   }

   @Override
   public void resetUser(String username, String password, String roles, boolean plaintext) throws Exception {
      if (AuditLogger.isEnabled()) {
         AuditLogger.resetUser(this.server, username, "****", roles, plaintext);
      }
      tcclInvoke(ActiveMQServerControlImpl.class.getClassLoader(), () -> internalresetUser(username, password, roles, plaintext));
   }

   @Override
   public void resetUser(String username, String password, String roles) throws Exception {
      resetUser(username, password, roles, true);
   }

   private void internalresetUser(String username, String password, String roles, boolean plaintext) throws Exception {
      PropertiesLoginModuleConfigurator config = getPropertiesLoginModuleConfigurator();
      // don't hash a null password even if plaintext = false
      config.updateUser(username, password == null ? password : plaintext ? password : PasswordMaskingUtil.getHashProcessor().hash(password), roles == null ? null : roles.split(","));
      config.save();
   }

   private PropertiesLoginModuleConfigurator getPropertiesLoginModuleConfigurator() throws Exception {
      URL configurationUrl = server.getConfiguration().getConfigurationUrl();
      if (configurationUrl == null) {
         throw ActiveMQMessageBundle.BUNDLE.failedToLocateConfigURL();
      }
      String path = configurationUrl.getPath();
      return new PropertiesLoginModuleConfigurator(getSecurityDomain(), path.substring(0, path.lastIndexOf("/")));
   }
}

