/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.paging.impl;

import io.netty.buffer.Unpooled;
import io.netty.util.internal.PlatformDependent;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.activemq.artemis.api.core.ActiveMQBuffer;
import org.apache.activemq.artemis.api.core.ICoreMessage;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.core.buffers.impl.ChannelBufferWrapper;
import org.apache.activemq.artemis.core.io.SequentialFile;
import org.apache.activemq.artemis.core.io.SequentialFileFactory;
import org.apache.activemq.artemis.core.io.mapped.MappedSequentialFileFactory;
import org.apache.activemq.artemis.core.io.nio.NIOSequentialFileFactory;
import org.apache.activemq.artemis.core.paging.PagedMessage;
import org.apache.activemq.artemis.core.paging.cursor.LivePageCache;
import org.apache.activemq.artemis.core.paging.cursor.PageSubscriptionCounter;
import org.apache.activemq.artemis.core.paging.impl.PagedMessageImpl;
import org.apache.activemq.artemis.core.persistence.StorageManager;
import org.apache.activemq.artemis.core.server.ActiveMQMessageBundle;
import org.apache.activemq.artemis.core.server.ActiveMQServerLogger;
import org.apache.activemq.artemis.core.server.LargeServerMessage;
import org.apache.activemq.artemis.utils.collections.ConcurrentHashSet;
import org.jboss.logging.Logger;

public final class Page
implements Comparable<Page> {
    private static final Logger logger = Logger.getLogger(Page.class);
    public static final int SIZE_RECORD = 6;
    private static final byte START_BYTE = 123;
    private static final byte END_BYTE = 125;
    private final int pageId;
    private boolean suspiciousRecords = false;
    private final AtomicInteger numberOfMessages = new AtomicInteger(0);
    private final SequentialFile file;
    private final SequentialFileFactory fileFactory;
    private volatile LivePageCache pageCache;
    private final AtomicInteger size = new AtomicInteger(0);
    private final StorageManager storageManager;
    private final SimpleString storeName;
    private Set<PageSubscriptionCounter> pendingCounters;
    private boolean canBeMapped;

    public Page(SimpleString storeName, StorageManager storageManager, SequentialFileFactory factory, SequentialFile file, int pageId) throws Exception {
        this.pageId = pageId;
        this.file = file;
        this.fileFactory = factory;
        this.storageManager = storageManager;
        this.storeName = storeName;
        this.canBeMapped = this.fileFactory instanceof NIOSequentialFileFactory || this.fileFactory instanceof MappedSequentialFileFactory;
    }

    public int getPageId() {
        return this.pageId;
    }

    public void setLiveCache(LivePageCache pageCache) {
        this.pageCache = pageCache;
    }

    public synchronized List<PagedMessage> read(StorageManager storage) throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug((Object)("reading page " + this.pageId + " on address = " + this.storeName));
        }
        if (!this.file.isOpen()) {
            throw ActiveMQMessageBundle.BUNDLE.invalidPageIO();
        }
        ArrayList<PagedMessage> messages = new ArrayList<PagedMessage>();
        this.size.lazySet((int)this.file.size());
        if (this.canBeMapped) {
            this.readFromMapped(storage, messages);
            this.file.position(this.file.size());
        } else {
            this.readFromSequentialFile(storage, messages);
        }
        this.numberOfMessages.lazySet(messages.size());
        return messages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readFromSequentialFile(StorageManager storage, List<PagedMessage> messages) throws Exception {
        int fileSize = (int)this.file.size();
        ByteBuffer buffer = this.fileFactory.newBuffer(fileSize);
        try {
            this.file.position(0L);
            this.file.read(buffer);
            buffer.rewind();
            assert (buffer.limit() == fileSize) : "buffer doesn't contains the whole file";
            ChannelBufferWrapper activeMQBuffer = this.wrapBuffer(fileSize, buffer);
            this.read(storage, (ActiveMQBuffer)activeMQBuffer, messages);
        }
        finally {
            this.fileFactory.releaseBuffer(buffer);
        }
    }

    private ChannelBufferWrapper wrapBuffer(int fileSize, ByteBuffer buffer) {
        ChannelBufferWrapper activeMQBuffer = new ChannelBufferWrapper(Unpooled.wrappedBuffer((ByteBuffer)buffer));
        return activeMQBuffer;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static MappedByteBuffer mapFileForRead(File file, int fileSize) {
        try (FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);){
            MappedByteBuffer mappedByteBuffer = channel.map(FileChannel.MapMode.READ_ONLY, 0L, fileSize);
            return mappedByteBuffer;
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int readFromMapped(StorageManager storage, List<PagedMessage> messages) throws IOException {
        this.file.position(0L);
        int mappedSize = this.size.get();
        MappedByteBuffer mappedByteBuffer = Page.mapFileForRead(this.file.getJavaFile(), mappedSize);
        ChannelBufferWrapper activeMQBuffer = this.wrapBuffer(mappedSize, mappedByteBuffer);
        try {
            int n = this.read(storage, (ActiveMQBuffer)activeMQBuffer, messages);
            return n;
        }
        finally {
            PlatformDependent.freeDirectBuffer((ByteBuffer)mappedByteBuffer);
        }
    }

    private int read(StorageManager storage, ActiveMQBuffer fileBuffer, List<PagedMessage> messages) {
        int readMessages = 0;
        while (fileBuffer.readable()) {
            int position = fileBuffer.readerIndex();
            byte byteRead = fileBuffer.readByte();
            if (byteRead == 123) {
                if (fileBuffer.readerIndex() + 4 >= fileBuffer.capacity()) continue;
                int messageSize = fileBuffer.readInt();
                int oldPos = fileBuffer.readerIndex();
                if (fileBuffer.readerIndex() + messageSize < fileBuffer.capacity() && fileBuffer.getByte(oldPos + messageSize) == 125) {
                    PagedMessageImpl msg = new PagedMessageImpl(this.storageManager);
                    msg.decode(fileBuffer);
                    byte b = fileBuffer.readByte();
                    if (b != 125) {
                        throw new IllegalStateException("Internal error, it wasn't possible to locate END_BYTE " + b);
                    }
                    msg.initMessage(storage);
                    if (logger.isTraceEnabled()) {
                        logger.trace((Object)("Reading message " + msg + " on pageId=" + this.pageId + " for address=" + this.storeName));
                    }
                    ++readMessages;
                    messages.add(msg);
                    continue;
                }
                this.markFileAsSuspect(this.file.getFileName(), position, messages.size());
                break;
            }
            this.markFileAsSuspect(this.file.getFileName(), position, messages.size());
            break;
        }
        return readMessages;
    }

    public synchronized void write(PagedMessage message) throws Exception {
        if (!this.file.isOpen()) {
            throw ActiveMQMessageBundle.BUNDLE.cannotWriteToClosedFile(this.file);
        }
        int messageEncodedSize = message.getEncodeSize();
        int bufferSize = messageEncodedSize + 6;
        ByteBuffer buffer = this.fileFactory.newBuffer(bufferSize);
        ChannelBufferWrapper activeMQBuffer = this.wrapBuffer(bufferSize, buffer);
        activeMQBuffer.clear();
        activeMQBuffer.writeByte((byte)123);
        activeMQBuffer.writeInt(messageEncodedSize);
        message.encode((ActiveMQBuffer)activeMQBuffer);
        activeMQBuffer.writeByte((byte)125);
        assert (activeMQBuffer.readableBytes() == bufferSize) : "messageEncodedSize is different from expected";
        assert (buffer.remaining() == bufferSize) : "buffer position or limit are changed";
        this.file.writeDirect(buffer, false);
        if (this.pageCache != null) {
            this.pageCache.addLiveMessage(message);
        }
        this.numberOfMessages.lazySet(this.numberOfMessages.get() + 1);
        this.size.lazySet(this.size.get() + bufferSize);
        this.storageManager.pageWrite(message, this.pageId);
    }

    public void sync() throws Exception {
        this.file.sync();
    }

    public void open() throws Exception {
        if (!this.file.isOpen()) {
            this.file.open();
        }
        this.size.set((int)this.file.size());
        this.file.position(0L);
    }

    public void close() throws Exception {
        this.close(false);
    }

    public synchronized void close(boolean sendEvent) throws Exception {
        if (sendEvent && this.storageManager != null) {
            this.storageManager.pageClosed(this.storeName, this.pageId);
        }
        if (this.pageCache != null) {
            this.pageCache.close();
            this.pageCache = null;
        }
        this.file.close();
        Set<PageSubscriptionCounter> counters = this.getPendingCounters();
        if (counters != null) {
            for (PageSubscriptionCounter counter : counters) {
                counter.cleanupNonTXCounters(this.getPageId());
            }
        }
    }

    public boolean isLive() {
        return this.pageCache != null;
    }

    public boolean delete(PagedMessage[] messages) throws Exception {
        if (this.storageManager != null) {
            this.storageManager.pageDeleted(this.storeName, this.pageId);
        }
        if (logger.isDebugEnabled()) {
            logger.debug((Object)("Deleting pageNr=" + this.pageId + " on store " + this.storeName));
        }
        if (messages != null) {
            for (PagedMessage msg : messages) {
                if (!(msg.getMessage() instanceof ICoreMessage) || !msg.getMessage().isLargeMessage()) continue;
                LargeServerMessage lmsg = (LargeServerMessage)msg.getMessage();
                lmsg.decrementDelayDeletionCount();
            }
        }
        try {
            if (this.suspiciousRecords) {
                ActiveMQServerLogger.LOGGER.pageInvalid(this.file.getFileName(), this.file.getFileName());
                this.file.renameTo(this.file.getFileName() + ".invalidPage");
            } else {
                this.file.delete();
            }
            return true;
        }
        catch (Exception e) {
            ActiveMQServerLogger.LOGGER.pageDeleteError(e);
            return false;
        }
    }

    public int getNumberOfMessages() {
        return this.numberOfMessages.intValue();
    }

    public int getSize() {
        return this.size.intValue();
    }

    public String toString() {
        return "Page::pageNr=" + this.pageId + ", file=" + this.file;
    }

    @Override
    public int compareTo(Page otherPage) {
        return otherPage.getPageId() - this.pageId;
    }

    protected void finalize() {
        try {
            if (this.file != null && this.file.isOpen()) {
                this.file.close(false);
            }
        }
        catch (Exception e) {
            ActiveMQServerLogger.LOGGER.pageFinaliseError(e);
        }
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        result = 31 * result + this.pageId;
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        Page other = (Page)obj;
        return this.pageId == other.pageId;
    }

    private void markFileAsSuspect(String fileName, int position, int msgNumber) {
        ActiveMQServerLogger.LOGGER.pageSuspectFile(fileName, position, msgNumber);
        this.suspiciousRecords = true;
    }

    public SequentialFile getFile() {
        return this.file;
    }

    public void addPendingCounter(PageSubscriptionCounter pageSubscriptionCounter) {
        Set<PageSubscriptionCounter> counter = this.getOrCreatePendingCounters();
        this.pendingCounters.add(pageSubscriptionCounter);
    }

    private synchronized Set<PageSubscriptionCounter> getPendingCounters() {
        return this.pendingCounters;
    }

    private synchronized Set<PageSubscriptionCounter> getOrCreatePendingCounters() {
        if (this.pendingCounters == null) {
            this.pendingCounters = new ConcurrentHashSet();
        }
        return this.pendingCounters;
    }
}

