/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.repository.graph;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.apache.atlas.ApplicationProperties;
import org.apache.atlas.AtlasErrorCode;
import org.apache.atlas.AtlasException;
import org.apache.atlas.GraphTransactionInterceptor;
import org.apache.atlas.RequestContext;
import org.apache.atlas.exception.AtlasBaseException;
import org.apache.atlas.exception.EntityNotFoundException;
import org.apache.atlas.model.instance.AtlasEntity;
import org.apache.atlas.model.instance.AtlasEntityHeader;
import org.apache.atlas.model.instance.AtlasObjectId;
import org.apache.atlas.model.instance.AtlasRelationship;
import org.apache.atlas.model.typedef.AtlasRelationshipDef;
import org.apache.atlas.repository.Constants;
import org.apache.atlas.repository.RepositoryException;
import org.apache.atlas.repository.graphdb.AtlasEdge;
import org.apache.atlas.repository.graphdb.AtlasEdgeDirection;
import org.apache.atlas.repository.graphdb.AtlasElement;
import org.apache.atlas.repository.graphdb.AtlasGraph;
import org.apache.atlas.repository.graphdb.AtlasGraphQuery;
import org.apache.atlas.repository.graphdb.AtlasVertex;
import org.apache.atlas.repository.graphdb.AtlasVertexQuery;
import org.apache.atlas.repository.store.graph.v2.AtlasGraphUtilsV2;
import org.apache.atlas.type.AtlasArrayType;
import org.apache.atlas.type.AtlasEntityType;
import org.apache.atlas.type.AtlasMapType;
import org.apache.atlas.type.AtlasStructType;
import org.apache.atlas.type.AtlasType;
import org.apache.atlas.util.AttributeValueMap;
import org.apache.atlas.util.IndexedInstance;
import org.apache.atlas.utils.AtlasPerfMetrics;
import org.apache.atlas.v1.model.instance.Id;
import org.apache.atlas.v1.model.instance.Referenceable;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.IteratorUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class GraphHelper {
    private static final Logger LOG = LoggerFactory.getLogger(GraphHelper.class);
    public static final String EDGE_LABEL_PREFIX = "__";
    public static final String RETRY_COUNT = "atlas.graph.storage.num.retries";
    public static final String RETRY_DELAY = "atlas.graph.storage.retry.sleeptime.ms";
    public static final String DEFAULT_REMOVE_PROPAGATIONS_ON_ENTITY_DELETE = "atlas.graph.remove.propagations.default";
    private AtlasGraph graph;
    private int maxRetries = 3;
    private long retrySleepTimeMillis = 1000L;
    private boolean removePropagations = false;
    @VisibleForTesting
    public static BiMap<String, String> RESERVED_CHARS_ENCODE_MAP = HashBiMap.create((Map)new HashMap<String, String>(){
        {
            this.put("{", "_o");
            this.put("}", "_c");
            this.put("\"", "_q");
            this.put("$", "_d");
            this.put("%", "_p");
        }
    });

    public GraphHelper(AtlasGraph graph) {
        this.graph = graph;
        try {
            this.maxRetries = ApplicationProperties.get().getInt(RETRY_COUNT, 3);
            this.retrySleepTimeMillis = ApplicationProperties.get().getLong(RETRY_DELAY, 1000L);
            this.removePropagations = ApplicationProperties.get().getBoolean(DEFAULT_REMOVE_PROPAGATIONS_ON_ENTITY_DELETE, false);
        }
        catch (AtlasException e) {
            LOG.error("Could not load configuration. Setting to default value for atlas.graph.storage.num.retries", (Throwable)e);
        }
    }

    public static boolean isTermEntityEdge(AtlasEdge edge) {
        return StringUtils.equals((String)edge.getLabel(), (String)"r:AtlasGlossarySemanticAssignment");
    }

    public AtlasEdge addClassificationEdge(AtlasVertex entityVertex, AtlasVertex classificationVertex, boolean isPropagated) {
        AtlasEdge ret = this.addEdge(entityVertex, classificationVertex, "classifiedAs");
        if (ret != null) {
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, GraphHelper.getTypeName((AtlasElement)classificationVertex));
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, isPropagated);
        }
        return ret;
    }

    public AtlasEdge addEdge(AtlasVertex fromVertex, AtlasVertex toVertex, String edgeLabel) {
        AtlasEdge ret;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Adding edge for {} -> label {} -> {}", new Object[]{GraphHelper.string(fromVertex), edgeLabel, GraphHelper.string(toVertex)});
        }
        if ((ret = this.graph.addEdge(fromVertex, toVertex, edgeLabel)) != null) {
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.STATE_PROPERTY_KEY, AtlasEntity.Status.ACTIVE.name());
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.TIMESTAMP_PROPERTY_KEY, RequestContext.get().getRequestTime());
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.MODIFICATION_TIMESTAMP_PROPERTY_KEY, RequestContext.get().getRequestTime());
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.CREATED_BY_KEY, RequestContext.get().getUser());
            AtlasGraphUtilsV2.setEncodedProperty(ret, Constants.MODIFIED_BY_KEY, RequestContext.get().getUser());
            if (LOG.isDebugEnabled()) {
                LOG.debug("Added {}", (Object)GraphHelper.string(ret));
            }
        }
        return ret;
    }

    public AtlasEdge getOrCreateEdge(AtlasVertex outVertex, AtlasVertex inVertex, String edgeLabel) throws RepositoryException {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("getOrCreateEdge");
        for (int numRetries = 0; numRetries < this.maxRetries; ++numRetries) {
            try {
                AtlasEdge edge;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Running edge creation attempt {}", (Object)numRetries);
                }
                if (inVertex.hasEdges(AtlasEdgeDirection.IN, edgeLabel) && outVertex.hasEdges(AtlasEdgeDirection.OUT, edgeLabel) && (edge = this.graph.getEdgeBetweenVertices(outVertex, inVertex, edgeLabel)) != null) {
                    return edge;
                }
                return this.addEdge(outVertex, inVertex, edgeLabel);
            }
            catch (Exception e) {
                LOG.warn(String.format("Exception while trying to create edge from %s to %s with label %s. Retrying", GraphHelper.vertexString(outVertex), GraphHelper.vertexString(inVertex), edgeLabel), (Throwable)e);
                if (numRetries == this.maxRetries - 1) {
                    LOG.error("Max retries exceeded for edge creation {} {} {} ", new Object[]{outVertex, inVertex, edgeLabel, e});
                    throw new RepositoryException("Edge creation failed after retries", e);
                }
                try {
                    LOG.info("Retrying with delay of {} ms ", (Object)this.retrySleepTimeMillis);
                    Thread.sleep(this.retrySleepTimeMillis);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.warn("Retry interrupted during edge creation ");
                    throw new RepositoryException("Retry interrupted during edge creation", ie);
                }
            }
        }
        RequestContext.get().endMetricRecord(metric);
        return null;
    }

    public AtlasEdge getEdgeByEdgeId(AtlasVertex outVertex, String edgeLabel, String edgeId) {
        if (edgeId == null) {
            return null;
        }
        return this.graph.getEdge(edgeId);
    }

    public AtlasVertex findVertex(Object ... args) throws EntityNotFoundException {
        return (AtlasVertex)this.findElement(true, args);
    }

    public AtlasEdge findEdge(Object ... args) throws EntityNotFoundException {
        return (AtlasEdge)this.findElement(false, args);
    }

    private AtlasElement findElement(boolean isVertexSearch, Object ... args) throws EntityNotFoundException {
        AtlasElement element;
        AtlasGraphQuery query = this.graph.query();
        for (int i = 0; i < args.length; i += 2) {
            query = query.has((String)args[i], args[i + 1]);
        }
        Iterator results = isVertexSearch ? query.vertices().iterator() : query.edges().iterator();
        AtlasElement atlasElement = element = results != null && results.hasNext() ? (AtlasElement)results.next() : null;
        if (element == null) {
            throw new EntityNotFoundException("Could not find " + (isVertexSearch ? "vertex" : "edge") + " with condition: " + this.getConditionString(args));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found {} with condition {}", (Object)GraphHelper.string(element), (Object)this.getConditionString(args));
        }
        return element;
    }

    public static Iterator<AtlasEdge> getAdjacentEdgesByLabel(AtlasVertex instanceVertex, AtlasEdgeDirection direction, String edgeLabel) {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("getAdjacentEdgesByLabel");
        if (LOG.isDebugEnabled()) {
            LOG.debug("Finding edges for {} with label {}", (Object)GraphHelper.string(instanceVertex), (Object)edgeLabel);
        }
        Iterator ret = null;
        if (instanceVertex != null && edgeLabel != null) {
            ret = instanceVertex.getEdges(direction, edgeLabel).iterator();
        }
        RequestContext.get().endMetricRecord(metric);
        return ret;
    }

    public static long getAdjacentEdgesCountByLabel(AtlasVertex instanceVertex, AtlasEdgeDirection direction, String edgeLabel) {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("getAdjacentEdgesCountByLabel");
        if (LOG.isDebugEnabled()) {
            LOG.debug("Finding edges for {} with label {}", (Object)GraphHelper.string(instanceVertex), (Object)edgeLabel);
        }
        long ret = 0L;
        if (instanceVertex != null && edgeLabel != null) {
            ret = instanceVertex.getEdgesCount(direction, edgeLabel);
        }
        RequestContext.get().endMetricRecord(metric);
        return ret;
    }

    public static boolean isPropagationEnabled(AtlasVertex classificationVertex) {
        boolean ret = false;
        if (classificationVertex != null) {
            Boolean enabled = AtlasGraphUtilsV2.getEncodedProperty(classificationVertex, Constants.CLASSIFICATION_VERTEX_PROPAGATE_KEY, Boolean.class);
            ret = enabled == null ? true : enabled;
        }
        return ret;
    }

    public static boolean getRemovePropagations(AtlasVertex classificationVertex) {
        boolean ret = false;
        if (classificationVertex != null) {
            Boolean enabled = AtlasGraphUtilsV2.getEncodedProperty(classificationVertex, Constants.CLASSIFICATION_VERTEX_REMOVE_PROPAGATIONS_KEY, Boolean.class);
            ret = enabled == null ? true : enabled;
        }
        return ret;
    }

    public static AtlasVertex getClassificationVertex(AtlasVertex entityVertex, String classificationName) {
        Iterator iterator;
        AtlasVertex ret = null;
        Iterable edges = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs").has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)false).has(Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, (Object)classificationName).edges();
        if (edges != null && (iterator = edges.iterator()).hasNext()) {
            AtlasEdge edge = (AtlasEdge)iterator.next();
            ret = edge != null ? edge.getInVertex() : null;
        }
        return ret;
    }

    public static AtlasEdge getClassificationEdge(AtlasVertex entityVertex, AtlasVertex classificationVertex) {
        Iterator iterator;
        AtlasEdge ret = null;
        Iterable edges = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs").has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)false).has(Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, (Object)GraphHelper.getTypeName((AtlasElement)classificationVertex)).edges();
        if (edges != null && (iterator = edges.iterator()).hasNext()) {
            AtlasEdge edge = (AtlasEdge)iterator.next();
            ret = edge != null && edge.getInVertex().equals(classificationVertex) ? edge : null;
        }
        return ret;
    }

    public static AtlasEdge getPropagatedClassificationEdge(AtlasVertex entityVertex, String classificationName, String associatedEntityGuid) {
        AtlasEdge ret = null;
        Iterable edges = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs").has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)true).has(Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, (Object)classificationName).edges();
        if (edges != null) {
            for (AtlasEdge edge : edges) {
                String guid;
                AtlasVertex classificationVertex = edge != null ? edge.getInVertex() : null;
                if (classificationVertex == null || !StringUtils.equals((String)(guid = AtlasGraphUtilsV2.getEncodedProperty(classificationVertex, Constants.CLASSIFICATION_ENTITY_GUID, String.class)), (String)associatedEntityGuid)) continue;
                ret = edge;
                break;
            }
        }
        return ret;
    }

    public static AtlasEdge getPropagatedClassificationEdge(AtlasVertex entityVertex, AtlasVertex classificationVertex) {
        AtlasEdge ret = null;
        Iterable edges = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs").has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)true).has(Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, (Object)GraphHelper.getTypeName((AtlasElement)classificationVertex)).edges();
        if (edges != null && classificationVertex != null) {
            Iterator iterator = edges.iterator();
            while (iterator != null && iterator.hasNext()) {
                AtlasEdge edge = (AtlasEdge)iterator.next();
                if (edge == null || !edge.getInVertex().equals(classificationVertex)) continue;
                ret = edge;
                break;
            }
        }
        return ret;
    }

    public static List<AtlasEdge> getPropagatedEdges(AtlasVertex classificationVertex) {
        ArrayList<AtlasEdge> ret = new ArrayList<AtlasEdge>();
        Iterable edges = classificationVertex.query().direction(AtlasEdgeDirection.IN).label("classifiedAs").has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)true).has(Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, (Object)GraphHelper.getTypeName((AtlasElement)classificationVertex)).edges();
        if (edges != null) {
            for (AtlasEdge edge : edges) {
                ret.add(edge);
            }
        }
        return ret;
    }

    public static boolean hasEntityReferences(AtlasVertex classificationVertex) {
        return classificationVertex.hasEdges(AtlasEdgeDirection.IN, "classifiedAs");
    }

    public static List<AtlasVertex> getAllPropagatedEntityVertices(AtlasVertex classificationVertex) {
        List<AtlasEdge> edges;
        ArrayList<AtlasVertex> ret = new ArrayList<AtlasVertex>();
        if (classificationVertex != null && CollectionUtils.isNotEmpty(edges = GraphHelper.getPropagatedEdges(classificationVertex))) {
            for (AtlasEdge edge : edges) {
                ret.add(edge.getOutVertex());
            }
        }
        return ret;
    }

    public static Iterator<AtlasEdge> getIncomingEdgesByLabel(AtlasVertex instanceVertex, String edgeLabel) {
        return GraphHelper.getAdjacentEdgesByLabel(instanceVertex, AtlasEdgeDirection.IN, edgeLabel);
    }

    public static Iterator<AtlasEdge> getOutGoingEdgesByLabel(AtlasVertex instanceVertex, String edgeLabel) {
        return GraphHelper.getAdjacentEdgesByLabel(instanceVertex, AtlasEdgeDirection.OUT, edgeLabel);
    }

    public static long getOutGoingEdgesCountByLabel(AtlasVertex instanceVertex, String edgeLabel) {
        return GraphHelper.getAdjacentEdgesCountByLabel(instanceVertex, AtlasEdgeDirection.OUT, edgeLabel);
    }

    public static long getInComingEdgesCountByLabel(AtlasVertex instanceVertex, String edgeLabel) {
        return GraphHelper.getAdjacentEdgesCountByLabel(instanceVertex, AtlasEdgeDirection.IN, edgeLabel);
    }

    public AtlasEdge getEdgeForLabel(AtlasVertex vertex, String edgeLabel, AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection edgeDirection) {
        AtlasEdge ret;
        switch (edgeDirection) {
            case IN: {
                ret = this.getEdgeForLabel(vertex, edgeLabel, AtlasEdgeDirection.IN);
                break;
            }
            case OUT: {
                ret = this.getEdgeForLabel(vertex, edgeLabel, AtlasEdgeDirection.OUT);
                break;
            }
            default: {
                ret = this.getEdgeForLabel(vertex, edgeLabel, AtlasEdgeDirection.BOTH);
            }
        }
        return ret;
    }

    public static Iterator<AtlasEdge> getEdgesForLabel(AtlasVertex vertex, String edgeLabel, AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection edgeDirection) {
        Iterator<AtlasEdge> ret = null;
        switch (edgeDirection) {
            case IN: {
                ret = GraphHelper.getIncomingEdgesByLabel(vertex, edgeLabel);
                break;
            }
            case OUT: {
                ret = GraphHelper.getOutGoingEdgesByLabel(vertex, edgeLabel);
                break;
            }
            case BOTH: {
                ret = GraphHelper.getAdjacentEdgesByLabel(vertex, AtlasEdgeDirection.BOTH, edgeLabel);
            }
        }
        return ret;
    }

    public AtlasEdge getEdgeForLabel(AtlasVertex vertex, String edgeLabel) {
        return this.getEdgeForLabel(vertex, edgeLabel, AtlasEdgeDirection.OUT);
    }

    public AtlasEdge getEdgeForLabel(AtlasVertex vertex, String edgeLabel, AtlasEdgeDirection edgeDirection) {
        Iterator<AtlasEdge> iterator = GraphHelper.getAdjacentEdgesByLabel(vertex, edgeDirection, edgeLabel);
        AtlasEdge latestDeletedEdge = null;
        long latestDeletedEdgeTime = Long.MIN_VALUE;
        while (iterator != null && iterator.hasNext()) {
            AtlasEdge edge = iterator.next();
            Id.EntityState edgeState = GraphHelper.getState((AtlasElement)edge);
            if (edgeState == null || edgeState == Id.EntityState.ACTIVE) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Found {}", (Object)GraphHelper.string(edge));
                }
                return edge;
            }
            Long modificationTime = (Long)edge.getProperty(Constants.MODIFICATION_TIMESTAMP_PROPERTY_KEY, Long.class);
            if (modificationTime == null || modificationTime < latestDeletedEdgeTime) continue;
            latestDeletedEdgeTime = modificationTime;
            latestDeletedEdge = edge;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found {}", (Object)(latestDeletedEdge == null ? "null" : GraphHelper.string(latestDeletedEdge)));
        }
        return latestDeletedEdge;
    }

    public static String vertexString(AtlasVertex vertex) {
        StringBuilder properties = new StringBuilder();
        for (String propertyKey : vertex.getPropertyKeys()) {
            Collection propertyValues = vertex.getPropertyValues(propertyKey, Object.class);
            properties.append(propertyKey).append("=").append(propertyValues.toString()).append(", ");
        }
        return "v[" + vertex.getIdForDisplay() + "], Properties[" + properties + "]";
    }

    public static String edgeString(AtlasEdge edge) {
        return "e[" + edge.getLabel() + "], [" + edge.getOutVertex() + " -> " + edge.getLabel() + " -> " + edge.getInVertex() + "]";
    }

    private static <T extends AtlasElement> String string(T element) {
        if (element instanceof AtlasVertex) {
            return GraphHelper.string((AtlasVertex)element);
        }
        if (element instanceof AtlasEdge) {
            return GraphHelper.string((AtlasEdge)element);
        }
        return element.toString();
    }

    public void removeEdge(AtlasEdge edge) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> removeEdge({})", (Object)GraphHelper.string(edge));
        }
        this.graph.removeEdge(edge);
        if (LOG.isDebugEnabled()) {
            LOG.info("<== removeEdge()");
        }
    }

    public void removeVertex(AtlasVertex vertex) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> GraphHelper.removeVertex({})", (Object)GraphHelper.string(vertex));
        }
        this.graph.removeVertex(vertex);
        if (LOG.isDebugEnabled()) {
            LOG.debug("<== GraphHelper.removeVertex()");
        }
    }

    public AtlasVertex getVertexForGUID(String guid) throws EntityNotFoundException {
        return this.findVertex(Constants.GUID_PROPERTY_KEY, guid);
    }

    public AtlasEdge getEdgeForGUID(String guid) throws AtlasBaseException {
        AtlasEdge ret;
        try {
            ret = this.findEdge(Constants.RELATIONSHIP_GUID_PROPERTY_KEY, guid);
        }
        catch (EntityNotFoundException e) {
            throw new AtlasBaseException(AtlasErrorCode.RELATIONSHIP_GUID_NOT_FOUND, new String[]{guid});
        }
        return ret;
    }

    public Map<String, AtlasVertex> getVerticesForPropertyValues(String property, List<String> values) {
        if (values.isEmpty()) {
            return Collections.emptyMap();
        }
        HashSet<String> nonNullValues = new HashSet<String>(values.size());
        for (String value : values) {
            if (value == null) continue;
            nonNullValues.add(value);
        }
        AtlasGraphQuery query = this.graph.query();
        query.in(property, nonNullValues);
        Iterable results = query.vertices();
        HashMap<String, AtlasVertex> result = new HashMap<String, AtlasVertex>(values.size());
        for (AtlasVertex vertex : results) {
            if (!vertex.exists()) continue;
            String propertyValue = (String)vertex.getProperty(property, String.class);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Found a vertex {} with {} =  {}", new Object[]{GraphHelper.string(vertex), property, propertyValue});
            }
            result.put(propertyValue, vertex);
        }
        return result;
    }

    public Map<String, AtlasVertex> getVerticesForGUIDs(List<String> guids) {
        return this.getVerticesForPropertyValues(Constants.GUID_PROPERTY_KEY, guids);
    }

    public static void updateModificationMetadata(AtlasVertex vertex) {
        AtlasGraphUtilsV2.setEncodedProperty(vertex, Constants.MODIFICATION_TIMESTAMP_PROPERTY_KEY, RequestContext.get().getRequestTime());
        AtlasGraphUtilsV2.setEncodedProperty(vertex, Constants.MODIFIED_BY_KEY, RequestContext.get().getUser());
    }

    public static String getQualifiedNameForMapKey(String prefix, String key) {
        return prefix + "." + key;
    }

    public static String getTraitLabel(String typeName, String attrName) {
        return attrName;
    }

    public static String getTraitLabel(String traitName) {
        return traitName;
    }

    public static List<String> getTraitNames(AtlasVertex entityVertex) {
        return GraphHelper.getTraitNames(entityVertex, false);
    }

    public static List<String> getPropagatedTraitNames(AtlasVertex entityVertex) {
        return GraphHelper.getTraitNames(entityVertex, true);
    }

    public static List<String> getAllTraitNames(AtlasVertex entityVertex) {
        return GraphHelper.getTraitNames(entityVertex, null);
    }

    public static List<String> getTraitNames(AtlasVertex entityVertex, Boolean propagated) {
        Iterable edges;
        ArrayList<String> ret = new ArrayList<String>();
        AtlasVertexQuery query = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs");
        if (propagated != null) {
            query = query.has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)propagated);
        }
        if ((edges = query.edges()) != null) {
            for (AtlasEdge edge : edges) {
                ret.add(AtlasGraphUtilsV2.getEncodedProperty(edge, Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, String.class));
            }
        }
        return ret;
    }

    public static List<AtlasVertex> getPropagatableClassifications(AtlasEdge edge) {
        ArrayList<AtlasVertex> ret = new ArrayList<AtlasVertex>();
        if (edge != null && GraphHelper.getStatus(edge) != AtlasEntity.Status.DELETED) {
            AtlasRelationshipDef.PropagateTags propagateTags = GraphHelper.getPropagateTags((AtlasElement)edge);
            AtlasVertex outVertex = edge.getOutVertex();
            AtlasVertex inVertex = edge.getInVertex();
            if (propagateTags == AtlasRelationshipDef.PropagateTags.ONE_TO_TWO || propagateTags == AtlasRelationshipDef.PropagateTags.BOTH) {
                ret.addAll(GraphHelper.getPropagationEnabledClassificationVertices(outVertex));
            }
            if (propagateTags == AtlasRelationshipDef.PropagateTags.TWO_TO_ONE || propagateTags == AtlasRelationshipDef.PropagateTags.BOTH) {
                ret.addAll(GraphHelper.getPropagationEnabledClassificationVertices(inVertex));
            }
        }
        return ret;
    }

    public static List<AtlasVertex> getPropagationEnabledClassificationVertices(AtlasVertex entityVertex) {
        Iterable edges;
        ArrayList<AtlasVertex> ret = new ArrayList<AtlasVertex>();
        if (entityVertex.hasEdges(AtlasEdgeDirection.OUT, "classifiedAs") && (edges = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs").edges()) != null) {
            for (AtlasEdge edge : edges) {
                AtlasVertex classificationVertex;
                if (edge == null || !GraphHelper.isPropagationEnabled(classificationVertex = edge.getInVertex())) continue;
                ret.add(classificationVertex);
            }
        }
        return ret;
    }

    public static List<AtlasEdge> getClassificationEdges(AtlasVertex entityVertex) {
        return GraphHelper.getClassificationEdges(entityVertex, false);
    }

    public static List<AtlasEdge> getPropagatedClassificationEdges(AtlasVertex entityVertex) {
        return GraphHelper.getClassificationEdges(entityVertex, true);
    }

    public static List<AtlasEdge> getAllClassificationEdges(AtlasVertex entityVertex) {
        return GraphHelper.getClassificationEdges(entityVertex, null);
    }

    public static List<AtlasEdge> getClassificationEdges(AtlasVertex entityVertex, Boolean propagated) {
        Iterable edges;
        ArrayList<AtlasEdge> ret = new ArrayList<AtlasEdge>();
        AtlasVertexQuery query = entityVertex.query().direction(AtlasEdgeDirection.OUT).label("classifiedAs");
        if (propagated != null) {
            query = query.has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)propagated);
        }
        if ((edges = query.edges()) != null) {
            for (AtlasEdge edge : edges) {
                if (edge == null) continue;
                ret.add(edge);
            }
        }
        return ret;
    }

    public static List<String> getSuperTypeNames(AtlasVertex<?, ?> entityVertex) {
        ArrayList<String> superTypes = new ArrayList<String>();
        Collection propertyValues = entityVertex.getPropertyValues(Constants.SUPER_TYPES_PROPERTY_KEY, String.class);
        if (CollectionUtils.isNotEmpty((Collection)propertyValues)) {
            for (String value : propertyValues) {
                superTypes.add(value);
            }
        }
        return superTypes;
    }

    public static String getEdgeLabel(AtlasStructType.AtlasAttribute aInfo) throws AtlasException {
        return aInfo.getRelationshipEdgeLabel();
    }

    public static Id getIdFromVertex(String dataTypeName, AtlasVertex vertex) {
        return new Id(GraphHelper.getGuid(vertex), GraphHelper.getVersion((AtlasElement)vertex).intValue(), dataTypeName, GraphHelper.getStateAsString((AtlasElement)vertex));
    }

    public static Id getIdFromVertex(AtlasVertex vertex) {
        return GraphHelper.getIdFromVertex(GraphHelper.getTypeName((AtlasElement)vertex), vertex);
    }

    public static String getRelationshipGuid(AtlasElement element) {
        return (String)element.getProperty(Constants.RELATIONSHIP_GUID_PROPERTY_KEY, String.class);
    }

    public static String getGuid(AtlasVertex vertex) {
        Object vertexId = vertex.getId();
        String ret = GraphTransactionInterceptor.getVertexGuidFromCache(vertexId);
        if (ret == null) {
            ret = (String)vertex.getProperty(Constants.GUID_PROPERTY_KEY, String.class);
            GraphTransactionInterceptor.addToVertexGuidCache(vertexId, ret);
        }
        return ret;
    }

    public static String getHomeId(AtlasElement element) {
        return (String)element.getProperty(Constants.HOME_ID_KEY, String.class);
    }

    public static Boolean isProxy(AtlasElement element) {
        return (Boolean)element.getProperty(Constants.IS_PROXY_KEY, Boolean.class);
    }

    public static Boolean isEntityIncomplete(AtlasElement element) {
        Integer value = (Integer)element.getProperty(Constants.IS_INCOMPLETE_PROPERTY_KEY, Integer.class);
        Boolean ret = value != null && value.equals(Constants.INCOMPLETE_ENTITY_VALUE) ? Boolean.TRUE : Boolean.FALSE;
        return ret;
    }

    public static Map getCustomAttributes(AtlasElement element) {
        Map ret = null;
        String customAttrsString = (String)element.getProperty(Constants.CUSTOM_ATTRIBUTES_PROPERTY_KEY, String.class);
        if (customAttrsString != null) {
            ret = (Map)AtlasType.fromJson((String)customAttrsString, Map.class);
        }
        return ret;
    }

    public static Set<String> getLabels(AtlasElement element) {
        return GraphHelper.parseLabelsString((String)element.getProperty(Constants.LABELS_PROPERTY_KEY, String.class));
    }

    public static Integer getProvenanceType(AtlasElement element) {
        return (Integer)element.getProperty(Constants.PROVENANCE_TYPE_KEY, Integer.class);
    }

    public static String getTypeName(AtlasElement element) {
        return (String)element.getProperty(Constants.ENTITY_TYPE_PROPERTY_KEY, String.class);
    }

    public static Id.EntityState getState(AtlasElement element) {
        String state = GraphHelper.getStateAsString(element);
        return state == null ? null : Id.EntityState.valueOf((String)state);
    }

    public static Long getVersion(AtlasElement element) {
        return (Long)element.getProperty(Constants.VERSION_PROPERTY_KEY, Long.class);
    }

    public static String getStateAsString(AtlasElement element) {
        return (String)element.getProperty(Constants.STATE_PROPERTY_KEY, String.class);
    }

    public static AtlasEntity.Status getStatus(AtlasVertex vertex) {
        Object vertexId = vertex.getId();
        AtlasEntity.Status ret = GraphTransactionInterceptor.getVertexStateFromCache(vertexId);
        if (ret == null) {
            ret = GraphHelper.getState((AtlasElement)vertex) == Id.EntityState.DELETED ? AtlasEntity.Status.DELETED : AtlasEntity.Status.ACTIVE;
            GraphTransactionInterceptor.addToVertexStateCache(vertexId, ret);
        }
        return ret;
    }

    public static AtlasEntity.Status getStatus(AtlasEdge edge) {
        Object edgeId = edge.getId();
        AtlasEntity.Status ret = GraphTransactionInterceptor.getEdgeStateFromCache(edgeId);
        if (ret == null) {
            ret = GraphHelper.getState((AtlasElement)edge) == Id.EntityState.DELETED ? AtlasEntity.Status.DELETED : AtlasEntity.Status.ACTIVE;
            GraphTransactionInterceptor.addToEdgeStateCache(edgeId, ret);
        }
        return ret;
    }

    public static AtlasRelationship.Status getEdgeStatus(AtlasElement element) {
        return GraphHelper.getState(element) == Id.EntityState.DELETED ? AtlasRelationship.Status.DELETED : AtlasRelationship.Status.ACTIVE;
    }

    public static String getClassificationName(AtlasVertex classificationVertex) {
        return AtlasGraphUtilsV2.getEncodedProperty(classificationVertex, Constants.CLASSIFICATION_VERTEX_NAME_KEY, String.class);
    }

    public static String getClassificationEntityGuid(AtlasVertex classificationVertex) {
        return AtlasGraphUtilsV2.getEncodedProperty(classificationVertex, Constants.CLASSIFICATION_ENTITY_GUID, String.class);
    }

    public static Integer getIndexValue(AtlasEdge edge) {
        Integer index = (Integer)edge.getProperty(Constants.ATTRIBUTE_INDEX_PROPERTY_KEY, Integer.class);
        return index == null ? 0 : index;
    }

    public static boolean isPropagatedClassificationEdge(AtlasEdge edge) {
        Boolean isPropagated;
        boolean ret = false;
        if (edge != null && (isPropagated = (Boolean)edge.getProperty(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, Boolean.class)) != null) {
            ret = isPropagated;
        }
        return ret;
    }

    public static boolean isClassificationEdge(AtlasEdge edge) {
        boolean ret = false;
        if (edge != null) {
            String edgeLabel = edge.getLabel();
            Boolean isPropagated = (Boolean)edge.getProperty(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, Boolean.class);
            if (edgeLabel != null && isPropagated != null) {
                ret = edgeLabel.equals("classifiedAs") && isPropagated == false;
            }
        }
        return ret;
    }

    public static List<String> getBlockedClassificationIds(AtlasEdge edge) {
        List ret = null;
        if (edge != null) {
            List classificationIds = AtlasGraphUtilsV2.getEncodedProperty(edge, Constants.RELATIONSHIPTYPE_BLOCKED_PROPAGATED_CLASSIFICATIONS_KEY, List.class);
            ret = CollectionUtils.isNotEmpty((Collection)classificationIds) ? classificationIds : Collections.emptyList();
        }
        return ret;
    }

    public static AtlasRelationshipDef.PropagateTags getPropagateTags(AtlasElement element) {
        String propagateTags = (String)element.getProperty(Constants.RELATIONSHIPTYPE_TAG_PROPAGATION_KEY, String.class);
        return propagateTags == null ? null : AtlasRelationshipDef.PropagateTags.valueOf((String)propagateTags);
    }

    public static AtlasEntity.Status getClassificationEntityStatus(AtlasElement element) {
        String status = (String)element.getProperty(Constants.CLASSIFICATION_ENTITY_STATUS, String.class);
        return status == null ? null : AtlasEntity.Status.valueOf((String)status);
    }

    public static String getCreatedByAsString(AtlasElement element) {
        return (String)element.getProperty(Constants.CREATED_BY_KEY, String.class);
    }

    public static String getModifiedByAsString(AtlasElement element) {
        return (String)element.getProperty(Constants.MODIFIED_BY_KEY, String.class);
    }

    public static long getCreatedTime(AtlasElement element) {
        return (Long)element.getProperty(Constants.TIMESTAMP_PROPERTY_KEY, Long.class);
    }

    public static long getModifiedTime(AtlasElement element) {
        return (Long)element.getProperty(Constants.MODIFICATION_TIMESTAMP_PROPERTY_KEY, Long.class);
    }

    public static boolean isActive(AtlasEntity entity) {
        return entity != null ? entity.getStatus() == AtlasEntity.Status.ACTIVE : false;
    }

    public AtlasVertex getVertexForInstanceByUniqueAttribute(AtlasEntityType classType, Referenceable instance) throws AtlasException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Checking if there is an instance with the same unique attributes for instance {}", (Object)instance.toShortString());
        }
        AtlasVertex result = null;
        for (AtlasStructType.AtlasAttribute attributeInfo : classType.getUniqAttributes().values()) {
            String propertyKey = attributeInfo.getQualifiedName();
            try {
                result = this.findVertex(propertyKey, instance.get(attributeInfo.getName()), Constants.ENTITY_TYPE_PROPERTY_KEY, classType.getTypeName(), Constants.STATE_PROPERTY_KEY, Id.EntityState.ACTIVE.name());
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Found vertex by unique attribute : {}={}", (Object)propertyKey, instance.get(attributeInfo.getName()));
            }
            catch (EntityNotFoundException entityNotFoundException) {}
        }
        return result;
    }

    public List<AtlasVertex> getVerticesForInstancesByUniqueAttribute(AtlasEntityType classType, List<? extends Referenceable> instancesForClass) throws AtlasException {
        HashMap<String, AttributeValueMap> map = new HashMap<String, AttributeValueMap>();
        for (AtlasStructType.AtlasAttribute attributeInfo : classType.getUniqAttributes().values()) {
            String propertyKey = attributeInfo.getQualifiedName();
            AttributeValueMap mapForAttribute = new AttributeValueMap();
            for (int i = 0; i < instancesForClass.size(); ++i) {
                Referenceable instance = instancesForClass.get(i);
                Object value = instance.get(attributeInfo.getName());
                mapForAttribute.put(value, instance, i);
            }
            map.put(propertyKey, mapForAttribute);
        }
        AtlasVertex[] result = new AtlasVertex[instancesForClass.size()];
        if (map.isEmpty()) {
            return Arrays.asList(result);
        }
        AtlasGraphQuery query = this.graph.query();
        query.has(Constants.ENTITY_TYPE_PROPERTY_KEY, (Object)classType.getTypeName());
        query.has(Constants.STATE_PROPERTY_KEY, (Object)Id.EntityState.ACTIVE.name());
        ArrayList<AtlasGraphQuery> orChildren = new ArrayList<AtlasGraphQuery>();
        for (Map.Entry entry : map.entrySet()) {
            AtlasGraphQuery orChild = query.createChildQuery();
            String propertyName = (String)entry.getKey();
            AttributeValueMap valueMap = (AttributeValueMap)entry.getValue();
            Set<Object> values = valueMap.getAttributeValues();
            if (values.size() == 1) {
                orChild.has(propertyName, values.iterator().next());
            } else if (values.size() > 1) {
                orChild.in(propertyName, values);
            }
            orChildren.add(orChild);
        }
        if (orChildren.size() == 1) {
            AtlasGraphQuery child = (AtlasGraphQuery)orChildren.get(0);
            query.addConditionsFrom(child);
        } else if (orChildren.size() > 1) {
            query.or(orChildren);
        }
        Iterable queryResult = query.vertices();
        for (AtlasVertex matchingVertex : queryResult) {
            Collection<IndexedInstance> matches = this.getInstancesForVertex(map, matchingVertex);
            for (IndexedInstance wrapper : matches) {
                result[wrapper.getIndex()] = matchingVertex;
            }
        }
        return Arrays.asList(result);
    }

    private Collection<IndexedInstance> getInstancesForVertex(Map<String, AttributeValueMap> map, AtlasVertex foundVertex) {
        for (Map.Entry<String, AttributeValueMap> entry : map.entrySet()) {
            Object vertexValue;
            String propertyName = entry.getKey();
            AttributeValueMap valueMap = entry.getValue();
            Collection<IndexedInstance> instances = valueMap.get(vertexValue = foundVertex.getProperty(propertyName, Object.class));
            if (instances == null || instances.size() <= 0) continue;
            return instances;
        }
        return Collections.emptyList();
    }

    public static List<String> getTypeNames(List<AtlasVertex> vertices) {
        ArrayList<String> ret = new ArrayList<String>();
        if (CollectionUtils.isNotEmpty(vertices)) {
            for (AtlasVertex vertex : vertices) {
                String entityTypeProperty = (String)vertex.getProperty(Constants.ENTITY_TYPE_PROPERTY_KEY, String.class);
                if (entityTypeProperty == null) continue;
                ret.add(GraphHelper.getTypeName((AtlasElement)vertex));
            }
        }
        return ret;
    }

    public static AtlasVertex getAssociatedEntityVertex(AtlasVertex classificationVertex) {
        Iterator iterator;
        AtlasVertex ret = null;
        Iterable edges = classificationVertex.query().direction(AtlasEdgeDirection.IN).label("classifiedAs").has(Constants.CLASSIFICATION_EDGE_IS_PROPAGATED_PROPERTY_KEY, (Object)false).has(Constants.CLASSIFICATION_EDGE_NAME_PROPERTY_KEY, (Object)GraphHelper.getTypeName((AtlasElement)classificationVertex)).edges();
        if (edges != null && (iterator = edges.iterator()) != null && iterator.hasNext()) {
            AtlasEdge edge = (AtlasEdge)iterator.next();
            ret = edge.getOutVertex();
        }
        return ret;
    }

    public AtlasGraph getGraph() {
        return this.graph;
    }

    public Boolean getDefaultRemovePropagations() {
        return this.removePropagations;
    }

    public static boolean isInternalType(AtlasVertex vertex) {
        return vertex != null && GraphHelper.isInternalType(GraphHelper.getTypeName((AtlasElement)vertex));
    }

    public static boolean isInternalType(String typeName) {
        return typeName != null && typeName.startsWith(EDGE_LABEL_PREFIX);
    }

    public static List<Object> getArrayElementsProperty(AtlasType elementType, AtlasVertex instanceVertex, AtlasStructType.AtlasAttribute attribute) {
        String propertyName = attribute.getVertexPropertyName();
        if (AtlasGraphUtilsV2.isReference(elementType)) {
            return GraphHelper.getCollectionElementsUsingRelationship(instanceVertex, attribute);
        }
        return instanceVertex.getListProperty(propertyName);
    }

    public static Map<String, Object> getMapElementsProperty(AtlasMapType mapType, AtlasVertex instanceVertex, String propertyName, AtlasStructType.AtlasAttribute attribute) {
        AtlasType mapValueType = mapType.getValueType();
        if (AtlasGraphUtilsV2.isReference(mapValueType)) {
            return GraphHelper.getReferenceMap(instanceVertex, attribute);
        }
        return (Map)instanceVertex.getProperty(propertyName, Map.class);
    }

    public static Map<String, Object> getReferenceMap(AtlasVertex instanceVertex, AtlasStructType.AtlasAttribute attribute) {
        HashMap<String, Object> ret = new HashMap<String, Object>();
        List<AtlasEdge> referenceEdges = GraphHelper.getCollectionElementsUsingRelationship(instanceVertex, attribute);
        for (AtlasEdge edge : referenceEdges) {
            String key = (String)edge.getProperty(Constants.ATTRIBUTE_KEY_PROPERTY_KEY, String.class);
            if (!StringUtils.isNotEmpty((String)key)) continue;
            ret.put(key, edge);
        }
        return ret;
    }

    public static List<AtlasEdge> getMapValuesUsingRelationship(AtlasVertex vertex, AtlasStructType.AtlasAttribute attribute) {
        String edgeLabel = attribute.getRelationshipEdgeLabel();
        AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection edgeDirection = attribute.getRelationshipEdgeDirection();
        Iterator<AtlasEdge> edgesForLabel = GraphHelper.getEdgesForLabel(vertex, edgeLabel, edgeDirection);
        return IteratorUtils.toList(edgesForLabel);
    }

    public static Map<String, Object> getPrimitiveMap(AtlasVertex instanceVertex, String propertyName) {
        Map ret = (Map)instanceVertex.getProperty(AtlasGraphUtilsV2.encodePropertyKey(propertyName), Map.class);
        return ret;
    }

    public static List<AtlasEdge> getCollectionElementsUsingRelationship(AtlasVertex vertex, AtlasStructType.AtlasAttribute attribute) {
        String edgeLabel = attribute.getRelationshipEdgeLabel();
        AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection edgeDirection = attribute.getRelationshipEdgeDirection();
        Iterator<AtlasEdge> edgesForLabel = GraphHelper.getEdgesForLabel(vertex, edgeLabel, edgeDirection);
        List ret = IteratorUtils.toList(edgesForLabel);
        GraphHelper.sortCollectionElements(attribute, ret);
        return ret;
    }

    private static void sortCollectionElements(AtlasStructType.AtlasAttribute attribute, List<AtlasEdge> edges) {
        if (attribute.getAttributeType() instanceof AtlasArrayType && CollectionUtils.isNotEmpty(edges) && edges.get(0).getProperty(Constants.ATTRIBUTE_INDEX_PROPERTY_KEY, Integer.class) != null) {
            Collections.sort(edges, (e1, e2) -> {
                Integer e1Index = GraphHelper.getIndexValue(e1);
                Integer e2Index = GraphHelper.getIndexValue(e2);
                return e1Index.compareTo(e2Index);
            });
        }
    }

    public static void dumpToLog(AtlasGraph<?, ?> graph) {
        LOG.debug("*******************Graph Dump****************************");
        LOG.debug("Vertices of {}", graph);
        for (AtlasVertex vertex : graph.getVertices()) {
            LOG.debug(GraphHelper.vertexString(vertex));
        }
        LOG.debug("Edges of {}", graph);
        for (AtlasEdge edge : graph.getEdges()) {
            LOG.debug(GraphHelper.edgeString(edge));
        }
        LOG.debug("*******************Graph Dump****************************");
    }

    public static String string(Referenceable instance) {
        return String.format("entity[type=%s guid=%s]", instance.getTypeName(), instance.getId()._getId());
    }

    public static String string(AtlasVertex<?, ?> vertex) {
        if (vertex == null) {
            return "vertex[null]";
        }
        if (LOG.isDebugEnabled()) {
            return GraphHelper.getVertexDetails(vertex);
        }
        return String.format("vertex[id=%s]", vertex.getIdForDisplay());
    }

    public static String getVertexDetails(AtlasVertex<?, ?> vertex) {
        return String.format("vertex[id=%s type=%s guid=%s]", vertex.getIdForDisplay(), GraphHelper.getTypeName(vertex), GraphHelper.getGuid(vertex));
    }

    public static String string(AtlasEdge<?, ?> edge) {
        if (edge == null) {
            return "edge[null]";
        }
        if (LOG.isDebugEnabled()) {
            return GraphHelper.getEdgeDetails(edge);
        }
        return String.format("edge[id=%s]", edge.getIdForDisplay());
    }

    public static String getEdgeDetails(AtlasEdge<?, ?> edge) {
        return String.format("edge[id=%s label=%s from %s -> to %s]", edge.getIdForDisplay(), edge.getLabel(), GraphHelper.string(edge.getOutVertex()), GraphHelper.string(edge.getInVertex()));
    }

    public static String decodePropertyKey(String key) {
        if (StringUtils.isBlank((String)key)) {
            return key;
        }
        for (String encodedStr : RESERVED_CHARS_ENCODE_MAP.values()) {
            key = key.replace(encodedStr, (CharSequence)RESERVED_CHARS_ENCODE_MAP.inverse().get((Object)encodedStr));
        }
        return key;
    }

    public Object getVertexId(String guid) throws EntityNotFoundException {
        AtlasVertex instanceVertex = this.getVertexForGUID(guid);
        Object instanceVertexId = instanceVertex.getId();
        return instanceVertexId;
    }

    public static boolean elementExists(AtlasElement v) {
        return v != null && v.exists();
    }

    public static void setListPropertyFromElementIds(AtlasVertex<?, ?> instanceVertex, String propertyName, List<AtlasElement> elements) {
        String actualPropertyName = AtlasGraphUtilsV2.encodePropertyKey(propertyName);
        instanceVertex.setPropertyFromElementsIds(actualPropertyName, elements);
    }

    public static void setPropertyFromElementId(AtlasVertex<?, ?> instanceVertex, String propertyName, AtlasElement value) {
        String actualPropertyName = AtlasGraphUtilsV2.encodePropertyKey(propertyName);
        instanceVertex.setPropertyFromElementId(actualPropertyName, value);
    }

    public static void setListProperty(AtlasVertex instanceVertex, String propertyName, ArrayList<String> value) {
        String actualPropertyName = AtlasGraphUtilsV2.encodePropertyKey(propertyName);
        instanceVertex.setListProperty(actualPropertyName, value);
    }

    public static List<String> getListProperty(AtlasVertex instanceVertex, String propertyName) {
        String actualPropertyName = AtlasGraphUtilsV2.encodePropertyKey(propertyName);
        return instanceVertex.getListProperty(actualPropertyName);
    }

    private String getConditionString(Object[] args) {
        StringBuilder condition = new StringBuilder();
        for (int i = 0; i < args.length; i += 2) {
            condition.append(args[i]).append(" = ").append(args[i + 1]).append(", ");
        }
        return condition.toString();
    }

    public String getRelationshipTypeName(AtlasVertex entityVertex, AtlasEntityType entityType, String attributeName) {
        String ret = null;
        Set relationshipTypes = entityType.getAttributeRelationshipTypes(attributeName);
        if (CollectionUtils.isNotEmpty((Collection)relationshipTypes)) {
            if (relationshipTypes.size() == 1) {
                ret = (String)relationshipTypes.iterator().next();
            } else {
                Iterator iter = entityVertex.getEdges(AtlasEdgeDirection.IN).iterator();
                while (iter.hasNext() && ret == null) {
                    String edgeTypeName = AtlasGraphUtilsV2.getTypeName((AtlasElement)iter.next());
                    if (!relationshipTypes.contains(edgeTypeName)) continue;
                    ret = edgeTypeName;
                    break;
                }
                if (ret == null) {
                    ret = (String)relationshipTypes.iterator().next();
                }
            }
        }
        return ret;
    }

    public static String getReferencedEntityTypeName(AtlasVertex entityVertex, String relation) {
        String ret = null;
        Iterator<AtlasEdge> edges = GraphHelper.getAdjacentEdgesByLabel(entityVertex, AtlasEdgeDirection.BOTH, relation);
        if (edges != null && edges.hasNext()) {
            AtlasEdge relationEdge = edges.next();
            AtlasVertex outVertex = relationEdge.getOutVertex();
            AtlasVertex inVertex = relationEdge.getInVertex();
            if (outVertex != null && inVertex != null) {
                String entityVertexId;
                String outVertexId = outVertex.getIdForDisplay();
                AtlasVertex endVertex = StringUtils.equals((String)outVertexId, (String)(entityVertexId = entityVertex.getIdForDisplay())) ? inVertex : outVertex;
                ret = GraphHelper.getTypeName((AtlasElement)endVertex);
            }
        }
        return ret;
    }

    public static boolean isRelationshipEdge(AtlasEdge edge) {
        if (edge == null) {
            return false;
        }
        String edgeLabel = edge.getLabel();
        return StringUtils.isNotEmpty((String)edge.getLabel()) ? edgeLabel.startsWith("r:") : false;
    }

    public static AtlasObjectId getReferenceObjectId(AtlasEdge edge, AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection relationshipDirection, AtlasVertex parentVertex) {
        AtlasObjectId ret = null;
        if (relationshipDirection == AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection.OUT) {
            ret = GraphHelper.getAtlasObjectIdForInVertex(edge);
        } else if (relationshipDirection == AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection.IN) {
            ret = GraphHelper.getAtlasObjectIdForOutVertex(edge);
        } else if (relationshipDirection == AtlasStructType.AtlasAttribute.AtlasRelationshipEdgeDirection.BOTH) {
            ret = GraphHelper.verticesEquals(parentVertex, edge.getOutVertex()) ? GraphHelper.getAtlasObjectIdForInVertex(edge) : GraphHelper.getAtlasObjectIdForOutVertex(edge);
        }
        return ret;
    }

    public static AtlasObjectId getAtlasObjectIdForOutVertex(AtlasEdge edge) {
        return new AtlasObjectId(GraphHelper.getGuid(edge.getOutVertex()), GraphHelper.getTypeName((AtlasElement)edge.getOutVertex()));
    }

    public static AtlasObjectId getAtlasObjectIdForInVertex(AtlasEdge edge) {
        return new AtlasObjectId(GraphHelper.getGuid(edge.getInVertex()), GraphHelper.getTypeName((AtlasElement)edge.getInVertex()));
    }

    private static boolean verticesEquals(AtlasVertex vertexA, AtlasVertex vertexB) {
        return StringUtils.equals((String)GraphHelper.getGuid(vertexB), (String)GraphHelper.getGuid(vertexA));
    }

    public static String getDelimitedClassificationNames(Collection<String> classificationNames) {
        String ret = null;
        if (CollectionUtils.isNotEmpty(classificationNames)) {
            ret = "|" + StringUtils.join(classificationNames, (String)"|") + "|";
        }
        return ret;
    }

    private static Set<String> parseLabelsString(String labels) {
        HashSet<String> ret = new HashSet<String>();
        if (StringUtils.isNotEmpty((String)labels)) {
            ret.addAll(Arrays.asList(StringUtils.split((String)labels, (String)"\\|")));
        }
        return ret;
    }

    public static class VertexInfo {
        private final AtlasEntityHeader entity;
        private final AtlasVertex vertex;

        public VertexInfo(AtlasEntityHeader entity, AtlasVertex vertex) {
            this.entity = entity;
            this.vertex = vertex;
        }

        public AtlasEntityHeader getEntity() {
            return this.entity;
        }

        public AtlasVertex getVertex() {
            return this.vertex;
        }

        public String getGuid() {
            return this.entity.getGuid();
        }

        public String getTypeName() {
            return this.entity.getTypeName();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            VertexInfo that = (VertexInfo)o;
            return Objects.equals(this.entity, that.entity) && Objects.equals(this.vertex, that.vertex);
        }

        public int hashCode() {
            return Objects.hash(this.entity, this.vertex);
        }
    }
}

