/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.repository.graph;

import com.google.common.annotations.VisibleForTesting;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.Iterator;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.inject.Inject;
import org.apache.atlas.AtlasException;
import org.apache.atlas.ha.HAConfiguration;
import org.apache.atlas.listener.ActiveStateChangeHandler;
import org.apache.atlas.repository.Constants;
import org.apache.atlas.repository.graphdb.AtlasGraph;
import org.apache.atlas.repository.graphdb.AtlasGraphQuery;
import org.apache.atlas.repository.graphdb.AtlasVertex;
import org.apache.atlas.repository.store.graph.v2.AtlasGraphUtilsV2;
import org.apache.atlas.service.Service;
import org.apache.commons.configuration.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

@Component
@Order(value=8)
public class IndexRecoveryService
implements Service,
ActiveStateChangeHandler {
    private static final Logger LOG = LoggerFactory.getLogger(IndexRecoveryService.class);
    private static final String DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
    private static final String INDEX_HEALTH_MONITOR_THREAD_NAME = "index-health-monitor";
    private static final String SOLR_STATUS_CHECK_RETRY_INTERVAL = "atlas.graph.index.status.check.frequency";
    private static final String SOLR_INDEX_RECOVERY_CONFIGURED_START_TIME = "atlas.graph.index.recovery.start.time";
    private static final long SOLR_STATUS_RETRY_DEFAULT_MS = 30000L;
    private final Thread indexHealthMonitor;
    private final RecoveryInfoManagement recoveryInfoManagement;
    private Configuration configuration;
    private boolean isIndexRecoveryEnabled;
    private RecoveryThread recoveryThread;

    @Inject
    public IndexRecoveryService(Configuration config, AtlasGraph graph) {
        this.configuration = config;
        this.isIndexRecoveryEnabled = config.getBoolean("atlas.index.recovery.enable", true);
        long recoveryStartTimeFromConfig = this.getRecoveryStartTimeFromConfig(config);
        long healthCheckFrequencyMillis = config.getLong(SOLR_STATUS_CHECK_RETRY_INTERVAL, 30000L);
        this.recoveryInfoManagement = new RecoveryInfoManagement(graph);
        this.recoveryThread = new RecoveryThread(this.recoveryInfoManagement, graph, recoveryStartTimeFromConfig, healthCheckFrequencyMillis);
        this.indexHealthMonitor = new Thread((Runnable)this.recoveryThread, INDEX_HEALTH_MONITOR_THREAD_NAME);
    }

    private long getRecoveryStartTimeFromConfig(Configuration config) {
        long ret = 0L;
        try {
            String time = config.getString(SOLR_INDEX_RECOVERY_CONFIGURED_START_TIME);
            SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
            dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            ret = dateFormat.parse(time).toInstant().toEpochMilli();
        }
        catch (Exception e) {
            LOG.debug("Error fetching: {}", (Object)SOLR_INDEX_RECOVERY_CONFIGURED_START_TIME);
        }
        return ret;
    }

    public void start() throws AtlasException {
        if (this.configuration == null || !HAConfiguration.isHAEnabled((Configuration)this.configuration)) {
            LOG.info("==> IndexRecoveryService.start()");
            this.startTxLogMonitoring();
            LOG.info("<== IndexRecoveryService.start()");
        }
    }

    public void stop() throws AtlasException {
        try {
            this.recoveryThread.shutdown();
            this.indexHealthMonitor.join();
        }
        catch (InterruptedException e) {
            LOG.error("indexHealthMonitor: Interrupted", (Throwable)e);
        }
    }

    public void instanceIsActive() throws AtlasException {
        LOG.info("==> IndexRecoveryService.instanceIsActive()");
        this.startTxLogMonitoring();
        LOG.info("<== IndexRecoveryService.instanceIsActive()");
    }

    public void instanceIsPassive() throws AtlasException {
        LOG.info("==> IndexRecoveryService.instanceIsPassive()");
        this.stop();
        LOG.info("<== IndexRecoveryService.instanceIsPassive()");
    }

    public int getHandlerOrder() {
        return ActiveStateChangeHandler.HandlerOrder.INDEX_RECOVERY.getOrder();
    }

    private void startTxLogMonitoring() {
        if (!this.isIndexRecoveryEnabled) {
            LOG.warn("IndexRecoveryService: Recovery should be enabled.");
            return;
        }
        try {
            if (this.indexHealthMonitor.getState() == Thread.State.NEW) {
                this.indexHealthMonitor.start();
            }
        }
        catch (Exception ex) {
            LOG.error("Error while starting Index Health Monitor", (Throwable)ex);
        }
    }

    @VisibleForTesting
    static class RecoveryInfoManagement {
        private static final String INDEX_RECOVERY_TYPE_NAME = "__solrIndexRecoveryInfo";
        private final AtlasGraph graph;

        public RecoveryInfoManagement(AtlasGraph graph) {
            this.graph = graph;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void updateStartTime(Long startTime) {
            try {
                Long prevStartTime = null;
                AtlasVertex vertex = this.findVertex();
                if (vertex == null) {
                    vertex = this.graph.addVertex();
                } else {
                    prevStartTime = this.getStartTime(vertex);
                }
                AtlasGraphUtilsV2.setEncodedProperty(vertex, Constants.PROPERTY_KEY_INDEX_RECOVERY_NAME, INDEX_RECOVERY_TYPE_NAME);
                AtlasGraphUtilsV2.setEncodedProperty(vertex, Constants.PROPERTY_KEY_INDEX_RECOVERY_START_TIME, startTime);
                AtlasGraphUtilsV2.setEncodedProperty(vertex, Constants.PROPERTY_KEY_INDEX_RECOVERY_PREV_TIME, prevStartTime);
            }
            catch (Exception ex) {
                LOG.error("Error: Updating: {}!", (Throwable)ex);
            }
            finally {
                this.graph.commit();
            }
        }

        public Long getStartTime() {
            AtlasVertex vertex = this.findVertex();
            return this.getStartTime(vertex);
        }

        private Long getStartTime(AtlasVertex vertex) {
            if (vertex == null) {
                LOG.warn("Vertex passed is NULL: Returned is 0");
                return 0L;
            }
            Long startTime = 0L;
            try {
                startTime = (Long)vertex.getProperty(Constants.PROPERTY_KEY_INDEX_RECOVERY_START_TIME, Long.class);
            }
            catch (Exception e) {
                LOG.error("Error retrieving startTime", (Throwable)e);
            }
            return startTime;
        }

        private AtlasVertex findVertex() {
            AtlasGraphQuery query = this.graph.query().has(Constants.PROPERTY_KEY_INDEX_RECOVERY_NAME, (Object)INDEX_RECOVERY_TYPE_NAME);
            Iterator results = query.vertices().iterator();
            return results.hasNext() ? (AtlasVertex)results.next() : null;
        }
    }

    private static class RecoveryThread
    implements Runnable {
        private final AtlasGraph graph;
        private final RecoveryInfoManagement recoveryInfoManagement;
        private long indexStatusCheckRetryMillis;
        private Object txRecoveryObject;
        private final AtomicBoolean shouldRun = new AtomicBoolean(false);

        private RecoveryThread(RecoveryInfoManagement recoveryInfoManagement, AtlasGraph graph, long startTimeFromConfig, long healthCheckFrequencyMillis) {
            this.graph = graph;
            this.recoveryInfoManagement = recoveryInfoManagement;
            this.indexStatusCheckRetryMillis = healthCheckFrequencyMillis;
            if (startTimeFromConfig > 0L) {
                this.recoveryInfoManagement.updateStartTime(startTimeFromConfig);
            }
        }

        @Override
        public void run() {
            this.shouldRun.set(true);
            LOG.info("Index Health Monitor: Starting...");
            while (this.shouldRun.get()) {
                try {
                    boolean isIdxHealthy = this.isIndexBackendHealthy();
                    if (this.txRecoveryObject == null && isIdxHealthy) {
                        this.startMonitoring();
                    }
                    if (this.txRecoveryObject == null || isIdxHealthy) continue;
                    this.stopMonitoring();
                }
                catch (Exception e) {
                    LOG.error("Error: Index recovery monitoring!", (Throwable)e);
                }
            }
        }

        public void shutdown() {
            try {
                LOG.info("Index Health Monitor: Shutdown: Starting...");
                if (!this.shouldRun.get()) {
                    return;
                }
                this.shouldRun.set(false);
            }
            finally {
                LOG.info("Index Health Monitor: Shutdown: Done!");
            }
        }

        private boolean isIndexBackendHealthy() throws AtlasException, InterruptedException {
            Thread.sleep(this.indexStatusCheckRetryMillis);
            return this.graph.getGraphIndexClient().isHealthy();
        }

        private void startMonitoring() {
            Long startTime = null;
            try {
                startTime = this.recoveryInfoManagement.getStartTime();
                this.txRecoveryObject = this.graph.getManagementSystem().startIndexRecovery(startTime.longValue());
                this.printIndexRecoveryStats();
            }
            catch (Exception e) {
                LOG.error("Index Recovery: Start: Error!", (Throwable)e);
            }
            finally {
                LOG.info("Index Recovery: Started! Recovery time: {}", (Object)Instant.ofEpochMilli(startTime));
            }
        }

        private void stopMonitoring() {
            Instant newStartTime = Instant.now().minusMillis(this.indexStatusCheckRetryMillis);
            try {
                this.graph.getManagementSystem().stopIndexRecovery(this.txRecoveryObject);
                this.recoveryInfoManagement.updateStartTime(newStartTime.toEpochMilli());
                this.printIndexRecoveryStats();
            }
            catch (Exception e) {
                LOG.info("Index Recovery: Stopped! Error!", (Throwable)e);
            }
            finally {
                this.txRecoveryObject = null;
                LOG.info("Index Recovery: Stopped! Recovery time: {}", (Object)newStartTime);
            }
        }

        private void printIndexRecoveryStats() {
            this.graph.getManagementSystem().printIndexRecoveryStats(this.txRecoveryObject);
        }
    }
}

