/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.util;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.concurrent.atomic.AtomicLong;

public class AtlasMetricsCounter {
    private final String name;
    private final Stats stats;
    private Clock clock;
    private Instant lastIncrTime;
    private Instant dayStartTime;
    private Instant dayEndTime;
    private Instant hourStartTime;
    private Instant hourEndTime;

    public AtlasMetricsCounter(String name) {
        this(name, Clock.systemUTC());
    }

    public AtlasMetricsCounter(String name, Clock clock) {
        this.name = name;
        this.stats = new Stats();
        this.init(clock);
    }

    public String getName() {
        return this.name;
    }

    public Instant getLastIncrTime() {
        return this.lastIncrTime;
    }

    public void incr() {
        this.incrByWithMeasure(1L, 0L);
    }

    public void incrBy(long count) {
        this.incrByWithMeasure(count, 0L);
    }

    public void incrWithMeasure(long measure) {
        this.incrByWithMeasure(1L, measure);
    }

    public void incrByWithMeasure(long count, long measure) {
        Instant instant = this.clock.instant();
        this.stats.addCount(Period.ALL, count);
        this.stats.addMeasure(Period.ALL, measure);
        if (instant.isAfter(this.dayStartTime)) {
            this.lastIncrTime = instant;
            this.updateForTime(instant);
            this.stats.addCount(Period.CURR_DAY, count);
            this.stats.addMeasure(Period.CURR_DAY, measure);
            if (instant.isAfter(this.hourStartTime)) {
                this.stats.addCount(Period.CURR_HOUR, count);
                this.stats.addMeasure(Period.CURR_HOUR, measure);
            }
        }
    }

    public StatsReport report() {
        this.updateForTime(this.clock.instant());
        return new StatsReport(this.stats, this.dayStartTime.toEpochMilli(), this.hourStartTime.toEpochMilli());
    }

    void init(Clock clock) {
        this.clock = clock;
        this.lastIncrTime = Instant.ofEpochSecond(0L);
        this.dayStartTime = Instant.ofEpochSecond(0L);
        this.dayEndTime = Instant.ofEpochSecond(0L);
        this.hourStartTime = Instant.ofEpochSecond(0L);
        this.hourEndTime = Instant.ofEpochSecond(0L);
        this.updateForTime(clock.instant());
    }

    protected void updateForTime(Instant now) {
        Instant dayEndTime = this.dayEndTime;
        Instant hourEndTime = this.hourEndTime;
        if (now.isAfter(dayEndTime)) {
            this.rolloverDay(dayEndTime, now);
            this.rolloverHour(hourEndTime, now);
        } else if (now.isAfter(hourEndTime)) {
            this.rolloverHour(hourEndTime, now);
        }
    }

    protected synchronized void rolloverDay(Instant fromDayEndTime, Instant now) {
        if (fromDayEndTime == this.dayEndTime) {
            Instant dayStartTime = AtlasMetricsCounter.getDayStartTime(now);
            if (dayStartTime.equals(this.dayEndTime)) {
                this.stats.copy(Period.CURR_DAY, Period.PREV_DAY);
            } else {
                this.stats.reset(Period.PREV_DAY);
            }
            this.stats.reset(Period.CURR_DAY);
            this.dayStartTime = dayStartTime;
            this.dayEndTime = AtlasMetricsCounter.getNextDayStartTime(now);
        }
    }

    protected synchronized void rolloverHour(Instant fromHourEndTime, Instant now) {
        if (fromHourEndTime == this.hourEndTime) {
            Instant hourStartTime = AtlasMetricsCounter.getHourStartTime(now);
            if (hourStartTime.equals(this.hourEndTime)) {
                this.stats.copy(Period.CURR_HOUR, Period.PREV_HOUR);
            } else {
                this.stats.reset(Period.PREV_HOUR);
            }
            this.stats.reset(Period.CURR_HOUR);
            this.hourStartTime = hourStartTime;
            this.hourEndTime = AtlasMetricsCounter.getNextHourStartTime(now);
        }
    }

    public static LocalDateTime getLocalDateTime(Instant instant) {
        return LocalDateTime.ofInstant(instant, ZoneOffset.UTC);
    }

    public static Instant getHourStartTime(Instant instant) {
        LocalDateTime time = AtlasMetricsCounter.getLocalDateTime(instant);
        return LocalDateTime.of(time.toLocalDate(), LocalTime.MIN).plusHours(time.getHour()).toInstant(ZoneOffset.UTC);
    }

    public static Instant getNextHourStartTime(Instant instant) {
        LocalDateTime time = AtlasMetricsCounter.getLocalDateTime(instant);
        return LocalDateTime.of(time.toLocalDate(), LocalTime.MIN).plusHours(time.getHour() + 1).toInstant(ZoneOffset.UTC);
    }

    public static Instant getDayStartTime(Instant instant) {
        LocalDateTime time = AtlasMetricsCounter.getLocalDateTime(instant);
        return LocalDateTime.of(time.toLocalDate(), LocalTime.MIN).toInstant(ZoneOffset.UTC);
    }

    public static Instant getNextDayStartTime(Instant instant) {
        LocalDateTime time = AtlasMetricsCounter.getLocalDateTime(instant);
        return LocalDateTime.of(time.toLocalDate().plusDays(1L), LocalTime.MIN).toInstant(ZoneOffset.UTC);
    }

    public static class StatsReport {
        private static final int NUM_PERIOD = Period.values().length;
        private final long dayStartTimeMs;
        private final long hourStartTimeMs;
        private final long[] count = new long[NUM_PERIOD];
        private final long[] measureSum = new long[NUM_PERIOD];
        private final long[] measureMin = new long[NUM_PERIOD];
        private final long[] measureMax = new long[NUM_PERIOD];

        public StatsReport(Stats other, long dayStartTimeMs, long hourStartTimeMs) {
            this.dayStartTimeMs = dayStartTimeMs;
            this.hourStartTimeMs = hourStartTimeMs;
            this.copy(other.count, this.count);
            this.copy(other.measureSum, this.measureSum);
            this.copy(other.measureMin, this.measureMin);
            this.copy(other.measureMax, this.measureMax);
        }

        public long getDayStartTimeMs() {
            return this.dayStartTimeMs;
        }

        public long getHourStartTimeMs() {
            return this.hourStartTimeMs;
        }

        public long getCount(Period period) {
            return this.count[period.ordinal()];
        }

        public long getMeasureSum(Period period) {
            return this.measureSum[period.ordinal()];
        }

        public long getMeasureMin(Period period) {
            return this.measureMin[period.ordinal()];
        }

        public long getMeasureMax(Period period) {
            return this.measureMax[period.ordinal()];
        }

        public long getMeasureAvg(Period period) {
            int idx = period.ordinal();
            long c = this.count[idx];
            return c != 0L ? this.measureSum[idx] / c : 0L;
        }

        private void copy(AtomicLong[] src, long[] dest) {
            for (int i = 0; i < dest.length; ++i) {
                dest[i] = src[i].get();
            }
        }
    }

    public static class Stats {
        private static final int NUM_PERIOD = Period.values().length;
        private final long dayStartTimeMs;
        private final long hourStartTimeMs;
        private final AtomicLong[] count = new AtomicLong[NUM_PERIOD];
        private final AtomicLong[] measureSum = new AtomicLong[NUM_PERIOD];
        private final AtomicLong[] measureMin = new AtomicLong[NUM_PERIOD];
        private final AtomicLong[] measureMax = new AtomicLong[NUM_PERIOD];

        public Stats() {
            this.dayStartTimeMs = 0L;
            this.hourStartTimeMs = 0L;
            for (Period period : Period.values()) {
                this.reset(period);
            }
        }

        public void addCount(Period period, long num) {
            this.count[period.ordinal()].addAndGet(num);
        }

        public void addMeasure(Period period, long measure) {
            int idx = period.ordinal();
            this.measureSum[idx].addAndGet(measure);
            if (this.measureMin[idx].get() > measure) {
                this.measureMin[idx].set(measure);
            }
            if (this.measureMax[idx].get() < measure) {
                this.measureMax[idx].set(measure);
            }
        }

        private void copy(Period src, Period dest) {
            int srcIdx = src.ordinal();
            int destIdx = dest.ordinal();
            this.count[destIdx].set(this.count[srcIdx].get());
            this.measureSum[destIdx].set(this.measureSum[srcIdx].get());
            this.measureMin[destIdx].set(this.measureMin[srcIdx].get());
            this.measureMax[destIdx].set(this.measureMax[srcIdx].get());
        }

        private void reset(Period period) {
            int idx = period.ordinal();
            this.count[idx] = new AtomicLong(0L);
            this.measureSum[idx] = new AtomicLong(0L);
            this.measureMin[idx] = new AtomicLong(Long.MAX_VALUE);
            this.measureMax[idx] = new AtomicLong(Long.MIN_VALUE);
        }
    }

    public static enum Period {
        ALL,
        CURR_DAY,
        CURR_HOUR,
        PREV_HOUR,
        PREV_DAY;

    }
}

