/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.hbase.bridge;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.atlas.ApplicationProperties;
import org.apache.atlas.AtlasClientV2;
import org.apache.atlas.hbase.model.HBaseDataTypes;
import org.apache.atlas.hook.AtlasHookException;
import org.apache.atlas.model.instance.AtlasEntity;
import org.apache.atlas.model.instance.AtlasEntityHeader;
import org.apache.atlas.model.instance.AtlasObjectId;
import org.apache.atlas.model.instance.EntityMutationResponse;
import org.apache.atlas.type.AtlasTypeUtil;
import org.apache.atlas.utils.AuthenticationUtil;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HBaseBridge {
    private static final Logger LOG = LoggerFactory.getLogger(HBaseBridge.class);
    private static final int EXIT_CODE_SUCCESS = 0;
    private static final int EXIT_CODE_FAILED = 1;
    private static final String ATLAS_ENDPOINT = "atlas.rest.address";
    private static final String DEFAULT_ATLAS_URL = "http://localhost:21000/";
    private static final String HBASE_CLUSTER_NAME = "atlas.cluster.name";
    private static final String DEFAULT_CLUSTER_NAME = "primary";
    private static final String QUALIFIED_NAME = "qualifiedName";
    private static final String NAME = "name";
    private static final String URI = "uri";
    private static final String OWNER = "owner";
    private static final String DESCRIPTION_ATTR = "description";
    private static final String CLUSTERNAME = "clusterName";
    private static final String NAMESPACE = "namespace";
    private static final String TABLE = "table";
    private static final String COLUMN_FAMILIES = "column_families";
    private static final String ATTR_TABLE_MAX_FILESIZE = "maxFileSize";
    private static final String ATTR_TABLE_ISREADONLY = "isReadOnly";
    private static final String ATTR_TABLE_ISCOMPACTION_ENABLED = "isCompactionEnabled";
    private static final String ATTR_TABLE_REPLICATION_PER_REGION = "replicasPerRegion";
    private static final String ATTR_TABLE_DURABLILITY = "durability";
    private static final String ATTR_CF_BLOOMFILTER_TYPE = "bloomFilterType";
    private static final String ATTR_CF_COMPRESSION_TYPE = "compressionType";
    private static final String ATTR_CF_COMPACTION_COMPRESSION_TYPE = "compactionCompressionType";
    private static final String ATTR_CF_ENCRYPTION_TYPE = "encryptionType";
    private static final String ATTR_CF_KEEP_DELETE_CELLS = "keepDeletedCells";
    private static final String ATTR_CF_MAX_VERSIONS = "maxVersions";
    private static final String ATTR_CF_MIN_VERSIONS = "minVersions";
    private static final String ATTR_CF_DATA_BLOCK_ENCODING = "dataBlockEncoding";
    private static final String ATTR_CF_TTL = "ttl";
    private static final String ATTR_CF_BLOCK_CACHE_ENABLED = "blockCacheEnabled";
    private static final String ATTR_CF_CACHED_BLOOM_ON_WRITE = "cacheBloomsOnWrite";
    private static final String ATTR_CF_CACHED_DATA_ON_WRITE = "cacheDataOnWrite";
    private static final String ATTR_CF_CACHED_INDEXES_ON_WRITE = "cacheIndexesOnWrite";
    private static final String ATTR_CF_EVICT_BLOCK_ONCLOSE = "evictBlocksOnClose";
    private static final String ATTR_CF_PREFETCH_BLOCK_ONOPEN = "prefetchBlocksOnOpen";
    private static final String ATTRIBUTE_QUALIFIED_NAME = "qualifiedName";
    private static final String HBASE_NAMESPACE_QUALIFIED_NAME = "%s@%s";
    private static final String HBASE_TABLE_QUALIFIED_NAME_FORMAT = "%s:%s@%s";
    private static final String HBASE_COLUMN_FAMILY_QUALIFIED_NAME_FORMAT = "%s:%s.%s@%s";
    private final String clusterName;
    private final AtlasClientV2 atlasClientV2;
    private final HBaseAdmin hbaseAdmin;

    public static void main(String[] args) {
        int exitCode = 1;
        try {
            AtlasClientV2 atlasClientV2;
            Options options = new Options();
            options.addOption("n", NAMESPACE, true, NAMESPACE);
            options.addOption("t", TABLE, true, "tablename");
            options.addOption("f", "filename", true, "filename");
            BasicParser parser = new BasicParser();
            CommandLine cmd = parser.parse(options, args);
            Object namespaceToImport = cmd.getOptionValue("n");
            Object tableToImport = cmd.getOptionValue("t");
            String fileToImport = cmd.getOptionValue("f");
            Configuration atlasConf = ApplicationProperties.get();
            String[] urls = atlasConf.getStringArray(ATLAS_ENDPOINT);
            if (urls == null || urls.length == 0) {
                urls = new String[]{DEFAULT_ATLAS_URL};
            }
            if (!AuthenticationUtil.isKerberosAuthenticationEnabled()) {
                String[] basicAuthUsernamePassword = AuthenticationUtil.getBasicAuthenticationInput();
                atlasClientV2 = new AtlasClientV2(urls, basicAuthUsernamePassword);
            } else {
                UserGroupInformation ugi = UserGroupInformation.getCurrentUser();
                atlasClientV2 = new AtlasClientV2(ugi, ugi.getShortUserName(), urls);
            }
            HBaseBridge importer = new HBaseBridge(atlasConf, atlasClientV2);
            if (StringUtils.isNotEmpty((String)fileToImport)) {
                File f = new File(fileToImport);
                if (f.exists() && f.canRead()) {
                    BufferedReader br = new BufferedReader(new FileReader(f));
                    String line = null;
                    while ((line = br.readLine()) != null) {
                        Object[] val = line.split(":");
                        if (!ArrayUtils.isNotEmpty((Object[])val)) continue;
                        namespaceToImport = val[0];
                        tableToImport = val.length > 1 ? val[1] : "";
                        importer.importHBaseEntities((String)namespaceToImport, (String)tableToImport);
                    }
                    exitCode = 0;
                } else {
                    LOG.error("Failed to read the file");
                }
            } else {
                importer.importHBaseEntities((String)namespaceToImport, (String)tableToImport);
                exitCode = 0;
            }
        }
        catch (ParseException e) {
            LOG.error("Failed to parse arguments. Error: ", (Object)e.getMessage());
            HBaseBridge.printUsage();
        }
        catch (Exception e) {
            System.out.println("ImportHBaseEntities failed. Please check the log file for the detailed error message");
            LOG.error("ImportHBaseEntities failed", (Throwable)e);
        }
        System.exit(exitCode);
    }

    public HBaseBridge(Configuration atlasConf, AtlasClientV2 atlasClientV2) throws Exception {
        this.atlasClientV2 = atlasClientV2;
        this.clusterName = atlasConf.getString(HBASE_CLUSTER_NAME, DEFAULT_CLUSTER_NAME);
        org.apache.hadoop.conf.Configuration conf = HBaseConfiguration.create();
        LOG.info("checking HBase availability..");
        HBaseAdmin.checkHBaseAvailable((org.apache.hadoop.conf.Configuration)conf);
        LOG.info("HBase is available");
        this.hbaseAdmin = new HBaseAdmin(conf);
    }

    private boolean importHBaseEntities(String namespaceToImport, String tableToImport) throws Exception {
        boolean ret = false;
        if (StringUtils.isEmpty((String)namespaceToImport) && StringUtils.isEmpty((String)tableToImport)) {
            this.importNameSpaceAndTable();
            ret = true;
        } else if (StringUtils.isNotEmpty((String)namespaceToImport)) {
            this.importNameSpaceWithTable(namespaceToImport, tableToImport);
            ret = true;
        } else if (StringUtils.isNotEmpty((String)tableToImport)) {
            this.importTable(tableToImport);
            ret = true;
        }
        return ret;
    }

    public void importNameSpace(String nameSpace) throws Exception {
        List<NamespaceDescriptor> matchingNameSpaceDescriptors = this.getMatchingNameSpaces(nameSpace);
        if (CollectionUtils.isNotEmpty(matchingNameSpaceDescriptors)) {
            for (NamespaceDescriptor namespaceDescriptor : matchingNameSpaceDescriptors) {
                this.createOrUpdateNameSpace(namespaceDescriptor);
            }
        } else {
            throw new AtlasHookException("No NameSpace found for the given criteria. NameSpace = " + nameSpace);
        }
    }

    public void importTable(String tableName) throws Exception {
        String tableNameStr = null;
        Object[] htds = this.hbaseAdmin.listTables(Pattern.compile(tableName));
        if (ArrayUtils.isNotEmpty((Object[])htds)) {
            for (Object htd : htds) {
                String tblNameWithNameSpace = htd.getTableName().getNameWithNamespaceInclAsString();
                String tblNameWithOutNameSpace = htd.getTableName().getNameAsString();
                if (tableName.equals(tblNameWithNameSpace)) {
                    tableNameStr = tblNameWithNameSpace;
                } else if (tableName.equals(tblNameWithOutNameSpace)) {
                    tableNameStr = tblNameWithOutNameSpace;
                } else if (tblNameWithNameSpace != null) {
                    tableNameStr = tblNameWithNameSpace;
                } else if (tblNameWithOutNameSpace != null) {
                    tableNameStr = tblNameWithOutNameSpace;
                }
                byte[] nsByte = htd.getTableName().getNamespace();
                String nsName = new String(nsByte);
                NamespaceDescriptor nsDescriptor = this.hbaseAdmin.getNamespaceDescriptor(nsName);
                AtlasEntity.AtlasEntityWithExtInfo entity = this.createOrUpdateNameSpace(nsDescriptor);
                HColumnDescriptor[] hcdts = htd.getColumnFamilies();
                this.createOrUpdateTable(nsName, tableNameStr, entity.getEntity(), (HTableDescriptor)htd, hcdts);
            }
        } else {
            throw new AtlasHookException("No Table found for the given criteria. Table = " + tableName);
        }
    }

    private void importNameSpaceAndTable() throws Exception {
        Object[] htds;
        Object[] namespaceDescriptors = this.hbaseAdmin.listNamespaceDescriptors();
        if (ArrayUtils.isNotEmpty((Object[])namespaceDescriptors)) {
            for (Object namespaceDescriptor : namespaceDescriptors) {
                String namespace = namespaceDescriptor.getName();
                this.importNameSpace(namespace);
            }
        }
        if (ArrayUtils.isNotEmpty((Object[])(htds = this.hbaseAdmin.listTables()))) {
            for (Object htd : htds) {
                String tableName = htd.getNameAsString();
                this.importTable(tableName);
            }
        }
    }

    private void importNameSpaceWithTable(String namespaceToImport, String tableToImport) throws Exception {
        this.importNameSpace(namespaceToImport);
        List<Object> hTableDescriptors = new ArrayList();
        if (StringUtils.isEmpty((String)tableToImport)) {
            List<NamespaceDescriptor> matchingNameSpaceDescriptors = this.getMatchingNameSpaces(namespaceToImport);
            if (CollectionUtils.isNotEmpty(matchingNameSpaceDescriptors)) {
                hTableDescriptors = this.getTableDescriptors(matchingNameSpaceDescriptors);
            }
        } else {
            tableToImport = namespaceToImport + ":" + tableToImport;
            HTableDescriptor[] htds = this.hbaseAdmin.listTables(Pattern.compile(tableToImport));
            hTableDescriptors.addAll(Arrays.asList(htds));
        }
        if (CollectionUtils.isNotEmpty(hTableDescriptors)) {
            for (HTableDescriptor hTableDescriptor : hTableDescriptors) {
                String tblName = hTableDescriptor.getTableName().getNameAsString();
                this.importTable(tblName);
            }
        }
    }

    private List<NamespaceDescriptor> getMatchingNameSpaces(String nameSpace) throws Exception {
        ArrayList<NamespaceDescriptor> ret = new ArrayList<NamespaceDescriptor>();
        NamespaceDescriptor[] namespaceDescriptors = this.hbaseAdmin.listNamespaceDescriptors();
        Pattern pattern = Pattern.compile(nameSpace);
        for (NamespaceDescriptor namespaceDescriptor : namespaceDescriptors) {
            String nmSpace = namespaceDescriptor.getName();
            Matcher matcher = pattern.matcher(nmSpace);
            if (!matcher.find()) continue;
            ret.add(namespaceDescriptor);
        }
        return ret;
    }

    private List<HTableDescriptor> getTableDescriptors(List<NamespaceDescriptor> namespaceDescriptors) throws Exception {
        ArrayList<HTableDescriptor> ret = new ArrayList<HTableDescriptor>();
        for (NamespaceDescriptor namespaceDescriptor : namespaceDescriptors) {
            HTableDescriptor[] tableDescriptors = this.hbaseAdmin.listTableDescriptorsByNamespace(namespaceDescriptor.getName());
            ret.addAll(Arrays.asList(tableDescriptors));
        }
        return ret;
    }

    protected AtlasEntity.AtlasEntityWithExtInfo createOrUpdateNameSpace(NamespaceDescriptor namespaceDescriptor) throws Exception {
        String nsName = namespaceDescriptor.getName();
        String nsQualifiedName = HBaseBridge.getNameSpaceQualifiedName(this.clusterName, nsName);
        AtlasEntity.AtlasEntityWithExtInfo nsEntity = this.findNameSpaceEntityInAtlas(nsQualifiedName);
        if (nsEntity == null) {
            LOG.info("Importing NameSpace: " + nsQualifiedName);
            AtlasEntity entity = this.getNameSpaceEntity(nsName, null);
            nsEntity = this.createEntityInAtlas(new AtlasEntity.AtlasEntityWithExtInfo(entity));
        } else {
            LOG.info("NameSpace already present in Atlas. Updating it..: " + nsQualifiedName);
            AtlasEntity entity = this.getNameSpaceEntity(nsName, nsEntity.getEntity());
            nsEntity.setEntity(entity);
            nsEntity = this.updateEntityInAtlas(nsEntity);
        }
        return nsEntity;
    }

    protected AtlasEntity.AtlasEntityWithExtInfo createOrUpdateTable(String nameSpace, String tableName, AtlasEntity nameSapceEntity, HTableDescriptor htd, HColumnDescriptor[] hcdts) throws Exception {
        AtlasEntity entity;
        String owner = htd.getOwnerString();
        String tblQualifiedName = HBaseBridge.getTableQualifiedName(this.clusterName, nameSpace, tableName);
        AtlasEntity.AtlasEntityWithExtInfo ret = this.findTableEntityInAtlas(tblQualifiedName);
        if (ret == null) {
            LOG.info("Importing Table: " + tblQualifiedName);
            entity = this.getTableEntity(nameSpace, tableName, owner, nameSapceEntity, htd, null);
            ret = this.createEntityInAtlas(new AtlasEntity.AtlasEntityWithExtInfo(entity));
        } else {
            LOG.info("Table already present in Atlas. Updating it..: " + tblQualifiedName);
            entity = this.getTableEntity(nameSpace, tableName, owner, nameSapceEntity, htd, ret.getEntity());
            ret.setEntity(entity);
            ret = this.updateEntityInAtlas(ret);
        }
        AtlasEntity tableEntity = ret.getEntity();
        if (tableEntity != null) {
            List<AtlasEntity.AtlasEntityWithExtInfo> cfEntities = this.createOrUpdateColumnFamilies(nameSpace, tableName, owner, hcdts, tableEntity);
            ArrayList<AtlasObjectId> cfIDs = new ArrayList<AtlasObjectId>();
            if (CollectionUtils.isNotEmpty(cfEntities)) {
                for (AtlasEntity.AtlasEntityWithExtInfo cfEntity : cfEntities) {
                    cfIDs.add(AtlasTypeUtil.getAtlasObjectId((AtlasEntity)cfEntity.getEntity()));
                }
            }
            tableEntity.setAttribute(COLUMN_FAMILIES, cfIDs);
        }
        return ret;
    }

    protected List<AtlasEntity.AtlasEntityWithExtInfo> createOrUpdateColumnFamilies(String nameSpace, String tableName, String owner, HColumnDescriptor[] hcdts, AtlasEntity tableEntity) throws Exception {
        ArrayList<AtlasEntity.AtlasEntityWithExtInfo> ret = new ArrayList<AtlasEntity.AtlasEntityWithExtInfo>();
        if (hcdts != null) {
            AtlasObjectId tableId = AtlasTypeUtil.getAtlasObjectId((AtlasEntity)tableEntity);
            for (HColumnDescriptor columnFamilyDescriptor : hcdts) {
                AtlasEntity entity;
                String cfName = columnFamilyDescriptor.getNameAsString();
                String cfQualifiedName = HBaseBridge.getColumnFamilyQualifiedName(this.clusterName, nameSpace, tableName, cfName);
                AtlasEntity.AtlasEntityWithExtInfo cfEntity = this.findColumnFamiltyEntityInAtlas(cfQualifiedName);
                if (cfEntity == null) {
                    LOG.info("Importing Column-family: " + cfQualifiedName);
                    entity = this.getColumnFamilyEntity(nameSpace, tableName, owner, columnFamilyDescriptor, tableId, null);
                    cfEntity = this.createEntityInAtlas(new AtlasEntity.AtlasEntityWithExtInfo(entity));
                } else {
                    LOG.info("ColumnFamily already present in Atlas. Updating it..: " + cfQualifiedName);
                    entity = this.getColumnFamilyEntity(nameSpace, tableName, owner, columnFamilyDescriptor, tableId, cfEntity.getEntity());
                    cfEntity.setEntity(entity);
                    cfEntity = this.updateEntityInAtlas(cfEntity);
                }
                ret.add(cfEntity);
            }
        }
        return ret;
    }

    private AtlasEntity.AtlasEntityWithExtInfo findNameSpaceEntityInAtlas(String nsQualifiedName) {
        AtlasEntity.AtlasEntityWithExtInfo ret = null;
        try {
            ret = this.findEntityInAtlas(HBaseDataTypes.HBASE_NAMESPACE.getName(), nsQualifiedName);
        }
        catch (Exception e) {
            ret = null;
        }
        return ret;
    }

    private AtlasEntity.AtlasEntityWithExtInfo findTableEntityInAtlas(String tableQualifiedName) {
        AtlasEntity.AtlasEntityWithExtInfo ret = null;
        try {
            ret = this.findEntityInAtlas(HBaseDataTypes.HBASE_TABLE.getName(), tableQualifiedName);
        }
        catch (Exception e) {
            ret = null;
        }
        return ret;
    }

    private AtlasEntity.AtlasEntityWithExtInfo findColumnFamiltyEntityInAtlas(String columnFamilyQualifiedName) {
        AtlasEntity.AtlasEntityWithExtInfo ret = null;
        try {
            ret = this.findEntityInAtlas(HBaseDataTypes.HBASE_COLUMN_FAMILY.getName(), columnFamilyQualifiedName);
        }
        catch (Exception e) {
            ret = null;
        }
        return ret;
    }

    private AtlasEntity.AtlasEntityWithExtInfo findEntityInAtlas(String typeName, String qualifiedName) throws Exception {
        Map<String, String> attributes = Collections.singletonMap("qualifiedName", qualifiedName);
        return this.atlasClientV2.getEntityByAttribute(typeName, attributes);
    }

    private AtlasEntity getNameSpaceEntity(String nameSpace, AtlasEntity nsEtity) {
        AtlasEntity ret = null;
        ret = nsEtity == null ? new AtlasEntity(HBaseDataTypes.HBASE_NAMESPACE.getName()) : nsEtity;
        String qualifiedName = HBaseBridge.getNameSpaceQualifiedName(this.clusterName, nameSpace);
        ret.setAttribute("qualifiedName", (Object)qualifiedName);
        ret.setAttribute(CLUSTERNAME, (Object)this.clusterName);
        ret.setAttribute(NAME, (Object)nameSpace);
        ret.setAttribute(DESCRIPTION_ATTR, (Object)nameSpace);
        return ret;
    }

    private AtlasEntity getTableEntity(String nameSpace, String tableName, String owner, AtlasEntity nameSpaceEntity, HTableDescriptor htd, AtlasEntity atlasEntity) {
        AtlasEntity ret = null;
        ret = atlasEntity == null ? new AtlasEntity(HBaseDataTypes.HBASE_TABLE.getName()) : atlasEntity;
        String tableQualifiedName = HBaseBridge.getTableQualifiedName(this.clusterName, nameSpace, tableName);
        ret.setAttribute("qualifiedName", (Object)tableQualifiedName);
        ret.setAttribute(CLUSTERNAME, (Object)this.clusterName);
        ret.setAttribute(NAMESPACE, (Object)AtlasTypeUtil.getAtlasObjectId((AtlasEntity)nameSpaceEntity));
        ret.setAttribute(NAME, (Object)tableName);
        ret.setAttribute(DESCRIPTION_ATTR, (Object)tableName);
        ret.setAttribute(OWNER, (Object)owner);
        ret.setAttribute(URI, (Object)tableName);
        ret.setAttribute(ATTR_TABLE_MAX_FILESIZE, (Object)htd.getMaxFileSize());
        ret.setAttribute(ATTR_TABLE_REPLICATION_PER_REGION, (Object)htd.getRegionReplication());
        ret.setAttribute(ATTR_TABLE_ISREADONLY, (Object)htd.isReadOnly());
        ret.setAttribute(ATTR_TABLE_ISCOMPACTION_ENABLED, (Object)htd.isCompactionEnabled());
        ret.setAttribute(ATTR_TABLE_DURABLILITY, htd.getDurability() != null ? htd.getDurability().name() : null);
        return ret;
    }

    private AtlasEntity getColumnFamilyEntity(String nameSpace, String tableName, String owner, HColumnDescriptor hcdt, AtlasObjectId tableId, AtlasEntity atlasEntity) {
        AtlasEntity ret = null;
        ret = atlasEntity == null ? new AtlasEntity(HBaseDataTypes.HBASE_COLUMN_FAMILY.getName()) : atlasEntity;
        String cfName = hcdt.getNameAsString();
        String cfQualifiedName = HBaseBridge.getColumnFamilyQualifiedName(this.clusterName, nameSpace, tableName, cfName);
        ret.setAttribute("qualifiedName", (Object)cfQualifiedName);
        ret.setAttribute(CLUSTERNAME, (Object)this.clusterName);
        ret.setAttribute(TABLE, (Object)tableId);
        ret.setAttribute(NAME, (Object)cfName);
        ret.setAttribute(DESCRIPTION_ATTR, (Object)cfName);
        ret.setAttribute(OWNER, (Object)owner);
        ret.setAttribute(ATTR_CF_BLOCK_CACHE_ENABLED, (Object)hcdt.isBlockCacheEnabled());
        ret.setAttribute(ATTR_CF_BLOOMFILTER_TYPE, hcdt.getBloomFilterType() != null ? hcdt.getBloomFilterType().name() : null);
        ret.setAttribute(ATTR_CF_CACHED_BLOOM_ON_WRITE, (Object)hcdt.isCacheBloomsOnWrite());
        ret.setAttribute(ATTR_CF_CACHED_DATA_ON_WRITE, (Object)hcdt.isCacheDataOnWrite());
        ret.setAttribute(ATTR_CF_CACHED_INDEXES_ON_WRITE, (Object)hcdt.isCacheIndexesOnWrite());
        ret.setAttribute(ATTR_CF_COMPACTION_COMPRESSION_TYPE, hcdt.getCompactionCompressionType() != null ? hcdt.getCompactionCompressionType().name() : null);
        ret.setAttribute(ATTR_CF_COMPRESSION_TYPE, hcdt.getCompressionType() != null ? hcdt.getCompressionType().name() : null);
        ret.setAttribute(ATTR_CF_DATA_BLOCK_ENCODING, hcdt.getDataBlockEncoding() != null ? hcdt.getDataBlockEncoding().name() : null);
        ret.setAttribute(ATTR_CF_ENCRYPTION_TYPE, (Object)hcdt.getEncryptionType());
        ret.setAttribute(ATTR_CF_EVICT_BLOCK_ONCLOSE, (Object)hcdt.isEvictBlocksOnClose());
        ret.setAttribute(ATTR_CF_KEEP_DELETE_CELLS, hcdt.getKeepDeletedCells() != null ? hcdt.getKeepDeletedCells().name() : null);
        ret.setAttribute(ATTR_CF_MAX_VERSIONS, (Object)hcdt.getMaxVersions());
        ret.setAttribute(ATTR_CF_MIN_VERSIONS, (Object)hcdt.getMinVersions());
        ret.setAttribute(ATTR_CF_PREFETCH_BLOCK_ONOPEN, (Object)hcdt.isPrefetchBlocksOnOpen());
        ret.setAttribute(ATTR_CF_TTL, (Object)hcdt.getTimeToLive());
        return ret;
    }

    private AtlasEntity.AtlasEntityWithExtInfo createEntityInAtlas(AtlasEntity.AtlasEntityWithExtInfo entity) throws Exception {
        AtlasEntity.AtlasEntityWithExtInfo ret = null;
        EntityMutationResponse response = this.atlasClientV2.createEntity(entity);
        List entities = response.getCreatedEntities();
        if (CollectionUtils.isNotEmpty((Collection)entities)) {
            AtlasEntity.AtlasEntityWithExtInfo getByGuidResponse;
            ret = getByGuidResponse = this.atlasClientV2.getEntityByGuid(((AtlasEntityHeader)entities.get(0)).getGuid());
            LOG.info("Created {} entity: name={}, guid={}", new Object[]{ret.getEntity().getTypeName(), ret.getEntity().getAttribute("qualifiedName"), ret.getEntity().getGuid()});
        }
        return ret;
    }

    private AtlasEntity.AtlasEntityWithExtInfo updateEntityInAtlas(AtlasEntity.AtlasEntityWithExtInfo entity) throws Exception {
        AtlasEntity.AtlasEntityWithExtInfo ret = null;
        EntityMutationResponse response = this.atlasClientV2.updateEntity(entity);
        if (response != null) {
            List entities = response.getUpdatedEntities();
            if (CollectionUtils.isNotEmpty((Collection)entities)) {
                AtlasEntity.AtlasEntityWithExtInfo getByGuidResponse;
                ret = getByGuidResponse = this.atlasClientV2.getEntityByGuid(((AtlasEntityHeader)entities.get(0)).getGuid());
                LOG.info("Updated {} entity: name={}, guid={} ", new Object[]{ret.getEntity().getTypeName(), ret.getEntity().getAttribute("qualifiedName"), ret.getEntity().getGuid()});
            } else {
                LOG.info("Entity: name={} ", (Object)(entity.toString() + " not updated as it is unchanged from what is in Atlas"));
                ret = entity;
            }
        } else {
            LOG.info("Entity: name={} ", (Object)(entity.toString() + " not updated as it is unchanged from what is in Atlas"));
            ret = entity;
        }
        return ret;
    }

    private static String getColumnFamilyQualifiedName(String clusterName, String nameSpace, String tableName, String columnFamily) {
        tableName = HBaseBridge.stripNameSpace(tableName.toLowerCase());
        return String.format(HBASE_COLUMN_FAMILY_QUALIFIED_NAME_FORMAT, nameSpace.toLowerCase(), tableName, columnFamily.toLowerCase(), clusterName);
    }

    private static String getTableQualifiedName(String clusterName, String nameSpace, String tableName) {
        tableName = HBaseBridge.stripNameSpace(tableName.toLowerCase());
        return String.format(HBASE_TABLE_QUALIFIED_NAME_FORMAT, nameSpace.toLowerCase(), tableName, clusterName);
    }

    private static String getNameSpaceQualifiedName(String clusterName, String nameSpace) {
        return String.format(HBASE_NAMESPACE_QUALIFIED_NAME, nameSpace.toLowerCase(), clusterName);
    }

    private static String stripNameSpace(String tableName) {
        tableName = tableName.substring(tableName.indexOf(":") + 1);
        return tableName;
    }

    private static void printUsage() {
        System.out.println("Usage 1: import-hbase.sh [-n <namespace regex> OR --namespace <namespace regex >] [-t <table regex > OR --table <table regex>]");
        System.out.println("Usage 2: import-hbase.sh [-f <filename>]");
        System.out.println("   Format:");
        System.out.println("        namespace1:tbl1");
        System.out.println("        namespace1:tbl2");
        System.out.println("        namespace2:tbl1");
    }
}

