/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Random;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.datanode.BlockSender;
import org.apache.hadoop.hdfs.server.datanode.BlockTransferThrottler;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.FSDataset;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.StringUtils;

class DataBlockScanner
implements Runnable {
    public static final Log LOG = LogFactory.getLog(DataBlockScanner.class);
    private static final int MAX_SCAN_RATE = 0x800000;
    private static final int MIN_SCAN_RATE = 0x100000;
    static final long DEFAULT_SCAN_PERIOD_HOURS = 504L;
    private static final long ONE_DAY = 86400000L;
    static final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS");
    static final String verificationLogFile = "dncp_block_verification.log";
    static final int verficationLogLimit = 5;
    private long scanPeriod = 1814400000L;
    DataNode datanode;
    FSDataset dataset;
    TreeSet<BlockScanInfo> blockInfoSet;
    HashMap<Block, BlockScanInfo> blockMap;
    long totalScans = 0L;
    long totalVerifications = 0L;
    long totalScanErrors = 0L;
    long totalTransientErrors = 0L;
    long currentPeriodStart = System.currentTimeMillis();
    long bytesLeft = 0L;
    long totalBytesToScan = 0L;
    private LogFileHandler verificationLog;
    Random random = new Random();
    BlockTransferThrottler throttler = null;

    DataBlockScanner(DataNode datanode, FSDataset dataset, Configuration conf) {
        this.datanode = datanode;
        this.dataset = dataset;
        this.scanPeriod = conf.getInt("dfs.datanode.scan.period.hours", 0);
        if (this.scanPeriod <= 0L) {
            this.scanPeriod = 504L;
        }
        this.scanPeriod *= 3600000L;
    }

    private synchronized boolean isInitiliazed() {
        return this.throttler != null;
    }

    private void updateBytesToScan(long len, long lastScanTime) {
        this.totalBytesToScan += len;
        if (lastScanTime < this.currentPeriodStart) {
            this.bytesLeft += len;
        }
    }

    private synchronized void addBlockInfo(BlockScanInfo info) {
        boolean added = this.blockInfoSet.add(info);
        this.blockMap.put(info.block, info);
        if (added) {
            LogFileHandler log = this.verificationLog;
            if (log != null) {
                log.setMaxNumLines(this.blockMap.size() * 5);
            }
            this.updateBytesToScan(info.block.getNumBytes(), info.lastScanTime);
        }
    }

    private synchronized void delBlockInfo(BlockScanInfo info) {
        boolean exists = this.blockInfoSet.remove(info);
        this.blockMap.remove(info.block);
        if (exists) {
            LogFileHandler log = this.verificationLog;
            if (log != null) {
                log.setMaxNumLines(this.blockMap.size() * 5);
            }
            this.updateBytesToScan(-info.block.getNumBytes(), info.lastScanTime);
        }
    }

    private synchronized void updateBlockInfo(LogEntry e) {
        BlockScanInfo info = this.blockMap.get(new Block(e.blockId, 0L, e.genStamp));
        if (info != null && e.verificationTime > 0L && info.lastScanTime < e.verificationTime) {
            this.delBlockInfo(info);
            info.lastScanTime = e.verificationTime;
            info.lastScanType = ScanType.VERIFICATION_SCAN;
            this.addBlockInfo(info);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void init() {
        FSDataset.FSVolume[] volumes;
        Block[] arr = this.dataset.getBlockReport();
        Collections.shuffle(Arrays.asList(arr));
        this.blockInfoSet = new TreeSet();
        this.blockMap = new HashMap();
        long scanTime = -1L;
        for (Block block : arr) {
            BlockScanInfo info = new BlockScanInfo(block);
            --scanTime;
            info.lastScanTime = info.lastScanTime;
            this.addBlockInfo(info);
        }
        File dir = null;
        for (FSDataset.FSVolume vol : volumes = this.dataset.volumes.volumes) {
            if (!LogFileHandler.isFilePresent(vol.getDir(), verificationLogFile)) continue;
            dir = vol.getDir();
            break;
        }
        if (dir == null) {
            dir = volumes[0].getDir();
        }
        try {
            this.verificationLog = new LogFileHandler(dir, verificationLogFile, 100);
        }
        catch (IOException e) {
            LOG.warn((Object)"Could not open verfication log. Verification times are not stored.");
        }
        DataBlockScanner dataBlockScanner = this;
        synchronized (dataBlockScanner) {
            this.throttler = new BlockTransferThrottler(200L, 0x800000L);
        }
    }

    private synchronized long getNewBlockScanTime() {
        long period = Math.min(this.scanPeriod, (long)(Math.max(this.blockMap.size(), 1) * 600) * 1000L);
        return System.currentTimeMillis() - this.scanPeriod + (long)this.random.nextInt((int)period);
    }

    synchronized void addBlock(Block block) {
        if (!this.isInitiliazed()) {
            return;
        }
        BlockScanInfo info = this.blockMap.get(block);
        if (info != null) {
            LOG.warn((Object)("Adding an already existing block " + block));
            this.delBlockInfo(info);
        }
        info = new BlockScanInfo(block);
        info.lastScanTime = this.getNewBlockScanTime();
        this.addBlockInfo(info);
        this.adjustThrottler();
    }

    synchronized void deleteBlock(Block block) {
        if (!this.isInitiliazed()) {
            return;
        }
        BlockScanInfo info = this.blockMap.get(block);
        if (info != null) {
            this.delBlockInfo(info);
        }
    }

    synchronized long getLastScanTime(Block block) {
        if (!this.isInitiliazed()) {
            return 0L;
        }
        BlockScanInfo info = this.blockMap.get(block);
        return info == null ? 0L : info.lastScanTime;
    }

    void deleteBlocks(Block[] blocks) {
        for (Block b : blocks) {
            this.deleteBlock(b);
        }
    }

    void verifiedByClient(Block block) {
        this.updateScanStatus(block, ScanType.REMOTE_READ, true);
    }

    private synchronized void updateScanStatus(Block block, ScanType type, boolean scanOk) {
        BlockScanInfo info = this.blockMap.get(block);
        if (info != null) {
            this.delBlockInfo(info);
        } else {
            info = new BlockScanInfo(block);
        }
        long now = System.currentTimeMillis();
        info.lastScanType = type;
        info.lastScanTime = now;
        info.lastScanOk = scanOk;
        this.addBlockInfo(info);
        if (type == ScanType.REMOTE_READ) {
            ++this.totalVerifications;
        }
        long diff = now - info.lastLogTime;
        if (!scanOk || type == ScanType.REMOTE_READ && diff < this.scanPeriod / 3L && diff < 86400000L) {
            return;
        }
        info.lastLogTime = now;
        LogFileHandler log = this.verificationLog;
        if (log != null) {
            log.appendLine(LogEntry.newEnry(block, now));
        }
    }

    private void handleScanFailure(Block block) {
        LOG.info((Object)("Reporting bad block " + block + " to namenode."));
        try {
            DatanodeInfo[] dnArr = new DatanodeInfo[]{new DatanodeInfo(this.datanode.dnRegistration)};
            LocatedBlock[] blocks = new LocatedBlock[]{new LocatedBlock(block, dnArr)};
            this.datanode.namenode.reportBadBlocks(blocks);
        }
        catch (IOException e) {
            LOG.warn((Object)("Failed to report bad block " + block + " to namenode : " + " Exception : " + StringUtils.stringifyException(e)));
        }
    }

    private synchronized void adjustThrottler() {
        long timeLeft = this.currentPeriodStart + this.scanPeriod - System.currentTimeMillis();
        long bw = Math.max(this.bytesLeft * 1000L / timeLeft, 0x100000L);
        this.throttler.setBandwidth(Math.min(bw, 0x800000L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyBlock(Block block) {
        BlockSender blockSender = null;
        for (int i = 0; i < 2; ++i) {
            boolean second = i > 0;
            try {
                this.adjustThrottler();
                blockSender = new BlockSender(block, 0L, -1L, false, false, true, this.datanode);
                DataOutputStream out = new DataOutputStream(new IOUtils.NullOutputStream());
                blockSender.sendBlock(out, null, this.throttler);
                LOG.info((Object)((second ? "Second " : "") + "Verification succeeded for " + block));
                if (second) {
                    ++this.totalTransientErrors;
                }
                this.updateScanStatus(block, ScanType.VERIFICATION_SCAN, true);
            }
            catch (IOException e) {
                block8: {
                    block7: {
                        try {
                            ++this.totalScanErrors;
                            this.updateScanStatus(block, ScanType.VERIFICATION_SCAN, false);
                            if (this.dataset.getFile(block) != null) break block7;
                            LOG.info((Object)("Verification failed for " + block + ". Its ok since " + "it not in datanode dataset anymore."));
                            this.deleteBlock(block);
                        }
                        catch (Throwable throwable) {
                            IOUtils.closeStream(blockSender);
                            this.datanode.getMetrics().blocksVerified.inc();
                            ++this.totalScans;
                            ++this.totalVerifications;
                            throw throwable;
                        }
                        IOUtils.closeStream(blockSender);
                        this.datanode.getMetrics().blocksVerified.inc();
                        ++this.totalScans;
                        ++this.totalVerifications;
                        return;
                    }
                    LOG.warn((Object)((second ? "Second " : "First ") + "Verification failed for " + block + ". Exception : " + StringUtils.stringifyException(e)));
                    if (!second) break block8;
                    this.datanode.getMetrics().blockVerificationFailures.inc();
                    this.handleScanFailure(block);
                    IOUtils.closeStream(blockSender);
                    this.datanode.getMetrics().blocksVerified.inc();
                    ++this.totalScans;
                    ++this.totalVerifications;
                    return;
                }
                IOUtils.closeStream(blockSender);
                this.datanode.getMetrics().blocksVerified.inc();
                ++this.totalScans;
                ++this.totalVerifications;
                continue;
            }
            IOUtils.closeStream(blockSender);
            this.datanode.getMetrics().blocksVerified.inc();
            ++this.totalScans;
            ++this.totalVerifications;
            return;
        }
    }

    private synchronized long getEarliestScanTime() {
        if (this.blockInfoSet.size() > 0) {
            return this.blockInfoSet.first().lastScanTime;
        }
        return Long.MAX_VALUE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyFirstBlock() {
        Block block = null;
        DataBlockScanner dataBlockScanner = this;
        synchronized (dataBlockScanner) {
            if (this.blockInfoSet.size() > 0) {
                block = this.blockInfoSet.first().block;
            }
        }
        if (block != null) {
            this.verifyBlock(block);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean assignInitialVerificationTimes() {
        int numBlocks = 1;
        DataBlockScanner dataBlockScanner = this;
        synchronized (dataBlockScanner) {
            numBlocks = Math.max(this.blockMap.size(), 1);
        }
        LogFileHandler.Reader logReader = null;
        try {
            if (this.verificationLog != null) {
                LogFileHandler logFileHandler = this.verificationLog;
                logFileHandler.getClass();
                logReader = logFileHandler.new LogFileHandler.Reader(false);
            }
        }
        catch (IOException e) {
            LOG.warn((Object)("Could not read previous verification times : " + StringUtils.stringifyException(e)));
        }
        if (this.verificationLog != null) {
            this.verificationLog.updateCurNumLines();
        }
        try {
            while (logReader != null && logReader.hasNext()) {
                if (!this.datanode.shouldRun || Thread.interrupted()) {
                    boolean e = false;
                    return e;
                }
                LogEntry entry = LogEntry.parseEntry(logReader.next());
                if (entry == null) continue;
                this.updateBlockInfo(entry);
            }
        }
        finally {
            IOUtils.closeStream(logReader);
        }
        long verifyInterval = (long)Math.min((double)this.scanPeriod / 2.0 / (double)numBlocks, 600000.0);
        long lastScanTime = System.currentTimeMillis() - this.scanPeriod;
        DataBlockScanner dataBlockScanner2 = this;
        synchronized (dataBlockScanner2) {
            if (this.blockInfoSet.size() > 0) {
                while (true) {
                    BlockScanInfo info = this.blockInfoSet.first();
                    if (info.lastScanTime >= 0L) break;
                    this.delBlockInfo(info);
                    info.lastScanTime = lastScanTime;
                    lastScanTime += verifyInterval;
                    this.addBlockInfo(info);
                }
            }
        }
        return true;
    }

    private synchronized void startNewPeriod() {
        LOG.info((Object)("Starting a new period : work left in prev period : " + String.format("%.2f%%", (double)this.bytesLeft * 100.0 / (double)this.totalBytesToScan)));
        this.bytesLeft = this.totalBytesToScan;
        this.currentPeriodStart = System.currentTimeMillis();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            this.init();
            if (!this.assignInitialVerificationTimes()) {
                return;
            }
            this.adjustThrottler();
            while (this.datanode.shouldRun && !Thread.interrupted()) {
                long now = System.currentTimeMillis();
                DataBlockScanner dataBlockScanner = this;
                synchronized (dataBlockScanner) {
                    if (now >= this.currentPeriodStart + this.scanPeriod) {
                        this.startNewPeriod();
                    }
                }
                if (now - this.getEarliestScanTime() >= this.scanPeriod) {
                    this.verifyFirstBlock();
                    continue;
                }
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        catch (RuntimeException e) {
            LOG.warn((Object)("RuntimeException during DataBlockScanner.run() : " + StringUtils.stringifyException(e)));
            throw e;
        }
        finally {
            this.shutdown();
            LOG.info((Object)"Exiting DataBlockScanner thread.");
        }
    }

    synchronized void shutdown() {
        LogFileHandler log = this.verificationLog;
        this.verificationLog = null;
        if (log != null) {
            log.close();
        }
    }

    synchronized void printBlockReport(StringBuilder buffer, boolean summaryOnly) {
        long oneHour = 3600000L;
        long oneDay = 24L * oneHour;
        long oneWeek = 7L * oneDay;
        long fourWeeks = 4L * oneWeek;
        int inOneHour = 0;
        int inOneDay = 0;
        int inOneWeek = 0;
        int inFourWeeks = 0;
        int inScanPeriod = 0;
        int neverScanned = 0;
        int total = this.blockInfoSet.size();
        long now = System.currentTimeMillis();
        Date date = new Date();
        for (BlockScanInfo info : this.blockInfoSet) {
            long scanTime = info.getLastScanTime();
            long diff = now - scanTime;
            if (diff <= oneHour) {
                ++inOneHour;
            }
            if (diff <= oneDay) {
                ++inOneDay;
            }
            if (diff <= oneWeek) {
                ++inOneWeek;
            }
            if (diff <= fourWeeks) {
                ++inFourWeeks;
            }
            if (diff <= this.scanPeriod) {
                ++inScanPeriod;
            }
            if (scanTime <= 0L) {
                ++neverScanned;
            }
            if (summaryOnly) continue;
            date.setTime(scanTime);
            String scanType = info.lastScanType == ScanType.REMOTE_READ ? "remote" : (info.lastScanType == ScanType.VERIFICATION_SCAN ? "local" : "none");
            buffer.append(String.format("%-26s : status : %-6s type : %-6s scan time : %-15d %s\n", info.block, info.lastScanOk ? "ok" : "failed", scanType, scanTime, scanTime <= 0L ? "not yet verified" : dateFormat.format(date)));
        }
        double pctPeriodLeft = (double)(this.scanPeriod + this.currentPeriodStart - now) * 100.0 / (double)this.scanPeriod;
        double pctProgress = this.totalBytesToScan == 0L ? 100.0 : (double)(this.totalBytesToScan - this.bytesLeft) * 10000.0 / (double)this.totalBytesToScan / (100.0 - pctPeriodLeft + 1.0E-10);
        buffer.append(String.format("\nTotal Blocks                 : %6d\nVerified in last hour        : %6d\nVerified in last day         : %6d\nVerified in last week        : %6d\nVerified in last four weeks  : %6d\nVerified in SCAN_PERIOD      : %6d\nNot yet verified             : %6d\nVerified since restart       : %6d\nScans since restart          : %6d\nScan errors since restart    : %6d\nTransient scan errors        : %6d\nCurrent scan rate limit KBps : %6d\nProgress this period         : %6.0f%%\nTime left in cur period      : %6.2f%%\n", total, inOneHour, inOneDay, inOneWeek, inFourWeeks, inScanPeriod, neverScanned, this.totalVerifications, this.totalScans, this.totalScanErrors, this.totalTransientErrors, Math.round((double)this.throttler.getBandwidth() / 1024.0), pctProgress, pctPeriodLeft));
    }

    public static class Servlet
    extends HttpServlet {
        @Override
        public void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException {
            response.setContentType("text/plain");
            DataBlockScanner blockScanner = (DataBlockScanner)this.getServletContext().getAttribute("datanode.blockScanner");
            boolean summary = request.getParameter("listblocks") == null;
            StringBuilder buffer = new StringBuilder(8192);
            if (blockScanner == null) {
                buffer.append("Periodic block scanner is not running. Please check the datanode log if this is unexpected.");
            } else if (blockScanner.isInitiliazed()) {
                blockScanner.printBlockReport(buffer, summary);
            } else {
                buffer.append("Periodic block scanner is not yet initialized. Please check back again after some time.");
            }
            response.getWriter().write(buffer.toString());
        }
    }

    private static class LogFileHandler {
        private static final String curFileSuffix = ".curr";
        private static final String prevFileSuffix = ".prev";
        private static final long minRollingPeriod = 21600000L;
        private static final long minWarnPeriod = 21600000L;
        private static final int minLineLimit = 1000;
        private File curFile;
        private File prevFile;
        private int maxNumLines = -1;
        private int curNumLines = -1;
        long lastWarningTime = 0L;
        private PrintStream out;
        int numReaders = 0;

        static boolean isFilePresent(File dir, String filePrefix) {
            return new File(dir, filePrefix + curFileSuffix).exists() || new File(dir, filePrefix + prevFileSuffix).exists();
        }

        LogFileHandler(File dir, String filePrefix, int maxNumLines) throws IOException {
            this.curFile = new File(dir, filePrefix + curFileSuffix);
            this.prevFile = new File(dir, filePrefix + prevFileSuffix);
            this.openCurFile();
            this.curNumLines = -1;
            this.setMaxNumLines(maxNumLines);
        }

        synchronized void setMaxNumLines(int maxNumLines) {
            this.maxNumLines = Math.max(maxNumLines, 1000);
        }

        synchronized boolean appendLine(String line) {
            this.out.println();
            this.out.print(line);
            this.curNumLines += this.curNumLines < 0 ? -1 : 1;
            try {
                this.rollIfRequired();
            }
            catch (IOException e) {
                this.warn("Rolling failed for " + this.curFile + " : " + e.getMessage());
                return false;
            }
            return true;
        }

        private synchronized void warn(String msg) {
            long now = System.currentTimeMillis();
            if (now - this.lastWarningTime >= 21600000L) {
                this.lastWarningTime = now;
                LOG.warn((Object)msg);
            }
        }

        private synchronized void openCurFile() throws FileNotFoundException {
            this.close();
            this.out = new PrintStream(new FileOutputStream(this.curFile, true));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void updateCurNumLines() {
            int count = 0;
            Reader it = null;
            try {
                it = new Reader(true);
                while (it.hasNext()) {
                    it.next();
                    ++count;
                }
            }
            catch (IOException iOException) {
            }
            finally {
                LogFileHandler logFileHandler = this;
                synchronized (logFileHandler) {
                    this.curNumLines = count;
                }
                IOUtils.closeStream(it);
            }
        }

        private void rollIfRequired() throws IOException {
            if (this.curNumLines < this.maxNumLines || this.numReaders > 0) {
                return;
            }
            long now = System.currentTimeMillis();
            if (now < 21600000L) {
                return;
            }
            if (!this.prevFile.delete() && this.prevFile.exists()) {
                throw new IOException("Could not delete " + this.prevFile);
            }
            this.close();
            if (!this.curFile.renameTo(this.prevFile)) {
                this.openCurFile();
                throw new IOException("Could not rename " + this.curFile + " to " + this.prevFile);
            }
            this.openCurFile();
            this.updateCurNumLines();
        }

        synchronized void close() {
            if (this.out != null) {
                this.out.close();
                this.out = null;
            }
        }

        private class Reader
        implements Iterator<String>,
        Closeable {
            BufferedReader reader;
            File file;
            String line;
            boolean closed = false;

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private Reader(boolean skipPrevFile) throws IOException {
                LogFileHandler logFileHandler2 = LogFileHandler.this;
                synchronized (logFileHandler2) {
                    ++LogFileHandler.this.numReaders;
                }
                this.reader = null;
                this.file = skipPrevFile ? LogFileHandler.this.curFile : LogFileHandler.this.prevFile;
                this.readNext();
            }

            private boolean openFile() throws IOException {
                for (int i = 0; i < 2; ++i) {
                    if (this.reader != null || i > 0) {
                        File file = this.file = this.file == LogFileHandler.this.prevFile ? LogFileHandler.this.curFile : null;
                    }
                    if (this.file == null) {
                        return false;
                    }
                    if (this.file.exists()) break;
                }
                if (this.reader != null) {
                    this.reader.close();
                    this.reader = null;
                }
                this.reader = new BufferedReader(new FileReader(this.file));
                return true;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void readNext() throws IOException {
                this.line = null;
                try {
                    if (this.reader != null && (this.line = this.reader.readLine()) != null) {
                        return;
                    }
                    if (this.line == null && this.openFile()) {
                        this.readNext();
                    }
                }
                finally {
                    if (!this.hasNext()) {
                        this.close();
                    }
                }
            }

            @Override
            public boolean hasNext() {
                return this.line != null;
            }

            @Override
            public String next() {
                String curLine = this.line;
                try {
                    this.readNext();
                }
                catch (IOException e) {
                    LOG.info((Object)("Could not reade next line in LogHandler : " + StringUtils.stringifyException(e)));
                }
                return curLine;
            }

            @Override
            public void remove() {
                throw new RuntimeException("remove() is not supported.");
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void close() throws IOException {
                if (!this.closed) {
                    try {
                        if (this.reader != null) {
                            this.reader.close();
                        }
                    }
                    finally {
                        this.file = null;
                        this.reader = null;
                        this.closed = true;
                        LogFileHandler logFileHandler = LogFileHandler.this;
                        synchronized (logFileHandler) {
                            --LogFileHandler.this.numReaders;
                            assert (LogFileHandler.this.numReaders >= 0);
                        }
                    }
                }
            }
        }
    }

    private static class LogEntry {
        long blockId = -1L;
        long verificationTime = -1L;
        long genStamp = 0L;
        private static Pattern entryPattern = Pattern.compile("\\G\\s*([^=\\p{Space}]+)=\"(.*?)\"\\s*");

        private LogEntry() {
        }

        static String newEnry(Block block, long time) {
            return "date=\"" + dateFormat.format(new Date(time)) + "\"\t " + "time=\"" + time + "\"\t " + "genstamp=\"" + block.getGenerationStamp() + "\"\t " + "id=\"" + block.getBlockId() + "\"";
        }

        static LogEntry parseEntry(String line) {
            LogEntry entry = new LogEntry();
            Matcher matcher = entryPattern.matcher(line);
            while (matcher.find()) {
                String name = matcher.group(1);
                String value = matcher.group(2);
                try {
                    if (name.equals("id")) {
                        entry.blockId = Long.valueOf(value);
                        continue;
                    }
                    if (name.equals("time")) {
                        entry.verificationTime = Long.valueOf(value);
                        continue;
                    }
                    if (!name.equals("genstamp")) continue;
                    entry.genStamp = Long.valueOf(value);
                }
                catch (NumberFormatException nfe) {
                    LOG.warn((Object)("Cannot parse line: " + line), (Throwable)nfe);
                    return null;
                }
            }
            return entry;
        }
    }

    static class BlockScanInfo
    implements Comparable<BlockScanInfo> {
        Block block;
        long lastScanTime = 0L;
        long lastLogTime = 0L;
        ScanType lastScanType = ScanType.NONE;
        boolean lastScanOk = true;

        BlockScanInfo(Block block) {
            this.block = block;
        }

        public int hashCode() {
            return this.block.hashCode();
        }

        public boolean equals(Object other) {
            return other instanceof BlockScanInfo && this.compareTo((BlockScanInfo)other) == 0;
        }

        long getLastScanTime() {
            return this.lastScanType == ScanType.NONE ? 0L : this.lastScanTime;
        }

        @Override
        public int compareTo(BlockScanInfo other) {
            long t1 = this.lastScanTime;
            long t2 = other.lastScanTime;
            return t1 < t2 ? -1 : (t1 > t2 ? 1 : this.block.compareTo(other.block));
        }
    }

    private static enum ScanType {
        REMOTE_READ,
        VERIFICATION_SCAN,
        NONE;

    }
}

