/*
 * Decompiled with CFR 0.152.
 */
package org.apache.avro.hadoop.file;

import java.io.Closeable;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import org.apache.avro.Schema;
import org.apache.avro.file.DataFileReader;
import org.apache.avro.file.DataFileWriter;
import org.apache.avro.generic.GenericData;
import org.apache.avro.generic.GenericDatumReader;
import org.apache.avro.generic.GenericDatumWriter;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.hadoop.io.AvroKeyValue;
import org.apache.avro.hadoop.util.AvroCharSequenceComparator;
import org.apache.avro.mapred.FsInput;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SortedKeyValueFile {
    private static final Logger LOG = LoggerFactory.getLogger(SortedKeyValueFile.class);
    public static final String DATA_FILENAME = "data";
    public static final String INDEX_FILENAME = "index";

    public static class Writer<K, V>
    implements Closeable {
        private final Schema mKeySchema;
        private final Schema mValueSchema;
        private final Schema mRecordSchema;
        private final Schema mIndexSchema;
        private final DataFileWriter<GenericRecord> mDataFileWriter;
        private final DataFileWriter<GenericRecord> mIndexFileWriter;
        private final int mIndexInterval;
        private long mRecordsWritten;
        private K mPreviousKey;

        public Writer(Options options) throws IOException {
            if (null == options.getConfiguration()) {
                throw new IllegalArgumentException("Configuration may not be null");
            }
            FileSystem fileSystem = options.getPath().getFileSystem(options.getConfiguration());
            this.mKeySchema = options.getKeySchema();
            if (null == this.mKeySchema) {
                throw new IllegalArgumentException("Key schema may not be null");
            }
            this.mValueSchema = options.getValueSchema();
            if (null == this.mValueSchema) {
                throw new IllegalArgumentException("Value schema may not be null");
            }
            this.mIndexInterval = options.getIndexInterval();
            if (!fileSystem.mkdirs(options.getPath())) {
                throw new IOException("Unable to create directory for SortedKeyValueFile: " + options.getPath());
            }
            LOG.debug("Created directory " + options.getPath());
            Path dataFilePath = new Path(options.getPath(), SortedKeyValueFile.DATA_FILENAME);
            LOG.debug("Creating writer for avro data file: " + dataFilePath);
            this.mRecordSchema = AvroKeyValue.getSchema(this.mKeySchema, this.mValueSchema);
            GenericDatumWriter datumWriter = new GenericDatumWriter(this.mRecordSchema);
            FSDataOutputStream dataOutputStream = fileSystem.create(dataFilePath);
            this.mDataFileWriter = new DataFileWriter(datumWriter).setSyncInterval(0x100000).create(this.mRecordSchema, dataOutputStream);
            Path indexFilePath = new Path(options.getPath(), SortedKeyValueFile.INDEX_FILENAME);
            LOG.debug("Creating writer for avro index file: " + indexFilePath);
            this.mIndexSchema = AvroKeyValue.getSchema(this.mKeySchema, Schema.create(Schema.Type.LONG));
            GenericDatumWriter indexWriter = new GenericDatumWriter(this.mIndexSchema);
            FSDataOutputStream indexOutputStream = fileSystem.create(indexFilePath);
            this.mIndexFileWriter = new DataFileWriter(indexWriter).create(this.mIndexSchema, indexOutputStream);
        }

        public void append(K key, V value) throws IOException {
            if (null != this.mPreviousKey && GenericData.get().compare(key, this.mPreviousKey, this.mKeySchema) < 0) {
                throw new IllegalArgumentException("Records must be inserted in sorted key order. Attempted to insert key " + key + " after " + this.mPreviousKey + ".");
            }
            this.mPreviousKey = GenericData.get().deepCopy(this.mKeySchema, key);
            AvroKeyValue<K, V> dataRecord = new AvroKeyValue<K, V>(new GenericData.Record(this.mRecordSchema));
            dataRecord.setKey(key);
            dataRecord.setValue(value);
            if (0L == this.mRecordsWritten++ % (long)this.mIndexInterval) {
                long position = this.mDataFileWriter.sync();
                AvroKeyValue<K, Long> indexRecord = new AvroKeyValue<K, Long>(new GenericData.Record(this.mIndexSchema));
                indexRecord.setKey(key);
                indexRecord.setValue(position);
                this.mIndexFileWriter.append(indexRecord.get());
            }
            this.mDataFileWriter.append(dataRecord.get());
        }

        @Override
        public void close() throws IOException {
            this.mIndexFileWriter.close();
            this.mDataFileWriter.close();
        }

        public static class Options {
            private Schema mKeySchema;
            private Schema mValueSchema;
            private Configuration mConf;
            private Path mPath;
            private int mIndexInterval = 128;

            public Options withKeySchema(Schema keySchema) {
                this.mKeySchema = keySchema;
                return this;
            }

            public Schema getKeySchema() {
                return this.mKeySchema;
            }

            public Options withValueSchema(Schema valueSchema) {
                this.mValueSchema = valueSchema;
                return this;
            }

            public Schema getValueSchema() {
                return this.mValueSchema;
            }

            public Options withConfiguration(Configuration conf) {
                this.mConf = conf;
                return this;
            }

            public Configuration getConfiguration() {
                return this.mConf;
            }

            public Options withPath(Path path) {
                this.mPath = path;
                return this;
            }

            public Path getPath() {
                return this.mPath;
            }

            public Options withIndexInterval(int indexInterval) {
                this.mIndexInterval = indexInterval;
                return this;
            }

            public int getIndexInterval() {
                return this.mIndexInterval;
            }
        }
    }

    public static class Reader<K, V>
    implements Closeable,
    Iterable<AvroKeyValue<K, V>> {
        private final NavigableMap<K, Long> mIndex;
        private final DataFileReader<GenericRecord> mDataFileReader;
        private final Schema mKeySchema;

        public Reader(Options options) throws IOException {
            this.mKeySchema = options.getKeySchema();
            Path indexFilePath = new Path(options.getPath(), SortedKeyValueFile.INDEX_FILENAME);
            LOG.debug("Loading the index from " + indexFilePath);
            this.mIndex = Reader.loadIndexFile(options.getConfiguration(), indexFilePath, this.mKeySchema);
            Path dataFilePath = new Path(options.getPath(), SortedKeyValueFile.DATA_FILENAME);
            LOG.debug("Loading the data file " + dataFilePath);
            Schema recordSchema = AvroKeyValue.getSchema(this.mKeySchema, options.getValueSchema());
            GenericDatumReader datumReader = new GenericDatumReader(recordSchema);
            this.mDataFileReader = new DataFileReader(new FsInput(dataFilePath, options.getConfiguration()), datumReader);
        }

        public V get(K key) throws IOException {
            LOG.debug("Looking up key " + key + " in the index.");
            Map.Entry<K, Long> indexEntry = this.mIndex.floorEntry(key);
            if (null == indexEntry) {
                LOG.debug("Key " + key + " was not found in the index (it is before the first entry)");
                return null;
            }
            LOG.debug("Key was found in the index, seeking to syncpoint " + indexEntry.getValue());
            this.mDataFileReader.seek(indexEntry.getValue());
            for (AvroKeyValue<K, V> record : this) {
                int comparison = GenericData.get().compare(record.getKey(), key, this.mKeySchema);
                if (0 == comparison) {
                    LOG.debug("Found record for key " + key);
                    return record.getValue();
                }
                if (comparison <= 0) continue;
                LOG.debug("Searched beyond the point where key " + key + " would appear in the file.");
                return null;
            }
            LOG.debug("Searched to the end of the file but did not find key " + key);
            return null;
        }

        @Override
        public Iterator<AvroKeyValue<K, V>> iterator() {
            return new AvroKeyValue.Iterator(this.mDataFileReader.iterator());
        }

        @Override
        public void close() throws IOException {
            this.mDataFileReader.close();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private static <K> NavigableMap<K, Long> loadIndexFile(Configuration conf, Path path, Schema keySchema) throws IOException {
            GenericDatumReader datumReader = new GenericDatumReader(AvroKeyValue.getSchema(keySchema, Schema.create(Schema.Type.LONG)));
            DataFileReader fileReader = new DataFileReader(new FsInput(path, conf), datumReader);
            TreeMap index = Schema.create(Schema.Type.STRING).equals(keySchema) ? new TreeMap(new AvroCharSequenceComparator()) : new TreeMap();
            try {
                for (GenericRecord genericRecord : fileReader) {
                    AvroKeyValue indexRecord = new AvroKeyValue(genericRecord);
                    index.put(indexRecord.getKey(), indexRecord.getValue());
                }
            }
            finally {
                fileReader.close();
            }
            return index;
        }

        public static class Options {
            private Configuration mConf;
            private Path mPath;
            private Schema mKeySchema;
            private Schema mValueSchema;

            public Options withConfiguration(Configuration conf) {
                this.mConf = conf;
                return this;
            }

            public Configuration getConfiguration() {
                return this.mConf;
            }

            public Options withPath(Path path) {
                this.mPath = path;
                return this;
            }

            public Path getPath() {
                return this.mPath;
            }

            public Options withKeySchema(Schema keySchema) {
                this.mKeySchema = keySchema;
                return this;
            }

            public Schema getKeySchema() {
                return this.mKeySchema;
            }

            public Options withValueSchema(Schema valueSchema) {
                this.mValueSchema = valueSchema;
                return this;
            }

            public Schema getValueSchema() {
                return this.mValueSchema;
            }
        }
    }
}

