/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.StorageObject;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.StorageStubProvider;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryDeterminer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.flogger.GoogleLogger;
import com.google.common.hash.Hashing;
import com.google.google.storage.v1.GetObjectMediaRequest;
import com.google.google.storage.v1.GetObjectMediaResponse;
import com.google.google.storage.v1.StorageGrpc;
import com.google.protobuf.ByteString;
import io.grpc.Context;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.util.Iterator;
import java.util.OptionalLong;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;

public class GoogleCloudStorageGrpcReadChannel
implements SeekableByteChannel {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    protected static final String METADATA_FIELDS = "contentEncoding,generation,size";
    private volatile StorageGrpc.StorageBlockingStub stub;
    private final StorageStubProvider stubProvider;
    private final StorageResourceId resourceId;
    private final long objectGeneration;
    private final long objectSize;
    private boolean channelIsOpen = true;
    private long positionInGrpcStream = 0L;
    private long bytesToSkipBeforeReading = 0L;
    @Nullable
    private ByteString bufferedContent = null;
    private int bufferedContentReadOffset = 0;
    @Nullable
    private Iterator<GetObjectMediaResponse> resIterator = null;
    private final GoogleCloudStorageReadOptions readOptions;
    private final GoogleCloudStorageImpl.BackOffFactory backOffFactory;
    @Nullable
    Context.CancellableContext requestContext;
    GoogleCloudStorageReadOptions.Fadvise readStrategy;
    @Nullable
    private final ByteString footerContent;
    private final long footerStartOffsetInBytes;
    private long contentChannelEndOffset = -1L;

    public static GoogleCloudStorageGrpcReadChannel open(StorageStubProvider stubProvider, Storage storage, ApiErrorExtractor errorExtractor, StorageResourceId resourceId, GoogleCloudStorageReadOptions readOptions) throws IOException {
        return GoogleCloudStorageGrpcReadChannel.open(stubProvider, storage, errorExtractor, resourceId, readOptions, GoogleCloudStorageImpl.BackOffFactory.DEFAULT);
    }

    @VisibleForTesting
    static GoogleCloudStorageGrpcReadChannel open(StorageStubProvider stubProvider, Storage storage, ApiErrorExtractor errorExtractor, StorageResourceId resourceId, GoogleCloudStorageReadOptions readOptions, GoogleCloudStorageImpl.BackOffFactory backOffFactory) throws IOException {
        try {
            return ResilientOperation.retry(() -> GoogleCloudStorageGrpcReadChannel.openChannel(stubProvider, storage, errorExtractor, resourceId, readOptions, backOffFactory), backOffFactory.newBackOff(), RetryDeterminer.ALL_ERRORS, IOException.class);
        }
        catch (Exception e) {
            throw new IOException(String.format("Error reading '%s'", resourceId), e);
        }
    }

    private static GoogleCloudStorageGrpcReadChannel openChannel(StorageStubProvider stubProvider, Storage storage, ApiErrorExtractor errorExtractor, StorageResourceId resourceId, GoogleCloudStorageReadOptions readOptions, GoogleCloudStorageImpl.BackOffFactory backOffFactory) throws IOException {
        StorageGrpc.StorageBlockingStub stub = stubProvider.newBlockingStub();
        Preconditions.checkArgument(storage != null, "GCS json client cannot be null");
        GoogleCloudStorageItemInfo itemInfo = GoogleCloudStorageGrpcReadChannel.getObjectMetadata(resourceId, errorExtractor, backOffFactory, storage);
        Preconditions.checkArgument(itemInfo != null, "object metadata cannot be null");
        String contentEncoding = itemInfo.getContentEncoding();
        if (contentEncoding != null && contentEncoding.contains("gzip")) {
            throw new IOException("Cannot read GZIP encoded files - content encoding support is disabled.");
        }
        int prefetchSizeInBytes = readOptions.getMinRangeRequestSize() / 2;
        long footerOffsetInBytes = Math.max(0L, itemInfo.getSize() - (long)prefetchSizeInBytes);
        ByteString footerContent = GoogleCloudStorageGrpcReadChannel.getFooterContent(resourceId, readOptions, stub, footerOffsetInBytes);
        return new GoogleCloudStorageGrpcReadChannel(stub, stubProvider, resourceId, itemInfo.getContentGeneration(), itemInfo.getSize(), footerOffsetInBytes, footerContent, readOptions, backOffFactory);
    }

    private static GoogleCloudStorageItemInfo getObjectMetadata(StorageResourceId resourceId, ApiErrorExtractor errorExtractor, GoogleCloudStorageImpl.BackOffFactory backOffFactory, Storage gcs) throws IOException {
        StorageObject object;
        try {
            Storage.Objects.Get metadataRequest = GoogleCloudStorageGrpcReadChannel.getMetadataRequest(gcs, resourceId).setFields(METADATA_FIELDS);
            object = ResilientOperation.retry(metadataRequest::execute, backOffFactory.newBackOff(), RetryDeterminer.SOCKET_ERRORS, IOException.class, Sleeper.DEFAULT);
        }
        catch (IOException e) {
            throw errorExtractor.itemNotFound(e) ? GoogleCloudStorageExceptions.createFileNotFoundException(resourceId, e) : new IOException("Error reading " + resourceId, e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Thread interrupt received.", e);
        }
        return GoogleCloudStorageItemInfo.createObject(resourceId, 0L, 0L, Preconditions.checkNotNull(object.getSize(), "size can not be null for '%s'", (Object)resourceId).longValue(), null, object.getContentEncoding(), null, Preconditions.checkNotNull(object.getGeneration(), "generation can not be null for '%s'", (Object)resourceId), 0L, null);
    }

    private static Storage.Objects.Get getMetadataRequest(Storage gcs, StorageResourceId resourceId) throws IOException {
        Storage.Objects.Get getObject = gcs.objects().get(resourceId.getBucketName(), resourceId.getObjectName());
        if (resourceId.hasGenerationId()) {
            getObject.setGeneration(resourceId.getGenerationId());
        }
        return getObject;
    }

    private static ByteString getFooterContent(StorageResourceId resourceId, GoogleCloudStorageReadOptions readOptions, StorageGrpc.StorageBlockingStub stub, long footerOffset) throws IOException {
        try {
            Iterator<GetObjectMediaResponse> footerContentResponse = ((StorageGrpc.StorageBlockingStub)stub.withDeadlineAfter(readOptions.getGrpcReadTimeoutMillis(), TimeUnit.MILLISECONDS)).getObjectMedia(GetObjectMediaRequest.newBuilder().setReadOffset(footerOffset).setBucket(resourceId.getBucketName()).setObject(resourceId.getObjectName()).build());
            ByteString footerContent = null;
            while (footerContentResponse.hasNext()) {
                GetObjectMediaResponse objectMediaResponse = footerContentResponse.next();
                if (!objectMediaResponse.hasChecksummedData()) continue;
                ByteString content = objectMediaResponse.getChecksummedData().getContent();
                if (footerContent == null) {
                    footerContent = content;
                    continue;
                }
                footerContent = footerContent.concat(content);
            }
            if (footerContent == null) {
                ((GoogleLogger.Api)logger.atFiner()).log("Prefetched footer content is null for resource '%s'", resourceId);
            } else {
                ((GoogleLogger.Api)logger.atFiner()).log("Prefetched %s bytes footer for '%s'", footerContent.size(), (Object)resourceId);
            }
            return footerContent;
        }
        catch (StatusRuntimeException e) {
            throw GoogleCloudStorageGrpcReadChannel.convertError(e, resourceId);
        }
    }

    private GoogleCloudStorageGrpcReadChannel(StorageGrpc.StorageBlockingStub gcsGrpcBlockingStub, StorageStubProvider stubProvider, StorageResourceId resourceId, long objectGeneration, long objectSize, long footerStartOffsetInBytes, ByteString footerContent, GoogleCloudStorageReadOptions readOptions, GoogleCloudStorageImpl.BackOffFactory backOffFactory) {
        this.stub = gcsGrpcBlockingStub;
        this.stubProvider = stubProvider;
        this.resourceId = resourceId;
        this.objectGeneration = objectGeneration;
        this.objectSize = objectSize;
        this.readOptions = readOptions;
        this.backOffFactory = backOffFactory;
        this.readStrategy = readOptions.getFadvise();
        this.footerStartOffsetInBytes = footerStartOffsetInBytes;
        this.footerContent = footerContent;
    }

    private static IOException convertError(StatusRuntimeException error, StorageResourceId resourceId) {
        String msg = String.format("Error reading '%s'", resourceId);
        switch (Status.fromThrowable(error).getCode()) {
            case NOT_FOUND: {
                return GoogleCloudStorageExceptions.createFileNotFoundException(resourceId.getBucketName(), resourceId.getObjectName(), new IOException(msg, error));
            }
            case OUT_OF_RANGE: {
                return (IOException)new EOFException(msg).initCause(error);
            }
        }
        return new IOException(msg, error);
    }

    private static void put(ByteString source, int offset, int size, ByteBuffer dest) {
        ByteString croppedSource = source.substring(offset, offset + size);
        for (ByteBuffer sourcePiece : croppedSource.asReadOnlyByteBufferList()) {
            dest.put(sourcePiece);
        }
    }

    private int readBufferedContentInto(ByteBuffer byteBuffer) {
        long bufferSkip = Math.min((long)(this.bufferedContent.size() - this.bufferedContentReadOffset), this.bytesToSkipBeforeReading);
        bufferSkip = Math.max(0L, bufferSkip);
        this.bufferedContentReadOffset = (int)((long)this.bufferedContentReadOffset + bufferSkip);
        this.bytesToSkipBeforeReading -= bufferSkip;
        this.positionInGrpcStream += bufferSkip;
        int remainingBufferedBytes = this.bufferedContent.size() - this.bufferedContentReadOffset;
        boolean remainingBufferedContentLargerThanByteBuffer = remainingBufferedBytes > byteBuffer.remaining();
        int bytesToWrite = remainingBufferedContentLargerThanByteBuffer ? byteBuffer.remaining() : remainingBufferedBytes;
        GoogleCloudStorageGrpcReadChannel.put(this.bufferedContent, this.bufferedContentReadOffset, bytesToWrite, byteBuffer);
        this.positionInGrpcStream += (long)bytesToWrite;
        if (remainingBufferedContentLargerThanByteBuffer) {
            this.bufferedContentReadOffset += bytesToWrite;
        } else {
            this.bufferedContent = null;
            this.bufferedContentReadOffset = 0;
        }
        return bytesToWrite;
    }

    @Override
    public int read(ByteBuffer byteBuffer) throws IOException {
        ((GoogleLogger.Api)logger.atFiner()).log("GCS gRPC read request for up to %d bytes at offset %d from object '%s'", byteBuffer.remaining(), this.position(), this.resourceId);
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        int bytesRead = 0;
        if (this.resIterator != null && this.isByteBufferBeyondCurrentRequestRange(byteBuffer)) {
            this.positionInGrpcStream += this.bytesToSkipBeforeReading;
            this.cancelCurrentRequest();
            this.bufferedContent = null;
            this.bufferedContentReadOffset = 0;
            this.bytesToSkipBeforeReading = 0L;
        }
        if (this.bufferedContent != null) {
            bytesRead += this.readBufferedContentInto(byteBuffer);
        }
        if (!byteBuffer.hasRemaining()) {
            return bytesRead;
        }
        if (this.positionInGrpcStream == this.objectSize) {
            return bytesRead > 0 ? bytesRead : -1;
        }
        long effectivePosition = this.positionInGrpcStream + this.bytesToSkipBeforeReading;
        if (this.footerContent != null && effectivePosition >= this.footerStartOffsetInBytes) {
            ((GoogleLogger.Api)logger.atFiner()).log("Read request responded with footer content at position '%s'", effectivePosition);
            return bytesRead += this.readFooterContentIntoBuffer(byteBuffer);
        }
        if (this.resIterator == null) {
            OptionalLong bytesToRead = this.getBytesToRead(byteBuffer);
            this.positionInGrpcStream += this.bytesToSkipBeforeReading;
            this.bytesToSkipBeforeReading = 0L;
            this.requestObjectMedia(bytesToRead);
            if (bytesToRead.isPresent()) {
                this.contentChannelEndOffset = this.positionInGrpcStream + bytesToRead.getAsLong();
            }
        }
        bytesRead += this.readObjectContentFromGCS(byteBuffer);
        if (this.hasMoreFooterContentToRead(byteBuffer)) {
            int bytesToWrite = Math.min(byteBuffer.remaining(), this.footerContent.size());
            int bytesToSkipInFooter = (int)(this.positionInGrpcStream - this.footerStartOffsetInBytes);
            GoogleCloudStorageGrpcReadChannel.put(this.footerContent, bytesToSkipInFooter, bytesToWrite, byteBuffer);
            this.positionInGrpcStream += (long)bytesToWrite;
            bytesRead += bytesToWrite;
        }
        return bytesRead;
    }

    private boolean isByteBufferBeyondCurrentRequestRange(ByteBuffer byteBuffer) {
        long effectivePosition = this.positionInGrpcStream + this.bytesToSkipBeforeReading;
        if (this.contentChannelEndOffset == -1L) {
            return false;
        }
        return effectivePosition + (long)byteBuffer.remaining() > this.contentChannelEndOffset;
    }

    private int readObjectContentFromGCS(ByteBuffer byteBuffer) throws IOException {
        int bytesRead = 0;
        while (this.moreServerContent() && byteBuffer.hasRemaining()) {
            GetObjectMediaResponse res = this.resIterator.next();
            ByteString content = res.getChecksummedData().getContent();
            if (this.bytesToSkipBeforeReading >= 0L && this.bytesToSkipBeforeReading < (long)content.size()) {
                content = res.getChecksummedData().getContent().substring((int)this.bytesToSkipBeforeReading);
                this.positionInGrpcStream += this.bytesToSkipBeforeReading;
                this.bytesToSkipBeforeReading = 0L;
            } else if (this.bytesToSkipBeforeReading >= (long)content.size()) {
                this.positionInGrpcStream += (long)content.size();
                this.bytesToSkipBeforeReading -= (long)content.size();
                continue;
            }
            if (this.readOptions.isGrpcChecksumsEnabled() && res.getChecksummedData().hasCrc32C()) {
                this.validateChecksum(res);
            }
            boolean responseSizeLargerThanRemainingBuffer = content.size() > byteBuffer.remaining();
            int bytesToWrite = responseSizeLargerThanRemainingBuffer ? byteBuffer.remaining() : content.size();
            GoogleCloudStorageGrpcReadChannel.put(content, 0, bytesToWrite, byteBuffer);
            bytesRead += bytesToWrite;
            this.positionInGrpcStream += (long)bytesToWrite;
            if (!responseSizeLargerThanRemainingBuffer) continue;
            this.bufferedContent = content;
            this.bufferedContentReadOffset = bytesToWrite;
        }
        return bytesRead;
    }

    private void validateChecksum(GetObjectMediaResponse res) throws IOException {
        int expectedChecksum;
        int calculatedChecksum = Hashing.crc32c().hashBytes(res.getChecksummedData().getContent().toByteArray()).asInt();
        if (calculatedChecksum != (expectedChecksum = res.getChecksummedData().getCrc32C().getValue())) {
            throw new IOException(String.format("Message checksum (%s) didn't match expected checksum (%s) for '%s'", expectedChecksum, calculatedChecksum, this.resourceId));
        }
    }

    private boolean hasMoreFooterContentToRead(ByteBuffer byteBuffer) {
        return this.footerContent != null && this.positionInGrpcStream >= this.footerStartOffsetInBytes && byteBuffer.hasRemaining();
    }

    private OptionalLong getBytesToRead(ByteBuffer byteBuffer) {
        OptionalLong optionalBytesToRead = OptionalLong.empty();
        if (this.readStrategy == GoogleCloudStorageReadOptions.Fadvise.RANDOM) {
            long rangeRequestSize = Math.max(this.readOptions.getInplaceSeekLimit(), (long)this.readOptions.getMinRangeRequestSize());
            optionalBytesToRead = OptionalLong.of(Math.max((long)byteBuffer.remaining(), rangeRequestSize));
        }
        if (this.footerContent == null) {
            return optionalBytesToRead;
        }
        long bytesToFooterOffset = this.footerStartOffsetInBytes - this.positionInGrpcStream;
        if (optionalBytesToRead.isPresent()) {
            return OptionalLong.of(Math.min(optionalBytesToRead.getAsLong(), bytesToFooterOffset));
        }
        return OptionalLong.of(bytesToFooterOffset);
    }

    private int readFooterContentIntoBuffer(ByteBuffer byteBuffer) {
        this.positionInGrpcStream += this.bytesToSkipBeforeReading;
        this.bytesToSkipBeforeReading = 0L;
        long bytesToSkipFromFooter = this.positionInGrpcStream - this.footerStartOffsetInBytes;
        long bytesToWriteFromFooter = (long)this.footerContent.size() - bytesToSkipFromFooter;
        int bytesToWrite = Math.toIntExact(Math.min((long)byteBuffer.remaining(), bytesToWriteFromFooter));
        GoogleCloudStorageGrpcReadChannel.put(this.footerContent, Math.toIntExact(bytesToSkipFromFooter), bytesToWrite, byteBuffer);
        this.positionInGrpcStream += (long)bytesToWrite;
        return bytesToWrite;
    }

    private void requestObjectMedia(OptionalLong bytesToRead) throws IOException {
        GetObjectMediaRequest.Builder requestBuilder = GetObjectMediaRequest.newBuilder().setBucket(this.resourceId.getBucketName()).setObject(this.resourceId.getObjectName()).setGeneration(this.objectGeneration).setReadOffset(this.positionInGrpcStream);
        bytesToRead.ifPresent(requestBuilder::setReadLimit);
        GetObjectMediaRequest request = requestBuilder.build();
        try {
            ResilientOperation.retry(() -> {
                try {
                    this.requestContext = Context.current().withCancellation();
                    Context toReattach = this.requestContext.attach();
                    try {
                        this.resIterator = ((StorageGrpc.StorageBlockingStub)this.stub.withDeadlineAfter(this.readOptions.getGrpcReadTimeoutMillis(), TimeUnit.MILLISECONDS)).getObjectMedia(request);
                    }
                    finally {
                        this.requestContext.detach(toReattach);
                    }
                }
                catch (StatusRuntimeException e) {
                    this.recreateStub(e);
                    throw GoogleCloudStorageGrpcReadChannel.convertError(e, this.resourceId);
                }
                return null;
            }, this.backOffFactory.newBackOff(), RetryDeterminer.ALL_ERRORS, IOException.class);
        }
        catch (Exception e) {
            throw new IOException(String.format("Error reading '%s'", this.resourceId), e);
        }
    }

    private void cancelCurrentRequest() {
        if (this.requestContext != null) {
            this.requestContext.close();
            this.requestContext = null;
        }
        if (this.resIterator != null) {
            this.resIterator = null;
        }
        this.contentChannelEndOffset = -1L;
    }

    private boolean moreServerContent() throws IOException {
        if (this.resIterator == null || this.requestContext == null || this.requestContext.isCancelled()) {
            return false;
        }
        try {
            return ResilientOperation.retry(() -> {
                try {
                    boolean moreDataAvailable = this.resIterator.hasNext();
                    if (!moreDataAvailable) {
                        this.cancelCurrentRequest();
                    }
                    return moreDataAvailable;
                }
                catch (StatusRuntimeException e) {
                    this.recreateStub(e);
                    throw GoogleCloudStorageGrpcReadChannel.convertError(e, this.resourceId);
                }
            }, this.backOffFactory.newBackOff(), RetryDeterminer.ALL_ERRORS, IOException.class);
        }
        catch (Exception e) {
            this.cancelCurrentRequest();
            throw new IOException(String.format("Error reading '%s'", this.resourceId), e);
        }
    }

    private void recreateStub(StatusRuntimeException e) {
        if (StorageStubProvider.isStubBroken(Status.fromThrowable(e).getCode())) {
            this.stub = this.stubProvider.newBlockingStub();
        }
    }

    @Override
    public int write(ByteBuffer byteBuffer) {
        throw new UnsupportedOperationException("Cannot mutate read-only channel: " + this);
    }

    @Override
    public long position() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        return this.positionInGrpcStream + this.bytesToSkipBeforeReading;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        Preconditions.checkArgument(newPosition >= 0L, "Read position must be non-negative, but was %s", newPosition);
        Preconditions.checkArgument(newPosition < this.size(), "Read position must be before end of file (%s), but was %s", this.size(), newPosition);
        if (newPosition == this.positionInGrpcStream) {
            return this;
        }
        long seekDistance = newPosition - this.positionInGrpcStream;
        if (seekDistance >= 0L && seekDistance <= this.readOptions.getInplaceSeekLimit()) {
            this.bytesToSkipBeforeReading = seekDistance;
            return this;
        }
        if (this.readStrategy == GoogleCloudStorageReadOptions.Fadvise.AUTO && (seekDistance < 0L || seekDistance > this.readOptions.getInplaceSeekLimit())) {
            this.readStrategy = GoogleCloudStorageReadOptions.Fadvise.RANDOM;
        }
        this.cancelCurrentRequest();
        this.bufferedContent = null;
        this.bufferedContentReadOffset = 0;
        this.bytesToSkipBeforeReading = 0L;
        this.positionInGrpcStream = newPosition;
        return this;
    }

    @Override
    public long size() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        return this.objectSize;
    }

    @Override
    public SeekableByteChannel truncate(long l) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.channelIsOpen;
    }

    @Override
    public void close() {
        this.cancelCurrentRequest();
        this.channelIsOpen = false;
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("resourceId", this.resourceId).add("generation", this.objectGeneration).toString();
    }
}

