/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio.cooplock;

import com.google.api.client.util.ExponentialBackOff;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.cooplock.CoopLockOperationType;
import com.google.cloud.hadoop.gcsio.cooplock.CoopLockRecord;
import com.google.cloud.hadoop.gcsio.cooplock.CoopLockRecords;
import com.google.cloud.hadoop.gcsio.cooplock.CooperativeLockingOptions;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.flogger.GoogleLogger;
import com.google.common.flogger.LazyArgs;
import com.google.common.util.concurrent.Uninterruptibles;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonPrimitive;
import java.io.IOException;
import java.lang.reflect.Type;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

public class CoopLockRecordsDao {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    public static final String LOCK_DIRECTORY = "_lock/";
    private static final String LOCK_FILE = "all.lock";
    public static final String LOCK_PATH = "_lock/all.lock";
    private static final String LOCK_METADATA_KEY = "lock";
    private static final int MIN_BACK_OFF_INTERVAL_MILLIS = 500;
    private static final int MAX_BACK_OFF_INTERVAL_MILLIS = 2000;
    private static final Duration RETRY_LOCK_INTERVAL = Duration.ofSeconds(2L);
    private static final Gson GSON = CoopLockRecordsDao.createGson();
    private static final CreateObjectOptions CREATE_NEW_OBJECT_OPTIONS = CreateObjectOptions.DEFAULT_NO_OVERWRITE;
    private final GoogleCloudStorageImpl gcs;
    private final CooperativeLockingOptions options;

    public CoopLockRecordsDao(GoogleCloudStorageImpl gcs) {
        this.gcs = gcs;
        this.options = gcs.getOptions().getCooperativeLockingOptions();
    }

    public Set<CoopLockRecord> getLockedOperations(String bucketName) throws IOException {
        long startMs = System.currentTimeMillis();
        StorageResourceId lockId = CoopLockRecordsDao.getLockId(bucketName);
        GoogleCloudStorageItemInfo lockInfo = this.gcs.getItemInfo(lockId);
        HashSet<CoopLockRecord> operations = !lockInfo.exists() || lockInfo.getMetaGeneration() == 0L || lockInfo.getMetadata().get(LOCK_METADATA_KEY) == null ? new HashSet<CoopLockRecord>() : CoopLockRecordsDao.getLockRecords(lockInfo).getLocks();
        ((GoogleLogger.Api)logger.atFine()).log("[%dms] getLockedOperations(%s): %s", System.currentTimeMillis() - startMs, bucketName, operations);
        return operations;
    }

    public void relockOperation(String bucketName, CoopLockRecord operationRecord) throws IOException {
        long startMs = System.currentTimeMillis();
        String operationId = operationRecord.getOperationId();
        String clientId = operationRecord.getClientId();
        this.modifyLock(records -> this.reacquireOperationLock((CoopLockRecords)records, operationRecord), bucketName, operationId);
        ((GoogleLogger.Api)logger.atFine()).log("[%dms] lockOperation(%s, %s)", System.currentTimeMillis() - startMs, operationId, clientId);
    }

    public void lockPaths(String operationId, Instant operationInstant, CoopLockOperationType operationType, StorageResourceId ... resources) throws IOException {
        long startMs = System.currentTimeMillis();
        Set<String> objects = CoopLockRecordsDao.validateResources(resources);
        String bucketName = resources[0].getBucketName();
        this.modifyLock(records -> this.addLockRecords((CoopLockRecords)records, operationId, operationInstant, operationType, objects), bucketName, operationId);
        ((GoogleLogger.Api)logger.atFine()).log("[%dms] lockPaths(%s, %s)", System.currentTimeMillis() - startMs, operationId, LazyArgs.lazy(() -> Arrays.toString(resources)));
    }

    public void unlockPaths(String operationId, StorageResourceId ... resources) throws IOException {
        long startMs = System.currentTimeMillis();
        Set<String> objects = CoopLockRecordsDao.validateResources(resources);
        String bucketName = resources[0].getBucketName();
        this.modifyLock(records -> CoopLockRecordsDao.removeLockRecords(records, operationId, objects), bucketName, operationId);
        ((GoogleLogger.Api)logger.atFine()).log("[%dms] unlockPaths(%s, %s)", System.currentTimeMillis() - startMs, operationId, LazyArgs.lazy(() -> Arrays.toString(resources)));
    }

    private static Set<String> validateResources(StorageResourceId[] resources) {
        Preconditions.checkNotNull(resources, "resources should not be null");
        Preconditions.checkArgument(resources.length > 0, "resources should not be empty");
        String bucketName = resources[0].getBucketName();
        Preconditions.checkState(Arrays.stream(resources).allMatch(r -> r.getBucketName().equals(bucketName)), "All resources should be in the same bucket");
        return Arrays.stream(resources).map(StorageResourceId::getObjectName).collect(ImmutableSet.toImmutableSet());
    }

    private void modifyLock(Function<CoopLockRecords, Boolean> modificationFn, String bucketName, String operationId) throws IOException {
        long startMs = System.currentTimeMillis();
        StorageResourceId lockId = CoopLockRecordsDao.getLockId(bucketName);
        ExponentialBackOff backOff = new ExponentialBackOff.Builder().setInitialIntervalMillis(500).setMultiplier(1.2).setMaxIntervalMillis(2000).setMaxElapsedTimeMillis(Integer.MAX_VALUE).build();
        block2: while (true) {
            try {
                CoopLockRecords lockRecords;
                GoogleCloudStorageItemInfo lockInfo;
                while (true) {
                    if (!(lockInfo = this.gcs.getItemInfo(lockId)).exists()) {
                        this.gcs.createEmptyObject(lockId, CREATE_NEW_OBJECT_OPTIONS);
                        lockInfo = this.gcs.getItemInfo(lockId);
                    }
                    CoopLockRecords coopLockRecords = lockRecords = lockInfo.getMetaGeneration() == 0L || lockInfo.getMetadata().get(LOCK_METADATA_KEY) == null ? new CoopLockRecords().setFormatVersion(3L) : CoopLockRecordsDao.getLockRecords(lockInfo);
                    if (!modificationFn.apply(lockRecords).booleanValue()) {
                        ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).atMostEvery(5, TimeUnit.SECONDS)).log("Failed to update %s entries in %s file: resources could be locked, retrying.", lockRecords.getLocks().size(), (Object)lockId);
                        Uninterruptibles.sleepUninterruptibly(RETRY_LOCK_INTERVAL);
                        continue;
                    }
                    if (lockRecords.getLocks().isEmpty()) {
                        this.gcs.deleteObject(lockInfo.getResourceId(), lockInfo.getMetaGeneration());
                        break block2;
                    }
                    if (lockRecords.getLocks().size() <= this.options.getMaxConcurrentOperations()) break;
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).atMostEvery(5, TimeUnit.SECONDS)).log("Skipping lock entries update in %s file: too many (%d) locked resources, retrying.", (Object)lockId, lockRecords.getLocks().size());
                    Uninterruptibles.sleepUninterruptibly(RETRY_LOCK_INTERVAL);
                }
                String lockContent = GSON.toJson((Object)lockRecords, (Type)((Object)CoopLockRecords.class));
                HashMap<String, byte[]> metadata = new HashMap<String, byte[]>(lockInfo.getMetadata());
                metadata.put(LOCK_METADATA_KEY, lockContent.getBytes(StandardCharsets.UTF_8));
                this.gcs.updateMetadata(lockInfo, metadata);
                ((GoogleLogger.Api)logger.atFine()).log("Updated lock file in %dms for %s operation", System.currentTimeMillis() - startMs, (Object)operationId);
            }
            catch (IOException e) {
                if (ApiErrorExtractor.INSTANCE.preconditionNotMet(e)) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).atMostEvery(5, TimeUnit.SECONDS)).log("Failed to update entries (condition not met) in %s file for operation %s, retrying.", (Object)lockId, (Object)operationId);
                } else if (ApiErrorExtractor.INSTANCE.itemNotFound(e)) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).atMostEvery(5, TimeUnit.SECONDS)).log("Failed to update entries (file not found) in %s file for operation %s, retrying.", (Object)lockId, (Object)operationId);
                } else {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause(e)).log("Failed to modify lock for %s operation, retrying.", operationId);
                }
                Uninterruptibles.sleepUninterruptibly(Duration.ofMillis(backOff.nextBackOffMillis()));
                continue;
            }
            break;
        }
    }

    private static StorageResourceId getLockId(String bucketName) {
        return new StorageResourceId(bucketName, LOCK_PATH);
    }

    private static CoopLockRecords getLockRecords(GoogleCloudStorageItemInfo lockInfo) {
        String lockContent = new String(lockInfo.getMetadata().get(LOCK_METADATA_KEY), StandardCharsets.UTF_8);
        CoopLockRecords lockRecords = GSON.fromJson(lockContent, CoopLockRecords.class);
        Preconditions.checkState(lockRecords.getFormatVersion() == 3L, "Unsupported metadata format: expected %s, but was %s", lockRecords.getFormatVersion(), 3L);
        return lockRecords;
    }

    private boolean reacquireOperationLock(CoopLockRecords lockRecords, CoopLockRecord operationRecord) {
        Optional<CoopLockRecord> operationOptional = lockRecords.getLocks().stream().filter(o -> o.equals(operationRecord)).findAny();
        Preconditions.checkState(operationOptional.isPresent(), "operation %s not found", (Object)operationRecord.getOperationId());
        operationOptional.get().setClientId(CoopLockRecordsDao.newClientId(operationRecord.getOperationId())).setLockExpiration(Instant.now().plusMillis(this.options.getLockExpirationTimeoutMilli()));
        return true;
    }

    private boolean addLockRecords(CoopLockRecords lockRecords, String operationId, Instant operationInstant, CoopLockOperationType operationType, Set<String> resourcesToAdd) {
        boolean atLestOneResourceAlreadyLocked = lockRecords.getLocks().stream().flatMap(operation -> operation.getResources().stream()).anyMatch(lockedResource -> {
            for (String resourceToAdd : resourcesToAdd) {
                if (!resourceToAdd.equals(lockedResource) && !CoopLockRecordsDao.isChildObject(lockedResource, resourceToAdd) && !CoopLockRecordsDao.isChildObject(resourceToAdd, lockedResource)) continue;
                return true;
            }
            return false;
        });
        if (atLestOneResourceAlreadyLocked) {
            return false;
        }
        CoopLockRecord record = new CoopLockRecord().setClientId(CoopLockRecordsDao.newClientId(operationId)).setOperationId(operationId).setOperationTime(operationInstant).setLockExpiration(Instant.now().plusMillis(this.options.getLockExpirationTimeoutMilli())).setOperationType(operationType).setResources(resourcesToAdd);
        lockRecords.getLocks().add(record);
        return true;
    }

    private static boolean isChildObject(String parent, String child) {
        return parent.startsWith(child.endsWith("/") ? child : child + "/");
    }

    private static boolean removeLockRecords(CoopLockRecords lockRecords, String operationId, Set<String> resourcesToRemove) {
        List recordsToRemove = lockRecords.getLocks().stream().filter(o -> o.getResources().stream().anyMatch(resourcesToRemove::contains)).collect(Collectors.toList());
        Preconditions.checkState(recordsToRemove.size() == 1, "Only %s operation with %s resources should be unlocked, but found %s operations:\n%s", (Object)operationId, resourcesToRemove, (Object)recordsToRemove.size(), recordsToRemove);
        CoopLockRecord operationToRemove = (CoopLockRecord)recordsToRemove.get(0);
        Preconditions.checkState(operationToRemove.getOperationId().equals(operationId), "All resources should be locked by %s operation, but they are locked by %s operation", (Object)operationId, (Object)operationToRemove.getOperationId());
        Preconditions.checkState(operationToRemove.getResources().equals(resourcesToRemove), "All of %s resources should be locked by operation, but was locked only %s resources", resourcesToRemove, operationToRemove.getResources());
        Preconditions.checkState(lockRecords.getLocks().remove(operationToRemove), "operation %s was not removed", (Object)operationToRemove);
        return true;
    }

    private static String newClientId(String operationId) {
        InetAddress localHost;
        try {
            localHost = InetAddress.getLocalHost();
        }
        catch (UnknownHostException e) {
            throw new RuntimeException(String.format("Failed to get clientId for %s operation", operationId), e);
        }
        String epochMillis = String.valueOf(Instant.now().toEpochMilli());
        return localHost.getCanonicalHostName() + "-" + epochMillis.substring(epochMillis.length() - 6);
    }

    public static Gson createGson() {
        return new GsonBuilder().registerTypeAdapter((Type)((Object)Instant.class), (instant, type, context) -> new JsonPrimitive(instant.toString())).registerTypeAdapter((Type)((Object)Instant.class), (json, type, context) -> Instant.parse(json.getAsJsonPrimitive().getAsString())).create();
    }
}

