/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.aws.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.CompleteMultipartUploadResult;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.CopyObjectResult;
import com.amazonaws.services.s3.model.CopyPartRequest;
import com.amazonaws.services.s3.model.CopyPartResult;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadResult;
import com.amazonaws.services.s3.model.ListObjectsV2Request;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.auto.value.AutoValue;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.annotations.VisibleForTesting;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.base.Preconditions;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.base.Strings;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.collect.ArrayListMultimap;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.collect.ImmutableList;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.collect.ImmutableSet;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.collect.Iterables;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.util.concurrent.ListeningExecutorService;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.util.concurrent.MoreExecutors;
import org.apache.beam.repackaged.beam_sdks_java_io_amazon_web_services.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.beam.sdk.io.FileSystem;
import org.apache.beam.sdk.io.aws.options.S3Options;
import org.apache.beam.sdk.io.aws.s3.AutoValue_S3FileSystem_ExpandedGlob;
import org.apache.beam.sdk.io.aws.s3.AutoValue_S3FileSystem_PathWithEncoding;
import org.apache.beam.sdk.io.aws.s3.S3ReadableSeekableByteChannel;
import org.apache.beam.sdk.io.aws.s3.S3ResourceId;
import org.apache.beam.sdk.io.aws.s3.S3WritableByteChannel;
import org.apache.beam.sdk.io.fs.CreateOptions;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.io.fs.ResourceId;
import org.apache.beam.sdk.util.MoreFutures;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class S3FileSystem
extends FileSystem<S3ResourceId> {
    private static final Logger LOG = LoggerFactory.getLogger(S3FileSystem.class);
    private static final long MAX_COPY_OBJECT_SIZE_BYTES = 0x140000000L;
    private static final int MAX_DELETE_OBJECTS_PER_REQUEST = 1000;
    private static final ImmutableSet<String> NON_READ_SEEK_EFFICIENT_ENCODINGS = ImmutableSet.of("gzip");
    private AmazonS3 amazonS3;
    private final S3Options options;
    private final ListeningExecutorService executorService;

    S3FileSystem(S3Options options) {
        this.options = Preconditions.checkNotNull(options, "options");
        if (Strings.isNullOrEmpty(options.getAwsRegion())) {
            LOG.info("The AWS S3 Beam extension was included in this build, but the awsRegion flag was not specified. If you don't plan to use S3, then ignore this message.");
        }
        this.amazonS3 = S3FileSystem.buildAmazonS3Client(options);
        Preconditions.checkNotNull(options.getS3StorageClass(), "storageClass");
        Preconditions.checkArgument(options.getS3ThreadPoolSize() > 0, "threadPoolSize");
        this.executorService = MoreExecutors.listeningDecorator(Executors.newFixedThreadPool(options.getS3ThreadPoolSize(), new ThreadFactoryBuilder().setDaemon(true).build()));
    }

    private static AmazonS3 buildAmazonS3Client(S3Options options) {
        AmazonS3ClientBuilder builder = (AmazonS3ClientBuilder)AmazonS3ClientBuilder.standard().withCredentials(options.getAwsCredentialsProvider());
        builder = Strings.isNullOrEmpty(options.getAwsServiceEndpoint()) ? (AmazonS3ClientBuilder)builder.withRegion(options.getAwsRegion()) : (AmazonS3ClientBuilder)builder.withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(options.getAwsServiceEndpoint(), options.getAwsRegion()));
        return (AmazonS3)builder.build();
    }

    protected String getScheme() {
        return "s3";
    }

    @VisibleForTesting
    void setAmazonS3Client(AmazonS3 amazonS3) {
        this.amazonS3 = amazonS3;
    }

    @VisibleForTesting
    AmazonS3 getAmazonS3Client() {
        return this.amazonS3;
    }

    protected List<MatchResult> match(List<String> specs) throws IOException {
        List paths = specs.stream().map(S3ResourceId::fromUri).collect(Collectors.toList());
        ArrayList<S3ResourceId> globs = new ArrayList<S3ResourceId>();
        ArrayList<S3ResourceId> nonGlobs = new ArrayList<S3ResourceId>();
        ArrayList<Boolean> isGlobBooleans = new ArrayList<Boolean>();
        for (S3ResourceId path : paths) {
            if (path.isWildcard()) {
                globs.add(path);
                isGlobBooleans.add(true);
                continue;
            }
            nonGlobs.add(path);
            isGlobBooleans.add(false);
        }
        Iterator<MatchResult> globMatches = this.matchGlobPaths(globs).iterator();
        Iterator<MatchResult> nonGlobMatches = this.matchNonGlobPaths(nonGlobs).iterator();
        ImmutableList.Builder matchResults = ImmutableList.builder();
        for (Boolean isGlob : isGlobBooleans) {
            if (isGlob.booleanValue()) {
                Preconditions.checkState(globMatches.hasNext(), "Expect globMatches has next.");
                matchResults.add(globMatches.next());
                continue;
            }
            Preconditions.checkState(nonGlobMatches.hasNext(), "Expect nonGlobMatches has next.");
            matchResults.add(nonGlobMatches.next());
        }
        Preconditions.checkState(!globMatches.hasNext(), "Expect no more elements in globMatches.");
        Preconditions.checkState(!nonGlobMatches.hasNext(), "Expect no more elements in nonGlobMatches.");
        return matchResults.build();
    }

    @VisibleForTesting
    List<MatchResult> matchGlobPaths(Collection<S3ResourceId> globPaths) throws IOException {
        ArrayList expandTasks = new ArrayList(globPaths.size());
        for (S3ResourceId path : globPaths) {
            expandTasks.add(() -> this.expandGlob(path));
        }
        HashMap<S3ResourceId, ExpandedGlob> expandedGlobByGlobPath = new HashMap<S3ResourceId, ExpandedGlob>();
        ArrayList contentTypeTasks = new ArrayList(globPaths.size());
        for (Object expandedGlob : this.callTasks(expandTasks)) {
            expandedGlobByGlobPath.put(((ExpandedGlob)expandedGlob).getGlobPath(), (ExpandedGlob)expandedGlob);
            if (((ExpandedGlob)expandedGlob).getExpandedPaths() == null) continue;
            for (S3ResourceId path : ((ExpandedGlob)expandedGlob).getExpandedPaths()) {
                contentTypeTasks.add(() -> this.getPathContentEncoding(path));
            }
        }
        HashMap<S3ResourceId, PathWithEncoding> exceptionByPath = new HashMap<S3ResourceId, PathWithEncoding>();
        for (PathWithEncoding pathWithException : this.callTasks(contentTypeTasks)) {
            exceptionByPath.put(pathWithException.getPath(), pathWithException);
        }
        ArrayList<MatchResult> results = new ArrayList<MatchResult>(globPaths.size());
        for (S3ResourceId globPath : globPaths) {
            ExpandedGlob expandedGlob = (ExpandedGlob)expandedGlobByGlobPath.get(globPath);
            if (expandedGlob.getException() != null) {
                results.add(MatchResult.create((MatchResult.Status)MatchResult.Status.ERROR, (IOException)expandedGlob.getException()));
                continue;
            }
            ArrayList<MatchResult.Metadata> metadatas = new ArrayList<MatchResult.Metadata>();
            IOException exception = null;
            for (S3ResourceId expandedPath : expandedGlob.getExpandedPaths()) {
                PathWithEncoding pathWithEncoding = (PathWithEncoding)exceptionByPath.get(expandedPath);
                if (pathWithEncoding.getException() != null) {
                    exception = pathWithEncoding.getException();
                    break;
                }
                metadatas.add(S3FileSystem.createBeamMetadata(pathWithEncoding.getPath(), pathWithEncoding.getContentEncoding()));
            }
            if (exception != null) {
                if (exception instanceof FileNotFoundException) {
                    results.add(MatchResult.create((MatchResult.Status)MatchResult.Status.NOT_FOUND, exception));
                    continue;
                }
                results.add(MatchResult.create((MatchResult.Status)MatchResult.Status.ERROR, exception));
                continue;
            }
            results.add(MatchResult.create((MatchResult.Status)MatchResult.Status.OK, metadatas));
        }
        return ImmutableList.copyOf(results);
    }

    private ExpandedGlob expandGlob(S3ResourceId glob) {
        Preconditions.checkArgument(glob.isWildcard(), "isWildcard");
        String keyPrefix = glob.getKeyNonWildcardPrefix();
        Pattern wildcardRegexp = Pattern.compile(S3FileSystem.wildcardToRegexp(glob.getKey()));
        LOG.debug("expanding bucket {}, prefix {}, against pattern {}", new Object[]{glob.getBucket(), keyPrefix, wildcardRegexp.toString()});
        ImmutableList.Builder expandedPaths = ImmutableList.builder();
        String continuationToken = null;
        do {
            ListObjectsV2Result result;
            ListObjectsV2Request request = new ListObjectsV2Request().withBucketName(glob.getBucket()).withPrefix(keyPrefix).withContinuationToken(continuationToken);
            try {
                result = this.amazonS3.listObjectsV2(request);
            }
            catch (AmazonClientException e) {
                return ExpandedGlob.create(glob, new IOException(e));
            }
            continuationToken = result.getNextContinuationToken();
            for (S3ObjectSummary objectSummary : result.getObjectSummaries()) {
                if (!wildcardRegexp.matcher(objectSummary.getKey()).matches()) continue;
                S3ResourceId expandedPath = S3ResourceId.fromComponents(objectSummary.getBucketName(), objectSummary.getKey()).withSize(objectSummary.getSize());
                LOG.debug("Expanded S3 object path {}", (Object)expandedPath);
                expandedPaths.add(expandedPath);
            }
        } while (continuationToken != null);
        return ExpandedGlob.create(glob, (List<S3ResourceId>)((Object)expandedPaths.build()));
    }

    private PathWithEncoding getPathContentEncoding(S3ResourceId path) {
        ObjectMetadata s3Metadata;
        try {
            s3Metadata = this.getObjectMetadata(path);
        }
        catch (AmazonClientException e) {
            if (e instanceof AmazonS3Exception && ((AmazonS3Exception)e).getStatusCode() == 404) {
                return PathWithEncoding.create(path, new FileNotFoundException());
            }
            return PathWithEncoding.create(path, new IOException(e));
        }
        return PathWithEncoding.create(path, Strings.nullToEmpty(s3Metadata.getContentEncoding()));
    }

    private List<MatchResult> matchNonGlobPaths(Collection<S3ResourceId> paths) throws IOException {
        ArrayList tasks = new ArrayList(paths.size());
        for (S3ResourceId path : paths) {
            tasks.add(() -> this.matchNonGlobPath(path));
        }
        return this.callTasks(tasks);
    }

    private ObjectMetadata getObjectMetadata(S3ResourceId s3ResourceId) throws AmazonClientException {
        GetObjectMetadataRequest request = new GetObjectMetadataRequest(s3ResourceId.getBucket(), s3ResourceId.getKey());
        request.setSSECustomerKey(this.options.getSSECustomerKey());
        return this.amazonS3.getObjectMetadata(request);
    }

    @VisibleForTesting
    MatchResult matchNonGlobPath(S3ResourceId path) {
        ObjectMetadata s3Metadata;
        try {
            s3Metadata = this.getObjectMetadata(path);
        }
        catch (AmazonClientException e) {
            if (e instanceof AmazonS3Exception && ((AmazonS3Exception)e).getStatusCode() == 404) {
                return MatchResult.create((MatchResult.Status)MatchResult.Status.NOT_FOUND, (IOException)new FileNotFoundException());
            }
            return MatchResult.create((MatchResult.Status)MatchResult.Status.ERROR, (IOException)new IOException(e));
        }
        return MatchResult.create((MatchResult.Status)MatchResult.Status.OK, ImmutableList.of(S3FileSystem.createBeamMetadata(path.withSize(s3Metadata.getContentLength()), Strings.nullToEmpty(s3Metadata.getContentEncoding()))));
    }

    private static MatchResult.Metadata createBeamMetadata(S3ResourceId path, String contentEncoding) {
        Preconditions.checkArgument(path.getSize().isPresent(), "path has size");
        Preconditions.checkNotNull(contentEncoding, "contentEncoding");
        boolean isReadSeekEfficient = !NON_READ_SEEK_EFFICIENT_ENCODINGS.contains(contentEncoding);
        return MatchResult.Metadata.builder().setIsReadSeekEfficient(isReadSeekEfficient).setResourceId((ResourceId)path).setSizeBytes(path.getSize().get().longValue()).build();
    }

    @VisibleForTesting
    static String wildcardToRegexp(String globExp) {
        StringBuilder dst = new StringBuilder();
        char[] src = globExp.replace("**/*", "**").toCharArray();
        int i = 0;
        block6: while (i < src.length) {
            char c = src[i++];
            switch (c) {
                case '*': {
                    if (i < src.length && src[i] == '*') {
                        dst.append(".*");
                        ++i;
                        continue block6;
                    }
                    dst.append("[^/]*");
                    continue block6;
                }
                case '?': {
                    dst.append("[^/]");
                    continue block6;
                }
                case '$': 
                case '(': 
                case ')': 
                case '+': 
                case '.': 
                case '^': 
                case '{': 
                case '|': 
                case '}': {
                    dst.append('\\').append(c);
                    continue block6;
                }
                case '\\': {
                    i = S3FileSystem.doubleSlashes(dst, src, i);
                    continue block6;
                }
            }
            dst.append(c);
        }
        return dst.toString();
    }

    private static int doubleSlashes(StringBuilder dst, char[] src, int i) {
        dst.append("\\\\");
        if (i - 1 != src.length) {
            dst.append(src[i]);
            ++i;
        } else {
            dst.append('\\');
        }
        return i;
    }

    protected WritableByteChannel create(S3ResourceId resourceId, CreateOptions createOptions) throws IOException {
        return new S3WritableByteChannel(this.amazonS3, resourceId, createOptions.mimeType(), this.options);
    }

    protected ReadableByteChannel open(S3ResourceId resourceId) throws IOException {
        return new S3ReadableSeekableByteChannel(this.amazonS3, resourceId, this.options);
    }

    protected void copy(List<S3ResourceId> sourcePaths, List<S3ResourceId> destinationPaths) throws IOException {
        Preconditions.checkArgument(sourcePaths.size() == destinationPaths.size(), "sizes of sourcePaths and destinationPaths do not match");
        ArrayList tasks = new ArrayList(sourcePaths.size());
        Iterator<S3ResourceId> sourcePathsIterator = sourcePaths.iterator();
        Iterator<S3ResourceId> destinationPathsIterator = destinationPaths.iterator();
        while (sourcePathsIterator.hasNext()) {
            S3ResourceId sourcePath = sourcePathsIterator.next();
            S3ResourceId destinationPath = destinationPathsIterator.next();
            tasks.add(() -> {
                this.copy(sourcePath, destinationPath);
                return null;
            });
        }
        this.callTasks(tasks);
    }

    @VisibleForTesting
    void copy(S3ResourceId sourcePath, S3ResourceId destinationPath) throws IOException {
        try {
            ObjectMetadata sourceObjectMetadata = this.getObjectMetadata(sourcePath);
            if (sourceObjectMetadata.getContentLength() < 0x140000000L) {
                this.atomicCopy(sourcePath, destinationPath, sourceObjectMetadata);
            } else {
                this.multipartCopy(sourcePath, destinationPath, sourceObjectMetadata);
            }
        }
        catch (AmazonClientException e) {
            throw new IOException(e);
        }
    }

    @VisibleForTesting
    CopyObjectResult atomicCopy(S3ResourceId sourcePath, S3ResourceId destinationPath, ObjectMetadata sourceObjectMetadata) throws AmazonClientException {
        CopyObjectRequest copyObjectRequest = new CopyObjectRequest(sourcePath.getBucket(), sourcePath.getKey(), destinationPath.getBucket(), destinationPath.getKey());
        copyObjectRequest.setNewObjectMetadata(sourceObjectMetadata);
        copyObjectRequest.setStorageClass(this.options.getS3StorageClass());
        copyObjectRequest.setSourceSSECustomerKey(this.options.getSSECustomerKey());
        copyObjectRequest.setDestinationSSECustomerKey(this.options.getSSECustomerKey());
        return this.amazonS3.copyObject(copyObjectRequest);
    }

    @VisibleForTesting
    CompleteMultipartUploadResult multipartCopy(S3ResourceId sourcePath, S3ResourceId destinationPath, ObjectMetadata sourceObjectMetadata) throws AmazonClientException {
        InitiateMultipartUploadRequest initiateUploadRequest = new InitiateMultipartUploadRequest(destinationPath.getBucket(), destinationPath.getKey()).withStorageClass(this.options.getS3StorageClass()).withObjectMetadata(sourceObjectMetadata);
        initiateUploadRequest.setSSECustomerKey(this.options.getSSECustomerKey());
        InitiateMultipartUploadResult initiateUploadResult = this.amazonS3.initiateMultipartUpload(initiateUploadRequest);
        String uploadId = initiateUploadResult.getUploadId();
        ArrayList<PartETag> eTags = new ArrayList<PartETag>();
        long objectSize = sourceObjectMetadata.getContentLength();
        if (objectSize == 0L) {
            CopyPartRequest copyPartRequest = new CopyPartRequest().withSourceBucketName(sourcePath.getBucket()).withSourceKey(sourcePath.getKey()).withDestinationBucketName(destinationPath.getBucket()).withDestinationKey(destinationPath.getKey()).withUploadId(uploadId).withPartNumber(1);
            copyPartRequest.setSourceSSECustomerKey(this.options.getSSECustomerKey());
            copyPartRequest.setDestinationSSECustomerKey(this.options.getSSECustomerKey());
            CopyPartResult copyPartResult = this.amazonS3.copyPart(copyPartRequest);
            eTags.add(copyPartResult.getPartETag());
        } else {
            long bytePosition = 0L;
            Integer uploadBufferSizeBytes = this.options.getS3UploadBufferSizeBytes();
            int partNumber = 1;
            while (bytePosition < objectSize) {
                CopyPartRequest copyPartRequest = new CopyPartRequest().withSourceBucketName(sourcePath.getBucket()).withSourceKey(sourcePath.getKey()).withDestinationBucketName(destinationPath.getBucket()).withDestinationKey(destinationPath.getKey()).withUploadId(uploadId).withPartNumber(partNumber).withFirstByte(Long.valueOf(bytePosition)).withLastByte(Long.valueOf(Math.min(objectSize - 1L, bytePosition + (long)uploadBufferSizeBytes.intValue() - 1L)));
                copyPartRequest.setSourceSSECustomerKey(this.options.getSSECustomerKey());
                copyPartRequest.setDestinationSSECustomerKey(this.options.getSSECustomerKey());
                CopyPartResult copyPartResult = this.amazonS3.copyPart(copyPartRequest);
                eTags.add(copyPartResult.getPartETag());
                bytePosition += (long)uploadBufferSizeBytes.intValue();
                ++partNumber;
            }
        }
        CompleteMultipartUploadRequest completeUploadRequest = new CompleteMultipartUploadRequest().withBucketName(destinationPath.getBucket()).withKey(destinationPath.getKey()).withUploadId(uploadId).withPartETags(eTags);
        return this.amazonS3.completeMultipartUpload(completeUploadRequest);
    }

    protected void rename(List<S3ResourceId> sourceResourceIds, List<S3ResourceId> destinationResourceIds) throws IOException {
        this.copy(sourceResourceIds, destinationResourceIds);
        this.delete(sourceResourceIds);
    }

    protected void delete(Collection<S3ResourceId> resourceIds) throws IOException {
        List nonDirectoryPaths = resourceIds.stream().filter(s3ResourceId -> !s3ResourceId.isDirectory()).collect(Collectors.toList());
        ArrayListMultimap<String, String> keysByBucket = ArrayListMultimap.create();
        for (S3ResourceId path : nonDirectoryPaths) {
            keysByBucket.put(path.getBucket(), path.getKey());
        }
        ArrayList tasks = new ArrayList();
        for (String bucket : keysByBucket.keySet()) {
            for (List keysPartition : Iterables.partition(keysByBucket.get(bucket), 1000)) {
                tasks.add(() -> {
                    this.delete(bucket, keysPartition);
                    return null;
                });
            }
        }
        this.callTasks(tasks);
    }

    private void delete(String bucket, Collection<String> keys) throws IOException {
        Preconditions.checkArgument(keys.size() <= 1000, "only %s keys can be deleted per request, but got %s", 1000, keys.size());
        List deleteKeyVersions = keys.stream().map(DeleteObjectsRequest.KeyVersion::new).collect(Collectors.toList());
        DeleteObjectsRequest request = new DeleteObjectsRequest(bucket).withKeys(deleteKeyVersions);
        try {
            this.amazonS3.deleteObjects(request);
        }
        catch (AmazonClientException e) {
            throw new IOException(e);
        }
    }

    protected S3ResourceId matchNewResource(String singleResourceSpec, boolean isDirectory) {
        if (isDirectory) {
            if (!singleResourceSpec.endsWith("/")) {
                singleResourceSpec = singleResourceSpec + "/";
            }
        } else {
            Preconditions.checkArgument(!singleResourceSpec.endsWith("/"), "Expected a file path, but [%s] ends with '/'. This is unsupported in S3FileSystem.", (Object)singleResourceSpec);
        }
        return S3ResourceId.fromUri(singleResourceSpec);
    }

    private <T> List<T> callTasks(Collection<Callable<T>> tasks) throws IOException {
        try {
            ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>(tasks.size());
            for (Callable<T> task : tasks) {
                futures.add(MoreFutures.supplyAsync(task::call, (ExecutorService)this.executorService));
            }
            return (List)MoreFutures.get((CompletionStage)MoreFutures.allAsList(futures));
        }
        catch (ExecutionException e) {
            if (e.getCause() != null) {
                if (e.getCause() instanceof IOException) {
                    throw (IOException)e.getCause();
                }
                throw new IOException(e.getCause());
            }
            throw new IOException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("executor service was interrupted");
        }
    }

    @AutoValue
    static abstract class PathWithEncoding {
        PathWithEncoding() {
        }

        abstract S3ResourceId getPath();

        @Nullable
        abstract String getContentEncoding();

        @Nullable
        abstract IOException getException();

        static PathWithEncoding create(S3ResourceId path, String contentEncoding) {
            Preconditions.checkNotNull(path, "path");
            Preconditions.checkNotNull(contentEncoding, "contentEncoding");
            return new AutoValue_S3FileSystem_PathWithEncoding(path, contentEncoding, null);
        }

        static PathWithEncoding create(S3ResourceId path, IOException exception) {
            Preconditions.checkNotNull(path, "path");
            Preconditions.checkNotNull(exception, "exception");
            return new AutoValue_S3FileSystem_PathWithEncoding(path, null, exception);
        }
    }

    @AutoValue
    static abstract class ExpandedGlob {
        ExpandedGlob() {
        }

        abstract S3ResourceId getGlobPath();

        @Nullable
        abstract List<S3ResourceId> getExpandedPaths();

        @Nullable
        abstract IOException getException();

        static ExpandedGlob create(S3ResourceId globPath, List<S3ResourceId> expandedPaths) {
            Preconditions.checkNotNull(globPath, "globPath");
            Preconditions.checkNotNull(expandedPaths, "expandedPaths");
            return new AutoValue_S3FileSystem_ExpandedGlob(globPath, expandedPaths, null);
        }

        static ExpandedGlob create(S3ResourceId globPath, IOException exception) {
            Preconditions.checkNotNull(globPath, "globPath");
            Preconditions.checkNotNull(exception, "exception");
            return new AutoValue_S3FileSystem_ExpandedGlob(globPath, null, exception);
        }
    }
}

