/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.common.concurrent;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.LongStream;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.common.util.OrderedScheduler;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class TestFutureUtils {
    @Test
    public void testComplete() throws Exception {
        CompletableFuture future = FutureUtils.createFuture();
        FutureUtils.complete((CompletableFuture)future, (Object)1024L);
        Assert.assertEquals((long)1024L, (long)((Long)FutureUtils.result((CompletableFuture)future)));
    }

    @Test(expected=TestException.class)
    public void testCompleteExceptionally() throws Exception {
        CompletableFuture future = FutureUtils.createFuture();
        FutureUtils.completeExceptionally((CompletableFuture)future, (Throwable)new TestException());
        FutureUtils.result((CompletableFuture)future);
    }

    @Test
    public void testWhenCompleteAsync() throws Exception {
        OrderedScheduler scheduler = (OrderedScheduler)OrderedScheduler.newSchedulerBuilder().name("test-when-complete-async").numThreads(1).build();
        AtomicLong resultHolder = new AtomicLong(0L);
        CountDownLatch latch = new CountDownLatch(1);
        CompletableFuture future = FutureUtils.createFuture();
        FutureUtils.whenCompleteAsync((CompletableFuture)future, (result, cause) -> {
            resultHolder.set((long)result);
            latch.countDown();
        }, (OrderedScheduler)scheduler, (Object)new Object());
        FutureUtils.complete((CompletableFuture)future, (Object)1234L);
        latch.await();
        Assert.assertEquals((long)1234L, (long)resultHolder.get());
    }

    @Test
    public void testProxyToSuccess() throws Exception {
        CompletableFuture src = FutureUtils.createFuture();
        CompletableFuture target = FutureUtils.createFuture();
        FutureUtils.proxyTo((CompletableFuture)src, (CompletableFuture)target);
        FutureUtils.complete((CompletableFuture)src, (Object)10L);
        Assert.assertEquals((long)10L, (long)((Long)FutureUtils.result((CompletableFuture)target)));
    }

    @Test(expected=TestException.class)
    public void testProxyToFailure() throws Exception {
        CompletableFuture src = FutureUtils.createFuture();
        CompletableFuture target = FutureUtils.createFuture();
        FutureUtils.proxyTo((CompletableFuture)src, (CompletableFuture)target);
        FutureUtils.completeExceptionally((CompletableFuture)src, (Throwable)new TestException());
        FutureUtils.result((CompletableFuture)target);
    }

    @Test
    public void testVoid() throws Exception {
        CompletableFuture voidFuture = FutureUtils.Void();
        Assert.assertTrue((boolean)voidFuture.isDone());
        Assert.assertFalse((boolean)voidFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)voidFuture.isCancelled());
    }

    @Test
    public void testCollectEmptyList() throws Exception {
        ArrayList futures = Lists.newArrayList();
        List result = (List)FutureUtils.result((CompletableFuture)FutureUtils.collect((List)futures));
        Assert.assertTrue((boolean)result.isEmpty());
    }

    @Test
    public void testCollectTenItems() throws Exception {
        ArrayList futures = Lists.newArrayList();
        ArrayList expectedResults = Lists.newArrayList();
        for (int i = 0; i < 10; ++i) {
            futures.add(FutureUtils.value((Object)i));
            expectedResults.add(i);
        }
        List results = (List)FutureUtils.result((CompletableFuture)FutureUtils.collect((List)futures));
        Assert.assertEquals((Object)expectedResults, (Object)results);
    }

    @Test(expected=TestException.class)
    public void testCollectFailures() throws Exception {
        ArrayList futures = Lists.newArrayList();
        ArrayList expectedResults = Lists.newArrayList();
        for (int i = 0; i < 10; ++i) {
            if (i == 9) {
                futures.add(FutureUtils.value((Object)i));
            } else {
                futures.add(FutureUtils.exception((Throwable)new TestException()));
            }
            expectedResults.add(i);
        }
        FutureUtils.result((CompletableFuture)FutureUtils.collect((List)futures));
    }

    @Test
    public void testWithinAlreadyDone() throws Exception {
        OrderedScheduler scheduler = (OrderedScheduler)Mockito.mock(OrderedScheduler.class);
        CompletableFuture doneFuture = FutureUtils.value((Object)1234L);
        CompletableFuture withinFuture = FutureUtils.within((CompletableFuture)doneFuture, (long)10L, (TimeUnit)TimeUnit.MILLISECONDS, (Throwable)new TestException(), (OrderedScheduler)scheduler, (Object)1234L);
        TimeUnit.MILLISECONDS.sleep(20L);
        Assert.assertTrue((boolean)withinFuture.isDone());
        Assert.assertFalse((boolean)withinFuture.isCancelled());
        Assert.assertFalse((boolean)withinFuture.isCompletedExceptionally());
        ((OrderedScheduler)Mockito.verify((Object)scheduler, (VerificationMode)Mockito.times((int)0))).scheduleOrdered((Object)ArgumentMatchers.eq((long)1234L), (Runnable)ArgumentMatchers.isA(Runnable.class), (long)ArgumentMatchers.eq((int)10), (TimeUnit)((Object)ArgumentMatchers.eq((Object)((Object)TimeUnit.MILLISECONDS))));
    }

    @Test
    public void testWithinZeroTimeout() throws Exception {
        OrderedScheduler scheduler = (OrderedScheduler)Mockito.mock(OrderedScheduler.class);
        CompletableFuture newFuture = FutureUtils.createFuture();
        CompletableFuture withinFuture = FutureUtils.within((CompletableFuture)newFuture, (long)0L, (TimeUnit)TimeUnit.MILLISECONDS, (Throwable)new TestException(), (OrderedScheduler)scheduler, (Object)1234L);
        TimeUnit.MILLISECONDS.sleep(20L);
        Assert.assertFalse((boolean)withinFuture.isDone());
        Assert.assertFalse((boolean)withinFuture.isCancelled());
        Assert.assertFalse((boolean)withinFuture.isCompletedExceptionally());
        ((OrderedScheduler)Mockito.verify((Object)scheduler, (VerificationMode)Mockito.times((int)0))).scheduleOrdered((Object)ArgumentMatchers.eq((long)1234L), (Runnable)ArgumentMatchers.isA(Runnable.class), (long)ArgumentMatchers.eq((int)10), (TimeUnit)((Object)ArgumentMatchers.eq((Object)((Object)TimeUnit.MILLISECONDS))));
    }

    @Test
    public void testWithinCompleteBeforeTimeout() throws Exception {
        OrderedScheduler scheduler = (OrderedScheduler)Mockito.mock(OrderedScheduler.class);
        ScheduledFuture scheduledFuture = (ScheduledFuture)Mockito.mock(ScheduledFuture.class);
        Mockito.when((Object)scheduler.scheduleOrdered(ArgumentMatchers.any(Object.class), (Runnable)ArgumentMatchers.any(Runnable.class), ArgumentMatchers.anyLong(), (TimeUnit)((Object)ArgumentMatchers.any(TimeUnit.class)))).thenAnswer(invocationOnMock -> scheduledFuture);
        CompletableFuture newFuture = FutureUtils.createFuture();
        CompletableFuture withinFuture = FutureUtils.within((CompletableFuture)newFuture, (long)Long.MAX_VALUE, (TimeUnit)TimeUnit.MILLISECONDS, (Throwable)new TestException(), (OrderedScheduler)scheduler, (Object)1234L);
        Assert.assertFalse((boolean)withinFuture.isDone());
        Assert.assertFalse((boolean)withinFuture.isCancelled());
        Assert.assertFalse((boolean)withinFuture.isCompletedExceptionally());
        newFuture.complete(5678L);
        Assert.assertTrue((boolean)withinFuture.isDone());
        Assert.assertFalse((boolean)withinFuture.isCancelled());
        Assert.assertFalse((boolean)withinFuture.isCompletedExceptionally());
        Assert.assertEquals((Object)5678L, (Object)FutureUtils.result((CompletableFuture)withinFuture));
        ((ScheduledFuture)Mockito.verify((Object)scheduledFuture, (VerificationMode)Mockito.times((int)1))).cancel(ArgumentMatchers.eq((boolean)true));
    }

    @Test
    public void testIgnoreSuccess() {
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture ignoredFuture = FutureUtils.ignore((CompletableFuture)underlyFuture);
        underlyFuture.complete(1234L);
        Assert.assertTrue((boolean)ignoredFuture.isDone());
        Assert.assertFalse((boolean)ignoredFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)ignoredFuture.isCancelled());
    }

    @Test
    public void testIgnoreFailure() {
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture ignoredFuture = FutureUtils.ignore((CompletableFuture)underlyFuture);
        underlyFuture.completeExceptionally(new TestException());
        Assert.assertTrue((boolean)ignoredFuture.isDone());
        Assert.assertFalse((boolean)ignoredFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)ignoredFuture.isCancelled());
    }

    @Test
    public void testEnsureSuccess() throws Exception {
        CountDownLatch ensureLatch = new CountDownLatch(1);
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture ensuredFuture = FutureUtils.ensure((CompletableFuture)underlyFuture, () -> ensureLatch.countDown());
        underlyFuture.complete(1234L);
        FutureUtils.result((CompletableFuture)ensuredFuture);
        Assert.assertTrue((boolean)ensuredFuture.isDone());
        Assert.assertFalse((boolean)ensuredFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)ensuredFuture.isCancelled());
        ensureLatch.await();
    }

    @Test
    public void testEnsureFailure() throws Exception {
        CountDownLatch ensureLatch = new CountDownLatch(1);
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture ensuredFuture = FutureUtils.ensure((CompletableFuture)underlyFuture, () -> ensureLatch.countDown());
        underlyFuture.completeExceptionally(new TestException());
        FutureUtils.result((CompletableFuture)FutureUtils.ignore((CompletableFuture)ensuredFuture));
        Assert.assertTrue((boolean)ensuredFuture.isDone());
        Assert.assertTrue((boolean)ensuredFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)ensuredFuture.isCancelled());
        ensureLatch.await();
    }

    @Test
    public void testRescueSuccess() throws Exception {
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        Function rescueFuc = (Function)Mockito.mock(Function.class);
        CompletableFuture rescuedFuture = FutureUtils.rescue((CompletableFuture)underlyFuture, (Function)rescueFuc);
        underlyFuture.complete(1234L);
        FutureUtils.result((CompletableFuture)rescuedFuture);
        Assert.assertTrue((boolean)rescuedFuture.isDone());
        Assert.assertFalse((boolean)rescuedFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)rescuedFuture.isCancelled());
        ((Function)Mockito.verify((Object)rescueFuc, (VerificationMode)Mockito.times((int)0))).apply((Throwable)ArgumentMatchers.any(Throwable.class));
    }

    @Test
    public void testRescueFailure() throws Exception {
        CompletableFuture futureCompletedAtRescue = FutureUtils.value((Object)3456L);
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture rescuedFuture = FutureUtils.rescue((CompletableFuture)underlyFuture, cause -> futureCompletedAtRescue);
        underlyFuture.completeExceptionally(new TestException());
        FutureUtils.result((CompletableFuture)rescuedFuture);
        Assert.assertTrue((boolean)rescuedFuture.isDone());
        Assert.assertFalse((boolean)rescuedFuture.isCompletedExceptionally());
        Assert.assertFalse((boolean)rescuedFuture.isCancelled());
        Assert.assertEquals((Object)3456L, (Object)FutureUtils.result((CompletableFuture)rescuedFuture));
    }

    @Test
    public void testStatsSuccess() throws Exception {
        OpStatsLogger statsLogger = (OpStatsLogger)Mockito.mock(OpStatsLogger.class);
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture statsFuture = FutureUtils.stats((CompletableFuture)underlyFuture, (OpStatsLogger)statsLogger, (Stopwatch)Stopwatch.createStarted());
        underlyFuture.complete(1234L);
        FutureUtils.result((CompletableFuture)statsFuture);
        ((OpStatsLogger)Mockito.verify((Object)statsLogger, (VerificationMode)Mockito.times((int)1))).registerSuccessfulEvent(ArgumentMatchers.anyLong(), (TimeUnit)((Object)ArgumentMatchers.eq((Object)((Object)TimeUnit.MICROSECONDS))));
    }

    @Test
    public void testStatsFailure() throws Exception {
        OpStatsLogger statsLogger = (OpStatsLogger)Mockito.mock(OpStatsLogger.class);
        CompletableFuture underlyFuture = FutureUtils.createFuture();
        CompletableFuture statsFuture = FutureUtils.stats((CompletableFuture)underlyFuture, (OpStatsLogger)statsLogger, (Stopwatch)Stopwatch.createStarted());
        underlyFuture.completeExceptionally(new TestException());
        FutureUtils.result((CompletableFuture)FutureUtils.ignore((CompletableFuture)statsFuture));
        ((OpStatsLogger)Mockito.verify((Object)statsLogger, (VerificationMode)Mockito.times((int)1))).registerFailedEvent(ArgumentMatchers.anyLong(), (TimeUnit)((Object)ArgumentMatchers.eq((Object)((Object)TimeUnit.MICROSECONDS))));
    }

    @Test
    public void testProcessListSuccess() throws Exception {
        ArrayList longList = Lists.newArrayList((Iterator)LongStream.range(0L, 10L).iterator());
        List expectedList = Lists.transform((List)longList, aLong -> 2L * aLong);
        Function<Long, CompletableFuture> sumFunc = value -> FutureUtils.value((Object)(2L * value));
        CompletableFuture totalFuture = FutureUtils.processList((List)longList, sumFunc, null);
        Assert.assertEquals((Object)expectedList, (Object)FutureUtils.result((CompletableFuture)totalFuture));
    }

    @Test
    public void testProcessEmptyList() throws Exception {
        ArrayList longList = Lists.newArrayList();
        List expectedList = Lists.transform((List)longList, aLong -> 2L * aLong);
        Function<Long, CompletableFuture> sumFunc = value -> FutureUtils.value((Object)(2L * value));
        CompletableFuture totalFuture = FutureUtils.processList((List)longList, sumFunc, null);
        Assert.assertEquals((Object)expectedList, (Object)FutureUtils.result((CompletableFuture)totalFuture));
    }

    @Test
    public void testProcessListFailures() throws Exception {
        ArrayList longList = Lists.newArrayList((Iterator)LongStream.range(0L, 10L).iterator());
        AtomicLong total = new AtomicLong(0L);
        Function<Long, CompletableFuture> sumFunc = value -> {
            if (value < 5L) {
                total.addAndGet((long)value);
                return FutureUtils.value((Object)(2L * value));
            }
            return FutureUtils.exception((Throwable)new TestException());
        };
        CompletableFuture totalFuture = FutureUtils.processList((List)longList, sumFunc, null);
        try {
            FutureUtils.result((CompletableFuture)totalFuture);
            Assert.fail((String)"Should fail with TestException");
        }
        catch (TestException testException) {
            // empty catch block
        }
        Assert.assertEquals((long)10L, (long)total.get());
    }

    static class TestException
    extends IOException {
        private static final long serialVersionUID = -6256482498453846308L;

        public TestException() {
            super("test-exception");
        }
    }
}

