/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie.storage.directentrylogger;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.util.ReferenceCountUtil;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import org.apache.bookkeeper.common.util.ExceptionMessageHelper;
import org.apache.bookkeeper.common.util.nativeio.NativeIO;
import org.apache.bookkeeper.shaded.com.google.common.base.Preconditions;

class Buffer {
    public static final byte PADDING_BYTE = -16;
    public static final int ALIGNMENT = 4096;
    private static final int MAX_ALIGNMENT = 0x7FFFF000;
    static final byte[] PADDING = Buffer.generatePadding();
    final NativeIO nativeIO;
    final int bufferSize;
    ByteBuf buffer;
    ByteBuffer byteBuffer;
    long pointer = 0L;

    Buffer(NativeIO nativeIO, int bufferSize) throws IOException {
        Preconditions.checkArgument(Buffer.isAligned(bufferSize), "Buffer size not aligned %d", bufferSize);
        this.buffer = this.allocateAligned(4096, bufferSize);
        this.nativeIO = nativeIO;
        this.bufferSize = bufferSize;
        this.byteBuffer = this.buffer.nioBuffer(0, bufferSize);
        this.byteBuffer.order(ByteOrder.BIG_ENDIAN);
    }

    private ByteBuf allocateAligned(int alignment, int bufferSize) {
        ByteBuf buf = PooledByteBufAllocator.DEFAULT.directBuffer(bufferSize + alignment);
        long addr = buf.memoryAddress();
        if ((addr & (long)(alignment - 1)) == 0L) {
            this.pointer = addr;
            return buf.slice(0, bufferSize);
        }
        int alignOffset = (int)((long)alignment - (addr & (long)(alignment - 1)));
        this.pointer = addr + (long)alignOffset;
        return buf.slice(alignOffset, bufferSize);
    }

    boolean hasSpace(int size) throws IOException {
        if (size > this.bufferSize) {
            throw new IOException(ExceptionMessageHelper.exMsg("Write too large").kv("writeSize", size).kv("maxSize", this.bufferSize).toString());
        }
        return this.byteBuffer.remaining() >= size;
    }

    boolean hasData(int offset, int size) {
        return offset + size <= this.bufferSize;
    }

    void writeInt(int value) throws IOException {
        this.byteBuffer.putInt(value);
    }

    void writeByteBuf(ByteBuf bytebuf) throws IOException {
        int bytesWritten = bytebuf.readableBytes();
        ByteBuffer bytesToPut = bytebuf.nioBuffer();
        this.byteBuffer.put(bytesToPut);
        bytebuf.skipBytes(bytesWritten);
    }

    int readInt(int offset) throws IOException {
        if (!this.hasData(offset, 4)) {
            throw new IOException(ExceptionMessageHelper.exMsg("Buffer cannot satify int read").kv("offset", offset).kv("bufferSize", this.bufferSize).toString());
        }
        try {
            return this.byteBuffer.getInt(offset);
        }
        catch (Exception e) {
            throw new IOException(ExceptionMessageHelper.exMsg("Error reading int").kv("byteBuffer", this.byteBuffer.toString()).kv("offset", offset).kv("bufferSize", this.bufferSize).toString(), e);
        }
    }

    long readLong(int offset) throws IOException {
        if (!this.hasData(offset, 8)) {
            throw new IOException(ExceptionMessageHelper.exMsg("Buffer cannot satify long read").kv("offset", offset).kv("bufferSize", this.bufferSize).toString());
        }
        try {
            return this.byteBuffer.getLong(offset);
        }
        catch (Exception e) {
            throw new IOException(ExceptionMessageHelper.exMsg("Error reading long").kv("byteBuffer", this.byteBuffer.toString()).kv("offset", offset).kv("bufferSize", this.bufferSize).toString(), e);
        }
    }

    int readByteBuf(ByteBuf buffer, int offset, int size) throws IOException {
        int originalLimit = this.byteBuffer.limit();
        this.byteBuffer.position(offset);
        int bytesToRead = Math.min(size, this.byteBuffer.capacity() - offset);
        this.byteBuffer.limit(offset + bytesToRead);
        try {
            buffer.writeBytes(this.byteBuffer);
        }
        catch (Exception e) {
            throw new IOException(ExceptionMessageHelper.exMsg("Error reading buffer").kv("byteBuffer", this.byteBuffer.toString()).kv("offset", offset).kv("size", size).kv("bufferSize", this.bufferSize).toString(), e);
        }
        finally {
            this.byteBuffer.limit(originalLimit);
        }
        return bytesToRead;
    }

    long pointer() {
        return this.pointer;
    }

    long pointer(long offset, long expectedWrite) {
        if (offset == 0L) {
            return this.pointer;
        }
        if (offset + expectedWrite > (long)this.byteBuffer.capacity()) {
            throw new IllegalArgumentException(ExceptionMessageHelper.exMsg("Buffer overflow").kv("offset", offset).kv("expectedWrite", expectedWrite).kv("capacity", this.byteBuffer.capacity()).toString());
        }
        return this.pointer + offset;
    }

    int position() {
        return this.byteBuffer.position();
    }

    int size() {
        return this.bufferSize;
    }

    int padToAlignment() {
        int bufferPos = this.byteBuffer.position();
        int nextAlignment = Buffer.nextAlignment(bufferPos);
        this.byteBuffer.put(PADDING, 0, nextAlignment - bufferPos);
        return nextAlignment;
    }

    void reset() {
        this.byteBuffer.clear();
    }

    void free() {
        ReferenceCountUtil.release((Object)this.buffer);
        this.buffer = null;
        this.byteBuffer = null;
    }

    private static byte[] generatePadding() {
        byte[] padding = new byte[4096];
        Arrays.fill(padding, (byte)-16);
        return padding;
    }

    static boolean isAligned(long size) {
        return size >= 0L && (0xFFFL & size) == 0L;
    }

    static int nextAlignment(int pos) {
        Preconditions.checkArgument(pos <= 0x7FFFF000, "position (0x%x) must be lower or equal to max alignment (0x%x)", pos, 0x7FFFF000);
        Preconditions.checkArgument(pos >= 0, "position (0x%x) must be positive", pos);
        return pos + 4095 & 0xFFFFF000;
    }
}

