/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BufferedChannel;
import org.apache.bookkeeper.bookie.FileInfo;
import org.apache.bookkeeper.bookie.IndexPersistenceMgr;
import org.apache.bookkeeper.bookie.Journal;
import org.apache.bookkeeper.bookie.JournalChannel;
import org.apache.bookkeeper.client.ClientUtil;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.conf.TestBKConfiguration;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@RunWith(value=PowerMockRunner.class)
@PrepareForTest(value={JournalChannel.class})
public class BookieJournalTest {
    private static final Logger LOG = LoggerFactory.getLogger(BookieJournalTest.class);
    final Random r = new Random(System.currentTimeMillis());
    final List<File> tempDirs = new ArrayList<File>();

    File createTempDir(String prefix, String suffix) throws IOException {
        File dir = IOUtils.createTempDir((String)prefix, (String)suffix);
        this.tempDirs.add(dir);
        return dir;
    }

    @After
    public void tearDown() throws Exception {
        for (File dir : this.tempDirs) {
            FileUtils.deleteDirectory((File)dir);
        }
        this.tempDirs.clear();
    }

    private void writeIndexFileForLedger(File indexDir, long ledgerId, byte[] masterKey) throws Exception {
        File fn = new File(indexDir, IndexPersistenceMgr.getLedgerName((long)ledgerId));
        fn.getParentFile().mkdirs();
        FileInfo fi = new FileInfo(fn, masterKey, 1);
        fi.write(new ByteBuffer[]{ByteBuffer.allocate(0)}, 0L);
        fi.close(true);
    }

    private void writePartialIndexFileForLedger(File indexDir, long ledgerId, byte[] masterKey, boolean truncateToMasterKey) throws Exception {
        File fn = new File(indexDir, IndexPersistenceMgr.getLedgerName((long)ledgerId));
        fn.getParentFile().mkdirs();
        FileInfo fi = new FileInfo(fn, masterKey, 1);
        fi.write(new ByteBuffer[]{ByteBuffer.allocate(0)}, 0L);
        fi.close(true);
        int headerLen = 12 + masterKey.length;
        int leftSize = truncateToMasterKey ? this.r.nextInt(headerLen) : headerLen + this.r.nextInt(1024 - headerLen);
        FileChannel fc = new RandomAccessFile(fn, "rw").getChannel();
        fc.truncate(leftSize);
        fc.close();
    }

    private static ByteBuf generateFenceEntry(long ledgerId) {
        ByteBuf bb = Unpooled.buffer();
        bb.writeLong(ledgerId);
        bb.writeLong(-8192L);
        return bb;
    }

    private static ByteBuf generateMetaEntry(long ledgerId, byte[] masterKey) {
        ByteBuf bb = Unpooled.buffer();
        bb.writeLong(ledgerId);
        bb.writeLong(-4096L);
        bb.writeInt(masterKey.length);
        bb.writeBytes(masterKey);
        return bb;
    }

    private void writeJunkJournal(File journalDir) throws Exception {
        long logId = System.currentTimeMillis();
        File fn = new File(journalDir, Long.toHexString(logId) + ".txn");
        FileChannel fc = new RandomAccessFile(fn, "rw").getChannel();
        ByteBuffer zeros = ByteBuffer.allocate(512);
        fc.write(zeros, 0x400000L);
        fc.position(0L);
        for (int i = 1; i <= 10; ++i) {
            fc.write(ByteBuffer.wrap("JunkJunkJunk".getBytes()));
        }
    }

    private void writePreV2Journal(File journalDir, int numEntries) throws Exception {
        long logId = System.currentTimeMillis();
        File fn = new File(journalDir, Long.toHexString(logId) + ".txn");
        FileChannel fc = new RandomAccessFile(fn, "rw").getChannel();
        ByteBuffer zeros = ByteBuffer.allocate(512);
        fc.write(zeros, 0x400000L);
        fc.position(0L);
        byte[] data = "JournalTestData".getBytes();
        long lastConfirmed = -1L;
        for (int i = 1; i <= numEntries; ++i) {
            ByteBuf packet = ClientUtil.generatePacket(1L, i, lastConfirmed, i * data.length, data);
            lastConfirmed = i;
            ByteBuffer lenBuff = ByteBuffer.allocate(4);
            lenBuff.putInt(packet.readableBytes());
            lenBuff.flip();
            fc.write(lenBuff);
            fc.write(packet.nioBuffer());
            packet.release();
        }
    }

    private static void moveToPosition(JournalChannel jc, long pos) throws IOException {
        jc.fc.position(pos);
        jc.bc.position = pos;
        jc.bc.writeBufferStartPosition.set(pos);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void updateJournalVersion(JournalChannel jc, int journalVersion) throws IOException {
        long prevPos = jc.fc.position();
        try {
            ByteBuffer versionBuffer = ByteBuffer.allocate(4);
            versionBuffer.putInt(journalVersion);
            versionBuffer.flip();
            jc.fc.position(4L);
            IOUtils.writeFully((WritableByteChannel)jc.fc, (ByteBuffer)versionBuffer);
            jc.fc.force(true);
        }
        finally {
            jc.fc.position(prevPos);
        }
    }

    private JournalChannel writeV2Journal(File journalDir, int numEntries) throws Exception {
        long logId = System.currentTimeMillis();
        JournalChannel jc = new JournalChannel(journalDir, logId);
        BookieJournalTest.moveToPosition(jc, 8L);
        BufferedChannel bc = jc.getBufferedChannel();
        byte[] data = new byte[1024];
        Arrays.fill(data, (byte)88);
        long lastConfirmed = -1L;
        for (int i = 1; i <= numEntries; ++i) {
            ByteBuf packet = ClientUtil.generatePacket(1L, i, lastConfirmed, i * data.length, data);
            lastConfirmed = i;
            ByteBuffer lenBuff = ByteBuffer.allocate(4);
            lenBuff.putInt(packet.readableBytes());
            lenBuff.flip();
            bc.write(Unpooled.wrappedBuffer((ByteBuffer)lenBuff));
            bc.write(packet);
            packet.release();
        }
        bc.flushAndForceWrite(false);
        BookieJournalTest.updateJournalVersion(jc, 2);
        return jc;
    }

    private JournalChannel writeV3Journal(File journalDir, int numEntries, byte[] masterKey) throws Exception {
        long logId = System.currentTimeMillis();
        JournalChannel jc = new JournalChannel(journalDir, logId);
        BookieJournalTest.moveToPosition(jc, 8L);
        BufferedChannel bc = jc.getBufferedChannel();
        byte[] data = new byte[1024];
        Arrays.fill(data, (byte)88);
        long lastConfirmed = -1L;
        for (int i = 0; i <= numEntries; ++i) {
            ByteBuf packet = i == 0 ? BookieJournalTest.generateMetaEntry(1L, masterKey) : ClientUtil.generatePacket(1L, i, lastConfirmed, i * data.length, data);
            lastConfirmed = i;
            ByteBuffer lenBuff = ByteBuffer.allocate(4);
            lenBuff.putInt(packet.readableBytes());
            lenBuff.flip();
            bc.write(Unpooled.wrappedBuffer((ByteBuffer)lenBuff));
            bc.write(packet);
            packet.release();
        }
        bc.flushAndForceWrite(false);
        BookieJournalTest.updateJournalVersion(jc, 3);
        return jc;
    }

    private JournalChannel writeV4Journal(File journalDir, int numEntries, byte[] masterKey) throws Exception {
        long logId = System.currentTimeMillis();
        JournalChannel jc = new JournalChannel(journalDir, logId);
        BookieJournalTest.moveToPosition(jc, 8L);
        BufferedChannel bc = jc.getBufferedChannel();
        byte[] data = new byte[1024];
        Arrays.fill(data, (byte)88);
        long lastConfirmed = -1L;
        for (int i = 0; i <= numEntries; ++i) {
            ByteBuf packet = i == 0 ? BookieJournalTest.generateMetaEntry(1L, masterKey) : ClientUtil.generatePacket(1L, i, lastConfirmed, i * data.length, data);
            lastConfirmed = i;
            ByteBuffer lenBuff = ByteBuffer.allocate(4);
            lenBuff.putInt(packet.readableBytes());
            lenBuff.flip();
            bc.write(Unpooled.wrappedBuffer((ByteBuffer)lenBuff));
            bc.write(packet);
            packet.release();
        }
        ByteBuf packet = BookieJournalTest.generateFenceEntry(1L);
        ByteBuf lenBuf = Unpooled.buffer();
        lenBuf.writeInt(packet.readableBytes());
        bc.write(lenBuf);
        bc.write(packet);
        bc.flushAndForceWrite(false);
        BookieJournalTest.updateJournalVersion(jc, 4);
        return jc;
    }

    static JournalChannel writeV5Journal(File journalDir, int numEntries, byte[] masterKey) throws Exception {
        return BookieJournalTest.writeV5Journal(journalDir, numEntries, masterKey, false);
    }

    static JournalChannel writeV5Journal(File journalDir, int numEntries, byte[] masterKey, boolean corruptLength) throws Exception {
        long logId = System.currentTimeMillis();
        JournalChannel jc = new JournalChannel(journalDir, logId);
        BufferedChannel bc = jc.getBufferedChannel();
        ByteBuf paddingBuff = Unpooled.buffer();
        paddingBuff.writeZero(1024);
        byte[] data = new byte[0x400000];
        Arrays.fill(data, (byte)88);
        long lastConfirmed = -1L;
        long length = 0L;
        for (int i = 0; i <= numEntries; ++i) {
            ByteBuf packet = i == 0 ? BookieJournalTest.generateMetaEntry(1L, masterKey) : ClientUtil.generatePacket(1L, i, lastConfirmed, length, data, 0, i);
            lastConfirmed = i;
            length += (long)i;
            ByteBuf lenBuff = Unpooled.buffer();
            if (corruptLength) {
                lenBuff.writeInt(-1);
            } else {
                lenBuff.writeInt(packet.readableBytes());
            }
            bc.write(lenBuff);
            bc.write(packet);
            packet.release();
            Journal.writePaddingBytes((JournalChannel)jc, (ByteBuf)paddingBuff, (int)512);
        }
        ByteBuf packet = BookieJournalTest.generateFenceEntry(1L);
        ByteBuf lenBuf = Unpooled.buffer();
        lenBuf.writeInt(packet.readableBytes());
        bc.write(lenBuf);
        bc.write(packet);
        Journal.writePaddingBytes((JournalChannel)jc, (ByteBuf)paddingBuff, (int)512);
        bc.flushAndForceWrite(false);
        BookieJournalTest.updateJournalVersion(jc, 5);
        return jc;
    }

    @Test
    public void testPreV2Journal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writePreV2Journal(Bookie.getCurrentDirectory((File)journalDir), 100);
        this.writeIndexFileForLedger(Bookie.getCurrentDirectory((File)ledgerDir), 1L, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = this.createBookieAndReadJournal(conf);
        b.readEntry(1L, 100L);
        try {
            b.readEntry(1L, 101L);
            Assert.fail((String)"Shouldn't have found entry 101");
        }
        catch (Bookie.NoEntryException noEntryException) {
            // empty catch block
        }
        b.shutdown();
    }

    @Test
    public void testV4Journal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writeV4Journal(Bookie.getCurrentDirectory((File)journalDir), 100, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = this.createBookieAndReadJournal(conf);
        b.readEntry(1L, 100L);
        try {
            b.readEntry(1L, 101L);
            Assert.fail((String)"Shouldn't have found entry 101");
        }
        catch (Bookie.NoEntryException noEntryException) {
            // empty catch block
        }
        Assert.assertTrue((boolean)b.handles.getHandle(1L, "testPasswd".getBytes()).isFenced());
        b.shutdown();
    }

    @Test
    public void testV5Journal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        BookieJournalTest.writeV5Journal(Bookie.getCurrentDirectory((File)journalDir), 1024, "testV5Journal".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = this.createBookieAndReadJournal(conf);
        for (int i = 1; i <= 1024; ++i) {
            b.readEntry(1L, (long)i);
        }
        try {
            b.readEntry(1L, 1025L);
            Assert.fail((String)"Shouldn't have found entry 1025");
        }
        catch (Bookie.NoEntryException noEntryException) {
            // empty catch block
        }
        Assert.assertTrue((boolean)b.handles.getHandle(1L, "testV5Journal".getBytes()).isFenced());
        b.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAllJunkJournal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writeJunkJournal(Bookie.getCurrentDirectory((File)journalDir));
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = null;
        try {
            b = new Bookie(conf);
            Assert.fail((String)"Shouldn't have been able to start without admin");
        }
        catch (Throwable throwable) {
        }
        finally {
            if (b != null) {
                b.shutdown();
            }
        }
    }

    @Test
    public void testEmptyJournal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writePreV2Journal(Bookie.getCurrentDirectory((File)journalDir), 0);
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = new Bookie(conf);
    }

    @Test
    public void testHeaderOnlyJournal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writeV2Journal(Bookie.getCurrentDirectory((File)journalDir), 0);
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = new Bookie(conf);
    }

    @Test
    public void testJunkEndedJournal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        JournalChannel jc = this.writeV2Journal(Bookie.getCurrentDirectory((File)journalDir), 0);
        jc.getBufferedChannel().write(Unpooled.wrappedBuffer((byte[])"JunkJunkJunk".getBytes()));
        jc.getBufferedChannel().flushAndForceWrite(false);
        this.writeIndexFileForLedger(ledgerDir, 1L, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = null;
        try {
            b = new Bookie(conf);
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    @Test
    public void testTruncatedInLenJournal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        JournalChannel jc = this.writeV2Journal(Bookie.getCurrentDirectory((File)journalDir), 100);
        ByteBuffer zeros = ByteBuffer.allocate(2048);
        jc.fc.position(jc.getBufferedChannel().position() - 1065L);
        jc.fc.write(zeros);
        jc.fc.force(false);
        this.writeIndexFileForLedger(Bookie.getCurrentDirectory((File)ledgerDir), 1L, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = this.createBookieAndReadJournal(conf);
        b.readEntry(1L, 99L);
        try {
            b.readEntry(1L, 100L);
            Assert.fail((String)"Shouldn't have found entry 100");
        }
        catch (Bookie.NoEntryException noEntryException) {
            // empty catch block
        }
    }

    @Test
    public void testTruncatedInEntryJournal() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        JournalChannel jc = this.writeV2Journal(Bookie.getCurrentDirectory((File)journalDir), 100);
        ByteBuffer zeros = ByteBuffer.allocate(2048);
        jc.fc.position(jc.getBufferedChannel().position() - 768L);
        jc.fc.write(zeros);
        jc.fc.force(false);
        this.writeIndexFileForLedger(Bookie.getCurrentDirectory((File)ledgerDir), 1L, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = this.createBookieAndReadJournal(conf);
        b.readEntry(1L, 99L);
        ByteBuf buf = b.readEntry(1L, 100L);
        Assert.assertEquals((String)"Ledger Id is wrong", (long)buf.readLong(), (long)1L);
        Assert.assertEquals((String)"Entry Id is wrong", (long)buf.readLong(), (long)100L);
        Assert.assertEquals((String)"Last confirmed is wrong", (long)buf.readLong(), (long)99L);
        Assert.assertEquals((String)"Length is wrong", (long)buf.readLong(), (long)102400L);
        buf.readLong();
        boolean allX = true;
        for (int i = 0; i < 1024; ++i) {
            byte x = buf.readByte();
            allX = allX && x == 88;
        }
        Assert.assertFalse((String)"Some of buffer should have been zeroed", (boolean)allX);
        try {
            b.readEntry(1L, 101L);
            Assert.fail((String)"Shouldn't have found entry 101");
        }
        catch (Bookie.NoEntryException noEntryException) {
            // empty catch block
        }
    }

    private Bookie createBookieAndReadJournal(ServerConfiguration conf) throws Exception {
        Bookie b = new Bookie(conf);
        for (Journal journal : b.journals) {
            Journal.LastLogMark lastLogMark = journal.getLastLogMark().markLog();
            b.readJournal();
            Assert.assertTrue((journal.getLastLogMark().getCurMark().compare(lastLogMark.getCurMark()) > 0 ? 1 : 0) != 0);
        }
        return b;
    }

    @Test
    public void testSortedLedgerStorageReplayWithSmallMaxArenaSize() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        JournalChannel jc = this.writeV2Journal(Bookie.getCurrentDirectory((File)journalDir), 100);
        jc.fc.force(false);
        this.writeIndexFileForLedger(Bookie.getCurrentDirectory((File)ledgerDir), 1L, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerStorageClass("org.apache.bookkeeper.bookie.SortedLedgerStorage");
        conf.setSkipListArenaMaxAllocSize(0);
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()});
        Bookie b = new Bookie(conf);
        b.readJournal();
        b.ledgerStorage.flush();
        b.readEntry(1L, 80L);
        b.readEntry(1L, 99L);
    }

    @Test
    public void testPartialFileInfoPreV3Journal1() throws Exception {
        this.testPartialFileInfoPreV3Journal(true);
    }

    @Test
    public void testPartialFileInfoPreV3Journal2() throws Exception {
        this.testPartialFileInfoPreV3Journal(false);
    }

    private void testPartialFileInfoPreV3Journal(boolean truncateMasterKey) throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writePreV2Journal(Bookie.getCurrentDirectory((File)journalDir), 100);
        this.writePartialIndexFileForLedger(Bookie.getCurrentDirectory((File)ledgerDir), 1L, "testPasswd".getBytes(), truncateMasterKey);
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        if (truncateMasterKey) {
            try {
                Bookie b = new Bookie(conf);
                b.readJournal();
                Assert.fail((String)"Should not reach here!");
            }
            catch (IOException b) {}
        } else {
            Bookie b = new Bookie(conf);
            b.readJournal();
            b.readEntry(1L, 100L);
            try {
                b.readEntry(1L, 101L);
                Assert.fail((String)"Shouldn't have found entry 101");
            }
            catch (Bookie.NoEntryException noEntryException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testPartialFileInfoPostV3Journal1() throws Exception {
        this.testPartialFileInfoPostV3Journal(true);
    }

    @Test
    public void testPartialFileInfoPostV3Journal2() throws Exception {
        this.testPartialFileInfoPostV3Journal(false);
    }

    private void testPartialFileInfoPostV3Journal(boolean truncateMasterKey) throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        byte[] masterKey = "testPasswd".getBytes();
        this.writeV3Journal(Bookie.getCurrentDirectory((File)journalDir), 100, masterKey);
        this.writePartialIndexFileForLedger(Bookie.getCurrentDirectory((File)ledgerDir), 1L, masterKey, truncateMasterKey);
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        Bookie b = new Bookie(conf);
        b.readJournal();
        b.readEntry(1L, 100L);
        try {
            b.readEntry(1L, 101L);
            Assert.fail((String)"Shouldn't have found entry 101");
        }
        catch (Bookie.NoEntryException noEntryException) {
            // empty catch block
        }
    }

    @Test
    public void testJournalScanIOException() throws Exception {
        File journalDir = this.createTempDir("bookie", "journal");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)journalDir));
        File ledgerDir = this.createTempDir("bookie", "ledger");
        Bookie.checkDirectoryStructure((File)Bookie.getCurrentDirectory((File)ledgerDir));
        this.writeV4Journal(Bookie.getCurrentDirectory((File)journalDir), 100, "testPasswd".getBytes());
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(journalDir.getPath()).setLedgerDirNames(new String[]{ledgerDir.getPath()}).setMetadataServiceUri(null);
        DummyJournalScan journalScanner = new DummyJournalScan();
        FileChannel fileChannel = (FileChannel)PowerMockito.mock(FileChannel.class);
        PowerMockito.when((Object)fileChannel.position(Mockito.anyLong())).thenThrow(new Throwable[]{new IOException()});
        PowerMockito.mockStatic(JournalChannel.class, (Class[])new Class[0]);
        PowerMockito.when((Object)JournalChannel.openFileChannel((RandomAccessFile)((RandomAccessFile)Mockito.any(RandomAccessFile.class)))).thenReturn((Object)fileChannel);
        Bookie b = new Bookie(conf);
        for (Journal journal : b.journals) {
            List journalIds = Journal.listJournalIds((File)journal.getJournalDirectory(), null);
            Assert.assertEquals((long)journalIds.size(), (long)1L);
            try {
                journal.scanJournal(((Long)journalIds.get(0)).longValue(), Long.MAX_VALUE, (Journal.JournalScanner)journalScanner);
                Assert.fail((String)"Should not have been able to scan the journal");
            }
            catch (Exception exception) {}
        }
        b.shutdown();
    }

    private class DummyJournalScan
    implements Journal.JournalScanner {
        private DummyJournalScan() {
        }

        public void process(int journalVersion, long offset, ByteBuffer entry) throws IOException {
            LOG.warn("Journal Version : " + journalVersion);
        }
    }
}

