/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.util.concurrent.MoreExecutors;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.UnpooledByteBufAllocator;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.stream.IntStream;
import org.apache.bookkeeper.bookie.BookieImpl;
import org.apache.bookkeeper.bookie.DefaultEntryLogger;
import org.apache.bookkeeper.bookie.EntryLogManagerBase;
import org.apache.bookkeeper.bookie.EntryLogManagerForEntryLogPerLedger;
import org.apache.bookkeeper.bookie.EntryLogManagerForSingleEntryLog;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.EntryLoggerAllocator;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.conf.TestBKConfiguration;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.test.TestStatsProvider;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.commons.lang3.mutable.MutableInt;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CreateNewLogTest {
    private static final Logger LOG = LoggerFactory.getLogger(CreateNewLogTest.class);
    private String[] ledgerDirs;
    private int numDirs = 100;

    @Before
    public void setUp() throws Exception {
        this.ledgerDirs = new String[this.numDirs];
        for (int i = 0; i < this.numDirs; ++i) {
            File temp = File.createTempFile("bookie", "test");
            temp.delete();
            temp.mkdir();
            File currentTemp = new File(temp.getAbsoluteFile() + "/current");
            currentTemp.mkdir();
            this.ledgerDirs[i] = temp.getPath();
        }
    }

    @After
    public void tearDown() throws Exception {
        for (int i = 0; i < this.numDirs; ++i) {
            File f = new File(this.ledgerDirs[i]);
            this.deleteRecursive(f);
        }
    }

    private void deleteRecursive(File f) {
        if (f.isDirectory()) {
            for (File c : f.listFiles()) {
                this.deleteRecursive(c);
            }
        }
        f.delete();
    }

    @Test
    public void testCreateNewLog() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        String logFileName = Long.toHexString(1L) + ".log";
        File dir = ledgerDirsManager.pickRandomWritableDir();
        LOG.info("Picked this directory: {}", (Object)dir);
        File newLogFile = new File(dir, logFileName);
        newLogFile.createNewFile();
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForSingleEntryLog entryLogManager = (EntryLogManagerForSingleEntryLog)el.getEntryLogManager();
        entryLogManager.createNewLog(0L);
        LOG.info("This is the current log id: {}", (Object)entryLogManager.getCurrentLogId());
        Assert.assertTrue((String)"Wrong log id", (entryLogManager.getCurrentLogId() > 1L ? 1 : 0) != 0);
    }

    @Test
    public void testCreateNewLogWithNoWritableLedgerDirs() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setIsForceGCAllowWhenNoSpace(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        String logFileName = Long.toHexString(1L) + ".log";
        File dir = ledgerDirsManager.pickRandomWritableDir();
        LOG.info("Picked this directory: {}", (Object)dir);
        File newLogFile = new File(dir, logFileName);
        newLogFile.createNewFile();
        List wDirs = ledgerDirsManager.getWritableLedgerDirs();
        for (File tdir : wDirs) {
            ledgerDirsManager.addToFilledDirs(tdir);
        }
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForSingleEntryLog entryLogManager = (EntryLogManagerForSingleEntryLog)el.getEntryLogManager();
        entryLogManager.createNewLog(0L);
        LOG.info("This is the current log id: {}", (Object)entryLogManager.getCurrentLogId());
        Assert.assertTrue((String)"Wrong log id", (entryLogManager.getCurrentLogId() > 1L ? 1 : 0) != 0);
    }

    void setSameThreadExecutorForEntryLoggerAllocator(EntryLoggerAllocator entryLoggerAllocator) {
        ExecutorService executorService = entryLoggerAllocator.allocatorExecutor;
        executorService.shutdown();
        entryLoggerAllocator.allocatorExecutor = MoreExecutors.newDirectExecutorService();
    }

    @Test
    public void testEntryLogPerLedgerCreationWithPreAllocation() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setIsForceGCAllowWhenNoSpace(true);
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLoggerAllocator entryLoggerAllocator = entryLogger.entryLoggerAllocator;
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        this.setSameThreadExecutorForEntryLoggerAllocator(entryLoggerAllocator);
        int expectedPreAllocatedLogID = -1;
        Assert.assertEquals((String)"PreallocatedlogId after initialization of Entrylogger", (long)expectedPreAllocatedLogID, (long)entryLoggerAllocator.getPreallocatedLogId());
        int numOfLedgers = 6;
        for (long i = 0L; i < (long)numOfLedgers; ++i) {
            entryLogManager.createNewLog(i);
        }
        expectedPreAllocatedLogID = numOfLedgers;
        Assert.assertEquals((String)"PreallocatedlogId after creation of logs for ledgers", (long)expectedPreAllocatedLogID, (long)entryLoggerAllocator.getPreallocatedLogId());
        Assert.assertEquals((String)"Number of current ", (long)numOfLedgers, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of LogChannels to flush", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
        String logFileName = Long.toHexString(expectedPreAllocatedLogID + 1) + ".log";
        File dir = ledgerDirsManager.pickRandomWritableDir();
        LOG.info("Picked this directory: " + dir);
        File newLogFile = new File(dir, logFileName);
        newLogFile.createNewFile();
        long rotatedLedger = 1L;
        entryLogManager.createNewLog(rotatedLedger);
        Assert.assertEquals((String)"PreallocatedlogId ", (long)(expectedPreAllocatedLogID += 2), (long)entryLoggerAllocator.getPreallocatedLogId());
        Assert.assertEquals((String)"Number of current ", (long)numOfLedgers, (long)entryLogManager.getCopyOfCurrentLogs().size());
        List rotatedLogChannels = entryLogManager.getRotatedLogChannels();
        Assert.assertEquals((String)"Number of LogChannels rotated", (long)1L, (long)rotatedLogChannels.size());
        Assert.assertEquals((String)"Rotated logchannel logid", (long)rotatedLedger, (long)((DefaultEntryLogger.BufferedLogChannel)rotatedLogChannels.iterator().next()).getLogId());
        entryLogger.flush();
        rotatedLogChannels = entryLogManager.getRotatedLogChannels();
        Assert.assertEquals((String)"Number of LogChannels rotated", (long)0L, (long)rotatedLogChannels.size());
        Assert.assertEquals((String)"Least UnflushedLoggerId", (long)0L, (long)entryLogger.getLeastUnflushedLogId());
        entryLogManager.createNewLog(0L);
        rotatedLogChannels = entryLogManager.getRotatedLogChannels();
        Assert.assertEquals((String)"Number of LogChannels rotated", (long)1L, (long)rotatedLogChannels.size());
        Assert.assertEquals((String)"Least UnflushedLoggerId", (long)0L, (long)entryLogger.getLeastUnflushedLogId());
        entryLogger.flush();
        Assert.assertEquals((String)"Least UnflushedLoggerId", (long)2L, (long)entryLogger.getLeastUnflushedLogId());
        ++expectedPreAllocatedLogID;
        for (int i = 0; i <= expectedPreAllocatedLogID; ++i) {
            EntryLogMetadata meta = entryLogger.getEntryLogMetadata((long)i);
            Assert.assertTrue((String)"EntryLogMetadata should be empty", (boolean)meta.isEmpty());
            Assert.assertTrue((String)"EntryLog usage should be 0", (meta.getTotalSize() == 0L ? 1 : 0) != 0);
        }
    }

    @Test
    public void testEntryLogCreationWithFilledDirs() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setIsForceGCAllowWhenNoSpace(false);
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLoggerAllocator entryLoggerAllocator = entryLogger.entryLoggerAllocator;
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        this.setSameThreadExecutorForEntryLoggerAllocator(entryLoggerAllocator);
        int expectedPreAllocatedLogIDDuringInitialization = -1;
        Assert.assertEquals((String)"PreallocatedlogId after initialization of Entrylogger", (long)expectedPreAllocatedLogIDDuringInitialization, (long)entryLoggerAllocator.getPreallocatedLogId());
        Assert.assertEquals((String)"Preallocation Future of this slot should be null", null, (Object)entryLogger.entryLoggerAllocator.preallocation);
        long ledgerId = 0L;
        entryLogManager.createNewLog(ledgerId);
        Assert.assertEquals((String)"PreallocatedlogId after initialization of Entrylogger", (long)(expectedPreAllocatedLogIDDuringInitialization + 1), (long)entryLoggerAllocator.getPreallocatedLogId());
        for (int i = 0; i < this.numDirs - 1; ++i) {
            ledgerDirsManager.addToFilledDirs(BookieImpl.getCurrentDirectory((File)new File(this.ledgerDirs[i])));
        }
        File nonFilledLedgerDir = BookieImpl.getCurrentDirectory((File)new File(this.ledgerDirs[this.numDirs - 1]));
        entryLogManager.createNewLog(ledgerId);
        DefaultEntryLogger.BufferedLogChannel newLogChannel = entryLogManager.getCurrentLogForLedger(ledgerId);
        Assert.assertEquals((String)"Directory of newly created BufferedLogChannel file", (Object)nonFilledLedgerDir.getAbsolutePath(), (Object)newLogChannel.getLogFile().getParentFile().getAbsolutePath());
        ledgerDirsManager.addToFilledDirs(BookieImpl.getCurrentDirectory((File)new File(this.ledgerDirs[this.numDirs - 1])));
        entryLogManager.createNewLog(ledgerId);
    }

    @Test
    public void testLedgerDirsUniformityDuringCreation() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entrylogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        for (long i = 0L; i < (long)this.ledgerDirs.length; ++i) {
            entrylogManager.createNewLog(i);
        }
        int numberOfLedgersCreated = this.ledgerDirs.length;
        Assert.assertEquals((String)"Highest frequency of entrylogs per ledgerdir", (long)1L, (long)this.highestFrequencyOfEntryLogsPerLedgerDir(entrylogManager.getCopyOfCurrentLogs()));
        long newLedgerId = numberOfLedgersCreated;
        entrylogManager.createNewLog(newLedgerId);
        Assert.assertEquals((String)"Highest frequency of entrylogs per ledgerdir", (long)2L, (long)this.highestFrequencyOfEntryLogsPerLedgerDir(entrylogManager.getCopyOfCurrentLogs()));
        for (long i = (long)(++numberOfLedgersCreated); i < (long)(2 * this.ledgerDirs.length); ++i) {
            entrylogManager.createNewLog(i);
        }
        Assert.assertEquals((String)"Highest frequency of entrylogs per ledgerdir", (long)2L, (long)this.highestFrequencyOfEntryLogsPerLedgerDir(entrylogManager.getCopyOfCurrentLogs()));
    }

    int highestFrequencyOfEntryLogsPerLedgerDir(Set<EntryLogManagerForEntryLogPerLedger.BufferedLogChannelWithDirInfo> copyOfCurrentLogsWithDirInfo) {
        HashMap<File, org.apache.commons.lang.mutable.MutableInt> frequencyOfEntryLogsInLedgerDirs = new HashMap<File, org.apache.commons.lang.mutable.MutableInt>();
        for (EntryLogManagerForEntryLogPerLedger.BufferedLogChannelWithDirInfo logChannelWithDirInfo : copyOfCurrentLogsWithDirInfo) {
            File parentDir = logChannelWithDirInfo.getLogChannel().getLogFile().getParentFile();
            if (frequencyOfEntryLogsInLedgerDirs.containsKey(parentDir)) {
                ((org.apache.commons.lang.mutable.MutableInt)frequencyOfEntryLogsInLedgerDirs.get(parentDir)).increment();
                continue;
            }
            frequencyOfEntryLogsInLedgerDirs.put(parentDir, new org.apache.commons.lang.mutable.MutableInt(1));
        }
        int highestFreq = ((org.apache.commons.lang.mutable.MutableInt)frequencyOfEntryLogsInLedgerDirs.entrySet().stream().max(Map.Entry.comparingByValue()).get().getValue()).intValue();
        return highestFreq;
    }

    @Test
    public void testConcurrentCreateNewLogWithEntryLogFilePreAllocationEnabled() throws Exception {
        this.testConcurrentCreateNewLog(true);
    }

    @Test
    public void testConcurrentCreateNewLogWithEntryLogFilePreAllocationDisabled() throws Exception {
        this.testConcurrentCreateNewLog(false);
    }

    public void testConcurrentCreateNewLog(boolean entryLogFilePreAllocationEnabled) throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(entryLogFilePreAllocationEnabled);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManager = (EntryLogManagerBase)el.getEntryLogManager();
        this.setSameThreadExecutorForEntryLoggerAllocator(el.getEntryLoggerAllocator());
        Assert.assertEquals((String)"previousAllocatedEntryLogId after initialization", (long)-1L, (long)el.getPreviousAllocatedEntryLogId());
        Assert.assertEquals((String)"leastUnflushedLogId after initialization", (long)0L, (long)el.getLeastUnflushedLogId());
        int createNewLogNumOfTimes = 10;
        AtomicBoolean receivedException = new AtomicBoolean(false);
        IntStream.range(0, createNewLogNumOfTimes).parallel().forEach(i -> {
            try {
                entryLogManager.createNewLog((long)i);
            }
            catch (IOException e) {
                LOG.error("Received exception while creating newLog", (Throwable)e);
                receivedException.set(true);
            }
        });
        Assert.assertFalse((String)"There shouldn't be any exceptions while creating newlog", (boolean)receivedException.get());
        int expectedPreviousAllocatedEntryLogId = createNewLogNumOfTimes - 1;
        if (entryLogFilePreAllocationEnabled) {
            expectedPreviousAllocatedEntryLogId = createNewLogNumOfTimes;
        }
        Assert.assertEquals((String)("previousAllocatedEntryLogId after " + createNewLogNumOfTimes + " number of times createNewLog is called"), (long)expectedPreviousAllocatedEntryLogId, (long)el.getPreviousAllocatedEntryLogId());
        Assert.assertEquals((String)"Number of RotatedLogChannels", (long)(createNewLogNumOfTimes - 1), (long)entryLogManager.getRotatedLogChannels().size());
    }

    @Test
    public void testCreateNewLogWithGaps() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(false);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManagerBase = (EntryLogManagerBase)el.getEntryLogManager();
        entryLogManagerBase.createNewLog(0L);
        Assert.assertEquals((String)"previousAllocatedEntryLogId after initialization", (long)0L, (long)el.getPreviousAllocatedEntryLogId());
        String logFileName = Long.toHexString(1L) + ".log";
        File dir = ledgerDirsManager.pickRandomWritableDir();
        LOG.info("Picked this directory: {}", (Object)dir);
        File newLogFile = new File(dir, logFileName);
        newLogFile.createNewFile();
        entryLogManagerBase.createNewLog(0L);
        Assert.assertEquals((String)"previousAllocatedEntryLogId since entrylogid 1 is already taken", (long)2L, (long)el.getPreviousAllocatedEntryLogId());
        logFileName = Long.toHexString(3L) + ".log";
        dir = ledgerDirsManager.pickRandomWritableDir();
        LOG.info("Picked this directory: {}", (Object)dir);
        newLogFile = new File(dir, logFileName);
        newLogFile.createNewFile();
        entryLogManagerBase.createNewLog(0L);
        Assert.assertEquals((String)"previousAllocatedEntryLogId since entrylogid 3 is already taken", (long)4L, (long)el.getPreviousAllocatedEntryLogId());
    }

    @Test
    public void testCreateNewLogAndCompactionLog() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        this.setSameThreadExecutorForEntryLoggerAllocator(el.getEntryLoggerAllocator());
        AtomicBoolean receivedException = new AtomicBoolean(false);
        IntStream.range(0, 2).parallel().forEach(i -> {
            try {
                if (i % 2 == 0) {
                    ((EntryLogManagerBase)el.getEntryLogManager()).createNewLog((long)i);
                } else {
                    el.newCompactionLog((long)i);
                }
            }
            catch (IOException e) {
                LOG.error("Received exception while creating newLog", (Throwable)e);
                receivedException.set(true);
            }
        });
        Assert.assertFalse((String)"There shouldn't be any exceptions while creating newlog", (boolean)receivedException.get());
        Assert.assertEquals((String)"previousAllocatedEntryLogId after 2 times createNewLog is called", (long)2L, (long)el.getPreviousAllocatedEntryLogId());
    }

    @Test
    public void testLastIdCompatibleBetweenDefaultAndDirectEntryLogger() throws Exception {
        File dir;
        int i;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(false);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManagerBase = (EntryLogManagerBase)el.getEntryLogManager();
        for (i = 0; i < 10; ++i) {
            entryLogManagerBase.createNewLog((long)i);
        }
        Assert.assertEquals((long)9L, (long)el.getPreviousAllocatedEntryLogId());
        for (i = 0; i < ledgerDirsManager.getAllLedgerDirs().size() / 2; ++i) {
            dir = (File)ledgerDirsManager.getAllLedgerDirs().get(i);
            LOG.info("Picked this directory: {}", (Object)dir);
            el.getEntryLoggerAllocator().setLastLogId(dir, 3L);
        }
        el = new DefaultEntryLogger(conf, ledgerDirsManager);
        Assert.assertEquals((long)9L, (long)el.getPreviousAllocatedEntryLogId());
        for (i = 0; i < ledgerDirsManager.getAllLedgerDirs().size(); ++i) {
            dir = (File)ledgerDirsManager.getAllLedgerDirs().get(i);
            LOG.info("Picked this directory: {}", (Object)dir);
            el.getEntryLoggerAllocator().setLastLogId(dir, 3L);
        }
        el = new DefaultEntryLogger(conf, ledgerDirsManager);
        Assert.assertEquals((long)9L, (long)el.getPreviousAllocatedEntryLogId());
    }

    @Test
    public void testConcurrentEntryLogCreations() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entrylogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        int numOfLedgers = 10;
        int numOfThreadsForSameLedger = 10;
        AtomicInteger createdEntryLogs = new AtomicInteger(0);
        CountDownLatch startLatch = new CountDownLatch(1);
        CountDownLatch createdLatch = new CountDownLatch(numOfLedgers * numOfThreadsForSameLedger);
        for (long i = 0L; i < (long)numOfLedgers; ++i) {
            for (int j = 0; j < numOfThreadsForSameLedger; ++j) {
                long ledgerId = i;
                new Thread(() -> {
                    try {
                        startLatch.await();
                        entrylogManager.createNewLog(ledgerId);
                        createdEntryLogs.incrementAndGet();
                        Thread.sleep(2000L);
                    }
                    catch (IOException | InterruptedException e) {
                        LOG.error("Got exception while trying to createNewLog for Ledger: " + ledgerId, (Throwable)e);
                    }
                    finally {
                        createdLatch.countDown();
                    }
                }).start();
            }
        }
        startLatch.countDown();
        createdLatch.await(20L, TimeUnit.SECONDS);
        Assert.assertEquals((String)"Created EntryLogs", (long)(numOfLedgers * numOfThreadsForSameLedger), (long)createdEntryLogs.get());
        Assert.assertEquals((String)"Active currentlogs size", (long)numOfLedgers, (long)entrylogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Rotated entrylogs size", (long)((numOfThreadsForSameLedger - 1) * numOfLedgers), (long)entrylogManager.getRotatedLogChannels().size());
        Assert.assertEquals((String)"PreviousAllocatedEntryLogId", (long)(numOfLedgers * numOfThreadsForSameLedger), (long)entryLogger.getPreviousAllocatedEntryLogId());
    }

    @Test
    public void testEntryLogManagerMetrics() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        TestStatsProvider statsProvider = new TestStatsProvider();
        TestStatsProvider.TestStatsLogger statsLogger = statsProvider.getStatsLogger("entrylogger");
        int maximumNumberOfActiveEntryLogs = 3;
        int entryLogPerLedgerCounterLimitsMultFactor = 2;
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setMaximumNumberOfActiveEntryLogs(maximumNumberOfActiveEntryLogs);
        conf.setEntryLogPerLedgerCounterLimitsMultFactor(entryLogPerLedgerCounterLimitsMultFactor);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(conf, ledgerDirsManager, null, (StatsLogger)statsLogger, (ByteBufAllocator)UnpooledByteBufAllocator.DEFAULT);
        EntryLogManagerForEntryLogPerLedger entrylogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        this.setSameThreadExecutorForEntryLoggerAllocator(entryLogger.getEntryLoggerAllocator());
        Counter numOfWriteActiveLedgers = statsLogger.getCounter("NUM_OF_WRITE_ACTIVE_LEDGERS");
        Counter numOfWriteLedgersRemovedCacheExpiry = statsLogger.getCounter("NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY");
        Counter numOfWriteLedgersRemovedCacheMaxSize = statsLogger.getCounter("NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE");
        Counter numLedgersHavingMultipleEntrylogs = statsLogger.getCounter("NUM_LEDGERS_HAVING_MULTIPLE_ENTRYLOGS");
        TestStatsProvider.TestOpStatsLogger entryLogsPerLedger = (TestStatsProvider.TestOpStatsLogger)statsLogger.getOpStatsLogger("ENTRYLOGS_PER_LEDGER");
        Assert.assertEquals((String)"NUM_OF_WRITE_ACTIVE_LEDGERS", (long)0L, (long)numOfWriteActiveLedgers.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY", (long)0L, (long)numOfWriteLedgersRemovedCacheExpiry.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE", (long)0L, (long)numOfWriteLedgersRemovedCacheMaxSize.get().intValue());
        Assert.assertEquals((String)"NUM_LEDGERS_HAVING_MULTIPLE_ENTRYLOGS", (long)0L, (long)numLedgersHavingMultipleEntrylogs.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)0L, (long)entryLogsPerLedger.getSuccessCount());
        int numOfEntrylogsForLedger1 = 3;
        CreateNewLogTest.createNewLogs(entrylogManager, 1L, numOfEntrylogsForLedger1);
        int numOfEntrylogsForLedger2 = 2;
        CreateNewLogTest.createNewLogs(entrylogManager, 2L, numOfEntrylogsForLedger2);
        CreateNewLogTest.createNewLogs(entrylogManager, 3L, 1);
        Assert.assertEquals((String)"NUM_OF_WRITE_ACTIVE_LEDGERS", (long)3L, (long)numOfWriteActiveLedgers.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY", (long)0L, (long)numOfWriteLedgersRemovedCacheExpiry.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE", (long)0L, (long)numOfWriteLedgersRemovedCacheMaxSize.get().intValue());
        Assert.assertEquals((String)"NUM_LEDGERS_HAVING_MULTIPLE_ENTRYLOGS", (long)2L, (long)numLedgersHavingMultipleEntrylogs.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)0L, (long)entryLogsPerLedger.getSuccessCount());
        CreateNewLogTest.createNewLogs(entrylogManager, 4L, 1);
        Assert.assertEquals((String)"NUM_OF_WRITE_ACTIVE_LEDGERS", (long)maximumNumberOfActiveEntryLogs, (long)numOfWriteActiveLedgers.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE", (long)1L, (long)numOfWriteLedgersRemovedCacheMaxSize.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)0L, (long)entryLogsPerLedger.getSuccessCount());
        CreateNewLogTest.createNewLogs(entrylogManager, 5L, 1);
        CreateNewLogTest.createNewLogs(entrylogManager, 6L, 1);
        CreateNewLogTest.createNewLogs(entrylogManager, 7L, 1);
        Assert.assertEquals((String)"NUM_OF_WRITE_ACTIVE_LEDGERS", (long)maximumNumberOfActiveEntryLogs, (long)numOfWriteActiveLedgers.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE", (long)4L, (long)numOfWriteLedgersRemovedCacheMaxSize.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)1L, (long)entryLogsPerLedger.getSuccessCount());
        Assert.assertTrue((String)"ENTRYLOGS_PER_LEDGER average value", (Double.compare(numOfEntrylogsForLedger1, entryLogsPerLedger.getSuccessAverage()) == 0 ? 1 : 0) != 0);
        CreateNewLogTest.createNewLogs(entrylogManager, 8L, 4);
        Assert.assertEquals((String)"NUM_OF_WRITE_ACTIVE_LEDGERS", (long)maximumNumberOfActiveEntryLogs, (long)numOfWriteActiveLedgers.get().intValue());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE", (long)5L, (long)numOfWriteLedgersRemovedCacheMaxSize.get().intValue());
        Assert.assertEquals((String)"NUM_LEDGERS_HAVING_MULTIPLE_ENTRYLOGS", (long)3L, (long)numLedgersHavingMultipleEntrylogs.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)2L, (long)entryLogsPerLedger.getSuccessCount());
        Assert.assertTrue((String)"ENTRYLOGS_PER_LEDGER average value", (Double.compare((double)(numOfEntrylogsForLedger1 + numOfEntrylogsForLedger2) / 2.0, entryLogsPerLedger.getSuccessAverage()) == 0 ? 1 : 0) != 0);
        CreateNewLogTest.createNewLogs(entrylogManager, 3L, 4);
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_MAXSIZE", (long)6L, (long)numOfWriteLedgersRemovedCacheMaxSize.get().intValue());
        Assert.assertEquals((String)"NUM_LEDGERS_HAVING_MULTIPLE_ENTRYLOGS", (long)4L, (long)numLedgersHavingMultipleEntrylogs.get().intValue());
        Assert.assertEquals((String)"Numofentrylogs for ledger: 3l", (long)5L, (long)((MutableInt)entrylogManager.entryLogsPerLedgerCounter.getCounterMap().get(3L)).intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)2L, (long)entryLogsPerLedger.getSuccessCount());
    }

    @Test
    public void testEntryLogManagerMetricsFromExpiryAspect() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        TestStatsProvider statsProvider = new TestStatsProvider();
        TestStatsProvider.TestStatsLogger statsLogger = statsProvider.getStatsLogger("entrylogger");
        int entrylogMapAccessExpiryTimeInSeconds = 1;
        int entryLogPerLedgerCounterLimitsMultFactor = 2;
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setEntrylogMapAccessExpiryTimeInSeconds(entrylogMapAccessExpiryTimeInSeconds);
        conf.setEntryLogPerLedgerCounterLimitsMultFactor(entryLogPerLedgerCounterLimitsMultFactor);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(conf, ledgerDirsManager, null, (StatsLogger)statsLogger, (ByteBufAllocator)UnpooledByteBufAllocator.DEFAULT);
        EntryLogManagerForEntryLogPerLedger entrylogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        this.setSameThreadExecutorForEntryLoggerAllocator(entryLogger.getEntryLoggerAllocator());
        Counter numOfWriteLedgersRemovedCacheExpiry = statsLogger.getCounter("NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY");
        TestStatsProvider.TestOpStatsLogger entryLogsPerLedger = (TestStatsProvider.TestOpStatsLogger)statsLogger.getOpStatsLogger("ENTRYLOGS_PER_LEDGER");
        int numOfEntrylogsForLedger1 = 3;
        CreateNewLogTest.createNewLogs(entrylogManager, 1L, numOfEntrylogsForLedger1);
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)0L, (long)entryLogsPerLedger.getSuccessCount());
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY", (long)0L, (long)numOfWriteLedgersRemovedCacheExpiry.get().intValue());
        Thread.sleep(entrylogMapAccessExpiryTimeInSeconds * 1000 + 100);
        entrylogManager.doEntryLogMapCleanup();
        entrylogManager.entryLogsPerLedgerCounter.doCounterMapCleanup();
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY", (long)1L, (long)numOfWriteLedgersRemovedCacheExpiry.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)0L, (long)entryLogsPerLedger.getSuccessCount());
        Thread.sleep(entrylogMapAccessExpiryTimeInSeconds * 1000 + 100);
        entrylogManager.doEntryLogMapCleanup();
        entrylogManager.entryLogsPerLedgerCounter.doCounterMapCleanup();
        Assert.assertEquals((String)"NUM_OF_WRITE_LEDGERS_REMOVED_CACHE_EXPIRY", (long)1L, (long)numOfWriteLedgersRemovedCacheExpiry.get().intValue());
        Assert.assertEquals((String)"ENTRYLOGS_PER_LEDGER SuccessCount", (long)1L, (long)entryLogsPerLedger.getSuccessCount());
        Assert.assertTrue((String)"ENTRYLOGS_PER_LEDGER average value", (Double.compare(numOfEntrylogsForLedger1, entryLogsPerLedger.getSuccessAverage()) == 0 ? 1 : 0) != 0);
    }

    private static void createNewLogs(EntryLogManagerForEntryLogPerLedger entrylogManager, long ledgerId, int numOfTimes) throws IOException {
        for (int i = 0; i < numOfTimes; ++i) {
            entrylogManager.createNewLog(ledgerId);
        }
    }

    @Test
    public void testLockConsistency() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.ledgerDirs);
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setMaximumNumberOfActiveEntryLogs(5);
        final CountDownLatch latch = new CountDownLatch(1);
        final AtomicInteger count = new AtomicInteger(0);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold())){

            public List<File> getWritableLedgerDirsForNewLog() throws LedgerDirsManager.NoWritableLedgerDirException {
                if (count.incrementAndGet() == 1) {
                    try {
                        latch.await();
                    }
                    catch (InterruptedException e) {
                        LOG.error("Got InterruptedException while awaiting for latch countdown", (Throwable)e);
                    }
                }
                return super.getWritableLedgerDirsForNewLog();
            }
        };
        DefaultEntryLogger el = new DefaultEntryLogger(conf, ledgerDirsManager);
        final EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)el.getEntryLogManager();
        final long firstLedgerId = 100L;
        final AtomicBoolean newLogCreated = new AtomicBoolean(false);
        Assert.assertFalse((String)"EntryLogManager cacheMap should not contain entry for firstLedgerId", (boolean)entryLogManager.getCacheAsMap().containsKey(firstLedgerId));
        Assert.assertEquals((String)"Value of the count should be 0", (long)0L, (long)count.get());
        new Thread(){

            @Override
            public void run() {
                try {
                    entryLogManager.createNewLog(firstLedgerId);
                    newLogCreated.set(true);
                }
                catch (IOException e) {
                    LOG.error("Got IOException while creating new log", (Throwable)e);
                }
            }
        }.start();
        while (!entryLogManager.getCacheAsMap().containsKey(firstLedgerId)) {
            Thread.sleep(200L);
        }
        Lock firstLedgersLock = entryLogManager.getLock(firstLedgerId);
        Thread.sleep(2000L);
        Assert.assertFalse((String)"New log shouldn't have created", (boolean)newLogCreated.get());
        for (int i = 1; i <= conf.getMaximumNumberOfActiveEntryLogs(); ++i) {
            entryLogManager.createNewLog(firstLedgerId + (long)i);
        }
        entryLogManager.doEntryLogMapCleanup();
        Assert.assertFalse((String)"Entry for that ledger shouldn't be there", (boolean)entryLogManager.getCacheAsMap().containsKey(firstLedgerId));
        latch.countDown();
        while (!newLogCreated.get()) {
            Thread.sleep(200L);
        }
        while (entryLogManager.getRotatedLogChannels().size() < 1) {
            Thread.sleep(200L);
        }
        Lock lockForThatLedgerAfterRemoval = entryLogManager.getLock(firstLedgerId);
        Assert.assertEquals((String)"For a given ledger lock should be the same before and after removal", (Object)firstLedgersLock, (Object)lockForThatLedgerAfterRemoval);
    }
}

