/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import bk-shade.com.google.common.util.concurrent.RateLimiter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.bookie.GarbageCollector;
import org.apache.bookkeeper.bookie.LedgerCache;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.ScanAndCompareGarbageCollector;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.SnapshotMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GarbageCollectorThread
extends Thread {
    private static final Logger LOG = LoggerFactory.getLogger(GarbageCollectorThread.class);
    private static final int SECOND = 1000;
    private Map<Long, EntryLogMetadata> entryLogMetaMap = new ConcurrentHashMap<Long, EntryLogMetadata>();
    final long gcWaitTime;
    boolean enableMinorCompaction = false;
    final double minorCompactionThreshold;
    final long minorCompactionInterval;
    boolean enableMajorCompaction = false;
    final double majorCompactionThreshold;
    final long majorCompactionInterval;
    long lastMinorCompactionTime;
    long lastMajorCompactionTime;
    final int maxOutstandingRequests;
    final int compactionRate;
    final CompactionScannerFactory scannerFactory;
    final EntryLogger entryLogger;
    final LedgerCache ledgerCache;
    final SnapshotMap<Long, Boolean> activeLedgers;
    final AtomicBoolean compacting = new AtomicBoolean(false);
    volatile boolean running = true;
    long scannedLogId = 0L;
    final GarbageCollector garbageCollector;
    final GarbageCollector.GarbageCleaner garbageCleaner;

    public GarbageCollectorThread(ServerConfiguration conf, final LedgerCache ledgerCache, EntryLogger entryLogger, SnapshotMap<Long, Boolean> activeLedgers, LedgerManager ledgerManager) throws IOException {
        super("GarbageCollectorThread");
        this.ledgerCache = ledgerCache;
        this.entryLogger = entryLogger;
        this.activeLedgers = activeLedgers;
        this.gcWaitTime = conf.getGcWaitTime();
        this.maxOutstandingRequests = conf.getCompactionMaxOutstandingRequests();
        this.compactionRate = conf.getCompactionRate();
        this.scannerFactory = new CompactionScannerFactory();
        entryLogger.addListener(this.scannerFactory);
        this.garbageCleaner = new GarbageCollector.GarbageCleaner(){

            @Override
            public void clean(long ledgerId) {
                try {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("delete ledger : " + ledgerId);
                    }
                    ledgerCache.deleteLedger(ledgerId);
                }
                catch (IOException e) {
                    LOG.error("Exception when deleting the ledger index file on the Bookie: ", (Throwable)e);
                }
            }
        };
        this.garbageCollector = new ScanAndCompareGarbageCollector(ledgerManager, activeLedgers);
        this.minorCompactionThreshold = conf.getMinorCompactionThreshold();
        this.minorCompactionInterval = conf.getMinorCompactionInterval() * 1000L;
        this.majorCompactionThreshold = conf.getMajorCompactionThreshold();
        this.majorCompactionInterval = conf.getMajorCompactionInterval() * 1000L;
        if (this.minorCompactionInterval > 0L && this.minorCompactionThreshold > 0.0) {
            if (this.minorCompactionThreshold > 1.0) {
                throw new IOException("Invalid minor compaction threshold " + this.minorCompactionThreshold);
            }
            if (this.minorCompactionInterval <= this.gcWaitTime) {
                throw new IOException("Too short minor compaction interval : " + this.minorCompactionInterval);
            }
            this.enableMinorCompaction = true;
        }
        if (this.majorCompactionInterval > 0L && this.majorCompactionThreshold > 0.0) {
            if (this.majorCompactionThreshold > 1.0) {
                throw new IOException("Invalid major compaction threshold " + this.majorCompactionThreshold);
            }
            if (this.majorCompactionInterval <= this.gcWaitTime) {
                throw new IOException("Too short major compaction interval : " + this.majorCompactionInterval);
            }
            this.enableMajorCompaction = true;
        }
        if (this.enableMinorCompaction && this.enableMajorCompaction && (this.minorCompactionInterval >= this.majorCompactionInterval || this.minorCompactionThreshold >= this.majorCompactionThreshold)) {
            throw new IOException("Invalid minor/major compaction settings : minor (" + this.minorCompactionThreshold + ", " + this.minorCompactionInterval + "), major (" + this.majorCompactionThreshold + ", " + this.majorCompactionInterval + ")");
        }
        LOG.info("Minor Compaction : enabled=" + this.enableMinorCompaction + ", threshold=" + this.minorCompactionThreshold + ", interval=" + this.minorCompactionInterval);
        LOG.info("Major Compaction : enabled=" + this.enableMajorCompaction + ", threshold=" + this.majorCompactionThreshold + ", interval=" + this.majorCompactionInterval);
        this.lastMinorCompactionTime = this.lastMajorCompactionTime = MathUtils.now();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (this.running) {
            GarbageCollectorThread garbageCollectorThread = this;
            synchronized (garbageCollectorThread) {
                try {
                    this.wait(this.gcWaitTime);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    continue;
                }
            }
            this.entryLogMetaMap = this.extractMetaFromEntryLogs(this.entryLogMetaMap);
            this.doGcLedgers();
            this.doGcEntryLogs();
            long curTime = MathUtils.now();
            if (this.enableMajorCompaction && curTime - this.lastMajorCompactionTime > this.majorCompactionInterval) {
                LOG.info("Enter major compaction");
                this.doCompactEntryLogs(this.majorCompactionThreshold);
                this.lastMinorCompactionTime = this.lastMajorCompactionTime = MathUtils.now();
                continue;
            }
            if (!this.enableMinorCompaction || curTime - this.lastMinorCompactionTime <= this.minorCompactionInterval) continue;
            LOG.info("Enter minor compaction");
            this.doCompactEntryLogs(this.minorCompactionThreshold);
            this.lastMinorCompactionTime = MathUtils.now();
        }
    }

    private void doGcLedgers() {
        this.garbageCollector.gc(this.garbageCleaner);
    }

    private void doGcEntryLogs() {
        for (Long entryLogId : this.entryLogMetaMap.keySet()) {
            EntryLogMetadata meta = this.entryLogMetaMap.get(entryLogId);
            for (Long entryLogLedger : meta.ledgersMap.keySet()) {
                if (this.activeLedgers.containsKey(entryLogLedger)) continue;
                meta.removeLedger(entryLogLedger);
            }
            if (!meta.isEmpty()) continue;
            LOG.info("Deleting entryLogId " + entryLogId + " as it has no active ledgers!");
            this.removeEntryLog(entryLogId);
        }
    }

    private void doCompactEntryLogs(double threshold) {
        EntryLogMetadata meta;
        LOG.info("Do compaction to compact those files lower than " + threshold);
        Comparator<EntryLogMetadata> sizeComparator = new Comparator<EntryLogMetadata>(){

            @Override
            public int compare(EntryLogMetadata m1, EntryLogMetadata m2) {
                long unusedSize1 = m1.totalSize - m1.remainingSize;
                long unusedSize2 = m2.totalSize - m2.remainingSize;
                if (unusedSize1 > unusedSize2) {
                    return -1;
                }
                if (unusedSize1 < unusedSize2) {
                    return 1;
                }
                return 0;
            }
        };
        ArrayList<EntryLogMetadata> logsToCompact = new ArrayList<EntryLogMetadata>();
        logsToCompact.addAll(this.entryLogMetaMap.values());
        Collections.sort(logsToCompact, sizeComparator);
        ArrayList<Long> toRemove = new ArrayList<Long>();
        Iterator i$ = logsToCompact.iterator();
        while (i$.hasNext() && !((meta = (EntryLogMetadata)i$.next()).getUsage() >= threshold)) {
            LOG.debug("Compacting entry log {} below threshold {}.", (Object)meta.entryLogId, (Object)threshold);
            try {
                this.compactEntryLog(this.scannerFactory, meta);
                toRemove.add(meta.entryLogId);
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
                LOG.warn("No writable ledger directory available, aborting compaction", (Throwable)nwlde);
                break;
            }
            catch (IOException ioe) {
                LOG.error("Error compacting entry log. Log won't be deleted", (Throwable)ioe);
            }
            if (this.running) continue;
            return;
        }
        try {
            this.scannerFactory.flush();
        }
        catch (IOException ioe) {
            LOG.error("Cannot flush compacted entries, skip removal", (Throwable)ioe);
            return;
        }
        for (Long l : toRemove) {
            this.removeEntryLog(l);
        }
    }

    public void shutdown() throws InterruptedException {
        this.running = false;
        if (this.compacting.compareAndSet(false, true)) {
            this.interrupt();
        }
        this.join();
    }

    private void removeEntryLog(long entryLogId) {
        if (this.entryLogger.removeEntryLog(entryLogId)) {
            this.entryLogMetaMap.remove(entryLogId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void compactEntryLog(CompactionScannerFactory scannerFactory, EntryLogMetadata entryLogMeta) throws IOException {
        if (!this.compacting.compareAndSet(false, true)) {
            return;
        }
        LOG.info("Compacting entry log : {}", (Object)entryLogMeta.entryLogId);
        try {
            this.entryLogger.scanEntryLog(entryLogMeta.entryLogId, scannerFactory.newScanner(entryLogMeta));
        }
        finally {
            this.compacting.set(false);
        }
    }

    protected Map<Long, EntryLogMetadata> extractMetaFromEntryLogs(Map<Long, EntryLogMetadata> entryLogMetaMap) {
        long curLogId = this.entryLogger.getCurrentLogId();
        boolean hasExceptionWhenScan = false;
        for (long entryLogId = this.scannedLogId; entryLogId < curLogId; ++entryLogId) {
            if (entryLogMetaMap.containsKey(entryLogId) || !this.entryLogger.logExists(entryLogId)) continue;
            LOG.info("Extracting entry log meta from entryLogId: {}", (Object)entryLogId);
            try {
                EntryLogMetadata entryLogMeta = GarbageCollectorThread.extractMetaFromEntryLog(this.entryLogger, entryLogId);
                entryLogMetaMap.put(entryLogId, entryLogMeta);
            }
            catch (IOException e) {
                hasExceptionWhenScan = true;
                LOG.warn("Premature exception when processing " + entryLogId + " recovery will take care of the problem", (Throwable)e);
            }
            if (hasExceptionWhenScan) continue;
            ++this.scannedLogId;
        }
        return entryLogMetaMap;
    }

    static EntryLogMetadata extractMetaFromEntryLog(EntryLogger entryLogger, long entryLogId) throws IOException {
        EntryLogMetadata entryLogMeta = new EntryLogMetadata(entryLogId);
        ExtractionScanner scanner = new ExtractionScanner(entryLogMeta);
        entryLogger.scanEntryLog(entryLogId, scanner);
        LOG.debug("Retrieved entry log meta data entryLogId: {}, meta: {}", (Object)entryLogId, (Object)entryLogMeta);
        return entryLogMeta;
    }

    static class ExtractionScanner
    implements EntryLogger.EntryLogScanner {
        EntryLogMetadata meta;

        public ExtractionScanner(EntryLogMetadata meta) {
            this.meta = meta;
        }

        @Override
        public boolean accept(long ledgerId) {
            return true;
        }

        @Override
        public void process(long ledgerId, long offset, ByteBuffer entry) {
            this.meta.addLedgerSize(ledgerId, entry.limit() + 4);
        }
    }

    static class EntryLogMetadata {
        long entryLogId;
        long totalSize;
        long remainingSize;
        ConcurrentHashMap<Long, Long> ledgersMap;

        public EntryLogMetadata(long logId) {
            this.entryLogId = logId;
            this.remainingSize = 0L;
            this.totalSize = 0L;
            this.ledgersMap = new ConcurrentHashMap();
        }

        public void addLedgerSize(long ledgerId, long size) {
            this.totalSize += size;
            this.remainingSize += size;
            Long ledgerSize = this.ledgersMap.get(ledgerId);
            if (null == ledgerSize) {
                ledgerSize = 0L;
            }
            ledgerSize = ledgerSize + size;
            this.ledgersMap.put(ledgerId, ledgerSize);
        }

        public void removeLedger(long ledgerId) {
            Long size = this.ledgersMap.remove(ledgerId);
            if (null == size) {
                return;
            }
            this.remainingSize -= size.longValue();
        }

        public boolean containsLedger(long ledgerId) {
            return this.ledgersMap.containsKey(ledgerId);
        }

        public double getUsage() {
            if (this.totalSize == 0L) {
                return 0.0;
            }
            return (double)this.remainingSize / (double)this.totalSize;
        }

        public boolean isEmpty() {
            return this.ledgersMap.isEmpty();
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("{ totalSize = ").append(this.totalSize).append(", remainingSize = ").append(this.remainingSize).append(", ledgersMap = ").append(this.ledgersMap).append(" }");
            return sb.toString();
        }
    }

    class CompactionScannerFactory
    implements EntryLogger.EntryLogListener {
        List<Offset> offsets = new ArrayList<Offset>();
        AtomicBoolean flushed = new AtomicBoolean(false);
        Object flushLock = new Object();

        CompactionScannerFactory() {
        }

        EntryLogger.EntryLogScanner newScanner(final EntryLogMetadata meta) {
            final RateLimiter rateLimiter = RateLimiter.create(GarbageCollectorThread.this.compactionRate);
            return new EntryLogger.EntryLogScanner(){

                @Override
                public boolean accept(long ledgerId) {
                    return meta.containsLedger(ledgerId);
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void process(long ledgerId, long offset, ByteBuffer entry) throws IOException {
                    rateLimiter.acquire();
                    CompactionScannerFactory compactionScannerFactory = CompactionScannerFactory.this;
                    synchronized (compactionScannerFactory) {
                        if (CompactionScannerFactory.this.offsets.size() > GarbageCollectorThread.this.maxOutstandingRequests) {
                            CompactionScannerFactory.this.waitEntrylogFlushed();
                        }
                        entry.getLong();
                        long entryId = entry.getLong();
                        entry.rewind();
                        long newoffset = GarbageCollectorThread.this.entryLogger.addEntry(ledgerId, entry);
                        CompactionScannerFactory.this.flushed.set(false);
                        CompactionScannerFactory.this.offsets.add(new Offset(ledgerId, entryId, newoffset));
                    }
                }
            };
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onEntryLogFlushed() {
            Object object = this.flushLock;
            synchronized (object) {
                this.flushed.set(true);
                this.flushLock.notifyAll();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private synchronized void waitEntrylogFlushed() throws IOException {
            try {
                Object object = this.flushLock;
                synchronized (object) {
                    while (!this.flushed.get() && GarbageCollectorThread.this.entryLogger.isFlushRequired() && GarbageCollectorThread.this.running) {
                        this.flushLock.wait(1000L);
                    }
                    if (!this.flushed.get() && GarbageCollectorThread.this.entryLogger.isFlushRequired() && !GarbageCollectorThread.this.running) {
                        throw new IOException("Shutdown before flushed");
                    }
                }
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                throw new IOException("Interrupted waiting for flush", ie);
            }
            for (Offset o : this.offsets) {
                GarbageCollectorThread.this.ledgerCache.putEntryOffset(o.ledger, o.entry, o.offset);
            }
            this.offsets.clear();
        }

        synchronized void flush() throws IOException {
            this.waitEntrylogFlushed();
            GarbageCollectorThread.this.ledgerCache.flushLedger(true);
        }
    }

    private static class Offset {
        final long ledger;
        final long entry;
        final long offset;

        Offset(long ledger, long entry, long offset) {
            this.ledger = ledger;
            this.entry = entry;
            this.offset = offset;
        }
    }
}

