/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.AbstractFuture;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.lang.invoke.LambdaMetafactory;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Formatter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.Cookie;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.bookie.IndexPersistenceMgr;
import org.apache.bookkeeper.bookie.Journal;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.LedgerEntryPage;
import org.apache.bookkeeper.bookie.LocalBookieEnsemblePlacementPolicy;
import org.apache.bookkeeper.bookie.LogMark;
import org.apache.bookkeeper.bookie.ReadOnlyEntryLogger;
import org.apache.bookkeeper.bookie.ReadOnlyFileInfo;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.BookKeeperAdmin;
import org.apache.bookkeeper.client.BookieInfoReader;
import org.apache.bookkeeper.client.LedgerEntry;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.client.LedgerMetadata;
import org.apache.bookkeeper.client.UpdateLedgerOp;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.discover.RegistrationManager;
import org.apache.bookkeeper.discover.ZKRegistrationManager;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bookkeeper.meta.LedgerUnderreplicationManager;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.replication.AuditorElector;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.bookkeeper.util.EntryFormatter;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.Tool;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.MissingArgumentException;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.configuration.CompositeConfiguration;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.mutable.MutableBoolean;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BookieShell
implements Tool {
    static final Logger LOG = LoggerFactory.getLogger(BookieShell.class);
    static final String ENTRY_FORMATTER_CLASS = "entryFormatterClass";
    static final String CMD_METAFORMAT = "metaformat";
    static final String CMD_BOOKIEFORMAT = "bookieformat";
    static final String CMD_RECOVER = "recover";
    static final String CMD_LEDGER = "ledger";
    static final String CMD_READ_LEDGER_ENTRIES = "readledger";
    static final String CMD_LISTLEDGERS = "listledgers";
    static final String CMD_LEDGERMETADATA = "ledgermetadata";
    static final String CMD_LISTUNDERREPLICATED = "listunderreplicated";
    static final String CMD_WHOISAUDITOR = "whoisauditor";
    static final String CMD_SIMPLETEST = "simpletest";
    static final String CMD_BOOKIESANITYTEST = "bookiesanity";
    static final String CMD_READLOG = "readlog";
    static final String CMD_READJOURNAL = "readjournal";
    static final String CMD_LASTMARK = "lastmark";
    static final String CMD_AUTORECOVERY = "autorecovery";
    static final String CMD_LISTBOOKIES = "listbookies";
    static final String CMD_LISTFILESONDISC = "listfilesondisc";
    static final String CMD_UPDATECOOKIE = "updatecookie";
    static final String CMD_EXPANDSTORAGE = "expandstorage";
    static final String CMD_UPDATELEDGER = "updateledgers";
    static final String CMD_DELETELEDGER = "deleteledger";
    static final String CMD_BOOKIEINFO = "bookieinfo";
    static final String CMD_DECOMMISSIONBOOKIE = "decommissionbookie";
    static final String CMD_LOSTBOOKIERECOVERYDELAY = "lostbookierecoverydelay";
    static final String CMD_TRIGGERAUDIT = "triggeraudit";
    static final String CMD_HELP = "help";
    final ServerConfiguration bkConf = new ServerConfiguration();
    File[] indexDirectories;
    File[] ledgerDirectories;
    File[] journalDirectories;
    EntryLogger entryLogger = null;
    List<Journal> journals = null;
    EntryFormatter formatter;
    int pageSize;
    int entriesPerPage;
    static final int LIST_BATCH_SIZE = 1000;
    final Map<String, MyCommand> commands = new HashMap<String, MyCommand>();

    public BookieShell() {
        this.commands.put(CMD_METAFORMAT, new MetaFormatCmd());
        this.commands.put(CMD_BOOKIEFORMAT, new BookieFormatCmd());
        this.commands.put(CMD_RECOVER, new RecoverCmd());
        this.commands.put(CMD_LEDGER, new LedgerCmd());
        this.commands.put(CMD_READ_LEDGER_ENTRIES, new ReadLedgerEntriesCmd());
        this.commands.put(CMD_LISTLEDGERS, new ListLedgersCmd());
        this.commands.put(CMD_LISTUNDERREPLICATED, new ListUnderreplicatedCmd());
        this.commands.put(CMD_WHOISAUDITOR, new WhoIsAuditorCmd());
        this.commands.put(CMD_LEDGERMETADATA, new LedgerMetadataCmd());
        this.commands.put(CMD_SIMPLETEST, new SimpleTestCmd());
        this.commands.put(CMD_BOOKIESANITYTEST, new BookieSanityTestCmd());
        this.commands.put(CMD_READLOG, new ReadLogCmd());
        this.commands.put(CMD_READJOURNAL, new ReadJournalCmd());
        this.commands.put(CMD_LASTMARK, new LastMarkCmd());
        this.commands.put(CMD_AUTORECOVERY, new AutoRecoveryCmd());
        this.commands.put(CMD_LISTBOOKIES, new ListBookiesCmd());
        this.commands.put(CMD_LISTFILESONDISC, new ListDiskFilesCmd());
        this.commands.put(CMD_UPDATECOOKIE, new UpdateCookieCmd());
        this.commands.put(CMD_EXPANDSTORAGE, new ExpandStorageCmd());
        this.commands.put(CMD_UPDATELEDGER, new UpdateLedgerCmd());
        this.commands.put(CMD_DELETELEDGER, new DeleteLedgerCmd());
        this.commands.put(CMD_BOOKIEINFO, new BookieInfoCmd());
        this.commands.put(CMD_DECOMMISSIONBOOKIE, new DecommissionBookieCmd());
        this.commands.put(CMD_HELP, new HelpCmd());
        this.commands.put(CMD_LOSTBOOKIERECOVERYDELAY, new LostBookieRecoveryDelayCmd());
        this.commands.put(CMD_TRIGGERAUDIT, new TriggerAuditCmd());
    }

    static void printLedgerMetadata(ReadMetadataCallback cb) throws Exception {
        LedgerMetadata md = (LedgerMetadata)cb.get();
        System.out.println("ledgerID: " + cb.getLedgerId());
        System.out.println(new String(md.serialize(), Charsets.UTF_8));
    }

    @Override
    public void setConf(CompositeConfiguration conf) throws Exception {
        this.bkConf.loadConf(conf);
        this.journalDirectories = Bookie.getCurrentDirectories(this.bkConf.getJournalDirs());
        this.ledgerDirectories = Bookie.getCurrentDirectories(this.bkConf.getLedgerDirs());
        this.indexDirectories = null == this.bkConf.getIndexDirs() ? this.ledgerDirectories : Bookie.getCurrentDirectories(this.bkConf.getIndexDirs());
        this.formatter = EntryFormatter.newEntryFormatter((Configuration)this.bkConf, ENTRY_FORMATTER_CLASS);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Using entry formatter {}", (Object)this.formatter.getClass().getName());
        }
        this.pageSize = this.bkConf.getPageSize();
        this.entriesPerPage = this.pageSize / 8;
    }

    private void printShellUsage() {
        System.err.println("Usage: BookieShell [-conf configuration] <command>");
        System.err.println();
        ArrayList<String> commandNames = new ArrayList<String>();
        for (MyCommand c : this.commands.values()) {
            commandNames.add("       " + c.getUsage());
        }
        Collections.sort(commandNames);
        for (String s : commandNames) {
            System.err.println(s);
        }
    }

    @VisibleForTesting
    public int execute(String ... args) throws Exception {
        return this.run(args);
    }

    @Override
    public int run(String[] args) throws Exception {
        if (args.length <= 0) {
            this.printShellUsage();
            return -1;
        }
        String cmdName = args[0];
        Command cmd = this.commands.get(cmdName);
        if (null == cmd) {
            System.err.println("ERROR: Unknown command " + cmdName);
            this.printShellUsage();
            return -1;
        }
        String[] newArgs = new String[args.length - 1];
        System.arraycopy(args, 1, newArgs, 0, newArgs.length);
        return cmd.runCmd(newArgs);
    }

    public static List<File> listFilesAndSort(File[] folderNames, String ... extensions) {
        ArrayList<File> completeFilesList = new ArrayList<File>();
        for (int i = 0; i < folderNames.length; ++i) {
            Collection filesCollection = FileUtils.listFiles((File)folderNames[i], (String[])extensions, (boolean)true);
            completeFilesList.addAll(filesCollection);
        }
        Collections.sort(completeFilesList, new FilesTimeComparator());
        return completeFilesList;
    }

    public static void main(String[] argv) throws Exception {
        BookieShell shell = new BookieShell();
        if (argv.length <= 0) {
            shell.printShellUsage();
            System.exit(-1);
        }
        CompositeConfiguration conf = new CompositeConfiguration();
        if ("-conf".equals(argv[0])) {
            if (argv.length <= 1) {
                shell.printShellUsage();
                System.exit(-1);
            }
            conf.addConfiguration((Configuration)new PropertiesConfiguration(new File(argv[1]).toURI().toURL()));
            String[] newArgv = new String[argv.length - 2];
            System.arraycopy(argv, 2, newArgv, 0, newArgv.length);
            argv = newArgv;
        }
        shell.setConf(conf);
        int res = shell.run(argv);
        System.exit(res);
    }

    private File getLedgerFile(long ledgerId) {
        File d;
        String ledgerName = IndexPersistenceMgr.getLedgerName(ledgerId);
        File lf = null;
        File[] fileArray = this.indexDirectories;
        int n = fileArray.length;
        for (int i = 0; i < n && !(lf = new File(d = fileArray[i], ledgerName)).exists(); ++i) {
            lf = null;
        }
        return lf;
    }

    ReadOnlyFileInfo getFileInfo(long ledgerId) throws IOException {
        File ledgerFile = this.getLedgerFile(ledgerId);
        if (null == ledgerFile) {
            throw new FileNotFoundException("No index file found for ledger " + ledgerId + ". It may be not flushed yet.");
        }
        ReadOnlyFileInfo fi = new ReadOnlyFileInfo(ledgerFile, null);
        fi.readHeader();
        return fi;
    }

    private synchronized void initEntryLogger() throws IOException {
        if (null == this.entryLogger) {
            this.entryLogger = new ReadOnlyEntryLogger(this.bkConf);
        }
    }

    protected void scanEntryLog(long logId, EntryLogger.EntryLogScanner scanner) throws IOException {
        this.initEntryLogger();
        this.entryLogger.scanEntryLog(logId, scanner);
    }

    private synchronized List<Journal> getJournals() throws IOException {
        if (null == this.journals) {
            this.journals = Lists.newArrayListWithCapacity((int)this.bkConf.getJournalDirs().length);
            for (File journalDir : this.bkConf.getJournalDirs()) {
                this.journals.add(new Journal(journalDir, this.bkConf, new LedgerDirsManager(this.bkConf, this.bkConf.getLedgerDirs(), new DiskChecker(this.bkConf.getDiskUsageThreshold(), this.bkConf.getDiskUsageWarnThreshold()))));
            }
        }
        return this.journals;
    }

    protected void scanJournal(Journal journal, long journalId, Journal.JournalScanner scanner) throws IOException {
        journal.scanJournal(journalId, 0L, scanner);
    }

    protected void readLedgerMeta(long ledgerId) throws Exception {
        System.out.println("===== LEDGER: " + ledgerId + " =====");
        ReadOnlyFileInfo fi = this.getFileInfo(ledgerId);
        byte[] masterKey = fi.getMasterKey();
        if (null == masterKey) {
            System.out.println("master key  : NULL");
        } else {
            System.out.println("master key  : " + BookieShell.bytes2Hex(fi.getMasterKey()));
        }
        long size = fi.size();
        if (size % 8L == 0L) {
            System.out.println("size        : " + size);
        } else {
            System.out.println("size : " + size + " (not aligned with 8, may be corrupted or under flushing now)");
        }
        System.out.println("entries     : " + size / 8L);
        System.out.println("isFenced    : " + fi.isFenced());
    }

    protected void readLedgerIndexEntries(long ledgerId) throws IOException {
        long curSize;
        System.out.println("===== LEDGER: " + ledgerId + " =====");
        ReadOnlyFileInfo fi = this.getFileInfo(ledgerId);
        long size = fi.size();
        System.out.println("size        : " + size);
        long curEntry = 0L;
        LedgerEntryPage lep = new LedgerEntryPage(this.pageSize, this.entriesPerPage);
        lep.usePage();
        try {
            for (curSize = 0L; curSize < size; curSize += (long)this.pageSize) {
                lep.setLedgerAndFirstEntry(ledgerId, curEntry);
                lep.readPage(fi);
                for (int i = 0; i < this.entriesPerPage; ++i) {
                    long offset = lep.getOffset(i * 8);
                    if (0L == offset) {
                        System.out.println("entry " + curEntry + "\t:\tN/A");
                    } else {
                        long entryLogId = offset >> 32;
                        long pos = offset & 0xFFFFFFFFL;
                        System.out.println("entry " + curEntry + "\t:\t(log:" + entryLogId + ", pos: " + pos + ")");
                    }
                    ++curEntry;
                }
            }
        }
        catch (IOException ie) {
            LOG.error("Failed to read index page : ", (Throwable)ie);
            if (curSize + (long)this.pageSize < size) {
                System.out.println("Failed to read index page @ " + curSize + ", the index file may be corrupted : " + ie.getMessage());
            }
            System.out.println("Failed to read last index page @ " + curSize + ", the index file may be corrupted or last index page is not fully flushed yet : " + ie.getMessage());
        }
    }

    protected void scanEntryLog(long logId, final boolean printMsg) throws Exception {
        System.out.println("Scan entry log " + logId + " (" + Long.toHexString(logId) + ".log)");
        this.scanEntryLog(logId, new EntryLogger.EntryLogScanner(){

            @Override
            public boolean accept(long ledgerId) {
                return true;
            }

            @Override
            public void process(long ledgerId, long startPos, ByteBuffer entry) {
                BookieShell.this.formatEntry(startPos, entry, printMsg);
            }
        });
    }

    protected void scanEntryLogForSpecificEntry(long logId, long ledgerId, final long entryId, final boolean printMsg) throws Exception {
        System.out.println("Scan entry log " + logId + " (" + Long.toHexString(logId) + ".log) for LedgerId " + ledgerId + (entryId == -1L ? "" : " for EntryId " + entryId));
        final MutableBoolean entryFound = new MutableBoolean(false);
        this.scanEntryLog(logId, new EntryLogger.EntryLogScanner(){

            @Override
            public boolean accept(long ledgerId) {
                return !entryFound.booleanValue() || entryId == -1L;
            }

            @Override
            public void process(long ledgerId, long startPos, ByteBuffer entry) {
                long entrysLedgerId = entry.getLong();
                long entrysEntryId = entry.getLong();
                entry.rewind();
                if (ledgerId == entrysLedgerId && entrysEntryId == entryId || entryId == -1L) {
                    entryFound.setValue(true);
                    BookieShell.this.formatEntry(startPos, entry, printMsg);
                }
            }
        });
        if (!entryFound.booleanValue()) {
            System.out.println("LedgerId " + ledgerId + (entryId == -1L ? "" : " EntryId " + entryId) + " is not available in the entry log " + logId + " (" + Long.toHexString(logId) + ".log)");
        }
    }

    protected void scanEntryLogForPositionRange(long logId, final long rangeStartPos, final long rangeEndPos, final boolean printMsg) throws Exception {
        System.out.println("Scan entry log " + logId + " (" + Long.toHexString(logId) + ".log) for PositionRange: " + rangeStartPos + " - " + rangeEndPos);
        final MutableBoolean entryFound = new MutableBoolean(false);
        this.scanEntryLog(logId, new EntryLogger.EntryLogScanner(){
            private MutableBoolean stopScanning = new MutableBoolean(false);

            @Override
            public boolean accept(long ledgerId) {
                return !this.stopScanning.booleanValue();
            }

            @Override
            public void process(long ledgerId, long entryStartPos, ByteBuffer entry) {
                if (!this.stopScanning.booleanValue()) {
                    if (rangeEndPos != -1L && entryStartPos > rangeEndPos) {
                        this.stopScanning.setValue(true);
                    } else {
                        int entrySize = entry.limit();
                        long entryEndPos = entryStartPos + (long)entrySize + 4L - 1L;
                        if ((rangeEndPos == -1L || entryStartPos <= rangeEndPos) && rangeStartPos <= entryEndPos) {
                            BookieShell.this.formatEntry(entryStartPos, entry, printMsg);
                            entryFound.setValue(true);
                        }
                    }
                }
            }
        });
        if (!entryFound.booleanValue()) {
            System.out.println("Entry log " + logId + " (" + Long.toHexString(logId) + ".log) doesn't has any entry in the range " + rangeStartPos + " - " + rangeEndPos + ". Probably the position range, you have provided is lesser than the LOGFILE_HEADER_SIZE (1024) or greater than the current log filesize.");
        }
    }

    protected void scanJournal(Journal journal, long journalId, final boolean printMsg) throws Exception {
        System.out.println("Scan journal " + journalId + " (" + Long.toHexString(journalId) + ".txn)");
        this.scanJournal(journal, journalId, new Journal.JournalScanner(){
            boolean printJournalVersion = false;

            @Override
            public void process(int journalVersion, long offset, ByteBuffer entry) throws IOException {
                if (!this.printJournalVersion) {
                    System.out.println("Journal Version : " + journalVersion);
                    this.printJournalVersion = true;
                }
                BookieShell.this.formatEntry(offset, entry, printMsg);
            }
        });
    }

    protected void printLastLogMark() throws IOException {
        for (Journal journal : this.getJournals()) {
            LogMark lastLogMark = journal.getLastLogMark().getCurMark();
            System.out.println("LastLogMark: Journal Id - " + lastLogMark.getLogFileId() + "(" + Long.toHexString(lastLogMark.getLogFileId()) + ".txn), Pos - " + lastLogMark.getLogFileOffset());
        }
    }

    private void formatEntry(LedgerEntry entry, boolean printMsg) {
        long ledgerId = entry.getLedgerId();
        long entryId = entry.getEntryId();
        long entrySize = entry.getLength();
        System.out.println("--------- Lid=" + ledgerId + ", Eid=" + entryId + ", EntrySize=" + entrySize + " ---------");
        if (printMsg) {
            this.formatter.formatEntry(entry.getEntry());
        }
    }

    private void formatEntry(long pos, ByteBuffer recBuff, boolean printMsg) {
        long ledgerId = recBuff.getLong();
        long entryId = recBuff.getLong();
        int entrySize = recBuff.limit();
        System.out.println("--------- Lid=" + ledgerId + ", Eid=" + entryId + ", ByteOffset=" + pos + ", EntrySize=" + entrySize + " ---------");
        if (entryId == -4096L) {
            int masterKeyLen = recBuff.getInt();
            byte[] masterKey = new byte[masterKeyLen];
            recBuff.get(masterKey);
            System.out.println("Type:           META");
            System.out.println("MasterKey:      " + BookieShell.bytes2Hex(masterKey));
            System.out.println();
            return;
        }
        if (entryId == -8192L) {
            System.out.println("Type:           META");
            System.out.println("Fenced");
            System.out.println();
            return;
        }
        long lastAddConfirmed = recBuff.getLong();
        System.out.println("Type:           DATA");
        System.out.println("LastConfirmed:  " + lastAddConfirmed);
        if (!printMsg) {
            System.out.println();
            return;
        }
        recBuff.position(40);
        System.out.println("Data:");
        System.out.println();
        try {
            byte[] ret = new byte[recBuff.remaining()];
            recBuff.get(ret);
            this.formatter.formatEntry(ret);
        }
        catch (Exception e) {
            System.out.println("N/A. Corrupted.");
        }
        System.out.println();
    }

    static String bytes2Hex(byte[] data) {
        StringBuilder sb = new StringBuilder(data.length * 2);
        Formatter formatter = new Formatter(sb);
        for (byte b : data) {
            formatter.format("%02x", b);
        }
        formatter.close();
        return sb.toString();
    }

    private static int getOptionIntValue(CommandLine cmdLine, String option, int defaultVal) {
        if (cmdLine.hasOption(option)) {
            String val = cmdLine.getOptionValue(option);
            try {
                return Integer.parseInt(val);
            }
            catch (NumberFormatException nfe) {
                System.err.println("ERROR: invalid value for option " + option + " : " + val);
                return defaultVal;
            }
        }
        return defaultVal;
    }

    private static long getOptionLongValue(CommandLine cmdLine, String option, long defaultVal) {
        if (cmdLine.hasOption(option)) {
            String val = cmdLine.getOptionValue(option);
            try {
                return Long.parseLong(val);
            }
            catch (NumberFormatException nfe) {
                System.err.println("ERROR: invalid value for option " + option + " : " + val);
                return defaultVal;
            }
        }
        return defaultVal;
    }

    private static boolean getOptionBooleanValue(CommandLine cmdLine, String option, boolean defaultVal) {
        if (cmdLine.hasOption(option)) {
            String val = cmdLine.getOptionValue(option);
            return Boolean.parseBoolean(val);
        }
        return defaultVal;
    }

    private static boolean getOptionalValue(String optValue, String optName) {
        return StringUtils.equals((String)optValue, (String)optName);
    }

    private static class FilesTimeComparator
    implements Comparator<File>,
    Serializable {
        private static final long serialVersionUID = 1L;

        private FilesTimeComparator() {
        }

        @Override
        public int compare(File file1, File file2) {
            Path file1Path = Paths.get(file1.getAbsolutePath(), new String[0]);
            Path file2Path = Paths.get(file2.getAbsolutePath(), new String[0]);
            try {
                BasicFileAttributes file1Attributes = Files.readAttributes(file1Path, BasicFileAttributes.class, new LinkOption[0]);
                BasicFileAttributes file2Attributes = Files.readAttributes(file2Path, BasicFileAttributes.class, new LinkOption[0]);
                FileTime file1CreationTime = file1Attributes.creationTime();
                FileTime file2CreationTime = file2Attributes.creationTime();
                int compareValue = file1CreationTime.compareTo(file2CreationTime);
                if (compareValue == 0) {
                    FileTime file1LastModifiedTime = file1Attributes.lastModifiedTime();
                    FileTime file2LastModifiedTime = file2Attributes.lastModifiedTime();
                    compareValue = file1LastModifiedTime.compareTo(file2LastModifiedTime);
                }
                return compareValue;
            }
            catch (IOException e) {
                return 0;
            }
        }
    }

    public static interface UpdateLedgerNotifier {
        public void progress(long var1, long var3);
    }

    class DecommissionBookieCmd
    extends MyCommand {
        Options lOpts;

        DecommissionBookieCmd() {
            super(BookieShell.CMD_DECOMMISSIONBOOKIE);
            this.lOpts = new Options();
        }

        @Override
        String getDescription() {
            return "Force trigger the Audittask and make sure all the ledgers stored in the decommissioning bookie are replicated.";
        }

        @Override
        String getUsage() {
            return BookieShell.CMD_DECOMMISSIONBOOKIE;
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            ClientConfiguration adminConf = new ClientConfiguration(BookieShell.this.bkConf);
            try (BookKeeperAdmin admin = new BookKeeperAdmin(adminConf);){
                BookieSocketAddress thisBookieAddress = Bookie.getBookieAddress(BookieShell.this.bkConf);
                admin.decommissionBookie(thisBookieAddress);
                int n = 0;
                return n;
            }
        }
    }

    class TriggerAuditCmd
    extends MyCommand {
        Options opts;

        TriggerAuditCmd() {
            super(BookieShell.CMD_TRIGGERAUDIT);
            this.opts = new Options();
        }

        @Override
        String getDescription() {
            return "Force trigger the Audit by resetting the lostBookieRecoveryDelay.";
        }

        @Override
        String getUsage() {
            return BookieShell.CMD_TRIGGERAUDIT;
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            ClientConfiguration adminConf = new ClientConfiguration(BookieShell.this.bkConf);
            try (BookKeeperAdmin admin = new BookKeeperAdmin(adminConf);){
                admin.triggerAudit();
            }
            return 0;
        }
    }

    class BookieInfoCmd
    extends MyCommand {
        Options lOpts;

        BookieInfoCmd() {
            super(BookieShell.CMD_BOOKIEINFO);
            this.lOpts = new Options();
        }

        @Override
        String getDescription() {
            return "Retrieve bookie info such as free and total disk space.";
        }

        @Override
        String getUsage() {
            return BookieShell.CMD_BOOKIEINFO;
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }

        String getReadable(long val) {
            double d;
            String[] unit = new String[]{"", "KB", "MB", "GB", "TB"};
            int cnt = 0;
            for (d = (double)val; d >= 1000.0 && cnt < unit.length - 1; d /= 1000.0, ++cnt) {
            }
            DecimalFormat df = new DecimalFormat("#.###");
            df.setRoundingMode(RoundingMode.DOWN);
            return cnt > 0 ? "(" + df.format(d) + unit[cnt] + ")" : unit[cnt];
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            ClientConfiguration clientConf = new ClientConfiguration(BookieShell.this.bkConf);
            clientConf.setDiskWeightBasedPlacementEnabled(true);
            BookKeeper bk = new BookKeeper(clientConf);
            Map<BookieSocketAddress, BookieInfoReader.BookieInfo> map = bk.getBookieInfo();
            if (map.size() == 0) {
                System.out.println("Failed to retrieve bookie information from any of the bookies");
                bk.close();
                return 0;
            }
            System.out.println("Free disk space info:");
            long totalFree = 0L;
            long total = 0L;
            for (Map.Entry<BookieSocketAddress, BookieInfoReader.BookieInfo> e : map.entrySet()) {
                BookieInfoReader.BookieInfo bInfo = e.getValue();
                System.out.println(e.getKey() + ":\tFree: " + bInfo.getFreeDiskSpace() + this.getReadable(bInfo.getFreeDiskSpace()) + "\tTotal: " + bInfo.getTotalDiskSpace() + this.getReadable(bInfo.getTotalDiskSpace()));
                totalFree += bInfo.getFreeDiskSpace();
                total += bInfo.getTotalDiskSpace();
            }
            System.out.println("Total free disk space in the cluster:\t" + totalFree + this.getReadable(totalFree));
            System.out.println("Total disk capacity in the cluster:\t" + total + this.getReadable(total));
            bk.close();
            return 0;
        }
    }

    class DeleteLedgerCmd
    extends MyCommand {
        Options lOpts;

        DeleteLedgerCmd() {
            super(BookieShell.CMD_DELETELEDGER);
            this.lOpts = new Options();
            this.lOpts.addOption("l", "ledgerid", true, "Ledger ID");
            this.lOpts.addOption("f", "force", false, "Whether to force delete the Ledger without prompt..?");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            long lid;
            String lidStr = cmdLine.getOptionValue("ledgerid");
            if (StringUtils.isBlank((String)lidStr)) {
                LOG.error("Invalid argument list!");
                this.printUsage();
                return -1;
            }
            try {
                lid = Long.parseLong(lidStr);
            }
            catch (NumberFormatException nfe) {
                System.err.println("ERROR: invalid ledger id " + lidStr);
                this.printUsage();
                return -1;
            }
            boolean force = cmdLine.hasOption("f");
            boolean confirm = false;
            if (!force) {
                confirm = IOUtils.confirmPrompt("Are you sure to delete Ledger : " + lid + "?");
            }
            try (BookKeeper bk = null;){
                if (force || confirm) {
                    ClientConfiguration conf = new ClientConfiguration();
                    conf.addConfiguration((Configuration)BookieShell.this.bkConf);
                    bk = new BookKeeper(conf);
                    bk.deleteLedger(lid);
                }
            }
            return 0;
        }

        @Override
        String getDescription() {
            return "Delete a ledger.";
        }

        @Override
        String getUsage() {
            return "deleteledger -ledgerid <ledgerid> [-force]";
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }
    }

    class UpdateLedgerCmd
    extends MyCommand {
        private final Options opts;

        UpdateLedgerCmd() {
            super(BookieShell.CMD_UPDATELEDGER);
            this.opts = new Options();
            this.opts.addOption("b", "bookieId", true, "Bookie Id");
            this.opts.addOption("s", "updatespersec", true, "Number of ledgers updating per second (default: 5 per sec)");
            this.opts.addOption("l", "limit", true, "Maximum number of ledgers to update (default: no limit)");
            this.opts.addOption("v", "verbose", true, "Print status of the ledger updation (default: false)");
            this.opts.addOption("p", "printprogress", true, "Print messages on every configured seconds if verbose turned on (default: 10 secs)");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Update bookie id in ledgers (this may take a long time).";
        }

        @Override
        String getUsage() {
            return "updateledger -bookieId <hostname|ip> [-updatespersec N] [-limit N] [-verbose true/false] [-printprogress N]";
        }

        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            long printprogress;
            String bookieId = cmdLine.getOptionValue("bookieId");
            if (StringUtils.isBlank((String)bookieId)) {
                LOG.error("Invalid argument list!");
                this.printUsage();
                return -1;
            }
            if (!StringUtils.equals((String)bookieId, (String)"hostname") && !StringUtils.equals((String)bookieId, (String)"ip")) {
                LOG.error("Invalid option value {} for bookieId, expected hostname/ip", (Object)bookieId);
                this.printUsage();
                return -1;
            }
            boolean useHostName = BookieShell.getOptionalValue(bookieId, "hostname");
            if (!BookieShell.this.bkConf.getUseHostNameAsBookieID() && useHostName) {
                LOG.error("Expects configuration useHostNameAsBookieID=true as the option value passed is 'hostname'");
                return -1;
            }
            if (BookieShell.this.bkConf.getUseHostNameAsBookieID() && !useHostName) {
                LOG.error("Expects configuration useHostNameAsBookieID=false as the option value passed is 'ip'");
                return -1;
            }
            int rate = BookieShell.getOptionIntValue(cmdLine, "updatespersec", 5);
            if (rate <= 0) {
                LOG.error("Invalid updatespersec {}, should be > 0", (Object)rate);
                return -1;
            }
            int limit = BookieShell.getOptionIntValue(cmdLine, "limit", Integer.MIN_VALUE);
            if (limit <= 0 && limit != Integer.MIN_VALUE) {
                LOG.error("Invalid limit {}, should be > 0", (Object)limit);
                return -1;
            }
            boolean verbose = BookieShell.getOptionBooleanValue(cmdLine, "verbose", false);
            if (!verbose) {
                if (cmdLine.hasOption("printprogress")) {
                    LOG.warn("Ignoring option 'printprogress', this is applicable when 'verbose' is true");
                }
                printprogress = Integer.MIN_VALUE;
            } else {
                printprogress = BookieShell.getOptionLongValue(cmdLine, "printprogress", 10L);
            }
            ClientConfiguration conf = new ClientConfiguration();
            conf.addConfiguration((Configuration)BookieShell.this.bkConf);
            BookKeeper bk = new BookKeeper(conf);
            BookKeeperAdmin admin = new BookKeeperAdmin(conf);
            UpdateLedgerOp updateLedgerOp = new UpdateLedgerOp(bk, admin);
            ServerConfiguration serverConf = new ServerConfiguration(BookieShell.this.bkConf);
            BookieSocketAddress newBookieId = Bookie.getBookieAddress(serverConf);
            serverConf.setUseHostNameAsBookieID(!useHostName);
            BookieSocketAddress oldBookieId = Bookie.getBookieAddress(serverConf);
            UpdateLedgerNotifier progressable = new UpdateLedgerNotifier(){
                long lastReport = System.nanoTime();

                @Override
                public void progress(long updated, long issued) {
                    if (printprogress <= 0L) {
                        return;
                    }
                    if (TimeUnit.MILLISECONDS.toSeconds(MathUtils.elapsedMSec((long)this.lastReport)) >= printprogress) {
                        LOG.info("Number of ledgers issued={}, updated={}", (Object)issued, (Object)updated);
                        this.lastReport = MathUtils.nowInNano();
                    }
                }
            };
            try {
                updateLedgerOp.updateBookieIdInLedgers(oldBookieId, newBookieId, rate, limit, progressable);
            }
            catch (IOException | BKException e) {
                LOG.error("Failed to update ledger metadata", (Throwable)e);
                return -1;
            }
            return 0;
        }
    }

    class ExpandStorageCmd
    extends MyCommand {
        Options opts;

        ExpandStorageCmd() {
            super(BookieShell.CMD_EXPANDSTORAGE);
            this.opts = new Options();
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Add new empty ledger/index directories. Update the directoriesinfo in the conf file before running the command.";
        }

        @Override
        String getUsage() {
            return BookieShell.CMD_EXPANDSTORAGE;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) {
            ServerConfiguration conf = new ServerConfiguration(BookieShell.this.bkConf);
            try (ZKRegistrationManager rm = new ZKRegistrationManager();){
                try {
                    rm.initialize(BookieShell.this.bkConf, () -> {}, (StatsLogger)NullStatsLogger.INSTANCE);
                }
                catch (BookieException e) {
                    LOG.error("Exception while establishing zookeeper connection.", (Throwable)e);
                    int n = -1;
                    rm.close();
                    return n;
                }
                ArrayList allLedgerDirs = Lists.newArrayList();
                allLedgerDirs.addAll(Arrays.asList(BookieShell.this.ledgerDirectories));
                if (BookieShell.this.indexDirectories != BookieShell.this.ledgerDirectories) {
                    allLedgerDirs.addAll(Arrays.asList(BookieShell.this.indexDirectories));
                }
                try {
                    conf.setAllowStorageExpansion(true);
                    Bookie.checkEnvironmentWithStorageExpansion(conf, rm, Lists.newArrayList((Object[])BookieShell.this.journalDirectories), allLedgerDirs);
                }
                catch (BookieException e) {
                    LOG.error("Exception while updating cookie for storage expansion", (Throwable)e);
                    int n = -1;
                    rm.close();
                    return n;
                }
                int n = 0;
                return n;
            }
        }
    }

    class UpdateCookieCmd
    extends MyCommand {
        Options opts;

        UpdateCookieCmd() {
            super(BookieShell.CMD_UPDATECOOKIE);
            this.opts = new Options();
            this.opts.addOption("b", "bookieId", true, "Bookie Id");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Update bookie id in cookie.";
        }

        @Override
        String getUsage() {
            return "updatecookie -bookieId <hostname|ip>";
        }

        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            String bookieId = cmdLine.getOptionValue("bookieId");
            if (StringUtils.isBlank((String)bookieId)) {
                LOG.error("Invalid argument list!");
                this.printUsage();
                return -1;
            }
            if (!StringUtils.equals((String)bookieId, (String)"hostname") && !StringUtils.equals((String)bookieId, (String)"ip")) {
                LOG.error("Invalid option value:" + bookieId);
                this.printUsage();
                return -1;
            }
            boolean useHostName = BookieShell.getOptionalValue(bookieId, "hostname");
            if (!BookieShell.this.bkConf.getUseHostNameAsBookieID() && useHostName) {
                LOG.error("Expects configuration useHostNameAsBookieID=true as the option value passed is 'hostname'");
                return -1;
            }
            if (BookieShell.this.bkConf.getUseHostNameAsBookieID() && !useHostName) {
                LOG.error("Expects configuration useHostNameAsBookieID=false as the option value passed is 'ip'");
                return -1;
            }
            return this.updateBookieIdInCookie(bookieId, useHostName);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Unable to fully structure code
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private int updateBookieIdInCookie(String bookieId, boolean useHostname) throws BookieException, InterruptedException {
            rm = new ZKRegistrationManager();
            try {
                rm.initialize(BookieShell.this.bkConf, (RegistrationManager.RegistrationListener)LambdaMetafactory.metafactory(null, null, null, ()V, lambda$updateBookieIdInCookie$0(), ()V)(), (StatsLogger)NullStatsLogger.INSTANCE);
                conf = new ServerConfiguration(BookieShell.this.bkConf);
                newBookieId = Bookie.getBookieAddress(conf).toString();
                oldCookie = null;
                try {
                    conf.setUseHostNameAsBookieID(useHostname == false);
                    oldCookie = Cookie.readFromRegistrationManager((RegistrationManager)rm, conf);
                }
                catch (BookieException.CookieNotFoundException nne) {
                    BookieShell.LOG.error("Either cookie already updated with UseHostNameAsBookieID={} or no cookie exists!", (Object)useHostname, (Object)nne);
                    var8_11 = -1;
                    if (rm == null) return var8_11;
                    rm.close();
                    return var8_11;
                }
                newCookie = Cookie.newBuilder(oldCookie.getValue()).setBookieHost(newBookieId).build();
                hasCookieUpdatedInDirs = this.verifyCookie(newCookie, BookieShell.this.journalDirectories[0]);
                var9_13 = BookieShell.this.ledgerDirectories;
                var10_16 = var9_13.length;
                for (var11_17 = 0; var11_17 < var10_16; hasCookieUpdatedInDirs &= this.verifyCookie(newCookie, dir), ++var11_17) {
                    dir = var9_13[var11_17];
                }
                if (BookieShell.this.indexDirectories != BookieShell.this.ledgerDirectories) {
                    var9_13 = BookieShell.this.indexDirectories;
                    var10_16 = var9_13.length;
                    for (var11_17 = 0; var11_17 < var10_16; hasCookieUpdatedInDirs &= this.verifyCookie(newCookie, dir), ++var11_17) {
                        dir = var9_13[var11_17];
                    }
                }
                if (hasCookieUpdatedInDirs) {
                    try {
                        conf.setUseHostNameAsBookieID(useHostname);
                        Cookie.readFromRegistrationManager((RegistrationManager)rm, conf);
                        conf.setUseHostNameAsBookieID(useHostname == false);
                        oldCookie.getValue().deleteFromRegistrationManager((RegistrationManager)rm, conf, oldCookie.getVersion());
                        var9_14 = 0;
                        return var9_14;
                    }
                    catch (BookieException.CookieNotFoundException nne) {
                        if (BookieShell.LOG.isDebugEnabled()) {
                            BookieShell.LOG.debug("Ignoring, cookie will be written to zookeeper");
                        }
lbl46:
                        // 5 sources

                        while (true) {
                            conf.setUseHostNameAsBookieID(useHostname);
                            newCookie.writeToRegistrationManager(rm, conf, Version.NEW);
                            conf.setUseHostNameAsBookieID(useHostname == false);
                            oldCookie.getValue().deleteFromRegistrationManager((RegistrationManager)rm, conf, oldCookie.getVersion());
                            return 0;
                        }
                    }
                }
                var9_13 = BookieShell.this.journalDirectories;
            }
            catch (IOException ioe) {
                BookieShell.LOG.error("IOException during cookie updation!", (Throwable)ioe);
                var5_7 = -1;
                return var5_7;
            }
            for (File journalDirectory : var9_13) {
                newCookie.writeToDirectory(journalDirectory);
                BookieShell.LOG.info("Updated cookie file present in journalDirectory {}", (Object)journalDirectory);
            }
            for (File dir : BookieShell.this.ledgerDirectories) {
                newCookie.writeToDirectory(dir);
            }
            BookieShell.LOG.info("Updated cookie file present in ledgerDirectories {}", (Object[])BookieShell.this.ledgerDirectories);
            if (BookieShell.this.ledgerDirectories == BookieShell.this.indexDirectories) ** GOTO lbl46
            for (File dir : BookieShell.this.indexDirectories) {
                newCookie.writeToDirectory(dir);
            }
            BookieShell.LOG.info("Updated cookie file present in indexDirectories {}", (Object[])BookieShell.this.indexDirectories);
            ** while (true)
            finally {
                if (rm != null) {
                    rm.close();
                }
            }
        }

        private boolean verifyCookie(Cookie oldCookie, File dir) throws IOException {
            try {
                Cookie cookie = Cookie.readFromDirectory(dir);
                cookie.verify(oldCookie);
            }
            catch (BookieException.InvalidCookieException e) {
                return false;
            }
            return true;
        }

        private static /* synthetic */ void lambda$updateBookieIdInCookie$0() {
        }
    }

    class WhoIsAuditorCmd
    extends MyCommand {
        Options opts;

        public WhoIsAuditorCmd() {
            super(BookieShell.CMD_WHOISAUDITOR);
            this.opts = new Options();
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Print the node which holds the auditor lock.";
        }

        @Override
        String getUsage() {
            return BookieShell.CMD_WHOISAUDITOR;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            try (ZooKeeperClient zk = null;){
                zk = ZooKeeperClient.newBuilder().connectString(BookieShell.this.bkConf.getZkServers()).sessionTimeoutMs(BookieShell.this.bkConf.getZkTimeout()).build();
                BookieSocketAddress bookieId = AuditorElector.getCurrentAuditor(BookieShell.this.bkConf, zk);
                if (bookieId == null) {
                    LOG.info("No auditor elected");
                    int n = -1;
                    return n;
                }
                LOG.info("Auditor: {}/{}:{}", new Object[]{bookieId.getSocketAddress().getAddress().getCanonicalHostName(), bookieId.getSocketAddress().getAddress().getHostAddress(), bookieId.getSocketAddress().getPort()});
            }
            return 0;
        }
    }

    class LostBookieRecoveryDelayCmd
    extends MyCommand {
        Options opts;

        public LostBookieRecoveryDelayCmd() {
            super(BookieShell.CMD_LOSTBOOKIERECOVERYDELAY);
            this.opts = new Options();
            this.opts.addOption("g", "get", false, "Get LostBookieRecoveryDelay value (in seconds)");
            this.opts.addOption("s", "set", true, "Set LostBookieRecoveryDelay value (in seconds)");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Setter and Getter for LostBookieRecoveryDelay value (in seconds) in Zookeeper.";
        }

        @Override
        String getUsage() {
            return "lostbookierecoverydelay [-get|-set <value>]";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            boolean getter = cmdLine.hasOption("g");
            boolean setter = cmdLine.hasOption("s");
            if (!getter && !setter || getter && setter) {
                LOG.error("One and only one of -get and -set must be specified");
                this.printUsage();
                return 1;
            }
            ClientConfiguration adminConf = new ClientConfiguration(BookieShell.this.bkConf);
            try (BookKeeperAdmin admin = new BookKeeperAdmin(adminConf);){
                if (getter) {
                    int lostBookieRecoveryDelay = admin.getLostBookieRecoveryDelay();
                    LOG.info("LostBookieRecoveryDelay value in ZK: {}", (Object)String.valueOf(lostBookieRecoveryDelay));
                } else {
                    int lostBookieRecoveryDelay = Integer.parseInt(cmdLine.getOptionValue("set"));
                    admin.setLostBookieRecoveryDelay(lostBookieRecoveryDelay);
                    LOG.info("Successfully set LostBookieRecoveryDelay value in ZK: {}", (Object)String.valueOf(lostBookieRecoveryDelay));
                }
            }
            return 0;
        }
    }

    class AutoRecoveryCmd
    extends MyCommand {
        Options opts;

        public AutoRecoveryCmd() {
            super(BookieShell.CMD_AUTORECOVERY);
            this.opts = new Options();
            this.opts.addOption("e", "enable", false, "Enable auto recovery of underreplicated ledgers");
            this.opts.addOption("d", "disable", false, "Disable auto recovery of underreplicated ledgers");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Enable or disable autorecovery in the cluster.";
        }

        @Override
        String getUsage() {
            return "autorecovery [-enable|-disable]";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            boolean disable = cmdLine.hasOption("d");
            boolean enable = cmdLine.hasOption("e");
            if (enable && disable) {
                LOG.error("Only one of -enable and -disable can be specified");
                this.printUsage();
                return 1;
            }
            try (ZooKeeperClient zk = null;){
                zk = ZooKeeperClient.newBuilder().connectString(BookieShell.this.bkConf.getZkServers()).sessionTimeoutMs(BookieShell.this.bkConf.getZkTimeout()).build();
                LedgerManagerFactory mFactory = LedgerManagerFactory.newLedgerManagerFactory(BookieShell.this.bkConf, zk);
                LedgerUnderreplicationManager underreplicationManager = mFactory.newLedgerUnderreplicationManager();
                if (!enable && !disable) {
                    boolean enabled = underreplicationManager.isLedgerReplicationEnabled();
                    System.out.println("Autorecovery is " + (enabled ? "enabled." : "disabled."));
                } else if (enable) {
                    if (underreplicationManager.isLedgerReplicationEnabled()) {
                        LOG.warn("Autorecovery already enabled. Doing nothing");
                    } else {
                        LOG.info("Enabling autorecovery");
                        underreplicationManager.enableLedgerReplication();
                    }
                } else if (!underreplicationManager.isLedgerReplicationEnabled()) {
                    LOG.warn("Autorecovery already disabled. Doing nothing");
                } else {
                    LOG.info("Disabling autorecovery");
                    underreplicationManager.disableLedgerReplication();
                }
            }
            return 0;
        }
    }

    class HelpCmd
    extends MyCommand {
        HelpCmd() {
            super(BookieShell.CMD_HELP);
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            String[] args = cmdLine.getArgs();
            if (args.length == 0) {
                BookieShell.this.printShellUsage();
                return 0;
            }
            String cmdName = args[0];
            Command cmd = BookieShell.this.commands.get(cmdName);
            if (null == cmd) {
                System.err.println("Unknown command " + cmdName);
                BookieShell.this.printShellUsage();
                return -1;
            }
            cmd.printUsage();
            return 0;
        }

        @Override
        String getDescription() {
            return "Describe the usage of this program or its subcommands.";
        }

        @Override
        String getUsage() {
            return "help         [COMMAND]";
        }

        @Override
        Options getOptions() {
            return new Options();
        }
    }

    class ListDiskFilesCmd
    extends MyCommand {
        Options opts;

        ListDiskFilesCmd() {
            super(BookieShell.CMD_LISTFILESONDISC);
            this.opts = new Options();
            this.opts.addOption("txn", "journal", false, "Print list of Journal Files");
            this.opts.addOption("log", "entrylog", false, "Print list of EntryLog Files");
            this.opts.addOption("idx", "index", false, "Print list of Index files");
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            boolean journal = cmdLine.hasOption("txn");
            boolean entrylog = cmdLine.hasOption("log");
            boolean index = cmdLine.hasOption("idx");
            boolean all = false;
            if (!(journal || entrylog || index || all)) {
                all = true;
            }
            if (all || journal) {
                File[] journalDirs = BookieShell.this.bkConf.getJournalDirs();
                List<File> journalFiles = BookieShell.listFilesAndSort(journalDirs, "txn");
                System.out.println("--------- Printing the list of Journal Files ---------");
                for (File journalFile : journalFiles) {
                    System.out.println(journalFile.getName());
                }
                System.out.println();
            }
            if (all || entrylog) {
                File[] ledgerDirs = BookieShell.this.bkConf.getLedgerDirs();
                List<File> ledgerFiles = BookieShell.listFilesAndSort(ledgerDirs, "log");
                System.out.println("--------- Printing the list of EntryLog/Ledger Files ---------");
                for (File ledgerFile : ledgerFiles) {
                    System.out.println(ledgerFile.getName());
                }
                System.out.println();
            }
            if (all || index) {
                File[] indexDirs = BookieShell.this.bkConf.getIndexDirs() == null ? BookieShell.this.bkConf.getLedgerDirs() : BookieShell.this.bkConf.getIndexDirs();
                List<File> indexFiles = BookieShell.listFilesAndSort(indexDirs, "idx");
                System.out.println("--------- Printing the list of Index Files ---------");
                for (File indexFile : indexFiles) {
                    System.out.println(indexFile.getName());
                }
            }
            return 0;
        }

        @Override
        String getDescription() {
            return "List the files in JournalDirectory/LedgerDirectories/IndexDirectories.";
        }

        @Override
        String getUsage() {
            return "listfilesondisc  [-journal|-entrylog|-index]";
        }

        @Override
        Options getOptions() {
            return this.opts;
        }
    }

    class ListBookiesCmd
    extends MyCommand {
        Options opts;

        ListBookiesCmd() {
            super(BookieShell.CMD_LISTBOOKIES);
            this.opts = new Options();
            this.opts.addOption("rw", "readwrite", false, "Print readwrite bookies");
            this.opts.addOption("ro", "readonly", false, "Print readonly bookies");
            this.opts.addOption("h", "hostnames", false, "Also print hostname of the bookie");
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            boolean readwrite = cmdLine.hasOption("rw");
            boolean readonly = cmdLine.hasOption("ro");
            if (!readwrite && !readonly || readwrite && readonly) {
                LOG.error("One and only one of -readwrite and -readonly must be specified");
                this.printUsage();
                return 1;
            }
            ClientConfiguration clientconf = new ClientConfiguration(BookieShell.this.bkConf).setZkServers(BookieShell.this.bkConf.getZkServers());
            BookKeeperAdmin bka = new BookKeeperAdmin(clientconf);
            int count = 0;
            ArrayList<BookieSocketAddress> bookies = new ArrayList<BookieSocketAddress>();
            if (cmdLine.hasOption("rw")) {
                Collection<BookieSocketAddress> availableBookies = bka.getAvailableBookies();
                bookies.addAll(availableBookies);
            } else if (cmdLine.hasOption("ro")) {
                Collection<BookieSocketAddress> roBookies = bka.getReadOnlyBookies();
                bookies.addAll(roBookies);
            }
            for (BookieSocketAddress b : bookies) {
                System.out.print(b);
                if (cmdLine.hasOption("h")) {
                    System.out.print("\t" + b.getSocketAddress().getHostName());
                }
                System.out.println("");
                ++count;
            }
            if (count == 0) {
                System.err.println("No bookie exists!");
                return 1;
            }
            return 0;
        }

        @Override
        String getDescription() {
            return "List the bookies, which are running as either readwrite or readonly mode.";
        }

        @Override
        String getUsage() {
            return "listbookies  [-readwrite|-readonly] [-hostnames]";
        }

        @Override
        Options getOptions() {
            return this.opts;
        }
    }

    class LastMarkCmd
    extends MyCommand {
        LastMarkCmd() {
            super(BookieShell.CMD_LASTMARK);
        }

        @Override
        public int runCmd(CommandLine c) throws Exception {
            BookieShell.this.printLastLogMark();
            return 0;
        }

        @Override
        String getDescription() {
            return "Print last log marker.";
        }

        @Override
        String getUsage() {
            return BookieShell.CMD_LASTMARK;
        }

        @Override
        Options getOptions() {
            return new Options();
        }
    }

    class ReadJournalCmd
    extends MyCommand {
        Options rjOpts;

        ReadJournalCmd() {
            super(BookieShell.CMD_READJOURNAL);
            this.rjOpts = new Options();
            this.rjOpts.addOption("dir", false, "Journal directory (needed if more than one journal configured)");
            this.rjOpts.addOption("m", "msg", false, "Print message body");
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            long journalId;
            String[] leftArgs = cmdLine.getArgs();
            if (leftArgs.length <= 0) {
                System.err.println("ERROR: missing journal id or journal file name");
                this.printUsage();
                return -1;
            }
            boolean printMsg = false;
            if (cmdLine.hasOption("m")) {
                printMsg = true;
            }
            Journal journal = null;
            if (BookieShell.this.getJournals().size() > 1) {
                if (!cmdLine.hasOption("dir")) {
                    System.err.println("ERROR: invalid or missing journal directory");
                    this.printUsage();
                    return -1;
                }
                File journalDirectory = new File(cmdLine.getOptionValue("dir"));
                for (Journal j : BookieShell.this.getJournals()) {
                    if (!j.getJournalDirectory().equals(journalDirectory)) continue;
                    journal = j;
                    break;
                }
                if (journal == null) {
                    System.err.println("ERROR: journal directory not found");
                    this.printUsage();
                    return -1;
                }
            } else {
                journal = (Journal)BookieShell.this.getJournals().get(0);
            }
            try {
                journalId = Long.parseLong(leftArgs[0]);
            }
            catch (NumberFormatException nfe) {
                File f = new File(leftArgs[0]);
                String name = f.getName();
                if (!name.endsWith(".txn")) {
                    System.err.println("ERROR: invalid journal file name " + leftArgs[0]);
                    this.printUsage();
                    return -1;
                }
                String idString = name.split("\\.")[0];
                journalId = Long.parseLong(idString, 16);
            }
            BookieShell.this.scanJournal(journal, journalId, printMsg);
            return 0;
        }

        @Override
        String getDescription() {
            return "Scan a journal file and format the entries into readable format.";
        }

        @Override
        String getUsage() {
            return "readjournal [-dir] [-msg] <journal_id | journal_file_name>";
        }

        @Override
        Options getOptions() {
            return this.rjOpts;
        }
    }

    class ReadLogCmd
    extends MyCommand {
        Options rlOpts;

        ReadLogCmd() {
            super(BookieShell.CMD_READLOG);
            this.rlOpts = new Options();
            this.rlOpts.addOption("m", "msg", false, "Print message body");
            this.rlOpts.addOption("l", "ledgerid", true, "Ledger ID");
            this.rlOpts.addOption("e", "entryid", true, "Entry ID");
            this.rlOpts.addOption("sp", "startpos", true, "Start Position");
            this.rlOpts.addOption("ep", "endpos", true, "End Position");
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            long logId;
            String[] leftArgs = cmdLine.getArgs();
            if (leftArgs.length <= 0) {
                System.err.println("ERROR: missing entry log id or entry log file name");
                this.printUsage();
                return -1;
            }
            boolean printMsg = false;
            if (cmdLine.hasOption("m")) {
                printMsg = true;
            }
            try {
                logId = Long.parseLong(leftArgs[0]);
            }
            catch (NumberFormatException nfe) {
                File f = new File(leftArgs[0]);
                String name = f.getName();
                if (!name.endsWith(".log")) {
                    System.err.println("ERROR: invalid entry log file name " + leftArgs[0]);
                    this.printUsage();
                    return -1;
                }
                String idString = name.split("\\.")[0];
                logId = Long.parseLong(idString, 16);
            }
            long lId = BookieShell.getOptionLongValue(cmdLine, "ledgerid", -1L);
            long eId = BookieShell.getOptionLongValue(cmdLine, "entryid", -1L);
            long startpos = BookieShell.getOptionLongValue(cmdLine, "startpos", -1L);
            long endpos = BookieShell.getOptionLongValue(cmdLine, "endpos", -1L);
            if (startpos != -1L) {
                if (endpos != -1L && endpos < startpos) {
                    System.err.println("ERROR: StartPosition of the range should be lesser than or equal to EndPosition");
                    return -1;
                }
                BookieShell.this.scanEntryLogForPositionRange(logId, startpos, endpos, printMsg);
            } else if (lId != -1L) {
                BookieShell.this.scanEntryLogForSpecificEntry(logId, lId, eId, printMsg);
            } else {
                BookieShell.this.scanEntryLog(logId, printMsg);
            }
            return 0;
        }

        @Override
        String getDescription() {
            return "Scan an entry file and format the entries into readable format.";
        }

        @Override
        String getUsage() {
            return "readlog      [-msg] <entry_log_id | entry_log_file_name> [-ledgerid <ledgerid> [-entryid <entryid>]] [-startpos <startEntryLogBytePos> [-endpos <endEntryLogBytePos>]]";
        }

        @Override
        Options getOptions() {
            return this.rlOpts;
        }
    }

    class BookieSanityTestCmd
    extends MyCommand {
        Options lOpts;

        BookieSanityTestCmd() {
            super(BookieShell.CMD_BOOKIESANITYTEST);
            this.lOpts = new Options();
            this.lOpts.addOption("e", "entries", true, "Total entries to be added for the test (default 10)");
            this.lOpts.addOption("t", "timeout", true, "Timeout for write/read operations in seconds (default 1)");
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }

        @Override
        String getDescription() {
            return "Sanity test for local bookie. Create ledger and write/reads entries on local bookie.";
        }

        @Override
        String getUsage() {
            return "bookiesanity [-entries N] [-timeout N]";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            int numberOfEntries = BookieShell.getOptionIntValue(cmdLine, "entries", 10);
            int timeoutSecs = BookieShell.getOptionIntValue(cmdLine, "timeout", 1);
            ClientConfiguration conf = new ClientConfiguration();
            conf.addConfiguration((Configuration)BookieShell.this.bkConf);
            conf.setEnsemblePlacementPolicy(LocalBookieEnsemblePlacementPolicy.class);
            conf.setAddEntryTimeout(timeoutSecs);
            conf.setReadEntryTimeout(timeoutSecs);
            BookKeeper bk = new BookKeeper(conf);
            LedgerHandle lh = null;
            try {
                lh = bk.createLedger(1, 1, BookKeeper.DigestType.MAC, new byte[0]);
                LOG.info("Created ledger {}", (Object)lh.getId());
                for (int i = 0; i < numberOfEntries; ++i) {
                    String content = "entry-" + i;
                    lh.addEntry(content.getBytes(Charsets.UTF_8));
                }
                LOG.info("Written {} entries in ledger {}", (Object)numberOfEntries, (Object)lh.getId());
                lh = bk.openLedger(lh.getId(), BookKeeper.DigestType.MAC, new byte[0]);
                if (lh.getLastAddConfirmed() != (long)(numberOfEntries - 1)) {
                    throw new Exception("Invalid last entry found on ledger. expecting: " + (numberOfEntries - 1) + " -- found: " + lh.getLastAddConfirmed());
                }
                Enumeration<LedgerEntry> entries = lh.readEntries(0L, numberOfEntries - 1);
                int i = 0;
                while (entries.hasMoreElements()) {
                    LedgerEntry entry = entries.nextElement();
                    String actualMsg = new String(entry.getEntry(), Charsets.UTF_8);
                    String expectedMsg = "entry-" + i++;
                    if (expectedMsg.equals(actualMsg)) continue;
                    throw new Exception("Failed validation of received message - Expected: " + expectedMsg + ", Actual: " + actualMsg);
                }
                LOG.info("Read {} entries from ledger {}", entries, (Object)lh.getId());
            }
            catch (Exception e) {
                LOG.warn("Error in bookie sanity test", (Throwable)e);
                int n = -1;
                return n;
            }
            finally {
                if (lh != null) {
                    bk.deleteLedger(lh.getId());
                    LOG.info("Deleted ledger {}", (Object)lh.getId());
                }
                bk.close();
            }
            LOG.info("Bookie sanity test succeeded");
            return 0;
        }
    }

    class SimpleTestCmd
    extends MyCommand {
        Options lOpts;

        SimpleTestCmd() {
            super(BookieShell.CMD_SIMPLETEST);
            this.lOpts = new Options();
            this.lOpts.addOption("e", "ensemble", true, "Ensemble size (default 3)");
            this.lOpts.addOption("w", "writeQuorum", true, "Write quorum size (default 2)");
            this.lOpts.addOption("a", "ackQuorum", true, "Ack quorum size (default 2)");
            this.lOpts.addOption("n", "numEntries", true, "Entries to write (default 1000)");
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            byte[] data = new byte[100];
            int ensemble = BookieShell.getOptionIntValue(cmdLine, "ensemble", 3);
            int writeQuorum = BookieShell.getOptionIntValue(cmdLine, "writeQuorum", 2);
            int ackQuorum = BookieShell.getOptionIntValue(cmdLine, "ackQuorum", 2);
            int numEntries = BookieShell.getOptionIntValue(cmdLine, "numEntries", 1000);
            ClientConfiguration conf = new ClientConfiguration();
            conf.addConfiguration((Configuration)BookieShell.this.bkConf);
            BookKeeper bk = new BookKeeper(conf);
            LedgerHandle lh = bk.createLedger(ensemble, writeQuorum, ackQuorum, BookKeeper.DigestType.MAC, new byte[0]);
            System.out.println("Ledger ID: " + lh.getId());
            long lastReport = System.nanoTime();
            for (int i = 0; i < numEntries; ++i) {
                lh.addEntry(data);
                if (TimeUnit.SECONDS.convert(System.nanoTime() - lastReport, TimeUnit.NANOSECONDS) <= 1L) continue;
                System.out.println(i + " entries written");
                lastReport = System.nanoTime();
            }
            lh.close();
            bk.close();
            System.out.println(numEntries + " entries written to ledger " + lh.getId());
            return 0;
        }

        @Override
        String getDescription() {
            return "Simple test to create a ledger and write entries to it.";
        }

        @Override
        String getUsage() {
            return "simpletest   [-ensemble N] [-writeQuorum N] [-ackQuorum N] [-numEntries N]";
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }
    }

    class LedgerMetadataCmd
    extends MyCommand {
        Options lOpts;

        LedgerMetadataCmd() {
            super(BookieShell.CMD_LEDGERMETADATA);
            this.lOpts = new Options();
            this.lOpts.addOption("l", "ledgerid", true, "Ledger ID");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            long lid = BookieShell.getOptionLongValue(cmdLine, "ledgerid", -1L);
            if (lid == -1L) {
                System.err.println("Must specify a ledger id");
                return -1;
            }
            ZooKeeperClient zk = null;
            LedgerManagerFactory mFactory = null;
            Closeable m = null;
            try {
                zk = ZooKeeperClient.newBuilder().connectString(BookieShell.this.bkConf.getZkServers()).sessionTimeoutMs(BookieShell.this.bkConf.getZkTimeout()).build();
                mFactory = LedgerManagerFactory.newLedgerManagerFactory(BookieShell.this.bkConf, zk);
                m = mFactory.newLedgerManager();
                ReadMetadataCallback cb = new ReadMetadataCallback(lid);
                m.readLedgerMetadata(lid, cb);
                BookieShell.printLedgerMetadata(cb);
            }
            finally {
                if (m != null) {
                    try {
                        m.close();
                        mFactory.uninitialize();
                    }
                    catch (IOException ioe) {
                        LOG.error("Failed to close ledger manager : ", (Throwable)ioe);
                    }
                }
                if (zk != null) {
                    zk.close();
                }
            }
            return 0;
        }

        @Override
        String getDescription() {
            return "Print the metadata for a ledger.";
        }

        @Override
        String getUsage() {
            return "ledgermetadata -ledgerid <ledgerid>";
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }
    }

    static class ReadMetadataCallback
    extends AbstractFuture<LedgerMetadata>
    implements BookkeeperInternalCallbacks.GenericCallback<LedgerMetadata> {
        final long ledgerId;

        ReadMetadataCallback(long ledgerId) {
            this.ledgerId = ledgerId;
        }

        long getLedgerId() {
            return this.ledgerId;
        }

        @Override
        public void operationComplete(int rc, LedgerMetadata result) {
            if (rc != 0) {
                this.setException(BKException.create(rc));
            } else {
                this.set(result);
            }
        }
    }

    class ListLedgersCmd
    extends MyCommand {
        Options lOpts;

        ListLedgersCmd() {
            super(BookieShell.CMD_LISTLEDGERS);
            this.lOpts = new Options();
            this.lOpts.addOption("m", "meta", false, "Print metadata");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            ZooKeeperClient zk = null;
            LedgerManagerFactory mFactory = null;
            Closeable m = null;
            try {
                zk = ZooKeeperClient.newBuilder().connectString(BookieShell.this.bkConf.getZkServers()).sessionTimeoutMs(BookieShell.this.bkConf.getZkTimeout()).build();
                mFactory = LedgerManagerFactory.newLedgerManagerFactory(BookieShell.this.bkConf, zk);
                m = mFactory.newLedgerManager();
                LedgerManager.LedgerRangeIterator iter = m.getLedgerRanges();
                if (cmdLine.hasOption("m")) {
                    ArrayList<ReadMetadataCallback> futures = new ArrayList<ReadMetadataCallback>(1000);
                    while (iter.hasNext()) {
                        LedgerManager.LedgerRange r = iter.next();
                        for (Long lid : r.getLedgers()) {
                            ReadMetadataCallback cb = new ReadMetadataCallback(lid);
                            m.readLedgerMetadata(lid, cb);
                            futures.add(cb);
                        }
                        if (futures.size() < 1000) continue;
                        while (futures.size() > 0) {
                            ReadMetadataCallback cb = (ReadMetadataCallback)futures.remove(0);
                            BookieShell.printLedgerMetadata(cb);
                        }
                    }
                    while (futures.size() > 0) {
                        ReadMetadataCallback cb = (ReadMetadataCallback)futures.remove(0);
                        BookieShell.printLedgerMetadata(cb);
                    }
                } else {
                    while (iter.hasNext()) {
                        LedgerManager.LedgerRange r = iter.next();
                        for (Long lid : r.getLedgers()) {
                            System.out.println(Long.toString(lid));
                        }
                    }
                }
            }
            finally {
                if (m != null) {
                    try {
                        m.close();
                        mFactory.uninitialize();
                    }
                    catch (IOException ioe) {
                        LOG.error("Failed to close ledger manager : ", (Throwable)ioe);
                    }
                }
                if (zk != null) {
                    zk.close();
                }
            }
            return 0;
        }

        @Override
        String getDescription() {
            return "List all ledgers on the cluster (this may take a long time).";
        }

        @Override
        String getUsage() {
            return "listledgers  [-meta]";
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }
    }

    class ListUnderreplicatedCmd
    extends MyCommand {
        Options opts;

        public ListUnderreplicatedCmd() {
            super(BookieShell.CMD_LISTUNDERREPLICATED);
            this.opts = new Options();
            this.opts.addOption("missingreplica", true, "Bookie Id of missing replica");
            this.opts.addOption("excludingmissingreplica", true, "Bookie Id of missing replica to ignore");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "List ledgers marked as underreplicated, with optional options to specify missingreplica (BookieId) and to exclude missingreplica.";
        }

        @Override
        String getUsage() {
            return "listunderreplicated [[-missingreplica <bookieaddress>] [-excludingmissingreplica <bookieaddress>]]";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            String includingBookieId = cmdLine.getOptionValue("missingreplica");
            String excludingBookieId = cmdLine.getOptionValue("excludingmissingreplica");
            Predicate<List<String>> predicate = null;
            if (!StringUtils.isBlank((String)includingBookieId) && !StringUtils.isBlank((String)excludingBookieId)) {
                predicate = replicasList -> replicasList.contains(includingBookieId) && !replicasList.contains(excludingBookieId);
            } else if (!StringUtils.isBlank((String)includingBookieId)) {
                predicate = replicasList -> replicasList.contains(includingBookieId);
            } else if (!StringUtils.isBlank((String)excludingBookieId)) {
                predicate = replicasList -> !replicasList.contains(excludingBookieId);
            }
            try (ZooKeeperClient zk = null;){
                zk = ZooKeeperClient.newBuilder().connectString(BookieShell.this.bkConf.getZkServers()).sessionTimeoutMs(BookieShell.this.bkConf.getZkTimeout()).build();
                LedgerManagerFactory mFactory = LedgerManagerFactory.newLedgerManagerFactory(BookieShell.this.bkConf, zk);
                LedgerUnderreplicationManager underreplicationManager = mFactory.newLedgerUnderreplicationManager();
                Iterator<Long> iter = underreplicationManager.listLedgersToRereplicate(predicate);
                while (iter.hasNext()) {
                    System.out.println(iter.next());
                }
            }
            return 0;
        }
    }

    class ReadLedgerEntriesCmd
    extends MyCommand {
        Options lOpts;

        ReadLedgerEntriesCmd() {
            super(BookieShell.CMD_READ_LEDGER_ENTRIES);
            this.lOpts = new Options();
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }

        @Override
        String getDescription() {
            return "Read a range of entries from a ledger.";
        }

        @Override
        String getUsage() {
            return "readledger <ledger_id> [<start_entry_id> [<end_entry_id>]]";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            long ledgerId;
            String[] leftArgs = cmdLine.getArgs();
            if (leftArgs.length <= 0) {
                System.err.println("ERROR: missing ledger id");
                this.printUsage();
                return -1;
            }
            long firstEntry = 0L;
            long lastEntry = -1L;
            try {
                ledgerId = Long.parseLong(leftArgs[0]);
                if (leftArgs.length >= 2) {
                    firstEntry = Long.parseLong(leftArgs[1]);
                }
                if (leftArgs.length >= 3) {
                    lastEntry = Long.parseLong(leftArgs[2]);
                }
            }
            catch (NumberFormatException nfe) {
                System.err.println("ERROR: invalid number " + nfe.getMessage());
                this.printUsage();
                return -1;
            }
            ClientConfiguration conf = new ClientConfiguration();
            conf.addConfiguration((Configuration)BookieShell.this.bkConf);
            try (BookKeeperAdmin bk = null;){
                bk = new BookKeeperAdmin(conf);
                for (LedgerEntry entry : bk.readEntries(ledgerId, firstEntry, lastEntry)) {
                    BookieShell.this.formatEntry(entry, true);
                }
            }
            return 0;
        }
    }

    class LedgerCmd
    extends MyCommand {
        Options lOpts;

        LedgerCmd() {
            super(BookieShell.CMD_LEDGER);
            this.lOpts = new Options();
            this.lOpts.addOption("m", "meta", false, "Print meta information");
        }

        @Override
        public int runCmd(CommandLine cmdLine) throws Exception {
            long ledgerId;
            String[] leftArgs = cmdLine.getArgs();
            if (leftArgs.length <= 0) {
                System.err.println("ERROR: missing ledger id");
                this.printUsage();
                return -1;
            }
            boolean printMeta = false;
            if (cmdLine.hasOption("m")) {
                printMeta = true;
            }
            try {
                ledgerId = Long.parseLong(leftArgs[0]);
            }
            catch (NumberFormatException nfe) {
                System.err.println("ERROR: invalid ledger id " + leftArgs[0]);
                this.printUsage();
                return -1;
            }
            if (printMeta) {
                BookieShell.this.readLedgerMeta(ledgerId);
            }
            BookieShell.this.readLedgerIndexEntries(ledgerId);
            return 0;
        }

        @Override
        String getDescription() {
            return "Dump ledger index entries into readable format.";
        }

        @Override
        String getUsage() {
            return "ledger       [-m] <ledger_id>";
        }

        @Override
        Options getOptions() {
            return this.lOpts;
        }
    }

    class RecoverCmd
    extends MyCommand {
        Options opts;

        public RecoverCmd() {
            super(BookieShell.CMD_RECOVER);
            this.opts = new Options();
            this.opts.addOption("q", "query", false, "Query the ledgers that contain given bookies");
            this.opts.addOption("dr", "dryrun", false, "Printing the recovery plan w/o doing actual recovery");
            this.opts.addOption("f", "force", false, "Force recovery without confirmation");
            this.opts.addOption("l", BookieShell.CMD_LEDGER, true, "Recover a specific ledger");
            this.opts.addOption("sk", "skipOpenLedgers", false, "Skip recovering open ledgers");
            this.opts.addOption("d", "deleteCookie", false, "Delete cookie node for the bookie.");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Recover the ledger data for failed bookie.";
        }

        @Override
        String getUsage() {
            return "recover [-deleteCookie] <bookieSrc[:bookieSrc]>";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            String[] args = cmdLine.getArgs();
            if (args.length < 1) {
                throw new MissingArgumentException("'bookieSrc' argument required");
            }
            if (args.length > 1) {
                System.err.println("The provided bookie dest " + args[1] + " will be ignored!");
            }
            boolean query = cmdLine.hasOption("q");
            boolean dryrun = cmdLine.hasOption("dr");
            boolean force = cmdLine.hasOption("f");
            boolean skipOpenLedgers = cmdLine.hasOption("sk");
            boolean removeCookies = !dryrun && cmdLine.hasOption("d");
            Long ledgerId = null;
            if (cmdLine.hasOption("l")) {
                try {
                    ledgerId = Long.parseLong(cmdLine.getOptionValue("l"));
                }
                catch (NumberFormatException nfe) {
                    throw new IOException("Invalid ledger id provided : " + cmdLine.getOptionValue("l"));
                }
            }
            String[] bookieStrs = args[0].split(",");
            HashSet<BookieSocketAddress> bookieAddrs = new HashSet<BookieSocketAddress>();
            for (String bookieStr : bookieStrs) {
                String[] bookieStrParts = bookieStr.split(":");
                if (bookieStrParts.length != 2) {
                    System.err.println("BookieSrcs has invalid bookie address format (host:port expected) : " + bookieStr);
                    return -1;
                }
                bookieAddrs.add(new BookieSocketAddress(bookieStrParts[0], Integer.parseInt(bookieStrParts[1])));
            }
            if (!force) {
                System.err.println("Bookies : " + bookieAddrs);
                if (!IOUtils.confirmPrompt("Are you sure to recover them : (Y/N)")) {
                    System.err.println("Give up!");
                    return -1;
                }
            }
            LOG.info("Constructing admin");
            ClientConfiguration adminConf = new ClientConfiguration(BookieShell.this.bkConf);
            LOG.info("Construct admin : {}", (Object)admin);
            try (BookKeeperAdmin admin = new BookKeeperAdmin(adminConf);){
                int n;
                if (query) {
                    n = this.bkQuery(admin, bookieAddrs);
                    return n;
                }
                if (null != ledgerId) {
                    n = this.bkRecoveryLedger(admin, ledgerId, bookieAddrs, dryrun, skipOpenLedgers, removeCookies);
                    return n;
                }
                n = this.bkRecovery(admin, bookieAddrs, dryrun, skipOpenLedgers, removeCookies);
                return n;
            }
        }

        private int bkQuery(BookKeeperAdmin bkAdmin, Set<BookieSocketAddress> bookieAddrs) throws InterruptedException, BKException {
            SortedMap<Long, LedgerMetadata> ledgersContainBookies = bkAdmin.getLedgersContainBookies(bookieAddrs);
            System.err.println("NOTE: Bookies in inspection list are marked with '*'.");
            for (Map.Entry<Long, LedgerMetadata> ledger : ledgersContainBookies.entrySet()) {
                System.out.println("ledger " + ledger.getKey() + " : " + (Object)((Object)ledger.getValue().getState()));
                Map<Long, Integer> numBookiesToReplacePerEnsemble = this.inspectLedger(ledger.getValue(), bookieAddrs);
                System.out.print("summary: [");
                for (Map.Entry<Long, Integer> entry : numBookiesToReplacePerEnsemble.entrySet()) {
                    System.out.print(entry.getKey() + "=" + entry.getValue() + ", ");
                }
                System.out.println("]");
                System.out.println();
            }
            System.err.println("Done");
            return 0;
        }

        private Map<Long, Integer> inspectLedger(LedgerMetadata metadata, Set<BookieSocketAddress> bookiesToInspect) {
            TreeMap<Long, Integer> numBookiesToReplacePerEnsemble = new TreeMap<Long, Integer>();
            for (Map.Entry<Long, ArrayList<BookieSocketAddress>> ensemble : metadata.getEnsembles().entrySet()) {
                ArrayList<BookieSocketAddress> bookieList = ensemble.getValue();
                System.out.print(ensemble.getKey() + ":\t");
                int numBookiesToReplace = 0;
                for (BookieSocketAddress bookie : bookieList) {
                    System.out.print(bookie);
                    if (bookiesToInspect.contains(bookie)) {
                        System.out.print("*");
                        ++numBookiesToReplace;
                    } else {
                        System.out.print(" ");
                    }
                    System.out.print(" ");
                }
                System.out.println();
                numBookiesToReplacePerEnsemble.put(ensemble.getKey(), numBookiesToReplace);
            }
            return numBookiesToReplacePerEnsemble;
        }

        private int bkRecoveryLedger(BookKeeperAdmin bkAdmin, long lid, Set<BookieSocketAddress> bookieAddrs, boolean dryrun, boolean skipOpenLedgers, boolean removeCookies) throws InterruptedException, BKException, KeeperException {
            bkAdmin.recoverBookieData(lid, bookieAddrs, dryrun, skipOpenLedgers);
            if (removeCookies) {
                this.deleteCookies(bkAdmin.getConf(), bookieAddrs);
            }
            return 0;
        }

        private int bkRecovery(BookKeeperAdmin bkAdmin, Set<BookieSocketAddress> bookieAddrs, boolean dryrun, boolean skipOpenLedgers, boolean removeCookies) throws InterruptedException, BKException, KeeperException {
            bkAdmin.recoverBookieData(bookieAddrs, dryrun, skipOpenLedgers);
            if (removeCookies) {
                this.deleteCookies(bkAdmin.getConf(), bookieAddrs);
            }
            return 0;
        }

        private void deleteCookies(ClientConfiguration conf, Set<BookieSocketAddress> bookieAddrs) throws BKException {
            ServerConfiguration serverConf = new ServerConfiguration(conf);
            try (ZKRegistrationManager rm = new ZKRegistrationManager();){
                rm.initialize(serverConf, () -> {}, (StatsLogger)NullStatsLogger.INSTANCE);
                for (BookieSocketAddress addr : bookieAddrs) {
                    this.deleteCookie(rm, addr);
                }
            }
        }

        private void deleteCookie(RegistrationManager rm, BookieSocketAddress bookieSrc) throws BookieException {
            try {
                Versioned<Cookie> cookie = Cookie.readFromRegistrationManager(rm, bookieSrc);
                cookie.getValue().deleteFromRegistrationManager(rm, bookieSrc, cookie.getVersion());
            }
            catch (BookieException.CookieNotFoundException nne) {
                LOG.warn("No cookie to remove for {} : ", (Object)bookieSrc, (Object)nne);
            }
        }
    }

    class BookieFormatCmd
    extends MyCommand {
        Options opts;

        public BookieFormatCmd() {
            super(BookieShell.CMD_BOOKIEFORMAT);
            this.opts = new Options();
            this.opts.addOption("n", "nonInteractive", false, "Whether to confirm if old data exists..?");
            this.opts.addOption("f", "force", false, "If [nonInteractive] is specified, then whether to force delete the old data without prompt..?");
            this.opts.addOption("d", "deleteCookie", false, "Delete its cookie on zookeeper");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Format the current server contents.";
        }

        @Override
        String getUsage() {
            return "bookieformat [-nonInteractive] [-force] [-deleteCookie]";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            boolean interactive = !cmdLine.hasOption("n");
            boolean force = cmdLine.hasOption("f");
            ServerConfiguration conf = new ServerConfiguration(BookieShell.this.bkConf);
            boolean result = Bookie.format(conf, interactive, force);
            if (cmdLine.hasOption("d")) {
                ZKRegistrationManager rm = new ZKRegistrationManager();
                rm.initialize(conf, () -> {}, (StatsLogger)NullStatsLogger.INSTANCE);
                try {
                    Versioned<Cookie> cookie = Cookie.readFromRegistrationManager((RegistrationManager)rm, conf);
                    cookie.getValue().deleteFromRegistrationManager((RegistrationManager)rm, conf, cookie.getVersion());
                }
                catch (BookieException.CookieNotFoundException nne) {
                    LOG.warn("No cookie to remove : ", (Throwable)nne);
                }
                finally {
                    rm.close();
                }
            }
            return result ? 0 : 1;
        }
    }

    class MetaFormatCmd
    extends MyCommand {
        Options opts;

        MetaFormatCmd() {
            super(BookieShell.CMD_METAFORMAT);
            this.opts = new Options();
            this.opts.addOption("n", "nonInteractive", false, "Whether to confirm if old data exists..?");
            this.opts.addOption("f", "force", false, "If [nonInteractive] is specified, then whether to force delete the old data without prompt.");
        }

        @Override
        Options getOptions() {
            return this.opts;
        }

        @Override
        String getDescription() {
            return "Format bookkeeper metadata in zookeeper.";
        }

        @Override
        String getUsage() {
            return "metaformat   [-nonInteractive] [-force]";
        }

        @Override
        int runCmd(CommandLine cmdLine) throws Exception {
            boolean interactive = !cmdLine.hasOption("n");
            boolean force = cmdLine.hasOption("f");
            ClientConfiguration adminConf = new ClientConfiguration(BookieShell.this.bkConf);
            boolean result = BookKeeperAdmin.format(adminConf, interactive, force);
            return result ? 0 : 1;
        }
    }

    abstract class MyCommand
    implements Command {
        String cmdName;

        abstract Options getOptions();

        abstract String getDescription();

        abstract String getUsage();

        abstract int runCmd(CommandLine var1) throws Exception;

        MyCommand(String cmdName) {
            this.cmdName = cmdName;
        }

        @Override
        public int runCmd(String[] args) throws Exception {
            try {
                BasicParser parser = new BasicParser();
                CommandLine cmdLine = parser.parse(this.getOptions(), args);
                return this.runCmd(cmdLine);
            }
            catch (ParseException e) {
                LOG.error("Error parsing command line arguments : ", (Throwable)e);
                this.printUsage();
                return -1;
            }
        }

        @Override
        public void printUsage() {
            HelpFormatter hf = new HelpFormatter();
            System.err.println(this.cmdName + ": " + this.getDescription());
            hf.printHelp(this.getUsage(), this.getOptions());
        }
    }

    static interface Command {
        public int runCmd(String[] var1) throws Exception;

        public void printUsage();
    }
}

