/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.CacheCallback;
import org.apache.bookkeeper.bookie.CheckpointSource;
import org.apache.bookkeeper.bookie.EntryKey;
import org.apache.bookkeeper.bookie.EntryKeyValue;
import org.apache.bookkeeper.bookie.Journal;
import org.apache.bookkeeper.bookie.SkipListArena;
import org.apache.bookkeeper.bookie.SkipListFlusher;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.MathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryMemTable {
    private static Logger logger = LoggerFactory.getLogger(Journal.class);
    volatile EntrySkipList kvmap;
    volatile EntrySkipList snapshot;
    final ServerConfiguration conf;
    final CheckpointSource checkpointSource;
    final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    final AtomicLong size;
    final long skipListSizeLimit;
    SkipListArena allocator;
    private final AtomicBoolean previousFlushSucceeded;
    private final OpStatsLogger snapshotStats;
    private final OpStatsLogger putEntryStats;
    private final OpStatsLogger getEntryStats;
    private final Counter flushBytesCounter;
    private final Counter throttlingCounter;

    private EntrySkipList newSkipList() {
        return new EntrySkipList(this.checkpointSource.newCheckpoint());
    }

    public EntryMemTable(ServerConfiguration conf, CheckpointSource source, StatsLogger statsLogger) {
        this.checkpointSource = source;
        this.kvmap = this.newSkipList();
        this.snapshot = EntrySkipList.EMPTY_VALUE;
        this.conf = conf;
        this.size = new AtomicLong(0L);
        this.allocator = new SkipListArena(conf);
        this.previousFlushSucceeded = new AtomicBoolean(true);
        this.skipListSizeLimit = conf.getSkipListSizeLimit();
        this.snapshotStats = statsLogger.getOpStatsLogger("SKIP_LIST_SNAPSHOT");
        this.putEntryStats = statsLogger.getOpStatsLogger("SKIP_LIST_PUT_ENTRY");
        this.getEntryStats = statsLogger.getOpStatsLogger("SKIP_LIST_GET_ENTRY");
        this.flushBytesCounter = statsLogger.getCounter("SKIP_LIST_FLUSH_BYTES");
        this.throttlingCounter = statsLogger.getCounter("SKIP_LIST_THROTTLING");
    }

    void dump() {
        for (EntryKey key : this.kvmap.keySet()) {
            logger.info(key.toString());
        }
        for (EntryKey key : this.snapshot.keySet()) {
            logger.info(key.toString());
        }
    }

    CheckpointSource.Checkpoint snapshot() throws IOException {
        return this.snapshot(CheckpointSource.Checkpoint.MAX);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CheckpointSource.Checkpoint snapshot(CheckpointSource.Checkpoint oldCp) throws IOException {
        CheckpointSource.Checkpoint cp = null;
        if (this.snapshot.isEmpty() && this.kvmap.compareTo(oldCp) < 0) {
            long startTimeNanos = MathUtils.nowInNano();
            this.lock.writeLock().lock();
            try {
                if (this.snapshot.isEmpty() && !this.kvmap.isEmpty() && this.kvmap.compareTo(oldCp) < 0) {
                    this.snapshot = this.kvmap;
                    this.kvmap = this.newSkipList();
                    cp = this.kvmap.cp;
                    this.size.set(0L);
                    this.allocator = new SkipListArena(this.conf);
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
            if (null != cp) {
                this.snapshotStats.registerSuccessfulEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            } else {
                this.snapshotStats.registerFailedEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            }
        }
        return cp;
    }

    long flush(SkipListFlusher flusher) throws IOException {
        try {
            long flushSize = this.flushSnapshot(flusher, CheckpointSource.Checkpoint.MAX);
            this.previousFlushSucceeded.set(true);
            return flushSize;
        }
        catch (IOException ioe) {
            this.previousFlushSucceeded.set(false);
            throw ioe;
        }
    }

    public long flush(SkipListFlusher flusher, CheckpointSource.Checkpoint checkpoint) throws IOException {
        try {
            long size = this.flushSnapshot(flusher, checkpoint);
            if (null != this.snapshot(checkpoint)) {
                size += this.flushSnapshot(flusher, checkpoint);
            }
            this.previousFlushSucceeded.set(true);
            return size;
        }
        catch (IOException ioe) {
            this.previousFlushSucceeded.set(false);
            throw ioe;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long flushSnapshot(SkipListFlusher flusher, CheckpointSource.Checkpoint checkpoint) throws IOException {
        long size = 0L;
        if (this.snapshot.compareTo(checkpoint) < 0) {
            long ledgerGC = -1L;
            EntryMemTable entryMemTable = this;
            synchronized (entryMemTable) {
                EntrySkipList keyValues = this.snapshot;
                if (keyValues.compareTo(checkpoint) < 0) {
                    for (EntryKey key : keyValues.keySet()) {
                        EntryKeyValue kv = (EntryKeyValue)key;
                        size += (long)kv.getLength();
                        long ledger = kv.getLedgerId();
                        if (ledgerGC == ledger) continue;
                        try {
                            flusher.process(ledger, kv.getEntryId(), kv.getValueAsByteBuffer().nioBuffer());
                        }
                        catch (Bookie.NoLedgerException exception) {
                            ledgerGC = ledger;
                        }
                    }
                    this.flushBytesCounter.add(size);
                    this.clearSnapshot(keyValues);
                }
            }
        }
        return size;
    }

    private void clearSnapshot(EntrySkipList keyValues) {
        assert (!keyValues.isEmpty());
        this.lock.writeLock().lock();
        try {
            assert (this.snapshot == keyValues);
            this.snapshot = EntrySkipList.EMPTY_VALUE;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void throttleWriters() {
        try {
            Thread.sleep(1L);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.throttlingCounter.inc();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long addEntry(long ledgerId, long entryId, ByteBuffer entry, CacheCallback cb) throws IOException {
        long size = 0L;
        long startTimeNanos = MathUtils.nowInNano();
        boolean success = false;
        try {
            if (this.isSizeLimitReached() || !this.previousFlushSucceeded.get()) {
                CheckpointSource.Checkpoint cp = this.snapshot();
                if (null != cp || !this.previousFlushSucceeded.get()) {
                    cb.onSizeLimitReached(cp);
                } else {
                    this.throttleWriters();
                }
            }
            this.lock.readLock().lock();
            try {
                EntryKeyValue toAdd = this.cloneWithAllocator(ledgerId, entryId, entry);
                size = this.internalAdd(toAdd);
            }
            finally {
                this.lock.readLock().unlock();
            }
            success = true;
            long l = size;
            return l;
        }
        finally {
            if (success) {
                this.putEntryStats.registerSuccessfulEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            } else {
                this.putEntryStats.registerFailedEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            }
        }
    }

    private long internalAdd(EntryKeyValue toAdd) throws IOException {
        long sizeChange = 0L;
        if (this.kvmap.putIfAbsent(toAdd, toAdd) == null) {
            sizeChange = toAdd.getLength();
            this.size.addAndGet(sizeChange);
        }
        return sizeChange;
    }

    private EntryKeyValue newEntry(long ledgerId, long entryId, ByteBuffer entry) {
        byte[] buf;
        int offset = 0;
        int length = entry.remaining();
        if (entry.hasArray()) {
            buf = entry.array();
            offset = entry.arrayOffset();
        } else {
            buf = new byte[length];
            entry.get(buf);
        }
        return new EntryKeyValue(ledgerId, entryId, buf, offset, length);
    }

    private EntryKeyValue cloneWithAllocator(long ledgerId, long entryId, ByteBuffer entry) {
        int len = entry.remaining();
        SkipListArena.MemorySlice alloc = this.allocator.allocateBytes(len);
        if (alloc == null) {
            return this.newEntry(ledgerId, entryId, entry);
        }
        assert (alloc.getData() != null);
        entry.get(alloc.getData(), alloc.getOffset(), len);
        return new EntryKeyValue(ledgerId, entryId, alloc.getData(), alloc.getOffset(), len);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public EntryKeyValue getEntry(long ledgerId, long entryId) throws IOException {
        EntryKey key = new EntryKey(ledgerId, entryId);
        EntryKeyValue value = null;
        long startTimeNanos = MathUtils.nowInNano();
        boolean success = false;
        this.lock.readLock().lock();
        try {
            value = (EntryKeyValue)this.kvmap.get(key);
            if (value == null) {
                value = (EntryKeyValue)this.snapshot.get(key);
            }
            success = true;
        }
        finally {
            this.lock.readLock().unlock();
            if (success) {
                this.getEntryStats.registerSuccessfulEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            } else {
                this.getEntryStats.registerFailedEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            }
        }
        return value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public EntryKeyValue getLastEntry(long ledgerId) throws IOException {
        EntryKey result = null;
        EntryKey key = new EntryKey(ledgerId, Long.MAX_VALUE);
        long startTimeNanos = MathUtils.nowInNano();
        boolean success = false;
        this.lock.readLock().lock();
        try {
            result = this.kvmap.floorKey(key);
            if (result == null || result.getLedgerId() != ledgerId) {
                result = this.snapshot.floorKey(key);
            }
            success = true;
        }
        finally {
            this.lock.readLock().unlock();
            if (success) {
                this.getEntryStats.registerSuccessfulEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            } else {
                this.getEntryStats.registerFailedEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
            }
        }
        if (result == null || result.getLedgerId() != ledgerId) {
            return null;
        }
        return (EntryKeyValue)result;
    }

    boolean isSizeLimitReached() {
        return this.size.get() >= this.skipListSizeLimit;
    }

    boolean isEmpty() {
        return this.size.get() == 0L && this.snapshot.isEmpty();
    }

    static class EntrySkipList
    extends ConcurrentSkipListMap<EntryKey, EntryKeyValue> {
        final CheckpointSource.Checkpoint cp;
        static final EntrySkipList EMPTY_VALUE = new EntrySkipList(CheckpointSource.Checkpoint.MAX){

            @Override
            public boolean isEmpty() {
                return true;
            }
        };

        EntrySkipList(CheckpointSource.Checkpoint cp) {
            super(EntryKey.COMPARATOR);
            this.cp = cp;
        }

        int compareTo(CheckpointSource.Checkpoint cp) {
            return this.cp.compareTo(cp);
        }

        @Override
        public EntryKeyValue put(EntryKey k, EntryKeyValue v) {
            return this.putIfAbsent(k, v);
        }

        @Override
        public EntryKeyValue putIfAbsent(EntryKey k, EntryKeyValue v) {
            assert (k.equals(v));
            return super.putIfAbsent(v, v);
        }

        @Override
        public boolean equals(Object o) {
            return this == o;
        }
    }
}

