/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.bookie.AbstractLogCompactor;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.EntryLogCompactor;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.bookie.GarbageCollector;
import org.apache.bookkeeper.bookie.ScanAndCompareGarbageCollector;
import org.apache.bookkeeper.bookie.TransactionalEntryLogCompactor;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.SafeRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GarbageCollectorThread
extends SafeRunnable {
    private static final Logger LOG = LoggerFactory.getLogger(GarbageCollectorThread.class);
    private static final int SECOND = 1000;
    private Map<Long, EntryLogMetadata> entryLogMetaMap = new ConcurrentHashMap<Long, EntryLogMetadata>();
    private final ScheduledExecutorService gcExecutor;
    Future<?> scheduledFuture = null;
    final long gcWaitTime;
    boolean enableMinorCompaction = false;
    final double minorCompactionThreshold;
    final long minorCompactionInterval;
    long lastMinorCompactionTime;
    boolean enableMajorCompaction = false;
    final double majorCompactionThreshold;
    final long majorCompactionInterval;
    long lastMajorCompactionTime;
    final boolean isForceGCAllowWhenNoSpace;
    final EntryLogger entryLogger;
    final AbstractLogCompactor compactor;
    final CompactableLedgerStorage ledgerStorage;
    final AtomicBoolean compacting = new AtomicBoolean(false);
    volatile boolean running = true;
    long scannedLogId = 0L;
    final AtomicBoolean forceGarbageCollection = new AtomicBoolean(false);
    final AtomicBoolean suspendMajorCompaction = new AtomicBoolean(false);
    final AtomicBoolean suspendMinorCompaction = new AtomicBoolean(false);
    final GarbageCollector garbageCollector;
    final GarbageCollector.GarbageCleaner garbageCleaner;
    final ServerConfiguration conf;

    public GarbageCollectorThread(ServerConfiguration conf, LedgerManager ledgerManager, CompactableLedgerStorage ledgerStorage) throws IOException {
        this.gcExecutor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("GarbageCollectorThread-%d").build());
        this.conf = conf;
        this.entryLogger = ledgerStorage.getEntryLogger();
        this.ledgerStorage = ledgerStorage;
        this.gcWaitTime = conf.getGcWaitTime();
        this.garbageCleaner = ledgerId -> {
            try {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("delete ledger : " + ledgerId);
                }
                ledgerStorage.deleteLedger(ledgerId);
            }
            catch (IOException e) {
                LOG.error("Exception when deleting the ledger index file on the Bookie: ", (Throwable)e);
            }
        };
        this.garbageCollector = new ScanAndCompareGarbageCollector(ledgerManager, ledgerStorage, conf);
        this.minorCompactionThreshold = conf.getMinorCompactionThreshold();
        this.minorCompactionInterval = conf.getMinorCompactionInterval() * 1000L;
        this.majorCompactionThreshold = conf.getMajorCompactionThreshold();
        this.majorCompactionInterval = conf.getMajorCompactionInterval() * 1000L;
        this.isForceGCAllowWhenNoSpace = conf.getIsForceGCAllowWhenNoSpace();
        this.compactor = conf.getUseTransactionalCompaction() ? new TransactionalEntryLogCompactor(this) : new EntryLogCompactor(this);
        if (this.minorCompactionInterval > 0L && this.minorCompactionThreshold > 0.0) {
            if (this.minorCompactionThreshold > 1.0) {
                throw new IOException("Invalid minor compaction threshold " + this.minorCompactionThreshold);
            }
            if (this.minorCompactionInterval <= this.gcWaitTime) {
                throw new IOException("Too short minor compaction interval : " + this.minorCompactionInterval);
            }
            this.enableMinorCompaction = true;
        }
        if (this.majorCompactionInterval > 0L && this.majorCompactionThreshold > 0.0) {
            if (this.majorCompactionThreshold > 1.0) {
                throw new IOException("Invalid major compaction threshold " + this.majorCompactionThreshold);
            }
            if (this.majorCompactionInterval <= this.gcWaitTime) {
                throw new IOException("Too short major compaction interval : " + this.majorCompactionInterval);
            }
            this.enableMajorCompaction = true;
        }
        if (this.enableMinorCompaction && this.enableMajorCompaction && (this.minorCompactionInterval >= this.majorCompactionInterval || this.minorCompactionThreshold >= this.majorCompactionThreshold)) {
            throw new IOException("Invalid minor/major compaction settings : minor (" + this.minorCompactionThreshold + ", " + this.minorCompactionInterval + "), major (" + this.majorCompactionThreshold + ", " + this.majorCompactionInterval + ")");
        }
        LOG.info("Minor Compaction : enabled=" + this.enableMinorCompaction + ", threshold=" + this.minorCompactionThreshold + ", interval=" + this.minorCompactionInterval);
        LOG.info("Major Compaction : enabled=" + this.enableMajorCompaction + ", threshold=" + this.majorCompactionThreshold + ", interval=" + this.majorCompactionInterval);
        this.lastMinorCompactionTime = this.lastMajorCompactionTime = MathUtils.now();
    }

    public void enableForceGC() {
        if (this.forceGarbageCollection.compareAndSet(false, true)) {
            LOG.info("Forced garbage collection triggered by thread: {}", (Object)Thread.currentThread().getName());
            this.triggerGC(true, this.suspendMajorCompaction.get(), this.suspendMinorCompaction.get());
        }
    }

    public void disableForceGC() {
        if (this.forceGarbageCollection.compareAndSet(true, false)) {
            LOG.info("{} disabled force garbage collection since bookie has enough space now.", (Object)Thread.currentThread().getName());
        }
    }

    Future<?> triggerGC(boolean force, boolean suspendMajor, boolean suspendMinor) {
        return this.gcExecutor.submit(() -> this.runWithFlags(force, suspendMajor, suspendMinor));
    }

    Future<?> triggerGC() {
        boolean force = this.forceGarbageCollection.get();
        boolean suspendMajor = this.suspendMajorCompaction.get();
        boolean suspendMinor = this.suspendMinorCompaction.get();
        return this.gcExecutor.submit(() -> this.runWithFlags(force, suspendMajor, suspendMinor));
    }

    public void suspendMajorGC() {
        if (this.suspendMajorCompaction.compareAndSet(false, true)) {
            LOG.info("Suspend Major Compaction triggered by thread: {}", (Object)Thread.currentThread().getName());
        }
    }

    public void resumeMajorGC() {
        if (this.suspendMajorCompaction.compareAndSet(true, false)) {
            LOG.info("{} Major Compaction back to normal since bookie has enough space now.", (Object)Thread.currentThread().getName());
        }
    }

    public void suspendMinorGC() {
        if (this.suspendMinorCompaction.compareAndSet(false, true)) {
            LOG.info("Suspend Minor Compaction triggered by thread: {}", (Object)Thread.currentThread().getName());
        }
    }

    public void resumeMinorGC() {
        if (this.suspendMinorCompaction.compareAndSet(true, false)) {
            LOG.info("{} Minor Compaction back to normal since bookie has enough space now.", (Object)Thread.currentThread().getName());
        }
    }

    public void start() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(false);
        }
        this.scheduledFuture = this.gcExecutor.scheduleAtFixedRate((Runnable)((Object)this), this.gcWaitTime, this.gcWaitTime, TimeUnit.MILLISECONDS);
    }

    public void safeRun() {
        boolean force = this.forceGarbageCollection.get();
        boolean suspendMajor = this.suspendMajorCompaction.get();
        boolean suspendMinor = this.suspendMinorCompaction.get();
        this.runWithFlags(force, suspendMajor, suspendMinor);
        if (force) {
            this.forceGarbageCollection.set(false);
        }
    }

    public void runWithFlags(boolean force, boolean suspendMajor, boolean suspendMinor) {
        if (force) {
            LOG.info("Garbage collector thread forced to perform GC before expiry of wait time.");
        }
        this.compactor.cleanUpAndRecover();
        this.entryLogMetaMap = this.extractMetaFromEntryLogs(this.entryLogMetaMap);
        this.doGcLedgers();
        this.doGcEntryLogs();
        if (suspendMajor) {
            LOG.info("Disk almost full, suspend major compaction to slow down filling disk.");
        }
        if (suspendMinor) {
            LOG.info("Disk full, suspend minor compaction to slow down filling disk.");
        }
        long curTime = MathUtils.now();
        if (this.enableMajorCompaction && !suspendMajor && (force || curTime - this.lastMajorCompactionTime > this.majorCompactionInterval)) {
            LOG.info("Enter major compaction, suspendMajor {}", (Object)suspendMajor);
            this.doCompactEntryLogs(this.majorCompactionThreshold);
            this.lastMinorCompactionTime = this.lastMajorCompactionTime = MathUtils.now();
            return;
        }
        if (this.enableMinorCompaction && !suspendMinor && (force || curTime - this.lastMinorCompactionTime > this.minorCompactionInterval)) {
            LOG.info("Enter minor compaction, suspendMinor {}", (Object)suspendMinor);
            this.doCompactEntryLogs(this.minorCompactionThreshold);
            this.lastMinorCompactionTime = MathUtils.now();
        }
    }

    private void doGcLedgers() {
        this.garbageCollector.gc(this.garbageCleaner);
    }

    private void doGcEntryLogs() {
        for (Map.Entry<Long, EntryLogMetadata> entry : this.entryLogMetaMap.entrySet()) {
            long entryLogId = entry.getKey();
            EntryLogMetadata meta = entry.getValue();
            for (Long entryLogLedger : meta.getLedgersMap().keySet()) {
                try {
                    if (this.ledgerStorage.ledgerExists(entryLogLedger)) continue;
                    meta.removeLedger(entryLogLedger);
                }
                catch (IOException e) {
                    LOG.error("Error reading from ledger storage", (Throwable)e);
                }
            }
            if (!meta.isEmpty()) continue;
            LOG.info("Deleting entryLogId " + entryLogId + " as it has no active ledgers!");
            this.removeEntryLog(entryLogId);
        }
    }

    @VisibleForTesting
    void doCompactEntryLogs(double threshold) {
        EntryLogMetadata meta;
        LOG.info("Do compaction to compact those files lower than {}", (Object)threshold);
        ArrayList<EntryLogMetadata> logsToCompact = new ArrayList<EntryLogMetadata>();
        logsToCompact.addAll(this.entryLogMetaMap.values());
        logsToCompact.sort(Comparator.comparing(EntryLogMetadata::getUsage));
        Iterator iterator = logsToCompact.iterator();
        while (iterator.hasNext() && !((meta = (EntryLogMetadata)iterator.next()).getUsage() >= threshold)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Compacting entry log {} below threshold {}", (Object)meta.getEntryLogId(), (Object)threshold);
            }
            this.compactEntryLog(meta);
            if (this.running) continue;
            return;
        }
    }

    public void shutdown() throws InterruptedException {
        this.running = false;
        LOG.info("Shutting down GarbageCollectorThread");
        while (!this.compacting.compareAndSet(false, true)) {
            Thread.sleep(100L);
        }
        this.gcExecutor.shutdownNow();
    }

    protected void removeEntryLog(long entryLogId) {
        if (this.entryLogger.removeEntryLog(entryLogId)) {
            LOG.info("Removing entry log metadata for {}", (Object)entryLogId);
            this.entryLogMetaMap.remove(entryLogId);
        }
    }

    protected void compactEntryLog(EntryLogMetadata entryLogMeta) {
        if (!this.compacting.compareAndSet(false, true)) {
            return;
        }
        this.compactor.compact(entryLogMeta);
        this.compacting.set(false);
    }

    protected Map<Long, EntryLogMetadata> extractMetaFromEntryLogs(Map<Long, EntryLogMetadata> entryLogMetaMap) {
        long curLogId = this.entryLogger.getLeastUnflushedLogId();
        boolean hasExceptionWhenScan = false;
        for (long entryLogId = this.scannedLogId; entryLogId < curLogId; ++entryLogId) {
            if (entryLogMetaMap.containsKey(entryLogId) || !this.entryLogger.logExists(entryLogId)) continue;
            LOG.info("Extracting entry log meta from entryLogId: {}", (Object)entryLogId);
            try {
                EntryLogMetadata entryLogMeta = this.entryLogger.getEntryLogMetadata(entryLogId);
                entryLogMetaMap.put(entryLogId, entryLogMeta);
            }
            catch (IOException e) {
                hasExceptionWhenScan = true;
                LOG.warn("Premature exception when processing " + entryLogId + " recovery will take care of the problem", (Throwable)e);
            }
            if (hasExceptionWhenScan) continue;
            ++this.scannedLogId;
        }
        return entryLogMetaMap;
    }

    EntryLogger getEntryLogger() {
        return this.entryLogger;
    }

    CompactableLedgerStorage getLedgerStorage() {
        return this.ledgerStorage;
    }
}

