/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.GarbageCollector;
import org.apache.bookkeeper.client.LedgerMetadata;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.meta.ZkLedgerUnderreplicationManager;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScanAndCompareGarbageCollector
implements GarbageCollector {
    static final Logger LOG = LoggerFactory.getLogger(ScanAndCompareGarbageCollector.class);
    static final int MAX_CONCURRENT_ZK_REQUESTS = 1000;
    private final LedgerManager ledgerManager;
    private final CompactableLedgerStorage ledgerStorage;
    private final ServerConfiguration conf;
    private final BookieSocketAddress selfBookieAddress;
    private ZooKeeper zk = null;
    private boolean enableGcOverReplicatedLedger;
    private final long gcOverReplicatedLedgerIntervalMillis;
    private long lastOverReplicatedLedgerGcTimeMillis;
    private final String zkLedgersRootPath;

    public ScanAndCompareGarbageCollector(LedgerManager ledgerManager, CompactableLedgerStorage ledgerStorage, ServerConfiguration conf) throws IOException {
        this.ledgerManager = ledgerManager;
        this.ledgerStorage = ledgerStorage;
        this.conf = conf;
        this.selfBookieAddress = Bookie.getBookieAddress(conf);
        this.gcOverReplicatedLedgerIntervalMillis = conf.getGcOverreplicatedLedgerWaitTimeMillis();
        this.lastOverReplicatedLedgerGcTimeMillis = MathUtils.now();
        if (this.gcOverReplicatedLedgerIntervalMillis > 0L) {
            this.enableGcOverReplicatedLedger = true;
        }
        this.zkLedgersRootPath = conf.getZkLedgersRootPath();
        LOG.info("Over Replicated Ledger Deletion : enabled=" + this.enableGcOverReplicatedLedger + ", interval=" + this.gcOverReplicatedLedgerIntervalMillis);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void gc(GarbageCollector.GarbageCleaner garbageCleaner) {
        if (null == this.ledgerManager) {
            return;
        }
        try {
            boolean checkOverreplicatedLedgers;
            TreeSet bkActiveLedgers = Sets.newTreeSet(this.ledgerStorage.getActiveLedgersInRange(0L, Long.MAX_VALUE));
            LedgerManager.LedgerRangeIterator ledgerRangeIterator = this.ledgerManager.getLedgerRanges();
            if (!ledgerRangeIterator.hasNext()) {
                Iterator iterator = bkActiveLedgers.iterator();
                while (iterator.hasNext()) {
                    long ledgerId = (Long)iterator.next();
                    garbageCleaner.clean(ledgerId);
                }
            }
            long lastEnd = -1L;
            long curTime = MathUtils.now();
            boolean bl = checkOverreplicatedLedgers = this.enableGcOverReplicatedLedger && curTime - this.lastOverReplicatedLedgerGcTimeMillis > this.gcOverReplicatedLedgerIntervalMillis;
            if (checkOverreplicatedLedgers) {
                this.zk = ZooKeeperClient.newBuilder().connectString(this.conf.getZkServers()).sessionTimeoutMs(this.conf.getZkTimeout()).build();
                Set<Long> overReplicatedLedgers = this.removeOverReplicatedledgers(bkActiveLedgers, garbageCleaner);
                if (overReplicatedLedgers.isEmpty()) {
                    LOG.info("No over-replicated ledgers found.");
                } else {
                    LOG.info("Removed over-replicated ledgers: {}", overReplicatedLedgers);
                }
                this.lastOverReplicatedLedgerGcTimeMillis = MathUtils.now();
            }
            while (ledgerRangeIterator.hasNext()) {
                LedgerManager.LedgerRange lRange = ledgerRangeIterator.next();
                Long start = lastEnd + 1L;
                Long end = lRange.end();
                if (!ledgerRangeIterator.hasNext()) {
                    end = Long.MAX_VALUE;
                }
                NavigableSet<Long> subBkActiveLedgers = bkActiveLedgers.subSet(start, true, end, true);
                Set<Long> ledgersInMetadata = lRange.getLedgers();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Active in metadata {}, Active in bookie {}", ledgersInMetadata, subBkActiveLedgers);
                }
                for (Long bkLid : subBkActiveLedgers) {
                    if (ledgersInMetadata.contains(bkLid)) continue;
                    garbageCleaner.clean(bkLid);
                }
                lastEnd = end;
            }
        }
        catch (Throwable t) {
            LOG.warn("Exception when iterating over the metadata {}", t);
        }
        finally {
            if (this.zk != null) {
                try {
                    this.zk.close();
                }
                catch (InterruptedException e) {
                    LOG.error("Error closing zk session", (Throwable)e);
                }
                this.zk = null;
            }
        }
    }

    private Set<Long> removeOverReplicatedledgers(Set<Long> bkActiveledgers, final GarbageCollector.GarbageCleaner garbageCleaner) throws InterruptedException, KeeperException {
        List<ACL> zkAcls = ZkUtils.getACLs(this.conf);
        final HashSet overReplicatedLedgers = Sets.newHashSet();
        final Semaphore semaphore = new Semaphore(1000);
        final CountDownLatch latch = new CountDownLatch(bkActiveledgers.size());
        for (final Long ledgerId : bkActiveledgers) {
            try {
                if (ZkLedgerUnderreplicationManager.isLedgerBeingReplicated(this.zk, this.zkLedgersRootPath, ledgerId)) {
                    latch.countDown();
                    continue;
                }
                ZkLedgerUnderreplicationManager.acquireUnderreplicatedLedgerLock(this.zk, this.zkLedgersRootPath, ledgerId, zkAcls);
                semaphore.acquire();
                this.ledgerManager.readLedgerMetadata(ledgerId, new BookkeeperInternalCallbacks.GenericCallback<LedgerMetadata>(){

                    @Override
                    public void operationComplete(int rc, LedgerMetadata ledgerMetadata) {
                        if (rc == 0) {
                            if (!ledgerMetadata.isClosed()) {
                                this.release();
                                return;
                            }
                            TreeMap<Long, ArrayList<BookieSocketAddress>> ensembles = ledgerMetadata.getEnsembles();
                            for (ArrayList ensemble : ensembles.values()) {
                                if (!ensemble.contains(ScanAndCompareGarbageCollector.this.selfBookieAddress)) continue;
                                this.release();
                                return;
                            }
                            overReplicatedLedgers.add(ledgerId);
                            garbageCleaner.clean(ledgerId);
                        }
                        this.release();
                    }

                    private void release() {
                        semaphore.release();
                        latch.countDown();
                        try {
                            ZkLedgerUnderreplicationManager.releaseUnderreplicatedLedgerLock(ScanAndCompareGarbageCollector.this.zk, ScanAndCompareGarbageCollector.this.zkLedgersRootPath, ledgerId);
                        }
                        catch (Throwable t) {
                            LOG.error("Exception when removing underreplicated lock for ledger {}", (Object)ledgerId, (Object)t);
                        }
                    }
                });
            }
            catch (Throwable t) {
                LOG.error("Exception when iterating through the ledgers to check for over-replication", t);
                latch.countDown();
            }
        }
        latch.await();
        bkActiveledgers.removeAll(overReplicatedLedgers);
        return overReplicatedLedgers;
    }
}

