/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.discover;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.discover.RegistrationManager;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.bookkeeper.versioning.LongVersion;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.bookkeeper.zookeeper.BoundExponentialBackoffRetryPolicy;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKRegistrationManager
implements RegistrationManager {
    private static final Logger log = LoggerFactory.getLogger(ZKRegistrationManager.class);
    private ServerConfiguration conf;
    private ZooKeeper zk;
    private List<ACL> zkAcls;
    private volatile boolean running = false;
    private String cookiePath;
    protected String bookieRegistrationPath;
    protected String bookieReadonlyRegistrationPath;
    private StatsLogger statsLogger;

    @Override
    public RegistrationManager initialize(ServerConfiguration conf, RegistrationManager.RegistrationListener listener, StatsLogger statsLogger) throws BookieException {
        if (null == conf.getZkServers()) {
            log.warn("No ZK servers passed to Bookie constructor so BookKeeper clients won't know about this server!");
            return null;
        }
        this.conf = conf;
        this.zkAcls = ZkUtils.getACLs(conf);
        this.statsLogger = statsLogger;
        this.cookiePath = conf.getZkLedgersRootPath() + "/" + "cookies";
        this.bookieRegistrationPath = conf.getZkAvailableBookiesPath();
        this.bookieReadonlyRegistrationPath = this.bookieRegistrationPath + "/" + "readonly";
        try {
            this.zk = this.newZookeeper(conf, listener);
        }
        catch (IOException | InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException(e);
        }
        return this;
    }

    @VisibleForTesting
    public void setZk(ZooKeeper zk) {
        this.zk = zk;
    }

    @VisibleForTesting
    public ZooKeeper getZk() {
        return this.zk;
    }

    private ZooKeeper newZookeeper(ServerConfiguration conf, RegistrationManager.RegistrationListener listener) throws InterruptedException, KeeperException, IOException {
        HashSet<Watcher> watchers = new HashSet<Watcher>();
        watchers.add(event -> {
            if (!this.running) {
                return;
            }
            if (event.getType().equals((Object)Watcher.Event.EventType.None) && event.getState().equals((Object)Watcher.Event.KeeperState.Expired)) {
                listener.onRegistrationExpired();
            }
        });
        return ZooKeeperClient.newBuilder().connectString(conf.getZkServers()).sessionTimeoutMs(conf.getZkTimeout()).watchers(watchers).operationRetryPolicy(new BoundExponentialBackoffRetryPolicy(conf.getZkRetryBackoffStartMs(), conf.getZkRetryBackoffMaxMs(), Integer.MAX_VALUE)).requestRateLimit(conf.getZkRequestRateLimit()).statsLogger(this.statsLogger.scope("bookie")).build();
    }

    @Override
    public void close() {
        if (null != this.zk) {
            try {
                this.zk.close();
            }
            catch (InterruptedException e) {
                log.warn("Interrupted on closing zookeeper client", (Throwable)e);
            }
        }
    }

    private String getCookiePath(String bookieId) {
        return this.cookiePath + "/" + bookieId;
    }

    protected boolean checkRegNodeAndWaitExpired(String regPath) throws IOException {
        final CountDownLatch prevNodeLatch = new CountDownLatch(1);
        Watcher zkPrevRegNodewatcher = new Watcher(){

            public void process(WatchedEvent event) {
                if (Watcher.Event.EventType.NodeDeleted == event.getType()) {
                    prevNodeLatch.countDown();
                }
            }
        };
        try {
            Stat stat = this.zk.exists(regPath, zkPrevRegNodewatcher);
            if (null != stat) {
                if (stat.getEphemeralOwner() != this.zk.getSessionId()) {
                    log.info("Previous bookie registration znode: {} exists, so waiting zk sessiontimeout: {} ms for znode deletion", (Object)regPath, (Object)this.conf.getZkTimeout());
                    if (!prevNodeLatch.await(this.conf.getZkTimeout(), TimeUnit.MILLISECONDS)) {
                        throw new KeeperException.NodeExistsException(regPath);
                    }
                    return false;
                }
                return true;
            }
            return false;
        }
        catch (KeeperException ke) {
            log.error("ZK exception checking and wait ephemeral znode {} expired : ", (Object)regPath, (Object)ke);
            throw new IOException("ZK exception checking and wait ephemeral znode " + regPath + " expired", ke);
        }
        catch (InterruptedException ie) {
            log.error("Interrupted checking and wait ephemeral znode {} expired : ", (Object)regPath, (Object)ie);
            throw new IOException("Interrupted checking and wait ephemeral znode " + regPath + " expired", ie);
        }
    }

    @Override
    public void registerBookie(String bookieId, boolean readOnly) throws BookieException {
        if (!readOnly) {
            String regPath = this.bookieRegistrationPath + "/" + bookieId;
            this.doRegisterBookie(regPath);
        } else {
            this.doRegisterReadOnlyBookie(bookieId);
        }
    }

    private void doRegisterBookie(String regPath) throws BookieException {
        try {
            if (!this.checkRegNodeAndWaitExpired(regPath)) {
                this.zk.create(regPath, new byte[0], this.zkAcls, CreateMode.EPHEMERAL);
            }
        }
        catch (KeeperException ke) {
            log.error("ZK exception registering ephemeral Znode for Bookie!", (Throwable)ke);
            throw new BookieException.MetadataStoreException(ke);
        }
        catch (InterruptedException ie) {
            log.error("Interrupted exception registering ephemeral Znode for Bookie!", (Throwable)ie);
            throw new BookieException.MetadataStoreException(ie);
        }
        catch (IOException e) {
            throw new BookieException.MetadataStoreException(e);
        }
    }

    private void doRegisterReadOnlyBookie(String bookieId) throws BookieException {
        try {
            if (null == this.zk.exists(this.bookieReadonlyRegistrationPath, false)) {
                try {
                    this.zk.create(this.bookieReadonlyRegistrationPath, new byte[0], this.zkAcls, CreateMode.PERSISTENT);
                }
                catch (KeeperException.NodeExistsException nodeExistsException) {
                    // empty catch block
                }
            }
            String regPath = this.bookieReadonlyRegistrationPath + "/" + bookieId;
            this.doRegisterBookie(regPath);
            regPath = this.bookieRegistrationPath + "/" + bookieId;
            try {
                this.zk.delete(regPath, -1);
            }
            catch (KeeperException.NoNodeException nne) {
                log.warn("No writable bookie registered node {} when transitioning to readonly", (Object)regPath, (Object)nne);
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException(e);
        }
    }

    @Override
    public void unregisterBookie(String bookieId, boolean readOnly) throws BookieException {
        String regPath = !readOnly ? this.bookieRegistrationPath + "/" + bookieId : this.bookieReadonlyRegistrationPath + "/" + bookieId;
        this.doUnregisterBookie(regPath);
    }

    private void doUnregisterBookie(String regPath) throws BookieException {
        try {
            this.zk.delete(regPath, -1);
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException(e);
        }
    }

    @Override
    public void writeCookie(String bookieId, Versioned<byte[]> cookieData) throws BookieException {
        block7: {
            String zkPath = this.getCookiePath(bookieId);
            try {
                if (Version.NEW == cookieData.getVersion()) {
                    if (this.zk.exists(this.cookiePath, false) == null) {
                        try {
                            this.zk.create(this.cookiePath, new byte[0], this.zkAcls, CreateMode.PERSISTENT);
                        }
                        catch (KeeperException.NodeExistsException nne) {
                            log.info("More than one bookie tried to create {} at once. Safe to ignore.", (Object)this.cookiePath);
                        }
                    }
                    this.zk.create(zkPath, cookieData.getValue(), this.zkAcls, CreateMode.PERSISTENT);
                    break block7;
                }
                if (!(cookieData.getVersion() instanceof LongVersion)) {
                    throw new BookieException.BookieIllegalOpException("Invalid version type, expected it to be LongVersion");
                }
                this.zk.setData(zkPath, cookieData.getValue(), (int)((LongVersion)cookieData.getVersion()).getLongVersion());
            }
            catch (InterruptedException | KeeperException e) {
                throw new BookieException.MetadataStoreException("Failed to write cookie for bookie " + bookieId);
            }
        }
    }

    @Override
    public Versioned<byte[]> readCookie(String bookieId) throws BookieException {
        String zkPath = this.getCookiePath(bookieId);
        try {
            Stat stat = this.zk.exists(zkPath, false);
            byte[] data = this.zk.getData(zkPath, false, stat);
            LongVersion version = new LongVersion(stat.getVersion());
            return new Versioned<byte[]>(data, version);
        }
        catch (KeeperException.NoNodeException nne) {
            throw new BookieException.CookieNotFoundException(bookieId);
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException("Failed to read cookie for bookie " + bookieId);
        }
    }

    @Override
    public void removeCookie(String bookieId, Version version) throws BookieException {
        String zkPath = this.getCookiePath(bookieId);
        try {
            this.zk.delete(zkPath, (int)((LongVersion)version).getLongVersion());
        }
        catch (KeeperException.NoNodeException e) {
            throw new BookieException.CookieNotFoundException(bookieId);
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException("Failed to delete cookie for bookie " + bookieId);
        }
        log.info("Removed cookie from {} for bookie {}.", (Object)this.cookiePath, (Object)bookieId);
    }

    @Override
    public String getClusterInstanceId() throws BookieException {
        String instanceId = null;
        try {
            if (this.zk.exists(this.conf.getZkLedgersRootPath(), null) == null) {
                log.error("BookKeeper metadata doesn't exist in zookeeper. Has the cluster been initialized? Try running bin/bookkeeper shell metaformat");
                throw new KeeperException.NoNodeException("BookKeeper metadata");
            }
            try {
                byte[] data = this.zk.getData(this.conf.getZkLedgersRootPath() + "/" + "INSTANCEID", false, null);
                instanceId = new String(data, StandardCharsets.UTF_8);
            }
            catch (KeeperException.NoNodeException e) {
                log.info("INSTANCEID not exists in zookeeper. Not considering it for data verification");
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException("Failed to get cluster instance id", e);
        }
        return instanceId;
    }
}

