/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.util.DiskChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LedgerDirsMonitor {
    private static final Logger LOG = LoggerFactory.getLogger(LedgerDirsMonitor.class);
    private final int interval;
    private final ServerConfiguration conf;
    private final ConcurrentMap<File, Float> diskUsages;
    private final DiskChecker diskChecker;
    private final LedgerDirsManager ldm;
    private long minUsableSizeForHighPriorityWrites;
    private ScheduledExecutorService executor;
    private ScheduledFuture<?> checkTask;

    public LedgerDirsMonitor(ServerConfiguration conf, DiskChecker diskChecker, LedgerDirsManager ldm) {
        this.interval = conf.getDiskCheckInterval();
        this.minUsableSizeForHighPriorityWrites = conf.getMinUsableSizeForHighPriorityWrites();
        this.conf = conf;
        this.diskChecker = diskChecker;
        this.diskUsages = ldm.getDiskUsages();
        this.ldm = ldm;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void check() {
        try {
            List<File> writableDirs = this.ldm.getWritableLedgerDirs();
            for (File dir : writableDirs) {
                try {
                    this.diskUsages.put(dir, Float.valueOf(this.diskChecker.checkDir(dir)));
                }
                catch (DiskChecker.DiskErrorException diskErrorException) {
                    LOG.error("Ledger directory {} failed on disk checking : ", (Object)dir, (Object)diskErrorException);
                    for (LedgerDirsManager.LedgerDirsListener listener : this.ldm.getListeners()) {
                        listener.diskFailed(dir);
                    }
                }
                catch (DiskChecker.DiskWarnThresholdException diskWarnThresholdException) {
                    this.diskUsages.compute(dir, (d, prevUsage) -> {
                        if (null == prevUsage || diskWarnThresholdException.getUsage() != prevUsage.floatValue()) {
                            LOG.warn("Ledger directory {} is almost full : usage {}", (Object)dir, (Object)Float.valueOf(diskWarnThresholdException.getUsage()));
                        }
                        return Float.valueOf(diskWarnThresholdException.getUsage());
                    });
                    for (LedgerDirsManager.LedgerDirsListener listener : this.ldm.getListeners()) {
                        listener.diskAlmostFull(dir);
                    }
                }
                catch (DiskChecker.DiskOutOfSpaceException diskOutOfSpaceException) {
                    this.diskUsages.compute(dir, (d, prevUsage) -> {
                        if (null == prevUsage || diskOutOfSpaceException.getUsage() != prevUsage.floatValue()) {
                            LOG.error("Ledger directory {} is out-of-space : usage {}", (Object)dir, (Object)Float.valueOf(diskOutOfSpaceException.getUsage()));
                        }
                        return Float.valueOf(diskOutOfSpaceException.getUsage());
                    });
                    this.ldm.addToFilledDirs(dir);
                }
            }
            this.ldm.getWritableLedgerDirs();
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            boolean highPriorityWritesAllowed = true;
            try {
                this.ldm.getDirsAboveUsableThresholdSize(this.minUsableSizeForHighPriorityWrites, false);
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException e1) {
                highPriorityWritesAllowed = false;
            }
            for (LedgerDirsManager.LedgerDirsListener ledgerDirsListener : this.ldm.getListeners()) {
                ledgerDirsListener.allDisksFull(highPriorityWritesAllowed);
            }
        }
        ArrayList<File> fullfilledDirs = new ArrayList<File>(this.ldm.getFullFilledLedgerDirs());
        boolean hasWritableLedgerDirs = this.ldm.hasWritableLedgerDirs();
        float totalDiskUsage = 0.0f;
        try {
            if (!hasWritableLedgerDirs) {
                float f;
                totalDiskUsage = this.diskChecker.getTotalDiskUsage(this.ldm.getAllLedgerDirs());
                if (!(f < this.conf.getDiskLowWaterMarkUsageThreshold())) {
                    LOG.debug("Current TotalDiskUsage: {} is greater than LWMThreshold: {}. So not adding any filledDir to WritableDirsList", (Object)Float.valueOf(totalDiskUsage), (Object)Float.valueOf(this.conf.getDiskLowWaterMarkUsageThreshold()));
                    return;
                }
            }
            Iterator iterator = fullfilledDirs.iterator();
            block18: while (iterator.hasNext()) {
                File dir = (File)iterator.next();
                try {
                    this.diskUsages.put(dir, Float.valueOf(this.diskChecker.checkDir(dir)));
                    this.ldm.addToWritableDirs(dir, true);
                }
                catch (DiskChecker.DiskErrorException e) {
                    Iterator<LedgerDirsManager.LedgerDirsListener> iterator2 = this.ldm.getListeners().iterator();
                    while (true) {
                        if (!iterator2.hasNext()) continue block18;
                        LedgerDirsManager.LedgerDirsListener listener = iterator2.next();
                        listener.diskFailed(dir);
                    }
                }
                catch (DiskChecker.DiskWarnThresholdException e) {
                    this.diskUsages.put(dir, Float.valueOf(e.getUsage()));
                    this.ldm.addToWritableDirs(dir, false);
                }
                catch (DiskChecker.DiskOutOfSpaceException e) {
                    this.diskUsages.put(dir, Float.valueOf(e.getUsage()));
                }
            }
            return;
        }
        catch (IOException iOException) {
            LOG.error("Got IOException while monitoring Dirs", (Throwable)iOException);
            Iterator<LedgerDirsManager.LedgerDirsListener> iterator = this.ldm.getListeners().iterator();
            while (iterator.hasNext()) {
                LedgerDirsManager.LedgerDirsListener listener;
                listener = iterator.next();
                listener.fatalError();
            }
            return;
        }
    }

    public void init() throws DiskChecker.DiskErrorException, LedgerDirsManager.NoWritableLedgerDirException {
        this.checkDirs(this.ldm.getWritableLedgerDirs());
    }

    public void start() {
        this.executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("LedgerDirsMonitorThread").setDaemon(true).build());
        this.checkTask = this.executor.scheduleAtFixedRate(() -> this.check(), this.interval, this.interval, TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        LOG.info("Shutting down LedgerDirsMonitor");
        if (null != this.checkTask && this.checkTask.cancel(true)) {
            LOG.debug("Failed to cancel check task in LedgerDirsMonitor");
        }
        if (null != this.executor) {
            this.executor.shutdown();
        }
    }

    public void checkDirs(List<File> writableDirs) throws DiskChecker.DiskErrorException, LedgerDirsManager.NoWritableLedgerDirException {
        for (File dir : writableDirs) {
            try {
                this.diskChecker.checkDir(dir);
            }
            catch (DiskChecker.DiskWarnThresholdException diskWarnThresholdException) {
            }
            catch (DiskChecker.DiskOutOfSpaceException e) {
                this.ldm.addToFilledDirs(dir);
            }
        }
        this.ldm.getWritableLedgerDirs();
    }
}

