/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.tls;

import com.google.common.base.Strings;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.OpenSsl;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.ssl.SslProvider;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import org.apache.bookkeeper.conf.AbstractConfiguration;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.tls.SecurityException;
import org.apache.bookkeeper.tls.SecurityHandlerFactory;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TLSContextFactory
implements SecurityHandlerFactory {
    private static final Logger LOG = LoggerFactory.getLogger(TLSContextFactory.class);
    private static final String TLSCONTEXT_HANDLER_NAME = "tls";
    private String[] protocols;
    private String[] ciphers;
    private SslContext sslContext;

    private String getPasswordFromFile(String path) throws IOException {
        File passwdFile = new File(path);
        if (passwdFile.length() == 0L) {
            return "";
        }
        byte[] pwd = FileUtils.readFileToByteArray((File)passwdFile);
        return new String(pwd, "UTF-8");
    }

    @SuppressFBWarnings(value={"OBL_UNSATISFIED_OBLIGATION"}, justification="work around for java 9: https://github.com/spotbugs/spotbugs/issues/493")
    private KeyStore loadKeyStore(String keyStoreType, String keyStoreLocation, String keyStorePassword) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
        KeyStore ks = KeyStore.getInstance(keyStoreType);
        try (FileInputStream ksin = new FileInputStream(keyStoreLocation);){
            ks.load(ksin, keyStorePassword.trim().toCharArray());
        }
        return ks;
    }

    @Override
    public String getHandlerName() {
        return TLSCONTEXT_HANDLER_NAME;
    }

    private KeyManagerFactory initKeyManagerFactory(String keyStoreType, String keyStoreLocation, String keyStorePasswordPath) throws SecurityException, KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, UnrecoverableKeyException, InvalidKeySpecException {
        KeyManagerFactory kmf = null;
        if (Strings.isNullOrEmpty((String)keyStoreLocation)) {
            LOG.error("Key store location cannot be empty when Mutual Authentication is enabled!");
            throw new SecurityException("Key store location cannot be empty when Mutual Authentication is enabled!");
        }
        String keyStorePassword = "";
        if (!Strings.isNullOrEmpty((String)keyStorePasswordPath)) {
            keyStorePassword = this.getPasswordFromFile(keyStorePasswordPath);
        }
        KeyStore ks = this.loadKeyStore(keyStoreType, keyStoreLocation, keyStorePassword);
        kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        kmf.init(ks, keyStorePassword.trim().toCharArray());
        return kmf;
    }

    private TrustManagerFactory initTrustManagerFactory(String trustStoreType, String trustStoreLocation, String trustStorePasswordPath) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, SecurityException {
        if (Strings.isNullOrEmpty((String)trustStoreLocation)) {
            LOG.error("Trust Store location cannot be empty!");
            throw new SecurityException("Trust Store location cannot be empty!");
        }
        String trustStorePassword = "";
        if (!Strings.isNullOrEmpty((String)trustStorePasswordPath)) {
            trustStorePassword = this.getPasswordFromFile(trustStorePasswordPath);
        }
        KeyStore ts = this.loadKeyStore(trustStoreType, trustStoreLocation, trustStorePassword);
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        tmf.init(ts);
        return tmf;
    }

    private SslProvider getTLSProvider(String sslProvider) {
        if (sslProvider.trim().equalsIgnoreCase("OpenSSL")) {
            if (OpenSsl.isAvailable()) {
                LOG.info("Security provider - OpenSSL");
                return SslProvider.OPENSSL;
            }
            Throwable causeUnavailable = OpenSsl.unavailabilityCause();
            LOG.warn("OpenSSL Unavailable: ", causeUnavailable);
            LOG.info("Security provider - JDK");
            return SslProvider.JDK;
        }
        LOG.info("Security provider - JDK");
        return SslProvider.JDK;
    }

    private void createClientContext(AbstractConfiguration conf) throws SecurityException, KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, UnrecoverableKeyException, InvalidKeySpecException, NoSuchProviderException {
        SslContextBuilder sslContextBuilder;
        if (!(conf instanceof ClientConfiguration)) {
            throw new SecurityException("Client configruation not provided");
        }
        ClientConfiguration clientConf = (ClientConfiguration)conf;
        SslProvider provider = this.getTLSProvider(clientConf.getTLSProvider());
        boolean clientAuthentication = clientConf.getTLSClientAuthentication();
        switch (KeyStoreType.valueOf(clientConf.getTLSTrustStoreType())) {
            case PEM: {
                if (Strings.isNullOrEmpty((String)clientConf.getTLSTrustStore())) {
                    throw new SecurityException("CA Certificate required");
                }
                sslContextBuilder = SslContextBuilder.forClient().trustManager(new File(clientConf.getTLSTrustStore())).ciphers(null).sessionCacheSize(0L).sessionTimeout(0L).sslProvider(provider).clientAuth(ClientAuth.REQUIRE);
                break;
            }
            case JKS: 
            case PKCS12: {
                TrustManagerFactory tmf = this.initTrustManagerFactory(clientConf.getTLSTrustStoreType(), clientConf.getTLSTrustStore(), clientConf.getTLSTrustStorePasswordPath());
                sslContextBuilder = SslContextBuilder.forClient().trustManager(tmf).ciphers(null).sessionCacheSize(0L).sessionTimeout(0L).sslProvider(provider).clientAuth(ClientAuth.REQUIRE);
                break;
            }
            default: {
                throw new SecurityException("Invalid Truststore type: " + clientConf.getTLSTrustStoreType());
            }
        }
        if (clientAuthentication) {
            switch (KeyStoreType.valueOf(clientConf.getTLSKeyStoreType())) {
                case PEM: {
                    if (Strings.isNullOrEmpty((String)clientConf.getTLSCertificatePath())) {
                        throw new SecurityException("Valid Certificate is missing");
                    }
                    if (Strings.isNullOrEmpty((String)clientConf.getTLSKeyStore())) {
                        throw new SecurityException("Valid Key is missing");
                    }
                    String keyPassword = !Strings.isNullOrEmpty((String)clientConf.getTLSKeyStorePasswordPath()) ? this.getPasswordFromFile(clientConf.getTLSKeyStorePasswordPath()) : null;
                    sslContextBuilder.keyManager(new File(clientConf.getTLSCertificatePath()), new File(clientConf.getTLSKeyStore()), keyPassword);
                    break;
                }
                case JKS: 
                case PKCS12: {
                    KeyManagerFactory kmf = this.initKeyManagerFactory(clientConf.getTLSKeyStoreType(), clientConf.getTLSKeyStore(), clientConf.getTLSKeyStorePasswordPath());
                    sslContextBuilder.keyManager(kmf);
                    break;
                }
                default: {
                    throw new SecurityException("Invalid Keyfile type" + clientConf.getTLSKeyStoreType());
                }
            }
        }
        this.sslContext = sslContextBuilder.build();
    }

    private void createServerContext(AbstractConfiguration conf) throws SecurityException, KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, UnrecoverableKeyException, InvalidKeySpecException, IllegalArgumentException {
        SslContextBuilder sslContextBuilder;
        if (!(conf instanceof ServerConfiguration)) {
            throw new SecurityException("Server configruation not provided");
        }
        ServerConfiguration serverConf = (ServerConfiguration)conf;
        SslProvider provider = this.getTLSProvider(serverConf.getTLSProvider());
        boolean clientAuthentication = serverConf.getTLSClientAuthentication();
        switch (KeyStoreType.valueOf(serverConf.getTLSKeyStoreType())) {
            case PEM: {
                if (Strings.isNullOrEmpty((String)serverConf.getTLSKeyStore())) {
                    throw new SecurityException("Key path is required");
                }
                if (Strings.isNullOrEmpty((String)serverConf.getTLSCertificatePath())) {
                    throw new SecurityException("Certificate path is required");
                }
                String keyPassword = !Strings.isNullOrEmpty((String)serverConf.getTLSKeyStorePasswordPath()) ? this.getPasswordFromFile(serverConf.getTLSKeyStorePasswordPath()) : null;
                sslContextBuilder = SslContextBuilder.forServer((File)new File(serverConf.getTLSCertificatePath()), (File)new File(serverConf.getTLSKeyStore()), (String)keyPassword).ciphers(null).sessionCacheSize(0L).sessionTimeout(0L).sslProvider(provider).startTls(true);
                break;
            }
            case JKS: 
            case PKCS12: {
                KeyManagerFactory kmf = this.initKeyManagerFactory(serverConf.getTLSKeyStoreType(), serverConf.getTLSKeyStore(), serverConf.getTLSKeyStorePasswordPath());
                sslContextBuilder = SslContextBuilder.forServer((KeyManagerFactory)kmf).ciphers(null).sessionCacheSize(0L).sessionTimeout(0L).sslProvider(provider).startTls(true);
                break;
            }
            default: {
                throw new SecurityException("Invalid Keyfile type" + serverConf.getTLSKeyStoreType());
            }
        }
        if (clientAuthentication) {
            sslContextBuilder.clientAuth(ClientAuth.REQUIRE);
            switch (KeyStoreType.valueOf(serverConf.getTLSTrustStoreType())) {
                case PEM: {
                    if (Strings.isNullOrEmpty((String)serverConf.getTLSTrustStore())) {
                        throw new SecurityException("CA Certificate chain is required");
                    }
                    sslContextBuilder.trustManager(new File(serverConf.getTLSTrustStore()));
                    break;
                }
                case JKS: 
                case PKCS12: {
                    TrustManagerFactory tmf = this.initTrustManagerFactory(serverConf.getTLSTrustStoreType(), serverConf.getTLSTrustStore(), serverConf.getTLSTrustStorePasswordPath());
                    sslContextBuilder.trustManager(tmf);
                    break;
                }
                default: {
                    throw new SecurityException("Invalid Truststore type" + serverConf.getTLSTrustStoreType());
                }
            }
        }
        this.sslContext = sslContextBuilder.build();
    }

    @Override
    public synchronized void init(SecurityHandlerFactory.NodeType type, AbstractConfiguration conf) throws SecurityException {
        String enabledCiphers = conf.getTLSEnabledCipherSuites();
        String enabledProtocols = conf.getTLSEnabledProtocols();
        try {
            switch (type) {
                case Client: {
                    this.createClientContext(conf);
                    break;
                }
                case Server: {
                    this.createServerContext(conf);
                    break;
                }
                default: {
                    throw new SecurityException(new IllegalArgumentException("Invalid NodeType"));
                }
            }
            if (enabledProtocols != null && !enabledProtocols.isEmpty()) {
                this.protocols = enabledProtocols.split(",");
            }
            if (enabledCiphers != null && !enabledCiphers.isEmpty()) {
                this.ciphers = enabledCiphers.split(",");
            }
        }
        catch (KeyStoreException e) {
            throw new RuntimeException("Standard keystore type missing", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Standard algorithm missing", e);
        }
        catch (CertificateException e) {
            throw new SecurityException("Unable to load keystore", e);
        }
        catch (IOException e) {
            throw new SecurityException("Error initializing SSLContext", e);
        }
        catch (UnrecoverableKeyException e) {
            throw new SecurityException("Unable to load key manager, possibly bad password", e);
        }
        catch (InvalidKeySpecException e) {
            throw new SecurityException("Unable to load key manager", e);
        }
        catch (IllegalArgumentException e) {
            throw new SecurityException("Invalid TLS configuration", e);
        }
        catch (NoSuchProviderException e) {
            throw new SecurityException("No such provider", e);
        }
    }

    @Override
    public SslHandler newTLSHandler() {
        SslHandler sslHandler = this.sslContext.newHandler((ByteBufAllocator)PooledByteBufAllocator.DEFAULT);
        if (this.protocols != null && this.protocols.length != 0) {
            sslHandler.engine().setEnabledProtocols(this.protocols);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Enabled cipher protocols: {} ", (Object)Arrays.toString(sslHandler.engine().getEnabledProtocols()));
        }
        if (this.ciphers != null && this.ciphers.length != 0) {
            sslHandler.engine().setEnabledCipherSuites(this.ciphers);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Enabled cipher suites: {} ", (Object)Arrays.toString(sslHandler.engine().getEnabledCipherSuites()));
        }
        return sslHandler;
    }

    public static enum KeyStoreType {
        PKCS12("PKCS12"),
        JKS("JKS"),
        PEM("PEM");

        private String str;

        private KeyStoreType(String str) {
            this.str = str;
        }

        public String toString() {
            return this.str;
        }
    }
}

