/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.collect.Sets;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.bookie.InterleavedLedgerStorage;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.LedgerStorage;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.conf.TestBKConfiguration;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryLogTest {
    private static final Logger LOG = LoggerFactory.getLogger(EntryLogTest.class);
    final List<File> tempDirs = new ArrayList<File>();
    private File rootDir;
    private File curDir;
    private ServerConfiguration conf;
    private LedgerDirsManager dirsMgr;
    private EntryLogger entryLogger;

    File createTempDir(String prefix, String suffix) throws IOException {
        File dir = IOUtils.createTempDir((String)prefix, (String)suffix);
        this.tempDirs.add(dir);
        return dir;
    }

    @Before
    public void setUp() throws Exception {
        this.rootDir = this.createTempDir("bkTest", ".dir");
        this.curDir = Bookie.getCurrentDirectory((File)this.rootDir);
        Bookie.checkDirectoryStructure((File)this.curDir);
        this.conf = TestBKConfiguration.newServerConfiguration();
        this.dirsMgr = new LedgerDirsManager(this.conf, new File[]{this.rootDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
    }

    @After
    public void tearDown() throws Exception {
        if (null != this.entryLogger) {
            this.entryLogger.shutdown();
        }
        for (File dir : this.tempDirs) {
            FileUtils.deleteDirectory((File)dir);
        }
        this.tempDirs.clear();
    }

    @Test
    public void testDeferCreateNewLog() throws Exception {
        this.entryLogger.shutdown();
        this.conf.setMinUsableSizeForEntryLogCreation(1L);
        this.dirsMgr = new LedgerDirsManager(this.conf, new File[]{this.rootDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.dirsMgr.addToFilledDirs(this.curDir);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        Assert.assertEquals((long)-57005L, (long)this.entryLogger.getCurrentLogId());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        Assert.assertEquals((long)2L, (long)this.entryLogger.getCurrentLogId());
    }

    @Test
    public void testDeferCreateNewLogWithoutEnoughDiskSpaces() throws Exception {
        this.entryLogger.shutdown();
        this.conf.setMinUsableSizeForEntryLogCreation(Long.MAX_VALUE);
        this.dirsMgr = new LedgerDirsManager(this.conf, new File[]{this.rootDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.dirsMgr.addToFilledDirs(this.curDir);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        Assert.assertEquals((long)-57005L, (long)this.entryLogger.getCurrentLogId());
        try {
            this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
            Assert.fail((String)"Should fail to append entry if there is no enough reserved space left");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            Assert.assertEquals((long)-57005L, (long)this.entryLogger.getCurrentLogId());
        }
    }

    @Test
    public void testCorruptEntryLog() throws Exception {
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        this.entryLogger.addEntry(3L, EntryLogTest.generateEntry(3L, 1L).nioBuffer());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L).nioBuffer());
        this.entryLogger.flush();
        this.entryLogger.shutdown();
        File f = new File(this.curDir, "0.log");
        RandomAccessFile raf = new RandomAccessFile(f, "rw");
        raf.setLength(raf.length() - 10L);
        raf.close();
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        EntryLogMetadata meta = this.entryLogger.getEntryLogMetadata(0L);
        LOG.info("Extracted Meta From Entry Log {}", (Object)meta);
        Assert.assertTrue((boolean)meta.getLedgersMap().containsKey(1L));
        Assert.assertFalse((boolean)meta.getLedgersMap().containsKey(2L));
        Assert.assertTrue((boolean)meta.getLedgersMap().containsKey(3L));
    }

    private static ByteBuf generateEntry(long ledger, long entry) {
        byte[] data = EntryLogTest.generateDataString(ledger, entry).getBytes();
        ByteBuf bb = Unpooled.buffer((int)(16 + data.length));
        bb.writeLong(ledger);
        bb.writeLong(entry);
        bb.writeBytes(data);
        return bb;
    }

    private static String generateDataString(long ledger, long entry) {
        return "ledger-" + ledger + "-" + entry;
    }

    @Test
    public void testMissingLogId() throws Exception {
        int i;
        int numLogs = 3;
        int numEntries = 10;
        long[][] positions = new long[2 * numLogs][];
        for (int i2 = 0; i2 < numLogs; ++i2) {
            positions[i2] = new long[numEntries];
            EntryLogger logger = new EntryLogger(this.conf, this.dirsMgr);
            for (int j = 0; j < numEntries; ++j) {
                positions[i2][j] = logger.addEntry((long)i2, EntryLogTest.generateEntry(i2, j).nioBuffer());
            }
            logger.flush();
            logger.shutdown();
        }
        File lastLogId = new File(this.curDir, "lastId");
        lastLogId.delete();
        for (int i3 = numLogs; i3 < 2 * numLogs; ++i3) {
            positions[i3] = new long[numEntries];
            EntryLogger logger = new EntryLogger(this.conf, this.dirsMgr);
            for (int j = 0; j < numEntries; ++j) {
                positions[i3][j] = logger.addEntry((long)i3, EntryLogTest.generateEntry(i3, j).nioBuffer());
            }
            logger.flush();
            logger.shutdown();
        }
        EntryLogger newLogger = new EntryLogger(this.conf, this.dirsMgr);
        for (i = 0; i < 2 * numLogs + 1; ++i) {
            File logFile = new File(this.curDir, Long.toHexString(i) + ".log");
            Assert.assertTrue((boolean)logFile.exists());
        }
        for (i = 0; i < 2 * numLogs; ++i) {
            for (int j = 0; j < numEntries; ++j) {
                String expectedValue = "ledger-" + i + "-" + j;
                ByteBuf value = newLogger.readEntry((long)i, (long)j, positions[i][j]);
                long ledgerId = value.readLong();
                long entryId = value.readLong();
                byte[] data = new byte[value.readableBytes()];
                value.readBytes(data);
                value.release();
                Assert.assertEquals((long)i, (long)ledgerId);
                Assert.assertEquals((long)j, (long)entryId);
                Assert.assertEquals((Object)expectedValue, (Object)new String(data));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testEntryLoggerShouldThrowFNFEIfDirectoriesDoesNotExist() throws Exception {
        File tmpDir = this.createTempDir("bkTest", ".dir");
        EntryLogger entryLogger = null;
        try {
            entryLogger = new EntryLogger(this.conf, new LedgerDirsManager(this.conf, new File[]{tmpDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold())));
            Assert.fail((String)"Expecting FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            Assert.assertEquals((Object)("Entry log directory '" + tmpDir + "/current' does not exist"), (Object)e.getLocalizedMessage());
        }
        finally {
            if (entryLogger != null) {
                entryLogger.shutdown();
            }
        }
    }

    @Test
    public void testAddEntryFailureOnDiskFull() throws Exception {
        File ledgerDir1 = this.createTempDir("bkTest", ".dir");
        File ledgerDir2 = this.createTempDir("bkTest", ".dir");
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(ledgerDir1.toString());
        conf.setLedgerDirNames(new String[]{ledgerDir1.getAbsolutePath(), ledgerDir2.getAbsolutePath()});
        Bookie bookie = new Bookie(conf);
        EntryLogger entryLogger = new EntryLogger(conf, bookie.getLedgerDirsManager());
        InterleavedLedgerStorage ledgerStorage = (InterleavedLedgerStorage)bookie.ledgerStorage;
        ledgerStorage.entryLogger = entryLogger;
        ledgerStorage.setMasterKey(1L, "key".getBytes());
        ledgerStorage.setMasterKey(2L, "key".getBytes());
        ledgerStorage.setMasterKey(3L, "key".getBytes());
        ledgerStorage.addEntry(EntryLogTest.generateEntry(1L, 1L));
        ledgerStorage.addEntry(EntryLogTest.generateEntry(2L, 1L));
        bookie.getLedgerDirsManager().addToFilledDirs(entryLogger.currentDir);
        ledgerStorage.addEntry(EntryLogTest.generateEntry(3L, 1L));
        Assert.assertTrue((0 == EntryLogTest.generateEntry(1L, 1L).compareTo(ledgerStorage.getEntry(1L, 1L)) ? 1 : 0) != 0);
        Assert.assertTrue((0 == EntryLogTest.generateEntry(2L, 1L).compareTo(ledgerStorage.getEntry(2L, 1L)) ? 1 : 0) != 0);
        Assert.assertTrue((0 == EntryLogTest.generateEntry(3L, 1L).compareTo(ledgerStorage.getEntry(3L, 1L)) ? 1 : 0) != 0);
    }

    @Test
    public void testRecoverFromLedgersMap() throws Exception {
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        this.entryLogger.addEntry(3L, EntryLogTest.generateEntry(3L, 1L).nioBuffer());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L).nioBuffer());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 2L).nioBuffer());
        this.entryLogger.rollLog();
        this.entryLogger.flushRotatedLogs();
        EntryLogMetadata meta = this.entryLogger.extractEntryLogMetadataFromIndex(0L);
        LOG.info("Extracted Meta From Entry Log {}", (Object)meta);
        Assert.assertEquals((long)60L, (long)meta.getLedgersMap().get(1L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(2L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(3L));
        Assert.assertFalse((boolean)meta.getLedgersMap().containsKey(4L));
        Assert.assertEquals((long)120L, (long)meta.getTotalSize());
        Assert.assertEquals((long)120L, (long)meta.getRemainingSize());
    }

    @Test
    public void testRecoverFromLedgersMapOnV0EntryLog() throws Exception {
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        this.entryLogger.addEntry(3L, EntryLogTest.generateEntry(3L, 1L).nioBuffer());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L).nioBuffer());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 2L).nioBuffer());
        this.entryLogger.rollLog();
        this.entryLogger.shutdown();
        File f = new File(this.curDir, "0.log");
        RandomAccessFile raf = new RandomAccessFile(f, "rw");
        raf.seek(4L);
        raf.write(new byte[12]);
        raf.close();
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        try {
            this.entryLogger.extractEntryLogMetadataFromIndex(0L);
            Assert.fail((String)"Should not be possible to recover from ledgers map index");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        EntryLogMetadata meta = this.entryLogger.getEntryLogMetadata(0L);
        LOG.info("Extracted Meta From Entry Log {}", (Object)meta);
        Assert.assertEquals((long)60L, (long)meta.getLedgersMap().get(1L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(2L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(3L));
        Assert.assertFalse((boolean)meta.getLedgersMap().containsKey(4L));
        Assert.assertEquals((long)120L, (long)meta.getTotalSize());
        Assert.assertEquals((long)120L, (long)meta.getRemainingSize());
    }

    @Test
    public void testPreAllocateLog() throws Exception {
        this.entryLogger.shutdown();
        this.conf.setEntryLogFilePreAllocationEnabled(true);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        Assert.assertNotNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        Assert.assertNotNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
        this.entryLogger.shutdown();
        this.conf.setEntryLogFilePreAllocationEnabled(false);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        Assert.assertNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        Assert.assertNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
    }

    @Test
    public void testGetEntryLogsSet() throws Exception {
        Assert.assertEquals((Object)Sets.newHashSet((Object[])new Long[]{0L, 1L}), (Object)this.entryLogger.getEntryLogsSet());
        this.entryLogger.rollLog();
        this.entryLogger.flushRotatedLogs();
        Assert.assertEquals((Object)Sets.newHashSet((Object[])new Long[]{0L, 1L, 2L}), (Object)this.entryLogger.getEntryLogsSet());
        this.entryLogger.rollLog();
        this.entryLogger.flushRotatedLogs();
        Assert.assertEquals((Object)Sets.newHashSet((Object[])new Long[]{0L, 1L, 2L, 3L}), (Object)this.entryLogger.getEntryLogsSet());
    }

    @Test
    public void testConcurrentWriteAndReadCallsOfInterleavedLedgerStorage() throws Exception {
        int i;
        File ledgerDir = this.createTempDir("bkTest", ".dir");
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(ledgerDir.toString());
        conf.setLedgerDirNames(new String[]{ledgerDir.getAbsolutePath()});
        conf.setLedgerStorageClass(InterleavedLedgerStorage.class.getName());
        Bookie bookie = new Bookie(conf);
        InterleavedLedgerStorage ledgerStorage = (InterleavedLedgerStorage)bookie.ledgerStorage;
        Random rand = new Random(0L);
        int numOfLedgers = 70;
        int numEntries = 1500;
        for (int i2 = 0; i2 < numOfLedgers; ++i2) {
            ledgerStorage.setMasterKey((long)i2, "key".getBytes());
        }
        ExecutorService executor = Executors.newFixedThreadPool(10);
        ArrayList<Callable<Boolean>> writeAndFlushTasks = new ArrayList<Callable<Boolean>>();
        for (int j = 0; j < numEntries; ++j) {
            for (i = 0; i < numOfLedgers; ++i) {
                writeAndFlushTasks.add(new LedgerStorageWriteTask(i, j, (LedgerStorage)ledgerStorage));
            }
        }
        for (int i3 = 0; i3 < numOfLedgers * numEntries / 500; ++i3) {
            writeAndFlushTasks.add(rand.nextInt(writeAndFlushTasks.size()), new LedgerStorageFlushTask((LedgerStorage)ledgerStorage));
        }
        executor.invokeAll(writeAndFlushTasks).forEach(future -> {
            try {
                future.get();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                LOG.error("Write/Flush task failed because of InterruptedException", (Throwable)ie);
                Assert.fail((String)"Write/Flush task interrupted");
            }
            catch (Exception ex) {
                LOG.error("Write/Flush task failed because of  exception", (Throwable)ex);
                Assert.fail((String)("Write/Flush task failed " + ex.getMessage()));
            }
        });
        ArrayList<Callable<Boolean>> readAndFlushTasks = new ArrayList<Callable<Boolean>>();
        for (int j = 0; j < numEntries; ++j) {
            for (int i4 = 0; i4 < numOfLedgers; ++i4) {
                readAndFlushTasks.add(new LedgerStorageReadTask(i4, j, (LedgerStorage)ledgerStorage));
            }
        }
        for (i = 0; i < numOfLedgers * numEntries / 500; ++i) {
            readAndFlushTasks.add(rand.nextInt(readAndFlushTasks.size()), new LedgerStorageFlushTask((LedgerStorage)ledgerStorage));
        }
        executor.invokeAll(readAndFlushTasks).forEach(future -> {
            try {
                future.get();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                LOG.error("Read/Flush task failed because of InterruptedException", (Throwable)ie);
                Assert.fail((String)"Read/Flush task interrupted");
            }
            catch (Exception ex) {
                LOG.error("Read/Flush task failed because of  exception", (Throwable)ex);
                Assert.fail((String)("Read/Flush task failed " + ex.getMessage()));
            }
        });
        executor.shutdownNow();
    }

    static class LedgerStorageReadTask
    implements Callable<Boolean> {
        long ledgerId;
        int entryId;
        LedgerStorage ledgerStorage;

        LedgerStorageReadTask(long ledgerId, int entryId, LedgerStorage ledgerStorage) {
            this.ledgerId = ledgerId;
            this.entryId = entryId;
            this.ledgerStorage = ledgerStorage;
        }

        @Override
        public Boolean call() throws IOException {
            try {
                ByteBuf expectedByteBuf = EntryLogTest.generateEntry(this.ledgerId, this.entryId);
                ByteBuf actualByteBuf = this.ledgerStorage.getEntry(this.ledgerId, (long)this.entryId);
                if (!expectedByteBuf.equals((Object)actualByteBuf)) {
                    LOG.error("Expected Entry: {} Actual Entry: {}", (Object)expectedByteBuf.toString(Charset.defaultCharset()), (Object)actualByteBuf.toString(Charset.defaultCharset()));
                    throw new IOException("Expected Entry: " + expectedByteBuf.toString(Charset.defaultCharset()) + " Actual Entry: " + actualByteBuf.toString(Charset.defaultCharset()));
                }
            }
            catch (IOException e) {
                LOG.error("Got Exception for GetEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, (Throwable)e);
                throw new IOException("Got Exception for GetEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, e);
            }
            return true;
        }
    }

    static class LedgerStorageFlushTask
    implements Callable<Boolean> {
        LedgerStorage ledgerStorage;

        LedgerStorageFlushTask(LedgerStorage ledgerStorage) {
            this.ledgerStorage = ledgerStorage;
        }

        @Override
        public Boolean call() throws IOException {
            try {
                this.ledgerStorage.flush();
            }
            catch (IOException e) {
                LOG.error("Got Exception for flush call", (Throwable)e);
                throw new IOException("Got Exception for Flush call", e);
            }
            return true;
        }
    }

    static class LedgerStorageWriteTask
    implements Callable<Boolean> {
        long ledgerId;
        int entryId;
        LedgerStorage ledgerStorage;

        LedgerStorageWriteTask(long ledgerId, int entryId, LedgerStorage ledgerStorage) {
            this.ledgerId = ledgerId;
            this.entryId = entryId;
            this.ledgerStorage = ledgerStorage;
        }

        @Override
        public Boolean call() throws IOException, BookieException {
            try {
                this.ledgerStorage.addEntry(EntryLogTest.generateEntry(this.ledgerId, this.entryId));
            }
            catch (IOException e) {
                LOG.error("Got Exception for AddEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, (Throwable)e);
                throw new IOException("Got Exception for AddEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, e);
            }
            return true;
        }
    }
}

