/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.stats.Gauge;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.DiskChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LedgerDirsManager {
    private static final Logger LOG = LoggerFactory.getLogger(LedgerDirsManager.class);
    private volatile List<File> filledDirs;
    private final List<File> ledgerDirectories;
    private volatile List<File> writableLedgerDirectories;
    private final List<LedgerDirsListener> listeners;
    private final Random rand = new Random();
    private final ConcurrentMap<File, Float> diskUsages = new ConcurrentHashMap<File, Float>();
    private final long entryLogSize;
    private long minUsableSizeForEntryLogCreation;
    private long minUsableSizeForIndexFileCreation;
    private final DiskChecker diskChecker;

    public LedgerDirsManager(ServerConfiguration conf, File[] dirs, DiskChecker diskChecker) {
        this(conf, dirs, diskChecker, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    public LedgerDirsManager(ServerConfiguration conf, File[] dirs, DiskChecker diskChecker, StatsLogger statsLogger) {
        this.ledgerDirectories = Arrays.asList(Bookie.getCurrentDirectories(dirs));
        this.writableLedgerDirectories = new ArrayList<File>(this.ledgerDirectories);
        this.filledDirs = new ArrayList<File>();
        this.listeners = new ArrayList<LedgerDirsListener>();
        this.entryLogSize = conf.getEntryLogSizeLimit();
        this.minUsableSizeForIndexFileCreation = conf.getMinUsableSizeForIndexFileCreation();
        this.minUsableSizeForEntryLogCreation = conf.getMinUsableSizeForEntryLogCreation();
        for (File dir : this.ledgerDirectories) {
            this.diskUsages.put(dir, Float.valueOf(0.0f));
            String statName = "dir_" + dir.getParent().replace('/', '_') + "_usage";
            final File targetDir = dir;
            statsLogger.registerGauge(statName, (Gauge)new Gauge<Number>(){

                public Number getDefaultValue() {
                    return 0;
                }

                public Number getSample() {
                    return Float.valueOf(((Float)LedgerDirsManager.this.diskUsages.get(targetDir)).floatValue() * 100.0f);
                }
            });
        }
        this.diskChecker = diskChecker;
        statsLogger.registerGauge("writable_dirs", (Gauge)new Gauge<Number>(){

            public Number getDefaultValue() {
                return 0;
            }

            public Number getSample() {
                return LedgerDirsManager.this.writableLedgerDirectories.size();
            }
        });
    }

    public List<File> getAllLedgerDirs() {
        return this.ledgerDirectories;
    }

    public List<LedgerDirsListener> getListeners() {
        return this.listeners;
    }

    public long getTotalFreeSpace(List<File> dirs) throws IOException {
        return this.diskChecker.getTotalFreeSpace(dirs);
    }

    public long getTotalDiskSpace(List<File> dirs) throws IOException {
        return this.diskChecker.getTotalDiskSpace(dirs);
    }

    public ConcurrentMap<File, Float> getDiskUsages() {
        return this.diskUsages;
    }

    public List<File> getWritableLedgerDirs() throws NoWritableLedgerDirException {
        if (this.writableLedgerDirectories.isEmpty()) {
            String errMsg = "All ledger directories are non writable";
            NoWritableLedgerDirException e = new NoWritableLedgerDirException(errMsg);
            LOG.error(errMsg, (Throwable)e);
            throw e;
        }
        return this.writableLedgerDirectories;
    }

    public boolean hasWritableLedgerDirs() {
        return !this.writableLedgerDirectories.isEmpty();
    }

    public List<File> getWritableLedgerDirsForNewLog() throws NoWritableLedgerDirException {
        if (!this.writableLedgerDirectories.isEmpty()) {
            return this.writableLedgerDirectories;
        }
        return this.getDirsAboveUsableThresholdSize(this.minUsableSizeForEntryLogCreation, true);
    }

    List<File> getDirsAboveUsableThresholdSize(long thresholdSize, boolean loggingNoWritable) throws NoWritableLedgerDirException {
        ArrayList<File> fullLedgerDirsToAccomodate = new ArrayList<File>();
        for (File dir : this.ledgerDirectories) {
            if (dir.getUsableSpace() <= thresholdSize) continue;
            fullLedgerDirsToAccomodate.add(dir);
        }
        if (!fullLedgerDirsToAccomodate.isEmpty()) {
            if (loggingNoWritable) {
                LOG.info("No writable ledger dirs below diskUsageThreshold. But Dirs that can accommodate {} are: {}", (Object)thresholdSize, fullLedgerDirsToAccomodate);
            }
            return fullLedgerDirsToAccomodate;
        }
        String errMsg = "All ledger directories are non writable and no reserved space (" + thresholdSize + ") left.";
        NoWritableLedgerDirException e = new NoWritableLedgerDirException(errMsg);
        if (loggingNoWritable) {
            LOG.error(errMsg, (Throwable)e);
        }
        throw e;
    }

    public List<File> getFullFilledLedgerDirs() {
        return this.filledDirs;
    }

    public boolean isDirFull(File dir) {
        return this.filledDirs.contains(dir);
    }

    @VisibleForTesting
    public void addToFilledDirs(File dir) {
        if (!this.filledDirs.contains(dir)) {
            LOG.warn(dir + " is out of space. Adding it to filled dirs list");
            ArrayList<File> updatedFilledDirs = new ArrayList<File>(this.filledDirs);
            updatedFilledDirs.add(dir);
            this.filledDirs = updatedFilledDirs;
            ArrayList<File> newDirs = new ArrayList<File>(this.writableLedgerDirectories);
            newDirs.removeAll(this.filledDirs);
            this.writableLedgerDirectories = newDirs;
            for (LedgerDirsListener listener : this.listeners) {
                listener.diskFull(dir);
            }
        }
    }

    public void addToWritableDirs(File dir, boolean underWarnThreshold) {
        if (this.writableLedgerDirectories.contains(dir)) {
            return;
        }
        LOG.info("{} becomes writable. Adding it to writable dirs list.", (Object)dir);
        ArrayList<File> updatedWritableDirs = new ArrayList<File>(this.writableLedgerDirectories);
        updatedWritableDirs.add(dir);
        this.writableLedgerDirectories = updatedWritableDirs;
        ArrayList<File> newDirs = new ArrayList<File>(this.filledDirs);
        newDirs.removeAll(this.writableLedgerDirectories);
        this.filledDirs = newDirs;
        for (LedgerDirsListener listener : this.listeners) {
            if (underWarnThreshold) {
                listener.diskWritable(dir);
                continue;
            }
            listener.diskJustWritable(dir);
        }
    }

    File pickRandomWritableDir() throws NoWritableLedgerDirException {
        return this.pickRandomWritableDir(null);
    }

    File pickRandomWritableDir(File excludedDir) throws NoWritableLedgerDirException {
        List<File> writableDirs = this.getWritableLedgerDirs();
        return this.pickRandomDir(writableDirs, excludedDir);
    }

    File pickRandomWritableDirForNewIndexFile(File excludedDir) throws NoWritableLedgerDirException {
        List<File> writableDirsForNewIndexFile = !this.writableLedgerDirectories.isEmpty() ? this.writableLedgerDirectories : this.getDirsAboveUsableThresholdSize(this.minUsableSizeForIndexFileCreation, true);
        return this.pickRandomDir(writableDirsForNewIndexFile, excludedDir);
    }

    File pickRandomDir(File excludedDir) throws NoWritableLedgerDirException {
        return this.pickRandomDir(this.getAllLedgerDirs(), excludedDir);
    }

    File pickRandomDir(List<File> dirs, File excludedDir) throws NoWritableLedgerDirException {
        int start;
        int idx = start = this.rand.nextInt(dirs.size());
        File candidate = dirs.get(idx);
        while (null != excludedDir && excludedDir.equals(candidate)) {
            if ((idx = (idx + 1) % dirs.size()) == start) {
                throw new NoWritableLedgerDirException("No writable directories found from  available writable dirs (" + dirs + ") : exclude dir " + excludedDir);
            }
            candidate = dirs.get(idx);
        }
        return candidate;
    }

    public void addLedgerDirsListener(LedgerDirsListener listener) {
        if (listener != null) {
            this.listeners.add(listener);
        }
    }

    public DiskChecker getDiskChecker() {
        return this.diskChecker;
    }

    public static interface LedgerDirsListener {
        default public void diskFailed(File disk) {
        }

        default public void diskAlmostFull(File disk) {
        }

        default public void diskFull(File disk) {
        }

        default public void diskWritable(File disk) {
        }

        default public void diskJustWritable(File disk) {
        }

        default public void allDisksFull(boolean highPriorityWritesAllowed) {
        }

        default public void fatalError() {
        }
    }

    public static class NoWritableLedgerDirException
    extends IOException {
        private static final long serialVersionUID = -8696901285061448421L;

        public NoWritableLedgerDirException(String errMsg) {
            super(errMsg);
        }
    }
}

