/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.GarbageCollector;
import org.apache.bookkeeper.client.LedgerMetadata;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.meta.ZkLedgerUnderreplicationManager;
import org.apache.bookkeeper.meta.zk.ZKMetadataDriverBase;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.Gauge;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScanAndCompareGarbageCollector
implements GarbageCollector {
    static final Logger LOG = LoggerFactory.getLogger(ScanAndCompareGarbageCollector.class);
    static final int MAX_CONCURRENT_ZK_REQUESTS = 1000;
    private final LedgerManager ledgerManager;
    private final CompactableLedgerStorage ledgerStorage;
    private final ServerConfiguration conf;
    private final BookieSocketAddress selfBookieAddress;
    private ZooKeeper zk = null;
    private boolean enableGcOverReplicatedLedger;
    private final long gcOverReplicatedLedgerIntervalMillis;
    private long lastOverReplicatedLedgerGcTimeMillis;
    private final String zkServers;
    private final String zkLedgersRootPath;
    private final boolean verifyMetadataOnGc;
    private int activeLedgerCounter;
    private Counter deletedLedgerCounter;

    public ScanAndCompareGarbageCollector(LedgerManager ledgerManager, CompactableLedgerStorage ledgerStorage, ServerConfiguration conf, StatsLogger statsLogger) throws IOException {
        this.ledgerManager = ledgerManager;
        this.ledgerStorage = ledgerStorage;
        this.conf = conf;
        this.selfBookieAddress = Bookie.getBookieAddress(conf);
        this.gcOverReplicatedLedgerIntervalMillis = conf.getGcOverreplicatedLedgerWaitTimeMillis();
        this.lastOverReplicatedLedgerGcTimeMillis = System.currentTimeMillis();
        if (this.gcOverReplicatedLedgerIntervalMillis > 0L) {
            this.enableGcOverReplicatedLedger = true;
        }
        this.zkServers = ZKMetadataDriverBase.resolveZkServers(conf);
        this.zkLedgersRootPath = ZKMetadataDriverBase.resolveZkLedgersRootPath(conf);
        LOG.info("Over Replicated Ledger Deletion : enabled=" + this.enableGcOverReplicatedLedger + ", interval=" + this.gcOverReplicatedLedgerIntervalMillis);
        this.verifyMetadataOnGc = conf.getVerifyMetadataOnGC();
        this.deletedLedgerCounter = statsLogger.getCounter("DELETED_LEDGER_COUNT");
        this.activeLedgerCounter = 0;
        statsLogger.registerGauge("ACTIVE_LEDGER_COUNT", (Gauge)new Gauge<Integer>(){

            public Integer getDefaultValue() {
                return 0;
            }

            public Integer getSample() {
                return ScanAndCompareGarbageCollector.this.activeLedgerCounter;
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void gc(GarbageCollector.GarbageCleaner garbageCleaner) {
        if (null == this.ledgerManager) {
            return;
        }
        try {
            boolean checkOverreplicatedLedgers;
            TreeSet bkActiveLedgers = Sets.newTreeSet(this.ledgerStorage.getActiveLedgersInRange(0L, Long.MAX_VALUE));
            this.activeLedgerCounter = bkActiveLedgers.size();
            long curTime = System.currentTimeMillis();
            boolean bl = checkOverreplicatedLedgers = this.enableGcOverReplicatedLedger && curTime - this.lastOverReplicatedLedgerGcTimeMillis > this.gcOverReplicatedLedgerIntervalMillis;
            if (checkOverreplicatedLedgers) {
                this.zk = ZooKeeperClient.newBuilder().connectString(this.zkServers).sessionTimeoutMs(this.conf.getZkTimeout()).build();
                Set<Long> overReplicatedLedgers = this.removeOverReplicatedledgers(bkActiveLedgers, garbageCleaner);
                if (overReplicatedLedgers.isEmpty()) {
                    LOG.info("No over-replicated ledgers found.");
                } else {
                    LOG.info("Removed over-replicated ledgers: {}", overReplicatedLedgers);
                }
                this.lastOverReplicatedLedgerGcTimeMillis = System.currentTimeMillis();
            }
            LedgerManager.LedgerRangeIterator ledgerRangeIterator = this.ledgerManager.getLedgerRanges();
            Set<Object> ledgersInMetadata = null;
            long end = -1L;
            boolean done = false;
            while (!done) {
                long start = end + 1L;
                if (ledgerRangeIterator.hasNext()) {
                    LedgerManager.LedgerRange lRange = ledgerRangeIterator.next();
                    ledgersInMetadata = lRange.getLedgers();
                    end = lRange.end();
                } else {
                    ledgersInMetadata = new TreeSet();
                    end = Long.MAX_VALUE;
                    done = true;
                }
                NavigableSet<Long> subBkActiveLedgers = bkActiveLedgers.subSet(start, true, end, true);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Active in metadata {}, Active in bookie {}", ledgersInMetadata, subBkActiveLedgers);
                }
                for (Long bkLid : subBkActiveLedgers) {
                    if (ledgersInMetadata.contains(bkLid)) continue;
                    if (this.verifyMetadataOnGc) {
                        CountDownLatch latch = new CountDownLatch(1);
                        AtomicInteger metaRC = new AtomicInteger(0);
                        this.ledgerManager.readLedgerMetadata(bkLid, (rc, x) -> {
                            metaRC.set(rc);
                            latch.countDown();
                        });
                        latch.await();
                        if (metaRC.get() != -7) {
                            LOG.warn("Ledger {} Missing in metadata list, but ledgerManager returned rc: {}.", (Object)bkLid, (Object)metaRC.get());
                            continue;
                        }
                    }
                    this.deletedLedgerCounter.inc();
                    garbageCleaner.clean(bkLid);
                }
            }
        }
        catch (Throwable t) {
            LOG.warn("Exception when iterating over the metadata {}", t);
        }
        finally {
            if (this.zk != null) {
                try {
                    this.zk.close();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    LOG.error("Error closing zk session", (Throwable)e);
                }
                this.zk = null;
            }
        }
    }

    private Set<Long> removeOverReplicatedledgers(Set<Long> bkActiveledgers, final GarbageCollector.GarbageCleaner garbageCleaner) throws InterruptedException, KeeperException {
        List<ACL> zkAcls = ZkUtils.getACLs(this.conf);
        final HashSet overReplicatedLedgers = Sets.newHashSet();
        final Semaphore semaphore = new Semaphore(1000);
        final CountDownLatch latch = new CountDownLatch(bkActiveledgers.size());
        for (final Long ledgerId : bkActiveledgers) {
            try {
                if (ZkLedgerUnderreplicationManager.isLedgerBeingReplicated(this.zk, this.zkLedgersRootPath, ledgerId)) {
                    latch.countDown();
                    continue;
                }
                ZkLedgerUnderreplicationManager.acquireUnderreplicatedLedgerLock(this.zk, this.zkLedgersRootPath, ledgerId, zkAcls);
                semaphore.acquire();
                this.ledgerManager.readLedgerMetadata(ledgerId, new BookkeeperInternalCallbacks.GenericCallback<LedgerMetadata>(){

                    @Override
                    public void operationComplete(int rc, LedgerMetadata ledgerMetadata) {
                        if (rc == 0) {
                            if (!ledgerMetadata.isClosed()) {
                                this.release();
                                return;
                            }
                            TreeMap<Long, ArrayList<BookieSocketAddress>> ensembles = ledgerMetadata.getEnsembles();
                            for (ArrayList ensemble : ensembles.values()) {
                                if (!ensemble.contains(ScanAndCompareGarbageCollector.this.selfBookieAddress)) continue;
                                this.release();
                                return;
                            }
                            overReplicatedLedgers.add(ledgerId);
                            garbageCleaner.clean(ledgerId);
                        }
                        this.release();
                    }

                    private void release() {
                        semaphore.release();
                        latch.countDown();
                        try {
                            ZkLedgerUnderreplicationManager.releaseUnderreplicatedLedgerLock(ScanAndCompareGarbageCollector.this.zk, ScanAndCompareGarbageCollector.this.zkLedgersRootPath, ledgerId);
                        }
                        catch (Throwable t) {
                            LOG.error("Exception when removing underreplicated lock for ledger {}", (Object)ledgerId, (Object)t);
                        }
                    }
                });
            }
            catch (Throwable t) {
                LOG.error("Exception when iterating through the ledgers to check for over-replication", t);
                latch.countDown();
            }
        }
        latch.await();
        bkActiveledgers.removeAll(overReplicatedLedgers);
        return overReplicatedLedgers;
    }
}

