/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.meta;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.net.URI;
import java.util.HashSet;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Function;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.discover.RegistrationManager;
import org.apache.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bookkeeper.meta.MetadataBookieDriver;
import org.apache.bookkeeper.meta.MetadataClientDriver;
import org.apache.bookkeeper.meta.exceptions.Code;
import org.apache.bookkeeper.meta.exceptions.MetadataException;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.ReflectionUtils;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MetadataDrivers {
    private static final Logger log = LoggerFactory.getLogger(MetadataDrivers.class);
    static final String ZK_CLIENT_DRIVER_CLASS = "org.apache.bookkeeper.meta.zk.ZKMetadataClientDriver";
    static final String ZK_BOOKIE_DRIVER_CLASS = "org.apache.bookkeeper.meta.zk.ZKMetadataBookieDriver";
    static final String BK_METADATA_CLIENT_DRIVERS_PROPERTY = "bookkeeper.metadata.client.drivers";
    static final String BK_METADATA_BOOKIE_DRIVERS_PROPERTY = "bookkeeper.metadata.bookie.drivers";
    private static final ConcurrentMap<String, MetadataClientDriverInfo> clientDrivers;
    private static final ConcurrentMap<String, MetadataBookieDriverInfo> bookieDrivers;
    private static boolean initialized;

    static void initialize() {
        if (initialized) {
            return;
        }
        MetadataDrivers.loadInitialDrivers();
        initialized = true;
        log.info("BookKeeper metadata driver manager initialized");
    }

    @VisibleForTesting
    static void loadInitialDrivers() {
        MetadataDrivers.loadInitialClientDrivers();
        MetadataDrivers.loadInitialBookieDrivers();
    }

    private static void loadInitialClientDrivers() {
        HashSet driverList = Sets.newHashSet();
        driverList.add(ZK_CLIENT_DRIVER_CLASS);
        String driversStr = System.getProperty(BK_METADATA_CLIENT_DRIVERS_PROPERTY);
        if (null != driversStr) {
            String[] driversArray;
            for (String driver : driversArray = StringUtils.split((String)driversStr, (char)':')) {
                driverList.add(driver);
            }
        }
        for (String driverClsName : driverList) {
            try {
                MetadataClientDriver driver = ReflectionUtils.newInstance(driverClsName, MetadataClientDriver.class);
                MetadataClientDriverInfo driverInfo = new MetadataClientDriverInfo(driver.getClass());
                clientDrivers.put(driver.getScheme().toLowerCase(), driverInfo);
            }
            catch (Exception e) {
                log.warn("Failed to load metadata client driver {}", (Object)driverClsName, (Object)e);
            }
        }
    }

    private static void loadInitialBookieDrivers() {
        HashSet driverList = Sets.newHashSet();
        driverList.add(ZK_BOOKIE_DRIVER_CLASS);
        String driversStr = System.getProperty(BK_METADATA_BOOKIE_DRIVERS_PROPERTY);
        if (null != driversStr) {
            String[] driversArray;
            for (String driver : driversArray = StringUtils.split((String)driversStr, (char)':')) {
                driverList.add(driver);
            }
        }
        for (String driverClsName : driverList) {
            try {
                MetadataBookieDriver driver = ReflectionUtils.newInstance(driverClsName, MetadataBookieDriver.class);
                MetadataBookieDriverInfo driverInfo = new MetadataBookieDriverInfo(driver.getClass());
                bookieDrivers.put(driver.getScheme().toLowerCase(), driverInfo);
            }
            catch (Exception e) {
                log.warn("Failed to load metadata bookie driver {}", (Object)driverClsName, (Object)e);
            }
        }
    }

    public static void registerClientDriver(String metadataBackendScheme, Class<? extends MetadataClientDriver> driver) {
        MetadataDrivers.registerClientDriver(metadataBackendScheme, driver, false);
    }

    @VisibleForTesting
    public static void registerClientDriver(String metadataBackendScheme, Class<? extends MetadataClientDriver> driver, boolean allowOverride) {
        String scheme;
        MetadataClientDriverInfo oldDriverInfo;
        if (!initialized) {
            MetadataDrivers.initialize();
        }
        if (null != (oldDriverInfo = (MetadataClientDriverInfo)clientDrivers.get(scheme = metadataBackendScheme.toLowerCase())) && !allowOverride) {
            return;
        }
        MetadataClientDriverInfo newDriverInfo = new MetadataClientDriverInfo(driver);
        oldDriverInfo = clientDrivers.putIfAbsent(scheme, newDriverInfo);
        if (null != oldDriverInfo) {
            log.debug("Metadata client driver for {} is already there.", (Object)scheme);
            if (allowOverride) {
                log.debug("Overriding client driver for {}", (Object)scheme);
                clientDrivers.put(scheme, newDriverInfo);
            }
        }
    }

    public static void registerBookieDriver(String metadataBackendScheme, Class<? extends MetadataBookieDriver> driver) {
        MetadataDrivers.registerBookieDriver(metadataBackendScheme, driver, false);
    }

    @VisibleForTesting
    public static void registerBookieDriver(String metadataBackendScheme, Class<? extends MetadataBookieDriver> driver, boolean allowOverride) {
        String scheme;
        MetadataBookieDriverInfo oldDriverInfo;
        if (!initialized) {
            MetadataDrivers.initialize();
        }
        if (null != (oldDriverInfo = (MetadataBookieDriverInfo)bookieDrivers.get(scheme = metadataBackendScheme.toLowerCase())) && !allowOverride) {
            return;
        }
        MetadataBookieDriverInfo newDriverInfo = new MetadataBookieDriverInfo(driver);
        oldDriverInfo = bookieDrivers.putIfAbsent(scheme, newDriverInfo);
        if (null != oldDriverInfo) {
            log.debug("Metadata bookie driver for {} is already there.", (Object)scheme);
            if (allowOverride) {
                log.debug("Overriding bookie driver for {}", (Object)scheme);
                bookieDrivers.put(scheme, newDriverInfo);
            }
        }
    }

    public static MetadataClientDriver getClientDriver(String scheme) {
        MetadataClientDriverInfo driverInfo;
        Preconditions.checkNotNull((Object)scheme, (Object)"Client Driver Scheme is null");
        if (!initialized) {
            MetadataDrivers.initialize();
        }
        if (null == (driverInfo = (MetadataClientDriverInfo)clientDrivers.get(scheme.toLowerCase()))) {
            throw new IllegalArgumentException("Unknown backend " + scheme);
        }
        return ReflectionUtils.newInstance(driverInfo.driverClass);
    }

    public static MetadataClientDriver getClientDriver(URI uri) {
        Preconditions.checkNotNull((Object)uri, (Object)"Metadata service uri is null");
        String scheme = uri.getScheme();
        Preconditions.checkNotNull((Object)scheme, (Object)("Invalid metadata service uri : " + uri));
        scheme = scheme.toLowerCase();
        String[] schemeParts = StringUtils.split((String)scheme, (char)'+');
        Preconditions.checkArgument((schemeParts.length > 0 ? 1 : 0) != 0, (Object)("Invalid metadata service scheme found : " + uri));
        return MetadataDrivers.getClientDriver(schemeParts[0]);
    }

    public static MetadataBookieDriver getBookieDriver(String scheme) {
        MetadataBookieDriverInfo driverInfo;
        Preconditions.checkNotNull((Object)scheme, (Object)"Bookie Driver Scheme is null");
        if (!initialized) {
            MetadataDrivers.initialize();
        }
        if (null == (driverInfo = (MetadataBookieDriverInfo)bookieDrivers.get(scheme.toLowerCase()))) {
            throw new IllegalArgumentException("Unknown backend " + scheme);
        }
        return ReflectionUtils.newInstance(driverInfo.driverClass);
    }

    public static MetadataBookieDriver getBookieDriver(URI uri) {
        Preconditions.checkNotNull((Object)uri, (Object)"Metadata service uri is null");
        String scheme = uri.getScheme();
        Preconditions.checkNotNull((Object)scheme, (Object)("Invalid metadata service uri : " + uri));
        scheme = scheme.toLowerCase();
        String[] schemeParts = StringUtils.split((String)scheme, (char)'+');
        Preconditions.checkArgument((schemeParts.length > 0 ? 1 : 0) != 0, (Object)("Invalid metadata service scheme found : " + uri));
        return MetadataDrivers.getBookieDriver(schemeParts[0]);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static <T> T runFunctionWithMetadataClientDriver(ClientConfiguration conf, Function<MetadataClientDriver, T> function, ScheduledExecutorService executorService) throws MetadataException, ExecutionException {
        try {
            Throwable throwable = null;
            try (MetadataClientDriver driver = MetadataDrivers.getClientDriver(URI.create(conf.getMetadataServiceUri()));){
                MetadataClientDriver metadataClientDriver;
                driver.initialize(conf, executorService, (StatsLogger)NullStatsLogger.INSTANCE, Optional.empty());
                try {
                    metadataClientDriver = function.apply(driver);
                }
                catch (Exception uee) {
                    try {
                        if (uee.getCause() instanceof MetadataException) {
                            throw (MetadataException)uee.getCause();
                        }
                        throw new ExecutionException(uee.getMessage(), uee.getCause());
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                }
                return (T)metadataClientDriver;
            }
        }
        catch (ConfigurationException e) {
            throw new MetadataException(Code.INVALID_METADATA_SERVICE_URI, (Throwable)e);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static <T> T runFunctionWithMetadataBookieDriver(ServerConfiguration conf, Function<MetadataBookieDriver, T> function) throws MetadataException, ExecutionException {
        try {
            Throwable throwable = null;
            try (MetadataBookieDriver driver = MetadataDrivers.getBookieDriver(URI.create(conf.getMetadataServiceUri()));){
                MetadataBookieDriver metadataBookieDriver;
                driver.initialize(conf, () -> {}, (StatsLogger)NullStatsLogger.INSTANCE);
                try {
                    metadataBookieDriver = function.apply(driver);
                }
                catch (Exception uee) {
                    try {
                        if (uee.getCause() instanceof MetadataException) {
                            throw (MetadataException)uee.getCause();
                        }
                        throw new ExecutionException(uee.getMessage(), uee.getCause());
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                }
                return (T)metadataBookieDriver;
            }
        }
        catch (ConfigurationException e) {
            throw new MetadataException(Code.INVALID_METADATA_SERVICE_URI, (Throwable)e);
        }
    }

    public static <T> T runFunctionWithRegistrationManager(ServerConfiguration conf, Function<RegistrationManager, T> function) throws MetadataException, ExecutionException {
        return (T)MetadataDrivers.runFunctionWithMetadataBookieDriver(conf, driver -> function.apply(driver.getRegistrationManager()));
    }

    public static <T> T runFunctionWithLedgerManagerFactory(ServerConfiguration conf, Function<LedgerManagerFactory, T> function) throws MetadataException, ExecutionException {
        return (T)MetadataDrivers.runFunctionWithMetadataBookieDriver(conf, driver -> {
            try {
                return function.apply(driver.getLedgerManagerFactory());
            }
            catch (MetadataException me) {
                throw new UncheckedExecutionException(me.getMessage(), (Throwable)me);
            }
        });
    }

    private MetadataDrivers() {
    }

    static ConcurrentMap<String, MetadataClientDriverInfo> getClientDrivers() {
        return clientDrivers;
    }

    static ConcurrentMap<String, MetadataBookieDriverInfo> getBookieDrivers() {
        return bookieDrivers;
    }

    static {
        initialized = false;
        clientDrivers = new ConcurrentHashMap<String, MetadataClientDriverInfo>();
        bookieDrivers = new ConcurrentHashMap<String, MetadataBookieDriverInfo>();
        MetadataDrivers.initialize();
    }

    static class MetadataBookieDriverInfo {
        final Class<? extends MetadataBookieDriver> driverClass;
        final String driverClassName;

        MetadataBookieDriverInfo(Class<? extends MetadataBookieDriver> driverClass) {
            this.driverClass = driverClass;
            this.driverClassName = driverClass.getName();
        }

        public String toString() {
            return "MetadataDrivers.MetadataBookieDriverInfo(driverClass=" + this.driverClass + ", driverClassName=" + this.driverClassName + ")";
        }
    }

    static class MetadataClientDriverInfo {
        final Class<? extends MetadataClientDriver> driverClass;
        final String driverClassName;

        MetadataClientDriverInfo(Class<? extends MetadataClientDriver> driverClass) {
            this.driverClass = driverClass;
            this.driverClassName = driverClass.getName();
        }

        public String toString() {
            return "MetadataDrivers.MetadataClientDriverInfo(driverClass=" + this.driverClass + ", driverClassName=" + this.driverClassName + ")";
        }
    }
}

