/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie.storage.ldb;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.protobuf.ByteString;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.storage.ldb.ArrayUtil;
import org.apache.bookkeeper.bookie.storage.ldb.DbLedgerStorageDataFormats;
import org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorage;
import org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorageFactory;
import org.apache.bookkeeper.bookie.storage.ldb.LedgerMetadataIndexStats;
import org.apache.bookkeeper.bookie.storage.ldb.LongWrapper;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.collections.ConcurrentLongHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LedgerMetadataIndex
implements Closeable {
    private final ConcurrentLongHashMap<DbLedgerStorageDataFormats.LedgerData> ledgers;
    private final AtomicInteger ledgersCount;
    private final KeyValueStorage ledgersDb;
    private final LedgerMetadataIndexStats stats;
    private final ConcurrentLinkedQueue<Map.Entry<Long, DbLedgerStorageDataFormats.LedgerData>> pendingLedgersUpdates;
    private final ConcurrentLinkedQueue<Long> pendingDeletedLedgers;
    private static final Logger log = LoggerFactory.getLogger(LedgerMetadataIndex.class);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public LedgerMetadataIndex(ServerConfiguration conf, KeyValueStorageFactory storageFactory, String basePath, StatsLogger stats) throws IOException {
        String ledgersPath = FileSystems.getDefault().getPath(basePath, "ledgers").toFile().toString();
        this.ledgersDb = storageFactory.newKeyValueStorage(ledgersPath, KeyValueStorageFactory.DbConfigType.Small, conf);
        this.ledgers = new ConcurrentLongHashMap();
        this.ledgersCount = new AtomicInteger();
        try (KeyValueStorage.CloseableIterator<Map.Entry<byte[], byte[]>> iterator = this.ledgersDb.iterator();){
            while (iterator.hasNext()) {
                Map.Entry<byte[], byte[]> entry = iterator.next();
                long ledgerId = ArrayUtil.getLong(entry.getKey(), 0);
                DbLedgerStorageDataFormats.LedgerData ledgerData = DbLedgerStorageDataFormats.LedgerData.parseFrom((byte[])entry.getValue());
                this.ledgers.put(ledgerId, ledgerData);
                this.ledgersCount.incrementAndGet();
            }
        }
        this.pendingLedgersUpdates = new ConcurrentLinkedQueue();
        this.pendingDeletedLedgers = new ConcurrentLinkedQueue();
        this.stats = new LedgerMetadataIndexStats(stats, () -> this.ledgersCount.get());
    }

    @Override
    public void close() throws IOException {
        this.ledgersDb.close();
    }

    public DbLedgerStorageDataFormats.LedgerData get(long ledgerId) throws IOException {
        DbLedgerStorageDataFormats.LedgerData ledgerData = this.ledgers.get(ledgerId);
        if (ledgerData == null) {
            if (log.isDebugEnabled()) {
                log.debug("Ledger not found {}", (Object)ledgerId);
            }
            throw new Bookie.NoLedgerException(ledgerId);
        }
        return ledgerData;
    }

    public void set(long ledgerId, DbLedgerStorageDataFormats.LedgerData ledgerData) throws IOException {
        if (this.ledgers.put(ledgerId, ledgerData = DbLedgerStorageDataFormats.LedgerData.newBuilder((DbLedgerStorageDataFormats.LedgerData)ledgerData).setExists(true).build()) == null) {
            if (log.isDebugEnabled()) {
                log.debug("Added new ledger {}", (Object)ledgerId);
            }
            this.ledgersCount.incrementAndGet();
        }
        this.pendingLedgersUpdates.add(new AbstractMap.SimpleEntry<Long, DbLedgerStorageDataFormats.LedgerData>(ledgerId, ledgerData));
        this.pendingDeletedLedgers.remove(ledgerId);
    }

    public void delete(long ledgerId) throws IOException {
        if (this.ledgers.remove(ledgerId) != null) {
            if (log.isDebugEnabled()) {
                log.debug("Removed ledger {}", (Object)ledgerId);
            }
            this.ledgersCount.decrementAndGet();
        }
        this.pendingDeletedLedgers.add(ledgerId);
        this.pendingLedgersUpdates.removeIf(e -> (Long)e.getKey() == ledgerId);
    }

    public Iterable<Long> getActiveLedgersInRange(final long firstLedgerId, final long lastLedgerId) throws IOException {
        return Iterables.filter(this.ledgers.keys(), (Predicate)new Predicate<Long>(){

            public boolean apply(Long ledgerId) {
                return ledgerId >= firstLedgerId && ledgerId < lastLedgerId;
            }
        });
    }

    public boolean setFenced(long ledgerId) throws IOException {
        DbLedgerStorageDataFormats.LedgerData ledgerData = this.get(ledgerId);
        if (ledgerData.getFenced()) {
            return false;
        }
        DbLedgerStorageDataFormats.LedgerData newLedgerData = DbLedgerStorageDataFormats.LedgerData.newBuilder((DbLedgerStorageDataFormats.LedgerData)ledgerData).setFenced(true).build();
        if (this.ledgers.put(ledgerId, newLedgerData) == null) {
            if (log.isDebugEnabled()) {
                log.debug("Re-inserted fenced ledger {}", (Object)ledgerId);
            }
            this.ledgersCount.incrementAndGet();
        } else if (log.isDebugEnabled()) {
            log.debug("Set fenced ledger {}", (Object)ledgerId);
        }
        this.pendingLedgersUpdates.add(new AbstractMap.SimpleEntry<Long, DbLedgerStorageDataFormats.LedgerData>(ledgerId, newLedgerData));
        this.pendingDeletedLedgers.remove(ledgerId);
        return true;
    }

    public void setMasterKey(long ledgerId, byte[] masterKey) throws IOException {
        DbLedgerStorageDataFormats.LedgerData ledgerData = this.ledgers.get(ledgerId);
        if (ledgerData == null) {
            ledgerData = DbLedgerStorageDataFormats.LedgerData.newBuilder().setExists(true).setFenced(false).setMasterKey(ByteString.copyFrom((byte[])masterKey)).build();
            if (log.isDebugEnabled()) {
                log.debug("Inserting new ledger {}", (Object)ledgerId);
            }
        } else {
            byte[] storedMasterKey = ledgerData.getMasterKey().toByteArray();
            if (ArrayUtil.isArrayAllZeros(storedMasterKey)) {
                ledgerData = DbLedgerStorageDataFormats.LedgerData.newBuilder((DbLedgerStorageDataFormats.LedgerData)ledgerData).setMasterKey(ByteString.copyFrom((byte[])masterKey)).build();
                if (log.isDebugEnabled()) {
                    log.debug("Replace old master key {} with new master key {}", (Object)storedMasterKey, (Object)masterKey);
                }
            } else if (!Arrays.equals(storedMasterKey, masterKey) && !ArrayUtil.isArrayAllZeros(masterKey)) {
                log.warn("Ledger {} masterKey in db can only be set once.", (Object)ledgerId);
                throw new IOException(BookieException.create(-100));
            }
        }
        if (this.ledgers.put(ledgerId, ledgerData) == null) {
            this.ledgersCount.incrementAndGet();
        }
        this.pendingLedgersUpdates.add(new AbstractMap.SimpleEntry<Long, DbLedgerStorageDataFormats.LedgerData>(ledgerId, ledgerData));
        this.pendingDeletedLedgers.remove(ledgerId);
    }

    public void flush() throws IOException {
        LongWrapper key = LongWrapper.get();
        int updatedLedgers = 0;
        while (!this.pendingLedgersUpdates.isEmpty()) {
            Map.Entry<Long, DbLedgerStorageDataFormats.LedgerData> entry = this.pendingLedgersUpdates.poll();
            key.set(entry.getKey());
            byte[] value = entry.getValue().toByteArray();
            this.ledgersDb.put(key.array, value);
            ++updatedLedgers;
        }
        if (log.isDebugEnabled()) {
            log.debug("Persisting updates to {} ledgers", (Object)updatedLedgers);
        }
        this.ledgersDb.sync();
        key.recycle();
    }

    public void removeDeletedLedgers() throws IOException {
        LongWrapper key = LongWrapper.get();
        int deletedLedgers = 0;
        while (!this.pendingDeletedLedgers.isEmpty()) {
            long ledgerId = this.pendingDeletedLedgers.poll();
            key.set(ledgerId);
            this.ledgersDb.delete(key.array);
            ++deletedLedgers;
        }
        if (log.isDebugEnabled()) {
            log.debug("Persisting deletes of ledgers {}", (Object)deletedLedgers);
        }
        this.ledgersDb.sync();
        key.recycle();
    }
}

