/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.collect.Sets;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.Unpooled;
import io.netty.buffer.UnpooledByteBufAllocator;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLongArray;
import java.util.concurrent.locks.Lock;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.EntryLogManager;
import org.apache.bookkeeper.bookie.EntryLogManagerBase;
import org.apache.bookkeeper.bookie.EntryLogManagerForEntryLogPerLedger;
import org.apache.bookkeeper.bookie.EntryLogManagerForSingleEntryLog;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.bookie.EntryMemTable;
import org.apache.bookkeeper.bookie.EntryMemTableWithParallelFlusher;
import org.apache.bookkeeper.bookie.InterleavedLedgerStorage;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.LedgerStorage;
import org.apache.bookkeeper.bookie.SortedLedgerStorage;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.conf.TestBKConfiguration;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.bookkeeper.util.collections.ConcurrentLongLongHashMap;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryLogTest {
    private static final Logger LOG = LoggerFactory.getLogger(EntryLogTest.class);
    final List<File> tempDirs = new ArrayList<File>();
    final Random rand = new Random();
    private File rootDir;
    private File curDir;
    private ServerConfiguration conf;
    private LedgerDirsManager dirsMgr;
    private EntryLogger entryLogger;

    File createTempDir(String prefix, String suffix) throws IOException {
        File dir = IOUtils.createTempDir((String)prefix, (String)suffix);
        this.tempDirs.add(dir);
        return dir;
    }

    @Before
    public void setUp() throws Exception {
        this.rootDir = this.createTempDir("bkTest", ".dir");
        this.curDir = Bookie.getCurrentDirectory((File)this.rootDir);
        Bookie.checkDirectoryStructure((File)this.curDir);
        this.conf = TestBKConfiguration.newServerConfiguration();
        this.dirsMgr = new LedgerDirsManager(this.conf, new File[]{this.rootDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
    }

    @After
    public void tearDown() throws Exception {
        if (null != this.entryLogger) {
            this.entryLogger.shutdown();
        }
        for (File dir : this.tempDirs) {
            FileUtils.deleteDirectory((File)dir);
        }
        this.tempDirs.clear();
    }

    @Test
    public void testDeferCreateNewLog() throws Exception {
        this.entryLogger.shutdown();
        this.conf.setMinUsableSizeForEntryLogCreation(1L);
        this.dirsMgr = new LedgerDirsManager(this.conf, new File[]{this.rootDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.dirsMgr.addToFilledDirs(this.curDir);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        EntryLogManagerForSingleEntryLog entryLogManager = (EntryLogManagerForSingleEntryLog)this.entryLogger.getEntryLogManager();
        Assert.assertEquals((long)-57005L, (long)entryLogManager.getCurrentLogId());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        Assert.assertEquals((long)0L, (long)entryLogManager.getCurrentLogId());
    }

    @Test
    public void testDeferCreateNewLogWithoutEnoughDiskSpaces() throws Exception {
        this.entryLogger.shutdown();
        this.conf.setMinUsableSizeForEntryLogCreation(Long.MAX_VALUE);
        this.dirsMgr = new LedgerDirsManager(this.conf, new File[]{this.rootDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.dirsMgr.addToFilledDirs(this.curDir);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        EntryLogManagerForSingleEntryLog entryLogManager = (EntryLogManagerForSingleEntryLog)this.entryLogger.getEntryLogManager();
        Assert.assertEquals((long)-57005L, (long)entryLogManager.getCurrentLogId());
        try {
            this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
            Assert.fail((String)"Should fail to append entry if there is no enough reserved space left");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            Assert.assertEquals((long)-57005L, (long)entryLogManager.getCurrentLogId());
        }
    }

    @Test
    public void testCorruptEntryLog() throws Exception {
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        this.entryLogger.addEntry(3L, EntryLogTest.generateEntry(3L, 1L).nioBuffer());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L).nioBuffer());
        this.entryLogger.flush();
        this.entryLogger.shutdown();
        File f = new File(this.curDir, "0.log");
        RandomAccessFile raf = new RandomAccessFile(f, "rw");
        raf.setLength(raf.length() - 10L);
        raf.close();
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        EntryLogMetadata meta = this.entryLogger.getEntryLogMetadata(0L);
        LOG.info("Extracted Meta From Entry Log {}", (Object)meta);
        Assert.assertTrue((boolean)meta.getLedgersMap().containsKey(1L));
        Assert.assertFalse((boolean)meta.getLedgersMap().containsKey(2L));
        Assert.assertTrue((boolean)meta.getLedgersMap().containsKey(3L));
    }

    private static ByteBuf generateEntry(long ledger, long entry) {
        byte[] data = EntryLogTest.generateDataString(ledger, entry).getBytes();
        ByteBuf bb = Unpooled.buffer((int)(16 + data.length));
        bb.writeLong(ledger);
        bb.writeLong(entry);
        bb.writeBytes(data);
        return bb;
    }

    private ByteBuf generateEntry(long ledger, long entry, int length) {
        ByteBuf bb = Unpooled.buffer((int)length);
        bb.writeLong(ledger);
        bb.writeLong(entry);
        byte[] randbyteArray = new byte[length - 8 - 8];
        this.rand.nextBytes(randbyteArray);
        bb.writeBytes(randbyteArray);
        return bb;
    }

    private static String generateDataString(long ledger, long entry) {
        return "ledger-" + ledger + "-" + entry;
    }

    @Test
    public void testMissingLogId() throws Exception {
        int i;
        int numLogs = 3;
        int numEntries = 10;
        long[][] positions = new long[2 * numLogs][];
        for (int i2 = 0; i2 < numLogs; ++i2) {
            positions[i2] = new long[numEntries];
            EntryLogger logger = new EntryLogger(this.conf, this.dirsMgr);
            for (int j = 0; j < numEntries; ++j) {
                positions[i2][j] = logger.addEntry((long)i2, EntryLogTest.generateEntry(i2, j).nioBuffer());
            }
            logger.flush();
            logger.shutdown();
        }
        File lastLogId = new File(this.curDir, "lastId");
        lastLogId.delete();
        for (int i3 = numLogs; i3 < 2 * numLogs; ++i3) {
            positions[i3] = new long[numEntries];
            EntryLogger logger = new EntryLogger(this.conf, this.dirsMgr);
            for (int j = 0; j < numEntries; ++j) {
                positions[i3][j] = logger.addEntry((long)i3, EntryLogTest.generateEntry(i3, j).nioBuffer());
            }
            logger.flush();
            logger.shutdown();
        }
        EntryLogger newLogger = new EntryLogger(this.conf, this.dirsMgr);
        for (i = 0; i < 2 * numLogs + 1; ++i) {
            File logFile = new File(this.curDir, Long.toHexString(i) + ".log");
            Assert.assertTrue((boolean)logFile.exists());
        }
        for (i = 0; i < 2 * numLogs; ++i) {
            for (int j = 0; j < numEntries; ++j) {
                String expectedValue = "ledger-" + i + "-" + j;
                ByteBuf value = newLogger.readEntry((long)i, (long)j, positions[i][j]);
                long ledgerId = value.readLong();
                long entryId = value.readLong();
                byte[] data = new byte[value.readableBytes()];
                value.readBytes(data);
                value.release();
                Assert.assertEquals((long)i, (long)ledgerId);
                Assert.assertEquals((long)j, (long)entryId);
                Assert.assertEquals((Object)expectedValue, (Object)new String(data));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testEntryLoggerShouldThrowFNFEIfDirectoriesDoesNotExist() throws Exception {
        File tmpDir = this.createTempDir("bkTest", ".dir");
        EntryLogger entryLogger = null;
        try {
            entryLogger = new EntryLogger(this.conf, new LedgerDirsManager(this.conf, new File[]{tmpDir}, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold())));
            Assert.fail((String)"Expecting FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            Assert.assertEquals((Object)("Entry log directory '" + tmpDir + "/current' does not exist"), (Object)e.getLocalizedMessage());
        }
        finally {
            if (entryLogger != null) {
                entryLogger.shutdown();
            }
        }
    }

    @Test
    public void testAddEntryFailureOnDiskFull() throws Exception {
        File ledgerDir1 = this.createTempDir("bkTest", ".dir");
        File ledgerDir2 = this.createTempDir("bkTest", ".dir");
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerStorageClass(InterleavedLedgerStorage.class.getName());
        conf.setJournalDirName(ledgerDir1.toString());
        conf.setLedgerDirNames(new String[]{ledgerDir1.getAbsolutePath(), ledgerDir2.getAbsolutePath()});
        Bookie bookie = new Bookie(conf);
        EntryLogger entryLogger = new EntryLogger(conf, bookie.getLedgerDirsManager());
        InterleavedLedgerStorage ledgerStorage = (InterleavedLedgerStorage)bookie.ledgerStorage.getUnderlyingLedgerStorage();
        ledgerStorage.entryLogger = entryLogger;
        ledgerStorage.setMasterKey(1L, "key".getBytes());
        ledgerStorage.setMasterKey(2L, "key".getBytes());
        ledgerStorage.setMasterKey(3L, "key".getBytes());
        ledgerStorage.addEntry(EntryLogTest.generateEntry(1L, 1L));
        ledgerStorage.addEntry(EntryLogTest.generateEntry(2L, 1L));
        bookie.getLedgerDirsManager().addToFilledDirs(((EntryLogManagerBase)entryLogger.getEntryLogManager()).getCurrentLogForLedger(-1L).getLogFile().getParentFile());
        ledgerStorage.addEntry(EntryLogTest.generateEntry(3L, 1L));
        Assert.assertTrue((0 == EntryLogTest.generateEntry(1L, 1L).compareTo(ledgerStorage.getEntry(1L, 1L)) ? 1 : 0) != 0);
        Assert.assertTrue((0 == EntryLogTest.generateEntry(2L, 1L).compareTo(ledgerStorage.getEntry(2L, 1L)) ? 1 : 0) != 0);
        Assert.assertTrue((0 == EntryLogTest.generateEntry(3L, 1L).compareTo(ledgerStorage.getEntry(3L, 1L)) ? 1 : 0) != 0);
    }

    @Test
    public void testRecoverFromLedgersMap() throws Exception {
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        this.entryLogger.addEntry(3L, EntryLogTest.generateEntry(3L, 1L).nioBuffer());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L).nioBuffer());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 2L).nioBuffer());
        EntryLogManagerBase entryLogManager = (EntryLogManagerBase)this.entryLogger.getEntryLogManager();
        entryLogManager.createNewLog(-1L);
        entryLogManager.flushRotatedLogs();
        EntryLogMetadata meta = this.entryLogger.extractEntryLogMetadataFromIndex(0L);
        LOG.info("Extracted Meta From Entry Log {}", (Object)meta);
        Assert.assertEquals((long)60L, (long)meta.getLedgersMap().get(1L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(2L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(3L));
        Assert.assertFalse((boolean)meta.getLedgersMap().containsKey(4L));
        Assert.assertEquals((long)120L, (long)meta.getTotalSize());
        Assert.assertEquals((long)120L, (long)meta.getRemainingSize());
    }

    @Test
    public void testRecoverFromLedgersMapOnV0EntryLog() throws Exception {
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        this.entryLogger.addEntry(3L, EntryLogTest.generateEntry(3L, 1L).nioBuffer());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L).nioBuffer());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 2L).nioBuffer());
        ((EntryLogManagerBase)this.entryLogger.getEntryLogManager()).createNewLog(-1L);
        this.entryLogger.shutdown();
        File f = new File(this.curDir, "0.log");
        RandomAccessFile raf = new RandomAccessFile(f, "rw");
        raf.seek(4L);
        raf.write(new byte[12]);
        raf.close();
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        try {
            this.entryLogger.extractEntryLogMetadataFromIndex(0L);
            Assert.fail((String)"Should not be possible to recover from ledgers map index");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        EntryLogMetadata meta = this.entryLogger.getEntryLogMetadata(0L);
        LOG.info("Extracted Meta From Entry Log {}", (Object)meta);
        Assert.assertEquals((long)60L, (long)meta.getLedgersMap().get(1L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(2L));
        Assert.assertEquals((long)30L, (long)meta.getLedgersMap().get(3L));
        Assert.assertFalse((boolean)meta.getLedgersMap().containsKey(4L));
        Assert.assertEquals((long)120L, (long)meta.getTotalSize());
        Assert.assertEquals((long)120L, (long)meta.getRemainingSize());
    }

    @Test
    public void testPreAllocateLog() throws Exception {
        this.entryLogger.shutdown();
        this.conf.setEntryLogFilePreAllocationEnabled(true);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        ((EntryLogManagerBase)this.entryLogger.getEntryLogManager()).createNewLog(-1L);
        Assert.assertNotNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
        this.entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        Assert.assertNotNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
        this.entryLogger.shutdown();
        this.conf.setEntryLogFilePreAllocationEnabled(false);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        Assert.assertNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
        this.entryLogger.addEntry(2L, EntryLogTest.generateEntry(1L, 1L).nioBuffer());
        Assert.assertNull((Object)this.entryLogger.getEntryLoggerAllocator().getPreallocationFuture());
    }

    @Test
    public void testGetEntryLogsSet() throws Exception {
        EntryLogManagerBase entryLogManagerBase = (EntryLogManagerBase)this.entryLogger.getEntryLogManager();
        Assert.assertEquals((Object)Sets.newHashSet(), (Object)this.entryLogger.getEntryLogsSet());
        entryLogManagerBase.createNewLog(-1L);
        entryLogManagerBase.flushRotatedLogs();
        Assert.assertEquals((Object)Sets.newHashSet((Object[])new Long[]{0L, 1L}), (Object)this.entryLogger.getEntryLogsSet());
        entryLogManagerBase.createNewLog(-1L);
        entryLogManagerBase.flushRotatedLogs();
        Assert.assertEquals((Object)Sets.newHashSet((Object[])new Long[]{0L, 1L, 2L}), (Object)this.entryLogger.getEntryLogsSet());
    }

    @Test
    public void testFlushOrder() throws Exception {
        this.entryLogger.shutdown();
        final int logSizeLimit = 262144;
        this.conf.setEntryLogPerLedgerEnabled(false);
        this.conf.setEntryLogFilePreAllocationEnabled(false);
        this.conf.setFlushIntervalInBytes(0L);
        this.conf.setEntryLogSizeLimit((long)logSizeLimit);
        this.entryLogger = new EntryLogger(this.conf, this.dirsMgr);
        EntryLogManagerBase entryLogManager = (EntryLogManagerBase)this.entryLogger.getEntryLogManager();
        final AtomicBoolean exceptionHappened = new AtomicBoolean(false);
        final CyclicBarrier barrier = new CyclicBarrier(2);
        exceptionHappened.set(false);
        this.addEntriesAndRotateLogs(this.entryLogger, 30);
        LinkedList rotatedLogChannels = new LinkedList(entryLogManager.getRotatedLogChannels());
        EntryLogger.BufferedLogChannel currentActiveChannel = entryLogManager.getCurrentLogForLedger(-1L);
        long currentActiveChannelUnpersistedBytes = currentActiveChannel.getUnpersistedBytes();
        Thread flushThread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    barrier.await();
                    EntryLogTest.this.entryLogger.flush();
                }
                catch (IOException | InterruptedException | BrokenBarrierException e) {
                    LOG.error("Exception happened for entryLogger.flush", (Throwable)e);
                    exceptionHappened.set(true);
                }
            }
        });
        Thread createdNewLogThread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    barrier.await();
                    EntryLogTest.this.entryLogger.addEntry(123L, EntryLogTest.this.generateEntry(123L, 456L, logSizeLimit), true);
                }
                catch (IOException | InterruptedException | BrokenBarrierException e) {
                    LOG.error("Exception happened for entryLogManager.createNewLog", (Throwable)e);
                    exceptionHappened.set(true);
                }
            }
        });
        flushThread.start();
        createdNewLogThread.start();
        flushThread.join();
        createdNewLogThread.join();
        Assert.assertFalse((String)"Exception happened in one of the operation", (boolean)exceptionHappened.get());
        Assert.assertTrue((String)("previous currentChannel unpersistedBytes should be less than " + currentActiveChannelUnpersistedBytes + ", but it is actually " + currentActiveChannel.getUnpersistedBytes()), (currentActiveChannel.getUnpersistedBytes() < currentActiveChannelUnpersistedBytes ? 1 : 0) != 0);
        for (EntryLogger.BufferedLogChannel rotatedLogChannel : rotatedLogChannels) {
            Assert.assertEquals((String)"previous rotated entrylog should be flushandforcewritten", (long)0L, (long)rotatedLogChannel.getUnpersistedBytes());
        }
    }

    void addEntriesAndRotateLogs(EntryLogger entryLogger, int numOfRotations) throws IOException {
        EntryLogManagerBase entryLogManager = (EntryLogManagerBase)entryLogger.getEntryLogManager();
        entryLogManager.setCurrentLogForLedgerAndAddToRotate(-1L, null);
        for (int i = 0; i < numOfRotations; ++i) {
            this.addEntries(entryLogger, 10);
            entryLogManager.setCurrentLogForLedgerAndAddToRotate(-1L, null);
        }
        this.addEntries(entryLogger, 10);
    }

    void addEntries(EntryLogger entryLogger, int noOfEntries) throws IOException {
        for (int j = 0; j < noOfEntries; ++j) {
            int ledgerId = Math.abs(this.rand.nextInt());
            int entryId = Math.abs(this.rand.nextInt());
            entryLogger.addEntry((long)ledgerId, EntryLogTest.generateEntry(ledgerId, entryId).nioBuffer());
        }
    }

    public void testConcurrentWriteAndReadCallsOfInterleavedLedgerStorage() throws Exception {
        this.testConcurrentWriteAndReadCalls(InterleavedLedgerStorage.class.getName(), false);
    }

    public void testConcurrentWriteAndReadCallsOfInterleavedLedgerStorageWithELPLEnabled() throws Exception {
        this.testConcurrentWriteAndReadCalls(InterleavedLedgerStorage.class.getName(), true);
    }

    public void testConcurrentWriteAndReadCallsOfSortedLedgerStorage() throws Exception {
        this.testConcurrentWriteAndReadCalls(SortedLedgerStorage.class.getName(), false);
    }

    public void testConcurrentWriteAndReadCallsOfSortedLedgerStorageWithELPLEnabled() throws Exception {
        this.testConcurrentWriteAndReadCalls(SortedLedgerStorage.class.getName(), true);
    }

    public void testConcurrentWriteAndReadCalls(String ledgerStorageClass, boolean entryLogPerLedgerEnabled) throws Exception {
        int i;
        File ledgerDir = this.createTempDir("bkTest", ".dir");
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setJournalDirName(ledgerDir.toString());
        conf.setLedgerDirNames(new String[]{ledgerDir.getAbsolutePath()});
        conf.setLedgerStorageClass(ledgerStorageClass);
        conf.setEntryLogPerLedgerEnabled(entryLogPerLedgerEnabled);
        Bookie bookie = new Bookie(conf);
        CompactableLedgerStorage ledgerStorage = (CompactableLedgerStorage)bookie.ledgerStorage;
        Random rand = new Random(0L);
        if (ledgerStorageClass.equals(SortedLedgerStorage.class.getName())) {
            Assert.assertEquals((String)"LedgerStorage Class", SortedLedgerStorage.class, ledgerStorage.getClass());
            if (entryLogPerLedgerEnabled) {
                Assert.assertEquals((String)"MemTable Class", EntryMemTableWithParallelFlusher.class, ((SortedLedgerStorage)ledgerStorage).memTable.getClass());
            } else {
                Assert.assertEquals((String)"MemTable Class", EntryMemTable.class, ((SortedLedgerStorage)ledgerStorage).memTable.getClass());
            }
        }
        int numOfLedgers = 70;
        int numEntries = 1500;
        for (int i2 = 0; i2 < numOfLedgers; ++i2) {
            ledgerStorage.setMasterKey((long)i2, "key".getBytes());
        }
        ExecutorService executor = Executors.newFixedThreadPool(10);
        ArrayList<Callable<Boolean>> writeAndFlushTasks = new ArrayList<Callable<Boolean>>();
        for (int j = 0; j < numEntries; ++j) {
            for (i = 0; i < numOfLedgers; ++i) {
                writeAndFlushTasks.add(new LedgerStorageWriteTask(i, j, (LedgerStorage)ledgerStorage));
            }
        }
        for (int i3 = 0; i3 < numOfLedgers * numEntries / 500; ++i3) {
            writeAndFlushTasks.add(rand.nextInt(writeAndFlushTasks.size()), new LedgerStorageFlushTask((LedgerStorage)ledgerStorage));
        }
        executor.invokeAll(writeAndFlushTasks).forEach(future -> {
            try {
                future.get();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                LOG.error("Write/Flush task failed because of InterruptedException", (Throwable)ie);
                Assert.fail((String)"Write/Flush task interrupted");
            }
            catch (Exception ex) {
                LOG.error("Write/Flush task failed because of  exception", (Throwable)ex);
                Assert.fail((String)("Write/Flush task failed " + ex.getMessage()));
            }
        });
        ArrayList<Callable<Boolean>> readAndFlushTasks = new ArrayList<Callable<Boolean>>();
        for (int j = 0; j < numEntries; ++j) {
            for (int i4 = 0; i4 < numOfLedgers; ++i4) {
                readAndFlushTasks.add(new LedgerStorageReadTask(i4, j, (LedgerStorage)ledgerStorage));
            }
        }
        for (i = 0; i < numOfLedgers * numEntries / 500; ++i) {
            readAndFlushTasks.add(rand.nextInt(readAndFlushTasks.size()), new LedgerStorageFlushTask((LedgerStorage)ledgerStorage));
        }
        executor.invokeAll(readAndFlushTasks).forEach(future -> {
            try {
                future.get();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                LOG.error("Read/Flush task failed because of InterruptedException", (Throwable)ie);
                Assert.fail((String)"Read/Flush task interrupted");
            }
            catch (Exception ex) {
                LOG.error("Read/Flush task failed because of  exception", (Throwable)ex);
                Assert.fail((String)("Read/Flush task failed " + ex.getMessage()));
            }
        });
        executor.shutdownNow();
    }

    @Test
    public void testEntryLoggersRecentEntryLogsStatus() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogger.RecentEntryLogsStatus recentlyCreatedLogsStatus = entryLogger.recentlyCreatedEntryLogsStatus;
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(0L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)0L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(0L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)1L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(1L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)1L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(2L));
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(3L));
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(4L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)1L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(1L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)2L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(3L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)2L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(2L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)4L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(4L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)5L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(5L));
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(7L));
        recentlyCreatedLogsStatus.createdEntryLog(Long.valueOf(9L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)5L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(5L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)6L, (long)entryLogger.getLeastUnflushedLogId());
        recentlyCreatedLogsStatus.flushRotatedEntryLog(Long.valueOf(7L));
        Assert.assertEquals((String)"entryLogger's leastUnflushedLogId ", (long)8L, (long)entryLogger.getLeastUnflushedLogId());
    }

    String[] createAndGetLedgerDirs(int numOfLedgerDirs) throws IOException {
        String[] ledgerDirsPath = new String[numOfLedgerDirs];
        for (int i = 0; i < numOfLedgerDirs; ++i) {
            File ledgerDir = this.createTempDir("bkTest", ".dir");
            File curDir = Bookie.getCurrentDirectory((File)ledgerDir);
            Bookie.checkDirectoryStructure((File)curDir);
            ledgerDirsPath[i] = ledgerDir.getAbsolutePath();
        }
        return ledgerDirsPath;
    }

    @Test
    public void testFlushIntervalInBytes() throws Exception {
        long flushIntervalInBytes = 5000L;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setFlushIntervalInBytes(flushIntervalInBytes);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManagerBase = (EntryLogManagerBase)entryLogger.getEntryLogManager();
        long ledgerId = 0L;
        int firstEntrySize = 1000;
        long entry0Position = entryLogger.addEntry(0L, this.generateEntry(ledgerId, 0L, firstEntrySize));
        long expectedUnpersistedBytes = 1024 + firstEntrySize + 4;
        Assert.assertEquals((String)"Unpersisted Bytes of entrylog", (long)expectedUnpersistedBytes, (long)entryLogManagerBase.getCurrentLogForLedger(ledgerId).getUnpersistedBytes());
        int secondEntrySize = (int)(flushIntervalInBytes - expectedUnpersistedBytes);
        long entry1Position = entryLogger.addEntry(0L, this.generateEntry(ledgerId, 1L, secondEntrySize));
        Assert.assertEquals((String)"Unpersisted Bytes of entrylog", (long)0L, (long)entryLogManagerBase.getCurrentLogForLedger(ledgerId).getUnpersistedBytes());
        conf.setEntryLogPerLedgerEnabled(false);
        EntryLogger newEntryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManager newEntryLogManager = newEntryLogger.getEntryLogManager();
        Assert.assertEquals((String)"EntryLogManager class type", EntryLogManagerForSingleEntryLog.class, newEntryLogManager.getClass());
        ByteBuf buf = newEntryLogger.readEntry(ledgerId, 0L, entry0Position);
        long readLedgerId = buf.readLong();
        long readEntryId = buf.readLong();
        Assert.assertEquals((String)"LedgerId", (long)ledgerId, (long)readLedgerId);
        Assert.assertEquals((String)"EntryId", (long)0L, (long)readEntryId);
        buf = newEntryLogger.readEntry(ledgerId, 1L, entry1Position);
        readLedgerId = buf.readLong();
        readEntryId = buf.readLong();
        Assert.assertEquals((String)"LedgerId", (long)ledgerId, (long)readLedgerId);
        Assert.assertEquals((String)"EntryId", (long)1L, (long)readEntryId);
    }

    @Test
    public void testEntryLogManagerInterfaceForEntryLogPerLedger() throws Exception {
        long i;
        long i2;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)0L, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of Rotated Logs ", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
        int numOfLedgers = 5;
        int numOfThreadsPerLedger = 10;
        this.validateLockAcquireAndRelease(numOfLedgers, numOfThreadsPerLedger, entryLogManager);
        for (i2 = 0L; i2 < (long)numOfLedgers; ++i2) {
            entryLogManager.setCurrentLogForLedgerAndAddToRotate(i2, this.createDummyBufferedLogChannel(entryLogger, i2, conf));
        }
        for (i2 = 0L; i2 < (long)numOfLedgers; ++i2) {
            Assert.assertEquals((String)("LogChannel for ledger: " + i2), (Object)entryLogManager.getCurrentLogIfPresent(i2), (Object)entryLogManager.getCurrentLogForLedger(i2));
        }
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)numOfLedgers, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of Rotated Logs ", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
        for (i2 = 0L; i2 < (long)numOfLedgers; ++i2) {
            entryLogManager.setCurrentLogForLedgerAndAddToRotate(i2, this.createDummyBufferedLogChannel(entryLogger, (long)numOfLedgers + i2, conf));
        }
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)numOfLedgers, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of Rotated Logs ", (long)numOfLedgers, (long)entryLogManager.getRotatedLogChannels().size());
        for (i2 = 0L; i2 < (long)numOfLedgers; ++i2) {
            entryLogManager.setCurrentLogForLedgerAndAddToRotate(i2, this.createDummyBufferedLogChannel(entryLogger, (long)(2 * numOfLedgers) + i2, conf));
        }
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)numOfLedgers, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of Rotated Logs ", (long)(2 * numOfLedgers), (long)entryLogManager.getRotatedLogChannels().size());
        for (EntryLogger.BufferedLogChannel logChannel : entryLogManager.getRotatedLogChannels()) {
            entryLogManager.getRotatedLogChannels().remove(logChannel);
        }
        Assert.assertEquals((String)"Number of Rotated Logs ", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
        for (i = 0L; i < (long)numOfLedgers; ++i) {
            Assert.assertEquals((String)("EntryLogid for Ledger " + i), (long)((long)(2 * numOfLedgers) + i), (long)entryLogManager.getCurrentLogForLedger(i).getLogId());
        }
        for (i = (long)(2 * numOfLedgers); i < (long)(3 * numOfLedgers); ++i) {
            Assert.assertTrue((String)("EntryLog with logId: " + i + " should be present"), (entryLogManager.getCurrentLogIfPresent(i) != null ? 1 : 0) != 0);
        }
    }

    private EntryLogger.BufferedLogChannel createDummyBufferedLogChannel(EntryLogger entryLogger, long logid, ServerConfiguration servConf) throws IOException {
        File tmpFile = File.createTempFile("entrylog", logid + "");
        tmpFile.deleteOnExit();
        FileChannel fc = new RandomAccessFile(tmpFile, "rw").getChannel();
        EntryLogger.BufferedLogChannel logChannel = new EntryLogger.BufferedLogChannel((ByteBufAllocator)UnpooledByteBufAllocator.DEFAULT, fc, 10, 10, logid, tmpFile, servConf.getFlushIntervalInBytes());
        return logChannel;
    }

    private void validateLockAcquireAndRelease(int numOfLedgers, int numOfThreadsPerLedger, EntryLogManagerForEntryLogPerLedger entryLogManager) throws InterruptedException {
        ExecutorService tpe = Executors.newFixedThreadPool(numOfLedgers * numOfThreadsPerLedger);
        CountDownLatch latchToStart = new CountDownLatch(1);
        CountDownLatch latchToWait = new CountDownLatch(1);
        AtomicInteger numberOfThreadsAcquiredLock = new AtomicInteger(0);
        AtomicBoolean irptExceptionHappened = new AtomicBoolean(false);
        Random rand = new Random();
        for (int i = 0; i < numOfLedgers * numOfThreadsPerLedger; ++i) {
            long ledgerId = i % numOfLedgers;
            tpe.submit(() -> {
                try {
                    latchToStart.await();
                    Lock lock = entryLogManager.getLock(ledgerId);
                    lock.lock();
                    numberOfThreadsAcquiredLock.incrementAndGet();
                    latchToWait.await();
                    lock.unlock();
                }
                catch (IOException | InterruptedException e) {
                    irptExceptionHappened.set(true);
                }
            });
        }
        Assert.assertEquals((String)"Number Of Threads acquired Lock", (long)0L, (long)numberOfThreadsAcquiredLock.get());
        latchToStart.countDown();
        Thread.sleep(1000L);
        int currentNumberOfThreadsAcquiredLock = numberOfThreadsAcquiredLock.get();
        Assert.assertTrue((String)("Number Of Threads acquired Lock " + currentNumberOfThreadsAcquiredLock), (currentNumberOfThreadsAcquiredLock > 0 && currentNumberOfThreadsAcquiredLock <= numOfLedgers ? 1 : 0) != 0);
        latchToWait.countDown();
        Thread.sleep(2000L);
        Assert.assertEquals((String)"Number Of Threads acquired Lock", (long)(numOfLedgers * numOfThreadsPerLedger), (long)numberOfThreadsAcquiredLock.get());
    }

    @Test
    public void testEntryLogManagerExpiryRemoval() throws Exception {
        int evictionPeriod = 1;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        conf.setEntrylogMapAccessExpiryTimeInSeconds(evictionPeriod);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        long ledgerId = 0L;
        EntryLogger.BufferedLogChannel logChannel = this.createDummyBufferedLogChannel(entryLogger, 0L, conf);
        entryLogManager.setCurrentLogForLedgerAndAddToRotate(ledgerId, logChannel);
        EntryLogger.BufferedLogChannel currentLogForLedger = entryLogManager.getCurrentLogForLedger(ledgerId);
        Assert.assertEquals((String)("LogChannel for ledger " + ledgerId + " should match"), (Object)logChannel, (Object)currentLogForLedger);
        Thread.sleep(evictionPeriod * 1000 + 100);
        entryLogManager.doEntryLogMapCleanup();
        currentLogForLedger = entryLogManager.getCurrentLogForLedger(ledgerId);
        Assert.assertEquals((String)("LogChannel for ledger " + ledgerId + " should be null"), null, (Object)currentLogForLedger);
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)0L, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of rotated EntryLogs ", (long)1L, (long)entryLogManager.getRotatedLogChannels().size());
        Assert.assertTrue((String)"CopyOfRotatedLogChannels should contain the created LogChannel", (boolean)entryLogManager.getRotatedLogChannels().contains(logChannel));
        Assert.assertTrue((String)"since mapentry must have been evicted, it should be null", (entryLogManager.getCacheAsMap().get(ledgerId) == null || ((EntryLogManagerForEntryLogPerLedger.EntryLogAndLockTuple)entryLogManager.getCacheAsMap().get(ledgerId)).getEntryLogWithDirInfo() == null ? 1 : 0) != 0);
    }

    @Test
    public void testCacheMaximumSizeEvictionPolicy() throws Exception {
        this.entryLogger.shutdown();
        int cacheMaximumSize = 20;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(1));
        conf.setMaximumNumberOfActiveEntryLogs(20);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        this.entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)this.entryLogger.getEntryLogManager();
        for (int i = 0; i < 30; ++i) {
            entryLogManager.createNewLog((long)i);
            int cacheSize = entryLogManager.getCacheAsMap().size();
            Assert.assertTrue((String)("Cache maximum size is expected to be less than 20 but current cacheSize is " + cacheSize), (cacheSize <= 20 ? 1 : 0) != 0);
        }
    }

    @Test
    public void testLongLedgerIdsWithEntryLogPerLedger() throws Exception {
        int entryId;
        long ledgerId;
        int i;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(1));
        conf.setLedgerStorageClass(InterleavedLedgerStorage.class.getName());
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        int numOfLedgers = 5;
        int numOfEntries = 4;
        long[][] pos = new long[numOfLedgers][numOfEntries];
        for (i = 0; i < numOfLedgers; ++i) {
            ledgerId = Long.MAX_VALUE - (long)i;
            entryLogManager.createNewLog(ledgerId);
            for (entryId = 0; entryId < numOfEntries; ++entryId) {
                pos[i][entryId] = entryLogger.addEntry(ledgerId, EntryLogTest.generateEntry(ledgerId, entryId).nioBuffer());
            }
        }
        entryLogger.checkpoint();
        for (i = 0; i < numOfLedgers; ++i) {
            ledgerId = Long.MAX_VALUE - (long)i;
            for (entryId = 0; entryId < numOfEntries; ++entryId) {
                String expectedValue = EntryLogTest.generateDataString(ledgerId, entryId);
                ByteBuf buf = entryLogger.readEntry(ledgerId, (long)entryId, pos[i][entryId]);
                long readLedgerId = buf.readLong();
                long readEntryId = buf.readLong();
                byte[] readData = new byte[buf.readableBytes()];
                buf.readBytes(readData);
                Assert.assertEquals((String)"LedgerId ", (long)ledgerId, (long)readLedgerId);
                Assert.assertEquals((String)"EntryId ", (long)entryId, (long)readEntryId);
                Assert.assertEquals((String)"Entry Data ", (Object)expectedValue, (Object)new String(readData));
            }
        }
    }

    @Test
    public void testAppendLedgersMapOnCacheRemoval() throws Exception {
        int cacheMaximumSize = 5;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(true);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(1));
        conf.setMaximumNumberOfActiveEntryLogs(5);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        long ledgerId = 0L;
        entryLogManager.createNewLog(ledgerId);
        int entrySize = 200;
        int numOfEntries = 4;
        for (int i = 0; i < numOfEntries; ++i) {
            entryLogger.addEntry(ledgerId, this.generateEntry(ledgerId, i, entrySize));
        }
        EntryLogger.BufferedLogChannel logChannelForledger = entryLogManager.getCurrentLogForLedger(ledgerId);
        long logIdOfLedger = logChannelForledger.getLogId();
        entryLogger.checkpoint();
        try {
            entryLogger.extractEntryLogMetadataFromIndex(logIdOfLedger);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        for (int i = 1; i <= 5; ++i) {
            entryLogManager.createNewLog((long)i);
        }
        entryLogger.checkpoint();
        EntryLogMetadata entryLogMetadata = entryLogger.extractEntryLogMetadataFromIndex(logIdOfLedger);
        ConcurrentLongLongHashMap ledgersMap = entryLogMetadata.getLedgersMap();
        Assert.assertEquals((String)"There should be only one entry in entryLogMetadata", (long)1L, (long)ledgersMap.size());
        Assert.assertTrue((String)"Usage should be 1", (Double.compare(1.0, entryLogMetadata.getUsage()) == 0 ? 1 : 0) != 0);
        Assert.assertEquals((String)"Total size of entries", (long)((entrySize + 4) * numOfEntries), (long)ledgersMap.get(ledgerId));
    }

    @Test
    public void testExpiryRemovalByAccessingOnAnotherThread() throws Exception {
        final int evictionPeriod = 1;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        conf.setEntrylogMapAccessExpiryTimeInSeconds(evictionPeriod);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        final EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        final long ledgerId = 0L;
        EntryLogger.BufferedLogChannel newLogChannel = this.createDummyBufferedLogChannel(entryLogger, 1L, conf);
        entryLogManager.setCurrentLogForLedgerAndAddToRotate(ledgerId, newLogChannel);
        Thread t = new Thread(){

            @Override
            public void run() {
                try {
                    Thread.sleep(evictionPeriod * 1000 / 2);
                    entryLogManager.getCurrentLogForLedger(ledgerId);
                }
                catch (IOException | InterruptedException exception) {
                    // empty catch block
                }
            }
        };
        t.start();
        Thread.sleep(evictionPeriod * 1000 + 100);
        entryLogManager.doEntryLogMapCleanup();
        EntryLogger.BufferedLogChannel currentLogForLedger = entryLogManager.getCurrentLogForLedger(ledgerId);
        Assert.assertEquals((String)("LogChannel for ledger " + ledgerId), (Object)newLogChannel, (Object)currentLogForLedger);
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)1L, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of rotated EntryLogs ", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
    }

    @Test
    public void testExpiryRemovalByAccessingNonCacheRelatedMethods() throws Exception {
        int evictionPeriod = 1;
        final ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        conf.setEntrylogMapAccessExpiryTimeInSeconds(evictionPeriod);
        final LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        final EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        final EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        long ledgerId = 0L;
        final EntryLogger.BufferedLogChannel newLogChannel = this.createDummyBufferedLogChannel(entryLogger, 1L, conf);
        entryLogManager.setCurrentLogForLedgerAndAddToRotate(ledgerId, newLogChannel);
        final AtomicBoolean exceptionOccured = new AtomicBoolean(false);
        Thread t = new Thread(){

            @Override
            public void run() {
                try {
                    Thread.sleep(500L);
                    entryLogManager.getCopyOfCurrentLogs();
                    entryLogManager.getRotatedLogChannels();
                    entryLogManager.getCurrentLogIfPresent(newLogChannel.getLogId());
                    entryLogManager.getDirForNextEntryLog(ledgerDirsManager.getWritableLedgerDirs());
                    long newLedgerId = 100L;
                    EntryLogger.BufferedLogChannel logChannelForNewLedger = EntryLogTest.this.createDummyBufferedLogChannel(entryLogger, newLedgerId, conf);
                    entryLogManager.setCurrentLogForLedgerAndAddToRotate(newLedgerId, logChannelForNewLedger);
                    entryLogManager.getCurrentLogIfPresent(newLedgerId);
                }
                catch (Exception e) {
                    LOG.error("Got Exception in thread", (Throwable)e);
                    exceptionOccured.set(true);
                }
            }
        };
        t.start();
        Thread.sleep(evictionPeriod * 1000 + 100);
        entryLogManager.doEntryLogMapCleanup();
        Assert.assertFalse((String)"Exception occured in thread, which is not expected", (boolean)exceptionOccured.get());
        EntryLogger.BufferedLogChannel currentLogForLedger = entryLogManager.getCurrentLogForLedger(ledgerId);
        Assert.assertEquals((String)("LogChannel for ledger " + ledgerId + " should be null"), null, (Object)currentLogForLedger);
        Assert.assertEquals((String)"Number of current active EntryLogs ", (long)1L, (long)entryLogManager.getCopyOfCurrentLogs().size());
        Assert.assertEquals((String)"Number of rotated EntryLogs ", (long)1L, (long)entryLogManager.getRotatedLogChannels().size());
        Assert.assertTrue((String)"CopyOfRotatedLogChannels should contain the created LogChannel", (boolean)entryLogManager.getRotatedLogChannels().contains(newLogChannel));
        Assert.assertTrue((String)"since mapentry must have been evicted, it should be null", (entryLogManager.getCacheAsMap().get(ledgerId) == null || ((EntryLogManagerForEntryLogPerLedger.EntryLogAndLockTuple)entryLogManager.getCacheAsMap().get(ledgerId)).getEntryLogWithDirInfo() == null ? 1 : 0) != 0);
    }

    @Test
    public void testEntryLogManagerForEntryLogPerLedger() throws Exception {
        long i;
        EntryLogger.BufferedLogChannel logChannel;
        long i2;
        long i3;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setFlushIntervalInBytes(10000000L);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManager = (EntryLogManagerBase)entryLogger.getEntryLogManager();
        Assert.assertEquals((String)"EntryLogManager class type", EntryLogManagerForEntryLogPerLedger.class, entryLogManager.getClass());
        int numOfActiveLedgers = 20;
        int numEntries = 5;
        for (int j = 0; j < numEntries; ++j) {
            for (i3 = 0L; i3 < (long)numOfActiveLedgers; ++i3) {
                entryLogger.addEntry(i3, EntryLogTest.generateEntry(i3, j));
            }
        }
        for (i2 = 0L; i2 < (long)numOfActiveLedgers; ++i2) {
            EntryLogger.BufferedLogChannel logChannel2 = entryLogManager.getCurrentLogForLedger(i2);
            Assert.assertTrue((String)"unpersistedBytes should be greater than LOGFILE_HEADER_SIZE", (logChannel2.getUnpersistedBytes() > 1024L ? 1 : 0) != 0);
        }
        for (i2 = 0L; i2 < (long)numOfActiveLedgers; ++i2) {
            entryLogManager.createNewLog(i2);
        }
        List rotatedLogs = entryLogManager.getRotatedLogChannels();
        Assert.assertEquals((String)"Number of rotated entrylogs", (long)numOfActiveLedgers, (long)rotatedLogs.size());
        for (i3 = 0L; i3 < (long)numOfActiveLedgers; ++i3) {
            logChannel = entryLogManager.getCurrentLogForLedger(i3);
            Assert.assertEquals((String)"unpersistedBytes should be LOGFILE_HEADER_SIZE", (long)1024L, (long)logChannel.getUnpersistedBytes());
        }
        for (int j = numEntries; j < 2 * numEntries; ++j) {
            for (long i4 = 0L; i4 < (long)numOfActiveLedgers; ++i4) {
                entryLogger.addEntry(i4, EntryLogTest.generateEntry(i4, j));
            }
        }
        for (i = 0L; i < (long)numOfActiveLedgers; ++i) {
            logChannel = entryLogManager.getCurrentLogForLedger(i);
            Assert.assertTrue((String)"unpersistedBytes should be greater than LOGFILE_HEADER_SIZE", (logChannel.getUnpersistedBytes() > 1024L ? 1 : 0) != 0);
        }
        Assert.assertEquals((String)"LeastUnflushedloggerID", (long)0L, (long)entryLogger.getLeastUnflushedLogId());
        entryLogger.flush();
        Assert.assertEquals((String)"Number of rotated entrylogs", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
        Assert.assertEquals((String)"LeastUnflushedloggerID", (long)numOfActiveLedgers, (long)entryLogger.getLeastUnflushedLogId());
        for (i = 0L; i < (long)numOfActiveLedgers; ++i) {
            logChannel = entryLogManager.getCurrentLogForLedger(i);
            Assert.assertEquals((String)"unpersistedBytes should be 0", (long)0L, (long)logChannel.getUnpersistedBytes());
        }
    }

    @Test
    public void testReadAddCallsOfMultipleEntryLogs() throws Exception {
        byte[] data;
        long entryId;
        long ledgerId;
        ByteBuf buf;
        int i;
        int j;
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        conf.setEntryLogFilePreAllocationEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManagerBase = (EntryLogManagerBase)entryLogger.getEntryLogManager();
        int numOfActiveLedgers = 10;
        int numEntries = 10;
        long[][] positions = new long[numOfActiveLedgers][];
        for (int i2 = 0; i2 < numOfActiveLedgers; ++i2) {
            positions[i2] = new long[numEntries];
        }
        for (j = 0; j < numEntries; ++j) {
            for (i = 0; i < numOfActiveLedgers; ++i) {
                positions[i][j] = entryLogger.addEntry((long)i, EntryLogTest.generateEntry(i, j));
                long entryLogId = positions[i][j] >> 32;
                Assert.assertEquals((String)("EntryLogId for ledger: " + i), (long)i, (long)entryLogId);
            }
        }
        for (j = 0; j < numEntries; ++j) {
            for (i = 0; i < numOfActiveLedgers; ++i) {
                String expectedValue = "ledger-" + i + "-" + j;
                buf = entryLogger.readEntry((long)i, (long)j, positions[i][j]);
                ledgerId = buf.readLong();
                entryId = buf.readLong();
                data = new byte[buf.readableBytes()];
                buf.readBytes(data);
                Assert.assertEquals((String)"LedgerId ", (long)i, (long)ledgerId);
                Assert.assertEquals((String)"EntryId ", (long)j, (long)entryId);
                Assert.assertEquals((String)"Entry Data ", (Object)expectedValue, (Object)new String(data));
            }
        }
        for (long i3 = 0L; i3 < (long)numOfActiveLedgers; ++i3) {
            entryLogManagerBase.createNewLog(i3);
        }
        entryLogManagerBase.flushRotatedLogs();
        for (int j2 = 0; j2 < numEntries; ++j2) {
            for (i = 0; i < numOfActiveLedgers; ++i) {
                String expectedValue = "ledger-" + i + "-" + j2;
                buf = entryLogger.readEntry((long)i, (long)j2, positions[i][j2]);
                ledgerId = buf.readLong();
                entryId = buf.readLong();
                data = new byte[buf.readableBytes()];
                buf.readBytes(data);
                Assert.assertEquals((String)"LedgerId ", (long)i, (long)ledgerId);
                Assert.assertEquals((String)"EntryId ", (long)j2, (long)entryId);
                Assert.assertEquals((String)"Entry Data ", (Object)expectedValue, (Object)new String(data));
            }
        }
    }

    @Test
    public void testConcurrentReadCallsAfterEntryLogsAreRotated() throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setFlushIntervalInBytes(25000L);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(3));
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        int numOfActiveLedgers = 15;
        int numEntries = 2000;
        AtomicLongArray positions = new AtomicLongArray(numOfActiveLedgers * numEntries);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        for (int i = 0; i < numOfActiveLedgers; ++i) {
            for (int j = 0; j < numEntries; ++j) {
                positions.set(i * numEntries + j, entryLogger.addEntry((long)i, EntryLogTest.generateEntry(i, j)));
                long entryLogId = positions.get(i * numEntries + j) >> 32;
                Assert.assertEquals((String)("EntryLogId for ledger: " + i), (long)i, (long)entryLogId);
            }
        }
        for (long i = 0L; i < (long)numOfActiveLedgers; ++i) {
            entryLogManager.createNewLog(i);
        }
        entryLogManager.flushRotatedLogs();
        ArrayList<ReadTask> readTasks = new ArrayList<ReadTask>();
        for (int i = 0; i < numOfActiveLedgers; ++i) {
            for (int j = 0; j < numEntries; ++j) {
                readTasks.add(new ReadTask(i, j, positions.get(i * numEntries + j), entryLogger));
            }
        }
        ExecutorService executor = Executors.newFixedThreadPool(40);
        executor.invokeAll(readTasks).forEach(future -> {
            try {
                future.get();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                LOG.error("Read/Flush task failed because of InterruptedException", (Throwable)ie);
                Assert.fail((String)"Read/Flush task interrupted");
            }
            catch (Exception ex) {
                LOG.error("Read/Flush task failed because of  exception", (Throwable)ex);
                Assert.fail((String)("Read/Flush task failed " + ex.getMessage()));
            }
        });
    }

    @Test
    public void testEntryLoggerAddEntryWhenLedgerDirsAreFull() throws Exception {
        int numberOfLedgerDirs = 3;
        ArrayList<File> ledgerDirs = new ArrayList<File>();
        String[] ledgerDirsPath = new String[numberOfLedgerDirs];
        ArrayList<File> curDirs = new ArrayList<File>();
        for (int i = 0; i < numberOfLedgerDirs; ++i) {
            File ledgerDir = this.createTempDir("bkTest", ".dir").getAbsoluteFile();
            File curDir = Bookie.getCurrentDirectory((File)ledgerDir);
            Bookie.checkDirectoryStructure((File)curDir);
            ledgerDirs.add(ledgerDir);
            ledgerDirsPath[i] = ledgerDir.getPath();
            curDirs.add(curDir);
        }
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogFilePreAllocationEnabled(false);
        conf.setEntryLogPerLedgerEnabled(true);
        conf.setLedgerDirNames(ledgerDirsPath);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerForEntryLogPerLedger entryLogManager = (EntryLogManagerForEntryLogPerLedger)entryLogger.getEntryLogManager();
        Assert.assertEquals((String)"EntryLogManager class type", EntryLogManagerForEntryLogPerLedger.class, entryLogManager.getClass());
        entryLogger.addEntry(0L, EntryLogTest.generateEntry(0L, 1L));
        entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, 1L));
        entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, 1L));
        File ledgerDirForLedger0 = entryLogManager.getCurrentLogForLedger(0L).getLogFile().getParentFile();
        File ledgerDirForLedger1 = entryLogManager.getCurrentLogForLedger(1L).getLogFile().getParentFile();
        File ledgerDirForLedger2 = entryLogManager.getCurrentLogForLedger(2L).getLogFile().getParentFile();
        HashSet<File> ledgerDirsSet = new HashSet<File>();
        ledgerDirsSet.add(ledgerDirForLedger0);
        ledgerDirsSet.add(ledgerDirForLedger1);
        ledgerDirsSet.add(ledgerDirForLedger2);
        Assert.assertEquals((String)"Current active LedgerDirs size", (long)3L, (long)ledgerDirs.size());
        Assert.assertEquals((String)"Number of rotated logchannels", (long)0L, (long)entryLogManager.getRotatedLogChannels().size());
        ledgerDirsManager.addToFilledDirs(ledgerDirForLedger0);
        this.addEntryAndValidateFolders(entryLogger, (EntryLogManagerBase)entryLogManager, 2, ledgerDirForLedger0, false, ledgerDirForLedger1, ledgerDirForLedger2);
        Assert.assertEquals((String)"Number of rotated logchannels", (long)1L, (long)entryLogManager.getRotatedLogChannels().size());
        ledgerDirsManager.addToFilledDirs(ledgerDirForLedger1);
        this.addEntryAndValidateFolders(entryLogger, (EntryLogManagerBase)entryLogManager, 3, ledgerDirForLedger2, true, ledgerDirForLedger2, ledgerDirForLedger2);
        Assert.assertTrue((String)"Number of rotated logchannels", (2 <= entryLogManager.getRotatedLogChannels().size() && entryLogManager.getRotatedLogChannels().size() <= 3 ? 1 : 0) != 0);
        int numOfRotatedLogChannels = entryLogManager.getRotatedLogChannels().size();
        ledgerDirsManager.addToFilledDirs(ledgerDirForLedger2);
        this.addEntryAndValidateFolders(entryLogger, (EntryLogManagerBase)entryLogManager, 4, ledgerDirForLedger2, true, ledgerDirForLedger2, ledgerDirForLedger2);
        Assert.assertEquals((String)"Number of rotated logchannels", (long)numOfRotatedLogChannels, (long)entryLogManager.getRotatedLogChannels().size());
        ledgerDirsManager.addToWritableDirs(ledgerDirForLedger1, true);
        this.addEntryAndValidateFolders(entryLogger, (EntryLogManagerBase)entryLogManager, 4, ledgerDirForLedger1, true, ledgerDirForLedger1, ledgerDirForLedger1);
        Assert.assertEquals((String)"Number of rotated logchannels", (long)(numOfRotatedLogChannels + 3), (long)entryLogManager.getRotatedLogChannels().size());
    }

    void addEntryAndValidateFolders(EntryLogger entryLogger, EntryLogManagerBase entryLogManager, int entryId, File expectedDirForLedger0, boolean equalsForLedger0, File expectedDirForLedger1, File expectedDirForLedger2) throws IOException {
        entryLogger.addEntry(0L, EntryLogTest.generateEntry(0L, entryId));
        entryLogger.addEntry(1L, EntryLogTest.generateEntry(1L, entryId));
        entryLogger.addEntry(2L, EntryLogTest.generateEntry(2L, entryId));
        if (equalsForLedger0) {
            Assert.assertEquals((String)("LedgerDir for ledger 0 after adding entry " + entryId), (Object)expectedDirForLedger0, (Object)entryLogManager.getCurrentLogForLedger(0L).getLogFile().getParentFile());
        } else {
            Assert.assertNotEquals((String)("LedgerDir for ledger 0 after adding entry " + entryId), (Object)expectedDirForLedger0, (Object)entryLogManager.getCurrentLogForLedger(0L).getLogFile().getParentFile());
        }
        Assert.assertEquals((String)("LedgerDir for ledger 1 after adding entry " + entryId), (Object)expectedDirForLedger1, (Object)entryLogManager.getCurrentLogForLedger(1L).getLogFile().getParentFile());
        Assert.assertEquals((String)("LedgerDir for ledger 2 after adding entry " + entryId), (Object)expectedDirForLedger2, (Object)entryLogManager.getCurrentLogForLedger(2L).getLogFile().getParentFile());
    }

    @Test
    public void testSwappingEntryLogManagerFromEntryLogPerLedgerToSingle() throws Exception {
        this.testSwappingEntryLogManager(true, false);
    }

    @Test
    public void testSwappingEntryLogManagerFromSingleToEntryLogPerLedger() throws Exception {
        this.testSwappingEntryLogManager(false, true);
    }

    public void testSwappingEntryLogManager(boolean initialEntryLogPerLedgerEnabled, boolean laterEntryLogPerLedgerEnabled) throws Exception {
        ServerConfiguration conf = TestBKConfiguration.newServerConfiguration();
        conf.setEntryLogPerLedgerEnabled(initialEntryLogPerLedgerEnabled);
        conf.setLedgerDirNames(this.createAndGetLedgerDirs(2));
        conf.setEntryLogFilePreAllocationEnabled(true);
        LedgerDirsManager ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger entryLogger = new EntryLogger(conf, ledgerDirsManager);
        EntryLogManagerBase entryLogManager = (EntryLogManagerBase)entryLogger.getEntryLogManager();
        Assert.assertEquals((String)"EntryLogManager class type", initialEntryLogPerLedgerEnabled ? EntryLogManagerForEntryLogPerLedger.class : EntryLogManagerForSingleEntryLog.class, entryLogManager.getClass());
        int numOfActiveLedgers = 10;
        int numEntries = 10;
        long[][] positions = new long[numOfActiveLedgers][];
        for (int i = 0; i < numOfActiveLedgers; ++i) {
            positions[i] = new long[numEntries];
        }
        for (int j = 0; j < numEntries; ++j) {
            for (int i = 0; i < numOfActiveLedgers; ++i) {
                positions[i][j] = entryLogger.addEntry((long)i, EntryLogTest.generateEntry(i, j));
                long entryLogId = positions[i][j] >> 32;
                if (initialEntryLogPerLedgerEnabled) {
                    Assert.assertEquals((String)("EntryLogId for ledger: " + i), (long)i, (long)entryLogId);
                    continue;
                }
                Assert.assertEquals((String)("EntryLogId for ledger: " + i), (long)0L, (long)entryLogId);
            }
        }
        for (long i = 0L; i < (long)numOfActiveLedgers; ++i) {
            entryLogManager.createNewLog(i);
        }
        entryLogManager.flushRotatedLogs();
        conf.setEntryLogPerLedgerEnabled(laterEntryLogPerLedgerEnabled);
        LedgerDirsManager newLedgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()));
        EntryLogger newEntryLogger = new EntryLogger(conf, newLedgerDirsManager);
        EntryLogManager newEntryLogManager = newEntryLogger.getEntryLogManager();
        Assert.assertEquals((String)"EntryLogManager class type", laterEntryLogPerLedgerEnabled ? EntryLogManagerForEntryLogPerLedger.class : EntryLogManagerForSingleEntryLog.class, newEntryLogManager.getClass());
        for (int j = 0; j < numEntries; ++j) {
            for (int i = 0; i < numOfActiveLedgers; ++i) {
                String expectedValue = "ledger-" + i + "-" + j;
                ByteBuf buf = newEntryLogger.readEntry((long)i, (long)j, positions[i][j]);
                long ledgerId = buf.readLong();
                long entryId = buf.readLong();
                byte[] data = new byte[buf.readableBytes()];
                buf.readBytes(data);
                Assert.assertEquals((String)"LedgerId ", (long)i, (long)ledgerId);
                Assert.assertEquals((String)"EntryId ", (long)j, (long)entryId);
                Assert.assertEquals((String)"Entry Data ", (Object)expectedValue, (Object)new String(data));
            }
        }
    }

    class ReadTask
    implements Callable<Boolean> {
        long ledgerId;
        int entryId;
        long position;
        EntryLogger entryLogger;

        ReadTask(long ledgerId, int entryId, long position, EntryLogger entryLogger) {
            this.ledgerId = ledgerId;
            this.entryId = entryId;
            this.position = position;
            this.entryLogger = entryLogger;
        }

        @Override
        public Boolean call() throws IOException {
            try {
                ByteBuf expectedByteBuf = EntryLogTest.generateEntry(this.ledgerId, this.entryId);
                ByteBuf actualByteBuf = this.entryLogger.readEntry(this.ledgerId, (long)this.entryId, this.position);
                if (!expectedByteBuf.equals((Object)actualByteBuf)) {
                    LOG.error("Expected Entry: {} Actual Entry: {}", (Object)expectedByteBuf.toString(Charset.defaultCharset()), (Object)actualByteBuf.toString(Charset.defaultCharset()));
                    throw new IOException("Expected Entry: " + expectedByteBuf.toString(Charset.defaultCharset()) + " Actual Entry: " + actualByteBuf.toString(Charset.defaultCharset()));
                }
            }
            catch (IOException e) {
                LOG.error("Got Exception for GetEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, (Throwable)e);
                throw new IOException("Got Exception for GetEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, e);
            }
            return true;
        }
    }

    static class LedgerStorageReadTask
    implements Callable<Boolean> {
        long ledgerId;
        int entryId;
        LedgerStorage ledgerStorage;

        LedgerStorageReadTask(long ledgerId, int entryId, LedgerStorage ledgerStorage) {
            this.ledgerId = ledgerId;
            this.entryId = entryId;
            this.ledgerStorage = ledgerStorage;
        }

        @Override
        public Boolean call() throws IOException {
            try {
                ByteBuf expectedByteBuf = EntryLogTest.generateEntry(this.ledgerId, this.entryId);
                ByteBuf actualByteBuf = this.ledgerStorage.getEntry(this.ledgerId, (long)this.entryId);
                if (!expectedByteBuf.equals((Object)actualByteBuf)) {
                    LOG.error("Expected Entry: {} Actual Entry: {}", (Object)expectedByteBuf.toString(Charset.defaultCharset()), (Object)actualByteBuf.toString(Charset.defaultCharset()));
                    throw new IOException("Expected Entry: " + expectedByteBuf.toString(Charset.defaultCharset()) + " Actual Entry: " + actualByteBuf.toString(Charset.defaultCharset()));
                }
            }
            catch (IOException e) {
                LOG.error("Got Exception for GetEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, (Throwable)e);
                throw new IOException("Got Exception for GetEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, e);
            }
            return true;
        }
    }

    static class LedgerStorageFlushTask
    implements Callable<Boolean> {
        LedgerStorage ledgerStorage;

        LedgerStorageFlushTask(LedgerStorage ledgerStorage) {
            this.ledgerStorage = ledgerStorage;
        }

        @Override
        public Boolean call() throws IOException {
            try {
                this.ledgerStorage.flush();
            }
            catch (IOException e) {
                LOG.error("Got Exception for flush call", (Throwable)e);
                throw new IOException("Got Exception for Flush call", e);
            }
            return true;
        }
    }

    static class LedgerStorageWriteTask
    implements Callable<Boolean> {
        long ledgerId;
        int entryId;
        LedgerStorage ledgerStorage;

        LedgerStorageWriteTask(long ledgerId, int entryId, LedgerStorage ledgerStorage) {
            this.ledgerId = ledgerId;
            this.entryId = entryId;
            this.ledgerStorage = ledgerStorage;
        }

        @Override
        public Boolean call() throws IOException, BookieException {
            try {
                this.ledgerStorage.addEntry(EntryLogTest.generateEntry(this.ledgerId, this.entryId));
            }
            catch (IOException e) {
                LOG.error("Got Exception for AddEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, (Throwable)e);
                throw new IOException("Got Exception for AddEntry call. LedgerId: " + this.ledgerId + " entryId: " + this.entryId, e);
            }
            return true;
        }
    }
}

