/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.aws.xray;

import com.amazonaws.xray.AWSXRay;
import com.amazonaws.xray.entities.Entity;
import com.amazonaws.xray.entities.Segment;
import com.amazonaws.xray.entities.Subsegment;
import com.amazonaws.xray.entities.TraceID;
import com.amazonaws.xray.exceptions.AlreadyEmittedException;
import java.net.URI;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelContextAware;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.ExtendedCamelContext;
import org.apache.camel.NamedNode;
import org.apache.camel.Route;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.StaticService;
import org.apache.camel.component.aws.xray.NoopTracingStrategy;
import org.apache.camel.component.aws.xray.SegmentDecorator;
import org.apache.camel.spi.CamelEvent;
import org.apache.camel.spi.EventNotifier;
import org.apache.camel.spi.InterceptStrategy;
import org.apache.camel.spi.RoutePolicy;
import org.apache.camel.spi.RoutePolicyFactory;
import org.apache.camel.support.EventNotifierSupport;
import org.apache.camel.support.RoutePolicySupport;
import org.apache.camel.support.service.ServiceHelper;
import org.apache.camel.support.service.ServiceSupport;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class XRayTracer
extends ServiceSupport
implements RoutePolicyFactory,
StaticService,
CamelContextAware {
    public static final String XRAY_TRACE_ID = "Camel-AWS-XRay-Trace-ID";
    public static final String XRAY_TRACE_ENTITY = "Camel-AWS-XRay-Trace-Entity";
    private static final Logger LOG = LoggerFactory.getLogger(XRayTracer.class);
    private static final Pattern SANITIZE_NAME_PATTERN = Pattern.compile("[^\\w.:/%&#=+\\-@]");
    private static Map<String, SegmentDecorator> decorators = new HashMap<String, SegmentDecorator>();
    private static final String CURRENT_SEGMENT = "CAMEL_PROPERTY_AWS_XRAY_CURRENT_SEGMENT";
    private final XRayEventNotifier eventNotifier = new XRayEventNotifier();
    private CamelContext camelContext;
    private Set<String> excludePatterns = new HashSet<String>();
    private InterceptStrategy tracingStrategy;

    public void setCamelContext(CamelContext camelContext) {
        this.camelContext = camelContext;
    }

    public CamelContext getCamelContext() {
        return this.camelContext;
    }

    public RoutePolicy createRoutePolicy(CamelContext camelContext, String routeId, NamedNode route) {
        this.init(camelContext);
        return new XRayRoutePolicy(routeId);
    }

    protected void doInit() throws Exception {
        ObjectHelper.notNull((Object)this.camelContext, (String)"CamelContext", (Object)((Object)this));
        this.camelContext.getManagementStrategy().addEventNotifier((EventNotifier)this.eventNotifier);
        if (!this.camelContext.getRoutePolicyFactories().contains((Object)this)) {
            this.camelContext.addRoutePolicyFactory((RoutePolicyFactory)this);
        }
        if (null == this.tracingStrategy) {
            LOG.info("No tracing strategy available. Defaulting to no-op strategy");
            this.tracingStrategy = new NoopTracingStrategy();
        }
        ((ExtendedCamelContext)this.camelContext.adapt(ExtendedCamelContext.class)).addInterceptStrategy(this.tracingStrategy);
        LOG.debug("Initialized XRay tracer");
    }

    protected void doShutdown() throws Exception {
        this.camelContext.getManagementStrategy().removeEventNotifier((EventNotifier)this.eventNotifier);
        ServiceHelper.stopAndShutdownService((Object)((Object)this.eventNotifier));
        this.camelContext.getRoutePolicyFactories().remove((Object)this);
        LOG.debug("XRay tracer shutdown");
    }

    public void init(CamelContext camelContext) {
        if (!camelContext.hasService((Object)this)) {
            try {
                LOG.debug("Initializing XRay tracer");
                camelContext.addService((Object)this, true, true);
            }
            catch (Exception e) {
                throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
            }
        }
    }

    public InterceptStrategy getTracingStrategy() {
        return this.tracingStrategy;
    }

    public void setTracingStrategy(InterceptStrategy tracingStrategy) {
        this.tracingStrategy = tracingStrategy;
    }

    public Set<String> getExcludePatterns() {
        return this.excludePatterns;
    }

    public void setExcludePatterns(Set<String> excludePatterns) {
        this.excludePatterns = excludePatterns;
    }

    public void addExcludePattern(String pattern) {
        this.excludePatterns.add(pattern);
    }

    private boolean isExcluded(String routeId) {
        if (!this.excludePatterns.isEmpty()) {
            for (String pattern : this.excludePatterns) {
                if (!pattern.equals(routeId)) continue;
                LOG.debug("Ignoring route with ID {}", (Object)routeId);
                return true;
            }
        }
        return false;
    }

    protected SegmentDecorator getSegmentDecorator(Endpoint endpoint) {
        SegmentDecorator sd = decorators.get(URI.create(endpoint.getEndpointUri()).getScheme());
        if (null == sd) {
            return SegmentDecorator.DEFAULT;
        }
        return sd;
    }

    protected Entity getTraceEntityFromExchange(Exchange exchange) {
        Entity entity = (Entity)exchange.getIn().getHeader(XRAY_TRACE_ENTITY, Entity.class);
        if (entity == null) {
            entity = (Entity)exchange.getProperty(CURRENT_SEGMENT);
        }
        return entity;
    }

    public static String sanitizeName(String name) {
        return SANITIZE_NAME_PATTERN.matcher(name).replaceAll("_");
    }

    static {
        ServiceLoader.load(SegmentDecorator.class).forEach(d -> {
            SegmentDecorator existing = decorators.get(d.getComponent());
            if (existing == null || existing.getClass().isInstance(d)) {
                Logger log = LoggerFactory.getLogger(XRayTracer.class);
                log.trace("Adding segment decorator {}", (Object)d.getComponent());
                decorators.put(d.getComponent(), (SegmentDecorator)d);
            }
        });
    }

    private final class XRayRoutePolicy
    extends RoutePolicySupport {
        private String routeId;

        XRayRoutePolicy(String routeId) {
            this.routeId = routeId;
        }

        public void onExchangeBegin(Route route, Exchange exchange) {
            TraceID traceID;
            Entity entity;
            boolean createSegment;
            if (XRayTracer.this.isExcluded(route.getId())) {
                return;
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("=> RoutePolicy-Begin: Route: {} - RouteId: {}", (Object)this.routeId, (Object)route.getId());
            }
            boolean bl = createSegment = (entity = XRayTracer.this.getTraceEntityFromExchange(exchange)) == null || !Objects.equals(entity.getName(), this.routeId);
            if (exchange.getIn().getHeaders().containsKey(XRayTracer.XRAY_TRACE_ID)) {
                traceID = TraceID.fromString((String)((String)exchange.getIn().getHeader(XRayTracer.XRAY_TRACE_ID, String.class)));
            } else {
                traceID = new TraceID();
                exchange.getIn().setHeader(XRayTracer.XRAY_TRACE_ID, (Object)traceID.toString());
            }
            AWSXRay.setTraceEntity((Entity)entity);
            SegmentDecorator sd = XRayTracer.this.getSegmentDecorator(route.getEndpoint());
            if (createSegment) {
                Segment segment = AWSXRay.beginSegment((String)XRayTracer.sanitizeName(route.getId()));
                segment.setParent(entity);
                segment.setTraceId(traceID);
                sd.pre((Entity)segment, exchange, route.getEndpoint());
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Created new XRay segment {} with name {}", (Object)segment.getId(), (Object)segment.getName());
                }
                exchange.setProperty(XRayTracer.CURRENT_SEGMENT, (Object)segment);
            } else {
                String segmentName = entity.getId();
                try {
                    Subsegment subsegment = AWSXRay.beginSubsegment((String)route.getId());
                    sd.pre((Entity)subsegment, exchange, route.getEndpoint());
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Creating new subsegment with ID {} and name {} (parent {}, references: {})", new Object[]{subsegment.getId(), subsegment.getName(), subsegment.getParentSegment().getId(), subsegment.getParentSegment().getReferenceCount()});
                    }
                    exchange.setProperty(XRayTracer.CURRENT_SEGMENT, (Object)subsegment);
                }
                catch (AlreadyEmittedException aeEx) {
                    LOG.warn("Ignoring opening of subsegment {} as its parent segment {} was already emitted before.", (Object)route.getId(), (Object)segmentName);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onExchangeDone(Route route, Exchange exchange) {
            if (XRayTracer.this.isExcluded(route.getId())) {
                return;
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("=> RoutePolicy-Done: Route: {} - RouteId: {}", (Object)this.routeId, (Object)route.getId());
            }
            Entity entity = XRayTracer.this.getTraceEntityFromExchange(exchange);
            AWSXRay.setTraceEntity((Entity)entity);
            try {
                SegmentDecorator sd = XRayTracer.this.getSegmentDecorator(route.getEndpoint());
                sd.post(entity, exchange, route.getEndpoint());
                entity.close();
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Closing down (sub)segment {} with name {} (parent {}, references: {})", new Object[]{entity.getId(), entity.getName(), entity.getParentSegment().getId(), entity.getParentSegment().getReferenceCount()});
                }
                exchange.setProperty(XRayTracer.CURRENT_SEGMENT, (Object)entity.getParent());
            }
            catch (AlreadyEmittedException aeEx) {
                LOG.warn("Ignoring closing of (sub)segment {} as the segment was already emitted.", (Object)route.getId());
            }
            catch (Exception e) {
                LOG.warn("Error closing entity");
            }
            finally {
                AWSXRay.setTraceEntity(null);
            }
        }

        public String toString() {
            return "XRayRoutePolicy";
        }
    }

    private final class XRayEventNotifier
    extends EventNotifierSupport {
        private XRayEventNotifier() {
        }

        public void notify(CamelEvent event) throws Exception {
            if (event instanceof CamelEvent.ExchangeSendingEvent) {
                SegmentDecorator sd;
                CamelEvent.ExchangeSendingEvent ese = (CamelEvent.ExchangeSendingEvent)event;
                if (LOG.isTraceEnabled()) {
                    LOG.trace("-> {} - target: {} (routeId: {})", new Object[]{event.getClass().getSimpleName(), ese.getEndpoint(), ese.getExchange().getFromRouteId()});
                }
                if (!(sd = XRayTracer.this.getSegmentDecorator(ese.getEndpoint())).newSegment()) {
                    return;
                }
                Entity entity = XRayTracer.this.getTraceEntityFromExchange(ese.getExchange());
                if (entity != null) {
                    AWSXRay.setTraceEntity((Entity)entity);
                    Object name = sd.getOperationName(ese.getExchange(), ese.getEndpoint());
                    if (sd.getComponent() != null) {
                        name = sd.getComponent() + ":" + (String)name;
                    }
                    name = XRayTracer.sanitizeName((String)name);
                    try {
                        Subsegment subsegment = AWSXRay.beginSubsegment((String)name);
                        sd.pre((Entity)subsegment, ese.getExchange(), ese.getEndpoint());
                        if (LOG.isTraceEnabled()) {
                            LOG.trace("Creating new subsegment with ID {} and name {} (parent {}, references: {})", new Object[]{subsegment.getId(), subsegment.getName(), subsegment.getParentSegment().getId(), subsegment.getParentSegment().getReferenceCount()});
                        }
                        ese.getExchange().setProperty(XRayTracer.CURRENT_SEGMENT, (Object)subsegment);
                    }
                    catch (AlreadyEmittedException aeEx) {
                        LOG.warn("Ignoring starting of subsegment {} as its parent segment was already emitted to AWS.", name);
                    }
                } else {
                    LOG.trace("Ignoring creation of XRay subsegment as no segment exists in the current thread");
                }
            } else if (event instanceof CamelEvent.ExchangeSentEvent) {
                Entity entity;
                CamelEvent.ExchangeSentEvent ese = (CamelEvent.ExchangeSentEvent)event;
                if (LOG.isTraceEnabled()) {
                    LOG.trace("-> {} - target: {} (routeId: {})", new Object[]{event.getClass().getSimpleName(), ese.getEndpoint(), ese.getExchange().getFromRouteId()});
                }
                if ((entity = XRayTracer.this.getTraceEntityFromExchange(ese.getExchange())) instanceof Subsegment) {
                    AWSXRay.setTraceEntity((Entity)entity);
                    SegmentDecorator sd = XRayTracer.this.getSegmentDecorator(ese.getEndpoint());
                    try {
                        Subsegment subsegment = (Subsegment)entity;
                        sd.post((Entity)subsegment, ese.getExchange(), ese.getEndpoint());
                        subsegment.close();
                        if (LOG.isTraceEnabled()) {
                            LOG.trace("Closing down subsegment with ID {} and name {}", (Object)subsegment.getId(), (Object)subsegment.getName());
                            LOG.trace("Setting trace entity for exchange {} to {}", (Object)ese.getExchange(), (Object)subsegment.getParent());
                        }
                        ese.getExchange().setProperty(XRayTracer.CURRENT_SEGMENT, (Object)subsegment.getParent());
                    }
                    catch (AlreadyEmittedException aeEx) {
                        LOG.warn("Ignoring close of subsegment {} as its parent segment was already emitted to AWS", (Object)entity.getName());
                    }
                }
            } else {
                LOG.trace("Received event {} from source {}", (Object)event, event.getSource());
            }
        }

        public boolean isEnabled(CamelEvent event) {
            return event instanceof CamelEvent.ExchangeSendingEvent || event instanceof CamelEvent.ExchangeSentEvent;
        }

        public String toString() {
            return "XRayEventNotifier";
        }
    }
}

