/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.aws2.athena;

import java.time.Clock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.camel.Exchange;
import org.apache.camel.component.aws2.athena.Athena2Configuration;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.athena.model.GetQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.QueryExecutionState;

class Athena2QueryHelper {
    private static final Logger LOG = LoggerFactory.getLogger(Athena2QueryHelper.class);
    private final Clock clock = Clock.systemUTC();
    private final long waitTimeout;
    private final long delay;
    private final Set<String> retry;
    private final int maxAttempts;
    private final boolean resetWaitTimeoutOnAttempt;
    private final long absoluteStartMs;
    private long currentDelay;
    private int attempts;
    private boolean isFailure;
    private boolean isSuccess;
    private boolean isRetry;
    private long startMs;
    private boolean interrupted;

    Athena2QueryHelper(Exchange exchange, Athena2Configuration configuration) {
        this.waitTimeout = this.determineWaitTimeout(exchange, configuration);
        this.delay = this.determineDelay(exchange, configuration);
        this.maxAttempts = this.determineMaxAttempts(exchange, configuration);
        this.retry = this.determineRetry(exchange, configuration);
        this.resetWaitTimeoutOnAttempt = this.determineResetWaitTimeoutOnRetry(exchange, configuration);
        this.absoluteStartMs = this.now();
        this.currentDelay = this.determineInitialDelay(exchange, configuration);
    }

    private long now() {
        return this.clock.millis();
    }

    long getElapsedMillis() {
        return this.now() - this.absoluteStartMs;
    }

    void markAttempt() {
        if (this.attempts == 0) {
            this.startMs = this.now();
        } else if (this.resetWaitTimeoutOnAttempt) {
            this.startMs = this.now();
        }
        ++this.attempts;
        this.isFailure = false;
        this.isSuccess = false;
        this.isRetry = false;
    }

    int getAttempts() {
        return this.attempts;
    }

    boolean shouldAttempt() {
        if (this.attempts >= this.maxAttempts) {
            LOG.trace("AWS Athena start query execution used all {} attempts", (Object)this.maxAttempts);
            return false;
        }
        if (this.interrupted) {
            LOG.trace("AWS Athena start query execution thread was interrupted, will try no more");
            return false;
        }
        if (this.isFailure) {
            LOG.trace("AWS Athena start query execution detected permanent failure");
            return false;
        }
        if (this.isSuccess) {
            LOG.trace("AWS Athena start query execution detected success, will try no more");
            return false;
        }
        return true;
    }

    boolean shouldWait() {
        long now = this.now();
        long millisWaited = now - this.startMs;
        if (millisWaited >= this.waitTimeout) {
            LOG.trace("AWS Athena start query execution waited for {}, which exceeded wait timeout of {}", (Object)millisWaited, (Object)this.waitTimeout);
            return false;
        }
        if (this.interrupted) {
            LOG.trace("AWS Athena start query execution thread was interrupted, will wait no longer");
            return false;
        }
        if (this.isFailure) {
            LOG.trace("AWS Athena start query execution detected failure, will wait no longer");
            return false;
        }
        if (this.isSuccess) {
            LOG.trace("AWS Athena start query execution detected success, will wait no longer");
            return false;
        }
        if (this.isRetry) {
            LOG.trace("AWS Athena start query execution detected retry, will immediately attempt retry");
            return false;
        }
        return true;
    }

    void doWait() {
        try {
            Thread.sleep(this.currentDelay);
        }
        catch (InterruptedException e) {
            this.interrupted = Thread.interrupted();
            LOG.trace("AWS Athena start query execution wait thread was interrupted; will return at earliest opportunity");
        }
        this.currentDelay = this.delay;
    }

    boolean isComplete(GetQueryExecutionResponse getQueryExecutionResponse) {
        QueryExecutionState state = getQueryExecutionResponse.queryExecution().status().state();
        return state == QueryExecutionState.SUCCEEDED || state == QueryExecutionState.FAILED || state == QueryExecutionState.CANCELLED || state == QueryExecutionState.UNKNOWN_TO_SDK_VERSION;
    }

    boolean wasSuccessful(GetQueryExecutionResponse getQueryExecutionResponse) {
        QueryExecutionState state = getQueryExecutionResponse.queryExecution().status().state();
        return state == QueryExecutionState.SUCCEEDED;
    }

    void setStatusFrom(GetQueryExecutionResponse getQueryExecutionResponse) {
        if (this.isComplete(getQueryExecutionResponse)) {
            if (this.wasSuccessful(getQueryExecutionResponse)) {
                this.isSuccess = true;
            } else if (this.shouldRetry(getQueryExecutionResponse)) {
                this.isRetry = true;
            } else {
                LOG.trace("AWS Athena start query execution detected failure ({})", (Object)getQueryExecutionResponse.queryExecution().status().state());
                this.isFailure = true;
            }
        }
    }

    boolean shouldRetry(GetQueryExecutionResponse getQueryExecutionResponse) {
        String stateChangeReason = getQueryExecutionResponse.queryExecution().status().stateChangeReason();
        if (this.retry.contains("never")) {
            LOG.trace("AWS Athena start query execution detected error ({}), marked as not retryable", (Object)stateChangeReason);
            return false;
        }
        if (this.retry.contains("always")) {
            LOG.trace("AWS Athena start query execution detected error ({}), marked as retryable", (Object)stateChangeReason);
            return true;
        }
        if (stateChangeReason != null && stateChangeReason.contains("GENERIC_INTERNAL_ERROR") && (this.retry.contains("generic") || this.retry.contains("retryable"))) {
            LOG.trace("AWS Athena start query execution detected generic error ({}), marked as retryable", (Object)stateChangeReason);
            return true;
        }
        if (stateChangeReason != null && stateChangeReason.contains("exhausted resources at this scale factor") && (this.retry.contains("exhausted") || this.retry.contains("retryable"))) {
            LOG.trace("AWS Athena start query execution detected resource exhaustion error ({}), marked as retryable", (Object)stateChangeReason);
            return true;
        }
        return false;
    }

    private long determineWaitTimeout(Exchange exchange, Athena2Configuration configuration) {
        Long waitTimeout = (Long)exchange.getIn().getHeader("CamelAwsAthenaWaitTimeout", Long.class);
        if (ObjectHelper.isEmpty((Object)waitTimeout)) {
            waitTimeout = configuration.getWaitTimeout();
            LOG.trace("AWS Athena wait timeout is missing, using default one [{}]", (Object)waitTimeout);
        }
        if (ObjectHelper.isEmpty((Object)waitTimeout)) {
            throw new IllegalArgumentException("AWS Athena wait timeout required.");
        }
        if (waitTimeout < 0L) {
            throw new IllegalArgumentException("AWS Athena wait timeout must be >= 0");
        }
        return waitTimeout;
    }

    private long determineDelay(Exchange exchange, Athena2Configuration configuration) {
        Long delay = (Long)exchange.getIn().getHeader("CamelAwsAthenaDelay", Long.class);
        if (ObjectHelper.isEmpty((Object)delay)) {
            delay = configuration.getDelay();
            LOG.trace("AWS Athena delay is missing, using default one [{}]", (Object)delay);
        }
        if (ObjectHelper.isEmpty((Object)delay)) {
            throw new IllegalArgumentException("AWS Athena delay is required.");
        }
        if (delay < 0L) {
            throw new IllegalArgumentException("AWS Athena delay must be >= 0");
        }
        return delay;
    }

    private long determineInitialDelay(Exchange exchange, Athena2Configuration configuration) {
        Long initialDelay = (Long)exchange.getIn().getHeader("CamelAwsAthenaInitialDelay", Long.class);
        if (ObjectHelper.isEmpty((Object)initialDelay)) {
            initialDelay = configuration.getInitialDelay();
        }
        if (ObjectHelper.isEmpty((Object)initialDelay)) {
            initialDelay = this.determineDelay(exchange, configuration);
        }
        if (initialDelay < 0L) {
            throw new IllegalArgumentException("AWS Athena initial delay must be >= 0");
        }
        return initialDelay;
    }

    private Set<String> determineRetry(Exchange exchange, Athena2Configuration configuration) {
        String retry = (String)exchange.getIn().getHeader("CamelAwsAthenaRetry", String.class);
        if (ObjectHelper.isEmpty((Object)retry)) {
            retry = configuration.getRetry();
            LOG.trace("AWS Athena retry is missing, using default one [{}]", (Object)retry);
        }
        if (ObjectHelper.isEmpty((Object)retry)) {
            retry = "never";
        }
        String[] parts = retry.split(",");
        HashSet<String> finalRetry = new HashSet<String>();
        for (String part : parts) {
            if (!ObjectHelper.isNotEmpty((Object)part)) continue;
            finalRetry.add(part);
        }
        if (finalRetry.size() > 1 && (finalRetry.contains("retryable") || finalRetry.contains("always") || finalRetry.contains("never"))) {
            throw new IllegalArgumentException("AWS Athena retry is invalid - provide only one mutually exclusive option (retryable, always, never)");
        }
        List<String> valid = Arrays.asList("never", "always", "retryable", "exhausted", "generic");
        ArrayList<String> invalid = new ArrayList<String>();
        for (String r : finalRetry) {
            if (valid.contains(r)) continue;
            invalid.add(r);
        }
        if (!invalid.isEmpty()) {
            throw new IllegalArgumentException("AWS Athena retry is invalid - invalid values provided: " + invalid + ".  Valid values: " + valid);
        }
        return Collections.unmodifiableSet(finalRetry);
    }

    private Integer determineMaxAttempts(Exchange exchange, Athena2Configuration configuration) {
        Integer maxAttempts = (Integer)exchange.getIn().getHeader("CamelAwsAthenaMaxAttempts", Integer.class);
        if (ObjectHelper.isEmpty((Object)maxAttempts)) {
            maxAttempts = configuration.getMaxAttempts();
            LOG.trace("AWS Athena max attempts is missing, using default one [{}]", (Object)maxAttempts);
        }
        if (ObjectHelper.isEmpty((Object)maxAttempts)) {
            throw new IllegalArgumentException("AWS Athena max attempts is required.");
        }
        if (maxAttempts < 1) {
            throw new IllegalArgumentException("AWS Athena max attempts must be >= 1");
        }
        return maxAttempts;
    }

    private boolean determineResetWaitTimeoutOnRetry(Exchange exchange, Athena2Configuration configuration) {
        Boolean resetWaitTimeoutOnRetry = (Boolean)exchange.getIn().getHeader("CamelAwsAthenaResetWaitTimeoutOnRetry", Boolean.class);
        if (ObjectHelper.isEmpty((Object)resetWaitTimeoutOnRetry)) {
            resetWaitTimeoutOnRetry = configuration.isResetWaitTimeoutOnRetry();
            LOG.trace("AWS Athena reset wait timeout on retry is missing, using default one [{}]", (Object)resetWaitTimeoutOnRetry);
        }
        if (ObjectHelper.isEmpty((Object)resetWaitTimeoutOnRetry)) {
            throw new IllegalArgumentException("AWS Athena reset wait timeout on retry is required.");
        }
        return resetWaitTimeoutOnRetry;
    }

    long getWaitTimeout() {
        return this.waitTimeout;
    }

    long getDelay() {
        return this.delay;
    }

    Set<String> getRetry() {
        return this.retry;
    }

    int getMaxAttempts() {
        return this.maxAttempts;
    }

    boolean isResetWaitTimeoutOnAttempt() {
        return this.resetWaitTimeoutOnAttempt;
    }

    long getAbsoluteStartMs() {
        return this.absoluteStartMs;
    }

    long getCurrentDelay() {
        return this.currentDelay;
    }

    boolean isFailure() {
        return this.isFailure;
    }

    boolean isSuccess() {
        return this.isSuccess;
    }

    boolean isRetry() {
        return this.isRetry;
    }

    long getStartMs() {
        return this.startMs;
    }

    boolean isInterrupted() {
        return this.interrupted;
    }
}

