/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.google.storage;

import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.CopyWriter;
import com.google.cloud.storage.Storage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.InvalidPayloadException;
import org.apache.camel.Message;
import org.apache.camel.WrappedFile;
import org.apache.camel.component.google.storage.GoogleCloudStorageConfiguration;
import org.apache.camel.component.google.storage.GoogleCloudStorageEndpoint;
import org.apache.camel.component.google.storage.GoogleCloudStorageOperations;
import org.apache.camel.support.DefaultProducer;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoogleCloudStorageProducer
extends DefaultProducer {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorageProducer.class);
    private GoogleCloudStorageEndpoint endpoint;

    public GoogleCloudStorageProducer(GoogleCloudStorageEndpoint endpoint) {
        super((Endpoint)endpoint);
        this.endpoint = endpoint;
    }

    public void process(Exchange exchange) throws Exception {
        GoogleCloudStorageOperations operation = this.determineOperation(exchange);
        if (ObjectHelper.isEmpty((Object)((Object)operation))) {
            this.processFile(this.getEndpoint().getStorageClient(), exchange);
        } else {
            switch (operation) {
                case copyObject: {
                    this.copyObject(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                case deleteObject: {
                    this.deleteObject(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                case listBuckets: {
                    this.listBuckets(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                case deleteBucket: {
                    this.deleteBucket(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                case listObjects: {
                    this.listObjects(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                case getObject: {
                    this.getObject(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                case createDownloadLink: {
                    this.createDownloadLink(this.getEndpoint().getStorageClient(), exchange);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unsupported operation");
                }
            }
        }
    }

    private void processFile(Storage storage, Exchange exchange) throws IOException, InvalidPayloadException {
        String cc;
        String md5;
        String ce;
        String cd;
        InputStream is;
        String bucketName = this.determineBucketName();
        String objectName = this.determineObjectName(exchange);
        Map<String, String> objectMetadata = this.determineMetadata(exchange);
        Object obj = exchange.getIn().getMandatoryBody();
        if (obj instanceof WrappedFile) {
            obj = ((WrappedFile)obj).getFile();
        }
        if (obj instanceof File) {
            File filePayload = (File)obj;
            is = new FileInputStream(filePayload);
        } else {
            is = (InputStream)exchange.getIn().getMandatoryBody(InputStream.class);
        }
        is = this.setContentLength(objectMetadata, is);
        BlobId blobId = BlobId.of((String)bucketName, (String)objectName);
        BlobInfo.Builder builder = BlobInfo.newBuilder((BlobId)blobId);
        String ct = objectMetadata.remove("Content-Type");
        if (ct != null) {
            builder.setContentType(ct);
        }
        if ((cd = objectMetadata.remove("Content-Disposition")) != null) {
            builder.setContentDisposition(ct);
        }
        if ((ce = objectMetadata.remove("Content-Encoding")) != null) {
            builder.setContentEncoding(ct);
        }
        if ((md5 = objectMetadata.remove("Content-Md5")) != null) {
            builder.setMd5(md5);
        }
        if ((cc = objectMetadata.remove("Cache-Control")) != null) {
            builder.setCacheControl(ct);
        }
        BlobInfo blobInfo = builder.setMetadata(objectMetadata).build();
        Blob createdBlob = storage.createFrom(blobInfo, is, new Storage.BlobWriteOption[0]);
        LOG.trace("created createdBlob [{}]", (Object)createdBlob);
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody((Object)createdBlob);
        IOHelper.close((Closeable)is);
    }

    private InputStream setContentLength(Map<String, String> objectMetadata, InputStream is) throws IOException {
        if (!objectMetadata.containsKey("Content-Length") || objectMetadata.get("Content-Length").equals("0")) {
            LOG.debug("The content length is not defined. It needs to be determined by reading the data into memory");
            ByteArrayOutputStream baos = this.determineLengthInputStream(is);
            objectMetadata.put("Content-Length", String.valueOf(baos.size()));
            return new ByteArrayInputStream(baos.toByteArray());
        }
        return is;
    }

    private ByteArrayOutputStream determineLengthInputStream(InputStream is) throws IOException {
        int count;
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        byte[] bytes = new byte[1024];
        while ((count = is.read(bytes)) > 0) {
            out.write(bytes, 0, count);
        }
        return out;
    }

    private Map<String, String> determineMetadata(Exchange exchange) {
        String contentMD5;
        String contentEncoding;
        String contentDisposition;
        String cacheControl;
        HashMap<String, String> objectMetadata = new HashMap<String, String>();
        Long contentLength = (Long)exchange.getIn().getHeader("CamelGoogleCloudStorageContentLength", Long.class);
        if (contentLength != null) {
            objectMetadata.put("Content-Length", String.valueOf(contentLength));
        } else if (ObjectHelper.isNotEmpty((Object)exchange.getProperty("Content-Length"))) {
            objectMetadata.put("Content-Length", (String)exchange.getProperty("Content-Length", String.class));
        }
        String contentType = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageContentType", String.class);
        if (contentType != null) {
            objectMetadata.put("Content-Type", contentType);
        }
        if ((cacheControl = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageCacheControl", String.class)) != null) {
            objectMetadata.put("Cache-Control", cacheControl);
        }
        if ((contentDisposition = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageContentDisposition", String.class)) != null) {
            objectMetadata.put("Content-Disposition", contentDisposition);
        }
        if ((contentEncoding = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageContentEncoding", String.class)) != null) {
            objectMetadata.put("Content-Encoding", contentEncoding);
        }
        if ((contentMD5 = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageContentMd5", String.class)) != null) {
            objectMetadata.put("Content-Md5", contentMD5);
        }
        return objectMetadata;
    }

    private void createDownloadLink(Storage storage, Exchange exchange) {
        String bucketName = this.determineBucketName();
        String objectName = this.determineObjectName(exchange);
        Long expirationMillis = (Long)exchange.getIn().getHeader("CamelGoogleCloudStorageDownloadLinkExpirationTime", (Object)300000L, Long.class);
        long milliSeconds = 0L;
        milliSeconds = expirationMillis != null ? (milliSeconds += expirationMillis.longValue()) : (milliSeconds += 3600000L);
        BlobId blobId = BlobId.of((String)bucketName, (String)objectName);
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).build();
        URL url = storage.signUrl(blobInfo, milliSeconds, TimeUnit.MILLISECONDS, new Storage.SignUrlOption[0]);
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody((Object)url.toString());
    }

    private void copyObject(Storage storage, Exchange exchange) {
        String bucketName = this.determineBucketName();
        String objectName = this.determineObjectName(exchange);
        String destinationObjectName = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageDestinationObjectName", String.class);
        String bucketNameDestination = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageDestinationBucketName", String.class);
        if (ObjectHelper.isEmpty((String)bucketNameDestination)) {
            throw new IllegalArgumentException("Bucket Name Destination must be specified for copyObject Operation");
        }
        if (ObjectHelper.isEmpty((String)destinationObjectName)) {
            throw new IllegalArgumentException("Destination Key must be specified for copyObject Operation");
        }
        Bucket destinationBucketCheck = storage.get(bucketNameDestination, new Storage.BucketGetOption[0]);
        if (destinationBucketCheck != null) {
            LOG.trace("destinationBucketCheck [{}] already exists", (Object)destinationBucketCheck.getName());
        } else {
            LOG.trace("Destination Bucket [{}] doesn't exist yet", (Object)bucketNameDestination);
            if (this.getConfiguration().isAutoCreateBucket()) {
                destinationBucketCheck = GoogleCloudStorageEndpoint.createNewBucket(bucketNameDestination, this.getConfiguration(), storage);
            }
        }
        BlobId sourceBlobId = BlobId.of((String)bucketName, (String)objectName);
        BlobId targetBlobId = BlobId.of((String)bucketNameDestination, (String)destinationObjectName);
        Storage.CopyRequest request = Storage.CopyRequest.of((BlobId)sourceBlobId, (BlobId)targetBlobId);
        CopyWriter copyWriter = storage.copy(request);
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody((Object)copyWriter);
    }

    private void deleteObject(Storage storage, Exchange exchange) {
        String bucketName = this.determineBucketName();
        String objectName = this.determineObjectName(exchange);
        BlobId blobId = BlobId.of((String)bucketName, (String)objectName);
        boolean result = storage.delete(blobId);
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody((Object)result);
    }

    private void deleteBucket(Storage storage, Exchange exchange) {
        String bucketName = this.determineBucketName();
        for (Blob blob : storage.list(bucketName, new Storage.BlobListOption[0]).iterateAll()) {
            storage.delete(blob.getBlobId());
        }
        boolean result = storage.delete(bucketName, new Storage.BucketSourceOption[0]);
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody((Object)result);
    }

    private void listBuckets(Storage storage, Exchange exchange) {
        LinkedList<Bucket> bucketsList = new LinkedList<Bucket>();
        for (Bucket bucket : storage.list(new Storage.BucketListOption[0]).iterateAll()) {
            bucketsList.add(bucket);
        }
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody(bucketsList);
    }

    private void getObject(Storage storage, Exchange exchange) {
        String bucketName = this.determineBucketName();
        String objectName = this.determineObjectName(exchange);
        Blob blob = storage.get(BlobId.of((String)bucketName, (String)objectName));
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody((Object)blob);
    }

    private void listObjects(Storage storage, Exchange exchange) {
        String bucketName = this.determineBucketName();
        LinkedList<Blob> bloblist = new LinkedList<Blob>();
        for (Blob blob : storage.list(bucketName, new Storage.BlobListOption[0]).iterateAll()) {
            bloblist.add(blob);
        }
        Message message = GoogleCloudStorageProducer.getMessageForResponse(exchange);
        message.setBody(bloblist);
    }

    private String determineObjectName(Exchange exchange) {
        String key = (String)exchange.getIn().getHeader("CamelGoogleCloudStorageObjectName", String.class);
        if (ObjectHelper.isEmpty((String)key)) {
            key = this.getConfiguration().getObjectName();
        }
        if (key == null) {
            throw new IllegalArgumentException("Google Cloud Storage object name header missing.");
        }
        return key;
    }

    private String determineBucketName() {
        String bucketName = this.getConfiguration().getBucketName();
        if (bucketName == null) {
            throw new IllegalArgumentException("Bucket name is missing or not configured.");
        }
        return bucketName;
    }

    public static Message getMessageForResponse(Exchange exchange) {
        return exchange.getMessage();
    }

    private GoogleCloudStorageOperations determineOperation(Exchange exchange) {
        GoogleCloudStorageOperations operation = (GoogleCloudStorageOperations)((Object)exchange.getIn().getHeader("CamelGoogleCloudStorageOperation", GoogleCloudStorageOperations.class));
        if (operation == null) {
            operation = this.getConfiguration().getOperation();
        }
        return operation;
    }

    public GoogleCloudStorageEndpoint getEndpoint() {
        return (GoogleCloudStorageEndpoint)super.getEndpoint();
    }

    private GoogleCloudStorageConfiguration getConfiguration() {
        return this.endpoint.getConfiguration();
    }
}

